/*
  Graph

  A simple example of communication from the Launchpad to the computer:
  the value of analog input 0 is sent out the serial port. We call this "serial"
  communication because the connection appears to both the Launchpad and the
  computer as a serial port, even though it may actually use
  a USB cable. Bytes are sent one after another (serially) from the Launchpad
  to the computer.

  You can use the Energia serial monitor to view the sent data, or it can
  be read by Processing or another program capable of reading 
  data from a serial port. The Processing code below graphs the data received 
  so you can see the value of the analog input changing over time.

  The circuit:
  Any analog input sensor can be attached to an analog pin.  
  For this example, a potentiometer is attached to analog pin 3. 

  created 2006
  by David A. Mellis
  modified 9 Apr 2012
  by Tom Igoe and Scott Fitzgerald
  Modified 15 April 2013
  By Sean Alvarado
  Modified 24 February 2016
  By Paul Gingirch

  Hardware Required:
  * MSP-EXP430F5529 LaunchPad
  
  This example code is in the public domain.
*/

#define NUMBER 1 // take number / 2
#define MAX_JSON_STRLEN 40
int greenLedState = HIGH;
int redLedState = LOW;
uint8_t i = 0;
uint32_t average = 0;
uint32_t values[NUMBER];
char jsonStrFromHost[MAX_JSON_STRLEN];
uint8_t numBytesRead = 0;
uint8_t j = 0;
uint8_t n = 0;
boolean flag = false;
boolean running = false;
int debounceCounter = 0;
// From http://forum.43oh.com/topic/4730-using-the-internal-temp-sensor-on-f5529/
static const uint16_t *ADC15TEMP30 = (uint16_t *)0x1A1A;
static const uint16_t *ADC15TEMP85 = (uint16_t *)0x1A1C;
static const uint16_t *ADC25TEMP30 = (uint16_t *)0x1A22;
static const uint16_t *ADC25TEMP85 = (uint16_t *)0x1A24;
void setup() {
  pinMode(RED_LED, OUTPUT);
  pinMode(GREEN_LED, OUTPUT);
  analogReference(INTERNAL1V5);
  analogRead(TEMPSENSOR); // first reading usually wrong
  Serial.setTimeout(6000); // 1s timeout when reading from serial port
  Serial.begin(9600);
  pinMode(PUSH2, INPUT_PULLUP);
  digitalWrite(RED_LED, HIGH);
  digitalWrite(GREEN_LED, LOW);

  for (int i=0; i < NUMBER; i++) {
    values[i]=0;
  }
  clearJsonStr();
  Serial.flush();
  average = 0;

}

void printDec(uint32_t ui) {
  if (ui < 9999){
    Serial.print(ui/100, DEC);
    Serial.print(".");
    if ((ui%100) < 10){
      Serial.print("0");  
    }  
    Serial.print(ui%100, DEC);
  } 
  else {
    Serial.print("99.99");
  }
}
// From http://forum.43oh.com/topic/4730-using-the-internal-temp-sensor-on-f5529/
int16_t GetTempX100(int oversampling)
{
  uint32_t reading=0;
  int16_t result = 0;
  
  analogReference(INTERNAL1V5); //use the 1.5V internal reference
  
  for(int i=0; i<(1<<oversampling); i++)
    reading = analogRead(TEMPSENSOR);
  /* Celsius calculations:
  return map(reading,*ADC15TEMP30,*ADC15TEMP85,300,850); //convert our reading to Celsius & return
  */
  /* Fahrenheit calculations: */
  result = map(reading,*ADC15TEMP30,*ADC15TEMP85,8600,18500); //convert our reading to Farenheit & return
  
  return (result);
}

void clearJsonStr(){
   int i;
   for (i=0; i < MAX_JSON_STRLEN; i++){
     jsonStrFromHost[i] = (char)0;
   }  
}
int getSerialString(){
  int i=0;
  char rxChar;
  boolean done = false;
  int waitCtr = 0;
  clearJsonStr();
  while (!done){
      if (Serial.available() > 0){
         rxChar = Serial.read();
         if (rxChar != -1){
           if (rxChar == (char)10){
             done = true;

           } else if (i < MAX_JSON_STRLEN) {  
             jsonStrFromHost[i++] = rxChar;
           } else {
             Serial.flush();
             done = true;
             i = 0;
           }
         }  
      } else {
        delayMicroseconds(100);     // pauses for 0.1 milliseconds
        waitCtr++;
        if (waitCtr > 2000) {
          done = true;
        }  
      }  
  }
  return(i);
  
}  
void loop() {
  if (running){
    

       numBytesRead = getSerialString();//Serial.readBytesUntil('\n',jsonStrFromHost,MAX_JSON_STRLEN);
       if (numBytesRead > 2){
	  // Data from the host needs to be of the form {"LEDs":{"Red":0,"Green":0}} to turn off 
          // the red and green LEDs, {"LEDs":{"Red":1,"Green":1}} to turn on the red and green LEDs.
          // (Code is not generic at all, but it is small  )
             
          for (int j=2; j < numBytesRead; j++){
            if ((jsonStrFromHost[j] == ':') && (j < (numBytesRead + 1))){
              if (jsonStrFromHost[j-2] == 'd'){
                 if (jsonStrFromHost[j+1] == '1'){
                   redLedState = HIGH;
                 }  else {
                   redLedState = LOW;
                 }
                 digitalWrite(RED_LED, redLedState);
                 flag=true;
              } else {
                 if (jsonStrFromHost[j-2] == 'n'){
                   if (jsonStrFromHost[j+1] == '1'){
                      greenLedState = HIGH;
                   }  else {
                      greenLedState = LOW;
                   }
                   digitalWrite(GREEN_LED, greenLedState);
                   flag=true;
                 }
              }
            } 
          }
          
       }
    

    // Formula: http://www.43oh.com/forum/viewtopic.php?p=18248#p18248
    average -= values[n];
    values[n] = (uint32_t)GetTempX100(1);  //2^1 oversampling
     average += values[n];
    // Print average as a JSON string of the form {"temp":99.9,"LED":0}
    if (flag) {
     Serial.print("{\"temp\":");
      printDec(average/NUMBER);
      Serial.print(",\"LEDs\":{\"Red\":");
      Serial.print(redLedState, DEC);
      Serial.print(",\"Green\":");
      Serial.print(greenLedState, DEC);
      Serial.print("}}\n");
      //flag = false;

    } 
    n++;
    if (n==NUMBER) {
      flag=true;
      n = 0;
    } 
  } 
  if (digitalRead(PUSH2)==LOW) {
    if (debounceCounter == 2){
      running = !running;
      if (running) {
        greenLedState = HIGH;
        redLedState = LOW;
      } else {
        greenLedState = LOW;
        redLedState = LOW;
      }   
      digitalWrite(GREEN_LED, greenLedState);
      digitalWrite(RED_LED, redLedState);
      debounceCounter = 9;
    } else {
      debounceCounter++;
    }
  } else {
    debounceCounter = 0;
  }
  delay(40);

}



