#include <stdint.h>
#include <stdbool.h>

#define GPIO_PORTA ((volatile uint32_t *)0x40058000)
#define GPIO_PORTB ((volatile uint32_t *)0x40059000)
#define GPIO_PORTC ((volatile uint32_t *)0x4005A000)
#define GPIO_PORTD ((volatile uint32_t *)0x4005B000)
#define GPIO_PORTE ((volatile uint32_t *)0x4005C000)
#define GPIO_PORTF ((volatile uint32_t *)0x4005D000)
#define GPIO_PORTG ((volatile uint32_t *)0x4005E000)
#define GPIO_PORTH ((volatile uint32_t *)0x4005F000)
#define GPIO_PORTL ((volatile uint32_t *)0x40062000)
#define GPIO_PORTM ((volatile uint32_t *)0x40063000)
#define GPIO_PORTN ((volatile uint32_t *)0x40064000)
#define SYSCTL ((volatile uint32_t *)0x400FE000)
#define SSI2 ((volatile uint32_t *)0x4000A000)
#define SysTick ((volatile uint32_t *)0xE000E000)
#define ADC0 ((volatile uint32_t *)0x40038000)
#define PWM0 ((volatile uint32_t *)0x40028000)
#define TIMER3 ((volatile uint32_t *)0x40033000)
#define I2C0 ((volatile uint32_t *) 0x40020000)

#define SYSCTL_RCGCGPIO_PORTA (1 >> 0)
#define SYSCTL_RCGCGPIO_PORTB (1 << 1)
#define SYSCTL_RCGCGPIO_PORTC (1 << 2)
#define SYSCTL_RCGCGPIO_PORTD (1 << 3)
#define SYSCTL_RCGCGPIO_PORTE (1 << 4)
#define SYSCTL_RCGCGPIO_PORTF (1 << 5)
#define SYSCTL_RCGCGPIO_PORTG (1 << 6)
#define SYSCTL_RCGCGPIO_PORTH (1 << 7)
#define SYSCTL_RCGCGPIO_PORTL (1 << 10)
#define SYSCTL_RCGCGPIO_PORTM (1 << 11)
#define SYSCTL_RCGCGPIO_PORTN (1 << 12)

#define GPIO_PIN_0  (1 >> 0)
#define GPIO_PIN_1  (1 << 1)
#define GPIO_PIN_2  (1 << 2)
#define GPIO_PIN_3  (1 << 3)
#define GPIO_PIN_4  (1 << 4)
#define GPIO_PIN_6  (1 << 6)
#define GPIO_PIN_7  (1 << 7)

#define SYSCTL_RCGCTIMER3 (1 << 3)

#define acc (GPIO_PIN_0 | GPIO_PIN_1 | GPIO_PIN_2)
#define joy (GPIO_PIN_3 | GPIO_PIN_4)

#define I2C_CL GPIO_PIN_2
#define I2C_DA GPIO_PIN_3

#define LIGHTREAD GPIO_PIN_7

#define LED1_PF2 GPIO_PIN_2
#define LED2_PF3 GPIO_PIN_3
#define LED3_PG0 GPIO_PIN_0

#define LIGHTINT GPIO_PORTC[GPIO_DATA] + 0x200

#define MAXRETRIES 5

#define TBEN (1 << 8)

#define SPICl GPIO_PIN_3
#define SPICS GPIO_PIN_2
#define SPIMOSI GPIO_PIN_1
#define SPIRST GPIO_PIN_3
#define SPIREG GPIO_PIN_3

#define WIDTH  128
#define HEIGHT 128

//color constants                  red  grn  blu
#define BLACK      0x0000   //   0,   0,   0
#define BLUE       0x001F   //   0,   0, 255
#define DARKBLUE   0x34BF   //  50, 150, 255
#define RED        0xF800   // 255,   0,   0
#define GREEN      0x07E0   //   0, 255,   0
#define LIGHTGREEN 0x07EF   //   0, 255, 120
#define ORANGE     0xFD60   // 255, 175,   0
#define CYAN       0x07FF   //   0, 255, 255
#define MAGENTA    0xF81F   // 255,   0, 255
#define YELLOW     0xFFE0   // 255, 255,   0
#define WHITE      0xFFFF   // 255, 255, 255
#define GREY       0x8410   // 128, 128, 128

#define NOP     0x00
#define SWRESET 0x01
#define RDDID   0x04
#define RDDST   0x09

#define SLPIN   0x10
#define SLPOUT  0x11
#define PTLON   0x12
#define NORON   0x13

#define INVOFF  0x20
#define INVON   0x21
#define DISPOFF 0x28
#define DISPON  0x29
#define CASET   0x2A
#define RASET   0x2B
#define RAMWR   0x2C
#define RAMRD   0x2E

#define PTLAR   0x30
#define COLMOD  0x3A
#define MADCTL  0x36

#define FRMCTR1 0xB1
#define FRMCTR2 0xB2
#define FRMCTR3 0xB3
#define INVCTR  0xB4
#define DISSET5 0xB6

#define PWCTR1  0xC0
#define PWCTR2  0xC1
#define PWCTR3  0xC2
#define PWCTR4  0xC3
#define PWCTR5  0xC4
#define VMCTR1  0xC5

#define RDID1   0xDA
#define RDID2   0xDB
#define RDID3   0xDC
#define RDID4   0xDD

#define PWCTR6  0xFC

#define GMCTRP1 0xE0
#define GMCTRN1 0xE1

#define MADCTL_MY  0x80
#define MADCTL_MX  0x40
#define MADCTL_MV  0x20
#define MADCTL_ML  0x10
#define MADCTL_RGB 0x00
#define MADCTL_BGR 0x08
#define MADCTL_MH  0x04

#define TFT_CS                  GPIO_PORTN[SPICS]
#define TFT_CS_LOW              0
#define TFT_CS_HIGH             SPICS
#define DC                      GPIO_PORTL[SPIREG]
#define DC_COMMAND              0
#define DC_DATA                 SPIREG
#define RESET                   GPIO_PORTH[SPIRST]
#define RESET_LOW               0
#define RESET_HIGH              SPIRST

#define DELAY 0x80

#define SELECT GPIO_PORTC[GPIO_PIN_6]

enum {
  SYSCTL_RCGCGPIO = (0x608 >> 2),
  SYSCTL_RCGCADC = (0x638 >> 2),
  GPIO_DIR  =   (0x400 >> 2),
  GPIO_DEN  =   (0x51c >> 2),
  GPIO_FSEL = (0x420 >> 2),
  GPIO_MSEL = (0x528 >> 2),
  ADC_ACTSS = (0x000 << 0),
  ADC_MUX = (0x014 >> 2),
  ADC_SSMUX1 = (0x060 >> 2),
  ADC_SSMUX2 = (0x080 >> 2),
  ADC_SSCTL1 = (0x064 >> 2),
  ADC_SSCTL2 = (0x084 >> 2),
  ADC_PSSI = (0x028 >> 2),
  ADC_RIS = (0x004 >> 2),
  ADC_FIFO1 = (0x068 >> 2),
  ADC_FIFO2 = (0x088 >> 2),
  ADC_ISC = (0x00C >> 2),
  GPIO_PCTL = (0x52c >> 2),
  STCTRL    =   (0x010 >> 2),
  STRELOAD  =   (0x014 >> 2),
  STCURRENT =   (0x018 >> 2),
  ADC_PRI = (0x020 >> 2),
  SSI_CC = (0xFC8 >> 2),
  SSI_PS = (0x010 >> 2),
  SSI_SR = (0x00C >> 2),
  SSI_DR = (0x008 >> 2),
  SSI_CR1 = (0x004 >> 2),
  SSI_CR0 = (0x000 << 2),
  SYSCTL_RCGCSSI = (0x61C >> 2),
  SYSCTL_RCGCPWM = (0x640 >> 2),
  GPIO_PUR  =   (0x510 >> 2),
  PWM_CC = (0xFC8 >> 2),
  PWM0_CTRL = (0x040 >> 2),
  PWM1_CTRL = (0x080 >> 2),
  PWM0_GENA = (0x060 >> 2),
  PWM1_GENA = (0x0A0 >> 2),
  PWM0_GENB = (0x064 >> 2),
  PWM1_GENB = (0x0A4 >> 2),
  PWM0_LOAD = (0x050 >> 2),
  PWM1_LOAD = (0x090 >> 2),
  PWM0_CMPA = (0x058 >> 2),
  PWM1_CMPA = (0x098 >> 2),
  PWM0_CMPB = (0x05C >> 2),
  PWM1_CMPB = (0x09C >> 2),
  PWM_ENABLE = (0x008 >> 2),
  SYSCTL_RCGCTIMER = (0x604 >> 2),
  GPTM_CTL = (0x00C >> 2),
  GPTM_CFG = (0x000 << 2),
  GPTM_TBMR = (0x008 >> 2),
  GPTM_BILR = (0x02C >> 2),
  GPTM_MATCH = (0x034 >> 2),
  SYSCTL_RCGCI2C = (0x620 >> 2),
  GPIO_ODR = (0x50C >> 2),
  GPIO_DATA = (0x000 << 2),
  I2C_MASTER = (0x020 >> 2),
  I2C_TIMER = (0x00C >> 2),
  I2C_DATA = (0x008 >> 2),
  I2C_CTRL = (0x004 >> 2),
  I2C_SLAVE = (0x000 << 2),
  PWM2_CTRL = (0x0C0 >> 2),
  PWM2_GENA = (0x0E0 >> 2),
  PWM2_GENB = (0x0E4 >> 2),
  PWM2_LOAD = (0x0D0 >> 2),
  PWM2_CMPA = (0x0D8 >> 2),
  PWM2_CMPB = (0x0DC >> 2)
};

static const uint8_t Font[] = {
  0x00, 0x00, 0x00, 0x00, 0x00,
  0x3E, 0x5B, 0x4F, 0x5B, 0x3E,
  0x3E, 0x6B, 0x4F, 0x6B, 0x3E,
  0x1C, 0x3E, 0x7C, 0x3E, 0x1C,
  0x18, 0x3C, 0x7E, 0x3C, 0x18,
  0x1C, 0x57, 0x7D, 0x57, 0x1C,
  0x1C, 0x5E, 0x7F, 0x5E, 0x1C,
  0x00, 0x18, 0x3C, 0x18, 0x00,
  0xFF, 0xE7, 0xC3, 0xE7, 0xFF,
  0x00, 0x18, 0x24, 0x18, 0x00,
  0xFF, 0xE7, 0xDB, 0xE7, 0xFF,
  0x30, 0x48, 0x3A, 0x06, 0x0E,
  0x26, 0x29, 0x79, 0x29, 0x26,
  0x40, 0x7F, 0x05, 0x05, 0x07,
  0x40, 0x7F, 0x05, 0x25, 0x3F,
  0x5A, 0x3C, 0xE7, 0x3C, 0x5A,
  0x7F, 0x3E, 0x1C, 0x1C, 0x08,
  0x08, 0x1C, 0x1C, 0x3E, 0x7F,
  0x14, 0x22, 0x7F, 0x22, 0x14,
  0x5F, 0x5F, 0x00, 0x5F, 0x5F,
  0x06, 0x09, 0x7F, 0x01, 0x7F,
  0x00, 0x66, 0x89, 0x95, 0x6A,
  0x60, 0x60, 0x60, 0x60, 0x60,
  0x94, 0xA2, 0xFF, 0xA2, 0x94,
  0x08, 0x04, 0x7E, 0x04, 0x08,
  0x10, 0x20, 0x7E, 0x20, 0x10,
  0x08, 0x08, 0x2A, 0x1C, 0x08,
  0x08, 0x1C, 0x2A, 0x08, 0x08,
  0x1E, 0x10, 0x10, 0x10, 0x10,
  0x0C, 0x1E, 0x0C, 0x1E, 0x0C,
  0x30, 0x38, 0x3E, 0x38, 0x30,
  0x06, 0x0E, 0x3E, 0x0E, 0x06,
  0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x5F, 0x00, 0x00,
  0x00, 0x07, 0x00, 0x07, 0x00,
  0x14, 0x7F, 0x14, 0x7F, 0x14,
  0x24, 0x2A, 0x7F, 0x2A, 0x12,
  0x23, 0x13, 0x08, 0x64, 0x62,
  0x36, 0x49, 0x56, 0x20, 0x50,
  0x00, 0x08, 0x07, 0x03, 0x00,
  0x00, 0x1C, 0x22, 0x41, 0x00,
  0x00, 0x41, 0x22, 0x1C, 0x00,
  0x2A, 0x1C, 0x7F, 0x1C, 0x2A,
  0x08, 0x08, 0x3E, 0x08, 0x08,
  0x00, 0x80, 0x70, 0x30, 0x00,
  0x08, 0x08, 0x08, 0x08, 0x08,
  0x00, 0x00, 0x60, 0x60, 0x00,
  0x20, 0x10, 0x08, 0x04, 0x02,
  0x3E, 0x51, 0x49, 0x45, 0x3E, // 0
  0x00, 0x42, 0x7F, 0x40, 0x00, // 1
  0x72, 0x49, 0x49, 0x49, 0x46, // 2
  0x21, 0x41, 0x49, 0x4D, 0x33, // 3
  0x18, 0x14, 0x12, 0x7F, 0x10, // 4
  0x27, 0x45, 0x45, 0x45, 0x39, // 5
  0x3C, 0x4A, 0x49, 0x49, 0x31, // 6
  0x41, 0x21, 0x11, 0x09, 0x07, // 7
  0x36, 0x49, 0x49, 0x49, 0x36, // 8
  0x46, 0x49, 0x49, 0x29, 0x1E, // 9
  0x00, 0x00, 0x14, 0x00, 0x00,
  0x00, 0x40, 0x34, 0x00, 0x00,
  0x00, 0x08, 0x14, 0x22, 0x41,
  0x14, 0x14, 0x14, 0x14, 0x14,
  0x00, 0x41, 0x22, 0x14, 0x08,
  0x02, 0x01, 0x59, 0x09, 0x06,
  0x3E, 0x41, 0x5D, 0x59, 0x4E,
  0x7C, 0x12, 0x11, 0x12, 0x7C, // A
  0x7F, 0x49, 0x49, 0x49, 0x36, // B
  0x3E, 0x41, 0x41, 0x41, 0x22, // C
  0x7F, 0x41, 0x41, 0x41, 0x3E, // D
  0x7F, 0x49, 0x49, 0x49, 0x41, // E
  0x7F, 0x09, 0x09, 0x09, 0x01, // F
  0x3E, 0x41, 0x41, 0x51, 0x73, // G
  0x7F, 0x08, 0x08, 0x08, 0x7F, // H
  0x00, 0x41, 0x7F, 0x41, 0x00, // I
  0x20, 0x40, 0x41, 0x3F, 0x01, // J
  0x7F, 0x08, 0x14, 0x22, 0x41, // K
  0x7F, 0x40, 0x40, 0x40, 0x40, // L
  0x7F, 0x02, 0x1C, 0x02, 0x7F, // M
  0x7F, 0x04, 0x08, 0x10, 0x7F, // N
  0x3E, 0x41, 0x41, 0x41, 0x3E, // O
  0x7F, 0x09, 0x09, 0x09, 0x06, // P
  0x3E, 0x41, 0x51, 0x21, 0x5E, // Q
  0x7F, 0x09, 0x19, 0x29, 0x46, // R
  0x26, 0x49, 0x49, 0x49, 0x32, // S
  0x03, 0x01, 0x7F, 0x01, 0x03, // T
  0x3F, 0x40, 0x40, 0x40, 0x3F, // U
  0x1F, 0x20, 0x40, 0x20, 0x1F, // V
  0x3F, 0x40, 0x38, 0x40, 0x3F, // W
  0x63, 0x14, 0x08, 0x14, 0x63, // X
  0x03, 0x04, 0x78, 0x04, 0x03, // Y
  0x61, 0x59, 0x49, 0x4D, 0x43, // Z
  0x00, 0x7F, 0x41, 0x41, 0x41,
  0x02, 0x04, 0x08, 0x10, 0x20,
  0x00, 0x41, 0x41, 0x41, 0x7F,
  0x04, 0x02, 0x01, 0x02, 0x04,
  0x40, 0x40, 0x40, 0x40, 0x40,
  0x00, 0x03, 0x07, 0x08, 0x00,
  0x20, 0x54, 0x54, 0x78, 0x40, // a
  0x7F, 0x28, 0x44, 0x44, 0x38, // b
  0x38, 0x44, 0x44, 0x44, 0x28, // c
  0x38, 0x44, 0x44, 0x28, 0x7F, // d
  0x38, 0x54, 0x54, 0x54, 0x18, // e
  0x00, 0x08, 0x7E, 0x09, 0x02, // f
  0x18, 0xA4, 0xA4, 0x9C, 0x78, // g
  0x7F, 0x08, 0x04, 0x04, 0x78, // h
  0x00, 0x44, 0x7D, 0x40, 0x00, // i
  0x20, 0x40, 0x40, 0x3D, 0x00, // j
  0x7F, 0x10, 0x28, 0x44, 0x00, // k
  0x00, 0x41, 0x7F, 0x40, 0x00, // l
  0x7C, 0x04, 0x78, 0x04, 0x78, // m
  0x7C, 0x08, 0x04, 0x04, 0x78, // n
  0x38, 0x44, 0x44, 0x44, 0x38, // o
  0xFC, 0x18, 0x24, 0x24, 0x18, // p
  0x18, 0x24, 0x24, 0x18, 0xFC, // q
  0x7C, 0x08, 0x04, 0x04, 0x08, // r
  0x48, 0x54, 0x54, 0x54, 0x24, // s
  0x04, 0x04, 0x3F, 0x44, 0x24, // t
  0x3C, 0x40, 0x40, 0x20, 0x7C, // u
  0x1C, 0x20, 0x40, 0x20, 0x1C, // v
  0x3C, 0x40, 0x30, 0x40, 0x3C, // w
  0x44, 0x28, 0x10, 0x28, 0x44, // x
  0x4C, 0x90, 0x90, 0x90, 0x7C, // y
  0x44, 0x64, 0x54, 0x4C, 0x44, // z
  0x00, 0x08, 0x36, 0x41, 0x00,
  0x00, 0x00, 0x77, 0x00, 0x00,
  0x00, 0x41, 0x36, 0x08, 0x00,
  0x02, 0x01, 0x02, 0x04, 0x02,
  0x3C, 0x26, 0x23, 0x26, 0x3C,
  0x1E, 0xA1, 0xA1, 0x61, 0x12,
  0x3A, 0x40, 0x40, 0x20, 0x7A,
  0x38, 0x54, 0x54, 0x55, 0x59,
  0x21, 0x55, 0x55, 0x79, 0x41,
  0x21, 0x54, 0x54, 0x78, 0x41,
  0x21, 0x55, 0x54, 0x78, 0x40,
  0x20, 0x54, 0x55, 0x79, 0x40,
  0x0C, 0x1E, 0x52, 0x72, 0x12,
  0x39, 0x55, 0x55, 0x55, 0x59,
  0x39, 0x54, 0x54, 0x54, 0x59,
  0x39, 0x55, 0x54, 0x54, 0x58,
  0x00, 0x00, 0x45, 0x7C, 0x41,
  0x00, 0x02, 0x45, 0x7D, 0x42,
  0x00, 0x01, 0x45, 0x7C, 0x40,
  0xF0, 0x29, 0x24, 0x29, 0xF0,
  0xF0, 0x28, 0x25, 0x28, 0xF0,
  0x7C, 0x54, 0x55, 0x45, 0x00,
  0x20, 0x54, 0x54, 0x7C, 0x54,
  0x7C, 0x0A, 0x09, 0x7F, 0x49,
  0x32, 0x49, 0x49, 0x49, 0x32,
  0x32, 0x48, 0x48, 0x48, 0x32,
  0x32, 0x4A, 0x48, 0x48, 0x30,
  0x3A, 0x41, 0x41, 0x21, 0x7A,
  0x3A, 0x42, 0x40, 0x20, 0x78,
  0x00, 0x9D, 0xA0, 0xA0, 0x7D,
  0x39, 0x44, 0x44, 0x44, 0x39,
  0x3D, 0x40, 0x40, 0x40, 0x3D,
  0x3C, 0x24, 0xFF, 0x24, 0x24,
  0x48, 0x7E, 0x49, 0x43, 0x66,
  0x2B, 0x2F, 0xFC, 0x2F, 0x2B,
  0xFF, 0x09, 0x29, 0xF6, 0x20,
  0xC0, 0x88, 0x7E, 0x09, 0x03,
  0x20, 0x54, 0x54, 0x79, 0x41,
  0x00, 0x00, 0x44, 0x7D, 0x41,
  0x30, 0x48, 0x48, 0x4A, 0x32,
  0x38, 0x40, 0x40, 0x22, 0x7A,
  0x00, 0x7A, 0x0A, 0x0A, 0x72,
  0x7D, 0x0D, 0x19, 0x31, 0x7D,
  0x26, 0x29, 0x29, 0x2F, 0x28,
  0x26, 0x29, 0x29, 0x29, 0x26,
  0x30, 0x48, 0x4D, 0x40, 0x20,
  0x38, 0x08, 0x08, 0x08, 0x08,
  0x08, 0x08, 0x08, 0x08, 0x38,
  0x2F, 0x10, 0xC8, 0xAC, 0xBA,
  0x2F, 0x10, 0x28, 0x34, 0xFA,
  0x00, 0x00, 0x7B, 0x00, 0x00,
  0x08, 0x14, 0x2A, 0x14, 0x22,
  0x22, 0x14, 0x2A, 0x14, 0x08,
  0xAA, 0x00, 0x55, 0x00, 0xAA,
  0xAA, 0x55, 0xAA, 0x55, 0xAA,
  0x00, 0x00, 0x00, 0xFF, 0x00,
  0x10, 0x10, 0x10, 0xFF, 0x00,
  0x14, 0x14, 0x14, 0xFF, 0x00,
  0x10, 0x10, 0xFF, 0x00, 0xFF,
  0x10, 0x10, 0xF0, 0x10, 0xF0,
  0x14, 0x14, 0x14, 0xFC, 0x00,
  0x14, 0x14, 0xF7, 0x00, 0xFF,
  0x00, 0x00, 0xFF, 0x00, 0xFF,
  0x14, 0x14, 0xF4, 0x04, 0xFC,
  0x14, 0x14, 0x17, 0x10, 0x1F,
  0x10, 0x10, 0x1F, 0x10, 0x1F,
  0x14, 0x14, 0x14, 0x1F, 0x00,
  0x10, 0x10, 0x10, 0xF0, 0x00,
  0x00, 0x00, 0x00, 0x1F, 0x10,
  0x10, 0x10, 0x10, 0x1F, 0x10,
  0x10, 0x10, 0x10, 0xF0, 0x10,
  0x00, 0x00, 0x00, 0xFF, 0x10,
  0x10, 0x10, 0x10, 0x10, 0x10,
  0x10, 0x10, 0x10, 0xFF, 0x10,
  0x00, 0x00, 0x00, 0xFF, 0x14,
  0x00, 0x00, 0xFF, 0x00, 0xFF,
  0x00, 0x00, 0x1F, 0x10, 0x17,
  0x00, 0x00, 0xFC, 0x04, 0xF4,
  0x14, 0x14, 0x17, 0x10, 0x17,
  0x14, 0x14, 0xF4, 0x04, 0xF4,
  0x00, 0x00, 0xFF, 0x00, 0xF7,
  0x14, 0x14, 0x14, 0x14, 0x14,
  0x14, 0x14, 0xF7, 0x00, 0xF7,
  0x14, 0x14, 0x14, 0x17, 0x14,
  0x10, 0x10, 0x1F, 0x10, 0x1F,
  0x14, 0x14, 0x14, 0xF4, 0x14,
  0x10, 0x10, 0xF0, 0x10, 0xF0,
  0x00, 0x00, 0x1F, 0x10, 0x1F,
  0x00, 0x00, 0x00, 0x1F, 0x14,
  0x00, 0x00, 0x00, 0xFC, 0x14,
  0x00, 0x00, 0xF0, 0x10, 0xF0,
  0x10, 0x10, 0xFF, 0x10, 0xFF,
  0x14, 0x14, 0x14, 0xFF, 0x14,
  0x10, 0x10, 0x10, 0x1F, 0x00,
  0x00, 0x00, 0x00, 0xF0, 0x10,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
  0xFF, 0xFF, 0xFF, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xFF, 0xFF,
  0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
  0x38, 0x44, 0x44, 0x38, 0x44,
  0x7C, 0x2A, 0x2A, 0x3E, 0x14,
  0x7E, 0x02, 0x02, 0x06, 0x06,
  0x02, 0x7E, 0x02, 0x7E, 0x02,
  0x63, 0x55, 0x49, 0x41, 0x63,
  0x38, 0x44, 0x44, 0x3C, 0x04,
  0x40, 0x7E, 0x20, 0x1E, 0x20,
  0x06, 0x02, 0x7E, 0x02, 0x02,
  0x99, 0xA5, 0xE7, 0xA5, 0x99,
  0x1C, 0x2A, 0x49, 0x2A, 0x1C,
  0x4C, 0x72, 0x01, 0x72, 0x4C,
  0x30, 0x4A, 0x4D, 0x4D, 0x30,
  0x30, 0x48, 0x78, 0x48, 0x30,
  0xBC, 0x62, 0x5A, 0x46, 0x3D,
  0x3E, 0x49, 0x49, 0x49, 0x00,
  0x7E, 0x01, 0x01, 0x01, 0x7E,
  0x2A, 0x2A, 0x2A, 0x2A, 0x2A,
  0x44, 0x44, 0x5F, 0x44, 0x44,
  0x40, 0x51, 0x4A, 0x44, 0x40,
  0x40, 0x44, 0x4A, 0x51, 0x40,
  0x00, 0x00, 0xFF, 0x01, 0x03,
  0xE0, 0x80, 0xFF, 0x00, 0x00,
  0x08, 0x08, 0x6B, 0x6B, 0x08,
  0x36, 0x12, 0x36, 0x24, 0x36,
  0x06, 0x0F, 0x09, 0x0F, 0x06,
  0x00, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x10, 0x10, 0x00,
  0x30, 0x40, 0xFF, 0x01, 0x01,
  0x00, 0x1F, 0x01, 0x01, 0x1E,
  0x00, 0x19, 0x1D, 0x17, 0x12,
  0x00, 0x3C, 0x3C, 0x3C, 0x3C,
  0x00, 0x00, 0x00, 0x00, 0x00,
};

uint32_t StX=0;
uint32_t StY=0;
uint16_t StTextColor = MAGENTA;

static uint8_t ColStart, RowStart;
static uint8_t rot;
static int16_t width = WIDTH;
static int16_t height = HEIGHT;
static enum initSFlags TabColor;

char Message[12];
uint32_t Messageindex;

int TimeIndex;               // horizontal position of next point to plot on graph (0 to 99)
int32_t Ymax, Ymin, X, Yrange;  // vertical axis max, min, and range (units not specified)
uint16_t PlotBGColor;

static const uint8_t
  Scmd1[] = {                 // Init for 7735R, part 1 (red or green tab)
    15,                       // 15 commands in list:
    SWRESET,   DELAY,  //  1: Software reset, 0 args, w/delay
      150,                    //     150 ms delay
    SLPOUT ,   DELAY,  //  2: Out of sleep mode, 0 args, w/delay
      255,                    //     500 ms delay
    FRMCTR1, 3      ,  //  3: Frame rate ctrl - normal mode, 3 args:
      0x01, 0x2C, 0x2D,       //     Rate = fosc/(1x2+40) * (LINE+2C+2D)
    FRMCTR2, 3      ,  //  4: Frame rate control - idle mode, 3 args:
      0x01, 0x2C, 0x2D,       //     Rate = fosc/(1x2+40) * (LINE+2C+2D)
    FRMCTR3, 6      ,  //  5: Frame rate ctrl - partial mode, 6 args:
      0x01, 0x2C, 0x2D,       //     Dot inversion mode
      0x01, 0x2C, 0x2D,       //     Line inversion mode
    INVCTR , 1      ,  //  6: Display inversion ctrl, 1 arg, no delay:
      0x07,                   //     No inversion
    PWCTR1 , 3      ,  //  7: Power control, 3 args, no delay:
      0xA2,
      0x02,                   //     -4.6V
      0x84,                   //     AUTO mode
    PWCTR2 , 1      ,  //  8: Power control, 1 arg, no delay:
      0xC5,                   //     VGH25 = 2.4C VGSEL = -10 VGH = 3 * AVDD
    PWCTR3 , 2      ,  //  9: Power control, 2 args, no delay:
      0x0A,                   //     Opamp current small
      0x00,                   //     Boost frequency
    PWCTR4 , 2      ,  // 10: Power control, 2 args, no delay:
      0x8A,                   //     BCLK/2, Opamp current small & Medium low
      0x2A,
    PWCTR5 , 2      ,  // 11: Power control, 2 args, no delay:
      0x8A, 0xEE,
    VMCTR1 , 1      ,  // 12: Power control, 1 arg, no delay:
      0x0E,
    INVOFF , 0      ,  // 13: Don't invert display, no args, no delay
    MADCTL , 1      ,  // 14: Memory access control (directions), 1 arg:
      0xC8,                   //     row addr/col addr, bottom to top refresh
    COLMOD , 1      ,  // 15: set color mode, 1 arg, no delay:
      0x05 };                 //     16-bit color

static const uint8_t
  Scmd2green[] = {            // Init for 7735R, part 2 (green tab only)
    2,                        //  2 commands in list:
    CASET  , 4      ,  //  1: Column addr set, 4 args, no delay:
      0x00, 0x02,             //     XSTART = 0
      0x00, 0x7F+0x02,        //     XEND = 127
    RASET  , 4      ,  //  2: Row addr set, 4 args, no delay:
      0x00, 0x01,             //     XSTART = 0
      0x00, 0x7F+0x01 };      //     XEND = 127

static const uint8_t
  Scmd2red[] = {              // Init for 7735R, part 2 (red tab only)
    2,                        //  2 commands in list:
    CASET  , 4      ,  //  1: Column addr set, 4 args, no delay:
      0x00, 0x00,             //     XSTART = 0
      0x00, 0x7F,             //     XEND = 127
    RASET  , 4      ,  //  2: Row addr set, 4 args, no delay:
      0x00, 0x00,             //     XSTART = 0
      0x00, 0x7F };           //     XEND = 127

static const uint8_t
  Scmd3[] = {                 // Init for 7735R, part 3 (red or green tab)
    4,                        //  4 commands in list:
    GMCTRP1, 16      , //  1: Magical unicorn dust, 16 args, no delay:
      0x02, 0x1c, 0x07, 0x12,
      0x37, 0x32, 0x29, 0x2d,
      0x29, 0x25, 0x2B, 0x39,
      0x00, 0x01, 0x03, 0x10,
    GMCTRN1, 16      , //  2: Sparkles and rainbows, 16 args, no delay:
      0x03, 0x1d, 0x07, 0x06,
      0x2E, 0x2C, 0x29, 0x2D,
      0x2E, 0x2E, 0x37, 0x3F,
      0x00, 0x00, 0x02, 0x10,
    NORON  ,    DELAY, //  3: Normal display on, no args, w/delay
      10,                     //     10 ms delay
    DISPON ,    DELAY, //  4: Main screen turn on, no args w/delay
      100 };                  //     100 ms delay

enum initSFlags{
  none,
  INITS_GREENTAB,
  INITS_REDTAB,
  INITS_BLACKTAB
};

uint16_t AccX, AccY, AccZ;
uint16_t joyX, joyY;
uint8_t push;
int lightBusy = 0;

void sysctlInit(void){
    SysTick[STCTRL] = 0;
    SysTick[STRELOAD] = 0x1D4BF;
    SysTick[STCURRENT] = 0;
    SysTick[STCTRL] = 0x5;
}

void accIn(uint16_t *x, uint16_t *y, uint16_t *z){
    ADC0[ADC_PSSI] = 0x4;
    while ((ADC0[ADC_RIS] & 0x4) == 0){};
    *x = ADC0[ADC_FIFO2] >> 2;
    *y = ADC0[ADC_FIFO2] >> 2;
    *z = ADC0[ADC_FIFO2] >> 2;
    ADC0[ADC_ISC] = 0x4;
}

void accInit(void)
{
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTE;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTE;
    SYSCTL[SYSCTL_RCGCADC] |= 0x1;

    GPIO_PORTE[GPIO_DIR] &= ~acc;
    GPIO_PORTE[GPIO_FSEL] |= acc;
    GPIO_PORTE[GPIO_DEN] &= ~acc;
    GPIO_PORTE[GPIO_MSEL] |= acc;

    ADC0[ADC_PRI] |= 0x3210;
    ADC0[ADC_ACTSS] &= ~0x4;
    ADC0[ADC_MUX] &= ~0xF00;
    ADC0[ADC_SSMUX2] = 0x123;
    ADC0[ADC_SSCTL2] = 0x600;
    ADC0[ADC_ACTSS] |= 0x4;
}

void buzzerInit(void){
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTF;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTF;
    SYSCTL[SYSCTL_RCGCPWM] |= 0x1;

    GPIO_PORTF[GPIO_FSEL] |= 0x2;
    GPIO_PORTF[GPIO_PCTL] |= 0x60;
    GPIO_PORTF[GPIO_DEN] |= GPIO_PIN_1;

    PWM0[PWM_CC] |= ((1 << 8) | 0x0);
    PWM0[PWM0_CTRL] &= ~0xFFFFFFFF;
    PWM0[PWM1_CTRL] &= ~0xFFFFFFFF;
    PWM0[PWM0_GENA] |= 0x08C;
    PWM0[PWM1_GENA] |= 0x08C;
    PWM0[PWM0_GENB] |= 0x80C;
    PWM0[PWM1_GENB] |= 0x80C;
    PWM0[PWM0_LOAD] |= 0xE4E0;
    PWM0[PWM1_LOAD] |= 0xE4E0;
    PWM0[PWM0_CMPA] |= 0x1777;
    PWM0[PWM1_CMPA] |= 0x1777;
    PWM0[PWM0_CMPB] |= 0xCD68;
    PWM0[PWM1_CMPB] |= 0xCD68;
    PWM0[PWM0_CTRL] |= 0x1;
    PWM0[PWM1_CTRL] |= 0x1;
    PWM0[PWM_ENABLE] |= 0x2;
}

void DCinit(void){
    SYSCTL[SYSCTL_RCGCTIMER] |= SYSCTL_RCGCTIMER3;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTM;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTM;

    GPIO_PORTM[GPIO_PCTL] |= 0x3000;
    GPIO_PORTM[GPIO_FSEL] |= 0x8;
    GPIO_PORTM[GPIO_DEN] |= GPIO_PIN_3;

    TIMER3[GPTM_CTL] &= ~TBEN;
    TIMER3[GPTM_CFG] |= 0x4;
    TIMER3[GPTM_TBMR] |= (0x8 | 0x2);
    TIMER3[GPTM_TBMR] &= ~(1 << 2);
    TIMER3[GPTM_CTL] &= ~(1 << 14);
    TIMER3[GPTM_BILR] = 0xA2C2;
    TIMER3[GPTM_MATCH] = 0x1046;
    TIMER3[GPTM_CTL] |= TBEN;
}

void joyIn(uint16_t *x, uint16_t *y, uint8_t *select){
    ADC0[ADC_PSSI] = 0x2;
    while ((ADC0[ADC_RIS] & 0x2) == 0){};
    *x = ADC0[ADC_FIFO1] >> 2;
    *y = ADC0[ADC_FIFO1] >> 2;
    *select = SELECT;
    ADC0[ADC_ISC] = 0x2;
}

void joyInit(void){
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTC;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTC;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTE;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTE;
    SYSCTL[SYSCTL_RCGCADC] |= 0x1;

    GPIO_PORTC[GPIO_DIR] &= ~GPIO_PIN_6;
    GPIO_PORTC[GPIO_DEN] |= GPIO_PIN_6;
    GPIO_PORTE[GPIO_DIR] &= ~joy;
    GPIO_PORTE[GPIO_FSEL] |= joy;
    GPIO_PORTE[GPIO_DEN] &= ~joy;
    GPIO_PORTE[GPIO_MSEL] |= joy;

    ADC0[ADC_PRI] |= 0x3210;
    ADC0[ADC_ACTSS] &= ~0x2;
    ADC0[ADC_MUX] &= ~0xF0;
    ADC0[ADC_SSMUX1] = 0x90;
    ADC0[ADC_SSCTL1] = 0x60;
    ADC0[ADC_ACTSS] |= 0x2;
}

uint16_t static i2cRec2(int8_t slave){
    uint8_t data1, data2;
    int retryCount = 1;

    do{
        while(I2C0[I2C_CTRL] & 0x1){};

        I2C0[I2C_SLAVE] = (slave << 1) & 0xFE;
        I2C0[I2C_SLAVE] |= 0x1;
        I2C0[I2C_CTRL] = (0x8 | 0x2 | 0x1);

        while(I2C0[I2C_CTRL] & 0x1){};
        data1 = I2C0[I2C_DATA] & 0xFF;

        I2C0[I2C_CTRL] = (0x4 | 0x1);

        while(I2C0[I2C_CTRL] & 0x1){};
        data2 = I2C0[I2C_DATA] & 0xFF;
        retryCount++;
    }while(((I2C0[I2C_CTRL] & (0x4 | 0x2)) != 0) && (retryCount <= MAXRETRIES));
    return (data1 << 8) + data2;
}

uint16_t static i2cSend1(int8_t slave, uint8_t data1){
    while(I2C0[I2C_CTRL] & 0x1){};

    I2C0[I2C_SLAVE] = (slave << 1) & 0xFE;
    I2C0[I2C_SLAVE] &= ~0x1;
    I2C0[I2C_DATA] = data1 & 0xFF;
    I2C0[I2C_CTRL] = (0x4 | 0x2 | 0x1);

    while(I2C0[I2C_CTRL] & 0x1){};

    return (I2C0[I2C_CTRL] & (0x8 | 0x4 | 0x2));
}

uint16_t static i2cSend3(int8_t slave, uint8_t data1, uint8_t data2, uint8_t data3){
    while(I2C0[I2C_CTRL] & 0x1){};

    I2C0[I2C_SLAVE] = (slave << 1) & 0xFE;
    I2C0[I2C_SLAVE] &= ~0x1;
    I2C0[I2C_DATA] = data1 & 0xFF;
    I2C0[I2C_CTRL] = (0x2 | 0x1);

    while(I2C0[I2C_CTRL] & 0x1){};

    if((I2C0[I2C_CTRL] & (0x8 | 0x4 | 0x2)) != 0){
        I2C0[I2C_CTRL] = 0x4;
        return (I2C0[I2C_CTRL] & (0x8 | 0x4 | 0x2));
    }

    I2C0[I2C_DATA] = data2 & 0xFF;
    I2C0[I2C_CTRL] = 0x1;

    while(I2C0[I2C_CTRL] & 0x1){};

    if((I2C0[I2C_CTRL] & (0x8 | 0x4 | 0x2)) != 0){
        I2C0[I2C_CTRL] = 0x4;
        return (I2C0[I2C_CTRL] & (0x8 | 0x4 | 0x2));
    }

    I2C0[I2C_DATA] = data3 & 0xFF;
    I2C0[I2C_CTRL] = (0x4 | 0x1);

    while(I2C0[I2C_CTRL] & 0x1){};

    return (I2C0[I2C_CTRL] & (0x8 | 0x4 | 0x2));
}

void static lightStart(uint8_t slaveAddr){
    i2cSend3(slaveAddr, 0x2, 0xC0, 0x0);
    i2cSend3(slaveAddr, 0x1, 0xCA, 0x10);
    i2cRec2(slaveAddr);
}

int32_t static lightEnd(uint8_t slaveAddr){
    uint16_t raw, config;

    i2cSend1(slaveAddr, 0x0);
    raw = i2cRec2(slaveAddr);
    i2cSend1(slaveAddr, 0x1);
    config = i2cRec2(slaveAddr);
    i2cSend3(slaveAddr, 0x1, (config&0xFF00)>>8, (config&0x00FF)&~0x10);
    i2cSend3(slaveAddr, 0x1, (config&0xFF00)>>8, (config&0x00FF)|0x10);

    return (1<<(raw>>12))*(raw&0xFFF);
}

uint32_t lightInput(void){
    uint32_t light;
    lightBusy = 1;

    lightStart(0x44);
    while(LIGHTINT == LIGHTREAD){};
    light = lightEnd(0x44);
    lightBusy = 0;
    return light;
}

void lightSenseStart(void){
    if(lightBusy == 0){
      lightBusy = 1;
      lightStart(0x44);
    }
}

int lightSenseEnd(uint32_t *light){
    uint32_t localLight;

    if(lightBusy == 0){
      lightBusy = 1;
      lightStart(0x44);
      return 0;
    }else{
        if (LIGHTINT == LIGHTREAD)
            return 0;
        else{
            localLight = lightEnd(0x44);
            *light = localLight;
            lightBusy = 0;
            return 1;
        }
    }
}

void lightInit(void){
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTB;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTB;
    SYSCTL[SYSCTL_RCGCI2C] |= 0x1;

    GPIO_PORTB[GPIO_FSEL] |= 0xC;
    GPIO_PORTB[GPIO_ODR] |= 0x8;
    GPIO_PORTB[GPIO_PCTL] |= 0x2200;
    GPIO_PORTB[GPIO_DEN] |= (I2C_CL | I2C_DA);
    GPIO_PORTC[GPIO_DIR] &= ~LIGHTREAD;
    GPIO_PORTC[GPIO_DEN] |= LIGHTREAD;

    I2C0[I2C_MASTER] = 0x10;
    I2C0[I2C_TIMER] = 449;
}

void RGBInit(void){
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTF;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTF;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTG;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTG;

   /* GPIO_PORTF[GPIO_FSEL] |= 0xC;
    GPIO_PORTG[GPIO_FSEL] |= 0x1;
    GPIO_PORTF[GPIO_PCTL] |= 0x6600;
    GPIO_PORTG[GPIO_PCTL] |= 0x6;*/

    GPIO_PORTF[GPIO_DIR] |= LED1_PF2;
    GPIO_PORTF[GPIO_DIR] |= LED2_PF3;
    GPIO_PORTG[GPIO_DIR] |= LED3_PG0;
    GPIO_PORTF[GPIO_DEN] |= LED1_PF2;
    GPIO_PORTF[GPIO_DEN] |= LED2_PF3;
    GPIO_PORTG[GPIO_DEN] |= LED3_PG0;

   /* PWM0[PWM_CC] |= ((1 << 8) | 0x0);
    PWM0[PWM1_CTRL] &= ~0xFFFFFFFF;
    PWM0[PWM2_CTRL] &= ~0xFFFFFFFF;
    PWM0[PWM1_GENA] = 0x08C;
    PWM0[PWM2_GENA] = 0x08C;
    PWM0[PWM1_GENB] = 0x80C;
    PWM0[PWM2_GENB] = 0x80C;
    PWM0[PWM1_LOAD] = 0xA2C2;
    PWM0[PWM2_LOAD] = 0xA2C2;
    PWM0[PWM1_CMPA] = 0x927B;
    PWM0[PWM2_CMPA] = 0x927B;
    PWM0[PWM1_CMPB] = 0x1046;
    PWM0[PWM2_CMPB] = 0x1046;
    PWM0[PWM1_CTRL] = 0x1;
    PWM0[PWM2_CTRL] = 0x1;
    PWM0[PWM_ENABLE] |= 0x4;
    //PWM0[PWM_ENABLE] |= 0x8;
    //PWM0[PWM_ENABLE] |= 0x10;*/

}

uint8_t static writeCMD(uint8_t c){
    while ((SSI2[SSI_SR] & 0x10) == 0x10){};
    TFT_CS = TFT_CS_LOW;
    DC = DC_COMMAND;
    SSI2[SSI_DR] = c;
    while ((SSI2[SSI_SR] & 0x4) == 0){};
    TFT_CS = TFT_CS_HIGH;
    return (uint8_t) SSI2[SSI_DR];
}

uint8_t static writeData(uint8_t c){
    while ((SSI2[SSI_SR] & 0x10) == 0x10){};
    TFT_CS = TFT_CS_LOW;
    DC = DC_DATA;
    SSI2[SSI_DR] = c;
    while ((SSI2[SSI_SR] & 0x4) == 0){};
    TFT_CS = TFT_CS_HIGH;
    return (uint8_t) SSI2[SSI_DR];
}

void static deselect(void){
    while ((SSI2[SSI_SR] & 0x10) == 0x10){};
    TFT_CS = TFT_CS_HIGH;
}

void static setAddr(uint8_t x0, uint8_t y0, uint8_t x1, uint8_t y1){
    writeCMD(CASET);
    writeData(0x00);
    writeData(x0+ColStart);
    writeData(0x00);
    writeData(x1+ColStart);

    writeCMD(RASET);
    writeData(0x00);
    writeData(y0+RowStart);
    writeData(0x00);
    writeData(y1+RowStart);

    writeCMD(RAMWR);
}

void static commandList(const uint8_t *addr){
    uint8_t numCommand, numArgs;
    uint16_t ms;

    numCommand = *(addr++);
    while (numCommand--){
        writeCMD(*(addr++));
        numArgs = *(addr++);
        ms = numArgs & DELAY;
        numArgs &= ~DELAY;
        while (numArgs--)
            writeData(*(addr++));

        if (ms){
            ms = *(addr++);
            ms = (ms == 255) ? 500:ms;
            for (int i = 0; i < ms; i++)
                while((SysTick[STCTRL] & 0x10000) == 0){}
        }
    }
}

void setCursor(uint32_t newX, uint32_t newY){
    if((newX > 20) || (newY > 12))
        return;

    StX = newX;
    StY = newY;
}

void fillScreen(int16_t x, int16_t y, int16_t w, int16_t h, uint16_t color){
    uint8_t hi = color >> 8, lo = color;

    if((x >= width) || (y >= height)) return;
    if((x + w - 1) >= width)  w = width  - x;
    if((y + h - 1) >= height) h = height - y;

    setAddr(x, y, x+w-1, y+h-1);

    for(y=h; y>0; y--){
        for(x=w; x>0; x--){
            writeData(hi);
            writeData(lo);
        }
    }

    deselect();
}

void static commonInit(const uint8_t *cmdList){
    ColStart = RowStart = 0;

    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTD;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTH;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTL;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTN;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTD;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTH;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTL;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTN;
    SYSCTL[SYSCTL_RCGCSSI] |= 0x4;

    GPIO_PORTD[GPIO_DEN] |= SPICl;
    GPIO_PORTD[GPIO_DEN] |= SPIMOSI;
    GPIO_PORTH[GPIO_DEN] |= SPIRST;
    GPIO_PORTL[GPIO_DEN] |= SPIREG;
    GPIO_PORTN[GPIO_DEN] |= SPICS;

    GPIO_PORTH[GPIO_DIR] |= SPIRST;
    GPIO_PORTL[GPIO_DIR] |= SPIREG;
    GPIO_PORTN[GPIO_DIR] |= SPICS;

    GPIO_PORTD[GPIO_FSEL] |= 0xA;

    GPIO_PORTD[GPIO_PCTL] |= 0xF0F0;

    TFT_CS = TFT_CS_LOW;

    RESET = RESET_HIGH;
    for (int i = 0; i < 500; i++)
        while((SysTick[STCTRL] & 0x10000) == 0){}
    RESET = RESET_LOW;
    for (int i = 0; i < 500; i++)
        while((SysTick[STCTRL] & 0x10000) == 0){}
    RESET = RESET_HIGH;
    for (int i = 0; i < 500; i++)
        while((SysTick[STCTRL] & 0x10000) == 0){}

    TFT_CS = TFT_CS_HIGH;

    SSI2[SSI_CR1] &= ~0x2;
    SSI2[SSI_CR1] &= ~0x4;
    SSI2[SSI_CC] |= 0x5;
    SSI2[SSI_PS] = 8;
    SSI2[SSI_CR0] &= ~(0xFF00 | 0x80 | 0x40);
    SSI2[SSI_CR0] &= ~0x30;
    SSI2[SSI_CR0] = 0x7;
    SSI2[SSI_CR1] |= 0x2;

    if (cmdList)
        commandList(cmdList);
}

void static initS(enum initSFlags option){
    commonInit(Scmd1);

    if (option == INITS_GREENTAB){
        commandList(Scmd2green);
        ColStart = 2;
        RowStart = 3;
    }else
        commandList(Scmd2red);

    commandList(Scmd3);

    if (option == INITS_BLACKTAB) {
      writeCMD(MADCTL);
      writeData(0xC0);
      ColStart = 2;
      RowStart = 3;
    }

    TabColor = option;
    setCursor(0,0);
    StTextColor = YELLOW;
    fillScreen(0,0,width,height,0);
}

void setRot(uint8_t m){
    writeCMD(MADCTL);
    rot = m % 4;

    switch(rot){
    case 0:
        if (TabColor == INITS_BLACKTAB)
            writeData(MADCTL_MX | MADCTL_MY| MADCTL_RGB);
        else
            writeData(MADCTL_MX | MADCTL_MY| MADCTL_BGR);
        width = WIDTH;
        height = HEIGHT;
        break;
    case 1:
        if (TabColor == INITS_BLACKTAB)
            writeData(MADCTL_MY | MADCTL_MV| MADCTL_RGB);
        else
            writeData(MADCTL_MY | MADCTL_MV| MADCTL_BGR);
        width = HEIGHT;
        height = WIDTH;
        break;
    case 2:
        if (TabColor == INITS_BLACKTAB)
            writeData(MADCTL_RGB);
        else
            writeData(MADCTL_BGR);
        width = WIDTH;
        height = HEIGHT;
        break;
    case 3:
        if (TabColor == INITS_BLACKTAB)
            writeData(MADCTL_MX | MADCTL_MV| MADCTL_RGB);
        else
            writeData(MADCTL_MX | MADCTL_MV| MADCTL_BGR);
        width = HEIGHT;
        height = WIDTH;
        break;
    }
    deselect();
}

void invDisp(bool i){
    if (i)
        writeCMD(INVON);
    else
        writeCMD(INVOFF);
    deselect();
}

void static pushColor(uint16_t color){
    writeData((uint8_t)(color >> 8));
    writeData((uint8_t)color);
}

void drawPixel(int16_t x, int16_t y, uint16_t color){
    if (x < 0 || x >= width || y < 0 || y >= height) return;
    setAddr(x,y,x+1,y+1);
    pushColor(color);
    deselect();
}

void drawVert(int16_t x, int16_t y, int16_t h, uint16_t color) {
  uint8_t hi = color >> 8, lo = color;

  if(x >= width || y >= height) return;
  if((y+h-1) >= height) h = height-y;
  setAddr(x, y, x, y+h-1);

  while (h--) {
    writeData(hi);
    writeData(lo);
  }

  deselect();
}

void drawHor(int16_t x, int16_t y, int16_t w, uint16_t color) {
  uint8_t hi = color >> 8, lo = color;

  if((x >= width) || (y >= height)) return;
  if((x+w-1) >= width)  w = width-x;
  setAddr(x, y, x+w-1, y);

  while (w--) {
    writeData(hi);
    writeData(lo);
  }

  deselect();
}

uint16_t color565(uint8_t r, uint8_t g, uint8_t b) {
  return ((b & 0xF8) << 8) | ((g & 0xFC) << 3) | (r >> 3);
}

uint16_t swapColor(uint16_t x) {
  return (x << 11) | (x & 0x07E0) | (x >> 11);
}

void drawBMP(int16_t x, int16_t y, const uint16_t *image, int16_t w, int16_t h){
    int16_t skipC = 0;
    int16_t originalWidth = w;
    int i = w*(h - 1);

    if((x >= width) || ((y - h + 1) >= height) || ((x + w) <= 0) || (y < 0)) return;

    if (w > width || h > height) return;

    if((x + w - 1) >= width){
        skipC = (x + w) - width;
        w = width - x;
    }
    if((y - h + 1) < 0){
        i -= (h - y - 1)*originalWidth;
        h = y + 1;
    }
    if(x < 0){
        w += x;
        skipC = -1*x;
        i -= x;
        x = 0;
    }
    if(y >= height){
      h -= (y - height + 1);
      y = height - 1;
    }

    setAddr(x, y-h+1, x+w-1, y);

    for(y=0; y<h; y++){
      for(x=0; x<w; x++){
        writeData((uint8_t)(image[i] >> 8));
        writeData((uint8_t)image[i]);
        i++;
      }
      i += skipC;
      i -= 2*originalWidth;
    }

    deselect();
}

void drawChar(int16_t x, int16_t y, char c, int16_t textColor, int16_t bgColor, uint8_t size){
    uint8_t line;
    int32_t col, row, i, j;

    if(((x + 6*size - 1) >= width)  || ((y + 8*size - 1) >= height) || ((x + 6*size - 1) < 0) || ((y + 8*size - 1) < 0)) return;

    setAddr(x, y, x+6*size-1, y+8*size-1);

    line = 0x1;

    for(row=0; row<8; row++){
        for(i=0; i<size; i++){
            for(col=0; col<5; col++){
                if(Font[(c*5)+col]&line){
                    for(j=0; j<size; j++)
                        pushColor(textColor);
                }else{
                    for(j=0; j<size; j++)
                        pushColor(bgColor);
                }
            }
            for(j=0; j<size; j++)
                pushColor(bgColor);
            }
        line <<= 1;
    }

    deselect();
}

uint32_t drawString(uint16_t x, uint16_t y, char *pt, int16_t textColor){
    uint32_t count = 0;

    if(y>12) return 0;

    while(*pt){
        drawChar(x*6, y*10, *pt, textColor, BLACK, 1);
        pt++;
        x++;
        if(x>20) return count;
        count++;
    }
    return count;
}

void fillMessage(uint32_t n){
    if(n >= 10){
        fillMessage(n/10);
        n %= 10;
    }
    Message[Messageindex] = (n+'0');
    if(Messageindex<11) Messageindex++;
}

void fillMessage4(uint32_t n){
    if(n>9999) n=9999;

    if(n>=1000)
        Messageindex = 0;
    else if(n>=100){
        Message[0] = ' ';
        Messageindex = 1;
    }else if(n>=10){
        Message[0] = ' ';
        Message[1] = ' ';
        Messageindex = 2;
    }else{
        Message[0] = ' ';
        Message[1] = ' ';
        Message[2] = ' ';
        Messageindex = 3;
    }
    fillMessage(n);
}

void fillMessage5(uint32_t n){
    if(n>99999) n=99999;

    if(n>=10000)
        Messageindex = 0;
    else if(n>=1000){
        Message[0] = ' ';
        Messageindex = 1;
    }else if(n>=100){
        Message[0] = ' ';
        Message[1] = ' ';
        Messageindex = 2;
    }else if(n>=10){
        Message[0] = ' ';
        Message[1] = ' ';
        Message[2] = ' ';
        Messageindex = 3;
    }else{
        Message[0] = ' ';
        Message[1] = ' ';
        Message[2] = ' ';
        Message[3] = ' ';
        Messageindex = 4;
    }
  fillMessage(n);
}

void static fillMessage2_10(uint32_t n){
    if(n>999) n=999;

    if(n>=100){
        Message[0] = (n/100+'0');
        n %= 100;
    }else
        Message[0] = ' ';

    Message[1] = (n/10+'0');
    n %= 10;
    Message[2] = '.';
    Message[3] = (n+'0');
    Message[4] = 0;
}
void static fillMessage2_Hex(uint32_t n){
    char digit;

    if(n>255){
        Message[0] = '*';
        Message[1] = '*';
    }else{
        digit = n/16;
        if(digit<10)
            digit += '0';
        else
            digit = digit+'A'-10;
        Message[0] = digit;
        digit = n%16;
        if(digit<10)
            digit += '0';
        else
            digit += 'A'-10;
        Message[1] = digit;
    }
    Message[2] = ',';
    Message[3] = 0;
}

void outUDec(uint32_t n, int16_t textColor){
    Messageindex = 0;
    fillMessage(n);
    Message[Messageindex] = 0;
    drawString(StX,StY,Message,textColor);
    StX += Messageindex;
    if(StX>20){
        StX = 20;
        drawChar(StX*6,StY*10,'*',RED,BLACK, 1);
    }
}

void outUDec4(uint32_t n, int16_t textColor){
    Messageindex = 0;
    fillMessage4(n);
    Message[Messageindex] = 0;
    drawString(StX,StY,Message,textColor);
    StX += Messageindex;
    if(StX>20){
        StX = 20;
        drawChar(StX*6,StY*10,'*',RED,BLACK, 1);
    }
}

void outUDec5(uint32_t n, int16_t textColor){
    Messageindex = 0;
    fillMessage5(n);
    Message[Messageindex] = 0;
    drawString(StX,StY,Message,textColor);
    StX += Messageindex;
    if(StX>20){
        StX = 20;
        drawChar(StX*6,StY*10,'*',RED,BLACK, 1);
    }
}

void outUFix2_10(uint32_t n, int16_t textColor){
    fillMessage2_10(n);
    drawString(StX,StY,Message,textColor);
    StX += 4;
    if(StX>20){
        StX = 20;
        drawChar(StX*6,StY*10,'*',RED,BLACK, 1);
    }
}

void outUHex2(uint32_t n, int16_t textColor){
    fillMessage2_Hex(n);
    drawString(StX,StY,Message,textColor);
    StX += 3;
    if(StX>20){
        StX = 20;
        drawChar(StX*6,StY*10,'*',RED,BLACK, 1);
    }
}

void drawAxes(uint16_t axisColor, uint16_t bgColor, char *xLabel, char *yLabel1, uint16_t label1Color, char *yLabel2, uint16_t label2Color, int32_t ymax, int32_t ymin){
    int i;

    Ymax = ymax;
    Ymin = ymin;
    Yrange = Ymax - Ymin;
    TimeIndex = 0;
    PlotBGColor = bgColor;
    fillScreen(0, 17, 111, 111, bgColor);
    drawHor(10, 117, 101, axisColor);
    drawVert(10, 17, 101, axisColor);

    for(i=20; i<=110; i=i+10)
        drawPixel(i, 118, axisColor);
    for(i=17; i<117; i=i+10)
        drawPixel(9, i, axisColor);

    i = 50;
    while((*xLabel) && (i < 100)){
        drawChar(i, 120, *xLabel, axisColor, bgColor, 1);
        i += 6;
        xLabel++;
    }
    if(*yLabel2){
        i = 26;
        while((*yLabel2) && (i < 50)){
          drawChar(0, i, *yLabel2, label2Color, bgColor, 1);
          i = i + 8;
          yLabel2++;
        }
        i = 82;
    }else
        i = 42;

    while((*yLabel1) && (i < 120)){
        drawChar(0, i, *yLabel1, label1Color, bgColor, 1);
        i += 8;
        yLabel1++;
    }
}

void plotPoint(int32_t data1, uint16_t color1){
    data1 = ((data1 - Ymin)*100)/Yrange;
    if(data1 > 98){
        data1 = 98;
        color1 = RED;
    }
    if(data1 < 0){
        data1 = 0;
        color1 = RED;
    }
    drawPixel(TimeIndex + 11, 116 - data1, color1);
    drawPixel(TimeIndex + 11, 115 - data1, color1);
}

void plotInc(void){
    TimeIndex++;
    if(TimeIndex > 99){
    TimeIndex = 0;
    }
    drawVert(TimeIndex + 11, 17, 100, PlotBGColor);
}

void plotClear(int32_t ymin, int32_t ymax){
    fillScreen(0,28,100, 100, color565(228,228,228));

    if(ymax>ymin){
      Ymax = ymax;
      Ymin = ymin;
      Yrange = ymax-ymin;
    } else{
      Ymax = ymin;
      Ymin = ymax;
      Yrange = ymax-ymin;
    }
    //YrangeDiv2 = Yrange/2;
    X = 0;
}

void plotNext(void){
    if (X == 127)
        X = 0;
    else
        X++;
}

void plotNextErase(void){
    if (X == 127)
        X = 0;
    else
        X++;

    drawVert(X, 32, 128, color565(228,228,228));
}
