#include <stdint.h>
#include <stdbool.h>
#include "..\BSP\BSP.h"
#include "..\pll\pll.h"

#define THREADFREQ 1000
#define NUMTHREADS 4
#define STACKSIZE 100
#define SOUNDRMSLENGTH 1000

#define BGCOLOR     BLACK
#define AXISCOLOR   ORANGE
#define MAGCOLOR    YELLOW
#define EWMACOLOR   CYAN
#define SOUNDCOLOR  CYAN
#define TEMPCOLOR   LIGHTGREEN
#define TOPTXTCOLOR WHITE
#define TOPNUMCOLOR ORANGE

uint32_t time;
uint32_t steps;
uint32_t mag;
uint32_t EWMA;
uint32_t soundData;
int32_t soundAvg;
uint32_t light;
int32_t new;
int32_t mutex;
int32_t mail;
uint32_t mailData = 0;
int redraw = false;

int16_t SoundArray[SOUNDRMSLENGTH];

struct tcb{
    int32_t *sp;
    struct tcb *next;
};
typedef struct tcb tcbType;

tcbType tcbs[NUMTHREADS];
tcbType *RunPt;
int32_t stacks[NUMTHREADS][STACKSIZE];

enum plotstate{
    Accel,
    Mic,
    Light
};

enum plotstate Plotstate = Accel;

enum {
  PRI3 = (0xD20 >> 0)
};

extern void disInt(void);
extern void enInt(void);
extern uint32_t startCrit(void);
extern void endCrit(uint32_t);
extern void startOS(void);

#define toggle0() GPIO_PORTN[GPIO_PIN_1] ^= GPIO_PIN_1
#define toggle1() GPIO_PORTN[GPIO_PIN_0] ^= GPIO_PIN_0
#define toggle2() GPIO_PORTF[GPIO_PIN_4] ^= GPIO_PIN_4
#define toggle3() GPIO_PORTF[GPIO_PIN_0] ^= GPIO_PIN_0

void toggleInit(void){
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTF;
    SYSCTL[SYSCTL_RCGCGPIO] |= SYSCTL_RCGCGPIO_PORTF;

    GPIO_PORTN[GPIO_DEN] |= GPIO_PIN_0 | GPIO_PIN_1;
    GPIO_PORTF[GPIO_DEN] |= GPIO_PIN_0 | GPIO_PIN_4;

    GPIO_PORTN[GPIO_DIR] |= GPIO_PIN_1;
    GPIO_PORTN[GPIO_DIR] |= GPIO_PIN_0;
    GPIO_PORTF[GPIO_DIR] |= GPIO_PIN_4;
    GPIO_PORTF[GPIO_DIR] |= GPIO_PIN_0;
}

void OS_init(void){
    disInt();
    pll();
}

void SetInitialStack(int i){
  tcbs[i].sp = &stacks[i][STACKSIZE-16]; // thread stack pointer
  stacks[i][STACKSIZE-1] = 0x01000000;   // thumb bit
  stacks[i][STACKSIZE-3] = 0x14141414;   // R14
  stacks[i][STACKSIZE-4] = 0x12121212;   // R12
  stacks[i][STACKSIZE-5] = 0x03030303;   // R3
  stacks[i][STACKSIZE-6] = 0x02020202;   // R2
  stacks[i][STACKSIZE-7] = 0x01010101;   // R1
  stacks[i][STACKSIZE-8] = 0x00000000;   // R0
  stacks[i][STACKSIZE-9] = 0x11111111;   // R11
  stacks[i][STACKSIZE-10] = 0x10101010;  // R10
  stacks[i][STACKSIZE-11] = 0x09090909;  // R9
  stacks[i][STACKSIZE-12] = 0x08080808;  // R8
  stacks[i][STACKSIZE-13] = 0x07070707;  // R7
  stacks[i][STACKSIZE-14] = 0x06060606;  // R6
  stacks[i][STACKSIZE-15] = 0x05050505;  // R5
  stacks[i][STACKSIZE-16] = 0x04040404;  // R4
}

int OS_AddThreads(void(*thread0)(void), void(*thread1)(void), void(*thread2)(void), void(*thread3)(void)){
  int32_t status;

  status = startCrit();
  tcbs[0].next = &tcbs[1]; // 0 points to 1
  tcbs[1].next = &tcbs[2]; // 1 points to 2
  tcbs[2].next = &tcbs[3]; // 2 points to 3
  tcbs[3].next = &tcbs[0]; // 3 points to 0
  SetInitialStack(0); stacks[0][STACKSIZE-2] = (int32_t)(thread0); // PC
  SetInitialStack(1); stacks[1][STACKSIZE-2] = (int32_t)(thread1); // PC
  SetInitialStack(2); stacks[2][STACKSIZE-2] = (int32_t)(thread2); // PC
  SetInitialStack(3); stacks[3][STACKSIZE-2] = (int32_t)(thread3); // PC
  RunPt = &tcbs[0];       // thread 0 will run first
  endCrit(status);

  return 1;               // successful
}

void OS_Launch(uint32_t timeSlice){
    SysTick[STCTRL] = 0;
    SysTick[STCURRENT] = 0;
    SysTick[STRELOAD] = timeSlice;
    SysTick[PRI3] = 0xE0000000;
    SysTick[STCTRL] = 0x7;
    startOS();
}

void scheduler(void){
    RunPt = RunPt -> next;
}

void OS_initSem(int32_t *semPt, int32_t value){
    int32_t status = startCrit();
    *semPt = value;
    endCrit(status);
}

void OS_wait(int32_t *semPt){
    int32_t status;

    disInt();
    do{
        status == startCrit();
        enInt();
        disInt();
        endCrit(status);
    }while(*semPt == 0);
    status = startCrit();
    (*semPt)--;
    endCrit(status);
    enInt();
}

void OS_sig(int32_t *semPt){
    int32_t status;

    status = startCrit();
    disInt();
    (*semPt)++;
    enInt();
    endCrit(status);
}

void OS_mailInit(void){
    OS_initSem(&mail,0);
}

void OS_mailSend(uint32_t data){
    uint32_t status;

    status = startCrit();
    mailData = data;
    endCrit(status);
    OS_sig(&mail);
}

uint32_t OS_mailRecv(void){
    uint32_t data, status;

    OS_wait(&mail);
    status = startCrit();
    data = mailData;
    endCrit(status);

    return data;
}

uint32_t sqrt32(uint32_t s){
    uint32_t t;
    int n;

    t = s/16 + 1;
    for (n = 16; n; --n)
        t = ((t*t+s)/t)/2;

    return t;
}

void task0_Init(void){
    joyInit();
    joyIn(&joyX, &joyY, &push);
    mag = sqrt32(joyX * joyX + joyY * joyY);
}

void task0(void){
    uint32_t squared;

    toggle0();
    joyIn(&joyX, &joyY, &push);
    squared = joyX * joyX + joyY * joyY;
    OS_mailSend(squared);
    time++;
}

void task1_Init(void){
    accInit();
    accIn(&AccX, &AccY, &AccZ);
    mag = sqrt32(AccX*AccX + AccY*AccY + AccZ*AccZ);
}

void task1(void){
    uint32_t squared;

    toggle1();
    accIn(&AccX, &AccY, &AccZ);
    squared = AccX*AccX + AccY*AccY + AccZ*AccZ;
    OS_mailSend(squared);
    time++;
}

void task2(void){
    while(1){
        toggle2();
        buzzerInit();
    }
}

void task3(void){
    while(1){
        toggle3();
        RGBInit();
    }
}

void task4(void){
    while(1){
        drawString(0, 5, "AccX=     ", color565(145,26,91));
        setCursor(5,0);
        outUDec((uint32_t) AccX, color565(255,0,255));
        drawString(0, 6, "AccY=     ", color565(145,26,91));
        setCursor(5,1);
        outUDec((uint32_t) AccY, color565(255,0,255));
        drawString(0, 7, "AccZ=     ", color565(145,26,91));
        setCursor(5,2);
        outUDec((uint32_t) AccZ, color565(255,0,255));
    }
}

void task5(void){
    while(1){
        drawString(0, 3, "JoyX=     ", color565(200,65,8));
        setCursor(5,3);
        outUDec((uint32_t) joyX, color565(255,0,255));
        drawString(0, 4, "JoyY=     ", color565(200,65,8));
        setCursor(5,4);
        outUDec((uint32_t) joyY, color565(255,0,255));
    }
}

void main(void)
{
    OS_init();
    sysctlInit();
    initS(INITS_BLACKTAB);
    toggleInit();
    task0_Init();
    task1_Init();
    time = 0;

    OS_AddThreads(&task2,&task3,&task4,&task5);
    OS_Launch(120000000/THREADFREQ);
}
