/* vim:ts=4:sw=4:expandtab
 * (No tabs, indent level is 4 spaces)  */
/*****************************************************************************
 * File:            usb_dev_serial.c
 * Creation Date:   2020.08.04
 * Description:     USB/CAN transparent bridge.
 *                  USB0 <=> CAN1
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of provider nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *****************************************************************************/


#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_gpio.h"
#include "inc/hw_uart.h"
#include "inc/hw_sysctl.h"
#include "driverlib/debug.h"
#include "driverlib/fpu.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
/* Define MCU type */
#define PART_TM4C123GH6ZRB      1
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/systick.h"
#include "driverlib/timer.h"
#include "driverlib/uart.h"
#include "inc/hw_can.h"
#include "driverlib/can.h"
#include "driverlib/usb.h"
#include "usblib/usblib.h"
#include "usblib/usbcdc.h"
#include "usblib/usb-ids.h"
#include "usblib/device/usbdevice.h"
#include "usblib/device/usbdcdc.h"
#include "utils/ustdlib.h"
#include "usb_serial_structs.h"
#include "utils/uartstdio.h"

/* MAX data length for CAN frame */
#define MAX_CAN_LEN             8

/* Message Identifiers and Objects
 * RXID is set to 0 so all messages are received
 */
#define CAN0RXID                0
#define RXOBJECT                1
#define CAN0TXID                2
#define TXOBJECT                2

/* Define the can frame header byte */
#define CAN_HEAD                0xad

/* Linux CAN frame structure */
typedef struct can_frame
{
    /* 32 bit CAN_ID + EFF/RTR/ERR flags */
    uint32_t   can_id;

    /* frame payload length in byte (0 .. CAN_MAX_DLEN) */
    uint32_t   can_dlc;

    /* Data buffer, 8 bytes for the most */
    uint8_t    data[MAX_CAN_LEN] __attribute__((aligned(8)));

} can_frame_t;

/* CAN message Objects for data being sent / received */
static tCANMsgObject g_sCAN1RxMessage;
static tCANMsgObject g_sCAN1TxMessage;
static can_frame_t   g_slinux_cf_rx_1;
static can_frame_t   g_slinux_cf_tx_1;
uint8_t              *cf_pointer_rx_1 = (uint8_t *)(&g_slinux_cf_rx_1);
uint8_t              *cf_pointer_tx_1 = (uint8_t *)(&g_slinux_cf_tx_1);

/* A flag for the interrupt handler to indicate that a message was received */
volatile bool        g_bRXFlag_1 = 0;

/* A flag to keep track of the errors that have been thrown so they may
 * be processed. This is necessary because reading the error register clears
 * the flags, so it is necessary to save them somewhere for processing.
 */
volatile uint32_t    g_ui32ErrFlag_1 = 0;

/* The system tick rate expressed both as ticks per second
 * and a millisecond period
 */
#define SYSTICKS_PER_SECOND 100
#define SYSTICK_PERIOD_MS (1000 / SYSTICKS_PER_SECOND)

/* Global system tick counter */
volatile uint32_t    g_ui32SysTickCount = 0;

/* Flags used to pass commands from interrupt context to the main loop */
#define COMMAND_PACKET_RECEIVED 0x00000001
#define COMMAND_STATUS_UPDATE   0x00000002

/* The usb connected status flag */
volatile uint32_t    g_ui32Flags = 0;
char                 *g_pcStatus = NULL;

/* Global flag indicating that a USB configuration has been set */
static volatile bool g_bUSBConfigured = false;

/* Internal function prototypes */
void USBRecv(void);
void USBSend(const uint8_t *pui8Buffer, uint32_t ui32Count);

/*****************************************************************************/
/** ControlHandler
 *
 * \desc            Handles CDC driver notifications related to control
 *                  and setup of the device.
 *
 * \param[in]       pvCBData, the client-supplied callback pointer for
 *                            this channel.
 *
 * \param[in]       ui32Event, identifies the event we are being notified about.
 *
 *
 * \param[in]       ui32MsgValue, an event-specific value.
 *
 *
 * \param[in]       pvMsgData, an event-specific pointer.
 *
 * \return          The return value is event-specific.
 *
 *****************************************************************************/
uint32_t ControlHandler(void *pvCBData, uint32_t ui32Event,
               uint32_t ui32MsgValue, void *pvMsgData)
{
    uint32_t ui32IntsOff = false;

    pvCBData = pvCBData;
    ui32MsgValue = ui32MsgValue;
    pvMsgData = pvMsgData;

    /* The event being asked to process. */
    switch (ui32Event)
    {
        /* We are connected to a host and communication is now possible. */
        case USB_EVENT_CONNECTED:
            g_bUSBConfigured = true;

            /* Flush our buffers. */
            USBBufferFlush(&g_sTxBuffer);
            USBBufferFlush(&g_sRxBuffer);

            /* Tell the main loop to update the display. */
            ui32IntsOff = MAP_IntMasterDisable();
            g_pcStatus = "Connected";
            g_ui32Flags |= COMMAND_STATUS_UPDATE;
            if (!ui32IntsOff)
            {
                MAP_IntMasterEnable();
            }

            break;

        /* The host has disconnected. */
        case USB_EVENT_DISCONNECTED:
            g_bUSBConfigured = false;
            ui32IntsOff = MAP_IntMasterDisable();
            g_pcStatus = "Disconnected";
            g_ui32Flags |= COMMAND_STATUS_UPDATE;
            if (!ui32IntsOff)
            {
                MAP_IntMasterEnable();
            }

            break;

        /* Return the current serial communication parameters. */
        case USBD_CDC_EVENT_GET_LINE_CODING:
            break;

        /* Set the current serial communication parameters. */
        case USBD_CDC_EVENT_SET_LINE_CODING:
            break;

        /* Set the current serial communication parameters. */
        case USBD_CDC_EVENT_SET_CONTROL_LINE_STATE:
            break;

        /* Send a break condition on the serial line. */
        case USBD_CDC_EVENT_SEND_BREAK:
            break;

        /* Clear the break condition on the serial line. */
        case USBD_CDC_EVENT_CLEAR_BREAK:
            break;

        /* Ignore SUSPEND and RESUME for now. */
        case USB_EVENT_SUSPEND:
        case USB_EVENT_RESUME:
            break;

        /* We don't expect to receive any other events. Ignore any that show
         * up in a release build or hang in a debug build.
         */
        default:
            break;
    }

    return(0);
}

/*****************************************************************************/
/** TxHandler
 *
 * \desc            Handles CDC driver notifications related to the transmit
 *                  channel (data to the USB host).
 *
 * \param[in]       ui32CBData, the client-supplied callback pointer for
 *                              this channel.
 *
 * \param[in]       ui32Event, identifies the event we are being notified about.
 *
 *
 * \param[in]       ui32MsgValue, an event-specific value.
 *
 *
 * \param[in]       pvMsgData, an event-specific pointer.
 *
 * \return          The return value is event-specific.
 *
 *****************************************************************************/
uint32_t TxHandler(void *pvCBData, uint32_t ui32Event, uint32_t ui32MsgValue,
          void *pvMsgData)
{
    pvCBData = pvCBData;
    ui32MsgValue = ui32MsgValue;
    pvMsgData = pvMsgData;

    /* The event we have been sent */
    switch (ui32Event)
    {
        case USB_EVENT_TX_COMPLETE:
            /* Since we are using the USBBuffer, we don't need
             * to do anything here.
             */
            break;

        /* We don't expect to receive any other events.  Ignore any that show
         * up in a release build or hang in a debug build.
         */
        default:
            break;
    }

    return(0);
}

/*****************************************************************************/
/** RxHandler
 *
 * \desc            Handles CDC driver notifications related to the receive
 *                  channel (data from the USB host).
 *
 * \param[in]       ui32CBData, the client-supplied callback data value
 *                              for this channel.
 *
 * \param[in]       ui32Event, identifies the event we are being notified about.
 *
 *
 * \param[in]       ui32MsgValue, an event-specific value.
 *
 *
 * \param[in]       pvMsgData, an event-specific pointer.
 *
 * \return          The return value is event-specific.
 *
 *****************************************************************************/
uint32_t RxHandler(void *pvCBData, uint32_t ui32Event, uint32_t ui32MsgValue,
          void *pvMsgData)
{
    pvCBData = pvCBData;
    ui32MsgValue = ui32MsgValue;
    pvMsgData = pvMsgData;

    /* The event we have been sent */
    switch (ui32Event)
    {
        /* A new packet has been received. */
        case USB_EVENT_RX_AVAILABLE:
            /* Handle the new packet */
            USBRecv();
            break;


        /* We are being asked how much unprocessed data we have still to
         * process. We return 0 if the UART is currently idle or 1 if it is
         * in the process of transmitting something. The actual number of
         * bytes in the UART FIFO is not important here, merely whether or
         * not everything previously sent to us has been transmitted.
         */
        case USB_EVENT_DATA_REMAINING:
            break;

        /* We are being asked to provide a buffer into which the next packet
         * can be read. We do not support this mode of receiving data so let
         * the driver know by returning 0. The CDC driver should not be sending
         * this message but this is included just for illustration and
         * completeness.
         */
        case USB_EVENT_REQUEST_BUFFER:
            break;

        /* We don't expect to receive any other events.  Ignore any that show
         * up in a release build or hang in a debug build.
         */
        default:
            break;
    }

    return(0);
}

/*****************************************************************************/
/** SysTickIntHandler
 *
 * \desc            Interrupt handler for the system tick counter.
 *
 * \param[in]       void
 *
 * \return          void
 *
 *****************************************************************************/
void SysTickIntHandler(void)
{
    /* Update our system time. */
    g_ui32SysTickCount++;
}

uint32_t fcount = 0;
/*****************************************************************************/
/** USBRecv
 *
 * \desc            The usb data received handler.
 *
 * \param[in]       void
 *
 * \return          void
 *
 *****************************************************************************/
void USBRecv(void)
{
    static uint32_t      usb_rx_cnt = 0;
    static uint8_t       ui8Char = 0;
    static uint8_t       hdchk[2] = {0};
    static uint32_t      hdchk_cnt = 0;
    static uint32_t      hdchk_flag = 1;

    /* Loop while there are characters in the receive FIFO */
    while (USBBufferDataAvailable((tUSBBuffer *)&g_sRxBuffer))
    {
        if (hdchk_flag)
        {
            /* Get a character from the buffer. */
            USBBufferRead((tUSBBuffer *)&g_sRxBuffer, &ui8Char, 1);
            hdchk[hdchk_cnt++] = ui8Char;

            if (hdchk_cnt >= 2)
            {
                /* The can frame head has been detected */
                if ((hdchk[0] == CAN_HEAD) && ((hdchk[1] == CAN_HEAD)))
                {
                    hdchk_flag = 0;

                    /* Clear the can frame head flag and
                     * continue read the data
                     */
                    continue;
                }
                else if ((hdchk[0] != CAN_HEAD) && ((hdchk[1] == CAN_HEAD)))
                {
                    /* Only the second byte is received, fix
                     * the first byte and detect again
                     */
                    hdchk[0] = CAN_HEAD;
                    hdchk_cnt = 1;
                    continue;
                }
                else
                {
                    /* This frame is not can frame and
                     * continue to detect
                     */
                    hdchk_cnt = 0;
                    continue;
                }
            }

            /* Continue to receive the second byte */
            continue;
        }

        /* Loop while there are characters in the receive FIFO */
        USBBufferRead((tUSBBuffer *)&g_sRxBuffer, &ui8Char, 1);
        cf_pointer_tx_1[usb_rx_cnt] = ui8Char;
        usb_rx_cnt++;

        /* Use >= instead of == to correct over-counting */
        if (usb_rx_cnt >= sizeof(g_slinux_cf_tx_1))
        {
            fcount++;
            if (fcount == 1000)
            {
                fcount = 0;
            }

            g_sCAN1TxMessage.ui32MsgID = g_slinux_cf_tx_1.can_id;
            g_sCAN1TxMessage.ui32MsgLen = g_slinux_cf_tx_1.can_dlc;

            CANMessageSet(CAN1_BASE, TXOBJECT, &g_sCAN1TxMessage,
                          MSG_OBJ_TYPE_TX);

            usb_rx_cnt = 0;
            hdchk_flag = 1;
            hdchk_cnt = 0;
        }
    }

    return;
}

/*****************************************************************************/
/** USBSend
 *
 * \desc            Send a string to USB.
 *
 * \param[in]       pui8Buffer, points to the array to be sent.
 *
 * \param[in]       ui32Count, array length.
 *
 * \return          void
 *
 *****************************************************************************/
void USBSend(const uint8_t *pui8Buffer, uint32_t ui32Count)
{
    uint8_t i = 2;
    uint8_t canhd[2] = {CAN_HEAD, CAN_HEAD};

    /* The can header to be sent */
    while (i--)
    {
        USBBufferWrite((tUSBBuffer *)&g_sTxBuffer, (uint8_t *)&canhd[i], 1);
    }

    /* Loop while there are more characters to send */
    while (ui32Count--)
    {
        USBBufferWrite((tUSBBuffer *)&g_sTxBuffer, (uint8_t *)pui8Buffer++, 1);
    }

    return;
}

/*****************************************************************************/
/** CAN1IntHandler
 *
 * \desc            CAN1 Interrupt Handler.
 *                  It checks for the cause of the interrupt, and set some
 *                  flags.
 *
 * \param[in]       void
 *
 * \return          void
 *
 *****************************************************************************/
void CAN1IntHandler(void)
{
    uint32_t ui32Status = 0;

    /* Read the CAN interrupt status to find the cause of the interrupt
     *
     * CAN_INT_STS_CAUSE register values
     * 0x0000        = No Interrupt Pending
     * 0x0001-0x0020 = Number of message object that caused the interrupt
     * 0x8000        = Status interrupt
     * all other numbers are reserved and have no meaning in this system
     */
    ui32Status = CANIntStatus(CAN1_BASE, CAN_INT_STS_CAUSE);

    /* If this was a status interrupt acknowledge it by reading the CAN
     * controller status register.
     */
    if (ui32Status == CAN_INT_INTID_STATUS)
    {
        /* Read the controller status. This will return a field of status
         * error bits that can indicate various errors. Refer to the
         * API documentation for details about the error status bits.
         * The act of reading this status will clear the interrupt.
         */
        ui32Status = CANStatusGet(CAN1_BASE, CAN_STS_CONTROL);

        /* Add ERROR flags to list of current errors. To be handled
         * later, because it would take too much time here in the
         * interrupt.
         */
        g_ui32ErrFlag_1 |= ui32Status;
    }

    /* Check if the cause is message object RXOBJECT, which we are using
     * for receiving messages.
     */
    else if (ui32Status == RXOBJECT)
    {
        /* Getting to this point means that the RX interrupt occurred on
         * message object RXOBJECT, and the message reception is complete.
         * Clear the message object interrupt.
         */
        CANIntClear(CAN1_BASE, RXOBJECT);

        /* Increment a counter to keep track of how many messages have been
         * received.  In a real application this could be used to set flags to
         * indicate when a message is received.
         */
        /*
        g_ui32RXMsgCount++;
         */

        /* Set flag to indicate received message is pending */
        g_bRXFlag_1 = true;

        /* Since a message was received, clear any error flags.
         * This is done because before the message is received it triggers
         * a Status Interrupt for RX complete. by clearing the flag here we
         * prevent unnecessary error handling from happeneing.
         */
        g_ui32ErrFlag_1 = 0;
    }

    /* Check if the cause is message object TXOBJECT, which we are using
     * for transmitting messages.
     */
    else if (ui32Status == TXOBJECT)
    {
        /* Getting to this point means that the TX interrupt occurred on
         * message object TXOBJECT, and the message reception is complete.
         * Clear the message object interrupt.
         */
        CANIntClear(CAN1_BASE, TXOBJECT);

       /* Increment a counter to keep track of how many messages have been
        * transmitted. In a real application this could be used to set
        * flags to indicate when a message is transmitted.
        */
        /*
        g_ui32TXMsgCount++;
        */

        /* Since a message was transmitted, clear any error flags.
         * This is done because before the message is transmitted it triggers
         * a Status Interrupt for TX complete. by clearing the flag here we
         * prevent unnecessary error handling from happeneing.
         */
        g_ui32ErrFlag_1 = 0;

    }

    /* Otherwise, something unexpected caused the interrupt.  This should
     * never happen.
     */
    else
    {
        /* Spurious interrupt handling can go here */
    }
}

/*****************************************************************************/
/** InitCAN
 *
 * \desc            CAN controller initialization.
 *                  In USB-CAN mode and two MCUs, only CAN1 need to
 *                  be initialized. Be caution that if hardware changed
 *                  as CAN0 <-> CAN0 or CAN0 <-> CAN1
 *
 * \param[in]       void
 *
 * \return          void
 *
 *****************************************************************************/
void InitCAN(void)
{
    /* For this example CAN1 is used with RX and TX pins on port E6 and E7.
     * GPIO port E needs to be enabled so these pins can be used.
     */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);

    /* Configure the GPIO pin muxing to select CAN1 functions for these pins.
     * This step selects which alternate function is available for these pins.
     */
    GPIOPinConfigure(GPIO_PE6_CAN1RX);
    GPIOPinConfigure(GPIO_PE7_CAN1TX);

    /* Enable the alternate function on the GPIO pins.  The above step selects
     * which alternate function is available.  This step actually enables the
     * alternate function instead of GPIO for these pins.
     */
    GPIOPinTypeCAN(GPIO_PORTE_BASE, GPIO_PIN_6 | GPIO_PIN_7);

    /* The GPIO port and pins have been set up for CAN.  The CAN peripheral
     * must be enabled.
     */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_CAN1);

    /* Initialize the CAN controller */
    CANInit(CAN1_BASE);

    /* Set up the bit rate for the CAN bus.  This function sets up the CAN
     * bus timing for a nominal configuration.  You can achieve more control
     * over the CAN bus timing by using the function CANBitTimingSet() instead
     * of this one, if needed.
     * In this example, the CAN bus is set to 500 kHz.
     */
    CANBitRateSet(CAN1_BASE, SysCtlClockGet(), 500000);

    /* Enable interrupts on the CAN peripheral.  This example uses static
     * allocation of interrupt handlers which means the name of the handler
     * is in the vector table of startup code.
     */
    CANIntEnable(CAN1_BASE, CAN_INT_MASTER | CAN_INT_ERROR | CAN_INT_STATUS);

    /* Enable the CAN interrupt on the processor (NVIC) */
    IntEnable(INT_CAN1);

    /* Enable the CAN for operation */
    CANEnable(CAN1_BASE);

    /* Initialize a message object to be used for receiving CAN messages with
     * any CAN ID.  In order to receive any CAN ID, the ID and mask must both
     * be set to 0, and the ID filter enabled.
     * A buffer for storing the received data must also be provided,
     * so set the buffer pointer within the message object.
     */
    g_sCAN1RxMessage.ui32MsgID = CAN0RXID;
    g_sCAN1RxMessage.ui32MsgIDMask = 0;
    g_sCAN1RxMessage.ui32Flags = MSG_OBJ_RX_INT_ENABLE | MSG_OBJ_USE_ID_FILTER;
    g_sCAN1RxMessage.ui32MsgLen = MAX_CAN_LEN;
    g_sCAN1RxMessage.pui8MsgData = g_slinux_cf_rx_1.data;

    /* Now load the message object into the CAN peripheral.  Once loaded the
     * CAN will receive any message on the bus, and an interrupt will occur.
     * Use message object RXOBJECT for receiving messages (this is not the
     * same as the CAN ID which can be any value).
     */
    CANMessageSet(CAN1_BASE, RXOBJECT, &g_sCAN1RxMessage, MSG_OBJ_TYPE_RX);

    /* Fixed tx data address */
    g_sCAN1TxMessage.pui8MsgData = g_slinux_cf_tx_1.data;
    g_sCAN1TxMessage.ui32MsgIDMask = 0;

    return;
}

/*****************************************************************************/
/** main
 *
 * \desc            Main function.
 *
 * \param[in]       void
 *
 * \return          void
 *
 *****************************************************************************/
int main(void)
{
    /* Initialze the can frame memory. */
    memset(&g_slinux_cf_rx_1, 0, sizeof(g_slinux_cf_rx_1));
    memset(&g_slinux_cf_tx_1, 0, sizeof(g_slinux_cf_tx_1));
    memset(&g_sCAN1RxMessage, 0, sizeof(g_sCAN1RxMessage));
    memset(&g_sCAN1TxMessage, 0, sizeof(g_sCAN1TxMessage));

    /* Enable lazy stacking for interrupt handlers.  This allows floating-point
     * instructions to be used within interrupt handlers, but at the expense of
     * extra stack usage.
     */
    MAP_FPULazyStackingEnable();

    /* Set the clocking to run from the PLL at 50MHz */
    MAP_SysCtlClockSet(SYSCTL_SYSDIV_4 | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN |
                       SYSCTL_XTAL_25MHZ);

    /* Configure the required pins for USB operation. */
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOL);
    MAP_GPIOPinTypeUSBAnalog(GPIO_PORTL_BASE, GPIO_PIN_6 | GPIO_PIN_7);

    /* Not configured initially. */
    g_bUSBConfigured = false;

    /* Enable the system tick. */
    MAP_SysTickPeriodSet(MAP_SysCtlClockGet() / SYSTICKS_PER_SECOND);
    MAP_SysTickIntEnable();
    MAP_SysTickEnable();

    /* Initialize the usb transmit and receive buffers. */
    USBBufferInit(&g_sTxBuffer);
    USBBufferInit(&g_sRxBuffer);

    /* Set the USB stack mode to Device mode with VBUS monitoring. */
    USBStackModeSet(0, eUSBModeForceDevice, 0);

    /* Initialize the CDC device. */
    USBDCDCInit(0, &g_sCDCDevice);

    /* Initialize CAN. */
    InitCAN();

    /* Main application loop. */
    while (1)
    {
        if (g_bRXFlag_1)
        {
            memset(g_slinux_cf_rx_1.data, 0, sizeof(g_slinux_cf_rx_1.data));

            /* Read the message from the CAN. Message object RXOBJECT is used
             * (which is not the same thing as CAN ID).
             */
            CANMessageGet(CAN1_BASE, RXOBJECT, &g_sCAN1RxMessage, 0);

            g_slinux_cf_rx_1.can_id = g_sCAN1RxMessage.ui32MsgID;
            g_slinux_cf_rx_1.can_dlc = g_sCAN1RxMessage.ui32MsgLen;

            USBSend(cf_pointer_rx_1, sizeof(g_slinux_cf_rx_1));

            /* Clear the pending message flag so that the interrupt handler can
             * set it again when the next message arrives.
             */
            g_bRXFlag_1 = 0;

            /* Check to see if there is an indication that some messages were
             * lost.
             */
            /*
            if(g_sCAN1RxMessage.ui32Flags & MSG_OBJ_DATA_LOST)
            {
                UARTprintf("\nCAN message loss detected\n");
            }
            */
        }
        else if (g_ui32ErrFlag_1)
        {
            memset(g_slinux_cf_rx_1.data, 0, sizeof(g_slinux_cf_rx_1.data));

            g_slinux_cf_rx_1.can_id = 0x1fffffff;
            g_slinux_cf_rx_1.can_dlc = 0;
            memcpy(g_slinux_cf_rx_1.data, (uint8_t *)(&g_ui32ErrFlag_1),
                   sizeof(g_ui32ErrFlag_1));

            USBSend(cf_pointer_rx_1, sizeof(g_slinux_cf_rx_1));

            /* Clear error flag */
            g_ui32ErrFlag_1 = 0;
        }
    }
}
