/****************************************************************************
Copyright (c) 2020 - 2028 Fujikon Industrial Co., Ltd.

FILE NAME
    i2c.c

DESCRIPTION
     I2C feature setup function

*/


#include "charging_case.h"


/*** Macro Definitions ********************************************************/
#define I2C_MAX_FIFO_ENTRIES				I2C_SYS_FENTRIES
#define NUMBER_OF_POLL_PER_OPERATION		(5)
#define POLLING_DELAY						(0.2)			// as 400KHz with 9bit per data (ACK) and 8 FIFO, so the total time should be 0.18ms.

#define I2C_USE_WHILE_FOR_POLLINGx


/*** Variables Definitions ****************************************************/


/*** Constant Variables *******************************************************/


/*** Public Global Variables **************************************************/
/* Data sent to the Target */
uint8_t gI2cTxPacket[I2C_MAX_FIFO_ENTRIES];


/*** Local Global Variables ***************************************************/


/*** Local Function Prototypes ************************************************/


/*** Local Functions **********************************************************/
bool I2C_StatusPolling( uint8_t flag, uint8_t wait4true )
{
	uint8_t pollCnt;
	bool status = FALSE;

	for ( pollCnt=0;pollCnt<NUMBER_OF_POLL_PER_OPERATION;pollCnt++ )
	{
		LoopDelayMS( POLLING_DELAY );
		if ( wait4true )
		{
			if ( DL_I2C_getControllerStatus(I2C_INST) & flag )
			{
				status = TRUE;
				break;
			}
		}
		else
		{
			if ( !(DL_I2C_getControllerStatus(I2C_INST) & flag) )
			{
				status = TRUE;
				break;
			}
		}
	}

//	I2C_DBG_HEX( pollCnt );
//	I2C_CR();
	return status;
}

bool I2C_DataPolling( void )
{
	uint8_t pollCnt;
	bool status = FALSE;

	for ( pollCnt=0;pollCnt<NUMBER_OF_POLL_PER_OPERATION;pollCnt++ )
	{
		LoopDelayMS( POLLING_DELAY );
		if ( !DL_I2C_isControllerRXFIFOEmpty(I2C_INST) )
		{
			status = TRUE;
			break;
		}
	}

//	I2C_DBG_HEX( pollCnt );
//	I2C_CR();
	return status;
}

void I2C_FlushFIFO( void )
{
	uint8_t pollCnt;

	if ( DL_I2C_getControllerTXFIFOCounter(I2C_INST) )
	{
		DL_I2C_startFlushControllerTXFIFO( I2C_INST );

		for ( pollCnt=0;pollCnt<NUMBER_OF_POLL_PER_OPERATION;pollCnt++ )
		{
			LoopDelayMS( POLLING_DELAY );
			if ( DL_I2C_getControllerTXFIFOCounter(I2C_INST) == 0 )
			{
				break;
			}
		}
		DL_I2C_stopFlushControllerTXFIFO( I2C_INST );		
	}

	if ( DL_I2C_getControllerRXFIFOCounter(I2C_INST) )
	{
		DL_I2C_startFlushControllerRXFIFO( I2C_INST );

		for ( pollCnt=0;pollCnt<NUMBER_OF_POLL_PER_OPERATION;pollCnt++ )
		{
			LoopDelayMS( POLLING_DELAY );
			if ( DL_I2C_getControllerRXFIFOCounter(I2C_INST) == 0 )
			{
				break;
			}
		}
		DL_I2C_stopFlushControllerRXFIFO( I2C_INST );		
	}
}


/*** Public Functions *********************************************************/
void I2C_Init(void)
{
}

I2cOperationStatus_t I2C_Read( uint8_t devAddr, uint8_t regAddr, uint8_t *dataBuf, uint8_t length )
{
	I2cOperationStatus_t status = I2C_SUCCESS;

	if ( length > I2C_MAX_FIFO_ENTRIES )
		return I2C_ERR_DATA_SIZE;

	DL_I2C_enableController( I2C_INST );

	gI2cTxPacket[0] = regAddr;
	/*
	 * Fill FIFO with data. This example will send a MAX of 8 bytes since it
	 * doesn't handle the case where FIFO is full
	 */
	DL_I2C_fillControllerTXFIFO( I2C_INST, &gI2cTxPacket[0], 1 );

	/* Wait for I2C to be Idle */
#if defined(I2C_USE_WHILE_FOR_POLLING)

	I2C_DEBUG("I2C:Rd FifoSetErr\n");
	while ( !(DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_IDLE) );

#else // I2C_USE_WHILE_FOR_POLLING

	if ( !I2C_StatusPolling(DL_I2C_CONTROLLER_STATUS_IDLE, TRUE) )
	{
		status = I2C_ERR_FIFO_SET;
		goto exit_on_err;
	}

#endif // I2C_USE_WHILE_FOR_POLLING

	/* Send the packet to the controller.
	 * This function will send Start + Stop automatically.
	 */
	DL_I2C_startControllerTransfer( I2C_INST, devAddr, DL_I2C_CONTROLLER_DIRECTION_TX, 1);

	/* Poll until the Controller writes all bytes */
#if defined(I2C_USE_WHILE_FOR_POLLING)

	I2C_DEBUG("I2C:Rd AddrWrTout\n");
	while ( DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_BUSY_BUS );

#else // I2C_USE_WHILE_FOR_POLLING

	if ( !I2C_StatusPolling(DL_I2C_CONTROLLER_STATUS_BUSY_BUS, FALSE) )
	{
		status = I2C_ERR_DEV_ADDRESS_WR_TIMEOUT;
		goto exit_on_err;
	}

#endif // I2C_USE_WHILE_FOR_POLLING

	/* Trap if there was an error */
	if ( DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_ERROR )
	{
//		DL_I2C_disableController(I2C_INST);
	    status = I2C_ERR_DEV_ADDRESS_WR;
		goto exit_on_err;
	}

	/* Wait for I2C to be Idle */
#if defined(I2C_USE_WHILE_FOR_POLLING)

	I2C_DEBUG("I2C:Rd IdleTout\n");
	while ( !(DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_IDLE) );

#else // I2C_USE_WHILE_FOR_POLLING

	if ( !I2C_StatusPolling(DL_I2C_CONTROLLER_STATUS_IDLE, TRUE) )
	{
		status = I2C_ERR_DEV_IDLE_TIMEOUT;
		goto exit_on_err;
	}

#endif // I2C_USE_WHILE_FOR_POLLING


	/* Add delay between transfers */
	delay_cycles(1000);

	/* Send a read request to Target */
	DL_I2C_startControllerTransfer( I2C_INST, devAddr, DL_I2C_CONTROLLER_DIRECTION_RX, length);

	/*
	 * Receive all bytes from target. LED will remain high if not all bytes
	 * are received
	 */
	for ( uint8_t i = 0; i < length; i++ )
	{
#if defined(I2C_USE_WHILE_FOR_POLLING)

		I2C_DEBUG("I2C:Rd DataRxTout\n");
		while ( DL_I2C_isControllerRXFIFOEmpty(I2C_INST) );

#else // I2C_USE_WHILE_FOR_POLLING

		if ( !I2C_DataPolling() )
		{
			status = I2C_ERR_DATA_RECEIVE_TIMEOUT;
			goto exit_on_err;
		}

#endif // I2C_USE_WHILE_FOR_POLLING

	    dataBuf[i] = DL_I2C_receiveControllerData( I2C_INST );
	}

exit_on_err:
	DL_I2C_disableController( I2C_INST );
	I2C_FlushFIFO();										// try to flush the FIFO in case of error.

	return status;
}

I2cOperationStatus_t I2C_WriteByte( uint8_t devAddr, uint8_t regAddr, uint8_t dataByte )
{
	I2cOperationStatus_t status = I2C_SUCCESS;

	DL_I2C_enableController( I2C_INST );

	gI2cTxPacket[0] = regAddr;
	gI2cTxPacket[1] = dataByte;
	/*
	 * Fill FIFO with data. This example will send a MAX of 8 bytes since it
	 * doesn't handle the case where FIFO is full
	 */
	DL_I2C_fillControllerTXFIFO( I2C_INST, &gI2cTxPacket[0], 2 );

	/* Wait for I2C to be Idle */
#if defined(I2C_USE_WHILE_FOR_POLLING)

	I2C_DEBUG("I2C:Wr FifoSetErr\n");
	while ( !(DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_IDLE) );

#else // I2C_USE_WHILE_FOR_POLLING

	if ( !I2C_StatusPolling(DL_I2C_CONTROLLER_STATUS_IDLE, TRUE) )
	{
		status = I2C_ERR_FIFO_SET;
		goto exit_on_err;
	}

#endif // I2C_USE_WHILE_FOR_POLLING

	/* Send the packet to the controller.
	 * This function will send Start + Stop automatically.
	 */
	DL_I2C_startControllerTransfer( I2C_INST, devAddr, DL_I2C_CONTROLLER_DIRECTION_TX, 2);

	/* Poll until the Controller writes all bytes */
#if defined(I2C_USE_WHILE_FOR_POLLING)

	I2C_DEBUG("I2C:Wr AddrWrTout\n");
	while ( DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_BUSY_BUS );

#else // I2C_USE_WHILE_FOR_POLLING

	if ( !I2C_StatusPolling(DL_I2C_CONTROLLER_STATUS_BUSY_BUS, FALSE) )
	{
		status = I2C_ERR_DEV_ADDRESS_WR_TIMEOUT;
		goto exit_on_err;
	}

#endif // I2C_USE_WHILE_FOR_POLLING

	/* Trap if there was an error */
	if ( DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_ERROR )
	{
//		DL_I2C_disableController(I2C_INST);
		status = I2C_ERR_DEV_ADDRESS_DATA_WR;
		goto exit_on_err;
	}

	/* Wait for I2C to be Idle */
#if defined(I2C_USE_WHILE_FOR_POLLING)

	I2C_DEBUG("I2C:Wr IdleTout\n");
	while ( !(DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_IDLE) );

#else // I2C_USE_WHILE_FOR_POLLING

	if ( !I2C_StatusPolling(DL_I2C_CONTROLLER_STATUS_IDLE, TRUE) )
	{
		status = I2C_ERR_DEV_IDLE_TIMEOUT;
		goto exit_on_err;
	}

#endif // I2C_USE_WHILE_FOR_POLLING

exit_on_err:
	DL_I2C_disableController( I2C_INST	);
	I2C_FlushFIFO();										// try to flush the FIFO in case of error.

	return status;
}

I2cOperationStatus_t I2C_WriteBytes( uint8_t devAddr, uint8_t regAddr, uint8_t *dataBuf, uint8_t dataSize )
{
	I2cOperationStatus_t status = I2C_SUCCESS;

	if ( dataSize > (I2C_MAX_FIFO_ENTRIES - 1) )
		return I2C_ERR_DATA_SIZE;

	DL_I2C_enableController( I2C_INST );

	gI2cTxPacket[0] = regAddr;
	memcpy( &gI2cTxPacket[1], dataBuf, dataSize );
	/*
	 * Fill FIFO with data. This example will send a MAX of 8 bytes since it
	 * doesn't handle the case where FIFO is full
	 */
	DL_I2C_fillControllerTXFIFO( I2C_INST, &gI2cTxPacket[0], (dataSize+1) );

	/* Wait for I2C to be Idle */
#if defined(I2C_USE_WHILE_FOR_POLLING)

	I2C_DEBUG("I2C:Wrs FifoSetErr\n");
	while ( !(DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_IDLE) );

#else // I2C_USE_WHILE_FOR_POLLING

	if ( !I2C_StatusPolling(DL_I2C_CONTROLLER_STATUS_IDLE, TRUE) )
	{
		status = I2C_ERR_FIFO_SET;
		goto exit_on_err;
	}

#endif // I2C_USE_WHILE_FOR_POLLING

	/* Send the packet to the controller.
	 * This function will send Start + Stop automatically.
	 */
	DL_I2C_startControllerTransfer( I2C_INST, devAddr, DL_I2C_CONTROLLER_DIRECTION_TX, 2);

	/* Poll until the Controller writes all bytes */
#if defined(I2C_USE_WHILE_FOR_POLLING)

	I2C_DEBUG("I2C:Wrs AddrWrTout\n");
	while ( DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_BUSY_BUS );

#else // I2C_USE_WHILE_FOR_POLLING

	if ( !I2C_StatusPolling(DL_I2C_CONTROLLER_STATUS_BUSY_BUS, FALSE) )
	{
		status = I2C_ERR_DEV_ADDRESS_WR_TIMEOUT;
		goto exit_on_err;
	}

#endif // I2C_USE_WHILE_FOR_POLLING

	/* Trap if there was an error */
	if ( DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_ERROR )
	{
//		DL_I2C_disableController(I2C_INST);
		status = I2C_ERR_DEV_ADDRESS_DATA_WR;
		goto exit_on_err;
	}

	/* Wait for I2C to be Idle */
#if defined(I2C_USE_WHILE_FOR_POLLING)

	I2C_DEBUG("I2C:Wrs IdleTout\n");
	while ( !(DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_IDLE) );

#else // I2C_USE_WHILE_FOR_POLLING

	if ( !I2C_StatusPolling(DL_I2C_CONTROLLER_STATUS_IDLE, TRUE) )
	{
		status = I2C_ERR_DEV_IDLE_TIMEOUT;
		goto exit_on_err;
	}

#endif // I2C_USE_WHILE_FOR_POLLING

exit_on_err:
	DL_I2C_disableController( I2C_INST	);
	I2C_FlushFIFO();										// try to flush the FIFO in case of error.

	return status;
}

