/*
 * my_calibration.c
 *
 *  Created on: gg/mmm/ccaa
 *      Author: cognome.nome
 *
 * funzioni usate in fase di collaudo per la calibrazione iniziale automatica dell'ADC
 *
 */

//==============================================================================
// Include files

#include "string.h"
#include "stdbool.h"
#include "stdint.h"

#include "driverlib/gpio.h"
#include "driverlib/sysctl.h"
#include "driverlib/rom.h"

#include "drivers/my_SPI.h"
#include "my_GPIO.h"
#include "my_timer.h"
#include "drivers/my_semaphore.h"
#include "drivers/AD7193.h"
#include "my_flash.h"

#include "my_calibration.h"

//==============================================================================
// Constants

#define CALIBRATION_NOT_VALID	0xFFFFFFFF
#define CORRECTION				600
#define CORRECTION_NO_BUF		2600

//==============================================================================
// Types

//==============================================================================
// Static global variables

static volatile unsigned long CalibrationCounter = 0;

//==============================================================================
// Static functions

static long SingleEndedChannelSelection(EChannelID ChannelID, TConfiguration *Configuration)
{
	// seleziono il canale 0
	switch(ChannelID)
	{
		case ChannelID1:
		{
			// seleziono il canale 1
			Configuration->CH0  = 1;
			Configuration->CH1  = 0;
			Configuration->CH2  = 0;
			Configuration->CH3  = 0;
			Configuration->CH4  = 0;
			Configuration->CH5  = 0;
			Configuration->CH6  = 0;
			Configuration->CH7  = 0;
			Configuration->Chop = 0;
		}
		break;
		case ChannelID2:
		{
			// seleziono il canale 2
			Configuration->CH0  = 0;
			Configuration->CH1  = 0;
			Configuration->CH2  = 1;
			Configuration->CH3  = 0;
			Configuration->CH4  = 0;
			Configuration->CH5  = 0;
			Configuration->CH6  = 0;
			Configuration->CH7  = 0;
			Configuration->Chop = 0;
		}
		break;
		case ChannelID3:
		{
			// seleziono il canale 3
			Configuration->CH0  = 0;
			Configuration->CH1  = 0;
			Configuration->CH2  = 0;
			Configuration->CH3  = 0;
			Configuration->CH4  = 1;
			Configuration->CH5  = 0;
			Configuration->CH6  = 0;
			Configuration->CH7  = 0;
			Configuration->Chop = 0;
		}
		break;
		case ChannelID4:
		{
			// seleziono il canale 4
			Configuration->CH0  = 0;
			Configuration->CH1  = 0;
			Configuration->CH2  = 0;
			Configuration->CH3  = 0;
			Configuration->CH4  = 0;
			Configuration->CH5  = 0;
			Configuration->CH6  = 1;
			Configuration->CH7  = 0;
			Configuration->Chop = 0;
		}
		break;
		default:{}
	}

	return 0;
}

static long DifferentialChannelSelection(EChannelID ChannelID, TConfiguration *Configuration)
{
	switch(ChannelID)
	{
		case ChannelID1:
		{
			// seleziono il canale 1
			Configuration->CH0  = 1;
			Configuration->CH1  = 0;
			Configuration->CH2  = 0;
			Configuration->CH3  = 0;
			Configuration->CH4  = 0;
			Configuration->CH5  = 0;
			Configuration->CH6  = 0;
			Configuration->CH7  = 0;
			Configuration->Chop = 0;
		}
		break;
		case ChannelID2:
		{
			// seleziono il canale 2
			Configuration->CH0  = 0;
			Configuration->CH1  = 0;
			Configuration->CH2  = 0;
			Configuration->CH3  = 0;
			Configuration->CH4  = 0;
			Configuration->CH5  = 1;
			Configuration->CH6  = 0;
			Configuration->CH7  = 0;
			Configuration->Chop = 0;
		}
		break;
		case ChannelID3:
		{
			// seleziono il canale 3
			Configuration->CH0  = 0;
			Configuration->CH1  = 0;
			Configuration->CH2  = 1;
			Configuration->CH3  = 0;
			Configuration->CH4  = 0;
			Configuration->CH5  = 0;
			Configuration->CH6  = 0;
			Configuration->CH7  = 0;
			Configuration->Chop = 0;
		}
		break;
		case ChannelID4:
		{
			// seleziono il canale 4
			Configuration->CH0  = 0;
			Configuration->CH1  = 0;
			Configuration->CH2  = 0;
			Configuration->CH3  = 0;
			Configuration->CH4  = 0;
			Configuration->CH5  = 0;
			Configuration->CH6  = 0;
			Configuration->CH7  = 1;
			Configuration->Chop = 0;
		}
		break;
		default:{}
	}

	return 0;
}

static long ModeRegister(TMode *Mode, EOperatingModes OperatingModes)
{
	switch(OperatingModes)
	{
		case Continuous_conversion_mode:
		{
			Mode->MD0 = 0;
			Mode->MD1 = 0;
			Mode->MD2 = 0;
		}break;
		case Single_conversion_mode:
		{
			Mode->MD0 = 1;
			Mode->MD1 = 0;
			Mode->MD2 = 0;
		}break;
		case Idle_mode:
		{
			Mode->MD0 = 0;
			Mode->MD1 = 1;
			Mode->MD2 = 0;
		}break;
		case Power_down_mode:
		{
			Mode->MD0 = 1;
			Mode->MD1 = 1;
			Mode->MD2 = 0;
		}break;
		case Internal_zero_scale_calibration:
		{
			Mode->MD0 = 0;
			Mode->MD1 = 0;
			Mode->MD2 = 1;
		}break;
		case Internal_full_scale_calibration:
		{
			Mode->MD0 = 1;
			Mode->MD1 = 0;
			Mode->MD2 = 1;
		}break;
		case System_zero_scale_calibration:
		{
			Mode->MD0 = 0;
			Mode->MD1 = 1;
			Mode->MD2 = 1;
		}break;
		case System_full_scale_calibration:
		{
			Mode->MD0 = 1;
			Mode->MD1 = 1;
			Mode->MD2 = 1;
		}break;
		default:{};
	}

	return 0;
}

static long WriteOffsetRegisterInternal(TOffset *Offset)
{
	WaitSemaphore(&sem_ADC);

	//------------------------------------------------------------------
	Communications.nWEN = 0;
	Communications.RnW  = 0;
	RegisterSelection(Offset_register);
	Communications.CREAD = 0;
	//------------------------------------------------------------------

	//------------------------------------------------------------------
	SPI.RegisterSelection = Offset_register;
	SPI.RnW               = 0;
	SPI.COMM_counter      = WORD_8;
	SPI.RD_counter        = WORD_NONE;
	SPI.WR_counter        = WORD_24;
	//------------------------------------------------------------------

	SPI.DataW.DataSPIAll = Offset->OffsetAll;

	// abilitazione interrupt gestione SPI
	SPI_WriteCommunication_Timer_Enable();

	return 0;
}

static long AD7193_WriteOffsetRegister(EAcquistionMode AcquisitionMode, TOffset *Offset, EChannelID ChannelID)
{
	if(Offset->OffsetAll == CALIBRATION_NOT_VALID)
		return -1;

	if(AcquisitionMode == Single_ended)
	{
		SingleEndedChannelSelection(ChannelID,&ConfigurationForCalib);
	}
	else
	{
		DifferentialChannelSelection(ChannelID,&ConfigurationForCalib);
	}

	ModeForCalib.FS = 480;
	ModeForCalib.AVG0 = 0;
	ModeForCalib.AVG1 = 0;
	// metto il dispositivo in idle
	ModeRegister(&ModeForCalib, Idle_mode);

	AD7193_WriteConfigurationRegister(&ConfigurationForCalib);
	AD7193_WriteModeRegister(&ModeForCalib);

	WriteOffsetRegisterInternal(Offset);

	//Ripristino il dispositivo
	AD7193_WriteConfigurationRegister(&Configuration);
    AD7193_WriteModeRegister(&Mode);

    return 0;
}


static long ApplyCalibrationOffset(void)
{
	EChannel ch;
	EChannelID ID;

	if(AcquisitionManage.AcquisitionMode == Single_ended)
	{
		for(ch = Ch1, ID = ChannelID1; ch < DimCh ; ch++, ID++)
		{
			switch(AcquisitionManage.Gain)
			{
				case 1:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain1].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 8:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain8].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 16:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain16].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 32:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain32].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 64:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain64].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 128:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain128].ChannelSingleEnded[ch]),ID);
				}
				break;
				default:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain1].ChannelSingleEnded[ch]),ID);
				}
			}
		}
	}
	else
	{
		for(ch = Ch1, ID = ChannelID1; ch < DimCh ; ch++, ID++)
		{
			switch(AcquisitionManage.Gain)
			{
				case 1:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain1].ChannelDifferential[ch]),ID);
				}
				break;
				case 8:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain8].ChannelDifferential[ch]),ID);
				}
				break;
				case 16:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain16].ChannelDifferential[ch]),ID);
				}
				break;
				case 32:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain32].ChannelDifferential[ch]),ID);
				}
				break;
				case 64:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain64].ChannelDifferential[ch]),ID);
				}
				break;
				case 128:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain128].ChannelDifferential[ch]),ID);
				}
				break;
				default:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain1].ChannelDifferential[ch]),ID);
				}
			}
		}
	}
	return 0;
}

static long WriteFullScaleRegisterInternal(TFullScale *FullScale)
{
	WaitSemaphore(&sem_ADC);

	//------------------------------------------------------------------
	Communications.nWEN = 0;
	Communications.RnW  = 0;
	RegisterSelection(Full_scale_register);
	Communications.CREAD = 0;
	//------------------------------------------------------------------

	//------------------------------------------------------------------
	SPI.RegisterSelection = Full_scale_register;
	SPI.RnW               = 0;
	SPI.COMM_counter      = WORD_8;
	SPI.RD_counter        = WORD_NONE;
	SPI.WR_counter        = WORD_24;
	//------------------------------------------------------------------

	SPI.DataW.DataSPIAll = FullScale->FullScaleAll;

	// abilitazione interrupt gestione SPI
	SPI_WriteCommunication_Timer_Enable();

	return 0;
}

static long AD7193_WriteFullScaleRegister(EAcquistionMode AcquisitionMode, TFullScale *FullScale, EChannelID ChannelID)
{
	if(FullScale->FullScaleAll == CALIBRATION_NOT_VALID)
		return -1;

	if(AcquisitionMode == Single_ended)
	{
		SingleEndedChannelSelection(ChannelID,&ConfigurationForCalib);
	}
	else
	{
		DifferentialChannelSelection(ChannelID,&ConfigurationForCalib);
	}

	ModeForCalib.FS = 480;
	ModeForCalib.AVG0 = 0;
	ModeForCalib.AVG1 = 0;
	// metto il dispositivo in idle
	ModeRegister(&ModeForCalib, Idle_mode);

	AD7193_WriteConfigurationRegister(&ConfigurationForCalib);
	AD7193_WriteModeRegister(&ModeForCalib);

	WriteFullScaleRegisterInternal(FullScale);

	//Ripristino il dispositivo
	AD7193_WriteConfigurationRegister(&Configuration);
    AD7193_WriteModeRegister(&Mode);

    return 0;
}

static long Calibration(EOperatingModes calibration)
{
	long i;//,j;
	EAcquistionMode j;
	EChannelID CHID;
	EChannel ch;
	EGainSelection current_gain;

	for( j = Single_ended ; j <= Differential ; j++ )
	//for( j = 0 ; j < 2 ; j++ )
	{
		InitAD7193ForCalibration(j, Gain1);
		//InitAD7193ForCalibration(Single_ended + j, Gain1);

		for( current_gain = Gain1 ; current_gain < DimGain ; current_gain++ )
		{
			if((calibration == Internal_full_scale_calibration) || (calibration == System_full_scale_calibration))
			{
				ApplyCalibrationOffset();
			}

			// selezione gain
			AcquisitionManage.GainSel = current_gain;
			GainSelector(&ConfigurationForCalib,&AcquisitionManage);

			for( i = 0, CHID = ChannelID1 , ch = Ch1 ; i < 4 ; i++, CHID++ , ch++ )
			{
				//---------------------------------------------------------
				// CH - leggi
				//---------------------------------------------------------
				if(AcquisitionManage.AcquisitionMode == Single_ended)
				{
					SingleEndedChannelSelection(CHID,&ConfigurationForCalib);
				}
				else
				{
					DifferentialChannelSelection(CHID,&ConfigurationForCalib);
				}

				ModeForCalib.FS = 480;
				ModeRegister(&ModeForCalib, calibration);
				ModeForCalib.AVG0 = 0;
				ModeForCalib.AVG1 = 0;

				AD7193_WriteConfigurationRegister(&ConfigurationForCalib);
				AD7193_WriteModeRegister(&ModeForCalib);

				ROM_SysCtlDelay(MILLI_DELAY);
				do
				{
					// aspetto che il pin vada a 0
					CalibrationCounter++;
				}while(GPIO_get_ADC_DOUT() == 1);

				switch(calibration)
				{
					case Internal_zero_scale_calibration:
					case System_zero_scale_calibration:
					{
						AD7193_ReadOffsetRegister();
					}
					break;
					case Internal_full_scale_calibration:
					case System_full_scale_calibration:
					{
						AD7193_ReadFullScaleRegister();
					}
					break;
				}

				//---------------------------------------------------------
				// CH - salva
				//---------------------------------------------------------
				if(AcquisitionManage.AcquisitionMode == Single_ended)
				{
					switch(calibration)
					{
						case Internal_zero_scale_calibration:
						case System_zero_scale_calibration:
						{
							switch(AcquisitionManage.Gain)
							{
								case 1:
								{
									FlashCalibration.Offset[Gain1].ChannelSingleEnded[ch] = Offset.OffsetAll;
								}
								break;
								case 8:
								{
									FlashCalibration.Offset[Gain8].ChannelSingleEnded[ch] = 0x800000;
								}
								break;
								case 16:
								{
									FlashCalibration.Offset[Gain16].ChannelSingleEnded[ch] = 0x800000;
								}
								break;
								case 32:
								{
									FlashCalibration.Offset[Gain32].ChannelSingleEnded[ch] = 0x800000;
								}
								break;
								case 64:
								{
									FlashCalibration.Offset[Gain64].ChannelSingleEnded[ch] = 0x800000;
								}
								break;
								case 128:
								{
									FlashCalibration.Offset[Gain128].ChannelSingleEnded[ch] = 0x800000;
								}
								break;
							}
						}
						break;
						case Internal_full_scale_calibration:
						case System_full_scale_calibration:
						{
							switch(AcquisitionManage.Gain)
							{
								case 1:
								{
									FlashCalibration.FScale[Gain1].ChannelSingleEnded[ch] = FullScale.FullScaleAll - CORRECTION;
								}
								break;
								case 8:
								{
									FlashCalibration.FScale[Gain8].ChannelSingleEnded[ch] = FullScale.FullScaleAll + CORRECTION_NO_BUF;
								}
								break;
								case 16:
								{
									FlashCalibration.FScale[Gain16].ChannelSingleEnded[ch] = FullScale.FullScaleAll + CORRECTION_NO_BUF;
								}
								break;
								case 32:
								{
									FlashCalibration.FScale[Gain32].ChannelSingleEnded[ch] = FullScale.FullScaleAll + CORRECTION_NO_BUF;
								}
								break;
								case 64:
								{
									FlashCalibration.FScale[Gain64].ChannelSingleEnded[ch] = FullScale.FullScaleAll + CORRECTION_NO_BUF;
								}
								break;
								case 128:
								{
									FlashCalibration.FScale[Gain128].ChannelSingleEnded[ch] = FullScale.FullScaleAll + CORRECTION_NO_BUF;
								}
								break;
							}
						}
						break;
					}
				}
				//Differential
				else
				{
					switch(calibration)
					{
						case Internal_zero_scale_calibration:
						case System_zero_scale_calibration:
						{
							switch(AcquisitionManage.Gain)
							{
								case 1:
								{
									FlashCalibration.Offset[Gain1].ChannelDifferential[ch] =  Offset.OffsetAll;
								}
								break;
								case 8:
								{
									FlashCalibration.Offset[Gain8].ChannelDifferential[ch] = 0x800000;
								}
								break;
								case 16:
								{
									FlashCalibration.Offset[Gain16].ChannelDifferential[ch] = 0x800000;
								}
								break;
								case 32:
								{
									FlashCalibration.Offset[Gain32].ChannelDifferential[ch] = 0x800000;
								}
								break;
								case 64:
								{
									FlashCalibration.Offset[Gain64].ChannelDifferential[ch] = 0x800000;
								}
								break;
								case 128:
								{
									FlashCalibration.Offset[Gain128].ChannelDifferential[ch] = 0x800000;
								}
								break;
							}
						}
						break;
						case Internal_full_scale_calibration:
						case System_full_scale_calibration:
						{
							switch(AcquisitionManage.Gain)
							{
								case 1:
								{
									FlashCalibration.FScale[Gain1].ChannelDifferential[ch] = FullScale.FullScaleAll - CORRECTION;
								}
								break;
								case 8:
								{
									FlashCalibration.FScale[Gain8].ChannelDifferential[ch] = FullScale.FullScaleAll + CORRECTION_NO_BUF;
								}
								break;
								case 16:
								{
									FlashCalibration.FScale[Gain16].ChannelDifferential[ch] = FullScale.FullScaleAll + CORRECTION_NO_BUF;
								}
								break;
								case 32:
								{
									FlashCalibration.FScale[Gain32].ChannelDifferential[ch] = FullScale.FullScaleAll + CORRECTION_NO_BUF;
								}
								break;
								case 64:
								{
									FlashCalibration.FScale[Gain64].ChannelDifferential[ch] = FullScale.FullScaleAll + CORRECTION_NO_BUF;
								}
								break;
								case 128:
								{
									FlashCalibration.FScale[Gain128].ChannelDifferential[ch] = FullScale.FullScaleAll + CORRECTION_NO_BUF;
								}
								break;
							}
						}
						break;
					}
				}
			}
		}
	}
	return 0;
}

//==============================================================================
// Global variables

//==============================================================================
// Global functions

long AD7193_Make_Calibration(EOperatingModes calibration)
{

	ReadCalibrationParameters(&FlashCalibration);

	switch(calibration)
	{
		case System_zero_scale_calibration:
		{
			Calibration(System_zero_scale_calibration);
		}
		break;
		case System_full_scale_calibration:
		{
			Calibration(System_full_scale_calibration);
		}
		break;
		default:
		{
			Calibration(System_full_scale_calibration);
		}
	}

	WriteCalibrationParameters(&FlashCalibration);

	return 0;
}


long ApplyCalibrationFromFlashOffset(void)
{
	EChannel ch;
	EChannelID ID;

	// legge dalla flash
	ReadCalibrationParameters(&FlashCalibration);

	if(AcquisitionManage.AcquisitionMode == Single_ended)
	{
		for(ch = Ch1, ID = ChannelID1; ch < DimCh ; ch++, ID++)
		{
			switch(AcquisitionManage.Gain)
			{
				case 1:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain1].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 8:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain8].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 16:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain16].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 32:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain32].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 64:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain64].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 128:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain128].ChannelSingleEnded[ch]),ID);
				}
				break;
				default:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain1].ChannelSingleEnded[ch]),ID);
				}
			}
		}
	}
	else
	{
		for(ch = Ch1, ID = ChannelID1; ch < DimCh ; ch++, ID++)
		{
			switch(AcquisitionManage.Gain)
			{
				case 1:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain1].ChannelDifferential[ch]),ID);
				}
				break;
				case 8:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain8].ChannelDifferential[ch]),ID);
				}
				break;
				case 16:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain16].ChannelDifferential[ch]),ID);
				}
				break;
				case 32:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain32].ChannelDifferential[ch]),ID);
				}
				break;
				case 64:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain64].ChannelDifferential[ch]),ID);
				}
				break;
				case 128:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain128].ChannelDifferential[ch]),ID);
				}
				break;
				default:
				{
					AD7193_WriteOffsetRegister(AcquisitionManage.AcquisitionMode,((TOffset *)&FlashCalibration.Offset[Gain1].ChannelDifferential[ch]),ID);
				}
			}
		}
	}

	return 0;
}


long ApplyCalibrationFromFlashFullScale(void)
{
	EChannel ch;
	EChannelID ID;

	// legge dalla flash
	ReadCalibrationParameters(&FlashCalibration);

	if(AcquisitionManage.AcquisitionMode == Single_ended)
	{
		for(ch = Ch1, ID = ChannelID1; ch < DimCh ; ch++, ID++)
		{
			switch(AcquisitionManage.Gain)
			{
				case 1:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain1].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 8:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain8].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 16:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain16].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 32:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain32].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 64:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain64].ChannelSingleEnded[ch]),ID);
				}
				break;
				case 128:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain128].ChannelSingleEnded[ch]),ID);
				}
				break;
				default:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain1].ChannelSingleEnded[ch]),ID);
				}
			}
		}
	}
	else
	{
		for(ch = Ch1, ID = ChannelID1; ch < DimCh ; ch++, ID++)
		{
			switch(AcquisitionManage.Gain)
			{
				case 1:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain1].ChannelDifferential[ch]),ID);
				}
				break;
				case 8:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain8].ChannelDifferential[ch]),ID);
				}
				break;
				case 16:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain16].ChannelDifferential[ch]),ID);
				}
				break;
				case 32:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain32].ChannelDifferential[ch]),ID);
				}
				break;
				case 64:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain64].ChannelDifferential[ch]),ID);
				}
				break;
				case 128:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain128].ChannelDifferential[ch]),ID);
				}
				break;
				default:
				{
					AD7193_WriteFullScaleRegister(AcquisitionManage.AcquisitionMode,((TFullScale *)&FlashCalibration.FScale[Gain1].ChannelDifferential[ch]),ID);
				}
			}
		}
	}

	return 0;
}
