//*****************************************************************************
//
// project0.c - Example to demonstrate minimal TivaWare setup
//
// Copyright (c) 2012-2018 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 2.1.4.178 of the EK-TM4C123GXL Firmware Package.
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_memmap.h"
#include "inc/hw_ints.h"
#include "driverlib/debug.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "driverlib/i2c.h"

#define MPU6050_I2C_ADDRESS 0x68
#define MPU6050_I2C_BASE I2C3_BASE

void MPU6050WriteRegister(uint8_t register, uint8_t value);
uint8_t MPU6050ReadRegister(uint8_t register);
void MPU6050BurstRead(uint8_t reg, unsigned int count, uint8_t *p8Data);
void MPU6050BurstWrite(uint8_t reg, unsigned int count, uint8_t *p8Data);

struct MPU6050
{
    uint8_t AccelXH;
    uint8_t AccelXL;
    uint8_t AccelYH;
    uint8_t AccelYL;
    uint8_t AccelZH;
    uint8_t AccelZL;
    uint8_t TempH;
    uint8_t TempL;
    uint8_t GyroXH;
    uint8_t GyroXL;
    uint8_t GyroYH;
    uint8_t GyroYL;
    uint8_t GyroZH;
    uint8_t GyroZL;
} mpu6050;

const struct MPU6050_CONFIG
{
    uint8_t Configuration;
    uint8_t GyroConfig;
    uint8_t AccelConfig;
} mpu6050Config = { 0x02, 0x00, 0x01 };

const uint8_t mpu6050Config2[] = { 0x4D, 0x8C, 0x02, 0x88};
const uint8_t mpu6050Config3[] = { 0x0C, 0x0A, 0x01, 0x84};
//*****************************************************************************
//
// Main 'C' Language entry point.  Toggle an LED using TivaWare.
//
//*****************************************************************************
int
main(void)
{
    //
    // Setup the system clock to run at 50 Mhz from PLL with crystal reference
    //
    SysCtlClockSet(SYSCTL_SYSDIV_4|SYSCTL_USE_PLL|SYSCTL_XTAL_16MHZ|
                    SYSCTL_OSC_MAIN);

    //
    // The I2C3 peripheral must be enabled before use.
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C3);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);

    //
    // Configure the pin muxing for I2C3 functions on port D0 and D1.
    //
    ROM_GPIOPinConfigure(GPIO_PD0_I2C3SCL);
    ROM_GPIOPinConfigure(GPIO_PD1_I2C3SDA);

    //
    // Select the I2C function for these pins.  This function will also
    // configure the GPIO pins pins for I2C operation, setting them to
    // open-drain operation with weak pull-ups.  Consult the data sheet
    // to see which functions are allocated per pin.
    //
    GPIOPinTypeI2CSCL(GPIO_PORTD_BASE, GPIO_PIN_0);
    ROM_GPIOPinTypeI2C(GPIO_PORTD_BASE, GPIO_PIN_1);

    // Initialize I2C peripheral.
    I2CMasterInitExpClk(MPU6050_I2C_BASE, SysCtlClockGet(), true);
    I2CMasterSlaveAddrSet(MPU6050_I2C_BASE, MPU6050_I2C_ADDRESS, false);

    // Use GPIOB2 as input, low when new sample is ready
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);
    ROM_GPIOPinTypeGPIOInput(GPIO_PORTB_BASE, GPIO_PIN_2);

    // Initialize the MPU6050
    MPU6050WriteRegister(0x6B, 0x80);
    // Wait until reset is complete (reg 0x6B = 0x40)
    while((MPU6050ReadRegister(0x6B) & 0x40) != 0x40);
    // Use PLL with X axis gyroscope reference
    MPU6050WriteRegister(0x6B, 0x01);
    // Enable I2C Master mode
    MPU6050WriteRegister(0x6A, 0x20);
    // Set sample rate divider
    MPU6050WriteRegister(0x19, 0x13);

    MPU6050WriteRegister(0x67, 0x11);
    MPU6050BurstWrite(0x24, sizeof(mpu6050Config2), (uint8_t *)&mpu6050Config2);
    MPU6050BurstWrite(0x31, sizeof(mpu6050Config3), (uint8_t *)&mpu6050Config3);


    // Configure Sample rate, Gyro and Accelerometer
    MPU6050BurstWrite(0x1A, sizeof(mpu6050Config), (uint8_t *)&mpu6050Config);
    // Enable Interrupt Pin
    MPU6050WriteRegister(0x37, 0xB0);
    MPU6050WriteRegister(0x38, 0x01);

    while(1)
    {
        // Wait for data ready
//        while(GPIOPinRead(GPIO_PORTB_BASE, GPIO_PIN_2) == GPIO_PIN_2);
        // Read the raw values
        MPU6050BurstRead(0x3B, 14, (uint8_t *)&mpu6050);
    }
}

void MPU6050WriteRegister(uint8_t reg, uint8_t value)
{
    I2CMasterSlaveAddrSet(MPU6050_I2C_BASE, MPU6050_I2C_ADDRESS, false);
    I2CMasterDataPut(MPU6050_I2C_BASE, reg);
    I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_BURST_SEND_START);
    while(I2CMasterBusy(MPU6050_I2C_BASE));
    I2CMasterDataPut(MPU6050_I2C_BASE, value);
    I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_BURST_SEND_FINISH);
    while(I2CMasterBusy(MPU6050_I2C_BASE));

}

uint8_t MPU6050ReadRegister(uint8_t reg)
{
    I2CMasterSlaveAddrSet(MPU6050_I2C_BASE, MPU6050_I2C_ADDRESS, false);
    I2CMasterDataPut(MPU6050_I2C_BASE, reg);
    I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_SINGLE_SEND);
    while(I2CMasterBusy(MPU6050_I2C_BASE));
    I2CMasterSlaveAddrSet(MPU6050_I2C_BASE, MPU6050_I2C_ADDRESS, true);
    I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_SINGLE_RECEIVE);
    while(I2CMasterBusy(MPU6050_I2C_BASE));
    return (uint8_t)I2CMasterDataGet(MPU6050_I2C_BASE);
}

void MPU6050BurstRead(uint8_t reg, unsigned int count, uint8_t *p8Data)
{
    unsigned int i;

    if(count < 2)
    {
        return;
    }
    I2CMasterSlaveAddrSet(MPU6050_I2C_BASE, MPU6050_I2C_ADDRESS, false);
    I2CMasterDataPut(MPU6050_I2C_BASE, reg);
    I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_SINGLE_SEND);
    while(I2CMasterBusy(MPU6050_I2C_BASE));
    I2CMasterSlaveAddrSet(MPU6050_I2C_BASE, MPU6050_I2C_ADDRESS, true);
    I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_BURST_RECEIVE_START);
    while(I2CMasterBusy(MPU6050_I2C_BASE));
    *p8Data++ = (uint8_t)I2CMasterDataGet(MPU6050_I2C_BASE);
    for(i = 0; i < (count - 2); i++)
    {
        I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_BURST_RECEIVE_CONT);
        while(I2CMasterBusy(MPU6050_I2C_BASE));
        *p8Data++ = (uint8_t)I2CMasterDataGet(MPU6050_I2C_BASE);
    }
    I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_BURST_RECEIVE_FINISH);
    while(I2CMasterBusy(MPU6050_I2C_BASE));
    *p8Data++ = (uint8_t)I2CMasterDataGet(MPU6050_I2C_BASE);
}

void MPU6050BurstWrite(uint8_t reg, unsigned int count, uint8_t *p8Data)
{
    unsigned int i;

    I2CMasterSlaveAddrSet(MPU6050_I2C_BASE, MPU6050_I2C_ADDRESS, false);
    I2CMasterDataPut(MPU6050_I2C_BASE, reg);
    I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_BURST_SEND_START);
    while(I2CMasterBusy(MPU6050_I2C_BASE));
    for(i = 0; i < (count - 1); i++)
    {
        I2CMasterDataPut(MPU6050_I2C_BASE, *p8Data++);
        I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_BURST_SEND_CONT);
        while(I2CMasterBusy(MPU6050_I2C_BASE));
    }
    I2CMasterDataPut(MPU6050_I2C_BASE, *p8Data);
    I2CMasterControl(MPU6050_I2C_BASE, I2C_MASTER_CMD_BURST_SEND_FINISH);
    while(I2CMasterBusy(MPU6050_I2C_BASE));

}
