#include <CUST_PHY_tja1101.h>
#include <custom_phy.h>

#define CUST_PHY_TJA1101_IEEE_IDENT                      (0x0180DD00u)
#define CUST_PHY_TJA1101_IEEE_IDENT_MASK                 (0xFFFFFFF0u)
#define CUST_PHY_TJA1101_AUTONEGTIMEOUT                  (1000u)

#define ETHPHY_TJA1101_BMCR_REG                          (0)
#define ETHPHY_TJA1101_BMSR_REG                          (1)
#define ETHPHY_TJA1101_EXSR_REG                          (15)

#define ETHPHY_TJA1101_MII_ECTRL_REG                     (17)
#define ETHPHY_TJA1101_MII_ECTRL_CONFIG_EN               ETHPHY_BIT(2)

#define ETHPHY_TJA1101_CFG1_REG                          (18)
#define ETHPHY_TJA1101_CFG2_REG                          (19)

#define ETHPHY_TJA1101_MII_COMMSTS_REG                   (23)

#define ETHPHY_TJA1101_MII_COMMCFG_REG                   (27)
#define ETHPHY_TJA1101_MII_COMMCFG_MASTER                0x8000
#define ETHPHY_TJA1101_MII_COMMCFG_CLKMODE               (0x00003000U)
#define ETHPHY_TJA1101_MII_COMMCFG_CLKMODE_25MHZ_CLK_OUT 0x1000
#define ETHPHY_TJA1101_MII_COMMCFG_CLKMODE_25MHZ_EXT_CLK 0x2000

/* DP8 global */
#define CLEARREGBIT(reg, bitNum)           (reg) = ((reg) & ~(1 << (bitNum)))

#define SETREGBIT(reg, bitNum)             (reg) = ((reg) | (1 << (bitNum)))

#define SETREGVAL(reg, shift, mask, value) (reg) = (((reg) & (~((mask) << (shift)))) | ((value) << (shift)))

static void CUST_PHY_TJA1101_softwareReset(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_softwareRestart(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_enableAutoMDIX(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_setMIIMode(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_configMLED(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_enableExtFD(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_enableODDNibbleDet(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_enableRxErrIdle(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_configLed(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_configLedBlink(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_enableFastLinkDownDet(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_enableFastRXDVDet(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_cofigSwStrapDone(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_setPowerMode(void *pAppCtxt_p, void *pStackCtxt_p, bool powerDown_p);
static bool CUST_PHY_TJA1101_getPowerMode(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_setLinkConfig(void *pAppCtxt_p, void *pStackCtxt_p, bool autoNeg_p, uint16_t linkSpeed_p, bool fullDuplex_p, uint32_t *pResult_p);
static bool CUST_PHY_TJA1101_getAutoNegotiation(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_setMdixMode(void *pAppCtxt_p, void *pStackCtxt_p, uint32_t mdixMode_p);
static uint32_t CUST_PHY_TJA1101_getMdixMode(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_disable1GbAdver(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_rgmiiLowLatencyEnable(void *pAppCtxt_p, void *pStackCtxt_p);
static void CUST_PHY_TJA1101_rgmiiTxHalfFullThreshold(void *pAppCtxt_p, void *pStackCtxt_p, uint32_t threshold_p);
static void CUST_PHY_TJA1101_rgmiiRxHalfFullThreshold(void *pAppCtxt_p, void *pStackCtxt_p, uint32_t threshold_p);
static void CUST_PHY_TJA1101_getSpeedDuplex(void *pAppCtxt_p, void *pStackCtxt_p, void *pData_p, uint32_t dataSize_p);
static uint32_t CUST_PHY_TJA1101_openFxn(void *pAppCtxt_p, void *pStackCtxt_p, void *pParam_p);
static uint32_t CUST_PHY_TJA1101_commandFxn(void *pAppCtxt_p, void *pStackCtxt_p, uint32_t command_p, void *pData_p, uint32_t dataSize_p);
static void CUST_PHY_TJA1101_closeFxn(void *pAppCtxt_p, void *pStackCtxt_p);

/*! <!-- Description: -->
 *
 *  \brief
 *  Detect Phy Type and setup access structures accordingly
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pPhyLibCtxt_p	Context of External PhyLib.
 *  \param[in]  phyId_p         Phy ID read from hardware
 *  \param[in]  pPhyLibDesc_p   External PhyLib Hooks
 *  \return     0 on success and Phy detected, error code otherwise
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */

int16_t CUST_PHY_TJA1101_detect(void *pPhyLibCtxt_p, uint32_t phyId_p, CUST_PHY_SPhyDescriptor_t *pPhyLibDesc_p)
{
    int16_t retVal = -CUST_PHY_STATUS_ERROR_UNKNOWN_PHY;

    /* exact match */
    if ((phyId_p & CUST_PHY_TJA1101_IEEE_IDENT_MASK) == CUST_PHY_TJA1101_IEEE_IDENT)
    {
        OSAL_printf("TJA1101 detected\r\n");
        pPhyLibDesc_p->softwareReset = CUST_PHY_TJA1101_softwareReset;
        pPhyLibDesc_p->softwareRestart = CUST_PHY_TJA1101_softwareRestart;
        pPhyLibDesc_p->enableAutoMDIX = CUST_PHY_TJA1101_enableAutoMDIX;
        pPhyLibDesc_p->setMiiMode = CUST_PHY_TJA1101_setMIIMode;
        pPhyLibDesc_p->setPowerMode = CUST_PHY_TJA1101_setPowerMode;
        pPhyLibDesc_p->getPowerMode = CUST_PHY_TJA1101_getPowerMode;
        pPhyLibDesc_p->configMLED = CUST_PHY_TJA1101_configMLED;
        pPhyLibDesc_p->enableExtFD = CUST_PHY_TJA1101_enableExtFD;
        pPhyLibDesc_p->enableODDNibbleDet = CUST_PHY_TJA1101_enableODDNibbleDet;
        pPhyLibDesc_p->enableRxErrIdle = CUST_PHY_TJA1101_enableRxErrIdle;
        pPhyLibDesc_p->configLed = CUST_PHY_TJA1101_configLed;
        pPhyLibDesc_p->configLedBlink = CUST_PHY_TJA1101_configLedBlink;
        pPhyLibDesc_p->enableFastLinkDownDet = CUST_PHY_TJA1101_enableFastLinkDownDet;
        pPhyLibDesc_p->enableFastRXDVDet = CUST_PHY_TJA1101_enableFastRXDVDet;
        pPhyLibDesc_p->configSwStrapDone = CUST_PHY_TJA1101_cofigSwStrapDone;
        pPhyLibDesc_p->setLinkConfig = CUST_PHY_TJA1101_setLinkConfig;
        pPhyLibDesc_p->getAutoNegotiation = CUST_PHY_TJA1101_getAutoNegotiation;
        pPhyLibDesc_p->setMdixMode = CUST_PHY_TJA1101_setMdixMode;
        pPhyLibDesc_p->getMdixMode = CUST_PHY_TJA1101_getMdixMode;
        pPhyLibDesc_p->disable1GbAdver = CUST_PHY_TJA1101_disable1GbAdver;
        pPhyLibDesc_p->rgmiiLowLatencyEnable = CUST_PHY_TJA1101_rgmiiLowLatencyEnable;
        pPhyLibDesc_p->rgmiiTxHalfFullThreshold = CUST_PHY_TJA1101_rgmiiTxHalfFullThreshold;
        pPhyLibDesc_p->rgmiiRxHalfFullThreshold = CUST_PHY_TJA1101_rgmiiRxHalfFullThreshold;
        pPhyLibDesc_p->getSpeedDuplex = CUST_PHY_TJA1101_getSpeedDuplex;
        pPhyLibDesc_p->openFxn = CUST_PHY_TJA1101_openFxn;
        pPhyLibDesc_p->commandFxn = CUST_PHY_TJA1101_commandFxn;
        pPhyLibDesc_p->closeFxn = CUST_PHY_TJA1101_closeFxn;

        retVal = 0;
    }
    else
    {
        /*PG: Implement NXP PHY */
    }

    return retVal;
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Issue PHY Reset by software (used if no Hard Reset is available)
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p      application context (not used)
 *  \param[in]  pStackCtxt_p    stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_softwareReset(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Issue PHY Restart by software (register left intact)
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p      application context (not used)
 *  \param[in]  pStackCtxt_p    stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_softwareRestart(void *pAppCtxt_p, void *pStackCtxt_p)
{
    uint16_t phyRegVal = 0;
    bool hadLink = false;

    OSALUNREF_PARM(pAppCtxt_p);

    // status = MDIO_phyRegRead(attrs->mdioBaseAddress, NULL, attrs->phyAddress, ETHPHY_TJA1101_MII_ECTRL_REG, &phyRegVal);
    CUST_PHY_readReg(pStackCtxt_p, ETHPHY_TJA1101_MII_ECTRL_REG, &phyRegVal);

    phyRegVal |= ETHPHY_TJA1101_MII_ECTRL_CONFIG_EN;
    // status = MDIO_phyRegWrite(attrs->mdioBaseAddress, NULL, attrs->phyAddress, ETHPHY_TJA1101_MII_ECTRL_REG, phyRegVal);
    CUST_PHY_writeReg(pStackCtxt_p, ETHPHY_TJA1101_MII_ECTRL_REG, phyRegVal);

    // Set Clock mode

    // status = MDIO_phyRegRead(attrs->mdioBaseAddress, NULL, attrs->phyAddress, ETHPHY_TJA1101_MII_COMMCFG_REG, &phyRegVal);
    CUST_PHY_readReg(pStackCtxt_p, ETHPHY_TJA1101_MII_COMMCFG_REG, &phyRegVal);
    phyRegVal &= (~ETHPHY_TJA1101_MII_COMMCFG_CLKMODE);
    // phyRegVal |= ETHPHY_TJA1101_MII_COMMCFG_CLKMODE_25MHZ_CLK_OUT;
    phyRegVal |= ETHPHY_TJA1101_MII_COMMCFG_CLKMODE_25MHZ_EXT_CLK;
    // status = MDIO_phyRegWrite(attrs->mdioBaseAddress, NULL, attrs->phyAddress, ETHPHY_TJA1101_MII_COMMCFG_REG, phyRegVal);
    CUST_PHY_writeReg(pStackCtxt_p, ETHPHY_TJA1101_MII_COMMCFG_REG, phyRegVal);

    // Set Master mode

    // status = MDIO_phyRegRead(attrs->mdioBaseAddress, NULL, attrs->phyAddress, ETHPHY_TJA1101_CFG1_REG, &phyRegVal);
    CUST_PHY_readReg(pStackCtxt_p, ETHPHY_TJA1101_CFG1_REG, &phyRegVal);
    phyRegVal |= ETHPHY_TJA1101_MII_COMMCFG_MASTER;
    // status = MDIO_phyRegWrite(attrs->mdioBaseAddress, NULL, attrs->phyAddress, ETHPHY_TJA1101_CFG1_REG, phyRegVal);
    CUST_PHY_writeReg(pStackCtxt_p, ETHPHY_TJA1101_CFG1_REG, phyRegVal);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Enable Auto MDIX
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_enableAutoMDIX(void *pAppCtxt_p, void *pStackCtxt_p)
{

    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Set MII Mode
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context (not used)
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_setMIIMode(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Configure PhyMLED to detect RxLink by MLED (e.g. TLK)
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_configMLED(void *pAppCtxt_p, void *pStackCtxt_p)
{

    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
#if 0
    /* LED 0 */
    phyRegVal = CUST_PHY_TJA1101_MLED_REG_LINK_LOST;
    CUST_PHY_writeReg(pStackCtxt_p, CUST_PHY_TJA1101_MLED_REG, phyRegVal);

    CUST_PHY_readReg(pStackCtxt_p, CUST_PHY_TJA1101_LEDCFG_REG, &phyRegVal);

    /* LED 1 */
    SETREGVAL(phyRegVal, CUST_PHY_TJA1101_LEDCFG_REG_LED1CTL_SHIFT,
                                                       CUST_PHY_TJA1101_LEDCFG_REG_LEDCTL_MASK,
                                                       CUST_PHY_TJA1101_MLED_REG_LINK_OK);

    /* LED 2 */
    SETREGVAL(phyRegVal, CUST_PHY_TJA1101_LEDCFG_REG_LED1CTL_SHIFT,
                                                       CUST_PHY_TJA1101_LEDCFG_REG_LEDCTL_MASK,
                                                       CUST_PHY_TJA1101_MLED_REG_LINK_OK_BLINK_ACT);

    /* LED 3 */
    SETREGVAL(phyRegVal, CUST_PHY_TJA1101_LEDCFG_REG_LED1CTL_SHIFT,
                                                       CUST_PHY_TJA1101_LEDCFG_REG_LEDCTL_MASK,
                                                       CUST_PHY_TJA1101_MLED_REG_MII_LINK);

    CUST_PHY_writeReg(pStackCtxt_p, CUST_PHY_TJA1101_LEDCFG_REG, phyRegVal);
#endif
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Set Ext Full Duplex enable
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_enableExtFD(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Set ODD Nibble detection enable
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_enableODDNibbleDet(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Set Rx Error Idle enable
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_enableRxErrIdle(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Configure PHY LEDs
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_configLed(void *pAppCtxt_p, void *pStackCtxt_p)
{

    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
#if 0
    CUST_PHY_readReg(pStackCtxt_p, CUST_PHY_TJA1101_LEDCR_REG, &phyRegVal);

    phyRegVal |= CUST_PHY_TJA1101_LEDCR_LINKLEDPOLARITY_HIGH;

    CUST_PHY_writeReg(pStackCtxt_p, CUST_PHY_TJA1101_LEDCR_REG, phyRegVal);
#endif
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Configure PHY Blink LED mode
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_configLedBlink(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
#if 0
    CUST_PHY_readReg(pStackCtxt_p, CUST_PHY_TJA1101_LEDCR_REG, &phyRegVal);

    switch (value)
    {
    case CUST_PHY_TJA1101_LED_BLINK_500: SETREGVAL(phyRegVal, CUST_PHY_TJA1101_LEDCR_LEDBLINK_SHIFT,
                                                   CUST_PHY_TJA1101_LEDCR_LEDBLINK_MASK,
                                                   CUST_PHY_TJA1101_LEDCR_LEDBLINK_500MS); break;
    case CUST_PHY_TJA1101_LED_BLINK_200: SETREGVAL(phyRegVal, CUST_PHY_TJA1101_LEDCR_LEDBLINK_SHIFT,
                                                   CUST_PHY_TJA1101_LEDCR_LEDBLINK_MASK,
                                                   CUST_PHY_TJA1101_LEDCR_LEDBLINK_200MS); break;
    case CUST_PHY_TJA1101_LED_BLINK_100: SETREGVAL(phyRegVal, CUST_PHY_TJA1101_LEDCR_LEDBLINK_SHIFT,
                                                   CUST_PHY_TJA1101_LEDCR_LEDBLINK_MASK,
                                                   CUST_PHY_TJA1101_LEDCR_LEDBLINK_100MS); break;
    case CUST_PHY_TJA1101_LED_BLINK_50:  SETREGVAL(phyRegVal, CUST_PHY_TJA1101_LEDCR_LEDBLINK_SHIFT,
                                                   CUST_PHY_TJA1101_LEDCR_LEDBLINK_MASK,
                                                   CUST_PHY_TJA1101_LEDCR_LEDBLINK_50MS); break;
    }

    CUST_PHY_writeReg(pStackCtxt_p, CUST_PHY_TJA1101_LEDCR_REG, phyRegVal);
#endif
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Set fast link down Detection enable
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_enableFastLinkDownDet(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Set Fast RX DV detection enable
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_enableFastRXDVDet(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Set SW Strap config done
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p         application context (not used)
 *  \param[in]  pStackCtxt_p       stack context (not used)
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_cofigSwStrapDone(void *pAppCtxt_p, void *pStackCtxt_p)
{
    /* no SW strap on DP 83869 */
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Set power mode
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p      application context (not used)
 *  \param[in]  pStackCtxt_p    stack context
 *  \param[in]  powerDown_p     False = Normal Operation - True = Power Down
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_setPowerMode(void *pAppCtxt_p, void *pStackCtxt_p, bool powerDown_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
    OSALUNREF_PARM(powerDown_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Get power down
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p      application context (not used)
 *  \param[in]  pStackCtxt_p    stack context
 *
 *  \return False = Normal Operation - True = Power Down
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
bool CUST_PHY_TJA1101_getPowerMode(void *pAppCtxt_p, void *pStackCtxt_p)
{

    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);

    return false;
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Configures PHY for link connection.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p          application context
 *  \param[in]  pStackCtxt_p        stack context
 *  \param[in]  autoNeg_p           selected auto-negotiation mode (false - auto-negotiation inactive, true - auto-negotiation active)
 *  \param[in]  linkSpeed_p         required link speed (10, 100 or 1000 Mbps)
 *  \param[in]  fullDuplex_p        required duplex mode (false - half-duplex, true - full duplex)
 *  \param[out] pResult_p           pointer to variable where result of configuration needs to be stored (0 - Success, other - Error)
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_setLinkConfig(void *pAppCtxt_p, void *pStackCtxt_p, bool autoNeg_p, uint16_t linkSpeed_p, bool fullDuplex_p, uint32_t *pResult_p)
{
    uint32_t error = 0;

    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
    OSALUNREF_PARM(autoNeg_p);
    OSALUNREF_PARM(linkSpeed_p);
    OSALUNREF_PARM(fullDuplex_p);
#if 0
    CUST_PHY_TJA1101_setPowerMode(pAppCtxt_p, pStackCtxt_p, true);

    if (autoNeg_p == true)
    {
        CUST_PHY_readReg(pStackCtxt_p, CUST_PHY_TJA1101_ANAR_REG, &phyRegVal);

        phyRegVal |= CUST_PHY_TJA1101_ANAR_100FD_SUPPORT;
        phyRegVal |= CUST_PHY_TJA1101_ANAR_100HD_SUPPORT;
        phyRegVal |= CUST_PHY_TJA1101_ANAR_10FD_SUPPORT;
        phyRegVal |= CUST_PHY_TJA1101_ANAR_10HD_SUPPORT;

        CUST_PHY_writeReg(pStackCtxt_p, CUST_PHY_TJA1101_ANAR_REG, phyRegVal);

        CUST_PHY_readReg(pStackCtxt_p, CUST_PHY_TJA1101_BMCR_REG, &phyRegVal);

        phyRegVal |=  CUST_PHY_TJA1101_BMCR_REG_ANEG_ENABLE |
                      CUST_PHY_TJA1101_BMCR_REG_ANEG_RESTART;

        CUST_PHY_writeReg(pStackCtxt_p, CUST_PHY_TJA1101_BMCR_REG, phyRegVal);
    }
    else
    {
        CUST_PHY_readReg(pStackCtxt_p, CUST_PHY_TJA1101_BMCR_REG, &phyRegVal);

        phyRegVal &= ~CUST_PHY_TJA1101_BMCR_REG_ANEG_ENABLE;
        phyRegVal &= ~CUST_PHY_TJA1101_BMCR_REG_SPEED_SEL_LSB;
        phyRegVal &= ~CUST_PHY_TJA1101_BMCR_REG_DUPLEX_EN;
        phyRegVal &= ~CUST_PHY_TJA1101_BMCR_REG_SPEED_SEL_MSB;

        switch(linkSpeed_p)
        {
            case 10:
                if (fullDuplex_p == true)
                {
                    phyRegVal |= CUST_PHY_TJA1101_BMCR_REG_DUPLEX_EN;
                }
                break;
            case 100:
                phyRegVal |= CUST_PHY_TJA1101_BMCR_REG_SPEED_SEL_LSB;

                // duplex mode
                if (fullDuplex_p == true)
                {
                    phyRegVal |= CUST_PHY_TJA1101_BMCR_REG_DUPLEX_EN;
                }
                break;
            default:
                // 1000 Mbps not supported (PRU-CODE limitation)
                error = ~0;
                break;
        }

        if (error == 0)
        {
            CUST_PHY_writeReg(pStackCtxt_p, CUST_PHY_TJA1101_BMCR_REG, phyRegVal);
        }
    }

    OSAL_waitTimerUs(10);

    CUST_PHY_TJA1101_setPowerMode(pAppCtxt_p, pStackCtxt_p, false);
#endif
    if (pResult_p != NULL)
    {
        *pResult_p = error;
    }
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Provides auto-negotiation enabled flag.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p          application context (not used)
 *  \param[in]  pStackCtxt_p        stack context
 *  \return     bool                auto-negotiation enabled flag (false - auto-negotiation disabled, true - auto-negotiation enabled)
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
bool CUST_PHY_TJA1101_getAutoNegotiation(void *pAppCtxt_p, void *pStackCtxt_p)
{

    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);

    return false;
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Configures PHY MDI crossover mode.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p          application context (not used)
 *  \param[in]  pStackCtxt_p        stack context
 *  \param[in]  mdixMode_p          required MDI crossover mode (0 - Manual MDI configuration, 1 - Manual MDI-X configuration, 2 - Enable automatic crossover)
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_setMdixMode(void *pAppCtxt_p, void *pStackCtxt_p, uint32_t mdixMode_p)
{

    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
    OSALUNREF_PARM(mdixMode_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Provides actually used PHY MDI crossover mode.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p          application context (not used)
 *  \param[in]  pStackCtxt_p        stack context
 *  \return     uint32_t            value of used MDI crossover mode (0 - Manual MDI configuration, 1 - Manual MDI-X configuration, 2 - Enable automatic crossover)
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
uint32_t CUST_PHY_TJA1101_getMdixMode(void *pAppCtxt_p, void *pStackCtxt_p)
{
    uint32_t mdixMode = CUST_PHY_MDIX_AUTO_CROSSOVER;

    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
#if 0
    CUST_PHY_readReg(pStackCtxt_p, CUST_PHY_TJA1101_PHYCR_REG, &phyRegVal);

    phyRegVal = phyRegVal & (CUST_PHY_TJA1101_PHYCR_REG_MDI_MANUAL_CROSSOVER | CUST_PHY_TJA1101_PHYCR_REG_MDI_AUTO_CROSSOVER);
    phyRegVal = phyRegVal >> 12;

    switch(phyRegVal)
    {
        case 0:
            mdixMode = CUST_PHY_MDI_MANUAL_CONFIG;
            break;
        case 1:
            mdixMode = CUST_PHY_MDIX_MANUAL_CONFIG;
            break;
        default:
            mdixMode = CUST_PHY_MDIX_AUTO_CROSSOVER;
            break;
    }
#endif
    return mdixMode;
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Disables 1Gbit Advertisment during autonegotiation.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p          application context (not used)
 *  \param[in]  pStackCtxt_p        stack context (not used)
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_disable1GbAdver(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Enables low latency in RGMII mode
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p          application context (not used)
 *  \param[in]  pStackCtxt_p        stack context (not used)
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_rgmiiLowLatencyEnable(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  RGMII_TX_HALF_FULL_THR[1:0] can be changed from the default of 0x2 to 0x1.
 *    This will reduce the latency by one clock period.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p          application context (not used)
 *  \param[in]  pStackCtxt_p        stack context (not used)
 *  \param[in]  threshold_p         Number of cycles for threshold (2, 1) (not used)
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_rgmiiTxHalfFullThreshold(void *pAppCtxt_p, void *pStackCtxt_p, uint32_t threshold_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
    OSALUNREF_PARM(threshold_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  RGMII_RX_HALF_FULL_THR[1:0] can be changed from the default of 0x2 to 0x1.
 *    This will reduce the latency by one clock period.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p          application context (not used)
 *  \param[in]  pStackCtxt_p        stack context (not used)
 *  \param[in]  threshold_p         Number of cycles for threshold (2, 1) (not used)
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_rgmiiRxHalfFullThreshold(void *pAppCtxt_p, void *pStackCtxt_p, uint32_t threshold_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
    OSALUNREF_PARM(threshold_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Provides actually used link speed and duplex mode.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p          application context (not used)
 *  \param[in]  pStackCtxt_p        stack context
 *  \param[in]  pData_p             pointer to data structure which contains link speed and duplex mode values
 *  \param[in]  dataSize_p          size of data structure
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
void CUST_PHY_TJA1101_getSpeedDuplex(void *pAppCtxt_p, void *pStackCtxt_p, void *pData_p, uint32_t dataSize_p)
{

    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);
    OSALUNREF_PARM(pData_p);
    OSALUNREF_PARM(dataSize_p);
#if 0
    pSpeedDuplexCfg = (CUST_PHY_SSpeedDuplexConfig_t*) pData_p;

    if (pSpeedDuplexCfg == NULL)
    {
        return;
    }

    CUST_PHY_readReg(pStackCtxt_p, CUST_PHY_TJA1101_PHYSTS_REG, &phyRegVal);

    switch(phyRegVal & CUST_PHY_TJA1101_PHYSTS_REG_SPEED)
    {
        case CUST_PHY_TJA1101_PHYSTS_REG_SPEED_100M:
            if (phyRegVal & CUST_PHY_TJA1101_PHYSTS_REG_DUPLEX)
            {
                pSpeedDuplexCfg->config = CUST_PHY_SPEED_DUPLEX_eCONFIG_100FD;
            }
            else
            {
                pSpeedDuplexCfg->config = CUST_PHY_SPEED_DUPLEX_eCONFIG_100HD;
            }
            break;
        case CUST_PHY_TJA1101_PHYSTS_REG_SPEED_10M:
            if (phyRegVal & CUST_PHY_TJA1101_PHYSTS_REG_DUPLEX)
            {
                pSpeedDuplexCfg->config = CUST_PHY_SPEED_DUPLEX_eCONFIG_10FD;
            }
            else
            {
                pSpeedDuplexCfg->config = CUST_PHY_SPEED_DUPLEX_eCONFIG_10HD;
            }
            break;
        default:
            break;
    }
#endif
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Open PHY interface function to be able connect to TI ETHPHY interface.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p                   application context (not used)
 *  \param[in]  pStackCtxt_p                 stack context
 *  \param[in]  pParam_p                     parameters (not used)
 *  \return     uint32_t                     status
 *  \retval     CUST_PHY_eSTATUS_FAIL        function failed
 *  \retval     CUST_PHY_eSTATUS_SUCCESS     function success
 *
 *  <!-- Group: -->
 *
 *  \ingroup CUST_PHY
 *
 * */
uint32_t CUST_PHY_TJA1101_openFxn(void *pAppCtxt_p, void *pStackCtxt_p, void *pParam_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pParam_p);

    return CUST_PHY_open(pStackCtxt_p);
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Command PHY interface function to be able connect to TI ETHPHY interface.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p                   application context
 *  \param[in]  pStackCtxt_p                 stack context
 *  \param[in]  command_p                    command id
 *  \param[in]  pData_p                      data
 *  \param[in]  dataSize_p                   data size
 *  \return     uint32_t                     status
 *  \retval     CUST_PHY_eSTATUS_FAIL        function failed
 *  \retval     CUST_PHY_eSTATUS_SUCCESS     function success
 *
 *  <!-- Group: -->
 *
 *  \ingroup PRU
 *
 * */
uint32_t CUST_PHY_TJA1101_commandFxn(void *pAppCtxt_p, void *pStackCtxt_p, uint32_t command_p, void *pData_p, uint32_t dataSize_p)
{
    int32_t status = CUST_PHY_eSTATUS_SUCCESS;

    switch (command_p)
    {
    case CUST_PHY_eCOMMAND_GET_SPEED_AND_DUPLEX_CONFIG:
        CUST_PHY_TJA1101_getSpeedDuplex(pAppCtxt_p, pStackCtxt_p, pData_p, dataSize_p);
        break;
    default:
        status = CUST_PHY_eSTATUS_FAIL;
        break;
    }

    return status;
}

/*! <!-- Description: -->
 *
 *  \brief
 *  Close PHY interface function to be able connect to TI ETHPHY interface.
 *
 *  <!-- Parameters and return values: -->
 *
 *  \param[in]  pAppCtxt_p                  application context (not used)
 *  \param[in]  pStackCtxt_p                stack context (not used)
 *
 *  <!-- Group: -->
 *
 *  \ingroup PRU
 *
 * */
void CUST_PHY_TJA1101_closeFxn(void *pAppCtxt_p, void *pStackCtxt_p)
{
    OSALUNREF_PARM(pAppCtxt_p);
    OSALUNREF_PARM(pStackCtxt_p);

    return;
}

//*************************************************************************************************
