/*
 *  COPYRIGHT (c) 2023.
 *  Reflex Aerospace GmbH.
 *
 *  Unauthorized copying of this file, via any medium is strictly prohibited.
 */

/**
 * \file
 *  rfx_eeprom
 *
 * \brief
 * Implementation of APIs for internal flash emulated EEPROM (Bank 7, 128KB)
 * using TI-FEE drivers
 *
 * \details
 * # emulated EEPROM driver
 *
 * TMS570LC4357 has 128KB of Data Flash for Emulated EEPROM with ECC
 * It contains 32 sectors each of size 4KB
 * sector 0  - 0xF0200000(Low address) - 0xF0200FFF(high address)
 * sector 31 - 0xF021F000(Low address) - 0xF021FFFF(high address)
 * 
 * The EEPROM Emulation Flash bank is divided into two or more virtual sectors. Each virtual
 * sector (user configurable) is further partitioned into several data blocks (user configurable).
 * 
 * RFX configuration
 * * FEE operating frequency is 150 MHz
 * * One flash sector is assigned to one virtual sector
 * * 32 virtual sectors
 * * 16 blocks of size 4095
 * * ECC enabled
 *  
 * TMS570LC4357 Data sheet
 * https://www.ti.com/product/TMS570LC4357
 * 
 * The TI-FEE drivers provides routines to erase, program 
 * and verifies flash memory on Texas Instruments micro controllers
 * 
 * https://www.ti.com/lit/ug/spnu518/spnu518.pdf?ts=1691568127124&ref_url=https%253A%252F%252Fwww.google.com%252F
 *
 * ## Features
 *
 * This driver implement the following functions:
 * 
 * Initialization:
 *  * rfx_eeprom_init() (reference 4.10.1.1)
 *  
 * API:
 *  * rfx_eeprom_format()      (reference 4.10.1.13)
 *  * rfx_eeprom_read_async()  (reference 4.10.1.4)
 *  * rfx_eeprom_write_async() (reference 4.10.1.2)
 *  
 *  During operations, the driver:
 *  * Locks and unlocks a mutex to protect multi threaded access
 *  * TI_Fee_MainFunction() function is called at 
 *    regular intervals to finish operation
 *  * TI_Fee_GetJobResult() is called to check job status
 *  * TI_FeeErrorCode is called to get error code
 *  * Maximum Timeout is experimental or tested value. 
 *    Datasheet doesn't provide any details.
 *
 *
 * ## known behavior 
 * * Erasing Flash using F021 Flash API library doesn't show all the erased flash locations as 0xFFFFFFFF
 * https://e2e.ti.com/support/microcontrollers/arm-based-microcontrollers-group/arm-based-microcontrollers/f/arm-based-microcontrollers-forum/974767/tms570lc4357-erasing-flash-using-f021-flash-api-library-doesn-t-show-all-the-erased-flash-locations-as-0xffffffff/3611242?tisearch=e2e-sitesearch&keymatch=%22Erase%22%20AND%20%22not%22%20AND%20%22FF%22#3611242
 * * Always call rfx_eeprom_init() after rfx_eeprom_format()
 * https://e2e.ti.com/support/microcontrollers/arm-based-microcontrollers-group/arm-based-microcontrollers/f/arm-based-microcontrollers-forum/1209102/tms570lc4357-ti-fee-driver-reading-block-after-formatting-or-invalidating-gets-into-strange-mode/4562672#4562672
 * \{
 *  
 */

#pragma once

#include <rfx_status.h>

#include <ti_fee.h>

#include <stdint.h>


/**
 * @brief      rfx_eeprom_init
 * This function provides functionality 
 * for initializing the TI FEE module
 * 
 * Performance note: TI_Fee_MainFunction() function should be called at 
 * regular intervals to finish operation. 
 * Maximum Timeout is set to EEPROM_MAX_INIT_TIME_MS
 * 
 * @return     RFX_STATUS_SUCCESS or RFX_STATUS_ERROR
 */
rfx_status_t rfx_eeprom_init(void);

/**
 * @brief      rfx_eeprom_format
 * Function to Format internal flash EEPROM - Bank7, 128KB
 * always call rfx_eeprom_init() after rfx_eeprom_format()
 * 
 * Performance note: TI_Fee_MainFunction() function should be called at 
 * regular intervals to finish the format operation. 
 * Maximum Timeout is set to EEPROM_MAX_BANK_FORMAT_TIME_MS
 * 
 * @param[in]  format_key  The format key
 *             EEPROM_FORMAT_KEY_1 or EEPROM_FORMAT_KEY_2
 *             
 * @return     RFX_STATUS_SUCCESS or RFX_STATUS_ERROR
 */
rfx_status_t rfx_eeprom_format(uint32_t format_key);

/**
 * @brief      rfx_eeprom_read_async
 * This function provides functionality for 
 * reading data from a EEPROM block asynchronously.
 * 
 * Performance note: TI_Fee_MainFunction() function should be called at 
 * regular intervals to finish the Asynchronous Write operation.
 * Maximum Timeout is input value from user. Default or tested value
 * for block size of 4095 is EEPROM_MAX_ASYNC_BLOCK_READ_TIME_MS
 * 
 * @param[in]  blk_num     The block number
 *                         1 to TI_FEE_NUMBER_OF_BLOCKS
 *                         (defined in ti_fee_cfg.h)
 * @param[in]  blk_offset  Address offset inside the block
 * @param[in]  length      Number of bytes to read
 * @param      buf_r       Pointer to data buffer
 * @param[in]  timeout_ms  Timeout value for polling TI_Fee_MainFunction()
 *                         default EEPROM_MAX_ASYNC_BLOCK_READ_TIME_MS
 * 
 * @return     RFX_STATUS_SUCCESS or RFX_STATUS_ERROR
 */
rfx_status_t rfx_eeprom_read_async(const uint16_t blk_num, const uint16_t blk_offset, const uint16_t length, uint8_t *buf_r, const uint32_t timeout_ms);

/**
 * @brief      rfx_eeprom_write_async
 * This function initiates an Asynchronous Write operation to a Data Block.
 * 
 * Performance note: TI_Fee_MainFunction() function should be called at 
 * regular intervals to finish the Asynchronous Write operation.
 * Maximum Timeout is input value from user. Default or tested value
 * for block size of 4095 is EEPROM_MAX_ASYNC_BLOCK_WRITE_TIME_MS
 * 
 * @param[in]  blk_num     The block number
 *                         1 to TI_FEE_NUMBER_OF_BLOCKS
 *                         (defined in ti_fee_cfg.h)
 * @param[in]  buf_w       Pointer to data buffer
 * @param[in]  timeout_ms  Timeout value for polling TI_Fee_MainFunction()
 *                         default EEPROM_MAX_ASYNC_BLOCK_WRITE_TIME_MS
 *
 * @return     RFX_STATUS_SUCCESS or RFX_STATUS_ERROR
 */
rfx_status_t rfx_eeprom_write_async(const uint16_t blk_num, const uint8_t *buf_w, const uint32_t timeout_ms);
