/*
 *  COPYRIGHT (c) 2023.
 *  Reflex Aerospace GmbH.
 *
 *  Unauthorized copying of this file, via any medium is strictly prohibited.
 */

#include <rfx_eeprom.h>
#include <rfx_eeprom_def.h>

#include <rfx_assert.h>

#include <ra_dbg_level.h>
#include <ra_stdio.h>

/*************************************************************************
** Static functions
*************************************************************************/
/**
 * @brief      rfx_eeprom_check_std_result
 * Function to check read/write return value
 * @param[in]  result  Std_ReturnType 
 *
 * @return     RFX_STATUS_SUCCESS or RFX_STATUS_ERROR
 */
static rfx_status_t rfx_eeprom_check_std_result(Std_ReturnType result)
{
    if(result ==  E_OK)
    {
        RA_DEBUG_INFO_WP("E_OK \r\n");
    }
    else if(result ==  E_NOT_OK)
    {
        RA_DEBUG_ERROR_WP("E_NOT_OK \r\n");
        return  RFX_STATUS_ERROR;
    }
    else
    {
        RA_DEBUG_ERROR_WP("Undefined error code %u\r\n", result);
        return RFX_STATUS_ERROR;
    }

    return RFX_STATUS_SUCCESS;
}

/**
 * @brief      rfx_eeprom_check_job_result
 * This function returns the result of the last job synchronously
 * 
 * @return     RFX_STATUS_SUCCESS or RFX_STATUS_ERROR
 */
static rfx_status_t rfx_eeprom_check_job_result(void)
{
    TI_FeeJobResultType job_result = TI_Fee_GetJobResult(EEPROM_INDEX);

    if(job_result ==  JOB_OK)
    {
        RA_DEBUG_INFO_WP("JOB_OK \r\n");
    }
    else if(job_result ==  JOB_FAILED)
    {
        RA_DEBUG_ERROR_WP("JOB_FAILED \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(job_result ==  JOB_PENDING)
    {
        RA_DEBUG_ERROR_WP("JOB_PENDING \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(job_result ==  JOB_CANCELLED)
    {
        RA_DEBUG_ERROR_WP("JOB_CANCELLED \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(job_result ==  BLOCK_INCONSISTENT)
    {
        RA_DEBUG_ERROR_WP("BLOCK_INCONSISTENT \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(job_result ==  BLOCK_INVALID)
    {
        RA_DEBUG_ERROR_WP("BLOCK_INVALID \r\n");
        return RFX_STATUS_ERROR;
    }
    else
    {
        RA_DEBUG_INFO_WP("Undefined error code %u\r\n", job_result);
        return RFX_STATUS_ERROR;
    }

    return RFX_STATUS_SUCCESS;
}

/**
 * @brief      rfx_eeprom_get_error_code
 * This function provides functionality to identify occurrence of an error
 * 
 * @return     RFX_STATUS_SUCCESS or RFX_STATUS_ERROR
 */
static rfx_status_t rfx_eeprom_get_error_code(void)
{
    TI_Fee_ErrorCodeType error_code =  TI_FeeErrorCode(EEPROM_INDEX);

    if(error_code == Error_Nil)
    {
        RA_DEBUG_INFO_WP("Error_Nil \r\n");
        return RFX_STATUS_SUCCESS;
    }
    else if(error_code == Error_TwoActiveVS)
    {
        RA_DEBUG_ERROR_WP(" Error_TwoActiveVS \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_TwoCopyVS)
    {
        RA_DEBUG_ERROR_WP(" Error_TwoCopyVS \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_SetupStateMachine)
    {
        RA_DEBUG_ERROR_WP("Error_SetupStateMachine \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_CopyButNoActiveVS)
    {
        RA_DEBUG_ERROR_WP("Error_CopyButNoActiveVS \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_NoActiveVS)
    {
        RA_DEBUG_ERROR_WP("Error_NoActiveVS \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_BlockInvalid)
    {
        RA_DEBUG_ERROR_WP("Error_BlockInvalid \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_NullDataPtr)
    {
        RA_DEBUG_ERROR_WP("Error_NullDataPtr \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_NoFreeVS)
    {
        RA_DEBUG_ERROR_WP("Error_NoFreeVS \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_InvalidVirtualSectorParameter)
    {
        RA_DEBUG_ERROR_WP("Error_InvalidVirtualSectorParameter \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_ExceedSectorOnBank)
    {
        RA_DEBUG_ERROR_WP("Error_ExceedSectorOnBank \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_EraseVS)
    {
        RA_DEBUG_ERROR_WP("Error_EraseVS \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_BlockOffsetGtBlockSize)
    {
        RA_DEBUG_ERROR_WP("Error_BlockOffsetGtBlockSize \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_LengthParam)
    {
        RA_DEBUG_ERROR_WP("Error_LengthParam \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_FeeUninit)
    {
        RA_DEBUG_ERROR_WP("Error_FeeUninit \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_Suspend)
    {
        RA_DEBUG_ERROR_WP(" Error_Suspend \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_InvalidBlockIndex)
    {
        RA_DEBUG_ERROR_WP("Error_InvalidBlockIndex \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_NoErase)
    {
        RA_DEBUG_ERROR_WP("Error_NoErase \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_CurrentAddress)
    {
        RA_DEBUG_ERROR_WP("Error_CurrentAddress \r\n");
        return RFX_STATUS_ERROR;
    }
    else if(error_code == Error_Exceed_No_Of_DataSets)
    {
        RA_DEBUG_ERROR_WP("Error_Exceed_No_Of_DataSets \r\n");
        return RFX_STATUS_ERROR;
    }
    else
    {
        RA_DEBUG_INFO_WP("Undefined error code %u\r\n", error_code);
        return RFX_STATUS_ERROR;
    }
}

/**
 * @brief      rfx_eeprom_check_operation_status
 * This function calls TI_Fee_MainFunction() at regular intervals 
 * to finish the operation with user defined timeout
 * @param[in]  wait_ms  The wait milliseconds
 *
 * @return     RFX_STATUS_SUCCESS or RFX_STATUS_ERROR
 */
static rfx_status_t rfx_eeprom_check_operation_status(const uint32_t wait_ms)
{
    do
    {
        TI_Fee_MainFunction();
    }while (TI_Fee_GetStatus(0) != IDLE);

    return RFX_STATUS_SUCCESS;
}

/*************************************************************************
** Global functions
*************************************************************************/
rfx_status_t rfx_eeprom_init(void)
{
    TI_Fee_Init();
    
    // TI_Fee_MainFunction function should be
    // called at regular intervals to finish the operation
    // reference 4.8.4
    rfx_status_t return_value = RFX_STATUS_SUCCESS;
    if(rfx_eeprom_check_operation_status(EEPROM_MAX_INIT_TIME_MS) != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR_WP("rfx_eeprom_check_operation_status() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    // Check job status and get error code
    if(rfx_eeprom_check_job_result() != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR("rfx_eeprom_check_job_result() failed \r\n");
        return_value =  RFX_STATUS_ERROR;
    }

    if(rfx_eeprom_get_error_code() != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR("rfx_eeprom_get_error_code() failed \r\n");
        return_value =  RFX_STATUS_ERROR;
    }

    return return_value;
}

rfx_status_t rfx_eeprom_format(uint32_t format_key)
{
    rfx_status_t return_value = RFX_STATUS_SUCCESS;

    if(TI_Fee_Format(format_key) == TRUE)
    {
        RA_DEBUG_ERROR("BANK 7 Format unsuccessful \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    // TI_Fee_MainFunction function should be
    // called at regular intervals to finish the operation
    // reference 4.8.4
    if(rfx_eeprom_check_operation_status(EEPROM_MAX_BANK_FORMAT_TIME_MS) != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR_WP("rfx_eeprom_check_operation_status() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    // Check job status and get error code
    if(rfx_eeprom_check_job_result() != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR("rfx_eeprom_check_job_result() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    if(rfx_eeprom_get_error_code() != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR("rfx_eeprom_get_error_code() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    if(return_value == RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_INFO_WP("BANK7 format successful \r\n");
    }

    return return_value;
}

rfx_status_t rfx_eeprom_read_async(const uint16_t blk_num, const uint16_t blk_offset, const uint16_t length, uint8_t *buf_r, const uint32_t timeout_ms)
{
    rfx_assert(buf_r != NULL);
    rfx_assert(blk_num <= TI_FEE_NUMBER_OF_BLOCKS);

    if(length == 0)
    {
        RA_DEBUG_ERROR("length == 0 \r\n");
        return RFX_STATUS_ERROR;

    }
    
    if((blk_num == 0) || (blk_num > TI_FEE_NUMBER_OF_BLOCKS))
    {
        RA_DEBUG_ERROR("Invalid block number \r\n");
        return RFX_STATUS_ERROR;
    }

    /* Read the block */
    RA_DEBUG_INFO_WP("Asynchronous read data set of block:%u, offset:%u and length:%u ..... \r\n", blk_num, blk_offset, length);
    Std_ReturnType std_return_value = TI_Fee_Read((uint16_t)blk_num, (uint16_t)blk_offset, buf_r, (uint16_t)length);
    
    // TI_Fee_MainFunction function should be
    // called at regular intervals to finish the operation
    // reference 4.8.4
    rfx_status_t return_value = RFX_STATUS_SUCCESS;
    if(rfx_eeprom_check_operation_status(EEPROM_MAX_ASYNC_BLOCK_READ_TIME_MS) != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR_WP("rfx_eeprom_check_operation_status() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    // Check read operation result
    if(rfx_eeprom_check_std_result(std_return_value) != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR_WP("rfx_eeprom_check_std_result() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    // Check job status and get error code
    if(rfx_eeprom_check_job_result() != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR_WP("rfx_eeprom_check_job_result() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    if(rfx_eeprom_get_error_code() != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR_WP("rfx_eeprom_get_error_code() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    if(return_value == RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_INFO_WP("BANK7 Asynchronous read done! \r\n");
    }
    else
    {
        RA_DEBUG_ERROR_WP("BANK7 Asynchronous read failed \r\n");
    }

    return return_value;
}

rfx_status_t rfx_eeprom_write_async(const uint16_t blk_num, const uint8_t *buf_w, const uint32_t timeout_ms)
{
    rfx_assert(buf_w != NULL);
    rfx_assert(blk_num <= TI_FEE_NUMBER_OF_BLOCKS);

    if((blk_num == 0) || (blk_num > TI_FEE_NUMBER_OF_BLOCKS))
    {
        RA_DEBUG_ERROR("Invalid block number \r\n");
        return RFX_STATUS_ERROR;
    }

    /* Write asynchronously the block */
    RA_DEBUG_INFO_WP("Asynchronous write data set of block:%u ..... \r\n", blk_num);
    Std_ReturnType std_return_value = TI_Fee_WriteAsync((uint16_t) blk_num, (uint8_t *)buf_w);

    // TI_Fee_MainFunction function should be
    // called at regular intervals to finish the operation
    // reference 4.8.4
    rfx_status_t return_value = RFX_STATUS_SUCCESS;
    if(rfx_eeprom_check_operation_status(EEPROM_MAX_ASYNC_BLOCK_WRITE_TIME_MS) != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR_WP("rfx_eeprom_check_operation_status() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    // Check read operation result
    if(rfx_eeprom_check_std_result(std_return_value) != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR_WP("rfx_eeprom_check_std_result() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    // Check job status and get error code
    if(rfx_eeprom_check_job_result() != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR_WP("rfx_eeprom_check_job_result() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    if(rfx_eeprom_get_error_code() != RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_ERROR_WP("rfx_eeprom_get_error_code() failed \r\n");
        return_value = RFX_STATUS_ERROR;
    }

    if(return_value == RFX_STATUS_SUCCESS)
    {
        RA_DEBUG_INFO_WP("BANK7 Asynchronous write done! \r\n");
    }
    else
    {
        RA_DEBUG_ERROR_WP("BANK7 Asynchronous write failed \r\n");
    }

    return return_value;
}
