/*
 * Copyright (c) 2015, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== empty.c ========
 */
/* XDCtools Header files */
#include <xdc/std.h>
#include <xdc/runtime/System.h>

/* BIOS Header files */
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>

/* TI-RTOS Header files */
// #include <ti/drivers/EMAC.h>
#include <ti/drivers/GPIO.h>
// #include <ti/drivers/I2C.h>
// #include <ti/drivers/SDSPI.h>
#include <ti/drivers/SPI.h>
// #include <ti/drivers/SSI.h>
// #include <ti/drivers/USBMSCHFatFs.h>
// #include <ti/drivers/Watchdog.h>
// #include <ti/drivers/WiFi.h>

#include <ti/sysbios/knl/Semaphore.h>
/* Board Header file */
#include "Board.h"
#include <ti/sysbios/knl/Swi.h>
#include <xdc/runtime/Error.h>
#include <ti/sysbios/family/arm/m3/Hwi.h>

#include <stdbool.h>
#include "driverlib/sysctl.h"
#include "driverlib/ssi.h"
#include "driverlib/udma.h"
#include "driverlib/interrupt.h"
#include <inc/hw_memmap.h>
#include <inc/hw_ints.h>
#include "inc/hw_ssi.h"
#include <stdbool.h>
#include <ti/sysbios/knl/Event.h>
#include <xdc/runtime/Error.h>

uint8_t pui8ControlTable[1024];

#define TASKSTACKSIZE   512

#define SSI_TXBUF_SIZE         1024
#define SSI_RXBUF_SIZE         64

#define CONTROL_TABLEBASE			0x20000400
#define SOURCE_ENDADDR              0x000000B0
#define DES_ENDADDR					0x000000B4
#define CONTROL_WORD                0x000000B8

#define SOURCE_ENDADDRR              0x000000A0
#define DES_ENDADDRR				 0x000000A4
#define CONTROL_WORDD                0x000000A8

Task_Struct task0Struct;
Char task0Stack[TASKSTACKSIZE];

uint8_t DMATxBufferpri[SSI_TXBUF_SIZE];
uint8_t DMATxBufferalt[SSI_TXBUF_SIZE];

uint8_t DMAPrimary_Buffer[SSI_RXBUF_SIZE];
uint8_t DMASecondary_Buffer[SSI_RXBUF_SIZE];

uint32_t g_ui32uDMAErrCount = 0;
uint32_t g_ui32BadISR = 0;
uint32_t ui32counter = 0;

uint8_t ui8SendBuf[SSI_TXBUF_SIZE];
uint8_t ui8ReceivedBuf[SSI_RXBUF_SIZE];



/* Error Block  Handler*/
Error_Block eb;
Event_Handle Event_Digital_Input_Handler;




void printreceivedbuf(uint8_t *data)
{
	uint8_t i = 0;

	for(i = 0;i < SSI_RXBUF_SIZE; i++)
	{
		System_printf("%x", *(data + i));
		System_flush();
	}

	System_printf("\n");

}

void uDMAErrorHandler(void)
{
    uint32_t ui32Status;

    //
    // Check for uDMA error bit
    //
    ui32Status = uDMAErrorStatusGet();

    //
    // If there is a uDMA error, then clear the error and increment
    // the error counter.
    //
    if(ui32Status)
    {
        uDMAErrorStatusClear();
    }
}

//*****************************************************************************
//
// The interrupt handler for SSI0.
//
//*****************************************************************************
void SSI0IntHandler(void)
{

    uint32_t ulMode;
    uint32_t ui32Status;
    //
	// Read the interrupt status of the SSI.
	//
	ui32Status = SSIIntStatus(SSI0_BASE, TRUE);


	//
	// Clear any pending status, even though there should be none since no SSI
	// interrupts were enabled.  If SSI error interrupts were enabled, then
	// those interrupts could occur here and should be handled.  Since uDMA is
	// used for both the RX and TX, then neither of those interrupts should be
	// enabled.
	//
	SSIIntClear(SSI0_BASE, ui32Status);


	//Checks the interrupt status for DMA Tx interrupt
	if(ui32Status & SSI_DMATX)
	{
		//check primary Tx channel
		ulMode = uDMAChannelModeGet(UDMA_CHANNEL_SSI0TX | UDMA_PRI_SELECT);
		if(ulMode == UDMA_MODE_STOP)
		{

			 uDMAChannelTransferSet(UDMA_CHANNEL_SSI0TX | UDMA_PRI_SELECT,
									 UDMA_MODE_PINGPONG, DMATxBufferpri,
									 (void *)(SSI0_BASE + SSI_O_DR), SSI_TXBUF_SIZE);

		}

		//check Alternate Tx channel
		ulMode = uDMAChannelModeGet(UDMA_CHANNEL_SSI0TX | UDMA_ALT_SELECT);
		if(ulMode == UDMA_MODE_STOP)  {

			uDMAChannelTransferSet(UDMA_CHANNEL_SSI0TX | UDMA_ALT_SELECT,
									UDMA_MODE_PINGPONG, DMATxBufferalt,
									(void *)(SSI0_BASE + SSI_O_DR), SSI_TXBUF_SIZE);
		}
	}


	//Checks the raw interrupt status for DMA Rx interrupt
	if(ui32Status & SSI_DMARX)
	{
		//check primary Rx channel
		ulMode = uDMAChannelModeGet( UDMA_CHANNEL_SSI0RX | UDMA_PRI_SELECT);
		if(ulMode == UDMA_MODE_STOP) {
			GPIO_toggle(Board_LED0);
		memcpy(ui8ReceivedBuf, DMAPrimary_Buffer, SSI_RXBUF_SIZE);
		uDMAChannelTransferSet(UDMA_CHANNEL_SSI0RX | UDMA_PRI_SELECT,
				UDMA_MODE_PINGPONG, (void *)(SSI0_BASE + SSI_O_DR),
				DMAPrimary_Buffer, SSI_RXBUF_SIZE);
	}


	//check Alternate Rx channel
	ulMode = uDMAChannelModeGet(UDMA_CHANNEL_SSI0RX | UDMA_ALT_SELECT);
	if(ulMode == UDMA_MODE_STOP)  {
		GPIO_toggle(Board_LED1);
	  memcpy(ui8ReceivedBuf, DMASecondary_Buffer, SSI_RXBUF_SIZE);
	  uDMAChannelTransferSet(UDMA_CHANNEL_SSI0RX | UDMA_ALT_SELECT,
			  UDMA_MODE_PINGPONG, (void *)(SSI0_BASE + SSI_O_DR),
			  DMASecondary_Buffer, SSI_RXBUF_SIZE);
		}
	}

//Counter used to see the number of times interrupt handled in ROV
	ui32counter++;

/*If i comment the below Event post, The code working fine. If i used the event post
  the interrupt bit cannot be cleared. */

//This post is used to handle the received data from master
	Event_post(Event_Digital_Input_Handler, Event_Id_00);
}


//*****************************************************************************
//
// Initializes the SSI0 transfer
//
//*****************************************************************************
void
InitSSI0Transfer(void)
{
    uint8_t i = 0;
	uint16_t ui8Idx = 0;

	/* Fill the TX buffer with a 1-64 simple data pattern. */
	for(ui8Idx = 0; ui8Idx < SSI_TXBUF_SIZE; ui8Idx++)
	{
		if((ui8Idx % 64) != 0)
		{
			ui8SendBuf[ui8Idx] = i+1;
			i++;
		}
		else
		{
			i = 0;
			ui8SendBuf[ui8Idx] = i+1;
			i++;
		}
	}
	memcpy(DMATxBufferpri, ui8SendBuf, SSI_TXBUF_SIZE);
	memcpy(DMATxBufferalt, ui8SendBuf, SSI_TXBUF_SIZE);

	/* Enable the SSI peripheral, and configure it to operate even
	 *  if the CPU is in sleep.
	 */
	SysCtlPeripheralEnable(SYSCTL_PERIPH_UDMA);
	SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_UDMA);

	 /* Enable the uDMA controller. */
	uDMAEnable();

	/* Point at the control table to use for channel control structures. */
	uDMAControlBaseSet(pui8ControlTable);

   /* Put the attributes in a known state for the uDMA SSI1RX channel.
    * These should already be disabled by default.
    */
//   uDMAChannelAttributeDisable(UDMA_CHANNEL_SSI0RX,UDMA_ATTR_USEBURST
//		   	   	   	   	   	   | UDMA_ATTR_REQMASK );
   uDMAChannelAttributeDisable(UDMA_CHANNEL_SSI0RX,UDMA_ATTR_USEBURST
		   	   	   	   	   	   | UDMA_ATTR_ALTSELECT | UDMA_ATTR_REQMASK );

   /* Configure the control parameters for the primary control structure for
    * the SSI RX channel.  The primary control structure is used for the "A"
    * part of the ping-pong receive.  The transfer data size is 8 bits, the
    * source address does not increment since it will be reading from a
    * register.  The destination address increment is byte 8-bit bytes.  The
    * arbitration size is set to 4 to match the RX FIFO trigger threshold.
    * The uDMA controller will use a 4 byte burst transfer if possible.  This
    * will be somewhat more efficient that single byte transfers.
    */
   uDMAChannelControlSet(UDMA_CHANNEL_SSI0RX | UDMA_PRI_SELECT,
                             UDMA_SIZE_8 | UDMA_SRC_INC_NONE | UDMA_DST_INC_8 |
							  UDMA_ARB_4);

   /* Set up the transfer parameters for the SSI RX primary control
    * structure.  The mode is set to ping-pong, the transfer source is the
    * SSI data register, and the destination is the receive "A" buffer.  The
    * transfer size is set to match the size of the buffer.
    */
    uDMAChannelTransferSet(UDMA_CHANNEL_SSI0RX | UDMA_PRI_SELECT,
                              UDMA_MODE_PINGPONG,
                              (void *)(SSI0_BASE + SSI_O_DR),
							   DMAPrimary_Buffer,SSI_RXBUF_SIZE);

   /* Configure the control parameters for the alternate control structure for
    * the SSI RX channel.  The alternate contol structure is used for the "B"
    * part of the ping-pong receive.  The configuration is identical to the
    * primary/A control structure.
    */
    uDMAChannelControlSet(UDMA_CHANNEL_SSI0RX | UDMA_ALT_SELECT,
                             UDMA_SIZE_8 | UDMA_SRC_INC_NONE | UDMA_DST_INC_8 |
							  UDMA_ARB_4);

   /* Set up the transfer parameters for the SSI RX alternate control
    * structure.  The mode is set to ping-pong, the transfer source is the
    * SSI data register, and the destination is the receive "B" buffer.  The
    * transfer size is set to match the size of the buffer.
    */
    uDMAChannelTransferSet(UDMA_CHANNEL_SSI0RX | UDMA_ALT_SELECT,
                              UDMA_MODE_PINGPONG,
                              (void *)(SSI0_BASE + SSI_O_DR),
							   DMASecondary_Buffer, SSI_RXBUF_SIZE);

    /* Put the attributes in a known state for the uDMA SSI1RX channel.  These
     * should already be disabled by default.
     */
//    uDMAChannelAttributeDisable(UDMA_CHANNEL_SSI0TX,UDMA_ATTR_USEBURST
//    						| UDMA_ATTR_REQMASK );

    uDMAChannelAttributeDisable(UDMA_CHANNEL_SSI0TX,UDMA_ATTR_USEBURST
    				| UDMA_ATTR_ALTSELECT | UDMA_ATTR_REQMASK );

	/* Configure the control parameters for the SSI TX.  The uDMA SSI TX
	 * channel is used to transfer a block of data from a buffer to the SSI.
	 * The data size is 8 bits.  The source address increment is 8-bit bytes
	 * since the data is coming from a buffer.  The destination increment is
	 * none since the data is to be written to the SSI data register.  The
	 * arbitration size is set to 4, which matches the SSI TX FIFO trigger
	 * threshold.
	 */
	uDMAChannelControlSet(UDMA_CHANNEL_SSI0TX | UDMA_PRI_SELECT,
							  UDMA_SIZE_8 | UDMA_SRC_INC_8 |
							  UDMA_DST_INC_NONE |
							  UDMA_ARB_4);

	/* Set up the transfer parameters for the uDMA SSI TX channel.  This will
	 * configure the transfer source and destination and the transfer size.
	 * Basic mode is used because the peripheral is making the uDMA transfer
	 * request.  The source is the TX buffer and the destination is the SSI
	 * data register.
	 */
	uDMAChannelTransferSet(UDMA_CHANNEL_SSI0TX | UDMA_PRI_SELECT,
							UDMA_MODE_PINGPONG, DMATxBufferpri,
							   (void *)(SSI0_BASE + SSI_O_DR),
							   SSI_TXBUF_SIZE );

	/* Configure the control parameters for the SSI TX.  The uDMA SSI TX
	 * channel is used to transfer a block of data from a buffer to the SSI.
	 * The data size is 8 bits.  The source address increment is 8-bit bytes
	 * since the data is coming from a buffer.  The destination increment is
	 * none since the data is to be written to the SSI data register.  The
	 * arbitration size is set to 4, which matches the SSI TX FIFO trigger
	 * threshold.
	 */
	uDMAChannelControlSet(UDMA_CHANNEL_SSI0TX | UDMA_ALT_SELECT,
							  UDMA_SIZE_8 | UDMA_SRC_INC_8 |
							  UDMA_DST_INC_NONE |
							  UDMA_ARB_4);

	/* Set up the transfer parameters for the uDMA SSI TX channel.  This will
	 * configure the transfer source and destination and the transfer size.
	 * Basic mode is used because the peripheral is making the uDMA transfer
	 * request.  The source is the TX buffer and the destination is the SSI
	 * data register.
	 */
	uDMAChannelTransferSet(UDMA_CHANNEL_SSI0TX | UDMA_ALT_SELECT,
							UDMA_MODE_PINGPONG, DMATxBufferalt,
							   (void *)(SSI0_BASE + SSI_O_DR),
							   SSI_TXBUF_SIZE );

    /* SSI 0 Configuration */
	SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI0);
	SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_SSI0);


    /* Configure the SSI communication parameters. */
    SSIConfigSetExpClk(SSI0_BASE, 120000000 /* SysCtlClockGet() */,
    					SSI_FRF_MOTO_MODE_0,
    					SSI_MODE_SLAVE, 4000000, 8);

    /* Enable the SSI for operation, and enable the uDMA interface for both TX
     * and RX channels.
     */
    SSIDMAEnable(SSI0_BASE, SSI_DMA_TX | SSI_DMA_RX );

    SSIClockSourceSet(SSI0_BASE, SSI_CLOCK_SYSTEM);

    /* Enable the SSI DMA TX/RX interrupts. */

	SSIIntEnable(SSI0_BASE, SSI_DMATX | SSI_DMARX);

    /* Now both the uDMA SSI TX and RX channels are primed to start a
     * transfer.  As soon as the channels are enabled, the peripheral will
     * issue a transfer request and the data transfers will begin.
     */

	uDMAChannelEnable(UDMA_CHANNEL_SSI0RX);
	uDMAChannelEnable(UDMA_CHANNEL_SSI0TX);

    SSIEnable(SSI0_BASE);
}


void ioc_RxTask(UArg arg0, UArg arg1)
{

	InitSSI0Transfer();
    while (1) {
//      Event_pend(Event_Digital_Input_Handler, Event_Id_00,
//    				   Event_Id_NONE, BIOS_WAIT_FOREVER);
    	Task_sleep(100);
       	System_printf("Counter increment in Task : %d\n", ui32counter);
       	System_flush();

    }
}


/*
 *  ======== main ========
 */
int main(void)
{
    Task_Params taskParams;
    /* Call board init functions */
    Board_initGeneral();
    Board_initGPIO();

    Board_initSPI();


	/* Error Block Handler */
	Error_init(&eb);

	/* Event created for ISR */
	/* Event create object for Digital Input ISR */
	Event_Digital_Input_Handler = Event_create(NULL, &eb);
	if (Event_Digital_Input_Handler == NULL)
	{
	System_abort("Event create failed");
	}

    /* Construct heartBeat Task  thread */
    Task_Params_init(&taskParams);
    taskParams.arg0 = 1000;
    taskParams.stackSize = TASKSTACKSIZE;
    taskParams.stack = &task0Stack;
    Task_construct(&task0Struct, (Task_FuncPtr)ioc_RxTask, &taskParams, NULL);

    /* Start BIOS */
    BIOS_start();

    return (0);
}
