//*****************************************************************************
//
// usb_host_serial.c - Main routines for the USB Host CDC serial example.
//
// Copyright (c) 2019 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include <string.h>
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "driverlib/debug.h"
#include "driverlib/gpio.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "driverlib/udma.h"
#include "usblib/usblib.h"
#include "usblib/usbcdc.h"
#include "usblib/host/usbhost.h"
#include "usblib/host/usbhcdc.h"
#include "utils/uartstdio.h"
#include "drivers/pinout.h"

//*****************************************************************************
//
//! \addtogroup example_list
//! <h1>USB Host CDC Serial example(usb_host_serial)</h1>
//!
//! This example application demonstrates data transfer between a CDC Host and
//! a CDC Serial Device using the EK-TM4C129XL evaluation kit.
//! To demonstrate the data transfer the CDC device example, "usb_dev_cdcserial",
//! should be run on another EK-TM4C129XL evaluation kit.  The data transferred
//! from the device is displayed on the UART terminal.
//!
//! The example automates the transfer of data between host and device.
//! Upon initial connection of the CDC device, the UART terminal will display
//! a certain number of 'Press Any Key' prompts and when the application
//! sends the character 'j' to the device, the device replies back with
//! 'Received the key press'.
//!
//! UART0, connected to the Virtual Serial Port and running at 115,200, 8-N-1,
//! is used to display messages from this application.
//!
//
//*****************************************************************************

//*****************************************************************************
//
// The number of SysTick ticks per second.
//
//*****************************************************************************
#define TICKS_PER_SECOND 100
#define MS_PER_SYSTICK (1000 / TICKS_PER_SECOND)

//*****************************************************************************
//
// Our running system tick counter and a global used to determine the time
// elapsed since last call to GetTickms().
//
//*****************************************************************************
uint32_t g_ui32SysTickCount;
uint32_t g_ui32LastTick;

//*****************************************************************************
//
// The size of the host controller's memory pool in bytes.
//
//*****************************************************************************
#define USB_TRANSFER_SIZE       64
#define USB_BUFFER_SIZE         1024
#define HCD_MEMORY_SIZE         128

//*****************************************************************************
//
// The memory pool to provide to the Host controller driver.
//
//*****************************************************************************
uint8_t g_pUSBPipeBuffer[USB_TRANSFER_SIZE];
uint8_t g_pUSBRxBuffer[USB_BUFFER_SIZE];
uint8_t g_pUSBTxBuffer[USB_BUFFER_SIZE];
uint8_t g_pHCDPool[HCD_MEMORY_SIZE];

//*****************************************************************************
//
// Declare the USB Events driver interface.
//
//*****************************************************************************
DECLARE_EVENT_DRIVER(g_sUSBEventDriver, 0, 0, USBHCDEvents);

//*****************************************************************************
//
// The global that holds all of the host drivers in use in the application.
// In this case, only the CDC class is loaded.
//
//*****************************************************************************
static tUSBHostClassDriver const * const g_ppHostClassDrivers[] =
{
    &g_sUSBCDCClassDriver,
    &g_sUSBEventDriver
};

//*****************************************************************************
//
// This global holds the number of class drivers in the g_ppHostClassDrivers
// list.
//
//*****************************************************************************
static const uint32_t g_ui32NumHostClassDrivers =
    sizeof(g_ppHostClassDrivers) / sizeof(tUSBHostClassDriver *);

//*****************************************************************************
//
// The global value used to store the CDC instance value.
//
//*****************************************************************************
static tCDCInstance *g_psCDCSerialInstance;

//*****************************************************************************
//
// This enumerated type is used to hold the states of the CDC device.
//
//*****************************************************************************
enum
{
    //
    // No device is present.
    //
    STATE_NO_DEVICE,

    //
    // CDC device has been detected and needs to be initialized in the main
    // loop.
    //
    STATE_CDC_DEVICE_INIT,

    //
    // CDC Device is connected and waiting for events.
    //
    STATE_CDC_DEVICE_CONNECTED,

    //
    // An unsupported device has been attached.
    //
    STATE_UNKNOWN_DEVICE,

    //
    // A power fault has occurred.
    //
    STATE_POWER_FAULT
}
g_eUSBState;

//*****************************************************************************
//
// Global variables used by the application to track sending and receiving
// of USB data in the USB callback.
//
//*****************************************************************************
volatile bool g_bSendData;
volatile bool g_bReceivedData;
volatile uint16_t g_ui16SentDataCount;
volatile uint16_t g_ui16ReceivedDataCount;

//*****************************************************************************
//
// The error routine that is called if the driver library encounters an error.
//
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, uint32_t ui32Line)
{
}
#endif

//*****************************************************************************
//
// This is the handler for this SysTick interrupt.
//
//*****************************************************************************
void
SysTickIntHandler(void)
{
    //
    // Update our tick counter.
    //
    g_ui32SysTickCount++;
}

//*****************************************************************************
//
// This function returns the number of ticks since the last time this function
// was called.
//
//*****************************************************************************
uint32_t
GetTickms(void)
{
    uint32_t ui32RetVal;
    uint32_t ui32Saved;

    ui32RetVal = g_ui32SysTickCount;
    ui32Saved = ui32RetVal;

    if(ui32Saved > g_ui32LastTick)
    {
        ui32RetVal = ui32Saved - g_ui32LastTick;
    }
    else
    {
        ui32RetVal = g_ui32LastTick - ui32Saved;
    }

    //
    // This could miss a few milliseconds but the timings here are on a
    // much larger scale.
    //
    g_ui32LastTick = ui32Saved;

    //
    // Return the number of milliseconds since the last time this was called.
    //
    return(ui32RetVal * MS_PER_SYSTICK);
}

//*****************************************************************************
//
// Simple function to fill the USB TX buffer with 1kB data for demonstration.
//
//*****************************************************************************

void
FillUSBTxBuffer(void)
{
    uint8_t ui8LoopCount1, ui8LoopCount2, ui8Count1, ui8Count2;
    uint8_t pui8Message[14] = {' ','U','S','B',' ','H', 'o','s','t',' ','C','D','C','\n' };


    ui8Count1 = 0;
    ui8Count2 = 0;

    for (ui8LoopCount1=0; ui8LoopCount1<64; ui8LoopCount1++)
    {
        g_pUSBTxBuffer[0 + (ui8LoopCount1*16)] = ui8Count2 + 0x30;
        g_pUSBTxBuffer[1 + (ui8LoopCount1*16)] = ui8Count1 + 0x30;

        for (ui8LoopCount2=2; ui8LoopCount2<16; ui8LoopCount2++)
        {
            g_pUSBTxBuffer[ui8LoopCount2 + (ui8LoopCount1*16)] = pui8Message[ui8LoopCount2-2];
        }

        if (ui8Count1 == 9)
        {
            ui8Count1 = 0;
            ui8Count2++;
        }
        else
        {
            ui8Count1++;
        }
    }
}

//*****************************************************************************
//
// This is the generic callback from host stack.
//
// \param pvData is actually a pointer to a tEventInfo structure.
//
// This function will be called to inform the application when a USB event has
// occurred that is outside those related to the CDC device.  At this
// point this is used to detect unsupported devices being inserted and removed.
// It is also used to inform the application when a power fault has occurred.
// This function is required when the g_USBGenericEventDriver is included in
// the host controller driver array that is passed in to the
// USBHCDRegisterDrivers() function.
//
// \return None.
//
//*****************************************************************************
void
USBHCDEvents(void *pvData)
{
    tEventInfo *pEventInfo;

    //
    // Cast this pointer to its actual type.
    //
    pEventInfo = (tEventInfo *)pvData;

    switch(pEventInfo->ui32Event)
    {
        //
        // New CDC device detected.
        //
        case USB_EVENT_CONNECTED:
        {
            //
            // See if this is a CDC device where control interface descriptor
            // (interface 0) is defined as CDC class and interface protocol is
            // Common AT commands (value of 02).
            //
            if((USBHCDDevClass(pEventInfo->ui32Instance, 0) == USB_CLASS_CDC) &&
               (USBHCDDevProtocol(pEventInfo->ui32Instance, 0) ==
                       USB_CDC_PROTOCOL_V25TER))
            {
                //
                // Indicate that the CDC device has been detected.
                //
                UARTprintf("\nCDC Device Connected\n");

                //
                // Proceed to the STATE_CDC_DEVICE_INIT state so that the main
                // loop can finish initialized the CDC device since
                // USBCDCSerialInit() cannot be called from within a callback.
                //
                g_eUSBState = STATE_CDC_DEVICE_INIT;
            }

            break;
        }
        //
        // Unsupported device detected.
        //
        case USB_EVENT_UNKNOWN_CONNECTED:
        {
            UARTprintf("Unsupported Device Class (0x%02x) Connected.\n",
                       pEventInfo->ui32Instance);

            //
            // An unknown device was detected.
            //
            g_eUSBState = STATE_UNKNOWN_DEVICE;

            break;
        }
        //
        // Device has been unplugged.
        //
        case USB_EVENT_DISCONNECTED:
        {
            //
            // Indicate that the device has been disconnected.
            //
            UARTprintf("\nDevice Disconnected\n");

            //
            // Change the state so that the main loop knows that the device
            // is no longer present.
            //
            g_eUSBState = STATE_NO_DEVICE;

            break;
        }
        //
        // Power Fault has occurred.
        //
        case USB_EVENT_POWER_FAULT:
        {
            UARTprintf("Power Fault\n");

            //
            // No power means no device is present.
            //
            g_eUSBState = STATE_POWER_FAULT;

            break;
        }
        default:
        {
            break;
        }
    }
}

//*****************************************************************************
//
// This is the callback from the USB CDC device handler.
//
// \param pvCBData is ignored by this function.
// \param ui32Event is one of the valid events for a CDC device.
// \param ui32MsgParam is defined by the event that occurs.
// \param pvMsgData is a pointer to data that is defined by the event that
// occurs.
//
// This function will be called to inform the application when a CDC device has
// been plugged in or removed and any time data is sent or received.
//
// \return This function will return 0.
//
//*****************************************************************************
uint32_t
CDCSerialCallback(void *pvCBData, uint32_t ui32Event,
                  uint32_t ui32MsgParam, void *pvMsgData)
{
    uint32_t ui32Count;
    uint32_t ui16BytesReceived;

    switch(ui32Event)
    {
        //
        // Data was detected.
        //
        case USB_EVENT_RX_AVAILABLE:
        {
            ui16BytesReceived = USBHCDCInPipeDataCount();

            //
            // Parse through 64 bytes of data to display.
            //
            for (ui32Count = 0; ui32Count<ui16BytesReceived; ui32Count++)
            {

                //
                // Transfer data from USB Buffer to Application Buffer.
                //
                g_pUSBRxBuffer[g_ui16ReceivedDataCount++] = g_pUSBPipeBuffer[ui32Count];

                //
                // Look for character to trigger data sending.
                //
                if (g_pUSBRxBuffer[g_ui16ReceivedDataCount-1] == 'Z')
                {
                    g_bSendData = true;
                }
            }

            //
            // Look for an LF-CR packet at the end to determine when a packet
            // has been received in full.
            //
            if ((g_pUSBRxBuffer[g_ui16ReceivedDataCount-1] == 0x0A) &&
                    (g_pUSBRxBuffer[g_ui16ReceivedDataCount-2] == 0x0D))
            {
                g_bReceivedData = true;
            }

            break;
        }
        case USB_EVENT_TX_COMPLETE:
        {
            //
            // Check if we have sent data (could replace w/ send data flag)
            //
            if (g_ui16SentDataCount != 0)
            {
                //
                // Check if all data has been sent
                //
                if (g_ui16SentDataCount > 1023)
                {
                    //
                    // Clear counter and flag
                    //
                    g_ui16SentDataCount = 0;
                }
                else
                {
                    //
                    // Queue up next packet.
                    //
                    USBHCDCScheduleWrite(g_psCDCSerialInstance,
                                     &g_pUSBTxBuffer[g_ui16SentDataCount], 64);

                    //
                    // Increase send counter.
                    //
                    g_ui16SentDataCount += 64;
                }
            }

            break;
        }
    }

    return(0);
}

//*****************************************************************************
//
// This is the main loop that runs the application.
//
//*****************************************************************************
int
main(void)
{
    uint32_t ui32SysClock;
    uint32_t ui32PLLRate;
    uint16_t ui16Count;
    unsigned char ucChar;

    //
    // Initialize global variables.
    //
    g_ui16SentDataCount = 0;
    g_ui16ReceivedDataCount = 0;
    g_bSendData = false;
    g_bReceivedData = false;

    //
    // Initially wait for device connection.
    //
    g_eUSBState = STATE_NO_DEVICE;

    //
    // Run from the PLL at 120 MHz.
    //
    ui32SysClock = SysCtlClockFreqSet((SYSCTL_XTAL_25MHZ |
                                           SYSCTL_OSC_MAIN |
                                           SYSCTL_USE_PLL |
                                           SYSCTL_CFG_VCO_480), 120000000);

    //
    // Enable the pins and peripherals used by this example.
    //
    PinoutSet(false,true);

    //
    // Enable the UART and print a brief message.
    //
    UARTStdioConfig(0, 115200, ui32SysClock);
    UARTprintf("\033[2J\033[H");
    UARTprintf("CDC Serial Application\n");

    //
    // Configure SysTick for a 100Hz interrupt.
    //
    ROM_SysTickPeriodSet(ui32SysClock / TICKS_PER_SECOND);
    ROM_SysTickEnable();
    ROM_SysTickIntEnable();

    //
    // Initialize the USB stack in host mode. No callback is needed at this
    // time.
    //
    USBStackModeSet(0, eUSBModeHost, 0);

    //
    // Register the host class drivers.
    //
    USBHCDRegisterDrivers(0, g_ppHostClassDrivers, g_ui32NumHostClassDrivers);

    //
    // Open an instance of the CDC driver.  The CDC device does not need
    // to be present at this time, this just saves a place for it and allows
    // the applications to be notified when a CDC device is present.
    //
    g_psCDCSerialInstance = USBHostCDCOpen(CDCSerialCallback, &g_pUSBPipeBuffer);

    //
    // Initialize the power configuration.  This sets the power enable signal
    // to be active high and does not enable the power fault.
    //
    USBHCDPowerConfigInit(0, USBHCD_VBUS_AUTO_HIGH | USBHCD_VBUS_FILTER);

    //
    // Tell the USB library the CPU clock and the PLL frequency.
    //
    SysCtlVCOGet(SYSCTL_XTAL_25MHZ, &ui32PLLRate);
    USBHCDFeatureSet(0, USBLIB_FEATURE_CPUCLK, &ui32SysClock);
    USBHCDFeatureSet(0, USBLIB_FEATURE_USBPLL, &ui32PLLRate);

#ifdef USE_ULPI
    //
    // Tell the USB library to use ULPI interface
    //
    ui32ULPI = USBLIB_FEATURE_ULPI_HS;
    USBHCDFeatureSet(0, USBLIB_FEATURE_USBULPI, &ui32ULPI);
#endif

    //
    // Initialize the USB controller for host operation.
    //
    USBHCDInit(0, g_pHCDPool, HCD_MEMORY_SIZE);

    //
    // Prepare USB data.
    //
    FillUSBTxBuffer();

    //
    // The main loop for the application.
    //
    while(1)
    {
        //
        // Tell the OTG library code how much time has passed in
        // milliseconds since the last call.
        //
        USBOTGMain(GetTickms());

        switch(g_eUSBState)
        {
            //
            // This state is entered when the CDC device is first detected.
            //
            case STATE_CDC_DEVICE_INIT:
            {
                //
                // Initialize the newly connected CDC device.
                //
                USBHCDCInitNewDevice(g_psCDCSerialInstance);

                //
                // Proceed to the connected CDC device state.
                //
                g_eUSBState = STATE_CDC_DEVICE_CONNECTED;

                break;
            }
            case STATE_CDC_DEVICE_CONNECTED:
            {
                //
                // Check if we need to trigger
                //
                if (g_bSendData)
                {
                    g_ui16SentDataCount = 0;

                    //
                    // Schedule next transaction for when first transmission
                    // is complete.
                    //
                    USBHCDCScheduleWrite(g_psCDCSerialInstance,
                                         &g_pUSBTxBuffer[g_ui16SentDataCount],
                                         64);

                    g_ui16SentDataCount += 64;
                    g_bSendData = false;
                }

                if (g_bReceivedData)
                {
                    for (ui16Count = 0; ui16Count<g_ui16ReceivedDataCount; ui16Count++)
                    {
                        //
                        // Retrieve the data value pointed to by ui16Count.
                        // Data will be in RX Buffer as that is what has been
                        // received on the USB IN Pipe.
                        //
                        ucChar = (unsigned char)g_pUSBRxBuffer[ui16Count];

                        if (ucChar != 0xBE)
                        {
                            //
                            // Print the data sent by attached CDC device out
                            // to the UART.
                            //
                            UARTprintf("%c", ucChar);
                        }
                    }

                    //
                    // Reset data pointer to load in new USB data.
                    //
                    g_ui16ReceivedDataCount = 0;
                    g_bReceivedData = false;
                }

                break;
            }
            case STATE_UNKNOWN_DEVICE:
            {
                //
                // Nothing to do as the device is unknown.
                //
                break;
            }
            case STATE_NO_DEVICE:
            {
                //
                // Nothing is currently done in the main loop when the CDC
                // device is not connected.
                //
                break;
            }
            default:
            {
                break;
            }
        }
    }
}
