/*
 * ahrs.c
 *
 *  Created on: 2016 febr. 7
 *      Author: Sandor
 */


#include <stdbool.h>
#include <stdint.h>
#include <math.h>
#include "ahrs.h"
#include "mpu.h"

extern uint32_t g_ui32SysTick;

void AHRS_init(tAHRSInst *psInst, tMPUInst *psMPUInst) {
	psInst->psMPUInst = psMPUInst;
	psInst->sampleFreq = 100;
	psInst->twoKp = 1.0;
	psInst->twoKi = 0;
    psInst->ui32SysTick = g_ui32SysTick;
	psInst->q0 = 1;
	psInst->ui8State = AHRS_STATE_IDLE;
}

//ha kesz az adat akkor frissiti a szurot
void AHRS_poll(tAHRSInst *psInst) {
	if(psInst->psMPUInst->bDataReady) {
		psInst->psMPUInst->bDataReady = false;
		AHRS_update(psInst);
	}
}

void AHRS_update(tAHRSInst *psInst) {
    float ax, ay, az, gx, gy, gz, mx, my, mz;
    float recipNorm;
    float q0q0, q0q1, q0q2, q0q3, q1q1, q1q2, q1q3, q2q2, q2q3, q3q3;
	float hx, hy, bx, bz;
	float halfvx, halfvy, halfvz, halfwx, halfwy, halfwz;
	float halfex, halfey, halfez;
	float qa, qb, qc;

	psInst->ui32SysTick = g_ui32SysTick;
	// apply axis transformation, scaling and offsets
	switch(psInst->psMPUInst->ui8AxisArrangement) {
		case MPU_AXIS_ARR_1: {
			ax = psInst->psMPUInst->i16ax * psInst->psMPUInst->faScaleFactor;
			ay = psInst->psMPUInst->i16ay * psInst->psMPUInst->faScaleFactor;
			az = psInst->psMPUInst->i16az * psInst->psMPUInst->faScaleFactor;
			gx = psInst->psMPUInst->i16gx * psInst->psMPUInst->fgScaleFactor;
			gy = psInst->psMPUInst->i16gy * psInst->psMPUInst->fgScaleFactor;
			gz = psInst->psMPUInst->i16gz * psInst->psMPUInst->fgScaleFactor;
			mx = psInst->psMPUInst->i16mx + psInst->psMPUInst->i16mxOffset;
			my = psInst->psMPUInst->i16mz + psInst->psMPUInst->i16mzOffset;
			mz = psInst->psMPUInst->i16my + psInst->psMPUInst->i16myOffset;
		}
	}

	// Skip update if accelerometer or magnetometer measurement invalid
	if(((mx == 0) && (my == 0) && (mz == 0)) || ((ax == 0) && (ay == 0) && (az == 0))) return;

	// Normalise accelerometer measurement
	recipNorm = 1/sqrt(ax * ax + ay * ay + az * az);
	ax *= recipNorm;
	ay *= recipNorm;
	az *= recipNorm;

	// Normalise magnetometer measurement
	recipNorm = 1/sqrt(mx * mx + my * my + mz * mz);
	mx *= recipNorm;
	my *= recipNorm;
	mz *= recipNorm;

    // Auxiliary variables to avoid repeated arithmetic
    q0q0 = psInst->q0 * psInst->q0;
    q0q1 = psInst->q0 * psInst->q1;
    q0q2 = psInst->q0 * psInst->q2;
    q0q3 = psInst->q0 * psInst->q3;
    q1q1 = psInst->q1 * psInst->q1;
    q1q2 = psInst->q1 * psInst->q2;
    q1q3 = psInst->q1 * psInst->q3;
    q2q2 = psInst->q2 * psInst->q2;
    q2q3 = psInst->q2 * psInst->q3;
    q3q3 = psInst->q3 * psInst->q3;

    // Reference direction of Earth's magnetic field
    hx = 2.0f * (mx * (0.5f - q2q2 - q3q3) + my * (q1q2 - q0q3) + mz * (q1q3 + q0q2));
    hy = 2.0f * (mx * (q1q2 + q0q3) + my * (0.5f - q1q1 - q3q3) + mz * (q2q3 - q0q1));
    bx = sqrt(hx * hx + hy * hy);
    bz = 2.0f * (mx * (q1q3 - q0q2) + my * (q2q3 + q0q1) + mz * (0.5f - q1q1 - q2q2));

	// Estimated direction of gravity and magnetic field
	halfvx = q1q3 - q0q2;
	halfvy = q0q1 + q2q3;
	halfvz = q0q0 - 0.5f + q3q3;
    halfwx = bx * (0.5f - q2q2 - q3q3) + bz * (q1q3 - q0q2);
    halfwy = bx * (q1q2 - q0q3) + bz * (q0q1 + q2q3);
    halfwz = bx * (q0q2 + q1q3) + bz * (0.5f - q1q1 - q2q2);

	// Error is sum of cross product between estimated direction and measured direction of field vectors
	halfex = (ay * halfvz - az * halfvy) + (my * halfwz - mz * halfwy);
	halfey = (az * halfvx - ax * halfvz) + (mz * halfwx - mx * halfwz);
	halfez = (ax * halfvy - ay * halfvx) + (mx * halfwy - my * halfwx);

	// Compute and apply integral feedback if enabled
	if(psInst->twoKi > 0.0f) {
		psInst->integralFBx += psInst->twoKi * halfex * (1.0f / psInst->sampleFreq);	// integral error scaled by Ki
		psInst->integralFBy += psInst->twoKi * halfey * (1.0f / psInst->sampleFreq);
		psInst->integralFBz += psInst->twoKi * halfez * (1.0f / psInst->sampleFreq);
		gx += psInst->integralFBx;	// apply integral feedback
		gy += psInst->integralFBy;
		gz += psInst->integralFBz;
	}
	else {
		psInst->integralFBx = 0.0f;	// prevent integral windup
		psInst->integralFBy = 0.0f;
		psInst->integralFBz = 0.0f;
	}

	// Apply proportional feedback
	gx += psInst->twoKp * halfex;
	gy += psInst->twoKp * halfey;
	gz += psInst->twoKp * halfez;

	// Integrate rate of change of quaternion
	gx *= (0.5f * (1.0f / psInst->sampleFreq));		// pre-multiply common factors
	gy *= (0.5f * (1.0f / psInst->sampleFreq));
	gz *= (0.5f * (1.0f / psInst->sampleFreq));
	qa = psInst->q0;
	qb = psInst->q1;
	qc = psInst->q2;
	psInst->q0 += (-qb * gx - qc * gy - psInst->q3 * gz);
	psInst->q1 += (qa * gx + qc * gz - psInst->q3 * gy);
	psInst->q2 += (qa * gy - qb * gz + psInst->q3 * gx);
	psInst->q3 += (qa * gz + qb * gy - qc * gx);

	// Normalise quaternion
	recipNorm = 1/sqrt(psInst->q0 * psInst->q0 + psInst->q1 * psInst->q1 + psInst->q2 * psInst->q2 + psInst->q3 * psInst->q3);
	psInst->q0 *= recipNorm;
	psInst->q1 *= recipNorm;
	psInst->q2 *= recipNorm;
	psInst->q3 *= recipNorm;

	psInst->yaw = TO_DEG * atan2(2 * psInst->q1 * psInst->q2 - 2 * psInst->q0 * psInst->q3, 2 * psInst->q0 * psInst->q0 + 2 * psInst->q1 * psInst->q1 - 1);		// psi
	psInst->pitch = TO_DEG * -asin(2 * psInst->q1 * psInst->q3 + 2 * psInst->q0 * psInst->q2);															// theta
	psInst->roll = TO_DEG * atan2(2 * psInst->q2 * psInst->q3 - 2 * psInst->q0 * psInst->q1, 2 * psInst->q0 * psInst->q0 + 2 * psInst->q3 * psInst->q3 - 1);		// phi

	psInst->bDataReady = true;
}

