/*
 * main.c
 */
#include <stdbool.h>
#include <inttypes.h>
#include <math.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>

#include "driverlib/cpu.h"
#include "driverlib/fpu.h"
#include "driverlib/pin_map.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/systick.h"
#include "driverlib/udma.h"
#include "driverlib/i2c.h"
#include "driverlib/uart.h"


#include "inc/hw_gpio.h"
#include "inc/hw_memmap.h"
#include "inc/hw_ints.h"
#include "inc/hw_sysctl.h"
#include "inc/hw_types.h"

#include "sensorlib/i2cm_drv.h"

#include "utils/ustdlib.h"


#include "main.h"
#include "startup_pll.h"
#include "halUART.h"
#include "halExtInt.h"
#include "halTimer.h"
#include "halPWM.h"
#include "mpu.h"
#include "common.h"
#include "ahrs.h"
#include "fifo.h"
#include "serialProtocol.h"
#include "I2C_Stellaris_API.h"

void Init();

//#define DEBUG

extern uint32_t savedValues[MAX_CHANNEL];

tI2CMInstance g_sI2CMInst;
uint32_t g_ui32SysTick = 0;
tErrorCounters g_sErrorCounters;
tMPUInst g_sMPUInst;
bool t_bI2C1Busy;
tAHRSInst g_sAHRSInst;															//definition to Mahony-Magdwick filter

//-------------------- Global Buffers --------------------
uint8_t UARTInputFIFO[BUFFER_SIZE];
uint8_t UARTOutputFIFO[BUFFER_SIZE];
_FIFOu8* InputFIFO;
_FIFOu8* OutputFIFO;

tProtocol* UART0Protocol;

_STATEMACHINE programState = INIT;

void main(void) {
	uint8_t out_fifo_cnt = 0;
	int pprotocol_cnt = 0;

	while(1)
	{
		switch(programState)
		{
			case INIT:
				Init();
				programState = READGYRO;
				break;

			case READGYRO:
				programState = READACCEL;
				break;

			case READACCEL:
				programState = READCOMPASS;
				break;

			case READCOMPASS:
				programState = READPRESSURE;
				break;

			case READPRESSURE:
				programState = READEXTINT;
				break;

			case READEXTINT:
				UARTPrintString("Measured 0 val: ");
				UARTPutNumber(savedValues[0]);
				UARTPrintString("\n\r");
				programState = READADC;
				break;

			case READADC:
				programState = CALCULATE;
				break;

			case CALCULATE:
				programState = SETPWM;
				break;

			case SETPWM:
				programState = SENDPROTOCOL;
				break;

			case SENDPROTOCOL:
				/* send the message from the OutputFIFO to the PC*/
				out_fifo_cnt = 0;
				while(*UARTOutputFIFO){
					UARTPutByte(*UARTOutputFIFO);
					UARTOutputFIFO[out_fifo_cnt++];
				}
				FIFO_Clear(OutputFIFO);
				programState = PROCESSPROTOCOL;
				break;

			case PROCESSPROTOCOL:
				/* process the message from the InputFIFO what I received from the PC*/

				for(pprotocol_cnt = 0; pprotocol_cnt < (InputFIFO->Number / PROTOCOL_LEN); pprotocol_cnt++){
					UART0Protocol = protocolProcess(InputFIFO);
					protocolProcessDoing(UART0Protocol);						/* make the necessary EEPROM things */
				}
				FIFO_Clear(InputFIFO);											/* Clear the Input FIFO 			*/

#ifdef DEBUG
				programState = PRINT;
#else
				programState = READGYRO;
#endif

				break;
			case PRINT:
				programState = READGYRO;
				break;
		}
	}
}

//-------------- Init functions -------------------------
void Init() {
	// FPU init
	FPULazyStackingEnable();
	FPUStackingEnable();

	// PLL init
	SysCtlClockSet(SYSCTL_SYSDIV_1 | SYSCTL_USE_OSC | SYSCTL_OSC_MAIN | SYSCTL_XTAL_16MHZ);
	SysCtlDelay(SysCtlClockGet()/1000);											//1ms delay

	SysCtlClockSet(SYSCTL_SYSDIV_4 | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN | SYSCTL_XTAL_16MHZ);   //200/4 = 50MHz

	/*
	 * LEDs init
	 */
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);
	GPIOPinTypeGPIOOutput(GPIO_PORTC_BASE, GPIO_PIN_4 | GPIO_PIN_5);
	GPIOPinWrite(GPIO_PORTC_BASE, GPIO_PIN_4 | GPIO_PIN_5, 0x00);

	// UART Init with 115200 baud, 1 stop bit, NO parity
	UARTInit();

	// Initialize the buffers
//	InputFIFO =  (uint8_t*)malloc(sizeof(UARTInputFIFO));
//	OutputFIFO = (uint8_t*)malloc(sizeof(UARTOutputFIFO));
	FIFO_Init(InputFIFO,UARTInputFIFO);
	FIFO_Init(OutputFIFO,UARTOutputFIFO);

	// external interrupt init for PORTF
	halExtInttInit();

	// Timer0 Init to measure the duty of receiver
	// 10us period time
	halTimer0Init();

	// PWM Init - frequency:50Hz
	halPWMMotorsInit();

    // I2C
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C1);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);
    ROM_GPIOPinConfigure(GPIO_PA7_I2C1SDA);
    ROM_GPIOPinTypeI2C(GPIO_PORTA_BASE, GPIO_PIN_7);
    ROM_GPIOPinConfigure(GPIO_PA6_I2C1SCL);
    ROM_GPIOPinTypeI2CSCL(GPIO_PORTA_BASE, GPIO_PIN_6);

	// I2C1 Master
	I2CMInit(&g_sI2CMInst, I2C1_BASE, INT_I2C1, 0xFF, 0xFF, SysCtlClockGet());

	// SysTick
	ROM_SysTickPeriodSet(SysCtlClockGet()/1000);
	ROM_SysTickIntEnable();
	ROM_SysTickEnable();

	//MPU init
	//MPU_init(&g_sMPUInst,&g_sI2CMInst,&t_bI2C1Busy, MPU_ADDRESS, NO_CALLBACK, NO_CALLBACK);
	//while(g_sMPUInst.ui8State != MPU_STATE_RUN);

	//default configuration to the MPU6050
	//MPU_default_config(&g_sMPUInst);

	//initialization of Mahony-Magdwick filter
#warning majd meg beallitani a mintaveteli frekit meg ilyeneket
	//AHRS_init(&g_sAHRSInst, &g_sMPUInst);
	//while(g_sAHRSInst.ui8State != AHRS_STATE_RUN);


	// All interrupt enable in the processor
	ROM_IntMasterEnable();
	GPIOPinWrite(GPIO_PORTF_BASE, RED_LED, 0 );
}

void SysTickHandler(void){
	g_ui32SysTick++;
	//MPU_poll(&g_sMPUInst);		//poll the MPU for timing
	// to be continue
}

/** @brief I2CMasterIntHandler
 *
 * 	@param psInst
 * 	@return
 *
 */
/*
void I2CMasterIntHandler(void) {
	I2CMIntHandler(&g_sI2CMInst);
}
*/
