/** @file mpu.c
 *  @brief
 *
 *  This file contains necessary settings for MPU6050
 *
 *  This is the first version.
 *
 *  @author 	Bedo Sandor
 *  @date 		2016.01.21
 *  @bug 		No known bugs.
 */


/* -- Includes -- */
#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_memmap.h"
#include "driverlib/rom.h"
#include "driverlib/gpio.h"
#include "driverlib/eeprom.h"
#include "sensorlib/i2cm_drv.h"

#include "common.h"
#include "mpu.h"

extern uint32_t g_ui32SysTick;
extern tErrorCounters g_sErrorCounters;
tMPUInst *psMPUInUse;

/** @brief MPU_init
 *
 * 	@param
 * 	@return
 *
 */
void
MPU_init(tMPUInst *psInst, tI2CMInstance *psI2CInst, bool *pbI2CBusy, uint_fast8_t ui8I2CAddr,
		           tSensorCallback *pfnCallback, void *pvCallbackData) {
	psInst->psI2CInst = psI2CInst;
	psInst->pbI2CBusy = pbI2CBusy;
	psInst->ui8Addr = ui8I2CAddr;
	psInst->faScaleFactor = 2 * 9.80665 / 32768;
	psInst->fgScaleFactor = TO_RAD * 500 / 32768;
	psInst->ui8AxisArrangement = MPU_AXIS_ARR_1;
	psInst->pfnCallback = pfnCallback;
	psInst->pvCallbackData = pvCallbackData;
	psInst->ui32SysTick = g_ui32SysTick;
	psMPUInUse = psInst;
//	MPU_I2CWrite(psInst, MPU_RA_PWR_MGMT_1, (1<<MPU_PWR1_DEVICE_RESET_BIT));	// reset device
	psInst->ui8State = MPU_STATE_WAIT_RESET;
}

/** @brief MPU_poll
 *
 * 	@param psInst
 * 	@return
 *
 */
void
MPU_poll(tMPUInst *psInst) {
	switch(psInst->ui8State) {
		case MPU_STATE_WAIT_RESET: {
			if(g_ui32SysTick > (psInst->ui32SysTick + MPU_INIT_TIME)) {
				psInst->ui8State = MPU_STATE_WAIT_INIT;
				MPU_default_config(psInst);
			}
		}
	}
}

/** @brief MPU_callback
 *
 *	Callback function for the MPU
 *
 * 	@param
 * 	@return
 *
 */
void
MPU_callback(void *pvCallbackData, uint_fast8_t ui8Status) {
	tMPUInst *psInst;
	psInst = pvCallbackData;

	if(ui8Status != I2CM_STATUS_SUCCESS) {
		g_sErrorCounters.ui16I2CMErrorCount++;
	}
	*(psInst->pbI2CBusy) = false;
}

/** @brief MPU_I2CRead
 *
 * 	@param psInst
 * 	@return
 *
 */
void
MPU_I2CRead(tMPUInst *psInst, uint8_t ui8Reg, uint8_t *pui8Value, uint16_t ui16Count) {
	psInst->uCommand.pui8Buffer[0] = ui8Reg;
	while(*(psInst->pbI2CBusy));
	*(psInst->pbI2CBusy) = true;
	I2CMCommand(psInst->psI2CInst, psInst->ui8Addr, psInst->uCommand.pui8Buffer, 1, 1, pui8Value, ui16Count, ui16Count, MPU_callback, psInst);
	while(*(psInst->pbI2CBusy));
}

/** @brief MPU_I2CRead16
 *
 * 	@param psInst
 * 	@return
 *
 */
void
MPU_I2CRead16(tMPUInst *psInst, uint8_t ui8Reg, uint16_t *pui16Value, uint16_t ui16Count) {
	while(*(psInst->pbI2CBusy));
	*(psInst->pbI2CBusy) = true;
	I2CMRead16BE(&(psInst->uCommand.sI2CRead16BE), psInst->psI2CInst, psInst->ui8Addr, ui8Reg, pui16Value, ui16Count, MPU_callback, psInst);
	while(*(psInst->pbI2CBusy));
}

/** @brief MPU_I2CWrite
 *
 *	This function sent the necessary data over I2C
 *
 * 	@param psInst This variable contain the necessarry data for the I2C
 * 	@param ui8Reg This value contain the name of the I2C's register
 * 	@param ui8Value This variable contain, what I want to send to the register
 */
void
MPU_I2CWrite(tMPUInst *psInst, uint8_t ui8Reg, uint8_t ui8Value) {
	psInst->uCommand.pui8Buffer[0] = ui8Reg;
	psInst->uCommand.pui8Buffer[1] = ui8Value;
	I2CMCommand(psInst->psI2CInst, psInst->ui8Addr, psInst->uCommand.pui8Buffer,
			2, 0, 0, 0, 0, MPU_callback, psInst);
}

/** @brief MPU_I2CWrite16
 *
 * 	@param
 * 	@return
 *
 */
void
MPU_I2CWrite16(tMPUInst *psInst, uint8_t ui8Reg, uint16_t *pui16Value, uint16_t ui16Count) {
	while(*(psInst->pbI2CBusy));
	*(psInst->pbI2CBusy) = true;
	I2CMWrite16BE(&(psInst->uCommand.sI2CWrite16BE), psInst->psI2CInst, psInst->ui8Addr, ui8Reg, pui16Value, ui16Count, MPU_callback, psInst);
	while(*(psInst->pbI2CBusy));
}

/** @brief MPU_default_config
 *
 * 	@param psInst
 * 	@return
 *
 */
void
MPU_default_config(tMPUInst* psInst){
	MPU_I2CWrite(psInst, MPU_RA_PWR_MGMT_1, MPU_CLOCK_PLL_XGYRO);				// set internal clock source to Gyro X, wake up from sleep
	MPU_I2CWrite(psInst, MPU_RA_SMPLRT_DIV, 9);									// set Sample Rate Divider, 249 = 4 Hz, 9 = 100 Hz, 10 = 92 Hz, 23 = 44 Hz. 3 = 250 Hz
	MPU_I2CWrite(psInst, MPU_RA_GYRO_CONFIG, (MPU_GYRO_FS_500<<3));			// set Gyro Full Scale Range
	MPU_I2CWrite(psInst, MPU_RA_ACCEL_CONFIG, (MPU_ACCEL_FS_2<<3));			// set Accel Full Scale Range
	MPU_I2CWrite(psInst, MPU_RA_CONFIG, MPU_DLPF_BW_98);					// set Digital Low Pass (DLP) Filter
	MPU_I2CWrite(psInst, MPU_RA_FIFO_EN, (1<<MPU_XG_FIFO_EN_BIT) | (1<<MPU_YG_FIFO_EN_BIT) |		// Select data to put to FIFO:
				(1<<MPU_ZG_FIFO_EN_BIT) | (1<<MPU_ACCEL_FIFO_EN_BIT) | (1<<MPU_SLV0_FIFO_EN_BIT));	// Gyro, Accel, Slave0 (Mag);
	MPU_I2CWrite(psInst, MPU_RA_I2C_MST_CTRL, MPU_CLOCK_DIV_400);			// I2C master clock 400 kHz

	// MPU_I2CWrite(psInst, MPU_RA_INT_PIN_CFG, (1<<MPU_INTCFG_I2C_BYPASS_EN_BIT));	// set I2C bypass to gain direct access to magnetometer

	MPU_I2CWrite(psInst, MPU_RA_I2C_SLV0_ADDR, HMC5983_ADDRESS | (1<<MPU_I2C_SLV_RW_BIT));	// slave 0 reads from compass
	MPU_I2CWrite(psInst, MPU_RA_I2C_SLV0_REG, HMC5983_DATA_OUT_START);						// reg to read values from
	MPU_I2CWrite(psInst, MPU_RA_I2C_SLV0_CTRL, (1<<MPU_I2C_SLV_EN_BIT) | 6);				// 6 bytes to read
	MPU_I2CWrite(psInst, MPU_RA_I2C_SLV1_ADDR, HMC5983_ADDRESS);							// slave 1 writes to compass
	MPU_I2CWrite(psInst, MPU_RA_I2C_SLV1_REG, HMC5983_MODE_REG);							// reg to write measure command to
	MPU_I2CWrite(psInst, MPU_RA_I2C_SLV1_CTRL, (1<<MPU_I2C_SLV_EN_BIT) | 1);				// enable slave 1, 1 byte to write
	MPU_I2CWrite(psInst, MPU_RA_I2C_SLV1_DO, 1);											// slave 1 writes 1 to the AKM_CTL register (start single measurement)

	MPU_I2CWrite(psInst, MPU_RA_INT_ENABLE, (1<<MPU_INTERRUPT_FIFO_OFLOW_BIT) | (1<<MPU_INTERRUPT_DATA_RDY_BIT));	// Enable Data Ready & FIFO overflow interrupt
	MPU_I2CWrite(psInst, MPU_RA_USER_CTRL, (1<<MPU_USERCTRL_FIFO_EN_BIT) | (1<<MPU_USERCTRL_I2C_MST_EN_BIT));		// Enable FIFO & interrupts
	psInst->ui8State = MPU_STATE_RUN;

}

