#include <stdint.h>
#include <stdbool.h>
#include <inc/hw_memmap.h>
#include "driverlib/gpio.h"
#include "driverlib/adc.h"
#include "driverlib/pin_map.h"
#include "driverlib/sysctl.h"

/**
 * Simple ADC example given from support forum to try to understand
 * how the ADC works on the Tiva.
 */
int main(void)
{
    unsigned int array[4];

    //Set clock to 16 MHz
    SysCtlClockSet(SYSCTL_SYSDIV_10 | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN | SYSCTL_XTAL_16MHZ);

    // Enable the GPIOE peripheral

    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);

    // Enable the ADC0 module.
    SysCtlPeripheralEnable(SYSCTL_PERIPH_ADC0);

    // Wait for the GPIOE module to be ready.
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOE))
    {
    }

    // Wait for the ADC0 module to be ready.
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_ADC0))
    {
    }

    //Configures ADC pins PE3 (AIN0) and PE2 (AIN1)
    //Here two analog pins are used, Arbitrarily, to show that more than one channel
    //can be selected in a sequence and not all of the available steps need to be used
    GPIOPinTypeADC(GPIO_PORTE_BASE, GPIO_PIN_3 | GPIO_PIN_2);

    //Configure ADC functions
    // Using Sequence 1 which has a 4 entry FIFO

    /* TivaWare Peripherals Driver Library (P. 38)
     *  function: Configures the trigger source and priority of a sample sequence.
     *  param: (ui32Base, ui32SequenceNum, ui32Trigger,ui32Priority)
     *      ui32Base - base address of the ADC module
     *      ui32SequenceNum - the sample sequence number
     *      ui32Trigger - the trigger source that initiates the sample sequence; must be one of the ADC_TRIGGER_ * values
     *      ui32Priority - the relative priority of the sample sequence with respect to the other sample sequences
     */
    ADCSequenceConfigure(ADC0_BASE, 1, ADC_TRIGGER_PROCESSOR, 0);

    /* TivaWare Peripherals Driver Library (P. 42)
     *  function: Configure a step of the sample sequencer
     *  param: (ui32Base, ui32SequenceNum, ui32Step, ui32Config)
     *      ui32base - base address of the ADC module
     *      ui32sequenceNum - the sample sequence number
     *      ui32Step -  the step to be configured
     *      ui32Config - configuration of this step; must be a logical OR of
     *       ADC_CTL_TS, ADC_CTL_IE, ADC_CTL_END, ADC_CTL_D, one of the input
     *       channel selects (ADC_CTL_CH0 through ADC_CTL_CH23), and one of the
     *       digital comparator selects (ADC_CTL_CMP0 through ADC_CTL_CMP7).
     */
    ADCSequenceStepConfigure(ADC0_BASE, 1, 0, ADC_CTL_CH0 );
    ADCSequenceStepConfigure(ADC0_BASE, 1, 1, ADC_CTL_CH1 | ADC_CTL_IE | ADC_CTL_END);

    /*
     * TivaWare Peripherals Driver Library (P. 41)
     *  function: Enables a sample sequence
     *  param: (ui32Base, ui32SequenceNum)
     *      ui32base - base address of the ADC module
     *      ui32SequenceNum - is the sample sequence number
     *  description: allows the specified sample sequence to be captured when
     *      its trigger is detected. A sample sequence must be configured
     *      before it is enabled.
     */
    ADCSequenceEnable(ADC0_BASE,1);

    /*
     * TivaWare Peripherals Driver Library (P. 30)
     *  function: Clears sample sequence interrupt source
     *  param: (ui32Base, ui32SequenceNum)
     *      ui32base - base address of the ADC module
     *      ui32SequenceNum - is the sample sequence number
     *  description: The specified sample sequence interrupt is cleared, so
     *      that it no longer asserts. This function must be called in the
     *      interrupr handler to keep the interrupt from being triggered
     *      again immediately upon exit.
     */
    ADCIntClear(ADC0_BASE, 1);
    ADCClockConfigSet(ADC0_BASE, ADC_CLOCK_SRC_PIOSC | ADC_CLOCK_RATE_FULL, 1);


    /*
     * Infinite while-loop
     */
    while(1)
    {
        /*
         *TivaWare Peripherals Driver Library (P. 36)
         *  function: Causes a processor trigger for a sample sequence
         *  param: (ui32Base, ui32SequenceNum)
         *      ui32Base - base address of the ADC module
         *      ui32SequenceNum - sample sequence number, with ADC_TRIGGER_WAIT or
         *          ADC_TRIGGER_SIGNAL optionally ORed into it
         *  description: This triggers a processor-initiated sample sequence
         *  if the sample sequence trigger is configured to ADC_TRIGGER_PROCESSOR.
         *  If ADC_TRIGGER_WAIT is ORed into the sequence number, the processor-initiated
         *  trigger is delayed until a later processor-initiated trigger to a different
         *  ADC module that specifies ADC_TRIGGER_SIGNAL, allowing multiple
         *  ADCs to start from a processor-initiated trigger in a synchronous
         *  manner
         */
        ADCProcessorTrigger(ADC0_BASE, 1);

        //wait for the conversion to be completed
        while(!ADCIntStatus(ADC0_BASE, 1, false))
        {
        }
        //clear Interrupt flag
        ADCIntClear(ADC0_BASE, 1);

        //Read ADC input and store PE3 in array[0], PE2 in array[1]
        ADCSequenceDataGet(ADC0_BASE, 1, array); //read the analog values
    }
}
