#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_adc.h"
#include "inc/hw_types.h"
#include "inc/hw_udma.h"
#include "driverlib/debug.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "driverlib/adc.h"
#include "driverlib/udma.h"
#include "driverlib/timer.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/systick.h"
//#include "uartstdio.h"

/**
 * Muhammad Rezuna with help from Bob Crosby
 * Setting up the ADC
 */

//"ADC_BUF_SIZE" is a constant with a value of 1024; Will determine the size of two buffers
#define ADC_BUF_SIZE 1024
#define MAX_SIZE 4096
#define NumOfSamples 2;
static numTimesInt = 1u; //Keep track of how many data transfers has occur
static numTransfer = MAX_SIZE / 1024;
#pragma DATA_ALIGN(ucControlTable, 1024)
uint8_t ucControlTable[1024];

static uint16_t data[MAX_SIZE]; //Store all the buffer data here - later used for correlation
static uint32_t g_ui32DMAErrCount = 0u;
static uint32_t g_ui32SysTickCount;

void setupADC(void);
void setupTimer(void);
void setupDMA(void);

/*
 * ADC interrupt handler. Also controls when to stop the ADC as well as the timer.
 */
void ADCseq0Handler()
{
    if(numTransfer > 3)
    {
        if(numTimesInt % 2 != 0)
        {
            uDMAChannelTransferSet(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT, UDMA_MODE_PINGPONG,
                                   (void*) (ADC0_BASE + ADC_O_SSFIFO0), &data[1024 + (1024*numTimesInt)], ADC_BUF_SIZE);
            uDMAChannelEnable(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT);
            numTimesInt++;
            numTransfer--;
        }
        else
        {
            uDMAChannelTransferSet(UDMA_CHANNEL_ADC0 | UDMA_ALT_SELECT, UDMA_MODE_PINGPONG,
                                   (void*) (ADC0_BASE + ADC_0_SSFIFO0), &data[1024 + (1024*numTimesInt)], ADC_BUF_SIZE);
            uDMAChannelEnable(UDMA_CHANNEL_ADC0|UDMA_ALT_SELECT);
            numTimesInt++;
            numTransfer--;
        }
        if(numTransfer == 2)
        {
            numTimesInterrupted++;
            numTransfer--;
        }
        else if (numTransfer ==1)
        {
            ADCSequenceDisable(ADC0_BASE, 0u);
            TimerDisable(Timer0_BASE, TIMER_A);
        }
    }
}
void uDMAErrorHandler(void)
{
    uint32_t ui32Status;
    ui32Status = MAP_uDMAErrorStatusGet();
    if (ui32Status)
    {
        MAP_uDMAErrorStatusClear();
        g_ui32DMAErrCount++;
    }
}

// Not used in this example, but used to debug to make sure timer interrupts happen
void Timer0AIntHandler(void)
{
    // Clear the timer interrupt flag.
    TimerIntClear(TIMER0_BASE, TIMER_TIMA_TIMEOUT);
}

void SysTickIntHandler(void)
{
    // Update our system tick counter.
    g_ui32SysTickCount++;
}

int main(void)
{
    SysCtlClockSet(
            SYSCTL_SYSDIV_2_5 | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN
                    | SYSCTL_XTAL_16MHZ); //Setting the clock to 80 MHz
    SysCtlDelay(20u); //Provides a small delay

    /*
     * For code intended to be shared between projects, and some of the projects
     * run on devices with a ROM and some without a ROM, it is convenient to have the
     * code automatically call the ROM
     */
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_TIMER0); //Enable the clock to TIMER0
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_ADC0); //Enable the clock to ADC module
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_UDMA);   //Enable the clock to uDMA
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE); //Enables the clock to PORT E
    MAP_SysCtlDelay(30u); //Provides a small delay; param provides the number of loop iterations to perform

    MAP_SysTickPeriodSet(SysCtlClockGet() / 100000u); //Sets the period of the SysTic counter to 10us
    MAP_SysTickIntEnable(); //Enables the SysTick interrupt, allowing it to be reflected to the processor.
    MAP_SysTickEnable();    //Enables and starts the SysTick counter

    setupDMA();
    setupADC();
    setupTimer();

    MAP_IntMasterEnable();
    MAP_TimerEnable(TIMER0_BASE, TIMER_A);
}

/*
 *  Setting up the uDMA (Micro Direct Memory Access) controller. The uDMA API provides
 *  functions to configure the Tiva uDMA controller. The uDMA controller is designed to work
 *  with the ARM Cortex-M processor and provides an efficient and low-overhead means of
 *  transferring blocks of data in the system.
 */
void setupDMA()
{
    //Must first enable the uDMA controller so that it can be configured and used
    uDMAEnable();

    /*
     *  Sets the base address for the channel control table. The parameter
     * is a pointer to the 1024-byte-aligned base address of the uDMA
     * channel control table (defined above main).
     *  This function table resides in system memory and holds control information
     *for each uDMA channel. The table must be aligned on a 1024-byte
     *boundary. The base address must be configured before any of the channel functions
     *can be used.
     *  The size of the channel control table depends on the number of uDMA
     *channels and the transfer modes that are used.
     */
    uDMAControlBaseSet(ucControlTable);

    /*
     *  Disable/Enable attributes of the specific uDMA channel selected. The first
     * parameter is the specific channel, the second is the attribute(s) you want to
     * disable/enable (if multiple, OR them)
     */
    uDMAChannelAttributeDisable(
            UDMA_CHANNEL_ADC0,
            UDMA_ATTR_ALTSELECT | UDMA_ATTR_HIGH_PRIORITY | UDMA_ATTR_REQMASK);
    uDMAChannelAttributeEnable(UDMA_CHANNEL_ADC0, UDMA_ATTR_USEBURST);

    /*
     *  Used to set control parameters for a uDMA transfer. These parameters are typically
     * not changed often. 2 parameters:
     *  1.) logical OR of channel number with UDMA_PRI_SELECT or UDMA_ALT_SELECT to choose
     * between the primary or alternate data structure is used.
     *  2.) logical OR of 5 values - data size, source address increment, destination
     * address increment, arbitration size, and the use burst flag. Possible values are
     * listed on page 591 of the TivaWare Peripheral User's Guide
     */
    uDMAChannelControlSet(
            UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT,
            UDMA_SIZE_8 | UDMA_SRC_INC_NONE | UDMA_DST_INC_8 | UDMA_ARB_1);
    uDMAChannelControlSet(
            UDMA_CHANNEL_ADC0 | UDMA_ALT_SELECT,
            UDMA_SIZE_8 | UDMA_SRC_INC_NONE | UDMA_DST_INC_8 | UDMA_ARB_1);

    /*
     *  Used to configure the parameters for a uDMA transfer. uDMAChannelControlSet()
     * MUST be called at least once for this channel prior to calling this function. 5 Params:
     *  1.)Logical OR of the channel number and structure (Pri or alt)
     *  2.)Type of uDMA transfer
     *  3.)source address for the transfer
     *  4.)destination address for the transfer
     *  5.)number of data items to transfer
     */
    uDMAChannelTransferSet(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT,
                           UDMA_MODE_PINGPONG,
                           (void*) (ADC0_BASE + ADC_O_SSFIFO0), &data[0],
                           ADC_BUF_SIZE);
    uDMAChannelTransferSet(UDMA_CHANNEL_ADC0 | UDMA_ALT_SELECT,
                           UDMA_MODE_PINGPONG,
                           (void*) (ADC0_BASE + ADC_O_SSFIFO0), &data[1024],
                           ADC_BUF_SIZE);

    //Enables a channel to perform data transfers
    uDMAChannelEnable(UDMA_CHANNEL_ADC0);
}

void setupADC()
{
    /*
     * GPIOPinTypeADC configures pin(s) to use as Analog-to-Digital Converter (ADC)
     * inputs. The first parameter is the base address of the GPIO port and the second
     * is the bit-packed representation of the pin(s).
     */
    GPIOPinTypeADC(GPIO_PORTE_BASE, GPIO_PIN_2);
    SysCtlDelay(80u); // delay

    /*
     * Use ADC0 sequence 0 to sample channel 0 once for each timer period
     *
     * This function sets the clock configuration for the ADC. The first parameter
     * is the base address of the ADC to configure (ADC0 was used for this example).
     * The second parameter is a combination (ORed value) of the ADC_CLOCK_SRC and
     * ADC_CLOCK_RATE to configure the ADC clock input - For TM4C123X devices if the
     * PLL is enabled, the PLL/25 is used as the ADC clock unless ADC_CLOCK_SRC_PIOSC
     * is specified (the internal PIOSC at 16MHz) & The clock rate controls how often
     * samples are provided back to the application: FULL, HALF, QUARTER, or EIGTH. The
     * last parameter is the input clock divider for the clock selected by the
     * ADC_CLOCK_SRC value (usually 1).
     */
    ADCClockConfigSet(ADC0_BASE, ADC_CLOCK_SRC_PIOSC | ADC_CLOCK_RATE_HALF, 1); // CHECK CLOCK RATE
    SysCtlDelay(10); // Time for the clock configuration to set

    /* Found in Chapter 17 - Interrupt Controller (NVIC) p. 345
     *  This call disables an interrupt - in this case "INT_ADC0SS0." The specified
     *  interrupt/param must be one of the valid INT_* values listed in the
     *  Driver Library User's Guide and defined in the inc/hw_ints.h header file.
     *  Other enables for the interrupt (such as at the peripheral level) are
     *  unaffected by this.
     */
    IntDisable(INT_ADC0SS0);

    /*
     * This function disables a sample sequence interrupt. The parameters are
     * the base address of the ADC module and the sample sequence number.
     */
    ADCIntDisable(ADC0_BASE, 0u);

    /*
     * Disables a sample sequence. Parameters are the base address of the ADC module
     * and the sample sequence number. This prevents the specified sequence from
     * being captured when its trigger is detected. A sample sequence must be disabled before it
     * is configured.
     */
    ADCSequenceDisable(ADC0_BASE, 0u);

    // With sequence disabled, it is now safe to load the new configuration parameters

    /*
     *  Configures the trigger source and priority of a sample sequence. The third
     * and fourth parameter is the trigger source that initiates the sample sequence
     * and priority respectively.
     *  For this example, the trigger source is a timer. The timer is enabled
     * and initiated elsewhere.
     *  The priority is 0 (highest)
     */
    ADCSequenceConfigure(ADC0_BASE, 0u, ADC_TRIGGER_TIMER, 0u);

    /*
     *  This function configures the ADC for one step of a sample sequence. The parameters
     *are: base address of the ADC module, sample sequence number, step to be configured,
     *and an OR combination.
     *  The first-two are self-explanatory. The third parameter determines the order
     *in which samples are captured by the ADC when the trigger occurs. It ranges from
     *0 to number of sample sequencer-1 of the module. E.g., sample sequencer 0
     *ranges from 0 to 7 (it has 8 sample sequencer) and sample sequencer 0 was
     *chosen by the function call. The third parameter "0" suggests that it is the highest
     *chosen priority amongst the samples so do it first.
     *  The last parameter is a logical OR of many possibilities.
     */
    ADCSequenceStepConfigure(ADC0_BASE, 0u, 0u,
                             ADC_CTL_CH0 | ADC_CTL_END | ADC_CTL_IE);

    ADCSequenceEnable(ADC0_BASE, 0u);

    ADCIntClear(ADC0_BASE, 0u); //Clears the interrupts
    ADCSequenceDMAEnable(ADC0_BASE, 0);
    IntEnable(INT_ADC0SS0);
}

/*
 *  Setting up the timer. Because we are sampling at a particular frequency (sample rate),
 *  a timer is used to trigger the ADC instead of a processor trigger.
 */
void setupTimer()
{
    /*
     *  This function configures the timer(s). The first parameter specifies the
     * base address of the timer module (arbitrary) meanwhile the second is the
     * configuration for the timer.
     *  For this example, two half-width timer is specified by "TIMER_CFG_SPLIT_PAIR."
     * In addition Timer A (Timer B is not used) is used and is a half-width periodic timer
     * specified by TIMER_CFG_A_PERIODIC.
     */
    MAP_TimerConfigure(TIMER0_BASE,
                       TIMER_CFG_SPLIT_PAIR | TIMER_CFG_A_PERIODIC);

    /*  Set sample frequency to 332.8KHz (every 3uS)
     *  First param is the base address of the timer module, second specifies the
     * timer(s) to adjust; must be either TIMER_A or TIMER_B or TIMER_BOTH (if
     * the timer is configured for full-width only use TIMER_A), and the last one
     * loads the value.
     */
    MAP_TimerLoadSet(TIMER0_BASE, TIMER_A, MAP_SysCtlClockGet() / 332800 - 1);

    /*
     *  Enables or disables trigger output. The first-two parameters are
     * self-explanatory, the last parameter specifies whether the timer is
     * enabled, if value is "true", or disabled, if "false."
     */
    MAP_TimerControlTrigger(TIMER0_BASE, TIMER_A, true);

    /* Assist in debug by stalling timer at breakpoints
     *      This function controls the stall handling. If the last parameter is
     * "true" then the timer stops counting if the processor enters debug mode;
     * otherwise the timer keeps running while in debug mode.
     */
    MAP_TimerControlStall(TIMER0_BASE, TIMER_A, true);
}
