/*
 *  Copyright (C) 2021 Texas Instruments Incorporated
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <kernel/dpl/DebugP.h>
#include "ti_drivers_config.h"
#include "ti_drivers_open_close.h"
#include "ti_board_open_close.h"
#include <drivers/i2c.h>

/*
 * This is an empty project provided for all cores present in the device.
 * User can use this project to start their application by adding more SysConfig modules.
 *
 * This application does driver and board init and just prints the pass string on the console.
 * In case of the main core, the print is redirected to the UART console.
 * For all other cores, CCS prints are used.
 */
static void i2c_read_error_handler(uint16_t sample, int32_t status);


void empty_main(void *args)
{
    /* Open drivers to open the UART driver for console */


        uint16_t        sample=0;
        uint32_t         status;
        uint8_t         rxBuffer[1U];
        uint8_t         txBuffer[1U];
        uint32_t        i2creadTargetAddr;
        I2C_Handle      i2cHandle;
        I2C_Transaction i2cTransaction;

        Drivers_open();
        Board_driversOpen();

        i2creadTargetAddr = 0x1C;
       i2cHandle = gI2cHandle[CONFIG_I2C0];

       DebugP_log("[I2C] Read data ... !!!\r\n");

       /* Set default transaction parameters */
       rxBuffer[0] = 0;
       I2C_Transaction_init(&i2cTransaction);

       /* Override with required transaction parameters */
       i2cTransaction.controllerMode = 0;
       i2cTransaction.readBuf = rxBuffer;
       i2cTransaction.readCount = 1U;

       status = I2C_transfer(i2cHandle, &i2cTransaction);


       if(status == SystemP_SUCCESS)
        {
           DebugP_log("transfer success");
           DebugP_log("[I2C] Sample %u: %u\r\n", sample, rxBuffer[0]);

           i2cTransaction.writeBuf = txBuffer;
           i2cTransaction.writeCount = 1U;
           i2cTransaction.targetAddress = i2creadTargetAddr;
           status = I2C_transfer(i2cHandle, &i2cTransaction);

        }
        else
        {
            i2c_read_error_handler(sample, status);
        }


    Board_driversClose();
    Drivers_close();
}

void myi2ccallback(I2C_Handle handle, I2C_Transaction *transaction, int32_t transferStatus){
    DebugP_log("callback!!\r\n");
}

static void i2c_read_error_handler(uint16_t sample, int32_t status)
{
    switch(status)
    {
        case I2C_STS_ERR:
            DebugP_logError("\n[I2C] Sample %u: Generic error occurred", sample);
            break;
        case I2C_STS_ERR_TIMEOUT:
            DebugP_logError("\n[I2C] Sample %u: Timeout error occurred", sample);
            break;
        case I2C_STS_ERR_NO_ACK:
            DebugP_logError("\n[I2C] Sample %u: No acknowledgement received", sample);
            break;
        case I2C_STS_ERR_ARBITRATION_LOST:
            DebugP_logError("\n[I2C] Sample %u: Arbitration lost", sample);
            break;
        default:
            DebugP_logError("\n[I2C] Sample %u: Bus Access error occurred", sample);
            break;
    }

    return;
}
