# simple-server.py

import socket
import random
from time import sleep
import threading
import sys

#HOST = "127.0.0.1"  # Standard loopback interface address (localhost)
HOST = ""  # Allow all IPv4 addresses to connect
PORT = 8888  # Port to listen on (non-privileged ports are > 1023)
CONNECTED = False
CAN_PACKET_SIZE_8_BYTES = 11
CAN_PACKET_HEADER_BYTES = 2
CAN_PACKET_SIZE = CAN_PACKET_SIZE_8_BYTES + CAN_PACKET_HEADER_BYTES
CAN_PACKET_HEADER_1 = 0xEE
CAN_PACKET_HEADER_2 = 0xAA

NUMBER_OF_FRAMES_TO_SEND = 40

def fakeCAN():
    can = bytearray([0xEE, 0xAA, 0x00, 0x14, 0x08, 0xDE, 0xAD, 0xBE, 0xEF, 0xDE, 0xAD, 0xBE, 0xEF])
    can_bytes = bytes(can)
    return can_bytes

class CANGenerator:
    def __init__(self) -> None:
        self.id = 0
        self.dlc = 0
        self.data = bytes()

    def SetId(self, id):
        self.id = id
    
    def SetDLC(self, dlc):
        self.dlc = dlc
    
    def SetData(self, data):
        self.data = data
        self.dlc = len(data)
    
    def Randomize(self):
        self.id = random.randint(0, 2048)
        #self.dlc = random.randint(0, 8)
        self.dlc = 8
        self.data = random.randbytes(self.dlc)
    
    def GetRawFrame(self) -> bytearray:
        idHighByte = (self.id >> 8) & 0xFF
        idLowByte = self.id & 0xFF
        frame = bytearray([idHighByte, idLowByte, self.dlc])
        
        for b in self.data:
            frame.append(b)
        
        return frame
    
    def GetFrameWithHeader(self) -> bytes:
        frame = bytearray([CAN_PACKET_HEADER_1, CAN_PACKET_HEADER_2])
        frame.extend(self.GetRawFrame())

        formatted_bytes = [f"0x{byte:02X}" for byte in frame]
        # Join the formatted bytes with commas
        hex_string = ', '.join(formatted_bytes)

        print(hex_string)

        return frame

    def Print(self):
        idHighByte = (self.id >> 8) & 0xFF
        idLowByte = self.id & 0xFF
        print("Id: " + str(self.id) + ", High byte: " + hex(idHighByte) + ", Low byte: " + hex(idLowByte))
        print("DLC: " + str(self.dlc))
        print("Data: " + str(self.data.hex(',')))

    def PrintFrame(self):
        formatted_bytes = [f"0x{byte:02X}" for byte in self.GetRawFrame()]
        # Join the formatted bytes with commas
        hex_string = ', '.join(formatted_bytes)

        print(hex_string)

    # Member variables
    id : int
    dlc : int
    data : bytes

class Socket:
    def __init__(self, host, port) -> None:
        self.host = host
        self.port = port
        self.isConnected = False

    def Init(self):
        self.s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        print("Binding " + self.host + ":" + str(self.port))
        self.s.bind((self.host, self.port))
        self.s.listen()

        self.stop_event = threading.Event()
    
    def Connect(self):
        print("Waiting for connection...")
        self.s.settimeout(5)
        self.conn, self.addr = self.s.accept()
        #self.conn.settimeout(0.5)
        print(f"Connected by {self.addr}")

    def Close(self):
        if self.IsConnected():
            self.s.shutdown(socket.SHUT_RDWR)
        self.s.close()

    def Receive(self) -> bytes:
        data = self.conn.recv(1024)
        return data
    
    def ReadAndPrint(self):
        self.stop_event.clear()
        while not self.stop_event.is_set():
            try:
                data = self.conn.recv(CAN_PACKET_SIZE)
                if data:
                    formatted_bytes = [f"0x{byte:02X}" for byte in data]
                    # Join the formatted bytes with commas
                    hex_string = ', '.join(formatted_bytes)

                    print("Received: " + hex_string)
                else:
                    continue
            except socket.error:
                break
    
    def StopContinuousReceive(self):
        self.stop_event.set()

    def ContinuousReceive(self):
        self.stop_event.clear()
        read_thread = threading.Thread(target=self.ReadAndPrint)
        read_thread.start()

        input("Press Enter to stop...\n")
        self.StopContinuousReceive()
        read_thread.join()
    
    def Send(self, data):
        self.conn.sendall(data)
    
    def IsConnected(self) -> bool:
        try:
            # Attempt to read bytes without blocking and without removing them from the buffer (peek only)
            data = self.s.recv(16, socket.MSG_DONTWAIT | socket.MSG_PEEK)
            if len(data) == 0:
                return False # Socket is closed
        except BlockingIOError:
            return True # Socket is open and reading from it would block
        except ConnectionResetError:
            return False # Socket was closed for some other reason
        except Exception as e:
            return False
        return True

    # Member variables
    conn : socket
    s : socket
    host : str
    port : int
    addr : any

if __name__ == '__main__':
    RUN = True

    s = Socket(HOST, PORT)
    s.Init()
    c = CANGenerator()

    while (not s.IsConnected()) and RUN:
        try:
            try:
                s.Connect()
            except socket.timeout:
                # Timeout, try again
                continue
            try:
                while RUN:
                    print("Choices:")
                    print("- 1: Send  \"randomized CAN\" to device")
                    print("- 2: Send multiple \"randomized CAN\" to device")
                    print("- 3: Send custom string to device")
                    print("- 4: Receive from device")
                    print("- 0: Quit")
                    choice = input("Enter your choice: ")

                    if choice == "1":
                        c.Randomize()
                        c.PrintFrame()
                        s.Send(c.GetFrameWithHeader())
                    elif choice == "2":
                        for x in range(0, NUMBER_OF_FRAMES_TO_SEND):
                            print(str(x) + ": ", end="")
                            c.Randomize()
                            s.Send(c.GetFrameWithHeader())
                    elif choice == "3":
                        stringToSend = input("String: ")
                        s.Send(str.encode(stringToSend))
                    elif choice == "4":
                        s.ContinuousReceive()
                    elif choice == "0":
                        print("Shutting down!")
                        RUN = False
                    else:
                        print("Invalid choice. Please try again.")
            except:
                print("Lost connection...")
        except KeyboardInterrupt:
            print('Server closing due to user interruption')
            RUN = False
    
    s.Close()