/*
 * cantasks.h
 *
 *  Created on: 3 maj 2024
 *      Author: tobjo
 */

#ifndef SRC_CANTASKS_H_
#define SRC_CANTASKS_H_

#include <stdio.h>
#include <kernel/dpl/DebugP.h>
#include <kernel/dpl/AddrTranslateP.h>
#include <kernel/dpl/SemaphoreP.h>
#include <drivers/mcan.h>
#include "ti_drivers_config.h"
#include "ti_drivers_open_close.h"
#include "ti_board_open_close.h"

#include "event_groups.h"

#include "can_message.h"

QueueHandle_t *xCanSendQueueCanFramesToSend;
QueueHandle_t *xCanSendQueueCanFramesReceived;

EventGroupHandle_t xCANSetupComplete;

#define BIT_0 ( 1 << 0 )

#define APP_MCAN_BASE_ADDR                       (CONFIG_MCAN0_BASE_ADDR)
#define APP_MCAN_INTR_NUM                        (CONFIG_MCAN0_INTR)
#define APP_MCAN_MSG_LOOP_COUNT                  (10U)
#define APP_MCAN_LOOPBACK_MODE_DISABLE           (FALSE)

/* Allocate Message RAM memory section to filter elements, buffers, FIFO */
/* Maximum STD Filter Element can be configured is 128 */
#define APP_MCAN_STD_ID_FILTER_CNT               (1U)
/* Maximum EXT Filter Element can be configured is 64 */
#define APP_MCAN_EXT_ID_FILTER_CNT               (0U)
/* Maximum TX Buffer + TX FIFO, combined can be configured is 32 */
#define APP_MCAN_TX_BUFF_CNT                     (1U)
#define APP_MCAN_TX_FIFO_CNT                     (0U)
/* Maximum TX Event FIFO can be configured is 32 */
#define APP_MCAN_TX_EVENT_FIFO_CNT               (0U)
/* Maximum RX FIFO 0 can be configured is 64 */
#define APP_MCAN_FIFO_0_CNT                      (0U)
/* Maximum RX FIFO 1 can be configured is 64 and
 * rest of the memory is allocated to RX buffer which is again of max size 64 */
#define APP_MCAN_FIFO_1_CNT                      (0U)

/* Standard Id configured in this app */
#define APP_MCAN_STD_ID                          (0xC0U)
#define APP_MCAN_STD_ID_MASK                     (0x7FFU)
#define APP_MCAN_STD_ID_SHIFT                    (18U)

#define APP_MCAN_EXT_ID_MASK                     (0x1FFFFFFFU)

/* Semaphore to indicate transfer completion */
static SemaphoreP_Object gMcanTxDoneSem, gMcanRxDoneSem;
static HwiP_Object       gMcanHwiObject;
static uint32_t          gMcanBaseAddr;

static void App_mcanInitMsgRamConfigParams(MCAN_MsgRAMConfigParams
                                           *msgRAMConfigParams)
{
    int32_t status;

    MCAN_initMsgRamConfigParams(msgRAMConfigParams);

    /* Configure the user required msg ram params */
    msgRAMConfigParams->lss = APP_MCAN_STD_ID_FILTER_CNT;
    msgRAMConfigParams->lse = APP_MCAN_EXT_ID_FILTER_CNT;
    msgRAMConfigParams->txBufCnt = APP_MCAN_TX_BUFF_CNT;
    msgRAMConfigParams->txFIFOCnt = APP_MCAN_TX_FIFO_CNT;
    /* Buffer/FIFO mode is selected */
    msgRAMConfigParams->txBufMode = MCAN_TX_MEM_TYPE_BUF;
    msgRAMConfigParams->txEventFIFOCnt = APP_MCAN_TX_EVENT_FIFO_CNT;
    msgRAMConfigParams->rxFIFO0Cnt = APP_MCAN_FIFO_0_CNT;
    msgRAMConfigParams->rxFIFO1Cnt = APP_MCAN_FIFO_1_CNT;
    /* FIFO blocking mode is selected */
    msgRAMConfigParams->rxFIFO0OpMode = MCAN_RX_FIFO_OPERATION_MODE_BLOCKING;
    msgRAMConfigParams->rxFIFO1OpMode = MCAN_RX_FIFO_OPERATION_MODE_BLOCKING;

    status = MCAN_calcMsgRamParamsStartAddr(msgRAMConfigParams);
    DebugP_assert(status == CSL_PASS);

    return;
}

static void App_mcanInitStdFilterElemParams(MCAN_StdMsgIDFilterElement *stdFiltElem,
                                            uint32_t bufNum)
{
    /* sfid1 defines the ID of the standard message to be stored. */
    stdFiltElem->sfid1 = APP_MCAN_STD_ID;
    /* As buffer mode is selected, sfid2 should be bufNum[0 - 63] */
    stdFiltElem->sfid2 = bufNum;
    //stdFiltElem->sfid2 = 0x7FF;
    /* Store message in buffer */
    stdFiltElem->sfec  = MCAN_STD_FILT_ELEM_BUFFER;
    /* Below configuration is ignored if message is stored in buffer */
    stdFiltElem->sft   = MCAN_STD_FILT_TYPE_RANGE;

    return;
}

static void App_mcanConfig(Bool enableInternalLpbk)
{
    MCAN_StdMsgIDFilterElement stdFiltElem[APP_MCAN_STD_ID_FILTER_CNT] = {0U};
    MCAN_InitParams            initParams = {0U};
    MCAN_ConfigParams          configParams = {0U};
    MCAN_MsgRAMConfigParams    msgRAMConfigParams = {0U};
    MCAN_BitTimingParams       bitTimes = {0U};
    /* MCAN Default Bit timing Parameters */
    MCAN_BitTimingParams gMcanBitTimingDefaultParams =
    {
        .nomRatePrescalar   = 0x7U,
        .nomTimeSeg1        = 0xFU,
        .nomTimeSeg2        = 0x2U,
        .nomSynchJumpWidth  = 0x1U,
        .dataRatePrescalar  = 0x13U,
        .dataTimeSeg1       = 0x5U,
        .dataTimeSeg2       = 0x0U,
        .dataSynchJumpWidth = 0x1U,
    };
    uint32_t                   i;

    /* Initialize MCAN module initParams */
    MCAN_initOperModeParams(&initParams);
    /* CAN FD Mode and Bit Rate Switch Enabled */
    //initParams.fdMode          = TRUE; // Example default
    //initParams.brsEnable       = TRUE; // Example default
    initParams.fdMode          = FALSE;
    initParams.brsEnable       = FALSE;

    /* Initialize MCAN module Global Filter Params */
    MCAN_initGlobalFilterConfigParams(&configParams);

    /* Initialize MCAN module Bit Time Params */
    /* Configuring default 1Mbps and 5Mbps as nominal and data bit-rate resp */
    MCAN_initSetBitTimeParams(&bitTimes); // Example default

    /* Initialize MCAN module Message Ram Params */
    App_mcanInitMsgRamConfigParams(&msgRAMConfigParams);

    /* Initialize Filter element to receive msg, should be same as tx msg id */
    for (i = 0U; i < APP_MCAN_STD_ID_FILTER_CNT; i++)
    {
        App_mcanInitStdFilterElemParams(&stdFiltElem[i], i);
    }
    /* wait for memory initialization to happen */
    while (FALSE == MCAN_isMemInitDone(gMcanBaseAddr))
    {}

    /* Put MCAN in SW initialization mode */
    MCAN_setOpMode(gMcanBaseAddr, MCAN_OPERATION_MODE_SW_INIT);
    while (MCAN_OPERATION_MODE_SW_INIT != MCAN_getOpMode(gMcanBaseAddr))
    {}

    /* Initialize MCAN module */
    MCAN_init(gMcanBaseAddr, &initParams);
    /* Configure MCAN module Gloabal Filter */
    MCAN_config(gMcanBaseAddr, &configParams);
    /* Configure Bit timings */
    //MCAN_setBitTime(gMcanBaseAddr, &bitTimes); // Example default
    MCAN_setBitTime(gMcanBaseAddr, &gMcanBitTimingDefaultParams);
    /* Configure Message RAM Sections */
    MCAN_msgRAMConfig(gMcanBaseAddr, &msgRAMConfigParams);
    /* Set Extended ID Mask */
    MCAN_setExtIDAndMask(gMcanBaseAddr, APP_MCAN_EXT_ID_MASK);

    /* Configure Standard ID filter element */
    for (i = 0U; i < APP_MCAN_STD_ID_FILTER_CNT; i++)
    {
        MCAN_addStdMsgIDFilter(gMcanBaseAddr, i, &stdFiltElem[i]);
    }
    if (TRUE == enableInternalLpbk)
    {
        MCAN_lpbkModeEnable(gMcanBaseAddr, MCAN_LPBK_MODE_INTERNAL, TRUE);
    }

    /* Take MCAN out of the SW initialization mode */
    MCAN_setOpMode(gMcanBaseAddr, MCAN_OPERATION_MODE_NORMAL);
    while (MCAN_OPERATION_MODE_NORMAL != MCAN_getOpMode(gMcanBaseAddr))
    {}

    return;
}

static void App_mcanIntrISR(void *arg)
{
    uint32_t intrStatus;

    intrStatus = MCAN_getIntrStatus(gMcanBaseAddr);
    MCAN_clearIntrStatus(gMcanBaseAddr, intrStatus);

    if (MCAN_INTR_SRC_TRANS_COMPLETE ==
        (intrStatus & MCAN_INTR_SRC_TRANS_COMPLETE))
    {
        SemaphoreP_post(&gMcanTxDoneSem);
    }

    /* If FIFO0/FIFO1 is used, then MCAN_INTR_SRC_DEDICATED_RX_BUFF_MSG macro
     * needs to be replaced by MCAN_INTR_SRC_RX_FIFO0_NEW_MSG/
     * MCAN_INTR_SRC_RX_FIFO1_NEW_MSG respectively */
    if (MCAN_INTR_SRC_DEDICATED_RX_BUFF_MSG ==
        (intrStatus & MCAN_INTR_SRC_DEDICATED_RX_BUFF_MSG))
    {
        SemaphoreP_post(&gMcanRxDoneSem);
    }

    return;
}

static void App_mcanEnableIntr(void)
{
    MCAN_enableIntr(gMcanBaseAddr, MCAN_INTR_MASK_ALL, (uint32_t)TRUE);
    MCAN_enableIntr(gMcanBaseAddr,
                    MCAN_INTR_SRC_RES_ADDR_ACCESS, (uint32_t)FALSE);
    /* Select Interrupt Line 0 */
    MCAN_selectIntrLine(gMcanBaseAddr, MCAN_INTR_MASK_ALL, MCAN_INTR_LINE_NUM_0);
    /* Enable Interrupt Line */
    MCAN_enableIntrLine(gMcanBaseAddr, MCAN_INTR_LINE_NUM_0, (uint32_t)TRUE);

    return;
}

bool MCAN_CheckForNewMessage(can_message_t *received_can_frame)
{
    MCAN_RxBufElement       rxMsg;
    MCAN_RxNewDataStatus    newDataStatus;
    MCAN_ErrCntStatus       errCounter;
    uint32_t bitPos, bufNum, fifoNum = 0U;
    TickType_t xTicksToWait = pdMS_TO_TICKS(10); // Convert 10ms to ticks

    if (received_can_frame == NULL) {
        return false; // Indicate failure if the provided pointer is NULL
    }

    // RX BEGIN
    /* Wait for Rx completion */
    //DebugP_logInfo("[MCAN_CheckForNewMessage] Waiting for RX completion\r\n");
    if(SemaphoreP_pend(&gMcanRxDoneSem, xTicksToWait) != SystemP_SUCCESS)
    {
        return false;
    }

//    SemaphoreP_pend(&gMcanRxDoneSem, SystemP_WAIT_FOREVER);

    DebugP_logInfo("[MCAN_CheckForNewMessage] RX completion\r\n");

    /* Checking for Rx Errors */
    MCAN_getErrCounters(gMcanBaseAddr, &errCounter);
    DebugP_assert((0U == errCounter.recErrCnt) &&
                  (0U == errCounter.canErrLogCnt));

    /* Get the new data status, indicates buffer num which received message */
    MCAN_getNewDataStatus(gMcanBaseAddr, &newDataStatus);
    MCAN_clearNewDataStatus(gMcanBaseAddr, &newDataStatus);

    /* Select buffer and fifo number, Buffer is used in this app */
    bufNum = 0U;
    fifoNum = MCAN_RX_FIFO_NUM_0;

    bitPos = (1U << bufNum);
    if (bitPos == (newDataStatus.statusLow & bitPos))
    {
        MCAN_readMsgRam(gMcanBaseAddr, MCAN_MEM_TYPE_BUF, bufNum, fifoNum, &rxMsg);

        received_can_frame->id = rxMsg.id;
        received_can_frame->dlc = rxMsg.dlc;

        for(int i = 0; i < rxMsg.dlc; i++)
        {
            received_can_frame->data[i] = rxMsg.data[i];
        }
    }
    else
    {
        DebugP_assert(FALSE);
    }
    // RX END
    return true;
}

void MCAN_SendMessage(can_message_t *frame)
{
    int32_t                 status = SystemP_SUCCESS;
    MCAN_TxBufElement       txMsg;
    MCAN_ProtocolStatus     protStatus;
    uint32_t                i, bufNum = 0U;

    // TX BEGIN
    /* Initialize message to transmit */
    MCAN_initTxBufElement(&txMsg);
    /* Standard message identifier 11 bit, stored into ID[28-18] */

    txMsg.id  = frame->id << 18; // Shift 18 bits since STD ID is stored in ID[28-18]
    txMsg.dlc = frame->dlc;
    txMsg.fdf = FALSE; /* Not CAN FD Frame Format */
    txMsg.xtd = FALSE; /* Extended id not configured */

    for (i = 0U; i < txMsg.dlc; i++)
    {
       txMsg.data[i] = frame->data[i];
    }

    DebugP_logInfo("[CanSend] Sending CAN frame with id: 0x%x\r\n", txMsg.id >> 18);
    DebugP_logInfo("[CanSend] Sending CAN frame with dlc: %d\r\n", txMsg.dlc);

    /* Select buffer number, 32 buffers available */
    bufNum = 0U;
    /* Enable Transmission interrupt for the selected buf num,
    * If FIFO is used, then need to send FIFO start index until FIFO count */
    status = MCAN_txBufTransIntrEnable(gMcanBaseAddr, bufNum, (uint32_t)TRUE);
    DebugP_assert(status == CSL_PASS);

    /* Write message to Msg RAM */
    MCAN_writeMsgRam(gMcanBaseAddr, MCAN_MEM_TYPE_BUF, bufNum, &txMsg);

    /* Add request for transmission, This function will trigger transmission */
    status = MCAN_txBufAddReq(gMcanBaseAddr, bufNum);
    DebugP_assert(status == CSL_PASS);

    /* Wait for Tx completion */
    DebugP_logInfo("[CanSend] Waiting for TX completion\r\n");
    SemaphoreP_pend(&gMcanTxDoneSem, SystemP_WAIT_FOREVER);
    DebugP_logInfo("[CanSend] TX completion\r\n");

    MCAN_getProtocolStatus(gMcanBaseAddr, &protStatus);
    /* Checking for Tx Errors */
    if (((MCAN_ERR_CODE_NO_ERROR != protStatus.lastErrCode) ||
        (MCAN_ERR_CODE_NO_CHANGE != protStatus.lastErrCode)) &&
       ((MCAN_ERR_CODE_NO_ERROR != protStatus.dlec) ||
        (MCAN_ERR_CODE_NO_CHANGE != protStatus.dlec)) &&
       (0U != protStatus.pxe))
    {
        DebugP_assert(FALSE);
    }
}

void vCanTask(void *pvParamters) {
    can_message_t receivedFrame;

    // Wait for setup to complete
    xEventGroupWaitBits(xCANSetupComplete, BIT_0, pdTRUE, pdFALSE, portMAX_DELAY);

    // 1. Check if xCanSendQueueCanFramesToSend has any packets to send
    //      a. If it does, send packets/clear queue
    // 2. Check if any new packets are available
    //      a. If there are, parse and add to xCanSendQueueCanFramesReceived
    while(1) {
        // 1.
        while (xQueueReceive(*xCanSendQueueCanFramesToSend, &receivedFrame, (TickType_t)0) == pdTRUE) {
            DebugP_logInfo("[vCanTask] Received from Queue\r\n");

            // a.
            MCAN_SendMessage(&receivedFrame);
        }

        // 2.
        if(MCAN_CheckForNewMessage(&receivedFrame)) {
            // a.
            DebugP_logInfo("[MCAN_CheckForNewMessage] - Sending to Queue\r\n");
            xQueueSend(*xCanSendQueueCanFramesReceived, &receivedFrame, (TickType_t) 0);
        }
    }
}

void initCAN(void *pvParameters)
{
    int32_t                 status = SystemP_SUCCESS;
    HwiP_Params             hwiPrms;
    uint32_t    gpioBaseAddr, pinNum;

    DebugP_logInfo("[CanInit] Begin\r\n");

    /* Enable MCAN transceiver */
    gpioBaseAddr = (uint32_t)AddrTranslateP_getLocalAddr(CONFIG_GPIO0_BASE_ADDR);
    pinNum       = CONFIG_GPIO0_PIN;
    GPIO_setDirMode(gpioBaseAddr, pinNum, GPIO_DIRECTION_OUTPUT);
    GPIO_pinWriteLow(gpioBaseAddr, pinNum);

    /* Construct Tx/Rx Semaphore objects */
    status = SemaphoreP_constructBinary(&gMcanTxDoneSem, 0);
    DebugP_assert(SystemP_SUCCESS == status);
    status = SemaphoreP_constructBinary(&gMcanRxDoneSem, 0);
    DebugP_assert(SystemP_SUCCESS == status);
    DebugP_logInfo("[CanInit] Constructed Tx/Rx Semaphores\r\n");

    /* Register interrupt */
    HwiP_Params_init(&hwiPrms);
    hwiPrms.intNum      = APP_MCAN_INTR_NUM;
    hwiPrms.callback    = &App_mcanIntrISR;
    status              = HwiP_construct(&gMcanHwiObject, &hwiPrms);
    DebugP_assert(status == SystemP_SUCCESS);
    DebugP_logInfo("[CanInit] Registered interrupts\r\n");

    /* Assign MCAN instance address */
    gMcanBaseAddr = (uint32_t) AddrTranslateP_getLocalAddr(APP_MCAN_BASE_ADDR);
    DebugP_logInfo("[CanInit] Assigned MCAN instance address\r\n");

    /* Configure MCAN module, Enable External LoopBack Mode */
    App_mcanConfig(APP_MCAN_LOOPBACK_MODE_DISABLE);
    DebugP_logInfo("[CanInit] Configured MCAN module, disabled external loopback\r\n");

    /* Enable Interrupts */
    App_mcanEnableIntr();
    DebugP_logInfo("[CanInit] Enabled interrupts\r\n");

    DebugP_logInfo("[CanInit] Setup Complete\r\n");
    xEventGroupSetBits(xCANSetupComplete, BIT_0);
    vTaskDelete(NULL);
}

#define CAN_TASK_SIZE (16384U/sizeof(configSTACK_DEPTH_TYPE))
StackType_t gCanInitTaskStack[CAN_TASK_SIZE] __attribute__((aligned(32)));
StackType_t gCanTaskStack[CAN_TASK_SIZE] __attribute__((aligned(32)));

void vCanTasks(QueueHandle_t *frameQueueIn, QueueHandle_t *frameQueueOut) {
    static const char CAN_INIT_TASK_NAME[] = "CanInit";
    static const char CAN_TASK_NAME[] = "CanTask";

    static StaticTask_t CanInitTaskHandle;
    static StaticTask_t CanTaskHandle;

    xCanSendQueueCanFramesToSend = frameQueueIn;
    xCanSendQueueCanFramesReceived = frameQueueOut;

    xCANSetupComplete = xEventGroupCreate();
    xEventGroupClearBits(xCANSetupComplete, BIT_0);

    xTaskCreateStatic(
        initCAN,
        CAN_INIT_TASK_NAME,
        CAN_TASK_SIZE,
        NULL,
        tskIDLE_PRIORITY + 2,
        gCanInitTaskStack,
        &CanInitTaskHandle);

    xTaskCreateStatic(
        vCanTask,
        CAN_TASK_NAME,
        CAN_TASK_SIZE,
        NULL,
        tskIDLE_PRIORITY + 1,
        gCanTaskStack,
        &CanTaskHandle);
}



#endif /* SRC_CANTASKS_H_ */
