/*
*
* Copyright (C) 2022-2023 Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
* license under copyrights and patents it now or hereafter owns or controls to make,
* have made, use, import, offer to sell and sell ("Utilize") this software subject to the
* terms herein.  With respect to the foregoing patent license, such license is granted
* solely to the extent that any such patent is necessary to Utilize the software alone.
* The patent license shall not apply to any combinations which include this software,
* other than combinations with devices manufactured by or for TI ("TI Devices").
* No hardware patent is licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this license
* (including the above copyright notice and the disclaimer and (if applicable) source
* code license limitations below) in the documentation and/or other materials provided
* with the distribution
*
* Redistribution and use in binary form, without modification, are permitted provided
* that the following conditions are met:
*
* *       No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* *       any redistribution and use are licensed by TI for use only with TI Devices.
*
* *       Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution of the
* source code are permitted provided that the following conditions are met:
*
* *       any redistribution and use of the source code, including any resulting derivative
* works, are licensed by TI for use only with TI Devices.
*
* *       any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

/**
 *  \file     Adc.h
 *
 *  \brief    This file contains interface header for ADC MCAL driver
 *
 */

/**
 *  \defgroup MCAL_ADC_API ADC Handler and Driver API
 *
 *  The ADC Driver MCAL provides low level access to ADC peripheral
 *  present on Device.<br>
 *  The ADC Driver implements a standardized interface specified in the
 *  AUTOSAR_SWS_ADCDriver document.<br>
 *
 *  Only the following combinations of modes are supported:
 *      - One-shot,   Software Trigger,          Single    Access Mode
 *      - Continuous, Software Trigger, Circular Single    Access Mode
 *      - Continuous, Software Trigger, Linear   Single    Access Mode
 *        (similar to one-shot mode)
 *      - Continuous, Software Trigger, Circular Streaming Access Mode
 *      - Continuous, Software Trigger, Linear   Streaming Access Mode
 *
 *  Following combinations of modes are not supported:
 *      - All hardware trigger modes
 *      - One-shot,   Software Trigger, Stream Access (Linear + Circular) Mode
 *
 *  @sa MCAL_ADC_CFG
 *  @{
 */

/*
 * Below are the global design requirements which are met by this ADC
 * driver which can't be mapped to a particular source ID
 */
/*
 * Design: MCAL-5723,MCAL-5861
 */

/*
 * Below are the ADC's module environment design requirements which can't
 * be mapped to this driver
 */
/*
 * Design: MCAL-5805,MCAL-5708,MCAL-5739,MCAL-5748
 */

 /*
  * Note: MCAL-5700,MCAL-5834 are prerequisites which are done by SBL/GEL files.
  */


#ifndef ADC_H_
#define ADC_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "Adc_Priv.h"

#ifdef __cplusplus
extern "C"
{
#endif

/* ========================================================================== */
/*                            Version Checks                                  */
/* ========================================================================== */

/**
 *  \name ADC Driver Module SW Version Info
 *
 *  Defines for ADC Driver version used for compatibility checks
 *  @{
 */
/** \brief Driver Implementation Major Version */
#define ADC_SW_MAJOR_VERSION            (8U)
/** \brief Driver Implementation Minor Version */
#define ADC_SW_MINOR_VERSION            (6U)
/** \brief Driver Implementation Patch Version */
#define ADC_SW_PATCH_VERSION            (2U)
/* @} */

/**
 *  \name ADC Driver Module AUTOSAR Version Info
 *
 *  Defines for ADC Driver AUTOSAR version used for compatibility checks
 *  @{
 */
/** \brief AUTOSAR Major version specification implemented by ADC Driver */
#define ADC_AR_RELEASE_MAJOR_VERSION    (4U)
/** \brief AUTOSAR Minor  version specification implemented by ADC Driver */
#define ADC_AR_RELEASE_MINOR_VERSION    (3U)
/** \brief AUTOSAR Patch version specification implemented by ADC Driver */
#define ADC_AR_RELEASE_REVISION_VERSION (1U)
/* @} */

/**
 *  \name ADC Driver ID Info
 *  @{
 */
/** \brief Texas Instruments Vendor ID */
#define ADC_VENDOR_ID                   ((uint16) 44U)
/** \brief ADC Driver Module ID */
#define ADC_MODULE_ID                   ((uint16) 123U)
/** \brief ADC Driver Instance ID */
#define ADC_INSTANCE_ID                 ((uint8) 0U)
/* @} */

/* Boolean Macros for ADC module. */
#define ADC_TRUE                         (TRUE)
#define ADC_FALSE                        (FALSE)

/**
 *  \brief The default channel resolution supported by the current ADC hardware.
 *  No other value is supported by the hardware.
 */
#define ADC_DEF_CHANNEL_RESOLUTION      ((uint8) 12U)
#define ADC_MAX_CHANNEL_VALUE_TYPE      ((uint8) 16U)

/**
 *  \brief Number of actual HW channels - in terms of ADC HW, this represents
 *  the actual channel input to the ADC module.
 *  Note: This is a fixed value as per the ADC module and can't be changed.
 */
#define ADC_NUM_HW_CHANNEL              (6U)
/** \brief Minimum value of HW channel ID */
#define ADC_MIN_HW_CHANNEL_ID           (0U)
/** \brief Maximum value of HW channel ID */
#define ADC_MAX_HW_CHANNEL_ID           (ADC_NUM_HW_CHANNEL - 1U)

/** \brief Minimum value of open delay */
#define ADC_MIN_OPEN_DELAY              (0x00U)
/** \brief Maximum value of open delay */
#define ADC_MAX_OPEN_DELAY              (0x3FFFFU)
/** \brief Minimum value of sample delay */
#define ADC_MIN_SAMPLE_DELAY            (0x00U)
/** \brief Maximum value of sample delay */
#define ADC_MAX_SAMPLE_DELAY            (0xFFU)
/** \brief Minimum value of range */
#define ADC_MIN_RANGE                   (0x00U)
/** \brief Maximum value of range */
#define ADC_MAX_RANGE                   (0xFFFU)

/**
 *  \name ADC Driver ISR category level
 *
 *  Defines for ADC Driver ISR ISR category level
 *  @{
 */
/** \brief void ISR type */
#define ADC_ISR_VOID                    (0x00U)
/** \brief Category 1 ISR type */
#define ADC_ISR_CAT1                    (0x01U)
/** \brief Category 2 ISR type */
#define ADC_ISR_CAT2                    (0x02U)
/* @} */

/*
 *Design: MCAL-5687
 */
/**
 *  \name ADC Error Codes
 *
 *  Error codes returned by ADC functions
 *  @{
 */
#ifndef ADC_E_UNINIT
/** \brief API service used without module initiali-zation */
#define ADC_E_UNINIT                    ((uint8) 0x0AU)
#endif
#ifndef ADC_E_BUSY
/** \brief API service called during ongoing process */
#define ADC_E_BUSY                      ((uint8) 0x0BU)
#endif
#ifndef ADC_E_IDLE
/** \brief API service called while no conversion is ongoing. */
#define ADC_E_IDLE                      ((uint8) 0x0CU)
#endif
#ifndef ADC_E_ALREADY_INITIALIZED
/**
 *  \brief API Adc_Init service called while the ADC driver has already been
 *  initialized
 */
#define ADC_E_ALREADY_INITIALIZED       ((uint8) 0x0DU)
#endif
#ifndef ADC_E_PARAM_CONFIG
/** \brief API service called with incorrect configuration parameter. */
#define ADC_E_PARAM_CONFIG              ((uint8) 0x0EU)
#endif
#ifndef ADC_E_PARAM_POINTER
/** \brief API servcie called with invalid data buffer pointer. */
#define ADC_E_PARAM_POINTER             ((uint8) 0x14U)
#endif
#ifndef ADC_E_PARAM_GROUP
/** \brief API servcie called with invalid group ID. */
#define ADC_E_PARAM_GROUP               ((uint8) 0x15U)
#endif
#ifndef ADC_E_WRONG_CONV_MODE
/**
 *  \brief API servcie called on a group with conversion mode configured as
 *  continuous.
 */
#define ADC_E_WRONG_CONV_MODE           ((uint8) 0x16U)
#endif
#ifndef ADC_E_WRONG_TRIGG_SRC
/** \brief API servcie called for group with wrong trigger source */
#define ADC_E_WRONG_TRIGG_SRC           ((uint8) 0x17U)
#endif
#ifndef ADC_E_NOTIF_CAPABILITY
/**
 *  \brief Enable/disable notification function for a group whose configuration
 *  set has no notification available.
 */
#define ADC_E_NOTIF_CAPABILITY          ((uint8) 0x18U)
#endif
#ifndef ADC_E_BUFFER_UNINIT
/** \brief Conversion started and result buffer pointer is not initialized. */
#define ADC_E_BUFFER_UNINIT             ((uint8) 0x19U)
#endif
#ifndef ADC_E_NOT_DISENGAGED
/** \brief One or more ADC group/channel not in IDLE state. */
#define ADC_E_NOT_DISENGAGED            ((uint8) 0x1AU)
#endif
#ifndef ADC_E_POWER_STATE_NOT_SUPPORTED
/** \brief Unsupported power state request. */
#define ADC_E_POWER_STATE_NOT_SUPPORTED ((uint8) 0x1BU)
#endif
#ifndef ADC_E_TRANSITION_NOT_POSSIBLE
/** \brief Requested power state can not be reached directly. */
#define ADC_E_TRANSITION_NOT_POSSIBLE   ((uint8) 0x1CU)
#endif
#ifndef ADC_E_PERIPHERAL_NOT_PREPARED
/** \brief ADC not prepared for target power state. */
#define ADC_E_PERIPHERAL_NOT_PREPARED   ((uint8) 0x1DU)
#endif
/* @} */

/**
 *  \name ADC Service Ids
 *
 *  The Service Id is one of the argument to Det_ReportError function and
 *  is used to identify the source of the error
 *  @{
 */
/** \brief Adc_Init() API Service ID */
#define ADC_SID_INIT                        ((uint8) 0x00U)
/** \brief Adc_DeInit() API Service ID */
#define ADC_SID_DEINIT                      ((uint8) 0x01U)
/** \brief Adc_StartGroupConversion() API Service ID */
#define ADC_SID_START_GROUP_CONVERSION      ((uint8) 0x02U)
/** \brief Adc_StopGroupConversion() API Service ID */
#define ADC_SID_STOP_GROUP_CONVERSION       ((uint8) 0x03U)
/** \brief Adc_ReadGroup() API Service ID */
#define ADC_SID_READ_GROUP                  ((uint8) 0x04U)
/** \brief Adc_EnableHardwareTrigger() API Service ID */
#define ADC_SID_ENABLE_HARDWARE_TRIGGER     ((uint8) 0x05U)
/** \brief Adc_DisableHardwareTrigger() API Service ID */
#define ADC_SID_DISABLE_HARDWARE_TRIGGER    ((uint8) 0x06U)
/** \brief Adc_EnableGroupNotification() API Service ID */
#define ADC_SID_ENABLE_GROUP_NOTIFICATION   ((uint8) 0x07U)
/** \brief Adc_DisableGroupNotification() API Service ID */
#define ADC_SID_DISABLE_GROUP_NOTIFICATION  ((uint8) 0x08U)
/** \brief Adc_GetGroupStatus() API Service ID */
#define ADC_SID_GET_GROUP_STATUS            ((uint8) 0x09U)
/** \brief Adc_GetVersionInfo() API Service ID */
#define ADC_SID_GET_VERSION_INFO            ((uint8) 0x0AU)
/** \brief Adc_GetStreamLastPointer() API Service ID */
#define ADC_SID_GET_STREAM_LAST_POINTER     ((uint8) 0x0BU)
/** \brief Adc_SetupResultBuffer() API Service ID */
#define ADC_SID_SETUP_RESULT_BUFFER         ((uint8) 0x0CU)
/** \brief Adc_SetPowerState() API Service ID */
#define ADC_SID_SET_POWER_STATE             ((uint8) 0x10U)
/** \brief Adc_GetCurrentPowerState() API Service ID */
#define ADC_SID_GET_CURRENT_POWER_STATE     ((uint8) 0x11U)
/** \brief Adc_GetTargetPowerState() API Service ID */
#define ADC_SID_GET_TARGET_POWER_STATE      ((uint8) 0x12U)
/** \brief Adc_PreparePowerState() API Service ID */
#define ADC_SID_PREPARE_POWER_STATE         ((uint8) 0x13U)
/** \brief Adc_Main_PowerTransitionManager() API Service ID */
#define ADC_SID_MAIN_POWER_TRANSITION_MANAGER       ((uint8) 0x14U)
/** \brief IoHwAb_AdcNotification() API Service ID */
#define ADC_SID_IOHWAB_NOTIFICATION                 ((uint8) 0x20U)
/** \brief IoHwAb_Adc_NotifyReadyForPowerState() API Service ID */
#define ADC_SID_IOHWAB_NOTIFY_READY_FOR_POWER_STATE ((uint8) 0x70U)
/* @} */
/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/**
 *  \brief Service for ADC initialization.
 *
 *  \verbatim
 *  Service name        : Adc_Init
 *  Syntax              : void Adc_Init( const Adc_ConfigType* CfgPtr )
 *  Mode                : Supervisor Mode (Privileged Mode)
 *  Service ID[hex]     : 0x00
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : Pointer to configuration set in Variant PB
 *                        (Variant PC requires a NULL_PTR).
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Initializes the ADC hardware units and driver
 *  \endverbatim
 */
FUNC(void, ADC_CODE) Adc_Init(
                    P2CONST(Adc_ConfigType, AUTOMATIC, ADC_PBCFG) CfgPtr);

#if (STD_ON == ADC_DEINIT_API)
/**
 *  \brief This service sets all ADC HW Units to a state comparable to their
 *  power on reset state
 *
 *  \verbatim
 *  Service name        : Adc_DeInit
 *  Syntax              : void Adc_DeInit(void)
 *  Mode                : Supervisor Mode (Privileged Mode)
 *  Service ID[hex]     : 0x01
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : None
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Returns all ADC HW Units to a state comparable to
 *                        their power on reset state
 *  \endverbatim
 */
FUNC(void, ADC_CODE) Adc_DeInit(void);
#endif  /* #if (STD_ON == ADC_DEINIT_API) */

/**
 *  \brief This service initializes ADC driver with the group specific result
 *  buffer start address where the conversion results will be stored.
 *
 *  \verbatim
 *  Service name        : Adc_SetupResultBuffer
 *  Syntax              : Std_ReturnType Adc_SetupResultBuffer(
 *                                          Adc_GroupType Group,
 *                                          Adc_ValueGroupType* DataBufferPtr)
 *  Mode                : User Mode (Non-Privileged Mode)
 *  Service ID[hex]     : 0x0C
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Group - Numeric ID of requested ADC channel group.
 *                        DataBufferPtr - pointer to result data buffer
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : Std_ReturnType
 *                        E_OK: result buffer pointer initialized correctly
 *                        E_NOT_OK: operation failed or development error occured
 *  Description         : Initializes ADC driver with the group specific result
 *                        buffer start address where the conversion results
 *                        will be stored. The application has to ensure that
 *                        the application buffer, where DataBufferPtr points
 *                        to, can hold all the conversion results of the
 *                        specified group. The initialization with
 *                        Adc_SetupResultBuffer is required after reset,
 *                        before a group conversion can be started.
 *  \endverbatim
 */
FUNC(Std_ReturnType, ADC_CODE) Adc_SetupResultBuffer(
                                    Adc_GroupType Group,
                                    const Adc_ValueGroupType *DataBufferPtr);

/**
 *  \brief Returns the conversion status of the requested ADC Channel group.
 *
 *  \verbatim
 *  Service name        : Adc_GetGroupStatus
 *  Syntax              : Adc_StatusType Adc_GetGroupStatus( Adc_GroupType Group )
 *  Mode                : User Mode (Non-Privileged Mode)
 *  Service ID[hex]     : 0x09
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Group - Numeric ID of requested ADC channel group.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : Adc_StatusType - Conversion status for the requested
 *                        group
 *  Description         : Returns the conversion status of the requested ADC
 *                        Channel group.
 *  \endverbatim
 */
FUNC(Adc_StatusType, ADC_CODE) Adc_GetGroupStatus(Adc_GroupType Group);

/**
 *  \brief Returns the number of valid samples per channel, stored in the
 *  result buffer.
 *
 *  \verbatim
 *  Service name        : Adc_GetStreamLastPointer
 *  Syntax              : Adc_StreamNumSampleType Adc_GetStreamLastPointer(
 *                              Adc_GroupType Group,
 *                              Adc_ValueGroupType** PtrToSamplePtr )
 *  Mode                : User Mode (Non-Privileged Mode)
 *  Service ID[hex]     : 0x0B
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Group - Numeric ID of requested ADC channel group.
 *  Parameters (inout)  : None
 *  Parameters (out)    : PtrToSamplePtr - Pointer to result buffer pointer.
 *  Return value        : Adc_StreamNumSampleType - Number of valid samples
 *                        per channel
 *  Description         : Returns the number of valid samples per channel,
 *                        stored in the result buffer. Reads a pointer,
 *                        pointing to a position in the group result buffer.
 *                        With the pointer position, the results of all group
 *                        channels of the last completed conversion round can
 *                        be accessed. With the pointer and the return value,
 *                        all valid group conversion results can be accessed
 *                        (the user has to take the layout of the result buffer
 *                        into account).
 *  \endverbatim
 */
FUNC(Adc_StreamNumSampleType, ADC_CODE) Adc_GetStreamLastPointer(
        Adc_GroupType Group,
        Adc_ValueGroupType **PtrToSamplePtr);

#if (STD_ON == ADC_ENABLE_START_STOP_GROUP_API)
/**
 *  \brief Starts the conversion of all channels of the requested ADC
 *  Channel group
 *
 *  \verbatim
 *  Service name        : Adc_StartGroupConversion
 *  Syntax              : void Adc_StartGroupConversion( Adc_GroupType Group )
 *  Mode                : Supervisor Mode (Privileged Mode)
 *  Service ID[hex]     : 0x02
 *  Sync/Async          : Asynchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Group - Numeric ID of requested ADC Channel group.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Starts the conversion of all channels of the
 *                        requested ADC Channel group.
 *  \endverbatim
 */
FUNC(void, ADC_CODE) Adc_StartGroupConversion(Adc_GroupType Group);

/**
 *  \brief Stops the conversion of the requested ADC Channel group
 *
 *  \verbatim
 *  Service name        : Adc_StopGroupConversion
 *  Syntax              : void Adc_StopGroupConversion( Adc_GroupType Group )
 *  Mode                : Supervisor Mode (Privileged Mode)
 *  Service ID[hex]     : 0x03
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Group - Numeric ID of requested ADC Channel group.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Stops the conversion of the requested ADC Channel
 *                        group.
 *  \endverbatim
 */
FUNC(void, ADC_CODE) Adc_StopGroupConversion(Adc_GroupType Group);
#endif  /* #if (STD_ON == ADC_ENABLE_START_STOP_GROUP_API) */

#if (STD_ON == ADC_READ_GROUP_API)
/**
 *  \brief Reads the group conversion result of the last completed conversion
 *  round of the requested group and stores the channel values starting at
 *  the DataBufferPtr address
 *
 *  \verbatim
 *  Service name        : Adc_ReadGroup
 *  Syntax              : Std_ReturnType Adc_ReadGroup(
 *                              Adc_GroupType Group,
 *                              Adc_ValueGroupType* DataBufferPtr )
 *  Mode                : User Mode (Non-Privileged Mode)
 *  Service ID[hex]     : 0x04
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Group - Numeric ID of requested ADC channel group.
 *                        DataBufferPtr - ADC results of all channels of the
 *                        selected group are stored in the data buffer
 *                        addressed with the pointer
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : Std_ReturnType
 *                        E_OK: results are available and written to the data buffer
 *                        E_NOT_OK: no results are available or development error occured
 *  Description         : Reads the group conversion result of the last
 *                        completed conversion round of the requested group
 *                        and stores the channel values starting at the
 *                        DataBufferPtr address. The group channel values are
 *                        stored in ascending channel number order (in contrast
 *                        to the storage layout of the result buffer if
 *                        streaming access is configured).
 *  \endverbatim
 */
FUNC(Std_ReturnType, ADC_CODE) Adc_ReadGroup(
            Adc_GroupType Group,
            Adc_ValueGroupType *DataBufferPtr);
#endif  /* #if (STD_ON == ADC_READ_GROUP_API) */

#if (STD_ON == ADC_GRP_NOTIF_CAPABILITY_API)
/**
 *  \brief Enables the notification mechanism for the requested ADC Channel
 *   group.
 *
 *  \verbatim
 *  Service name        : Adc_EnableGroupNotification
 *  Syntax              : void Adc_EnableGroupNotification( Adc_GroupType Group )
 *  Mode                : User Mode (Non-Privileged Mode)
 *  Service ID[hex]     : 0x07
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Group - Numeric ID of requested ADC channel group.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Enables the notification mechanism for the requested
 *                        ADC Channel group.
 *  \endverbatim
 */
FUNC(void, ADC_CODE) Adc_EnableGroupNotification(Adc_GroupType Group);

/**
 *  \brief Disables the notification mechanism for the requested ADC Channel
 *   group.
 *
 *  \verbatim
 *  Service name        : Adc_DisableGroupNotification
 *  Syntax              : void Adc_DisableGroupNotification( Adc_GroupType Group )
 *  Mode                : User Mode (Non-Privileged Mode)
 *  Service ID[hex]     : 0x08
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Group - Numeric ID of requested ADC channel group.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Disables the notification mechanism for the requested
 *                        ADC Channel group.
 *  \endverbatim
 */
FUNC(void, ADC_CODE) Adc_DisableGroupNotification(Adc_GroupType Group);
#endif  /* #if (STD_ON == ADC_GRP_NOTIF_CAPABILITY_API) */

#if (STD_ON == ADC_VERSION_INFO_API)
/**
 *  \brief This service returns the version information of this module.
 *
 *  \verbatim
 *  Service name        : Adc_GetVersionInfo
 *  Syntax              : void Adc_GetVersionInfo( Std_VersionInfoType* versioninfo )
 *  Mode                : User Mode (Non-Privileged Mode)
 *  Service ID[hex]     : 0x0A
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : None
 *  Parameters (inout)  : None
 *  Parameters (out)    : versioninfo - Pointer to where to store the version
 *                        information of this module
 *  Return value        : None
 *  Description         : Returns the version information of this module
 *  \endverbatim
 */
FUNC(void, ADC_CODE) Adc_GetVersionInfo(
        P2VAR(Std_VersionInfoType, AUTOMATIC, ADC_APPL_DATA) versioninfo);
#endif  /* #if (STD_ON == ADC_VERSION_INFO_API) */

#if (STD_ON == ADC_REGISTER_READBACK_API)
/**
 *  \brief This function reads the important registers of the hardware unit
 *  and returns the value in the structure.
 *
 *  This API should be called after Adc_Init is called. Otherwise this API will
 *  return E_NOT_OK.
 *
 *  This API could be used to readback the register contents after Adc_Init
 *  and then the readback value could be compared during ADC execution to
 *  check the correctness of the HW unit. Since this API is used for this
 *  purpose, the register returned are the ones which doesn't change after init
 *  based on conv or channel config.
 *
 *  \verbatim
 *  Service name        : Adc_RegisterReadback
 *  Mode                : Supervisor Mode (Privileged Mode)
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : HWUnit - ADC Hardware microcontroller peripheral
 *                        unit ID. If this is invalid, then the API will
 *                        return E_NOT_OK.
 *  Parameters (inout)  : RegRbPtr - Pointer to where to store the readback
 *                        values. If this pointer is NULL_PTR, then the API
 *                        will return E_NOT_OK.
 *  Return value        : Std_ReturnType
 *                        E_OK: Register read back has been done
 *                        E_NOT_OK: Register read back failed
 *  \endverbatim
 */
FUNC(Std_ReturnType, ADC_CODE) Adc_RegisterReadback(
    Adc_HWUnitType HWUnit,
    P2VAR(Adc_RegisterReadbackType, AUTOMATIC, ADC_APPL_DATA) RegRbPtr);
#endif  /* #if (STD_ON == ADC_REGISTER_READBACK_API) */

#if (ADC_HW_TRIGGER_API == STD_ON)
/**
 *  \brief Enables the hardware trigger for the requested ADC Channel group.
 *
 *  \verbatim
 *  Service name        : Adc_EnableHardwareTrigger
 *  Syntax              : void Adc_EnableHardwareTrigger( Adc_GroupType Group )
 *  Mode                : Supervisor Mode (Privileged Mode)
 *  Service ID[hex]     : 0x05
 *  Sync/Async          : Asynchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Group - Numeric ID of requested ADC Channel group.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : This function will enable the HW trigger source for the requested
 *                        ADC channel group.
 *  \endverbatim
 */
FUNC(void, ADC_CODE) Adc_EnableHardwareTrigger
(
    VAR(Adc_GroupType, AUTOMATIC) Group
);

/**
 *  \brief Disables the hardware trigger for the requested ADC Channel group.
 *
 *  \verbatim
 *  Service name        : Adc_DisableHardwareTrigger
 *  Syntax              : void Adc_DisableHardwareTrigger( Adc_GroupType Group )
 *  Mode                : Supervisor Mode (Privileged Mode)
 *  Service ID[hex]     : 0x06
 *  Sync/Async          : Asynchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Group - Numeric ID of requested ADC Channel group.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : This function will disable the HW trigger source for the requested
 *                        ADC channel group.
 *  \endverbatim
 */
FUNC(void, ADC_CODE) Adc_DisableHardwareTrigger
(
    VAR(Adc_GroupType, AUTOMATIC) Group
);

#endif

#ifdef __cplusplus
}
#endif

#endif  /* #ifndef ADC_H_ */

/* @} */
