/*
*    
* Copyright (C) 2022-2023 Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
* license under copyrights and patents it now or hereafter owns or controls to make,
* have made, use, import, offer to sell and sell ("Utilize") this software subject to the
* terms herein.  With respect to the foregoing patent license, such license is granted
* solely to the extent that any such patent is necessary to Utilize the software alone.
* The patent license shall not apply to any combinations which include this software,
* other than combinations with devices manufactured by or for TI ("TI Devices").
* No hardware patent is licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this license
* (including the above copyright notice and the disclaimer and (if applicable) source
* code license limitations below) in the documentation and/or other materials provided
* with the distribution
*
* Redistribution and use in binary form, without modification, are permitted provided
* that the following conditions are met:
*
* *       No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* *       any redistribution and use are licensed by TI for use only with TI Devices.
*
* *       Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution of the
* source code are permitted provided that the following conditions are met:
*
* *       any redistribution and use of the source code, including any resulting derivative
* works, are licensed by TI for use only with TI Devices.
*
* *       any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

/**
 *  \file     Adc_Priv.c
 *
 *  \brief    This file contains ADC MCAL driver internal functions
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "Adc_Cfg.h"
#include "Adc.h"
#include "Adc_Dbg.h"
#include "Adc_Priv.h"
#include "Adc_Irq.h"
#include "Dem.h"
#include "Adc_RegResult.h"

#if (STD_ON == ADC_DMA_MODE)
#include "Cdd_Dma.h"
#endif

#if ((STD_ON == ADC_ERR_LOG) || (STD_ON == ADC_GROUP_LOG))
#include "Os.h"
#endif
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */
static void Adc_scheduleGroup(Adc_GroupObjType *groupObj);
static void Adc_checkAndSchedule(Adc_HwUnitObjType *hwUnitObj);

static void Adc_procIsr(Adc_HwUnitObjType *hwUnitObj,
                        Adc_GroupObjType  *groupObj);
static void Adc_resetGroupObjOnResume(Adc_GroupObjType *groupObj);

static void Adc_hwConfig(Adc_GroupObjType *groupObj, uint32 baseAddr);
static void Adc_hwStart(uint32 baseAddr);
static void Adc_hwStop(uint32 baseAddr, const Adc_GroupObjType *groupObj);
static void Adc_hwErrRecovery(Adc_GroupObjType *groupObj, uint32 baseAddr);
static void Adc_hwSetDefReg(uint32 baseAddr);
static void Adc_setGroupStatusPostIsr(Adc_HwUnitObjType *hwUnitObj,
                                      Adc_GroupObjType  *groupObj,
                                      uint32             convComplete,
                                      uint32             streamComplete);
static Std_ReturnType Adc_startGroupCheckStatus(const Adc_GroupObjType *groupObj);

#if (STD_ON == ADC_DEV_ERROR_DETECT)
static Std_ReturnType Adc_checkGroupCfgRangeParameters(
const Adc_GroupConfigType *groupCfg);
static void Adc_checkChannelParams(const Adc_ChannelConfigType *chCfg);
static Std_ReturnType Adc_checkGroupParameters(
const Adc_GroupConfigType *groupCfg, const Adc_ConfigType *cfgPtr);
#endif  /* #if (STD_ON == ADC_DEV_ERROR_DETECT) */

#if (STD_ON == ADC_DMA_MODE)
static void Adc_IrqDmaTxRx();
static void AdcDma_FreeModuleChannelConfigured(uint32 dma_ch);
#endif  /* #if (STD_ON == ADC_DMA_MODE) */
/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
#if (STD_ON == ADC_DMA_MODE)
#define ADC_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Adc_MemMap.h"
/** \brief ADC DMA object */
extern VAR(Adc_DmaChannelObjType, ADC_VAR_CLEARED) 
Adc_DmaConfigObj[ADC_DMA_CHANNELS_CONFIGURED];
#define ADC_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Adc_MemMap.h"
#endif  /* #if (STD_ON == ADC_DMA_MODE) */


#if (STD_ON == ADC_GROUP_LOG)
#define ADC_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Adc_MemMap.h"
/** \brief ADC group log object */
VAR(Adc_GroupLogType, ADC_VAR_CLEARED) Adc_GroupLogObj;
#define ADC_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Adc_MemMap.h"
#endif  /* #if (STD_ON == ADC_GROUP_LOG) */

#if (STD_ON == ADC_ERR_LOG)
#define ADC_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Adc_MemMap.h"
/** \brief ADC FIFO error log object */
VAR(Adc_ErrLogType, ADC_VAR_CLEARED) Adc_ErrLogObj;
#define ADC_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Adc_MemMap.h"
#endif  /* #if (STD_ON == ADC_ERR_LOG) */
/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#define ADC_START_SEC_CODE
#include "Adc_MemMap.h"

/*
 * Design: MCAL-5797,MCAL-5812,MCAL-5884
 */
void Adc_hwUnitInit(Adc_HwUnitObjType *hwUnitObj)
{
    uint32   baseAddr;

    /* Assign base address */
    hwUnitObj->baseAddr = hwUnitObj->hwUnitCfg.baseAddr;
    hwUnitObj->resultBaseAddr = hwUnitObj->hwUnitCfg.baseAddr;

    /* Get the Result Register. */
    switch(hwUnitObj->hwUnitCfg.hwUnitId)
    {
        case 0: hwUnitObj->resultBaseAddr = MCAL_CONTROLSS_ADC0_RESULT; break;
        case 1: hwUnitObj->resultBaseAddr = MCAL_CONTROLSS_ADC1_RESULT; break;
        case 2: hwUnitObj->resultBaseAddr = MCAL_CONTROLSS_ADC2_RESULT; break;
        case 3: hwUnitObj->resultBaseAddr = MCAL_CONTROLSS_ADC3_RESULT; break;
        case 4: hwUnitObj->resultBaseAddr = MCAL_CONTROLSS_ADC4_RESULT; break;
    }


    /*
     * Init the HW instance
     */
    baseAddr = hwUnitObj->baseAddr;

    /* Set the Prescaler for ADC HW unit. */
    ADC_setPrescaler(baseAddr, hwUnitObj->hwUnitCfg.prescale);

    /* Set the Resolution for ADC HW unit. */
    ADC_setMode(baseAddr, hwUnitObj->hwUnitCfg.resolution, ADC_MODE_SINGLE_ENDED);

    /* Disable all Interrupt */
    ADC_disableInterrupt(baseAddr, 0);
    ADC_disableInterrupt(baseAddr, 1);
    ADC_disableInterrupt(baseAddr, 2);
    ADC_disableInterrupt(baseAddr, 3);

    /* Disable continous mode */
    ADC_disableContinuousMode(baseAddr, 0);
    ADC_disableContinuousMode(baseAddr, 1);
    ADC_disableContinuousMode(baseAddr, 2);
    ADC_disableContinuousMode(baseAddr, 3);
    
    /* Clear Interruptstatus */
    ADC_clearInterruptStatus(baseAddr, ADC_INT_NUMBER1);
    ADC_clearInterruptStatus(baseAddr, ADC_INT_NUMBER2);
    ADC_clearInterruptStatus(baseAddr, ADC_INT_NUMBER3);
    ADC_clearInterruptStatus(baseAddr, ADC_INT_NUMBER4);
    
    /* Clear Status Register. */
    HW_WR_REG16((baseAddr + MCAL_CSL_ADC_ADCEVTCLR), 0xFFFF);    
    
    /*
     * Power up Analog Front End
     */
    ADC_enableConverter(baseAddr);

    return;
}

/*
 * Design: MCAL-5792,MCAL-5847
 */
void Adc_hwUnitDeInit(Adc_HwUnitObjType *hwUnitObj)
{
    uint32 baseAddr;
#if (STD_ON == ADC_DMA_MODE)
    Adc_DriverObjType *drvObj = &Adc_DrvObj;
    Adc_GroupObjType *groupObj;
    CddDma_HandleType adcDmaHandle;
    uint32 dmaadcChIdx, dmaCh, dmaChIdx, regionId;
#endif
    baseAddr = hwUnitObj->baseAddr;
    /*
     * De-Init the HW instance
     */
    Adc_hwSetDefReg(baseAddr);    
    
#if (STD_ON == ADC_DMA_MODE)  
        adcDmaHandle = Cdd_Dma_GetHandle();
        baseAddr = CDD_EDMA_lld_getBaseAddr(adcDmaHandle);        
        regionId = CDD_EDMA_lld_getRegionId(adcDmaHandle);
        
        /* Check if, CDD is Init */
        if(ADC_TRUE == Cdd_Dma_GetInitStatus())
        {
            /* Only initialize configured resources for DMA channels. */
			/* Only initialize configured resources for DMA channels. */
            for (dmaCh = 0U; dmaCh < ADC_MAX_DMA_CHANNELS; dmaCh++)
            {
                Adc_DmaConfigObj[dmaCh].adclastChannelId = 0xFFU;
				Adc_DmaConfigObj[dmaCh].dmaChannelId = ADC_INVALID_INDEX;
                for (dmaChIdx = 0U; dmaChIdx < drvObj->maxGroup; dmaChIdx++)
                {    
                    groupObj = &drvObj->groupObj[dmaChIdx];
					/* Check if Group is DMA Access. */
                    if(ADC_GROUP_DMA_ACCESS == groupObj->groupCfg.groupDataAccessMode)
                    {
                                              
                        for (dmaadcChIdx = 0U; 
                             dmaadcChIdx < groupObj->groupCfg.numChannels; 
                             dmaadcChIdx++)
                        {
                            if(dmaCh == groupObj->groupCfg.channelConfig[dmaadcChIdx].adcDMAChannel)
                            {							
								
								CDD_EDMA_lld_freeChannelRegion(baseAddr, 
								regionId, CDD_EDMA_CHANNEL_TYPE_DMA, 
				                Adc_DmaConfigObj[dmaCh].dmaChannelId, 
				                CDD_EDMA_TRIG_MODE_EVENT, Adc_DmaConfigObj[dmaCh].adctccAlloc,0);
                                CDD_EDMA_lld_unregisterIntr
								(adcDmaHandle, &Adc_DmaConfigObj[dmaCh].adcDmaObj);
                                CDD_EDMA_lld_freeDmaChannel
								(adcDmaHandle, &Adc_DmaConfigObj[dmaCh].adcDmaChPtr);                                
                                CDD_EDMA_lld_freeTcc
								(adcDmaHandle, &Adc_DmaConfigObj[dmaCh].adctccAlloc);
                                CDD_EDMA_lld_freeParam
								(adcDmaHandle, &Adc_DmaConfigObj[dmaCh].adcParamPtr);
                            }
                            else
                            {
                                /* No Actions Required. */
                            }
                        }
                    }
                }
            }
        }        
#endif
    
    
    return;
}

/*
 * Design: MCAL-5747,MCAL-5752,MCAL-5837,MCAL-5722,MCAL-5819,MCAL-5751,MCAL-5741,MCAL-5882,MCAL-5714,MCAL-5733,MCAL-5768,MCAL-5791,MCAL-5701
 */
Std_ReturnType Adc_startGroup(Adc_GroupObjType *groupObj)
{
    Std_ReturnType        retVal = (Std_ReturnType) E_OK;
    Adc_HwUnitObjType    *hwUnitObj;
    Adc_GroupObjType     *curGroupObj;
    Adc_GroupPriorityType effPriority;
    boolean hwQue = ADC_FALSE;

    retVal = Adc_startGroupCheckStatus(groupObj);
    if (((Std_ReturnType) E_OK) == retVal)
    {
        hwUnitObj = groupObj->hwUnitObj;

        /* Check if we have any group already going on in hardware */
        if (NULL_PTR == hwUnitObj->curGroupObj)
        {
#if (ADC_PRIORITY_NONE == ADC_PRIORITY_IMPLEMENTATION)
            /* HW is free, schedule this group */
            hwUnitObj->curGroupObj = groupObj;
            groupObj->lastSocAssigned = (Adc_mcalSOCNumber_t)((uint16)(ADC_SOC_NUMBER15));
            if(hwUnitObj->curGroupObj->groupCfg.numChannels == 1U)
            {
                groupObj->socAssigned = (Adc_mcalSOCNumber_t)((uint16)(ADC_SOC_NUMBER15));
            }
            else
            {
                groupObj->socAssigned = (Adc_mcalSOCNumber_t)((uint16)(ADC_SOC_NUMBER15)) -
                                    (hwUnitObj->curGroupObj->groupCfg.numChannels - 1U);
            }
            hwUnitObj->socHwPtr = (uint16)(((uint16)groupObj->socAssigned) - 1U);
            groupObj->groupStatus = ADC_BUSY;
            groupObj->groupInterruptSrc = ADC_INT_NUMBER1;
            Adc_scheduleGroup(groupObj);
#endif

#if ((ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION) || \
                                                   (ADC_PRIORITY_HW == ADC_PRIORITY_IMPLEMENTATION))
            /* HW is free, schedule this group */
            hwUnitObj->curGroupObj = groupObj;
            groupObj->lastSocAssigned = (Adc_mcalSOCNumber_t)((uint16)(ADC_SOC_NUMBER15));
            if(hwUnitObj->curGroupObj->groupCfg.numChannels == 1U)
            {
                groupObj->socAssigned = (Adc_mcalSOCNumber_t)((uint16)(ADC_SOC_NUMBER15));
            }
            else
            {
                groupObj->socAssigned = (Adc_mcalSOCNumber_t)((uint16)(ADC_SOC_NUMBER15)) -
                                    (hwUnitObj->curGroupObj->groupCfg.numChannels - 1U);
            }
            hwUnitObj->socHwPtr = (uint16)(((uint16)groupObj->socAssigned) - 1U);
            groupObj->groupStatus = ADC_BUSY;
            groupObj->groupInterruptSrc = ADC_INT_NUMBER1;
            hwUnitObj->numHwGroupsQue = (uint16)(ADC_INT_NUMBER1);
            Adc_scheduleGroup(groupObj);
#endif
        }
        else
        {
#if (ADC_ENABLE_QUEUING == STD_OFF)
            /* Queuing is disabled - return error */
            retVal = (Std_ReturnType) E_NOT_OK;
            Adc_reportDetRuntimeError(ADC_SID_START_GROUP_CONVERSION, ADC_E_BUSY);
#endif
            /* Check if, error */
            if (((Std_ReturnType) E_OK) == retVal)
            {
                /* Use priority only when enabled */
                effPriority = groupObj->groupCfg.groupPriority;
#if ((ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION) || \
                                                   (ADC_PRIORITY_HW == ADC_PRIORITY_IMPLEMENTATION))
                /* Check if this group's priority is more than current group's
                 * priority */
                curGroupObj = hwUnitObj->curGroupObj;
                if ((effPriority > curGroupObj->groupCfg.groupPriority) ||
                    (curGroupObj->groupCfg.triggSrc == ADC_TRIGG_SRC_SW))
                {
                    groupObj->lastSocAssigned = (Adc_mcalSOCNumber_t)((uint16)(hwUnitObj->socHwPtr));
                        
                    /* Assign the Queue number. */
                    if(((uint16)((uint16)(ADC_SOC_NUMBER15) - 
                            hwUnitObj->socHwPtr + groupObj->groupCfg.numChannels))
                              <= ((uint16)(ADC_SOC_NUMBER15)))
                    {
                        if(groupObj->groupCfg.numChannels == 1U)
                        {
                           groupObj->socAssigned = (Adc_mcalSOCNumber_t)(((uint16)(hwUnitObj->socHwPtr)));
                        }
                        else
                        {
                           groupObj->socAssigned = (Adc_mcalSOCNumber_t)(((uint16)(hwUnitObj->socHwPtr)) -
                                                                 (groupObj->groupCfg.numChannels - 1U));
                        }
                        
                        hwUnitObj->socHwPtr = (uint16)(((uint16)groupObj->socAssigned) - 1U);
                        
                        /* Check if Interrupt is Valid. */
                        if(hwUnitObj->numHwGroupsQue >= ADC_INVALID_HW_INT)
                        {
                            retVal = (Std_ReturnType) E_NOT_OK;
                        }
                        else
                        {
                            hwUnitObj->numHwGroupsQue++;
                            retVal = E_OK;
                            groupObj->groupInterruptSrc = (Adc_mcalIntNumber_t)hwUnitObj->numHwGroupsQue;
                        }          
                    }
                    else
                    {
                        retVal = (Std_ReturnType) E_NOT_OK;
                    }       
                
                    if(retVal == E_OK)
                    {
                        /* Schedule the Group to HW Queue. */
                        groupObj->groupStatus = ADC_BUSY;
                        Adc_scheduleGroup(groupObj);
                        hwQue = ADC_TRUE;
                    }
                    else
                    {
                        if(curGroupObj->groupCfg.triggSrc == ADC_TRIGG_SRC_SW)
                        {
                            /* Allow the Group to store in Software Queue. */
                            hwQue = ADC_FALSE;
                        }
                        else
                        {
                            hwQue = ADC_TRUE;
                        }
                    }
                }
#if (ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION)
                else
                {
                    if(curGroupObj->groupCfg.triggSrc == ADC_TRIGG_SRC_SW)
                    {
                        /* Allow the Group to store in Software Queue. */
                        hwQue = ADC_FALSE;
                    }
                    else
                    {
                        hwQue = ADC_TRUE;
                    }
                }
#else
                else
                {
                    /* Queuing is disabled - return error */
                    retVal = (Std_ReturnType) E_NOT_OK;
                    hwQue = ADC_TRUE;
                }
#endif
#else
                /* Use same priority so that request are processed on first
                 * come first serve basis */
                effPriority = 0U;
#endif


#if ((ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION) || \
      ((ADC_PRIORITY_NONE == ADC_PRIORITY_IMPLEMENTATION) && (ADC_ENABLE_QUEUING == STD_ON)))
                /* Check if this group's priority is more than current group's
                 * priority */
                curGroupObj = hwUnitObj->curGroupObj;
                if ((effPriority > curGroupObj->groupCfg.groupPriority) && (hwQue == ADC_FALSE))
                {
                    /*
                     * This is high priority group. Stop the current group,
                     * push back to the group queue and schedule the new group
                     */

                    /* Check if Group is HW or SW triggered group. */
                    if(groupObj->groupCfg.triggSrc == ADC_TRIGG_SRC_SW)
                    {
                        /* Stop current group and push back to hardware queue */
                        Adc_hwStop(hwUnitObj->baseAddr, groupObj);
                        curGroupObj->isPaused = (uint32) ADC_TRUE;
                        curGroupObj->isQueued = (uint32) ADC_TRUE;
                        Adc_utilsLinkNodePri(
                            &hwUnitObj->groupList,
                            &curGroupObj->nodeObj,
                            curGroupObj,
                            curGroupObj->groupCfg.groupPriority,
                            curGroupObj->isPaused);
                    }

                    /* schedule the new group */
                    groupObj->groupStatus = ADC_BUSY;
                    hwUnitObj->curGroupObj = groupObj;
#if (ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION)
                    /* Check if, HW Queue if filled or not. */ 
                    if(((uint16)( ((uint16)(ADC_SOC_NUMBER15)) - hwUnitObj->socHwPtr + groupObj->groupCfg.numChannels))
                         <= ((uint16)(ADC_SOC_NUMBER15)))
                    {
                        /* Assign the Queue number. */
                        groupObj->lastSocAssigned = (Adc_mcalSOCNumber_t)((uint16)(hwUnitObj->socHwPtr));
                        if(groupObj->groupCfg.numChannels == 1U)
                        {
                            groupObj->socAssigned = (Adc_mcalSOCNumber_t)(((uint16)(hwUnitObj->socHwPtr)));
                        }
                        else
                        {
                            groupObj->socAssigned = (Adc_mcalSOCNumber_t)(((uint16)(hwUnitObj->socHwPtr)) -
                                                                 (groupObj->groupCfg.numChannels - 1U));
                        }
                        
                        hwUnitObj->socHwPtr = (uint16)(((uint16)groupObj->socAssigned) - 1U);
                        
                        /* Check if Interrupt is Valid. */
                        if(hwUnitObj->numHwGroupsQue >= ADC_INVALID_HW_INT)
                        {
                            retVal = (Std_ReturnType) E_NOT_OK;
                        }
                        else
                        {
                            hwUnitObj->numHwGroupsQue++;
                            retVal = E_OK;
                            groupObj->groupInterruptSrc = (Adc_mcalIntNumber_t)hwUnitObj->numHwGroupsQue;
                        }
                    }
                    else
                    {
                        retVal = (Std_ReturnType) E_NOT_OK;
                    }
                    
                    /* Check if there is DET Error. */
                    if(retVal == E_OK)
#endif                    
                    {
                        /* Schedule the Group to HW Queue. */
                        Adc_scheduleGroup(groupObj);
                    }
                }
                else
                {
                    /* Check if Group is HW or SW triggered group. */
                    if(groupObj->groupCfg.triggSrc == ADC_TRIGG_SRC_SW)
                    {
                        /* Some high priority group already on-going - queue this to
                         * hardware queue */
                        groupObj->groupStatus = ADC_BUSY;
                        groupObj->isQueued = (uint32) ADC_TRUE;
                        Adc_utilsLinkNodePri(
                            &hwUnitObj->groupList,
                            &groupObj->nodeObj,
                            groupObj,
                            effPriority,
                            ADC_FALSE);
                    }
                }
#endif
            }
        }
    }

    return (retVal);
}

/*
 *Design: MCAL-5763,MCAL-5873,MCAL-5718,MCAL-5746,MCAL-5870,MCAL-5867
 */
void Adc_stopGroup(Adc_GroupObjType *groupObj, boolean SwHw)
{
    Adc_HwUnitObjType *hwUnitObj;

    hwUnitObj = groupObj->hwUnitObj;

    /* Check if on-going conversion is same as this group */
    if (groupObj == hwUnitObj->curGroupObj)
    {
        /* Stop ADC */
        Adc_hwStop(hwUnitObj->baseAddr, groupObj);
        hwUnitObj->curGroupObj = NULL;

        /* Check and schedule any pending groups for this HW */
        Adc_checkAndSchedule(hwUnitObj);
    }
    else
    {
        if(SwHw == ADC_TRUE)
        {
           /* Just remove request from the queue */
           Adc_utilsUnLinkNodePri(&hwUnitObj->groupList, &groupObj->nodeObj);
           groupObj->isQueued = (uint32) ADC_FALSE;
        }
        
    }

    /* Disable group notification */
    groupObj->isNotifyOn = (uint32) ADC_FALSE;
    /* Set status to IDLE */
    groupObj->groupStatus = ADC_IDLE;
    
    /* Reset the SOC pointer. */
    groupObj->socAssigned = (Adc_mcalSOCNumber_t)ADC_INVALID_HW_SOC;
    groupObj->lastSocAssigned = (Adc_mcalSOCNumber_t)ADC_INVALID_HW_SOC;
    
    return;
}

/*
 *Design:
 */
void Adc_groupPolling(Adc_GroupObjType *groupObj, Adc_ValueGroupType *DataPtr)
{
    Adc_HwUnitObjType *hwUnitObj;
    uint32 baseAddr;
    uint32 streamComplete = (uint32) ADC_FALSE;
    uint32 data;
    Adc_ChannelObjType *chObj;
    uint8 socAssigned;
    uint8 channnelPtr;
    uint8 numChnls;
    uint8 numChannels;
    Adc_ValueGroupType *resultbufptrch;

    hwUnitObj = groupObj->hwUnitObj;
    baseAddr = hwUnitObj->baseAddr;
    socAssigned = (uint8)groupObj->socAssigned;

    if(ADC_TRUE == ADC_isBusy(baseAddr))
    {
        groupObj->groupStatus = ADC_BUSY;
    }
    else
    {
        if(groupObj->groupCfg.accessMode == ADC_ACCESS_MODE_SINGLE)
        {
            /* Get Number of channels to be stored  */
            numChnls = groupObj->groupCfg.numChannels;
            channnelPtr = 0;
            resultbufptrch = (Adc_ValueGroupType *)groupObj->resultBufPtr;

            while (numChnls > 0U)
            {
                /* Get the data from Register. */
                data =
                (uint16)ADC_readResult(hwUnitObj->resultBaseAddr, (Adc_mcalSOCNumber_t)socAssigned);
                socAssigned++;

                /* Fill the data in buffer. */
                resultbufptrch[channnelPtr] = data;
                channnelPtr++;
                numChnls--;
            }
            
            /* Conversion Completed */
            streamComplete = ADC_TRUE;
        }
        else
        {
            /* Get Number of Samples to be stored  */
            numChannels = groupObj->groupCfg.numChannels;
            groupObj->curCh = 0;
            
            /* Read all the samples. */
            while (numChannels > 0U)
            {
                data = (uint16)ADC_readResult
                            (hwUnitObj->resultBaseAddr, (Adc_mcalSOCNumber_t)socAssigned);
                socAssigned++;
            
                chObj = &groupObj->chObj[groupObj->curCh];
                *chObj->curResultBufPtr = data;
            
                /* Move to next buffer pointer for the channel */
                chObj->curNumSamples++;
                chObj->curResultBufPtr++;
                if ((chObj->curResultBufPtr) >=
                    (chObj->chResultBufPtr + (groupObj->groupCfg.streamNumSamples)))
                {
                    chObj->curNumSamples   = 0U;
                    chObj->curResultBufPtr = chObj->chResultBufPtr;
                }
            
                /* Move to next channel */
                groupObj->curCh++;
                if (groupObj->curCh >= groupObj->groupCfg.numChannels)
                {
                    /* All channel conversion completed */
                    groupObj->curCh = 0U;
                    /* Increment valid count and limit to stream max samples */
                    groupObj->validSampleCount++;
                    if (groupObj->validSampleCount >
                    ((uint32) groupObj->groupCfg.streamNumSamples))
                    {
                        /* This happens if app fails to read data faster than driver
                         * generates */
                        groupObj->validSampleCount =
                            (uint32) groupObj->groupCfg.streamNumSamples;
                    }
            
                    /* Check if all streams are completed - check last channel pointer.
                     * If it is at start then stream completed */
                    if (chObj->curResultBufPtr == chObj->chResultBufPtr)
                    {
                        streamComplete = (uint32) ADC_TRUE;
                        /* Incase of one-shot/linear stream, stop conversion and
                         * don't update the buffers - break from reading the FIFO and
                         * storing data */
                        if (((uint32) ADC_FALSE) == groupObj->isExplicitStopMode)
                        {
                            break;
                        }
                    }
                }
                numChannels--;
            }
        }


        /* Change the status of Group. */
        if((groupObj->groupCfg.convMode == ADC_CONV_MODE_ONESHOT) ||
           (groupObj->groupCfg.streamBufMode == ADC_STREAM_BUFFER_LINEAR))
        {        
            /* Set group status */
            if (((uint32) ADC_TRUE) == streamComplete)
            {
                /* Conversion over for all channels and all streams */
                groupObj->groupStatus = ADC_STREAM_COMPLETED;
            }
            else
            {
                /* Atleast one set of conversion over for all channels */
                groupObj->groupStatus = ADC_COMPLETED;
                
                /* Trigger the conversion again. */
                /* Configure ADC steps and start conversion */
                Adc_hwConfig(groupObj, hwUnitObj->baseAddr);
                
            }
            
            /* Stop ADC in case of one-shot/linear stream */
            if ((((uint32) ADC_TRUE) == streamComplete) &&
                (((uint32) ADC_FALSE) == groupObj->isExplicitStopMode))
            {
                Adc_hwStop(hwUnitObj->baseAddr, groupObj);
                hwUnitObj->socHwPtr = hwUnitObj->socHwPtr + groupObj->groupCfg.numChannels;
                if(hwUnitObj->numHwGroupsQue > 0U)
                {
                   hwUnitObj->numHwGroupsQue--;
                }
                
                if(ADC_INVALID_HW_SOC <= hwUnitObj->socHwPtr)
                {
                    hwUnitObj->socHwPtr = (uint16)ADC_SOC_NUMBER15;
                }
                else
                {
                    /* No Actions Required. */
                }
                groupObj->socAssigned = (Adc_mcalSOCNumber_t)ADC_INVALID_HW_SOC;
                groupObj->lastSocAssigned = (Adc_mcalSOCNumber_t)ADC_INVALID_HW_SOC;
                
                
                
                #if ((ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION) || \
                     ((ADC_PRIORITY_NONE == ADC_PRIORITY_IMPLEMENTATION) && (ADC_ENABLE_QUEUING == STD_ON)))
                hwUnitObj->curGroupObj = NULL;
                                
                /* Check and schedule any pending groups for this HW */
                Adc_checkAndSchedule(hwUnitObj);
                #endif
            }
        }
        else
        {
           /* Change the status. */
           groupObj->groupStatus = ADC_COMPLETED;
        }
    }

    return;
}

void Adc_IrqTxRx(Adc_HwUnitObjType *hwUnitObj,  Adc_mcalIntNumber_t InterruptNum, uint8 Soc)
{
    uint32 baseAddr, GroupCount;
    Adc_GroupObjType *groupObj;
    boolean allowFlag = ADC_TRUE;
    boolean errorOcuured = ADC_FALSE;
    uint8 adcSoc;
    uint8 adcSocCfg;
	Adc_HwUnitObjType *grpHwUnitObj;
#if (ADC_ENABLE_LIMIT_CHECK == STD_ON)
    uint8 LimitCheck;
#endif
    baseAddr = hwUnitObj->baseAddr;
    groupObj = hwUnitObj->curGroupObj;

    adcSoc = 0;
    adcSoc = (uint8) (Soc);
    
#if ((ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION) || \
                                                  (ADC_PRIORITY_HW == ADC_PRIORITY_IMPLEMENTATION))
    allowFlag = ADC_FALSE;
#if (ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION)
    /* Check if, Group is from SW queue */
    if((hwUnitObj->swGroupSchduled == ADC_FALSE) && (errorOcuured == ADC_FALSE))
#endif
    {
        for(GroupCount = 0U; GroupCount <= Adc_DrvObj.maxGroup; GroupCount++)
        {
            groupObj = &Adc_DrvObj.groupObj[GroupCount];
            adcSocCfg = (uint8)(groupObj->lastSocAssigned);

            /* Check if, SOC is correct. */
            if((adcSoc == adcSocCfg) && (groupObj->groupInterruptSrc == InterruptNum))
            {
                grpHwUnitObj = groupObj->hwUnitObj;
				if(baseAddr == grpHwUnitObj->baseAddr)
				{
				  allowFlag = ADC_TRUE;
                  break;
				}
            }
        }
    }
#if (ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION)
    else
    {
        /* Get the Current Group. */
        hwUnitObj->swGroupSchduled = ADC_FALSE;
        groupObj = hwUnitObj->curGroupObj;
        allowFlag = ADC_TRUE;
    }
#endif
#endif

#if (ADC_ENABLE_LIMIT_CHECK == STD_ON)
    if(ADC_TRUE == groupObj->groupCfg.grouplimitcheck)
    {
        /* Get the status of ADC Limit Values. */
        LimitCheck = (HW_RD_REG16(baseAddr + MCAL_CSL_ADC_ADCEVTSTAT));

        /* Check if, Error limits are occured or not.  */
        if
        (
        (MCAL_CSL_ADC_ADCEVTSTAT_PPB1TRIPHI_MASK == (LimitCheck & MCAL_CSL_ADC_ADCEVTSTAT_PPB1TRIPHI_MASK)) ||
        (MCAL_CSL_ADC_ADCEVTSTAT_PPB1TRIPLO_MASK == (LimitCheck & MCAL_CSL_ADC_ADCEVTSTAT_PPB1TRIPLO_MASK)) ||
        (MCAL_CSL_ADC_ADCEVTSTAT_PPB2TRIPHI_MASK == (LimitCheck & MCAL_CSL_ADC_ADCEVTSTAT_PPB2TRIPHI_MASK)) ||
        (MCAL_CSL_ADC_ADCEVTSTAT_PPB2TRIPLO_MASK == (LimitCheck & MCAL_CSL_ADC_ADCEVTSTAT_PPB2TRIPLO_MASK)) ||
        (MCAL_CSL_ADC_ADCEVTSTAT_PPB3TRIPHI_MASK == (LimitCheck & MCAL_CSL_ADC_ADCEVTSTAT_PPB3TRIPHI_MASK)) ||
        (MCAL_CSL_ADC_ADCEVTSTAT_PPB3TRIPLO_MASK == (LimitCheck & MCAL_CSL_ADC_ADCEVTSTAT_PPB3TRIPLO_MASK)) ||
        (MCAL_CSL_ADC_ADCEVTSTAT_PPB4TRIPHI_MASK == (LimitCheck & MCAL_CSL_ADC_ADCEVTSTAT_PPB4TRIPHI_MASK)) ||
        (MCAL_CSL_ADC_ADCEVTSTAT_PPB4TRIPLO_MASK == (LimitCheck & MCAL_CSL_ADC_ADCEVTSTAT_PPB4TRIPLO_MASK))
        )
        {
            allowFlag = ADC_FALSE;
        }

        /* Clear Status Register. */
        HW_WR_REG16((baseAddr + MCAL_CSL_ADC_ADCEVTCLR), LimitCheck);
    }
    else
    {
        /* No Actions Required. */
    }
#endif

#if (STD_ON == ADC_DMA_MODE)
    allowFlag = ADC_FALSE;
    
    /* Check if, Data access mode is Interrupt mode. */
    if(ADC_GROUP_INTERRUPT_ACCESS == groupObj->groupCfg.groupDataAccessMode)
    {
        /* Allow for processing the Group. */
        allowFlag = ADC_TRUE;
    }
    else
    {
        /* disable required interrupts */
        ADC_disableInterrupt(baseAddr, groupObj->groupInterruptSrc);
    }
#endif

    /* Clear interrupts immediately before handling events. This should avoid
     * missing the same interrupts again while handling the events */
    ADC_clearInterruptStatus(baseAddr, InterruptNum);

    /*
     * Process the group
     */
    if ((NULL_PTR != groupObj) && (allowFlag == ADC_TRUE))
    {
        /* Process the Interrupt */
        Adc_procIsr(hwUnitObj, groupObj);
    }
    
    /* Check if overflow has occurred */
    if(TRUE == ADC_getInterruptOverflowStatus(baseAddr, InterruptNum))
    {
        ADC_clearInterruptOverflowStatus(baseAddr, InterruptNum);
        ADC_clearInterruptStatus(baseAddr, InterruptNum);
    }
    
    return;
}

#if (STD_ON == ADC_DMA_MODE)
static void Adc_IrqDmaTxRx()
{
    uint32 baseAddr, irqStatus, GroupCount, hwunitid;
    boolean allowFlag = ADC_TRUE;
    boolean convComplete = ADC_FALSE;
    boolean streamComplete = ADC_FALSE;
    uint8 adcSoc = 0;    
    Adc_HwUnitObjType *hwUnitObj =  NULL_PTR;
    Adc_GroupObjType *groupObj = NULL_PTR;
    uint8 adcSocNumber = 0;
    uint32 intRegAddr = 0;
    uint8 groupCount = 0;
    uint8 Channel = 0;
    
    for(groupCount = 0; groupCount <= ADC_MAX_DMA_CHANNELS; groupCount++)
    {
            Channel = Adc_DmaConfigObj[groupCount].adclastChannelId;
            
            /* Channel count checking. */
            if(Channel != 0xFF)  
            {
              hwUnitObj = Adc_DmaConfigObj[groupCount].hwUnitObj;
              groupObj = Adc_DmaConfigObj[groupCount].groupObj;
              allowFlag = ADC_TRUE;
              Adc_DmaConfigObj[groupCount].adclastChannelId = 0xFFU;
              break;
            }
    }
    
    if ((NULL_PTR != groupObj) && (allowFlag == ADC_TRUE))
    {
        intRegAddr = hwUnitObj->baseAddr + MCAL_CSL_ADC_ADCINTSEL3N4 +
                            ((uint32)ADC_INT_NUMBER4 >> 1) * MCAL_ADC_ADCINTSELxNy_STEP;
        
        adcSocNumber = (uint8)
              ((HW_RD_REG16(intRegAddr) & MCAL_CSL_ADC_ADCINTSEL3N4_INT3SEL_MASK) >> 8U);
        
        baseAddr = hwUnitObj->baseAddr;   
        adcSoc = (uint8) (groupObj->lastSocAssigned);
        
        ADC_clearInterruptStatus(baseAddr, groupObj->groupInterruptSrc);
        
        /* Check if overflow has occurred */
        if(ADC_TRUE == 
          ADC_getInterruptOverflowStatus(baseAddr, groupObj->groupInterruptSrc))
        {
            ADC_clearInterruptOverflowStatus
                                        (baseAddr, groupObj->groupInterruptSrc);
        }
        
        /*
         * Process the group
         */
        if ((NULL_PTR != groupObj) && (allowFlag == ADC_TRUE))
        {
            /* Stop the conversion, if required. */
            convComplete = (uint32) ADC_TRUE;                        
            streamComplete = (uint32) ADC_TRUE;
            
            /* Set Group Status and Call group end notification */
            Adc_setGroupStatusPostIsr
                            (hwUnitObj, groupObj, convComplete, streamComplete);            
        }
        
        
        /* Stop the ADCs by removing the trigger for SOC0 */
        ADC_setInterruptSOCTrigger(baseAddr, adcSoc, ADC_INT_SOC_TRIGGER_NONE);
    }
    else
    {
        /* No Actions Required. */
    }
    return;
}
#endif

/**
 *  \brief Returns the the HW object for the corresponding HW unit ID
 */
Adc_HwUnitObjType *Adc_getHwUnitObj(Adc_HWUnitType HWUnit)
{
    Adc_HwUnitObjType *hwObj  = (Adc_HwUnitObjType *) NULL_PTR;
    Adc_DriverObjType *drvObj = &Adc_DrvObj;

    /* Get the HW unit.  */
    hwObj = &drvObj->hwUnitObj[HWUnit];
    
    if (hwObj == NULL_PTR)
    {
#if (STD_ON == ADC_DEV_ERROR_DETECT)
        Adc_reportDetError(ADC_SID_INIT, ADC_E_PARAM_CONFIG);
#endif  /* #if (STD_ON == ADC_DEV_ERROR_DETECT) */
    }
    
    return (hwObj);
}

/*
 * Design: MCAL-5854,MCAL-5808,MCAL-5892
 */
void Adc_resetDrvObj(Adc_DriverObjType *drvObj)
{
    uint8          hwUnitIdx, grpIdx;
    uint32         chIdx;
    Adc_UtilsNode *nodeObj;

    for (grpIdx = 0U; grpIdx < ADC_MAX_GROUP; grpIdx++)
    {
        drvObj->groupObj[grpIdx].groupStatus        = ADC_IDLE;
        drvObj->groupObj[grpIdx].isExplicitStopMode = (uint32) ADC_FALSE;
        drvObj->groupObj[grpIdx].isNotifyOn         = (uint32) ADC_FALSE;
        drvObj->groupObj[grpIdx].isPaused           = (uint32) ADC_FALSE;
        drvObj->groupObj[grpIdx].isQueued           = (uint32) ADC_FALSE;
        drvObj->groupObj[grpIdx].validSampleCount   = 0U;
        drvObj->groupObj[grpIdx].curCh        = 0U;
        drvObj->groupObj[grpIdx].resultBufPtr =
            NULL;
        for (chIdx = 0U; chIdx < ADC_NUM_CHANNEL; chIdx++)
        {
            drvObj->groupObj[grpIdx].chObj[chIdx].chResultBufPtr =
                NULL;
            drvObj->groupObj[grpIdx].chObj[chIdx].curResultBufPtr =
                NULL;
            drvObj->groupObj[grpIdx].chObj[chIdx].curNumSamples = 0U;
        }

        nodeObj = &drvObj->groupObj[grpIdx].nodeObj;
        Adc_utilsInitNodeObject(nodeObj);
    }
    for (hwUnitIdx = 0U; hwUnitIdx < ADC_MAX_HW_UNIT; hwUnitIdx++)
    {
        drvObj->hwUnitObj[hwUnitIdx].curGroupObj =
            NULL;
    }
    drvObj->maxGroup  = 0U;
    drvObj->maxHwUnit = 0U;

    return;
}

void Adc_copyConfig(Adc_DriverObjType *drvObj, const Adc_ConfigType *cfgPtr)
{
    uint8 hwUnitIdx, grpIdx;
    const Adc_HwUnitConfigType *hwCfg;
    Adc_GroupObjType           *groupObj;
    Adc_HwUnitObjType           *hwObj;
    const Adc_GroupConfigType  *groupCfg;    
    uint32         chIdx;
#if (STD_ON == ADC_DEV_ERROR_DETECT)
    const Adc_ChannelConfigType *chCfg;
    Std_ReturnType retVal;
#endif  /* #if (STD_ON == ADC_DEV_ERROR_DETECT) */

    drvObj->maxGroup  = cfgPtr->maxGroup;
    drvObj->maxHwUnit = cfgPtr->maxHwUnit;
    for (hwUnitIdx = 0U; hwUnitIdx < cfgPtr->maxHwUnit; hwUnitIdx++)
    {
        hwCfg = &cfgPtr->hwUnitCfg[hwUnitIdx];

        /* Copy configuration */
        (void) memcpy(
            &drvObj->hwUnitObj[hwUnitIdx].hwUnitCfg,
            hwCfg,
            sizeof (Adc_HwUnitConfigType));
        
        /* Reset the SOC pointer to last SOC. */
        hwObj = &drvObj->hwUnitObj[hwUnitIdx];
        hwObj->socHwPtr = (uint16)(ADC_SOC_NUMBER15);
    }

    for (grpIdx = 0U; grpIdx < cfgPtr->maxGroup; grpIdx++)
    {
        groupCfg = &cfgPtr->groupCfg[grpIdx];
        groupObj = &drvObj->groupObj[grpIdx];
#if (STD_ON == ADC_DEV_ERROR_DETECT)
        retVal = Adc_checkGroupParameters(groupCfg, cfgPtr);
        if (((Std_ReturnType) E_OK) == retVal)
#endif      /* #if (STD_ON == ADC_DEV_ERROR_DETECT) */
        {                      
            groupObj->groupCfg.groupId = groupCfg->groupId;
            groupObj->groupCfg.groupPriority = groupCfg->groupPriority;
            groupObj->groupCfg.hwUnitId = groupCfg->hwUnitId;
            groupObj->groupCfg.Adc_GroupEndNotification = groupCfg->Adc_GroupEndNotification;
            groupObj->groupCfg.streamNumSamples = groupCfg->streamNumSamples;
            groupObj->groupCfg.resolution = groupCfg->resolution;
            groupObj->groupCfg.convMode = groupCfg->convMode;
            groupObj->groupCfg.triggSrc = groupCfg->triggSrc;
            groupObj->groupCfg.accessMode = groupCfg->accessMode;
            groupObj->groupCfg.streamBufMode = groupCfg->streamBufMode;
            groupObj->groupCfg.hwTrigSignal = groupCfg->hwTrigSignal;
            groupObj->groupCfg.hwTrigTimer = groupCfg->hwTrigTimer;
            groupObj->groupCfg.groupReplacement = groupCfg->groupReplacement;
            groupObj->groupCfg.groupChannelMask = groupCfg->groupChannelMask;
#if (STD_ON == ADC_ENABLE_LIMIT_CHECK)
            groupObj->groupCfg.grouplimitcheck = groupCfg->grouplimitcheck;
#endif            
            groupObj->groupCfg.groupDataAccessMode = groupCfg->groupDataAccessMode;            
            
            groupObj->groupCfg.numChannels = groupCfg->numChannels;
            
            for (chIdx = 0U; chIdx < groupCfg->numChannels; chIdx++)
            {                
                groupObj->groupCfg.channelConfig[chIdx]
                = groupCfg->channelConfig[chIdx];
            }
            
            /* Store HW object pointer for easy reference */
            groupObj->hwUnitObj = &Adc_DrvObj.hwUnitObj[groupCfg->hwUnitId];

            /* Reset the SOC to invalid. */
            groupObj->socAssigned = ADC_INVALID_HW_SOC;
            groupObj->lastSocAssigned = ADC_INVALID_HW_SOC;
            groupObj->groupInterruptSrc = ADC_INVALID_HW_INT;

            /* All modes except single shot and linear streaming needs to be
             * stopped explicitly */
            groupObj->isExplicitStopMode = (uint32) ADC_TRUE;
            if ((ADC_CONV_MODE_ONESHOT == groupCfg->convMode) ||
                ((ADC_ACCESS_MODE_STREAMING == groupCfg->accessMode) &&
                 (ADC_STREAM_BUFFER_LINEAR == groupCfg->streamBufMode)))
            {
                groupObj->isExplicitStopMode = (uint32) ADC_FALSE;
            }

#if (STD_ON == ADC_DEV_ERROR_DETECT)
            /* Check channel configuration */
            for (chIdx = 0U; chIdx < groupCfg->numChannels; chIdx++)
            {
                chCfg = &groupCfg->channelConfig[chIdx];
                Adc_checkChannelParams(chCfg);
            }
#endif      /* #if (STD_ON == ADC_DEV_ERROR_DETECT) */
        }
#if (STD_ON == ADC_DEV_ERROR_DETECT)
        else
        {
            /* No Actions Required. */
        }
#endif      /* #if (STD_ON == ADC_DEV_ERROR_DETECT) */
    }

    return;
}

uint32 Adc_checkHwUnitObj(const Adc_ConfigType *cfgPtr)
{
    uint8 hwUnitIdx = 0U;
    const Adc_HwUnitConfigType *hwCfg;
    uint32 found = (uint32) ADC_FALSE;

    for (hwUnitIdx = 0U; hwUnitIdx < cfgPtr->maxHwUnit; hwUnitIdx++)
    {
        hwCfg = &cfgPtr->hwUnitCfg[hwUnitIdx];
        if (hwCfg->hwUnitId < ADC_MAX_HW_UNIT)
        {
            found = (uint32) ADC_TRUE;
            break;      /* Match found */
        }
    }
    
    return (found);
}

/*
 * Design:  MCAL-5759,MCAL-5713,MCAL-5720
 */
static Std_ReturnType Adc_startGroupCheckStatus(
    const Adc_GroupObjType *groupObj)
{
    Std_ReturnType retVal = (Std_ReturnType) E_OK;

    if (((uint32) ADC_TRUE) == groupObj->isExplicitStopMode)
    {
        /* For explicit stop mode, group in idle status can be started */
        if (ADC_IDLE != groupObj->groupStatus)
        {
            /* Group is already in progress - return error */
            retVal = (Std_ReturnType) E_NOT_OK;
        }
    }
    else
    {
        /* For implicit stop mode, group in idle or stream completed status
         * can be started */
        if ((ADC_IDLE != groupObj->groupStatus) &&
            (ADC_STREAM_COMPLETED != groupObj->groupStatus))
        {
            /* Group is already in progress - return error */
            retVal = (Std_ReturnType) E_NOT_OK;
        }
    }

    return retVal;
}

/* check channel parameters */
#if (STD_ON == ADC_DEV_ERROR_DETECT)
static void Adc_checkChannelParams(const Adc_ChannelConfigType *chCfg)
{
    if (chCfg->hwChannelId > ADC_MAX_HW_CHANNEL_ID)
    {
        Adc_reportDetError(ADC_SID_INIT, ADC_E_PARAM_CONFIG);
    }
    return;
}

#endif

#if (STD_ON == ADC_DEV_ERROR_DETECT)
static Std_ReturnType Adc_checkGroupParameters(
    const Adc_GroupConfigType *groupCfg, const Adc_ConfigType *cfgPtr)
{
    Std_ReturnType retVal = (Std_ReturnType) E_OK;
    /* ID is used as index, can't exceed array size */
    if (groupCfg->groupId >= cfgPtr->maxGroup)
    {
        retVal = (Std_ReturnType) E_NOT_OK;
        Adc_reportDetError(ADC_SID_INIT, ADC_E_PARAM_CONFIG);
    }
    /* check group parameters */
    else if ((groupCfg->streamNumSamples == 0U) ||
             ((ADC_ACCESS_MODE_SINGLE == groupCfg->accessMode) &&
              (groupCfg->streamNumSamples != 1U)) ||
             ((ADC_CONV_MODE_ONESHOT == groupCfg->convMode) &&
              (groupCfg->accessMode != ADC_ACCESS_MODE_SINGLE)))
    {
        retVal = (Std_ReturnType) E_NOT_OK;
        Adc_reportDetError(ADC_SID_INIT, ADC_E_PARAM_CONFIG);

        /* Dynamic MC/DC coverage for the above 'if' statement is not covered 100%
         * because MC/DC cannot be achieved for C5. The only combination to achieve
             * 100% coverage is when both C2 and C5 are either ADC_TRUE or ADC_FALSE and this
             * is not possible as the condition C2 and C5 cannot be ADC_TRUE or ADC_FALSE
             * at the same time.
         */
    }
    else if (NULL_PTR == Adc_getHwUnitObj(groupCfg->hwUnitId))
    {
        /* DET already reported by Adc_getHwUnitObj() */
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else
    {
        retVal = Adc_checkGroupCfgRangeParameters(groupCfg);
    }

    return retVal;
}

#endif  /* #if (STD_ON == ADC_DEV_ERROR_DETECT) */

#if (STD_ON == ADC_DEV_ERROR_DETECT)
static Std_ReturnType Adc_checkGroupCfgRangeParameters(
    const Adc_GroupConfigType *groupCfg)
{
    Std_ReturnType retVal = (Std_ReturnType) E_OK;

    if ((0U == groupCfg->numChannels) ||
        (groupCfg->numChannels > ADC_NUM_CHANNEL) ||
        (groupCfg->resolution != ADC_DEF_CHANNEL_RESOLUTION))
    {
        retVal = (Std_ReturnType) E_NOT_OK;
        Adc_reportDetError(ADC_SID_INIT, ADC_E_PARAM_CONFIG);
    }

    return retVal;
}

#endif  /* #if (STD_ON == ADC_DEV_ERROR_DETECT) */

/*
 * Design: MCAL-5738,MCAL-5802,MCAL-5831
 */
static void Adc_scheduleGroup(Adc_GroupObjType *groupObj)
{
    uint32 chIdx;
    Adc_HwUnitObjType     *hwUnitObj;
    Adc_ValueGroupType    *tempResultBufPtr;
#if (STD_ON == ADC_GROUP_LOG)
    Adc_GroupLogEntryType *logEntry;
    uint32 timeStamp;
    StatusType status;
#endif  /* #if (STD_ON == ADC_GROUP_LOG) */

    hwUnitObj = groupObj->hwUnitObj;

    if (((uint32) ADC_TRUE) == groupObj->isPaused)
    {
        /* For both type we always start with first channel and discard any
         * intermediate results */
        if ((ADC_GROUP_REPL_ABORT_RESTART ==
             groupObj->groupCfg.groupReplacement) ||
            (ADC_GROUP_REPL_SUSPEND_RESUME ==
             groupObj->groupCfg.groupReplacement))
        {
            Adc_resetGroupObjOnResume(groupObj);
        }
    }
    else
    {
        /* Set the states for normal start */
        groupObj->validSampleCount = 0U;
        groupObj->curCh  = 0U;
        tempResultBufPtr = (Adc_ValueGroupType *)groupObj->resultBufPtr;
        for (chIdx = 0U; chIdx < groupObj->groupCfg.numChannels; chIdx++)
        {
            groupObj->chObj[chIdx].chResultBufPtr  = tempResultBufPtr;
            groupObj->chObj[chIdx].curResultBufPtr = tempResultBufPtr;
            groupObj->chObj[chIdx].curNumSamples   = 0U;

            /* Move to next channel buffer */
            tempResultBufPtr += groupObj->groupCfg.streamNumSamples;
        }
    }
#if (STD_ON == ADC_GROUP_LOG)
    /* Log the group */
    logEntry =
        &Adc_GroupLogObj.logEntry[Adc_GroupLogObj.logIndex];

    status = GetCounterValue(ADC_OS_COUNTER_ID, &timeStamp);
    if (((StatusType) E_OK) == status)
    {
        logEntry->timeStamp = timeStamp;
        logEntry->groupId   = groupObj->groupCfg.groupId;
        logEntry->hwUnitId      = hwUnitObj->hwUnitCfg.hwUnitId;
        logEntry->isReScheduled = groupObj->isPaused;

        /* Move to next entry */
        Adc_GroupLogObj.logIndex++;
        Adc_GroupLogObj.totalLog++;
        if (Adc_GroupLogObj.logIndex >= ADC_MAX_GROUP_LOG)
        {
            Adc_GroupLogObj.logIndex = 0U;
        }
    }
#endif  /* #if (STD_ON == ADC_GROUP_LOG) */

    groupObj->isPaused = (uint32) ADC_FALSE;    /* Reset flag after log */

    /* Configure ADC steps and start conversion */
    Adc_hwConfig(groupObj, hwUnitObj->baseAddr);
    Adc_hwStart(hwUnitObj->baseAddr);

    return;
}

static void Adc_hwErrRecovery(Adc_GroupObjType *groupObj, uint32 baseAddr)
{
#if (STD_ON == ADC_ERR_LOG)
    Adc_FifoErrLogEntryType *logEntry;
    uint32 timeStamp;
    StatusType status;
#endif  /* #if (STD_ON == ADC_ERR_LOG) */

    /* Stop ADC - this will clear the FIFO */
    Adc_hwStop(baseAddr, groupObj);

    /* if no active group, clear only the HW event, no SW update needed */
    if (NULL_PTR != groupObj)
    {
        
#if (STD_ON == ADC_ERR_LOG)
    /* Log the group */
    logEntry = &Adc_ErrLogObj.logEntry[Adc_ErrLogObj.logIndex];
    status = GetCounterValue(ADC_OS_COUNTER_ID, &timeStamp);
    if (((StatusType) E_OK) == status)
    {
        logEntry->timeStamp = timeStamp;
        logEntry->groupId   = groupObj->groupCfg.groupId;
        logEntry->hwUnitId  = groupObj->hwUnitObj->hwUnitCfg.hwUnitId;

        /* Move to next entry */
        Adc_ErrLogObj.logIndex++;
        Adc_ErrLogObj.totalLog++;
        if (Adc_ErrLogObj.logIndex >= ADC_MAX_ERR_LOG)
        {
            Adc_ErrLogObj.logIndex = 0U;
        }
    }
#endif  /* #if (STD_ON == ADC_ERR_LOG) */

        Adc_resetGroupObjOnResume(groupObj);

        /* Configure ADC steps and start conversion */
        Adc_hwConfig(groupObj, baseAddr);
    }

    return;
}

static void Adc_checkAndSchedule(Adc_HwUnitObjType *hwUnitObj)
{
    Adc_UtilsNode    *headNodeObj;
    Adc_GroupObjType *nextGroupObj;
    Std_ReturnType retVal = E_OK;

    /* Check if any new group is pending */
    headNodeObj = Adc_utilsGetHeadNode(&hwUnitObj->groupList);
    if (NULL_PTR != headNodeObj)
    {
        /* Some group pending - schedule it */
        nextGroupObj = (Adc_GroupObjType *) headNodeObj->data;
        Adc_utilsUnLinkNodePri((&(hwUnitObj->groupList)), headNodeObj);
        nextGroupObj->isQueued = (uint32) ADC_FALSE;
        
        hwUnitObj->swGroupSchduled = ADC_TRUE;
        
        if(((uint16)(((uint16)(ADC_SOC_NUMBER15)) - hwUnitObj->socHwPtr + nextGroupObj->groupCfg.numChannels))
                             <= ((uint16)(ADC_SOC_NUMBER15)))
        {
            /* Assign the Queue number. */
            nextGroupObj->lastSocAssigned = (Adc_mcalSOCNumber_t)((uint16)(hwUnitObj->socHwPtr));
            if(nextGroupObj->groupCfg.numChannels == 1U)
            {
                nextGroupObj->socAssigned = (Adc_mcalSOCNumber_t)(((uint16)(hwUnitObj->socHwPtr)));
            }
            else
            {
                nextGroupObj->socAssigned = (Adc_mcalSOCNumber_t)(((uint16)(hwUnitObj->socHwPtr)) -
                                                     (nextGroupObj->groupCfg.numChannels - 1U));
            }
            
            hwUnitObj->socHwPtr = (uint16)(((uint16)nextGroupObj->socAssigned) - 1U);
            
            /* Check if Interrupt is Valid. */
            if(hwUnitObj->numHwGroupsQue >= ADC_INVALID_HW_INT)
            {
                retVal = (Std_ReturnType) E_NOT_OK;
                
                /* Check if Group is HW or SW triggered group. */
                if(nextGroupObj->groupCfg.triggSrc == ADC_TRIGG_SRC_SW)
                {
                    Adc_reportDetRuntimeError(ADC_SID_START_GROUP_CONVERSION, ADC_E_BUSY);
                }
            }
            else
            {
                hwUnitObj->numHwGroupsQue++;
                nextGroupObj->groupInterruptSrc = (Adc_mcalIntNumber_t)hwUnitObj->numHwGroupsQue;
                hwUnitObj->curGroupObj = nextGroupObj;
            } 
        }
        else
        {
            retVal = (Std_ReturnType) E_NOT_OK;
        }        
        

        if(retVal == E_OK)
        {
           /* Schedule Group. */
           Adc_scheduleGroup(nextGroupObj);
        }
        else
        {
            /* Some high priority group already on-going - queue this to
                     * hardware queue */
            nextGroupObj->groupStatus = ADC_BUSY;
            nextGroupObj->isQueued = (uint32) ADC_TRUE;
            Adc_utilsLinkNodePri(
                        &hwUnitObj->groupList,
                        &nextGroupObj->nodeObj,
                        nextGroupObj,
                        nextGroupObj->groupCfg.groupPriority,
                        ADC_FALSE);
        }
    }
    return;
}

/*
 * Design: MCAL-5888,MCAL-5761,MCAL-5857,MCAL-5860,MCAL-5719,MCAL-5753,MCAL-3234,MCAL-5762
 */
static void Adc_procIsr(Adc_HwUnitObjType *hwUnitObj,
                        Adc_GroupObjType  *groupObj)
{
    uint32 convComplete   = (uint32) ADC_FALSE;
    uint32 streamComplete = (uint32) ADC_FALSE;
    uint16 data, numChannels, channnelPtr;
    Adc_ChannelObjType *chObj;
    uint32 socAssigned;
    Adc_ValueGroupType *resultBufPtrCh;
    
#if (ADC_ALIGN_LEFT == ADC_RESULT_ALIGNMENT)
    uint16 data2, shiftResol;
#endif  /* #if (ADC_ALIGN_LEFT == ADC_RESULT_ALIGNMENT) */

    socAssigned = (uint32)groupObj->socAssigned;

    /* Check the mode of Group. */
    if(groupObj->groupCfg.accessMode == ADC_ACCESS_MODE_SINGLE)
    {
            /* Get Number of channels to be stored  */
            numChannels = groupObj->groupCfg.numChannels;
            channnelPtr = 0;
            resultBufPtrCh = (Adc_ValueGroupType *)groupObj->resultBufPtr;

            while (numChannels > 0U)
            {
                /* Get the Data. */
                data = (uint16)ADC_readResult
                        (hwUnitObj->resultBaseAddr, (Adc_mcalSOCNumber_t)socAssigned);
                        
#if (ADC_ALIGN_LEFT == ADC_RESULT_ALIGNMENT)
                shiftResol = (ADC_MAX_CHANNEL_VALUE_TYPE - groupObj->groupCfg.resolution);
                data2 = (uint16)((uint32)data << (uint32)shiftResol);
                data = data2;
#endif  /* #if (ADC_ALIGN_LEFT == ADC_RESULT_ALIGNMENT) */

                /* Get the Increment the Data ptr. */
                socAssigned++;

                /* Fill data to result buffer. */
                resultBufPtrCh[channnelPtr] = data;

                /* Point to next channel. */
                channnelPtr++;
                numChannels--;
            }

            /* Stop the conversion, if required. */
            convComplete = (uint32) ADC_TRUE;
            if (((uint32) ADC_FALSE) == groupObj->isExplicitStopMode)
            {
                streamComplete = (uint32) ADC_TRUE;
            }
    }
    else
    {
        /* Get Number of Samples to be stored  */
        numChannels = groupObj->groupCfg.numChannels;
        groupObj->curCh = 0;
        
        /* Read all the samples. */
        while (numChannels > 0U)
        {
            data = (uint16)ADC_readResult
                        (hwUnitObj->resultBaseAddr, (Adc_mcalSOCNumber_t)socAssigned);
            socAssigned++;

            chObj = &groupObj->chObj[groupObj->curCh];
            *chObj->curResultBufPtr = data;

            /* Move to next buffer pointer for the channel */
            chObj->curNumSamples++;
            chObj->curResultBufPtr++;
            if ((chObj->curResultBufPtr) >=
                (chObj->chResultBufPtr + (groupObj->groupCfg.streamNumSamples)))
            {
                chObj->curNumSamples   = 0U;
                chObj->curResultBufPtr = chObj->chResultBufPtr;
            }

            /* Move to next channel */
            groupObj->curCh++;
            if (groupObj->curCh >= groupObj->groupCfg.numChannels)
            {
                /* All channel conversion completed */
                groupObj->curCh = 0U;
                /* Increment valid count and limit to stream max samples */
                groupObj->validSampleCount++;
                if (groupObj->validSampleCount >
                ((uint32) groupObj->groupCfg.streamNumSamples))
                {
                    /* This happens if app fails to read data faster than driver
                     * generates */
                    groupObj->validSampleCount =
                        (uint32) groupObj->groupCfg.streamNumSamples;
                }

                /* Check if all streams are completed - check last channel pointer.
                 * If it is at start then stream completed */
                convComplete = (uint32) ADC_TRUE;
                if (chObj->curResultBufPtr == chObj->chResultBufPtr)
                {
                    streamComplete = (uint32) ADC_TRUE;
                    /* Incase of one-shot/linear stream, stop conversion and
                     * don't update the buffers - break from reading the FIFO and
                     * storing data */
                    if (((uint32) ADC_FALSE) == groupObj->isExplicitStopMode)
                    {
                        break;
                    }
                }
            }
            numChannels--;
        }
    }


    /* Set Group Status and Call group end notification */
    Adc_setGroupStatusPostIsr(hwUnitObj, groupObj, convComplete, streamComplete);

    return;
}

static void Adc_setGroupStatusPostIsr(Adc_HwUnitObjType *hwUnitObj,
                                      Adc_GroupObjType  *groupObj,
                                      uint32             convComplete,
                                      uint32             streamComplete)
{
    Adc_GroupEndNotifyType groupEndNotification = NULL;
    if (((uint32) ADC_TRUE) == convComplete)
    {
        /* Set group status */
        if (((uint32) ADC_TRUE) == streamComplete)
        {
            /* Conversion over for all channels and all streams */
            groupObj->groupStatus = ADC_STREAM_COMPLETED;
        }
        else
        {
            /* Atleast one set of conversion over for all channels */
            groupObj->groupStatus = ADC_COMPLETED;
        }

        /* Call group end notification */
        if ((((uint32) ADC_TRUE) == groupObj->isNotifyOn) &&
            ((Adc_GroupEndNotifyType)NULL_PTR !=
                groupObj->groupCfg.Adc_GroupEndNotification))
        {
            groupEndNotification =
                groupObj->groupCfg.Adc_GroupEndNotification;
            groupEndNotification();

            /* Dynamic MC/DC coverage for the above 'if' statement is not covered 100%
             * because MC/DC cannot be achieved for C2. The only combination to achieve
                 * 100% coverage is when C1 is ADC_TRUE and C2 is ADC_FALSE and this is not possible
                   * because if isNotifyOn is ADC_TRUE then groupCfg.Adc_GroupEndNotification is
                 * assigned to a callback fxn pointer.
             */
        }

        /* Stop ADC in case of one-shot/linear stream */
        if ((((uint32) ADC_TRUE) == streamComplete) &&
            (((uint32) ADC_FALSE) == groupObj->isExplicitStopMode))
        {
            Adc_hwStop(hwUnitObj->baseAddr, groupObj);
            hwUnitObj->socHwPtr = hwUnitObj->socHwPtr + groupObj->groupCfg.numChannels;
            if(hwUnitObj->numHwGroupsQue > 0U)
            {
               hwUnitObj->numHwGroupsQue--;
            }
            
            if(ADC_INVALID_HW_SOC <= hwUnitObj->socHwPtr)
            {
                hwUnitObj->socHwPtr = (uint16)ADC_SOC_NUMBER15;
            }
            else
            {
                /* No Actions Required. */
            }
            groupObj->socAssigned = (Adc_mcalSOCNumber_t)ADC_INVALID_HW_SOC;
            groupObj->lastSocAssigned = (Adc_mcalSOCNumber_t)ADC_INVALID_HW_SOC;
            hwUnitObj->curGroupObj = NULL;
            
#if ((ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION) || \
      ((ADC_PRIORITY_NONE == ADC_PRIORITY_IMPLEMENTATION) && (ADC_ENABLE_QUEUING == STD_ON)))            
            /* Check and schedule any pending groups for this HW */
            Adc_checkAndSchedule(hwUnitObj);
#endif
        }
    }
    return;
}

static void Adc_resetGroupObjOnResume(Adc_GroupObjType *groupObj)
{
    uint32 chIdx;

    /* When we re-start, the ADC will start conversion from 0th chan, so
     * un-do previous conversion if half-way through */
    for (chIdx = 0U; chIdx < groupObj->curCh; chIdx++)
    {
        if (groupObj->chObj[chIdx].curNumSamples > 0U)
        {
            groupObj->chObj[chIdx].curNumSamples--;
            groupObj->chObj[chIdx].curResultBufPtr--;

            /* Handle wrap around */
            if ((groupObj->chObj[chIdx].curResultBufPtr) <
                (groupObj->chObj[chIdx].chResultBufPtr))
            {
                /* We were at start pointer, so move to last pointer */
                groupObj->chObj[chIdx].curResultBufPtr =
                    groupObj->chObj[chIdx].chResultBufPtr +
                    (groupObj->groupCfg.streamNumSamples - 1U);

                /* Dynamic Code coverage for this statement is not covered
                 * because curResultBufPtr cannot be less than chResultBufPtr.
                         * This 'if statement' is added as an additional safety mechanism.
                 */
            }
        }
    }

    /* Set the Current channel to 0 */
    groupObj->curCh = 0U;

    return;
}

static void Adc_hwConfig(Adc_GroupObjType *groupObj, uint32 baseAddr)
{
    uint32 chnlId, socInc;
    uint32 threshold;
    uint16 groupMask = 0U;
    Adc_mcalSOCNumber_t adcSoc, adcLastSoc;
    const Adc_GroupConfigType   *groupCfg;
    const Adc_ChannelConfigType *chCfg;

#if (STD_ON == ADC_DMA_MODE)    
    uint32 dmaDataAddr;
    uint8 dmaDataBufferPtr = 0;
    Adc_HwUnitObjType  *adchwUnitObj;
    CddDma_HandleType adcEdmaHandle;
#endif
 
#if (STD_ON == ADC_DMA_MODE) 
    adchwUnitObj = groupObj->hwUnitObj;
    adcEdmaHandle = Cdd_Dma_GetHandle();
#endif

    groupCfg = &groupObj->groupCfg;
    adcSoc = groupObj->socAssigned;
    adcLastSoc = groupObj->lastSocAssigned;
    socInc = (uint32)adcSoc;

#if ((ADC_PRIORITY_NONE == ADC_PRIORITY_IMPLEMENTATION) && (ADC_ENABLE_QUEUING == STD_ON))
    /* Set the Priority to SOC.*/
    ADC_setSOCPriority(baseAddr, ADC_PRI_ALL_ROUND_ROBIN);
#endif

#if ((ADC_PRIORITY_HW == ADC_PRIORITY_IMPLEMENTATION) || (ADC_PRIORITY_HW_SW == ADC_PRIORITY_IMPLEMENTATION))
    /* Set the Priority to SOC.*/
    ADC_setSOCPriority(baseAddr, (Adc_mcalPriorityMode_t)(adcLastSoc+1U));
#endif

    /* Number of channels to every conversion of all channels */
    threshold = groupCfg->numChannels - 1U;


    for (chnlId = 0U; chnlId <= threshold; chnlId++)
    {
        chCfg = &groupCfg->channelConfig[chnlId];

#if (ADC_ENABLE_LIMIT_CHECK == STD_ON)
        if(ADC_TRUE == groupObj->groupCfg.grouplimitcheck)
        {
            /* Clear the Status Register. */
            HW_WR_REG16((baseAddr + MCAL_CSL_ADC_ADCEVTCLR), 0xFFFF);

            /* Setup the PPB . */
            ADC_setupPPB(baseAddr, (Adc_mcalPPBNumber_t)(chnlId+1U), (Adc_mcalSOCNumber_t)adcSoc);

            /* Set the Channel Limits. */
            ADC_setPPBTripLimits( baseAddr,
                                 (Adc_mcalPPBNumber_t)(chnlId+1U),
                                 chCfg->highRange,
                                 chCfg->lowRange
                                 );
        }
        
#endif
        ADC_setupSOC( baseAddr,
                      (Adc_mcalSOCNumber_t)adcSoc,
                      (Adc_HwTriggerTimerType)groupCfg->hwTrigTimer,
                      (Adc_mcalChannel_t)chCfg->hwChannelId,
                      (Adc_HwTriggerTimerType)chCfg->samplewindow
                    );
                    
        ADC_setInterruptSOCTrigger( baseAddr,
                                    (Adc_mcalSOCNumber_t)adcSoc,
                                    ADC_INT_SOC_TRIGGER_NONE);
                                    
        /* Check the access mode. */
        switch(groupObj->groupCfg.groupDataAccessMode)
        {
            case ADC_GROUP_INTERRUPT_ACCESS:        
            {
                ADC_setInterruptSOCTrigger( baseAddr,(Adc_mcalSOCNumber_t)adcSoc,
                                                          ADC_INT_SOC_TRIGGER_NONE);
                                                          
                /* Check if, Group conversion mode. */
                if(groupCfg->convMode == ADC_CONV_MODE_ONESHOT)
                {
                    ADC_setInterruptSOCTrigger( baseAddr,
                                                     (Adc_mcalSOCNumber_t)adcSoc,
                                                      ADC_INT_SOC_TRIGGER_NONE);
                }
                else
                {
                    ADC_setInterruptSOCTrigger( baseAddr,
                                                   (Adc_mcalSOCNumber_t)adcSoc,
                                                   ADC_INT_SOC_TRIGGER_ADCINT1);
                }
            }
            break;
            
            case ADC_GROUP_POLLING_ACCESS:
            {
                /* Check if, Group conversion mode. */
                if(groupCfg->convMode == ADC_CONV_MODE_ONESHOT)
                {
                    ADC_setInterruptSOCTrigger( baseAddr,
                                                (Adc_mcalSOCNumber_t)adcSoc,
                                                ADC_INT_SOC_TRIGGER_NONE
                                               );
                }
                else
                {
                    ADC_setInterruptSOCTrigger( baseAddr,
                                                (Adc_mcalSOCNumber_t)adcSoc,
                                                ADC_INT_SOC_TRIGGER_ADCINT1
                                               );
                }                
            }
            break;
            
            case ADC_GROUP_DMA_ACCESS:
            {
#if (STD_ON == ADC_DMA_MODE)                
                /* Get the Address. */
                dmaDataAddr = ADC_readResultbaseaddr
                              (adchwUnitObj->resultBaseAddr, adcSoc);            
                
                /* Add the Physical Channel ID and its data register.  */
                Adc_DmaConfigObj[chCfg->adcDMAChannel].adcDataRegAddr = 
                                                                    dmaDataAddr;
                Adc_DmaConfigObj[chCfg->adcDMAChannel].adcDataRegAddr = 
                                                                    dmaDataAddr;                                                                    
                if(chnlId == threshold)
                {                    
                    Adc_DmaConfigObj[chCfg->adcDMAChannel].adclastChannelId 
                                                                    = chnlId;                                        
                }
                else
                {
                    Adc_DmaConfigObj[chCfg->adcDMAChannel].adclastChannelId 
                                                                    = 0xFF;   
                }
                
                /* Add the Physical Channel ID and its data register.  */
                Adc_DmaConfigObj[chCfg->adcDMAChannel].groupObj = groupObj; 
                Adc_DmaConfigObj[chCfg->adcDMAChannel].hwUnitObj = adchwUnitObj; 
                                
                /* Check if, CDD is Initialized. */
                if(ADC_TRUE == Cdd_Dma_GetInitStatus())
                {
                    /* Configure ADC DMA Channel for each ADC channel.  */
                    AdcDma_ModuleChannelConfigure(
                    (const uint16 *)(groupObj->resultBufPtr+dmaDataBufferPtr), 
                    ADC_DMA_HANDLE_ID,
                    groupObj->groupCfg.streamNumSamples,
                    Adc_DmaConfigObj[chCfg->adcDMAChannel].dmaChannelId, 
                    dmaDataAddr, 
                    &Adc_DmaConfigObj[chCfg->adcDMAChannel].adctccAlloc, 
                    &Adc_DmaConfigObj[chCfg->adcDMAChannel].adcDmaChPtr, 
                    &Adc_DmaConfigObj[chCfg->adcDMAChannel].adcParamPtr, 
                    ADC_TRUE);
                    
                    if(groupObj->groupCfg.accessMode == ADC_ACCESS_MODE_SINGLE)
                    {
                        /* Buffer Pointer to be incremented to next channel. */
                        dmaDataBufferPtr++;
                    }
                    else
                    {
                        /* Buffer Pointer to be incremented to next channel 
                           as per samples. */
                        dmaDataBufferPtr = dmaDataBufferPtr + 
                                            groupObj->groupCfg.streamNumSamples;
                    }               

                    /* Register the Adc module for all channels */
                    Adc_DmaConfigObj[chCfg->adcDMAChannel].adcDmaObj.tccNum 
                          = Adc_DmaConfigObj[chCfg->adcDMAChannel].adctccAlloc;
                    Adc_DmaConfigObj[chCfg->adcDMAChannel].adcDmaObj.cbFxn  = 
                                                                    &Adc_IrqDmaTxRx;
                    Adc_DmaConfigObj[chCfg->adcDMAChannel].dmaSemData = threshold;
                    Adc_DmaConfigObj[chCfg->adcDMAChannel].adcDmaObj.appData  = 
                    (void *)&Adc_DmaConfigObj[chCfg->adcDMAChannel].dmaSemData;
                    CDD_EDMA_lld_registerIntr
                                (adcEdmaHandle, &Adc_DmaConfigObj[chCfg->adcDMAChannel].adcDmaObj);
                }
#endif                            
            }
            break;
        }
        groupMask |=  (1U  << socInc);
        socInc++;
        adcSoc = (Adc_mcalSOCNumber_t)socInc;
    }
    
    /* Check the access mode. */
    switch(groupObj->groupCfg.groupDataAccessMode)
    {
        case ADC_GROUP_INTERRUPT_ACCESS:        
        {
            /* Clear all previous interrupt status */
            ADC_clearInterruptStatus(baseAddr, groupObj->groupInterruptSrc);
                    
            /* Check if, Group conversion mode. */
            if(ADC_CONV_MODE_ONESHOT != groupCfg->convMode)
            {
                ADC_enableContinuousMode(baseAddr, groupObj->groupInterruptSrc);
            }
        
            /* Enable required interrupts */
            ADC_enableInterrupt(baseAddr, groupObj->groupInterruptSrc);
        
            /* Set the ADC Group Last Channel with  Interrupt trigger.*/
            ADC_setInterruptSource( baseAddr, groupObj->groupInterruptSrc, adcLastSoc);
            
            if(groupCfg->triggSrc == ADC_TRIGG_SRC_SW)
            {
               /* Start ADC */
               ADC_forceMultipleSOC(baseAddr, groupMask);
            }
        }
        break;
        
        case ADC_GROUP_POLLING_ACCESS:
        {
            /* Check if, Group conversion mode. */
            if(ADC_CONV_MODE_ONESHOT != groupCfg->convMode)
            {
                ADC_enableContinuousMode(baseAddr, groupObj->groupInterruptSrc);
            }
            
            /* disable required interrupts */
            ADC_disableInterrupt(baseAddr, groupObj->groupInterruptSrc);
            
            /* Set the ADC Group Last Channel with  Interrupt trigger.*/
            ADC_setInterruptSource( baseAddr, groupObj->groupInterruptSrc, adcLastSoc);
            
            if(groupCfg->triggSrc == ADC_TRIGG_SRC_SW)
            {
               /* Start ADC */
               ADC_forceMultipleSOC(baseAddr, groupMask);
            }
        }
        break;
        
        case ADC_GROUP_DMA_ACCESS:
        {    
#if (STD_ON == ADC_DMA_MODE)
            /* Clear all previous interrupt status */
            ADC_clearInterruptStatus(baseAddr, groupObj->groupInterruptSrc);
            
            /* Set the ADC Group Last Channel with  Interrupt trigger.*/
            ADC_setInterruptSource
                               (baseAddr, groupObj->groupInterruptSrc, adcLastSoc);
                    
            /* Check if, Group conversion mode. */
            if(ADC_CONV_MODE_ONESHOT != groupCfg->convMode)
            {
                ADC_setInterruptSOCTrigger(baseAddr,
                                            (Adc_mcalSOCNumber_t)adcLastSoc,
                                             ADC_INT_SOC_TRIGGER_ADCINT1);
                                          
                ADC_enableContinuousMode(baseAddr, groupObj->groupInterruptSrc);    
            }    
        
            /* Enable required interrupts */
            ADC_enableInterrupt(baseAddr, groupObj->groupInterruptSrc);            
            
            if(groupCfg->triggSrc == ADC_TRIGG_SRC_SW)
            {
               /* Start ADC */
               ADC_forceMultipleSOC(baseAddr, groupMask);
            }
#endif
        }
        break;
    }    

    return;
}

static void Adc_hwStart(uint32 baseAddr)
{
   /* Enable Converter. */
   return;
}

static void Adc_hwStop(uint32 baseAddr, const Adc_GroupObjType *groupObj)
{

    const Adc_GroupConfigType   *groupCfg;
    #if (STD_ON == ADC_DMA_MODE)
    const Adc_ChannelConfigType *chCfg;
    #endif
    uint8 ChnlId;
    #if (STD_ON == ADC_DMA_MODE)  
    CddDma_HandleType adcDmaHandle;
    uint32 adcDmabaseAddr, regionId;
    #endif

    groupCfg = &groupObj->groupCfg; 
    
    #if (STD_ON == ADC_DMA_MODE)  
    adcDmaHandle = Cdd_Dma_GetHandle();
    adcDmabaseAddr = CDD_EDMA_lld_getBaseAddr(adcDmaHandle);
    regionId = CDD_EDMA_lld_getRegionId(adcDmaHandle);
    #endif

    /* Disable  */
    ADC_disableInterrupt(baseAddr, groupObj->groupInterruptSrc);
    
    /* Stop ADC in continous mode. */
    ADC_disableContinuousMode(baseAddr, groupObj->groupInterruptSrc);
    
    
    

    for (ChnlId = 0U; ChnlId < groupCfg->numChannels; ChnlId++)
    {
        /* Check if, Group conversion mode. */
        if(groupCfg->convMode != ADC_CONV_MODE_ONESHOT)
        {
            ADC_setInterruptSOCTrigger( baseAddr,
                                        (Adc_mcalSOCNumber_t)ChnlId,
                                        ADC_INT_SOC_TRIGGER_NONE
                                       );
        }
        
#if (STD_ON == ADC_DMA_MODE)
        chCfg = &groupCfg->channelConfig[ChnlId];    
        if(groupCfg->groupDataAccessMode == ADC_GROUP_DMA_ACCESS)
        {            
            if(ADC_TRUE == Cdd_Dma_GetInitStatus())
            {
                /* DeRegister the DMA Channel */
                CDD_EDMA_lld_unregisterIntr
                   (adcDmaHandle, &Adc_DmaConfigObj[chCfg->adcDMAChannel].adcDmaObj);
                CDD_EDMA_lld_disableTransferRegion
                (adcDmabaseAddr, regionId, Adc_DmaConfigObj[chCfg->adcDMAChannel].adclastChannelId, CDD_EDMA_TRIG_MODE_EVENT);
            }
        }    
#endif
    }

    return;
}


static void Adc_hwSetDefReg(uint32 baseAddr)
{
    uint32 index;

    /* Power down analog Front End */
    ADC_disableConverter(baseAddr);

    /* Reset the Registers. */
    for (index = 0U; index <= 4; index++)
    {
        ADC_disableContinuousMode(baseAddr, ( Adc_mcalIntNumber_t)index );
        ADC_clearInterruptStatus(baseAddr, ( Adc_mcalIntNumber_t)index );
        ADC_clearInterruptOverflowStatus(baseAddr, ( Adc_mcalIntNumber_t)index );
    }
    
    

    return;
}

#if (STD_ON == ADC_DMA_MODE)
/* Configures the DMA Channel to particular external module register. */
boolean AdcDma_ModuleChannelConfigure(
        const uint16 *table, 
        uint8 inst,
        uint16 table_size,
        uint32 dma_ch,
        uint32 srceaddr, 
        uint32 *tccAlloc,
        uint32 *dmchstr,
        uint32 *paramptr,
        boolean param_enable)
{

    uint32            baseAddr, regionId;
    CDD_EDMACCEDMACCPaRAMEntry  edmaParam;
    uint32            dmaCh, tcc, param;
    boolean           testStatus = TRUE;
    CddDma_HandleType CddDma_gEdmaHandle; 
    
    CddDma_gEdmaHandle =  Cdd_Dma_GetHandle();    
                
    baseAddr = CDD_EDMA_lld_getBaseAddr(CddDma_gEdmaHandle);

    regionId = CDD_EDMA_lld_getRegionId(CddDma_gEdmaHandle);
    
    if(param_enable == ADC_TRUE)
    {

        dmaCh = dma_ch;
        testStatus = CDD_EDMA_lld_allocDmaChannel(CddDma_gEdmaHandle, &dmaCh);
        *dmchstr = dmaCh;

        tcc = CDD_EDMA_RESOURCE_ALLOC_ANY;
        testStatus = CDD_EDMA_lld_allocTcc(CddDma_gEdmaHandle, &tcc);
        *tccAlloc = tcc;

        param = CDD_EDMA_RESOURCE_ALLOC_ANY;
        testStatus = CDD_EDMA_lld_allocParam(CddDma_gEdmaHandle, &param);
        *paramptr = param;

        /* Request channel */
        testStatus = CDD_EDMA_lld_configureChannelRegion(baseAddr, regionId, 
                                                     CDD_EDMA_CHANNEL_TYPE_DMA,
                                                     dmaCh, tcc, param, 
                                                     0U);
    }
                                                 
    if(param_enable == ADC_TRUE)
    {        
        /* Program Param Set */
        param = *paramptr;
        tcc = *tccAlloc;
        dmaCh = *dmchstr;
        CDD_EDMA_lld_ccPaRAMEntry_init(&edmaParam);
        edmaParam.srcAddr       = (uint32) (Cdd_Dma_SOC_virtToPhy((void *)srceaddr));
        edmaParam.destAddr      = (uint32) Cdd_Dma_SOC_virtToPhy((void *)table);
        edmaParam.aCnt          = (uint16) (2);
        edmaParam.bCnt          = (uint16) table_size;
        edmaParam.cCnt          = (uint16) 1;
        edmaParam.bCntReload    = 0;
        edmaParam.srcBIdx       = (sint16) 0;
        edmaParam.destBIdx      = (sint16) 2;
        edmaParam.srcCIdx       = (sint16) 0;
        edmaParam.destCIdx      = (sint16) 2;
        edmaParam.linkAddr      = 0xFFFFU;
        edmaParam.opt           = (CDD_EDMA_OPT_TCINTEN_MASK |
                                  ((((uint32)tcc) << CDD_EDMA_OPT_TCC_SHIFT) & CDD_EDMA_OPT_TCC_MASK));
        
        CDD_EDMA_lld_setPaRAM(baseAddr, param, &edmaParam);
        
        testStatus = CDD_EDMA_lld_enableTransferRegion(baseAddr, regionId, dmaCh, CDD_EDMA_TRIG_MODE_EVENT);
    }
    
    return testStatus;
}

/* Frees the DMA channel. */
static void AdcDma_FreeModuleChannelConfigured(uint32 dma_ch)
{
    uint32 baseAddr, regionId, dmaCh;
    CddDma_HandleType adcEdmaHandle;

    adcEdmaHandle = Cdd_Dma_GetHandle();
    
    baseAddr = CDD_EDMA_lld_getBaseAddr(adcEdmaHandle);

    regionId = CDD_EDMA_lld_getRegionId(adcEdmaHandle);

    dmaCh = dma_ch;
    
    CDD_EDMA_lld_disableTransferRegion(baseAddr, regionId, dmaCh, CDD_EDMA_TRIG_MODE_EVENT);
}
#endif /* #if (STD_ON == ADC_DMA_MODE) */

#if (STD_ON == ADC_DEV_ERROR_DETECT)
void Adc_reportDetError(uint8 apiId, uint8 errorId)
{
    (void) Det_ReportError(ADC_MODULE_ID, ADC_INSTANCE_ID, apiId, errorId);
    return;
}

#endif  /* #if (STD_ON == ADC_DEV_ERROR_DETECT) */

void Adc_reportDetRuntimeError(uint8 apiId, uint8 errorId)
{
    (void) Det_ReportRuntimeError(ADC_MODULE_ID, ADC_INSTANCE_ID, apiId, errorId);
    return;
}

#define ADC_STOP_SEC_CODE
#include "Adc_MemMap.h"
