/* ======================================================================
*   Copyright (C) 2022-2023 Texas Instruments Incorporated
*
*   All rights reserved. Property of Texas Instruments Incorporated.
*   Restricted rights to use, duplicate or disclose this code are
*   granted through contract.
*
*   The program may not be used without the written permission
*   of Texas Instruments Incorporated or against the terms and conditions
*   stipulated in the agreement under which this program has been
*   supplied.
* ==================================================================== */

/**
 *  \file     Can.h
 *
 *  \brief    This file contains interface header for CAN MCAL driver
 *
 */

/**
 *
 * \defgroup MCAL_CAN_API CAN Driver API
 *
 * The Can Driver MCAL provides low level access to onbaord MCAN peripheral
 * present on AM263x Device.<br>
 * The Can Driver implements a standardized interface specified in the
 * AUTOSAR_SWS_CANDriver document.<br>
 * The Can Driver is part of the Communication Stack in the AUTOSAR Basic
 * Software (BSW)<br>
 * The CAN Hardware Unit cntrolled by CAN Driver combines one or several CAN
 * controllers, <br>
 * which may be located on-chip or as external standalone devices of the same
 * type, with common or separate Hardware Objects<br>
 * @sa MCAL_CAN_CFG
 *  @{
 */
/*
 * Below are the global requirements which are met by this CAN
 * driver which can't be mapped to a particular source ID
 */
/* Design : CAN_DesignId_001,CAN_DesignId_002,CAN_DesignId_003,
 * CAN_DesignId_007 */

/*Requirements :SWS_Can_00079 , SWS_Can_00077, SWS_Can_00284, SWS_Can_00385,
 * SWS_Can_00386, SWS_Can_00427, SWS_Can_00401, SWS_Can_00402, SWS_Can_00403,
 * SWS_Can_00060, SWS_Can_00497, SWS_Can_00462, SWS_Can_00202, SWS_Can_00110,
 * SWS_Can_00234, SWS_Can_00022, SWS_Can_00024, SWS_Can_00220, SWS_Can_00221
 * SWS_Can_00487  MCAL-200, MCAL-207, MCAL-213, MCAL-237, MCAL-244, MCAL-225 */
#ifndef CAN_H_
#define CAN_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

/* Requirements :SWS_Can_00034, SWS_Can_00435, SWS_Can_00436, SWS_Can_00388,
 * SWS_Can_00035, SWS_Can_00390, SWS_Can_00391, SWS_Can_00397, SWS_Can_00429,
 * SWS_Can_00417, SWS_Can_00416, SWS_Can_00415, SWS_Can_00413, SWS_Can_00222,
 * SWS_Can_00439
 */
#include "Std_Types.h"
#include "ComStack_Types.h"
#include "Can_GeneralTypes.h"
#include "Can_Cfg.h"
#include "Can_Irq.h"
#include "Os.h"

#ifdef __cplusplus
extern "C"
{
#endif

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/**
 *  \name Can Driver Module SW Version Info
 *
 *  Defines for CAN Driver version used for compatibility checks
 *  @{
 */
/** \brief  Driver Implementation Major Version.Used for version compatibiltiy
 *check */
#define CAN_SW_MAJOR_VERSION             (8U)
/** \brief  Driver Implementation Minor Version.Used for version compatibiltiy
 *check */
#define CAN_SW_MINOR_VERSION             (6U)
/** \brief  Driver Implementation Patch Version */
/* @} */
#define CAN_SW_PATCH_VERSION             (2U)

/**
 *  \name Can Driver Module AUTOSAR Version Info
 *
 *  Defines for CAN Driver AUTOSAR version used for compatibility checks
 *  @{
 */
/** \brief  AUTOSAR Major version specification implemented by CanDriver*/
#define CAN_AR_RELEASE_MAJOR_VERSION     (4U)
/** \brief  AUTOSAR Minor  version specification implemented by CanDriver */
#define CAN_AR_RELEASE_MINOR_VERSION     (3U)
/** \brief  AUTOSAR Patch version specification implemented by CanDriver*/
/* @} */
#define CAN_AR_RELEASE_REVISION_VERSION  (1U)

/* Global Macros */
/**
 *  \name Can Driver ID Info
 *  @{
 */
/** \brief  Texas Instruments Vendor ID*/
#define CAN_VENDOR_ID   (44U)
/** \brief  Can Driver Module ID       */
#define CAN_MODULE_ID   (80U)
/** \brief CAN Driver Instance ID */
#define CAN_INSTANCE_ID (0U)
/* @} */

/**
 *  \name Can Driver Service Id
 *
  *  The Service Id is one of the argument to Det_ReportError function and is
 *  used to identify the source of the error
 *  @{
 */
/** \brief  Can_Init() */
#define CAN_INIT_ID                  (0x00U)
/** \brief  Can_MainFunction_Write()  */
#define CAN_MAINFCT_WRITE_ID         (0x01U)
/** \brief  Can_SetControllerMode()  */
#define CAN_SETCTR_ID                (0x03U)
/** \brief  Can_DisableControllerInterrupts() */
#define CAN_DIINT_ID                 (0x04U)
/** \brief  Can_EnableControllerInterrupts() */
#define CAN_ENINT_ID                 (0x05U)
/** \brief  Can_Write()  */
#define CAN_WRITE_ID                 (0x06U)
/** \brief  Can_GetVersionInfo() */
#define CAN_VERSION_ID               (0x07U)
/** \brief  Can_MainFunction_Read() */
#define CAN_MAINFCT_READ_ID          (0x08U)
/** \brief  Can_MainFunction_BusOff()  */
#define CAN_MAINFCT_BO_ID            (0x09U)
/** \brief  Can_MainFunction_Wakeup() */
#define CAN_MAINFCT_WU_ID            (0x0AU)
/** \brief  Can_CheckWakeup()         */
#define CAN_CKWAKEUP_ID              (0x0BU)
/** \brief  Can_MainFunction_Mode()   */
#define CAN_MAINFCT_MODE_ID          (0x0CU)
/** \brief  Can_SetBaudrate()         */
#define CAN_SETBAUDRATE_ID           (0x0FU)
/** \brief  Can_DeInit()              */
#define CAN_DEINIT_ID                (0x10U)
/** \brief  Can_GetControllerErrorState() */
#define CAN_GETCTRERRST_ID           (0x11U)
/** \brief  Can_GetControllerMode() */
#define CAN_GETCTRMODE_ID            (0x12U)
/** \brief  Can_RxProcess_Interrupt() */
#define CAN_RXPROCESS_ID_INTERRUPT   (0x21U)
/** \brief  Can_RxProcess_Polling()   */
#define CAN_RXPROCESS_ID_POLLING     (0x22U)
/** \brief  Can_TestLoopBackModeEnable() */
#define CAN_LOOPBACK_ENABLE_ID       (0x14U)
/** \brief  Can_TestLoopBackModeDisable() */
#define CAN_LOOPBACK_DISABLE_ID      (0x15U)


/* @} */

/**
 *  \name Can Driver CallOut function id
 *
 *  The Service Id is one of the argument to Det_ReportError functiona and is
 *                     used to
 *  identify the source of the error
 *  @{
 */
/** \brief  CanIf_RxIndication callback */
#define CAN_RXINDI_ID               (0x10U)
/** \brief  CanIf_TxConfirmation callback */
#define CAN_TXCNF_ID                (0x11U)
/** \brief  Controller Wakeup indication callback */
#define CAN_CTRWAKEUP_ID            (0x12U)
/** \brief  Controller Off indication callback */
/* @} */
#define CAN_CTRBUSOFF_ID            (0x13U)

/**
 *  \name Can Error Codes
 *
 *  Error codes returned by Can functions
 *  @{
 */
#ifndef CAN_E_PARAM_POINTER
#define CAN_E_PARAM_POINTER          (0x01U)
/** \brief  ERROR:NULL_PTR passed as parameter */
#endif
#ifndef CAN_E_PARAM_HANDLE
#define CAN_E_PARAM_HANDLE           (0x02U)
/** \brief  ERROR:Invalid Controller Handle */
#endif
#ifndef CAN_E_PARAM_DATA_LENGTH
#define CAN_E_PARAM_DATA_LENGTH              (0x03U)
/** \brief  ERROR:Invalid length of msg */
#endif
#ifndef CAN_E_PARAM_CONTROLLER
#define CAN_E_PARAM_CONTROLLER       (0x04U)
/** \brief  ERROR:Invalid controller id */
#endif
#ifndef CAN_E_UNINIT
#define CAN_E_UNINIT                 (0x05U)
/** \brief  ERROR:Eror due to API invoked without  performing Can_Init() */
#endif
#ifndef CAN_E_TRANSITION
#define CAN_E_TRANSITION             (0x06U)
/** \brief  ERROR:Eror during Controller state  transition */
#endif
#ifndef CAN_E_PARAM_BAUDRATE
#define CAN_E_PARAM_BAUDRATE         (0x07U)
/** \brief  ERROR:Invalid Baudrate configuration */
#endif
#ifndef CAN_E_ICOM_CONFIG_INVALID
#define CAN_E_ICOM_CONFIG_INVALID            (0x08U)
/** \brief  ERROR:Invalid config ID*/
#endif
#ifndef CAN_E_INIT_FAILED
#define CAN_E_INIT_FAILED            (0x09U)
/** \brief  ERROR:Timeout occured */
#endif
#ifndef CAN_E_DATALOST
#define CAN_E_DATALOST               (0x01U)
/** \brief  ERROR:Data lost/dropped . Runtime Error*/
#endif
/* @} */

/**
 *  \name CAN Config Ids
 *
 *  The Config Ids used for different
 *  CAN Configuration builds
 *  @{
 */
/** \brief Config 0 (All configurations ON) */
#define  CAN_CFG_ID_0     (0x01U)
/** \brief Config 1 (All configurations OFF).
 *         CAN_CFG_ID_1 is used only for compile check */
#define  CAN_CFG_ID_1     (0x02U)
/** \brief Config 2 (All configurations ON except DET OFF) */
#define  CAN_CFG_ID_2     (0x04U)
/** \brief Config 3 (Configurator auto generated file) */
#define  CAN_CFG_ID_3     (0x08U)
/** \brief Config 4 (Configurator auto generated file MCAN
  *                  with pre-compile variant ON) */
#define  CAN_CFG_ID_4     (0x10U)
/** \brief Config 5 (All configurations ON but used only for testing
  *                   WakeUp by external HW event and Performance tests) */
#define  CAN_CFG_ID_5     (0x20U)

/** \brief Config 6 (Used for MCAN Testing) */
#define  CAN_CFG_ID_6     (0x40U)

/* @} */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/**
 *  \addtogroup MCAL_CAN_CFG Can Configuration
 *  @{
 */

/** \brief Typedef for error notification function pointer */
typedef P2FUNC(void, CAN_APPL_CODE, Can_ErrNotifyType)(uint32 canErrorType);

/**
 * \brief Can Mailbox direction enum
 */
typedef enum
{
    CAN_CONTROLLER_INSTANCE_MCAN0 = 0x0U,
    /**< MCAN0 Controller Instance */
    CAN_CONTROLLER_INSTANCE_MCAN1 = 0x1U,
    /**< MCAN1 Controller Instance */
    CAN_CONTROLLER_INSTANCE_MCAN2 = 0x2U,
    /**< MCAN2 Controller Instance */
    CAN_CONTROLLER_INSTANCE_MCAN3 = 0x3U
    /**< MCAN3 Controller Instance */
} Can_ControllerInstance;

/**
 * \brief Can Mailbox direction enum
 */
typedef enum
{
    CAN_MAILBOX_DIRECTION_RX = 0x0U,
    /**< Mailbox is for receive */
    CAN_MAILBOX_DIRECTION_TX = 0x1U
    /**< Mailbox is for transmit */
} Can_MailBoxDirectionType;

/**
 * \brief Can Handle Type enum
 */
typedef enum
{
    CAN_FULL = 0x0U,
    /**< For only one L-PDU (identifier) is handled by the hardware object */								   
    CAN_BASIC = 0x1U
    /**< For several L-PDUs are handled by the hardware object */
} Can_HandleType;

/**
 * \brief Can Tx/Rx processing enum
 */
typedef enum
{
    CAN_TX_RX_PROCESSING_INTERRUPT = 0x0U,
    /**< Interrupt Mode of operation */
    CAN_TX_RX_PROCESSING_MIXED = 0x1U,
    /**< Mixed Mode of operation */
    CAN_TX_RX_PROCESSING_POLLING = 0x2U
    /**< Polling Mode of operation */
} Can_TxRxProcessingType;

/**
 * \brief Can Controller Error Types
 */
typedef enum
{
    CAN_ECC_ERROR_TYPE_PARITY = 0x0U,
    /**< Parity Error */
    CAN_ECC_ERROR_TYPE_SEC = 0x1U,
    /**< Single Bit Error Detected and corrected */
    CAN_ECC_ERROR_TYPE_DED = 0x2U
    /**< Double Bit Error Detected */
} Can_ECCErrorType;



/**
 * \brief Can Controller State enum type
 */
typedef uint8 CanControllerState_Type;

/**
 * \brief Can Interrupt Mask type
 */
typedef uint32 Can_InterruptMask_Type;

/**
 * \brief Structure defining the filter mask to be used
 */
typedef struct Can_MaskStruct
{
    uint32 MaskValue;
/**< Filter mask value. */
}Can_MaskType;

/**
 * \brief Structure defining the HW filter to be used
 */
typedef struct Can_HwFilterStruct
{
    Can_IdType               IDValue;
    /**< ID value extended will have MSB set */
    uint32                   Mask;
    /**< Mask for the RX filtering */
}Can_HwFilterType;

/**
 * \brief Structure defining the CAN FD data phase baud rate configuration
 */
typedef struct Can_FdBaudConfigStruct
{
    uint16 Baud;
    /**< In Kbps */
    uint8  PropSeg;
    /**< Prop Segement value */
    uint8  Pseg1;
    /**< Phase Segment 1 */
    uint8  Pseg2;
    /**< Phase Segment 2 */
    uint8  Sjw;
    /**< Sync jump width */
    uint16 TimingValues;
    /**< Sum of all timing parameters * */
    uint16 BrpValue;
    /**< Controller BRP value for Baud **/
    uint16 TrcvCompDelay;
    /**< Specifies the Transceiver Delay Compensation Offset in ns **/
    boolean BrsSwitch;
    /**< Specifies if the bit rate switching shall be used for transmissions.**/
}Can_FdBaudConfigType;

/**
 * \brief Structure defining the CAN baud rate configuration
 */
typedef struct Can_BaudConfigStruct
{
    uint16 Baud;
    /**< In Kbps */
    uint8  PropSeg;
    /**< Prop Segement value */
    uint8  Pseg1;
    /**< Phase Segment 1 */
    uint8  Pseg2;
    /**< Phase Segment 2 */
    uint8  Sjw;
    /**< Sync jump width */
    uint16  TimingValues;
    /**< Sum of all timing parameters */
    uint16 BrpValue;
    /**< Controller BRP value for Baud */
    Can_FdBaudConfigType  BaudFdRateConfig;
    /**< List of available Data phase Baud rates for CAN FD */
}Can_BaudConfigType;

/**
 * \brief Can Controller Configuration definition
 */
typedef struct Can_ControllerStruct
{
    Can_BaudConfigType    *DefaultBaud;
	/**< pointer to default Baud structure */
    Can_BaudConfigType  **BaudRateConfigList;
	/**< List of available Baud rates */
}Can_ControllerType;

/**
 * \brief Can Controller Pre Compile Configuration definition
 */
typedef struct Can_ControllerStruct_PC
{
    uint8                 ControllerId;
    /**< Id as provided by GUI **/
    boolean               CntrActive;
    /**< Contoller is used=1 or not_used=0*/
    uint32                CntrAddr;
    /**< Pointer to CanReg structure */
    boolean               CanBusWakeupSupport;
    /**< Automatic wakeup via bus activity */
    Can_TxRxProcessingType    RxProcessingType;
    /**< Type can be Interrupt, Polling or Mixed */
    Can_TxRxProcessingType    TxProcessingType;
    /**< Type can be Interrupt, Polling or Mixed */
    boolean               BusOffProcessingInterrupt;
    /**< TRUE = Interrupt FALSE = Polling */
    boolean               WakeupProcessingInterrupt;
    /**< TRUE = Interrupt FALSE = Polling */
    boolean               DisableAutoRetransmission;
    /**< TRUE = Disable Auto Retransmission */
    Can_ErrNotifyType     Can_ErrorNotification;
    /**< Group end notification callback fxn pointer */
    Can_ControllerInstance CanControllerInst;
    /**< Can Controller Instance **/
}Can_ControllerType_PC;

/**
 * \brief Can mailox configuration definition
 */
typedef struct Can_MailboxStruct
{
    uint8                    CanHandleType;
    /**< CanHandleType 0=Full, *1=Basic */
    uint32                   MBIdType;
    /**< CanIdType 0=standard 1=Extended 2= Mixed */
    Can_HwHandleType         HwHandle;
    /**< Mailbox - Hw object in the controller*/
    uint16                   CanHwObjectCount;
    /**< Number of hardware objects used to implement one HOH*/
    Can_MailBoxDirectionType MBDir;
    /**< CanObjectType - Direction of Mailbox @sa Can_MailBoxDirectionType */
    const Can_ControllerType_PC    *Controller;
    /**< Reference to CAN Controller to which the HOH is associated to. */
    Can_HwFilterType         **HwFilterList;
    /**< List of HW Filter */
    uint32                   HwFilterCnt;
    /**< HW Filter Count */
    uint8                    CanFdPaddingValue;
    /**< If PduInfo->SduLength does not match possible DLC values
     *   CanDrv will use the next higher valid DLC for transmission with
     *   initialization of unused bytes to the value of
     *   the corresponding CanFdPaddingValue.
     */
    boolean                  CanHardwareObjectUsesPolling;
    /**< TRUE = Polling, FALSE = Interrupt */
	boolean                  CanTriggerTransmitEnable;
    /**< TRUE = Enable, FALSE = Disable */
}Can_MailboxType;

/**
 * \brief Can mailox Pre compile configuration definition
 */
typedef struct Can_MailboxStruct_PC
{
    uint16                   CanObjectId;
    /**< CanObjectId - Holds handle Id */
}Can_MailboxType_PC;
/**
 * \brief Can MCAL root configuration structure.
 */
typedef struct Can_ConfigType_s
{
    Can_ControllerType **CanControllerList;
    /**< List of enabled Controllers */
    uint8                CanMaxControllerCount;
    /**< MaxCount of Controller in Controller List */
    Can_MailboxType    **MailBoxList;
    /**< MB array for all controllers */
    uint32               MaxMbCnt;
    /**< MaxMbCount in MB list in all controller */
    uint32               MaxBaudConfigID[CAN_NUM_CONTROLLER];
    /**< Max Baud Config Index in BaudRateConfigList in all controller */
}Can_ConfigType;

#if (STD_ON == CAN_SAFETI_API)
/**
 * \brief  Register Readback Structure
 *
 * Structure to read back key CAN control registers.
 *
 * The Safety API to readback CAN register vales will populate this structure
 */
typedef struct
{
    uint32 CanReadBackRegCTL;
    /**< Control Register  */
    uint32 CanReadBackRegBTR;
    /**< Bit Timing Register  */
    uint32 CanReadBackRegABOTR;
    /**< Auto Bus On Time Register  */
    uint32 CanReadBackRegINTMUXx[4U];
    /**< Interrupt Multiplexer Registers */
} Can_RegisterReadbackType;
#endif /* (STD_ON == CAN_SAFETI_API) */

/* @} */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

#if (CAN_VERSION_INFO_API == STD_ON)
/**
 * \brief Function returns the version information of this module.
 *
 * \verbatim
 * Service name   :Can_GetVersionInfo
 * Syntax         :void Can_GetVersionInfo( Std_VersionInfoType* versioninfo )
 * Service ID[hex]:0x07
 * Sync/Async     :Synchronous
 * Reentrancy     :Reentrant
 * Parameters (in):None
 * Parameters (inout): None
 * Parameters (out)  : versioninfo
 *                     Pointer to where to store the version information of this
 *                     module.
 * Return value      : None
 * Description       : This function returns the version information of this
 *                     module
 * \endverbatim
 */
FUNC(void,
     CAN_CODE) Can_GetVersionInfo(P2VAR(Std_VersionInfoType, AUTOMATIC,
                                        CAN_APPL_DATA) Can_VersionInfo);
#endif

/**
 * \brief This function initializes the module.
 *
 * \verbatim
 * Service name      : Can_Init
 * Syntax            : void Can_Init( const Can_ConfigType* CfgPtr )
 * Service ID[hex]   : 0x00
 * Sync/Async        : Synchronous
 * Reentrancy        : Non Reentrant
 * Parameters (in)   : CfgPtr
 *                     Pointer to driver configuration.
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : None
 * Description       : This function initializes the module.
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_Init(P2CONST(Can_ConfigType, AUTOMATIC,
                                    CAN_PBCFG) CfgPtr);

/**
 * \brief This function performs software triggered state transitions of the CAN
 *                     controller State machine.
 *
 * \verbatim
 * Service name:Can_SetControllerMode
 * Syntax:Can_ReturnType Can_SetControllerMode( uint8 Controller,
 *                     Can_ControllerStateType Transition )
 * Service ID[hex]:0x03
 * Sync/Async:Asynchronous
 * Reentrancy:Non Reentrant
 * Parameters (in):Controller
 *                 - CAN controller for which the status shall be changed
 *                 Transition
 *                 - Transition value to request new CAN controller state
 * Parameters (inout):None
 * Parameters (out):  None
 * Return value: Can_ReturnType
 *               - CAN_OK: request accepted
 *               - CAN_NOT_OK: request not accepted, a development error
 *                     occurred
 * Description:  This function performs software triggered state transitions of
 *                     the CAN controller State machine.
 * \endverbatim
 */

FUNC(Std_ReturnType, CAN_CODE) Can_SetControllerMode(
    uint8 Controller, Can_ControllerStateType Transition);

/**
 * \brief This function is called by CanIf to pass a CAN message to CanDrv for
 *                     transmission.
 *
 * \verbatim
 * Service name:Can_Write
 * Syntax      :Can_ReturnType Can_Write( Can_HwHandleType Hth, const
 *                     Can_PduType* PduInfo )
 * Service ID[hex]:0x06
 * Sync/Async     :Synchronous
 * Reentrancy     :Reentrant (thread-safe)
 * Parameters (in):Hth
 *                 - Information which HW-transmit handle shall be used for
 *                  transmit. Implicitly this is also the information about
 *                  the controller to use because the Hth numbers are unique
 *                  inside one hardware unit.
 *                 PduInfo
 *                 - Pointer to SDU user memory, DLC and Identifier.
 * Parameters (inout):None
 * Parameters (out)  :None
 * Return value:Can_ReturnType
 *              - CAN_OK: Write command has been accepted
 *              - CAN_NOT_OK: development error occurred
 *              - CAN_BUSY: No TX hardware buffer available or pre-emptive call
 *                     of Can_Write that can't be implemented re-entrant
 * Description:This function is called by CanIf to pass a CAN message to CanDrv
 *                     for transmission.
 * \endverbatim
 */
FUNC(Std_ReturnType, CAN_CODE) Can_Write(Can_HwHandleType Hth,
                                         const Can_PduType *PduInfo);
#if (CAN_WAKEUP_FUNCTIONALITY_API == STD_ON)
/**
 * \brief This function checks if a wakeup has occurred for the given
 *                     controller.
 *
 * \verbatim
 * Service name      :Can_CheckWakeup
 * Syntax            :Can_ReturnType Can_CheckWakeup( uint8 Controller )
 * Service ID[hex]   :0x0b
 * Sync/Async        :Synchronous
 * Reentrancy        :Non Reentrant
 * Parameters (in)   :Controller
 *                    - Controller to be checked for a wakeup.
 * Parameters (inout):None
 * Parameters (out)  :None
 * Return value      :Can_ReturnType
 *                    - CAN_OK: API call has been accepted
 *                    - CAN_NOT_OK: API call has not been accepted
 * Description       :This function checks if a wakeup has occurred for the
 *                     given controller.
 * \endverbatim
 */
FUNC(Can_ReturnType, CAN_CODE) Can_CheckWakeup(uint8 Controller);
#endif
/**
 * \brief This function disables all interrupts for this CAN controller
 *
 * \verbatim
 * Service name      : Can_DisableControllerInterrupts
 * Syntax            :void Can_DisableControllerInterrupts( uint8 Controller )
 * Service ID[hex]   :0x04
 * Sync/Async        :Synchronous
 * Reentrancy        :Reentrant
 * Parameters (in)   :Controller
 *                    - CAN controller for which interrupts shall be disabled.
 * Parameters (inout):None
 * Parameters (out)  :None
 * Return value      :None
 * Description       :This function disables all interrupts for this CAN
 *                     controller
 *
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_DisableControllerInterrupts(uint8 Controller);

/**
 * \brief This function enables all allowed interrupts
 *
 * \verbatim
 * Service name      : Can_EnableControllerInterrupts
 * Syntax            :void Can_EnableControllerInterrupts( uint8 Controller )
 * Service ID[hex]   :0x05
 * Sync/Async        :Synchronous
 * Reentrancy        :Reentrant
 * Parameters (in)   :Controller
 *                    - CAN controller for which interrupts shall be disabled.
 * Parameters (inout):None
 * Parameters (out)  :None
 * Return value      :None
 * Description       :This function enables all allowed interrupts
 *
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_EnableControllerInterrupts(uint8 Controller);

/* polling functions (always available for non polled systems also)*/

/**
 * \brief This function performs the polling of TX confirmation when
 *                     CAN_TX_PROCESSING is set to POLLING.
 *
 * \verbatim
 * Service name   :Can_MainFunction_Write
 * Syntax         :void Can_MainFunction_Write( void )
 * Service ID[hex]:0x01
 * Description    :This function performs the polling of TX confirmation when
 *                     CAN_TX_PROCESSING is set to POLLING.
 *
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_MainFunction_Write(void);

/**
 * \brief This function performs the polling of bus-off events that are
 *                     configured statically as 'to be polled'
 *
 * \verbatim
 * Service name   :Can_MainFunction_BusOff
 * Syntax         :void Can_MainFunction_BusOff( void )
 * Service ID[hex]:0x09
 * Description    :This function performs the polling of bus-off events that are
 *                     configured statically as 'to be polled'.
 *
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_MainFunction_BusOff(void);

/**
 * \brief This function performs the polling of RX indications when
 *                     CAN_RX_PROCESSING is set to POLLING
 *
 * \verbatim
 * Service name   :Can_MainFunction_Read
 * Syntax         :void Can_MainFunction_Read( void )
 * Service ID[hex]:0x08
 * Description    :This function performs the polling of RX indications when
 *                     CAN_RX_PROCESSING is set to POLLING.
 *
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_MainFunction_Read(void);

/**
 * \brief This function performs the polling of wake-up events that are
 *                     configured statically as 'to be polled
 *
 * \verbatim
 * Service name   :Can_MainFunction_Wakeup
 * Syntax         :void Can_MainFunction_Wakeup( void )
 * Service ID[hex]:0x0A
 * Description    :This function performs the polling of wake-up events that are
 *                     configured statically as 'to be polled.
 *
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_MainFunction_Wakeup(void);

/**
 * \brief This function performs the polling of CAN controller mode transitions
 *
 * \verbatim
 * Service name   :Can_MainFunction_Mode
 * Syntax         :void Can_MainFunction_Mode( void )
 * Service ID[hex]:0x0C
 * Description    :This function performs the polling of CAN controller mode
 *                     transitions.
 *
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_MainFunction_Mode(void);


#if (CAN_LOOPBACK_ENABLE == STD_ON)
/**
 * \brief This service will enable CAN loopback mode.
 *
 * \verbatim
 *
 *  Service name      :Can_TestLoopBackModeEnable
 *  Syntax            :Std_ReturnType Can_TestLoopBackModeEnable( uint8
 *                     Controller, uint8 mode )
 *  Service ID[hex]   :0x0f
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Reentrant
 *  Parameters (in)   :Controller
 *                     - CAN Controller to enable loopback mode
 *                     mode
 *                     - Loopback mode
 *                     -# 0 - Digital Loopback
 *                     -# 1 - Analog  Loopback
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :Std_ReturnType
 *                     - E_OK: Baudrate supported by the CAN Controller
 *                     - E_NOT_OK: Baudrate not supported / invalid CAN
 *  Description       : This service will enable CAN loopback mode.
 * \endverbatim
 */
FUNC(Std_ReturnType, CAN_CODE) Can_TestLoopBackModeEnable(uint8 controller,
                                                          uint8 mode);

/**
 * \brief This service will disable CAN loopback mode.
 *
 * \verbatim
 *
 *  Service name      :Can_TestLoopBackModeDisable
 *  Syntax            :Std_ReturnType Can_TestLoopBackModeDisable( uint8
 *                     Controller, uint8 mode )
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Reentrant
 *  Parameters (in)   :Controller
 *                     - CAN Controller to enable loopback mode
 *                     mode
 *                     Loopback mode
 *                     -# 0 - Digital Loopback
 *                     -# 1 - Analog  Loopback
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :Std_ReturnType
 *                     - E_OK: Loopback mode enabled in the CAN Controller
 *                     - E_NOT_OK: Loopback mode enable failed
 *  Description       : This service will disable CAN loopback mode.
 * \endverbatim
 */
FUNC(Std_ReturnType, CAN_CODE) Can_TestLoopBackModeDisable(uint8 controller,
                                                           uint8 mode);
#endif


#if defined(CAN_CONTROLLER_MCAN0)

/**
 * \brief This function is the ISR for CAN controller 0/INT 0
 *
 * \verbatim
 *
 *  Function name     :Can_0_Int0ISR_Fun
 *  Syntax            :void Can_0_Int0ISR_Fun( void )
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :None
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function is the ISR for CAN controller 0/INT 0 <br>
 *                     This function should be invoked either by OS Interrupt
 *                     frame handler <br>
 *                     CAN Interrupt frame handler for Category 1 ISR
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_0_Int0ISR_Fun(void);

/**
 * \brief This function is the ISR for CAN controller 0/INT 1
 *
 * \verbatim
 *
 *  Function name     :Can_0_Int1ISR_Fun
 *  Syntax            :void Can_0_Int1ISR_Fun( void )
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :None
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function is the ISR for CAN controller 0/INT 1 <br>
 *                     This function should be invoked either by OS Interrupt
 *                     frame handler <br>
 *                     CAN Interrupt frame handler for Category 1 ISR
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_0_Int1ISR_Fun(void);
#endif

#if defined (CAN_CONTROLLER_MCAN1)
/**
 * \brief This function is the ISR for CAN controller 1/INT 0
 *
 * \verbatim
 *
 *  Function name     :Can_1_Int0ISR_Fun
 *  Syntax            :void Can_1_Int0ISR_Fun( void )
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :None
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function is the ISR for CAN controller 1/INT 0 <br>
 *                     This function should be invoked either by OS Interrupt
 *                     frame handler <br>
 *                     CAN Interrupt frame handler for Category 1 ISR
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_1_Int0ISR_Fun(void);

/**
 * \brief This function is the ISR for CAN controller 1/INT 1
 *
 * \verbatim
 *
 *  Function name     :Can_1_Int1ISR_Fun
 *  Syntax            :void Can_1_Int1ISR_Fun( void )
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :None
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function is the ISR for CAN controller 1/INT 1 <br>
 *                     This function should be invoked either by OS Interrupt
 *                     frame handler <br>
 *                     CAN Interrupt frame handler for Category 1 ISR
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_1_Int1ISR_Fun(void);
#endif

#if defined (CAN_CONTROLLER_MCAN2)
/**
 * \brief This function is the ISR for CAN controller 3/INT 0
 *
 * \verbatim
 *
 *  Function name     :Can_2_Int0ISR_Fun
 *  Syntax            :void Can_2_Int0ISR_Fun( void )
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :None
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function is the ISR for CAN controller 3/INT 0 <br>
 *                     This function should be invoked either by OS Interrupt
 *                     frame handler <br>
 *                     CAN Interrupt frame handler for Category 1 ISR
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_2_Int0ISR_Fun(void);

/* #if (STD_ON == CAN_ECC_ENABLE) */
/**
 * \brief This function is the ISR for CAN controller 3/INT 1
 *
 * \verbatim
 *
 *  Function name     :Can_2_Int1ISR_Fun
 *  Syntax            :void Can_2_Int1ISR_Fun( void )
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :None
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function is the ISR for CAN controller 3/INT 1 <br>
 *                     This function should be invoked either by OS Interrupt
 *                     frame handler <br>
 *                     CAN Interrupt frame handler for Category 1 ISR
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_2_Int1ISR_Fun(void);
/* #endif */
#endif

#if defined (CAN_CONTROLLER_MCAN3)
/**
 * \brief This function is the ISR for CAN controller 4/INT 0
 *
 * \verbatim
 *
 *  Function name     :Can_3_Int0ISR_Fun
 *  Syntax            :void Can_3_Int0ISR_Fun( void )
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :None
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function is the ISR for CAN controller 4/INT 0 <br>
 *                     This function should be invoked either by OS Interrupt
 *                     frame handler <br>
 *                     CAN Interrupt frame handler for Category 1 ISR
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_3_Int0ISR_Fun(void);

/* #if (STD_ON == CAN_ECC_ENABLE) */
/**
 * \brief This function is the ISR for CAN controller 4/INT 1
 *
 * \verbatim
 *
 *  Function name     :Can_3_Int1ISR_Fun
 *  Syntax            :void Can_3_Int1ISR_Fun( void )
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :None
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function is the ISR for CAN controller 4/INT 1 <br>
 *                     This function should be invoked either by OS Interrupt
 *                     frame handler <br>
 *                     CAN Interrupt frame handler for Category 1 ISR
 * \endverbatim
 */
FUNC(void, CAN_CODE) Can_3_Int1ISR_Fun(void);
/* #endif */
#endif

/**
 * \brief This function Sets can controller baud rate selected by
 **BaudRateConfigID
 *
 * \verbatim
 *
 *  Function name     :Can_SetBaudrate
 *  Syntax            :FUNC(Std_ReturnType, CAN_CODE) Can_SetBaudrate(uint8
 **Controller, uint16 BaudRateConfigID)
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :Controller
 *                     -Controller whose Baud Rate is being set
 *                     BaudRateConfigID
 *                     -ID number corresponding to a Baud Rate Configuration
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function Sets can controller baud rate selected by
 **BaudRateConfigID
 *
 * \endverbatim
 */
FUNC(Std_ReturnType, CAN_CODE) Can_SetBaudrate(uint8 Controller,
                                               uint16 BaudRateConfigID);

#if (STD_ON == CAN_SAFETI_API)
/**
 * \brief This service will readback CAN registers
 *
 * \verbatim
 *
 *  Service name      :Can_ConfigRegReadBack
 *  Syntax            :Std_ReturnType Can_ConfigRegReadBack( uint8 Controller )
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Reentrant
 *  Parameters (in)   :Controller
 *                     - CAN Controller to readback registers from
 *  Parameters (inout):RegRbPtr - Pointer to where to store the readback
 *                     values. If this pointer is NULL_PTR, then the API
 *                     will return E_NOT_OK.
 *  Parameters (out)  :None
 *  Return value      :Std_ReturnType
 *                     - E_OK: Loopback mode disabled in the CAN Controller
 *                     - E_NOT_OK: Loopback mode disable failed
 *  Description       : This service will disable CAN loopback mode.
 * \endverbatim
 */
Std_ReturnType Can_RegisterReadback(VAR(uint8, AUTOMATIC)controller,
                    P2VAR(Can_RegisterReadbackType, AUTOMATIC,
                                    CAN_APPL_DATA) RegRbPtr);
#endif
#if (STD_ON == CAN_ECC_ENABLE)
/**
 * \brief This function Enables/Disables Parity
 *
 * \verbatim
 *
 *  Function name     :Can_EccParityEnable
 *  Syntax            :void Can_EccParityEnable
 *                     (uint8 controller, uint8 flag, uint8 ramInitEnable)
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :controller - Controller Id, flag - Enable/Disable Parity
 *                     ramInitEnable - Enable/Disable Ram Init
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function used to test ECC/Parity error.
 *                     ramInitEnable should be set to TRUE if ECC
 *                     enabled for normal operations otherwise pass false
 *                     if it is diagnostic purpose.
 *
 * \endverbatim
 */
Std_ReturnType Can_EccParityEnable(VAR(uint8, AUTOMATIC)controller,
                                        uint8 flag,
                                        uint8 ramInitEnable);
/**
 * \brief This function returns corrupted mailbox number
 *
 * \verbatim
 *
 *  Function name     :Can_EccDiagnosticGetErrMsgBoxNum
 *  Syntax            :Std_ReturnType Can_EccDiagnosticGetErrMsgBoxNum
                       (VAR(uint8,AUTOMATIC)controller, uint8 *msgBox)
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :Parameters (in)   :controller - Controller Id
 *  Parameters (inout):msgBox - Message number which caused ECC error.
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function returns mailbox number whose RAM is
 **Corrupted
 * \endverbatim
 */
Std_ReturnType Can_EccDiagnosticGetErrMsgBoxNum(VAR(uint8, AUTOMATIC)
                                              controller, uint8 *msgBox);
/**
 * \brief This function Sets ECC diagnostic mode
 *
 * \verbatim
 *
 *  Function name     :Can_EccDiagnosticModeEnable
 *  Syntax            :Std_ReturnType Can_EccDiagnosticModeEnable
                       (VAR(uint8, AUTOMATIC)controller, uint8 flag)
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :controller - Controller Id, flag - Enable/Disable Parity
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function used to test Single bit ECC correction
 *
 * \endverbatim
 */
Std_ReturnType Can_EccDiagnosticModeEnable(VAR(uint8,
                                         AUTOMATIC) controller, uint8 flag);
/**
 * \brief This API will read the ECC error status information of MCAN
 *            controller
 *
 * \verbatim
 *
 *  Function name     :Can_GetEccErrStatus
 *  Syntax            :Std_ReturnType Can_GetEccErrStatus(uint8 controller,
                                                         uint32 *ecccErrStatus)
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :controller - Controller Id
 *  Parameters (inout):ecccErrStatus
 *  Parameters (out)  :retVal - ECC error status
 *  Return value      :Returns the ECC error status information
 *  Description       :This function used to test Double bit error detection
 *
 * \endverbatim
 */
Std_ReturnType Can_EccGetErrStatus(uint8 controller, uint32 *ecccErrStatus);
/**
 * \brief This API clears the ECC error status register
 *
 *
 * \verbatim
 *
 *  Function name     :Can_ClrEccErrStatus
 *  Syntax            :Std_ReturnType Can_ClrEccErrStatus(uint8 controller)
 *  Service ID[hex]   :N/A
 *  Sync/Async        :Synchronous
 *  Reentrancy        :Non-Reentrant
 *  Parameters (in)   :controller - Controller Id
 *  Parameters (inout):None
 *  Parameters (out)  :None
 *  Return value      :None
 *  Description       :This function used to clear ECC error status register
 *
 * \endverbatim
 */
Std_ReturnType Can_EccClrErrStatus(uint8 controller);
#endif /* (STD_ON == CAN_ECC_ENABLE) */

#if (CAN_DEINIT_API == STD_ON)
/**
 *  \brief This service is only used for UT testing and not for app use.
 *
 *  \verbatim
 *  Service name      : Can_DeInit
 *  Syntax            : void Can_DeInit(void)
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non-Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : This API is not standard AUTOSAR API.
 *                      This service sets the internal Can Controller state to
 *                      UNINIT state so that Can_Init can be invoked again.
 *                      This is required for UT test cases with diff Config.
 *  \endverbatim
 */
FUNC(void, CAN_CODE) Can_DeInit(void);

#endif /* (STD_ON == CAN_DEINIT_API) */

/**
 * \brief This function updates the error state of the controller.
 *
 * \verbatim
 * Service name: Can_GetControllerErrorState
 * Syntax      : Std_ReturnType Can_GetControllerErrorState( uint8 ControllerId,
 *               Can_ErrorStateType* ErrorStatePtr )
 * Service ID[hex]:0x11
 * Sync/Async     :Synchronous
 * Reentrancy     :Non Reentrant
 * Parameters (in):ControllerId
 *                 - ID of the CAN controller whose error state is requested.
 * Parameters (inout):None
 * Parameters (out)  :ErrorStatePtr 
 *                    - Pointer where the error state is updated.
 * Return value:Std_ReturnType
 *              - E_OK: Error state request accepted
 *              - E_NOT_OK: Error state request not accepted
 * Description:This function updates the error state of the controller.
 * \endverbatim
 */
/* [SWS_Can_91004] */
FUNC(Std_ReturnType, CAN_CODE) Can_GetControllerErrorState(uint8 ControllerId,
                                 Can_ErrorStateType* ErrorStatePtr);



/**
 * \brief This service reports about the current status of the requested
 *        CAN controller.
 *
 * \verbatim
 * Service name: Can_GetControllerMode
 * Syntax: Std_ReturnType Can_GetControllerMode(uint8 Controller,
 *                     Can_ControllerStateType* ControllerModePtr)
 * Service ID[hex]: 0x12
 * Sync/Async: Synchronous
 * Reentrancy: Non Reentrant
 * Parameters (in): Controller
 *                  - CAN controller for which the status shall be changed
 * Parameters (out): ControllerModePtr
 *                  - Pointer to a memory location,where 
 *                    the current mode of the CAN controller will be stored.
 * Parameters (inout):  None
 * Return value: Std_ReturnType
 *               - E_OK: request accepted
 *               - E_NOT_OK: request not accepted.
 * Description:  This service reports about the current status of the requested
 *               CAN controller.
 * \endverbatim
 */
/*  [SWS_Can_91014] */
FUNC(Std_ReturnType, CAN_CODE) Can_GetControllerMode(uint8 Controller,
  Can_ControllerStateType* ControllerModePtr);


#ifdef __cplusplus
}
#endif

#endif /* CAN_H_ end file*/
/*  End of File: Can.h */

/* @} */
