/* ======================================================================
*   Copyright (C) 2022-2023 Texas Instruments Incorporated
*
*   All rights reserved. Property of Texas Instruments Incorporated.
*   Restricted rights to use, duplicate or disclose this code are
*   granted through contract.
*
*   The program may not be used without the written permission
*   of Texas Instruments Incorporated or against the terms and conditions
*   stipulated in the agreement under which this program has been
*   supplied.
* ==================================================================== */

/* -------------------------------------------------------------------------------------------------------------------
 *  FILE DESCRIPTION
 *  -------------------------------------------------------------------------------------------------------------------
 *         File:  Can.c
 *      Project:  AM263x MCAL driver
 *       Module:  CAN Driver
 *
 *  Description:    The CAN driver provides services for CAN communication

 *---------------------------------------------------------------------------------------------------------------------
 *********************************************************************************************************************/

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "string.h"
#include "SchM_Can.h"
#include "Can.h"
#include "CanIf_Cbk.h"
#include "EcuM_Cbk.h"
/* #include "Os.h" */

#if (CAN_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif


#include "soc.h"
#define CAN_START_SEC_CODE
#include "Can_MemMap.h"
#include "hw_types.h"
#define CAN_STOP_SEC_CODE
#include "Can_MemMap.h"
#include "Can_Priv.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* AUTOSAR version information check has to match definition in header file */
#if ((CAN_AR_RELEASE_MAJOR_VERSION != (4U))   \
    || (CAN_AR_RELEASE_MINOR_VERSION != (3U)) \
    || (CAN_AR_RELEASE_REVISION_VERSION != (1U)))
#error "CAN: AUTOSAR Version Numbers of CAN are different"
#endif

#if ((CAN_SW_MAJOR_VERSION != (8U)) || (CAN_SW_MINOR_VERSION != (6U)))
#error "Vendor specific version numbers of Can.c and Can.h are inconsistent"
#endif

#if ((CAN_CFG_MAJOR_VERSION != (8U)) || (CAN_CFG_MINOR_VERSION != (6U)))
#error "Version numbers of Can.c and Can_Cfg.h are inconsistent!"
#endif

/* FMEA requirement */
#if (STD_OFF == CAN_ECC_ENABLE)
/* TODO #warn "CAN parity check is not enabled " */
#endif

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
/* None */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */
#if (CAN_DEV_ERROR_DETECT == STD_ON)
static boolean Can_CheckInitDet(const Can_ConfigType *ConfigPtr);
static boolean Can_CheckControllerConfig(const Can_ConfigType *ConfigPtr);
static boolean Can_CheckMbConfig(const Can_ConfigType *ConfigPtr);
static boolean Can_CheckWriteDet(const uint32 Hth,
                                 const Can_PduType *PduInfo,
                                 const Can_DriverObjType *canDrvObj);
static boolean Can_CheckEnableDet(uint8 Controller,
                                  const Can_DriverObjType * canDrvObj);
static boolean Can_CheckDisableDet(uint8 Controller,
                                   const Can_DriverObjType * canDrvObj);
static boolean Can_CheckSetControllerModeDet(uint8 Controller,
                                   const Can_DriverObjType * canDrvObj);
static boolean Can_CheckWakeupDet(uint8 Controller,
                                  const Can_DriverObjType * canDrvObj);
static boolean Can_SetBaudrateDet(uint8 Controller,
                                  const Can_DriverObjType * canDrvObj);
#endif

static void Can_resetDrvObj(Can_DriverObjType *drvObj);
static void Can_initDrvObj(Can_DriverObjType *drvObj,
                           const Can_ConfigType *ConfigPtr);

/*******************************************************************************
 *  CONSTANT MACRO DEFINITION
 ******************************************************************************/

/* Interrupt source */
#define INT_IE0                  (0x1U)
#define INT_IE1                  (0x2U)

/**
 *  \brief CAN INVALID BASE ADDRESS
 */
#define CAN_INVALID_BASE_ADDR           (0x00000000U)

#define CAN_ID_CAN_CONTROLLER_TYPE_MASK         (0x40000000U)
#define CAN_ID_CAN_ID_TYPE_MASK                 (0x80000000U)

/*******************************************************************************
 * MACRO DEFINITIONS
 ******************************************************************************/

/*******************************************************************************
 *  LOCAL DATA DEFINITIONS
 ******************************************************************************/

#define CAN_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Can_MemMap.h"

/*LDRA_INSPECTED 94 S : MISRAC_2012_R.11.1
 * "Reason - NULL pointer is defined as void pointer to zero. " */
/*LDRA_INSPECTED 95 S : MISRAC_2012_R.11.4
 * "Reason - NULL pointer is defined as void pointer to zero. " */
static VAR(Can_DriverObjType, CAN_VAR_NOINIT) Can_DriverObj;

#define CAN_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Can_MemMap.h"

#define CAN_START_SEC_VAR_INIT_8
#include "Can_MemMap.h"

VAR(uint8, CAN_VAR_INIT) Can_DrvState = CAN_UNINIT;

#define CAN_STOP_SEC_VAR_INIT_8
#include "Can_MemMap.h"

#define CAN_START_SEC_CODE
#include "Can_MemMap.h"

/*Requirements :SWS_Can_00079 , SWS_Can_00077, SWS_Can_00284, SWS_Can_00385,
 * SWS_Can_00386, SWS_Can_00427, SWS_Can_00401, SWS_Can_00402, SWS_Can_00403,
 * SWS_Can_00060,
 * SWS_Can_00202, SWS_Can_00110, SWS_Can_00234,
 * SWS_Can_00022,
 * SWS_Can_00024, SWS_Can_00221
 * SWS_Can_00490, SWS_Can_00440, SWS_Can_00242, SWS_Can_00398, SWS_Can_00397,
 * SWS_Can_00390  */

/* Optional Requirements which not implemented */
/* SWS_Can_00235, SWS_Can_00236, SWS_Can_00235, SWS_Can_00504, SWS_Can_00503 */

/* Not Applicable */
/* SWS_Can_00391, SWS_Can_00300, SWS_Can_00299, SWS_Can_00281 */

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */
/*******************************************************************************
 * Can_resetDrvObj
 ******************************************************************************/
/*! \brief      This function will reset default  Config structure.
 *
 *  \param[in]  Can_DriverObjType drvObj Pointer to Driver Config structure.
 *
 *  \context
 ******************************************************************************/
static void Can_resetDrvObj(Can_DriverObjType *drvObj)
{
    uint32 controllerIndx, mbIndx, loopIndx;

    for (controllerIndx = 0U; controllerIndx < (uint32)CAN_NUM_CONTROLLER;
         controllerIndx++)
    {
        if (Can_DrvState == CAN_UNINIT)
        {
            drvObj->canController[controllerIndx].canState = CAN_CS_UNINIT;
        }
        drvObj->canController[controllerIndx].canBaud = 0U;
        drvObj->canController[controllerIndx].canBusOffRecoveryStatus =
                                                            (boolean) FALSE;
        drvObj->canController[controllerIndx].canInterruptCounter = (uint8)0U;
        drvObj->canController[controllerIndx].canInterruptMask = (uint32)0U;
        for (mbIndx = 0U; mbIndx < (uint32)CAN_FD_PAYLOAD_MAX_BYTES; mbIndx++)
        {
            drvObj->canController[controllerIndx].canSduPtr[mbIndx] = (uint8)0U;
        }
		drvObj->canController[controllerIndx].canFDMsgRamConfig.stdFilterNum = 0U;
		drvObj->canController[controllerIndx].canFDMsgRamConfig.extFilterNum = 0U;
    }
	for (mbIndx = 0U; mbIndx < (uint32)CAN_NUM_TX_MAILBOXES; mbIndx++)
	{
		drvObj->canTxMessageObj[mbIndx].freeHwObjectCount = 0U;
		drvObj->canTxMessageObj[mbIndx].buffIdx = 0U;
	}	
    return;
}

/*******************************************************************************
 * Can_initDrvObj
 ******************************************************************************/
/*! \brief      This function will initialize Driver Config structure.
 *
 *  \param[in]  Can_DriverObjType drvObj Pointer to Driver Config structure.
 *  \param[in]  Can_ConfigType ConfigPtr Pointer to Can controllers
 *              Config structure.
 *
 *  \context
 ******************************************************************************/
static void Can_initDrvObj(Can_DriverObjType *drvObj,
                           const Can_ConfigType *ConfigPtr)
{
    uint32 controllerIndx, mbIndx;
	uint32 htrh;
    const Can_MailboxType *mailboxCfg;
    const Can_MailboxObjType *mailBoxCfgList;
	Can_MailboxObjTxType *canTxMessageObj;
	uint16 hth;
	uint32 maxMbCnt;
	uint32 hwObjectCount = 0U;

	for (controllerIndx = 0U ; controllerIndx < (uint32)CAN_NUM_CONTROLLER ;
         controllerIndx++)
    {
        (void) memcpy(
            &drvObj->canController[controllerIndx].canControllerConfig,
            ConfigPtr->CanControllerList[controllerIndx],
            sizeof(Can_ControllerType));
        (void) memcpy(
            &drvObj->canController[controllerIndx].canControllerConfig_PC,
            CanConfigSet_CanController_List_PC[controllerIndx],
            sizeof(Can_ControllerType_PC));
        drvObj->canController[controllerIndx].maxBaudConfigID =
                                ConfigPtr->MaxBaudConfigID[controllerIndx];
        drvObj->controllerIDMap[
        CanConfigSet_CanController_List_PC[controllerIndx]->CanControllerInst] =
            controllerIndx;
    }
    for (mbIndx = 0U ; mbIndx < (uint32)CAN_NUM_MAILBOXES ; mbIndx++)
    {
        (void) memcpy(
            &drvObj->canMailbox[mbIndx].mailBoxConfig,
            ConfigPtr->MailBoxList[mbIndx],
            sizeof(Can_MailboxType));
        (void) memcpy(
            &drvObj->canMailbox[mbIndx].mailBoxConfig_PC,
            CanConfigSet_CanHardwareObject_List_PC[mbIndx],
            sizeof(Can_MailboxType_PC));
    }
    drvObj->canMaxControllerCount = ConfigPtr->CanMaxControllerCount;
    drvObj->maxMbCnt              = ConfigPtr->MaxMbCnt;
	
	mailBoxCfgList 	= &drvObj->canMailbox[0U];
	canTxMessageObj = &drvObj->canTxMessageObj[0];
	maxMbCnt 		= drvObj->maxMbCnt;
    /* Initialize Can_MailboxObjTxType params according to configured mailboxes */
    for (htrh = 0U; htrh < maxMbCnt; htrh++)
    {
        mailboxCfg = &mailBoxCfgList[htrh].mailBoxConfig;
		
		if (CAN_MAILBOX_DIRECTION_TX == mailboxCfg->MBDir)
		{
			hth = mailboxCfg->HwHandle;
			canTxMessageObj[hth].freeHwObjectCount = mailboxCfg->CanHwObjectCount;
        }
	}
}	

/*******************************************************************************
 * Can_SetControllerMode
 ******************************************************************************/
/*! \brief      This function will transition the controller to different
 * states, Its AUTOSAR defined states.
 *
 *  \param[in]  uint8 Controller -  The controller for which the state change
 *              needs to be performed/
 *              Can_ControllerStateType Transition - the state to which
 *              controller needs to move in.
 *
 *
 *  \context
 ******************************************************************************/
/*Requirements: SWS_Can_00261, SWS_Can_00262, SWS_Can_00257, SWS_Can_00265 ,
 * SWS_Can_00267, SWS_Can_00230
 * SWS_Can_00017, SWS_Can_00384, SWS_Can_00196, SWS_Can_00425, SWS_Can_00197,
 * SWS_Can_00426, SWS_Can_00198,
 * SWS_Can_00199, SWS_Can_00200, SWS_Can_00039*/
/* Design : CAN_DesignId_009 */
FUNC(Std_ReturnType, CAN_CODE) Can_SetControllerMode(
    uint8 Controller, Can_ControllerStateType Transition)
{
    Std_ReturnType retVal = E_NOT_OK;

    /*check for the validity of the controller parameter if det is enabled*/
#if (CAN_DEV_ERROR_DETECT == STD_ON)
    if (Can_CheckSetControllerModeDet(Controller,
                                      &Can_DriverObj) == (boolean) FALSE)
    {
        /* Det Error is already reported */
    }
    else
#endif
    {
		switch (Transition)
		{
			case CAN_CS_STARTED:
			{
				retVal = Can_hwUnitStart(&Can_DriverObj.canController[Controller]);
				break;
			}

			case CAN_CS_STOPPED:
			{
				retVal = Can_hwUnitStop(&Can_DriverObj.canController[Controller],
										&Can_DriverObj.canMailbox[0], 
										&Can_DriverObj.canTxMessageObj[0], 
										Can_DriverObj.maxMbCnt);
				break;
			}

			case CAN_CS_SLEEP:
			{
				retVal = Can_hwUnitSleep(&Can_DriverObj.canController[Controller]);
				break;
			}

			default:
			{
	#if (CAN_DEV_ERROR_DETECT == STD_ON)
				retVal = E_NOT_OK;
				(void) Det_ReportError((uint16) CAN_MODULE_ID,
									   (uint8) CAN_INSTANCE_ID,
									   (uint8) CAN_SETCTR_ID,
									   (uint8) CAN_E_TRANSITION);
	#endif
			break;
			}
		}
    }

    return retVal;
}

/*******************************************************************************
 * Can_CheckInitDet
 ******************************************************************************/
/*! \brief      This function will check for NULL pointers for Config structure.
 *
 *  \param[in]  Can_ConfigType ConfigPtr Pointer to Driver Config structure.
 *
 *  \context
 ******************************************************************************/
/* Design :CAN_DesignId_006*/
#if (CAN_DEV_ERROR_DETECT == STD_ON)
static boolean Can_CheckInitDet(const Can_ConfigType *ConfigPtr)
{
    boolean returnstatus = (boolean) TRUE;

    returnstatus = Can_CheckControllerConfig(ConfigPtr);
    if ((boolean) TRUE == returnstatus)
    {
        returnstatus = Can_CheckMbConfig(ConfigPtr);
    }

    return returnstatus;
}

static boolean Can_CheckControllerConfig(const Can_ConfigType *ConfigPtr)
{
    boolean returnstatus = (boolean) TRUE;
    uint32 loopCnt, cnt;

    for (loopCnt = 0U ; loopCnt < CAN_NUM_CONTROLLER ; loopCnt++)
    {
        if (NULL_PTR == ConfigPtr->CanControllerList[loopCnt])
        {
            returnstatus = (boolean) FALSE;
        }
        else if (NULL_PTR == ConfigPtr->CanControllerList[loopCnt]->DefaultBaud)
        {
            returnstatus = (boolean) FALSE;
        }
        else if (CAN_INVALID_BASE_ADDR == 
                                CanConfigSet_CanController_List_PC[loopCnt]->CntrAddr)
        {
            returnstatus = (boolean) FALSE;
        }
        else
        {
            for (cnt = 0U ; cnt <= ConfigPtr->MaxBaudConfigID[loopCnt]; cnt++)
            {
                if (NULL_PTR ==
                ConfigPtr->CanControllerList[loopCnt]->BaudRateConfigList[cnt])
                {
                    returnstatus = (boolean) FALSE;
                }
            }
        }

        if ( returnstatus == (boolean)FALSE)
        {
            break;
        }
    }

    return returnstatus;
}

static boolean Can_CheckMbConfig(const Can_ConfigType *ConfigPtr)
{
    boolean returnstatus = (boolean) TRUE;
    uint32 loopCnt;

    for (loopCnt = 0U ; loopCnt < CAN_NUM_MAILBOXES ; loopCnt++)
    {
        if (NULL_PTR == ConfigPtr->MailBoxList[loopCnt])
        {
            returnstatus = (boolean) FALSE;
        }
        else if (NULL_PTR == ConfigPtr->MailBoxList[loopCnt]->Controller)
        {
            returnstatus = (boolean) FALSE;
        }
        else
        {
            /*Do Nothing*/
        }

        if ( returnstatus == (boolean)FALSE)
        {
            break;
        }
    }
    return returnstatus;
}
#endif

/*******************************************************************************
 * Can_CheckSetControllerModeDet
 ******************************************************************************/
/*! \brief      This function will check DET for SetControllerMode API.
 *
 *  \param[in]  uint8 Controller Controller Number in the can hardware its 0-3
 *
 *  \context
 ******************************************************************************/
#if (CAN_DEV_ERROR_DETECT == STD_ON)
/* Design :CAN_DesignId_006*/
static boolean Can_CheckSetControllerModeDet(uint8 Controller,
                                   const Can_DriverObjType * canDrvObj)
{
    boolean returnstatus;
    if (Can_DrvState == CAN_UNINIT)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_SETCTR_ID, (uint8) CAN_E_UNINIT);
        returnstatus = (boolean) FALSE;
    }
    else if (Controller >= (canDrvObj->canMaxControllerCount))
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_SETCTR_ID,
                               (uint8) CAN_E_PARAM_CONTROLLER);
        returnstatus = (boolean) FALSE;
    }
    else if (canDrvObj->canController[Controller].
                        canControllerConfig_PC.CntrActive == (boolean) FALSE)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_SETCTR_ID, (uint8) CAN_E_UNINIT);
        returnstatus = (boolean) FALSE;
    }
    else
    {
        /* MISRA C Compliance */
        returnstatus = (boolean) TRUE;
    }
    return returnstatus;
}

#endif

/*******************************************************************************
 * Can_CheckDisableDet
 ******************************************************************************/
/*! \brief      This function will check DET for Disable API.
 *
 *  \param[in]  uint8 Controller Controller Number in the can hardware its 0-3
 *
 *  \context
 ******************************************************************************/
#if (CAN_DEV_ERROR_DETECT == STD_ON)
/* Design :CAN_DesignId_006*/
static boolean Can_CheckDisableDet(uint8 Controller,
                                   const Can_DriverObjType * canDrvObj)
{
    boolean returnstatus;
    if (Can_DrvState == CAN_UNINIT)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_DIINT_ID, (uint8) CAN_E_UNINIT);
        returnstatus = (boolean) FALSE;
    }
    else if (Controller >= (canDrvObj->canMaxControllerCount))
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_DIINT_ID,
                               (uint8) CAN_E_PARAM_CONTROLLER);
        returnstatus = (boolean) FALSE;
    }
    else if (canDrvObj->canController[Controller].
                        canControllerConfig_PC.CntrActive == (boolean) FALSE)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_DIINT_ID, (uint8) CAN_E_UNINIT);
        returnstatus = (boolean) FALSE;
    }
    else
    {
        returnstatus = (boolean) TRUE;
        /* MISRA C Compliance */
    }
    return returnstatus;
}

#endif

/*******************************************************************************
 * Can_CheckEnableDet
 ******************************************************************************/
/*! \brief      This function will check DET for Enable API.
 *
 *  \param[in]  uint8 Controller Controller Number in the can hardware its 0-3
 *
 *  \context
 ******************************************************************************/
#if (CAN_DEV_ERROR_DETECT == STD_ON)
/* Design :CAN_DesignId_006*/
static boolean Can_CheckEnableDet(uint8 Controller,
                                  const Can_DriverObjType * canDrvObj)
{
    boolean returnstatus;
    if (Can_DrvState == CAN_UNINIT)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_ENINT_ID, (uint8) CAN_E_UNINIT);
        returnstatus = (boolean) FALSE;
    }
    else if (Controller >= (canDrvObj->canMaxControllerCount))
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_ENINT_ID,
                               (uint8) CAN_E_PARAM_CONTROLLER);
        returnstatus = (boolean) FALSE;
    }
    else if (canDrvObj->canController[Controller].
                        canControllerConfig_PC.CntrActive == (boolean) FALSE)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_ENINT_ID, (uint8) CAN_E_UNINIT);
        returnstatus = (boolean) FALSE;
    }
    else
    {
        returnstatus = (boolean) TRUE;
        /* MISRA C Compliance */
    }
    return returnstatus;
}

#endif

/*******************************************************************************
 * Can_CheckWakeupDet
 ******************************************************************************/
/*! \brief      This function will check DET for Can_CheckWakeup API.
 *
 *  \param[in]  uint8 Controller Controller Number in the can hardware its 0-3
 *
 *  \context
 ******************************************************************************/
#if (CAN_DEV_ERROR_DETECT == STD_ON)
/* Design :CAN_DesignId_006*/
static boolean Can_CheckWakeupDet(uint8 Controller,
                                  const Can_DriverObjType * canDrvObj)
{
    boolean returnstatus;
    if (Can_DrvState == CAN_UNINIT)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_CKWAKEUP_ID,
                               (uint8) CAN_E_UNINIT);
        returnstatus = (boolean) FALSE;
    }
    else if (Controller >= (canDrvObj->canMaxControllerCount))
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_CKWAKEUP_ID,
                               (uint8) CAN_E_PARAM_CONTROLLER);
        returnstatus = (boolean) FALSE;
    }
    else
    {
        returnstatus = (boolean) TRUE;
        /* MISRA C Compliance */
    }
    return returnstatus;
}

#endif

/*******************************************************************************
 *  API DEFINITIONS
 ******************************************************************************/

/*******************************************************************************
 * Can_Init
 ******************************************************************************/
/*! \brief     This function initilises the CAN controller
 *
 *              All the required initilisation for the CAN driver and hardware
 *              is performed in this function.
 *
 *  \param[in]  gCan_ConfigPtr  Pointer to post-build configuration data.
 *
 *  \context    App
 ******************************************************************************/
/*Requirements: SWS_Can_00291, SWS_Can_00021, SWS_Can_00250, SWS_Can_00053,
 * SWS_Can_00259, SWS_Can_00174, SWS_Can_00408, SWS_Can_00245,
 * SWS_Can_00407, SWS_Can_00104, SWS_Can_00026, SWS_Can_00056, SWS_Can_00255,
 *  SWS_Can_00246, SWS_Can_00238, SWS_Can_00220
 * SWS_Can_00239, SWS_Can_00240, SWS_Can_00244, SWS_Can_00419, SWS_Can_00091,
 * SWS_Can_00089, SWS_Can_00223, SWS_Can_00174, SWS_Can_00408, SWS_Can_00103,
 */
/* Design : CAN_DesignId_008 */
 FUNC(void, CAN_CODE) Can_Init(P2CONST(Can_ConfigType, AUTOMATIC,
                                    CAN_PBCFG) CfgPtr)
{
    uint8 controller_cntr;
    const Can_ConfigType *ConfigPtr = CfgPtr;
#if (STD_ON == CAN_VARIANT_PRE_COMPILE)
    if (NULL_PTR == ConfigPtr)
    {
        ConfigPtr = &CAN_INIT_CONFIG_PC;
    }
#endif  /* (STD_ON == CAN_CFG_PRE_COMPILE_VARIANT) */
#if (CAN_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR == ConfigPtr)
    {
        /* Can config pointer is NOT valid */
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_INIT_ID,
                               (uint8) CAN_E_PARAM_POINTER);
    }
    else if (Can_CheckInitDet(ConfigPtr) == (boolean) FALSE)
    {
        /* Reporting Det Error */
        (void) Det_ReportError((uint16) CAN_MODULE_ID,
                               (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_INIT_ID,
                               (uint8) CAN_E_PARAM_POINTER);
    }
    else
#endif  /* (CAN_DEV_ERROR_DETECT == STD_ON) */
    {
        Can_resetDrvObj(&Can_DriverObj);
        Can_initDrvObj(&Can_DriverObj, ConfigPtr);
        for (controller_cntr = 0U;
             controller_cntr < Can_DriverObj.canMaxControllerCount;
             controller_cntr++)
        {
#if (CAN_DEV_ERROR_DETECT == STD_ON)
            if ((Can_DriverObj.canController[controller_cntr].canState!= CAN_CS_UNINIT)  ||
                (Can_DrvState != CAN_UNINIT) )   
            {
                /* Can controller is in not in UNINT state */
                (void) Det_ReportError((uint16) CAN_MODULE_ID,
                                       (uint8) CAN_INSTANCE_ID,
                                       (uint8) CAN_INIT_ID,
                                       (uint8) CAN_E_TRANSITION);
            }
            else
#endif
            {
                /* If Controller is active then only do initialization */
                if (Can_DriverObj.canController[controller_cntr].
                            canControllerConfig_PC.CntrActive ==
                                                            (boolean) TRUE)
                {
                    /* Init individual controller (may be moved to
                     * SetControllerMode(start state)) */
                    Can_hwUnitConfig(&Can_DriverObj.
                                     canController[controller_cntr],
                                     &Can_DriverObj.canMailbox[0U],
									 &Can_DriverObj.canTxMessageObj[0U],
                                     Can_DriverObj.maxMbCnt);
                    /* initialize the interrupt counter to zero */
                    Can_DriverObj.canController[controller_cntr].
                    canInterruptCounter = 0U;
                    /* change the state to stopped from the UNINIT state */
                    Can_DriverObj.canController[controller_cntr].canState =
                                                                CAN_CS_STOPPED;
                    /* Update BusOff Recovery status */
                    Can_DriverObj.canController[controller_cntr].
                                canBusOffRecoveryStatus = (boolean) FALSE;
                }
            }
        }
        Can_DrvState = CAN_READY;
    }
}

#if (CAN_VERSION_INFO_API == STD_ON)
/*******************************************************************************
 * Can_GetVersionInfo
 ******************************************************************************/
/*! \brief      This function will return the version for this module
 *
 *  \param[in]  None
 *
 *  \param[out]    Std_VersionInfoType VersionInfo - the structure that will
 *                  contain version info after fxn call.
 *
 *  \context    App
 ******************************************************************************/
/*Requirements: SWS_Can_00224, SWS_Can_00177 */
/* Design : CAN_DesignId_018 */
 FUNC(void,
     CAN_CODE)  Can_GetVersionInfo(P2VAR(Std_VersionInfoType, AUTOMATIC,
                                         CAN_APPL_DATA) Can_VersionInfo)
{
#if (CAN_DEV_ERROR_DETECT == STD_ON)
    if (Can_VersionInfo == NULL_PTR)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_VERSION_ID,
                               (uint8) CAN_E_PARAM_POINTER);
    }
    else
#endif
    {
        Can_VersionInfo->vendorID         = (uint16) CAN_VENDOR_ID;
        Can_VersionInfo->moduleID         = (uint16) CAN_MODULE_ID;
        Can_VersionInfo->sw_major_version = (uint8) CAN_SW_MAJOR_VERSION;
        Can_VersionInfo->sw_minor_version = (uint8) CAN_SW_MINOR_VERSION;
        Can_VersionInfo->sw_patch_version = (uint8) CAN_SW_PATCH_VERSION;
    }
}
#endif

/*******************************************************************************
 * Can_CheckWriteDet
 ******************************************************************************/
/*! \brief      This function will check DET for WriteAPI.
 *
 *  \param[in]  uint8 Hth - Hardware Handle
 *              const Can_PduType *PduInfo - pointer to pdu.
 *
 *  \context
 ******************************************************************************/
#if (CAN_DEV_ERROR_DETECT == STD_ON)
/* Design :CAN_DesignId_006*/
static boolean Can_CheckWriteDet(const uint32 Hth,
                                 const Can_PduType *PduInfo,
                                 const Can_DriverObjType *canDrvObj)
{
    boolean returnstatus = (boolean) FALSE;
    uint8          MsgCntrlr = 0U;
	if (Can_DrvState == CAN_UNINIT)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_WRITE_ID,
                               (uint8) CAN_E_UNINIT);
        returnstatus = (boolean) FALSE;
    }
    else if (canDrvObj->maxMbCnt <= Hth)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_WRITE_ID,
                               (uint8) CAN_E_PARAM_HANDLE);
        returnstatus = (boolean) FALSE;
    }
	else if ((PduInfo == NULL_PTR) || (PduInfo->sdu == NULL_PTR && 
				(Can_DriverObj.canMailbox[Hth].mailBoxConfig.
				CanTriggerTransmitEnable == FALSE)))
    {
        /* Can config pointer is not valid */
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_WRITE_ID,
                               (uint8) CAN_E_PARAM_POINTER);
        returnstatus = (boolean) FALSE;
    }
	else if(Can_DriverObj.canMailbox[Hth].mailBoxConfig.MBDir != 
											CAN_MAILBOX_DIRECTION_TX)
	{
		(void) Det_ReportError((uint16) CAN_MODULE_ID,
                                   (uint8) CAN_INSTANCE_ID,
                                   (uint8) CAN_WRITE_ID,
                                   (uint8) CAN_E_PARAM_HANDLE);
		returnstatus = (boolean) FALSE;
	}
    else
    {
        MsgCntrlr = canDrvObj->canMailbox[Hth].
                                    mailBoxConfig.Controller->ControllerId;
        returnstatus = (boolean) TRUE;
    }
    if (returnstatus == (boolean) TRUE)
    {
        if (returnstatus == (boolean) TRUE)
        {
            if (PduInfo->length > 64U)
            {
                (void) Det_ReportError((uint16) CAN_MODULE_ID,
                                       (uint8) CAN_INSTANCE_ID,
                                       (uint8) CAN_WRITE_ID,
                                       (uint8)CAN_E_PARAM_DATA_LENGTH);
                returnstatus = (boolean) FALSE;
            }
            else if ((PduInfo->length > 8U) &&
                    (CAN_ID_CAN_CONTROLLER_TYPE_MASK !=
                            (CAN_ID_CAN_CONTROLLER_TYPE_MASK & PduInfo->id)))
            {
                (void) Det_ReportError((uint16) CAN_MODULE_ID,
                                       (uint8) CAN_INSTANCE_ID,
                                       (uint8) CAN_WRITE_ID,
                                       (uint8)CAN_E_PARAM_DATA_LENGTH);
                returnstatus = (boolean) FALSE;
            }
            else if (canDrvObj->canController[MsgCntrlr].
                    canControllerConfig_PC.CntrActive == (boolean) FALSE)
			{
				/* Can controller is not valid */
				(void) Det_ReportError((uint16) CAN_MODULE_ID,
									   (uint8) CAN_INSTANCE_ID,
									   (uint8) CAN_WRITE_ID,
									   (uint8) CAN_E_UNINIT);
				returnstatus = (boolean) FALSE;
			}
			else
			{
				/* MISRA C Compliance */
				returnstatus = (boolean) TRUE;
			}
        }
		
        
    }
    return returnstatus;
}

#endif

/*******************************************************************************
 * Can_Write
 ******************************************************************************/
/*! \brief      This function will write the data into Can mail boxes.
 *
 *  \param[in]  uint8 Hth -  This is the Msg box number its a unique number.
 *              const Can_PduType *PduInfo - the CAN message to be transmitted.
 *
 *
 *  \context    App
 ******************************************************************************/
/*Requirements: SWS_Can_00233, SWS_Can_00059, SWS_Can_00276, SWS_Can_00011,
 * SWS_Can_00212, SWS_Can_00213, SWS_Can_00501
 * SWS_Can_00214, SWS_Can_00275, SWS_Can_00216, SWS_Can_00217, SWS_Can_00218,
 * SWS_Can_00219, SWS_Can_00505, SWS_Can_00506, SWS_Can_00486, SWS_Can_00502
 * */
/* Design : CAN_DesignId_010 */
FUNC(Std_ReturnType, CAN_CODE) Can_Write(Can_HwHandleType Hth, 
                                         const Can_PduType *PduInfo)
{
    Std_ReturnType status = E_OK;
    uint32         messageBox, controller;
    uint8          MsgCntrlr;
    Can_HwHandleType HwHandle;

#if (CAN_DEV_ERROR_DETECT == STD_ON)
    if (Can_CheckWriteDet((uint32) Hth, PduInfo, &Can_DriverObj) ==
        (boolean) FALSE)
    {
        status = E_NOT_OK;
    }
    else
#endif
    {
        HwHandle =  Can_DriverObj.canMailbox[Hth].mailBoxConfig.HwHandle;
		MsgCntrlr = Can_DriverObj.canMailbox[Hth].
									mailBoxConfig.Controller->ControllerId;
		Can_mcanCancelledMessagesReset(&Can_DriverObj.canController[MsgCntrlr],
									   &Can_DriverObj.canMailbox[0],
									   &Can_DriverObj.canTxMessageObj[0]);
		if(Can_DriverObj.canTxMessageObj[HwHandle].freeHwObjectCount == 0)
		{
			status = CAN_BUSY;
		}
		else
		{
			/* CanSM has triggered a new write after L1 timeout. Stop the BusOff
			 * recovery.
			 * CanSM will trigger a new bus off recovery sequence. */
			Can_DriverObj.canController[MsgCntrlr].canBusOffRecoveryStatus =
															   (boolean) FALSE;
			SchM_Enter_Can_CAN_EXCLUSIVE_AREA_0();

			if (MsgCntrlr < CAN_NUM_CONTROLLER)
			{               
				
				messageBox = Can_DriverObj.canTxMessageObj[HwHandle].buffIdx -
					Can_DriverObj.canTxMessageObj[HwHandle].freeHwObjectCount;
				
				status = Can_writeTxMailbox(
							&Can_DriverObj.canMailbox[Hth].mailBoxConfig,
							&Can_DriverObj.canController[MsgCntrlr],
							Hth,
							messageBox,
							PduInfo);
				if(status == E_OK)
				{
					/* Decreases the Hardware Object Count. */
					Can_DriverObj.canTxMessageObj[HwHandle].freeHwObjectCount--;
				}
			}
			else
			{
				status = E_NOT_OK;
			}
			SchM_Exit_Can_CAN_EXCLUSIVE_AREA_0();
        }
    }
    return status;
}

/*******************************************************************************
 * Can_DisableControllerInterrupts
 ******************************************************************************/
/*! \brief      This function will Disable the interrupts for the Controller
 * desired.
 *              Basically it will Disable the CAN interrupts(reset bit 1 in CTL
 * reg(IE0=0)).
 *
 *
 *  \param[in]  uint8 Controller Controller Number in the can hardware its 0-3
 *
 *  \context
 ******************************************************************************/
/*Requirements: SWS_Can_00231, SWS_Can_00049, SWS_Can_00205, SWS_Can_00206,
 *SWS_Can_00204*/
/* Design : CAN_DesignId_012 */
 FUNC(void, CAN_CODE)  Can_DisableControllerInterrupts(uint8 Controller)
{

#if (CAN_DEV_ERROR_DETECT == STD_ON)
    if (Can_CheckDisableDet(Controller, &Can_DriverObj) == (boolean) FALSE)
    {
        /* Det Error is already reported */
    }
    else
#endif
    {
        SchM_Enter_Can_CAN_EXCLUSIVE_AREA_0();
#if (CAN_DEV_ERROR_DETECT == STD_ON)

        if ((Can_DriverObj.canController[Controller].canControllerConfig_PC.
                                                RxProcessingType == CAN_TX_RX_PROCESSING_POLLING)
            && (Can_DriverObj.canController[Controller].canControllerConfig_PC.
                                                TxProcessingType == CAN_TX_RX_PROCESSING_POLLING))
        {
            (void) Det_ReportError((uint16) CAN_MODULE_ID,
                                   (uint8) CAN_INSTANCE_ID,
                                   (uint8) CAN_DIINT_ID,
                                   (uint8) CAN_E_PARAM_CONTROLLER);
        }
        else
#endif
        {
            Can_hwUnitDisableInterrupts(
                                &Can_DriverObj.canController[Controller]);
        }
        SchM_Exit_Can_CAN_EXCLUSIVE_AREA_0();
    }
}

/*******************************************************************************
 * Can_EnableControllerInterrupts
 ******************************************************************************/
/*! \brief      This function will Enable the interrupts for the Controller
 *              desired. It will restore the interrupts
 *              that were defined in the configuration. Basically it will enable
 *              the CAN interrupts.
 *
 *  \param[in]  uint8 Controller Controller Number in the can hardware its 0-3
 *
 *  \context
 ******************************************************************************/
/*Requirements: SWS_Can_00232, SWS_Can_00050, SWS_Can_00208, SWS_Can_00209,
 *SWS_Can_00210, SWS_Can_00033 */
/* Design : CAN_DesignId_013 */
FUNC(void, CAN_CODE)  Can_EnableControllerInterrupts(uint8 Controller)
{
#if (CAN_DEV_ERROR_DETECT == STD_ON)
    if (Can_CheckEnableDet(Controller, &Can_DriverObj) == (boolean) FALSE)
    {
        /* Det Error is already reported */
    }
    else
#endif
    {
        SchM_Enter_Can_CAN_EXCLUSIVE_AREA_0();
        Can_hwUnitEnableInterrupts(&Can_DriverObj.canController[Controller]);
        SchM_Exit_Can_CAN_EXCLUSIVE_AREA_0();
    }
}

/*******************************************************************************
 * Can_MainFunction_Write
 ******************************************************************************/
/*! \brief      This function performs the polling of TX confirmation and TX
 * cancellation confirmation
 *              when CAN_TX_PROCESSING is set to POLLING.
 *
 *  \param[in]  void
 *
 *  \context
 ******************************************************************************/
/*Requirements: SWS_Can_00058, SWS_Can_00016, SWS_Can_00225, SWS_Can_00031,
 *SWS_Can_00178,SWS_Can_00179,SWS_Can_00441 */
/* Design : CAN_DesignId_014 */
FUNC(void, CAN_CODE)  Can_MainFunction_Write(void)
{
#if  (CAN_TX_POLLING == STD_ON)
    uint8  htrh;
    uint8  Controller2;
	uint8  ctlrIndx;
	Can_TxRxProcessingType txProcessingType;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_0();
	for (ctlrIndx = 0U ;
		 ctlrIndx < (uint8) CAN_NUM_CONTROLLER;
		 ctlrIndx++)
	{
		txProcessingType = Can_DriverObj.canController[ctlrIndx].
						canControllerConfig_PC.TxProcessingType;
		/* only for the Tx Mailbox */
		if (txProcessingType != CAN_TX_RX_PROCESSING_INTERRUPT)
		{
			if (Can_DriverObj.canController[ctlrIndx].
							canControllerConfig_PC.CntrActive == (boolean) TRUE)
			{
				Can_mcanProcessTx(&Can_DriverObj.canController[ctlrIndx],
									&Can_DriverObj.canMailbox[0U],
									&Can_DriverObj.canTxMessageObj[0],
									POLLING_MASK);
			}
		}
	}
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_0();
	
#endif
}

/*******************************************************************************
 * Can_CheckWakeup
 ******************************************************************************/
/*! \brief      This function will check the wakeup status for that controller.
 *
 *  \param[in]  uint8 Controller -  The controller for which the wake status
 * need to be checked
 *
 *  \return      Can_ReturnType  E_OK      - if wakeup is detected.
 *                               E_NOT_OK  - if Wakeup is not detected
 *
 *  \context
 ******************************************************************************/
/*Requirements: SWS_Can_00360, SWS_Can_00361, SWS_Can_00362, SWS_Can_00363,
 *SWS_Can_00485 */
/* Design : CAN_DesignId_011 */
#if (CAN_WAKEUP_FUNCTIONALITY_API == STD_ON)
FUNC(Can_ReturnType, CAN_CODE) Can_CheckWakeup(uint8 Controller)
{
    uint32         baseAddr;
    Can_ReturnType status = E_NOT_OK;

#if (CAN_DEV_ERROR_DETECT == STD_ON)
    if (Can_CheckWakeupDet(Controller, &Can_DriverObj) == (boolean) FALSE)
    {
        /* Det Error is already reported */
    }
    else
#endif
    {
        baseAddr = Can_DriverObj.canController[Controller].
                                            canControllerConfig_PC.CntrAddr;
        status = Can_hwUnitCheckWakeup(
                            baseAddr);
    }
    return status;
}
#endif

/*******************************************************************************
 * Can_SetBaudrateDet
 ******************************************************************************/
/*! \brief      This function will check DET for Can_ChangeBaudrate API.
 *
 *  \param[in]  uint8 Controller Controller Number in the can hardware its 0-3
 *
 *  \context
 ******************************************************************************/
/*Requirements: SWS_CAN_00492, SWS_CAN_00494 */
#if (CAN_DEV_ERROR_DETECT == STD_ON)
static boolean Can_SetBaudrateDet(uint8 Controller,
                                   const Can_DriverObjType * canDrvObj)
{
    boolean returnstatus = (boolean) TRUE;
    if (Can_DrvState == CAN_UNINIT)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID,
                               (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_SETBAUDRATE_ID,
                               (uint8) CAN_E_UNINIT);
        returnstatus = (boolean) FALSE;
    }
    else if (Controller >= (canDrvObj->canMaxControllerCount))
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID,
                               (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_SETBAUDRATE_ID,
                               (uint8) CAN_E_PARAM_CONTROLLER);
        returnstatus = (boolean) FALSE;
    }
    else if (canDrvObj->canController[Controller].canState !=
                                                CAN_CS_STOPPED)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID,
                               (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_SETBAUDRATE_ID,
                               (uint8) CAN_E_TRANSITION);
        returnstatus = (boolean) FALSE;
    }
    else
    {
        /* MISRA C Compliance */
    }
    return returnstatus;
}

#endif

/*******************************************************************************
 * Can_SetBaudrate
 ******************************************************************************/
/*! \brief      This service shall set the baudrate of the CAN controller.
 *
 *  \param[in]  uint8 Controller -  The controller for which the Baud is to be
 *              changed
 *              BaudRateConfigID -  references a baud rate configuration by ID
 *
 *  \return     Std_ReturnType -  E_OK  - Service request accepted, setting
 *                                           of (new) baud rate started
 *
 *                                E_NOT_OK  - Service request not accepted
 *
 *  \context
 ******************************************************************************/
/* Design : CAN_DesignId_033 */
/*Requirements: SWS_Can_00493,SWS_Can_00500, SWS_Can_00256,
 *SWS_Can_00062, SWS_Can_00260 , SWS_Can_00422*/
#if (CAN_SET_BAUDRATE_API == STD_ON)
FUNC(Std_ReturnType, CAN_CODE) Can_SetBaudrate(uint8 Controller,
                                               uint16 BaudRateConfigID)
{
    const Can_ControllerType_PC *canControllerCfg_PC;
    Std_ReturnType        status = E_NOT_OK;
    Can_BaudConfigType   *setBaud;

#if (CAN_DEV_ERROR_DETECT == STD_ON)
    if (Can_SetBaudrateDet(Controller, &Can_DriverObj) == (boolean) FALSE)
    {
        /* Det Error is already reported */
    }
    else
#endif
    {
        SchM_Enter_Can_CAN_EXCLUSIVE_AREA_0();
		canControllerCfg_PC = &Can_DriverObj.canController[Controller].
														canControllerConfig_PC;
        setBaud = Can_DriverObj.canController[Controller].
                    canControllerConfig.BaudRateConfigList[BaudRateConfigID];

        if ((uint32)BaudRateConfigID >
            (Can_DriverObj.canController[Controller].maxBaudConfigID))
        {
#if (CAN_DEV_ERROR_DETECT == STD_ON)
            (void) Det_ReportError((uint16) CAN_MODULE_ID,
                                   (uint8) CAN_INSTANCE_ID,
                                   (uint8) CAN_SETBAUDRATE_ID,
                                   (uint8) CAN_E_PARAM_BAUDRATE);
#endif
        }
        else
        {
            Can_mcanSetBaudrate(canControllerCfg_PC, setBaud);

            status = (Std_ReturnType) E_OK;
        }

        SchM_Exit_Can_CAN_EXCLUSIVE_AREA_0();
    }
    return status;
}
#endif

/*******************************************************************************
 * Can_MainFunction_Wakeup
 ******************************************************************************/
/*! \brief      This function performs the polling of wake-up events that are
 * configured statically as 'to be polled'.
 *
 *  \param[in]  void
 *
 *  \context
 ******************************************************************************/
/*Requirements: SWS_Can_00270, SWS_Can_00271, SWS_Can_00269, SWS_Can_00228,
 *SWS_Can_00112, SWS_Can_00185, SWS_Can_00186*/
/* Design : CAN_DesignId_017 */
FUNC(void, CAN_CODE)  Can_MainFunction_Wakeup(void)
{
#if  (CAN_WAKEUP_POLLING == STD_ON)
    uint8  controller_cntr;
	for (controller_cntr = 0U;
		 controller_cntr < Can_DriverObj.canMaxControllerCount;
		 controller_cntr++)
	{
		Can_MainFunction_WakeupProcess(
							&Can_DriverObj.canController[controller_cntr]);
	}
#endif /* #if (CAN_WAKEUP_POLLING == STD_ON)  */
}
/*******************************************************************************
 * Can_MainFunction_BusOff
 ******************************************************************************/
/*! \brief      This function performs the polling of bus-off events that are
 * configured statically as 'to be polled'.
 *
 *  \param[in]  void
 *
 *  \context
 ******************************************************************************/
/*Requirements:SWS_Can_00280, SWS_Can_00007, SWS_Can_00227, SWS_Can_00109,
 *SWS_Can_00183, SWS_Can_00184 */
/* Design : CAN_DesignId_015 */
FUNC(void, CAN_CODE)  Can_MainFunction_BusOff(void)
{
#if (CAN_BUSOFF_POLLING == STD_ON)
    uint8  controller_cntr;

	for (controller_cntr = 0U;
		 controller_cntr < Can_DriverObj.canMaxControllerCount;
		 controller_cntr++)
	{
		/* If the controller is not activated just skip its checking */
		if (Can_DriverObj.canController[controller_cntr].
					canControllerConfig_PC.CntrActive == (boolean) TRUE)
		{
			Can_MainFunction_BusOffProcess(&Can_DriverObj.canController[controller_cntr],
									&Can_DriverObj.canMailbox[0U],
									&Can_DriverObj.canTxMessageObj[0],
									Can_DriverObj.maxMbCnt);
		}
	}
#endif /* #if (CAN_BUSOFF_POLLING == STD_ON) */
}

/*******************************************************************************
 * Can_MainFunction_Mode
 ******************************************************************************/
/*! \brief      This function performs the polling of CAN controller mode
 * transitions
 *              This will indicate only the CanController SW state and cannot
 *              indicate any HW state as there is not
 *              HW state exposed from the Can Controller.
 *
 *  \param[in]  void
 *
 *  \context
 ******************************************************************************/
/*Requirements:SWS_Can_00370, SWS_Can_00373, SWS_Can_00368, SWS_Can_00369,
 * SWS_Can_00379, SWS_Can_00372 */
/* Design : CAN_DesignId_019 */
FUNC(void, CAN_CODE)  Can_MainFunction_Mode(void)
{
    uint8  controller_cntr;

	for (controller_cntr = 0U;
		 controller_cntr < Can_DriverObj.canMaxControllerCount;
		 controller_cntr++)
	{
		Can_MainFunction_ModeProcess(
			&Can_DriverObj.canController[controller_cntr]);
	}
    
}

/*******************************************************************************
 * Can_MainFunction_Read
 ******************************************************************************/
/*! \brief      This function performs the polling of CAN controller mode
 * transitions
 *
 *  \param[in]  void
 *
 *  \context
 ******************************************************************************/
/*Requirements: SWS_Can_00396, SWS_Can_00395, SWS_Can_00007, SWS_Can_00226,
 * SWS_Can_00108, SWS_Can_00180, SWS_Can_00181
 * SWS_Can_00442 , SWS_Can_00423 */
/* Design : CAN_DesignId_016 */
FUNC(void, CAN_CODE)  Can_MainFunction_Read(void)
{
#if (CAN_RX_POLLING == STD_ON)

	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_0();
	uint32 loopCnt;
	Can_TxRxProcessingType rxProcessingType;

	for(loopCnt = 0U ; loopCnt < CAN_NUM_CONTROLLER; loopCnt++)
	{
		rxProcessingType = Can_DriverObj.canController[loopCnt].
						canControllerConfig_PC.RxProcessingType;
		/* only for the Tx Mailbox */
		if (rxProcessingType != CAN_TX_RX_PROCESSING_INTERRUPT)
		{
			if (Can_DriverObj.canController[loopCnt].
						canControllerConfig_PC.CntrActive == (boolean) TRUE)
			{
				/* Read Messages stored in  buffers */
				Can_mcanReadRxBuff(&Can_DriverObj.canController[loopCnt],
									&Can_DriverObj.canMailbox[0U],
								   (uint32)POLLING_MASK);

				/* Empty FIFO messages */
				Can_mcanReadRxFIFO(&Can_DriverObj.canController[loopCnt],
									&Can_DriverObj.canMailbox[0U],
								   (uint32)MCAN_RX_FIFO_NUM_1);
			}
		}
	}
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_0();
#endif /* #if (CAN_RX_POLLING == STD_ON) */
}

#if (CAN_LOOPBACK_ENABLE == STD_ON)
/*******************************************************************************
 * Can_TestLoopBackModeEnable
 ******************************************************************************/
/*! \brief      This function enables the Loopback
 *
 *  \param[in]  uint8 controller for which loopback needs to be enabled
 *              uint8 mode 0 - Enable Digital mode. 1 - Enable Analog mode.
 *
 *  \context
 ******************************************************************************/
/* Design : CAN_DesignId_022 */
FUNC(Std_ReturnType, CAN_CODE) Can_TestLoopBackModeEnable(uint8 controller,
                                                          uint8 mode)
{
    uint32 baseAddr;
    VAR(Std_ReturnType, AUTOMATIC) retVal;

    retVal = (Std_ReturnType) E_NOT_OK;
#if (CAN_DEV_ERROR_DETECT == STD_ON)
    if (Can_DrvState == CAN_READY)
#endif
    {
        if ((controller < (Can_DriverObj.canMaxControllerCount)) &&
            (mode < 2U))
        {
            baseAddr = Can_DriverObj.canController[controller].
                                                canControllerConfig_PC.CntrAddr;
            retVal = Can_hwUnitTestLoopBackModeEnable(baseAddr,
                                    mode);
        }
    }
    return (retVal);
}

/*******************************************************************************
 * Can_TestLoopBackModeDisable
 ******************************************************************************/
/*! \brief      This function Disable the Loopback
 *
 *  \param[in]  uint8 controller for which loopback needs to be Disabled
 *              uint8 mode 0 - Disable Digital mode. 1 - Disable Analog mode.
 *
 *  \context
 ******************************************************************************/
/* Design : CAN_DesignId_023 */
FUNC(Std_ReturnType, CAN_CODE) Can_TestLoopBackModeDisable(uint8 controller,
                                                           uint8 mode)
{
    uint32 baseAddr;
    VAR(Std_ReturnType, AUTOMATIC) retVal;

    retVal = (Std_ReturnType) E_NOT_OK;
#if (CAN_DEV_ERROR_DETECT == STD_ON)
    if (Can_DrvState ==CAN_READY)
#endif
    {
        if ((controller < (Can_DriverObj.canMaxControllerCount)) &&
            (mode < 2U))
        {
            baseAddr = Can_DriverObj.canController[controller].
                                            canControllerConfig_PC.CntrAddr;
            retVal = Can_hwUnitTestLoopBackModeDisable(baseAddr,
                                    mode);
        }
    }
    return (retVal);
}
#endif

/* Design : CAN_DesignId_034 */
/**
 * \brief API used only by UT.Not to be used by application
 */
#if (CAN_DEINIT_API == STD_ON)
/* [SWS_Can_91002],[SWS_Can_ 91009],[SWS_Can_91010],[SWS_Can_91011] */
FUNC(void, CAN_CODE) Can_DeInit(void)
{
    uint8 controllerIdx;
	uint8 tmpStatus = (uint8)E_OK;
#if (STD_ON == CAN_DEV_ERROR_DETECT)
    if (Can_DrvState != CAN_READY)
    {
        (void) Det_ReportError((uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                               (uint8) CAN_DEINIT_ID, (uint8)CAN_E_TRANSITION);
    }
    else
#endif  /* #if (STD_ON == CAN_DEV_ERROR_DETECT) */
    {
		/* Check if any of the CAN controllers is in state STARTED */
        for (controllerIdx = 0U;
             controllerIdx < Can_DriverObj.canMaxControllerCount;
             controllerIdx++)
        {
            if (Can_DriverObj.canController[controllerIdx].canState ==
                    CAN_CS_STARTED)
                {
                    tmpStatus = (uint8) E_NOT_OK;
                    break;
                }
        }
		
		if (tmpStatus == (uint8) E_OK)
        {
            for (controllerIdx = 0U;
                 controllerIdx < Can_DriverObj.canMaxControllerCount;
                 controllerIdx++)
            {
                Can_DriverObj.canController[controllerIdx].canState = 
																CAN_CS_UNINIT;
				Can_hwDeInit(&Can_DriverObj.canController[controllerIdx]);
            }
			Can_DrvState = CAN_UNINIT;
            Can_resetDrvObj(&Can_DriverObj);
        }
        else
        {
#if (STD_ON == CAN_DEV_ERROR_DETECT)
            (void) Det_ReportError(
                (uint16) CAN_MODULE_ID, (uint8) CAN_INSTANCE_ID,
                    (uint8) CAN_DEINIT_ID, (uint8) CAN_E_TRANSITION);
#endif  /* #if (STD_ON == CAN_DEV_ERROR_DETECT) */
        }
    }
}
#endif /* (CAN_DEINIT_API == STD_ON) */

 #define CAN_STOP_SEC_CODE
#include "Can_MemMap.h"
/*******************************************************************************
 * INTERRUPT SERVICE ROUTINES
 ******************************************************************************/

#define CAN_START_SEC_ISR_CODE
#include "Can_MemMap.h"
#if  defined(CAN_CONTROLLER_MCAN0)
/*******************************************************************************
 * Can_0_Int0ISR_Fun
 ******************************************************************************/
/*! \brief      This is the ISR function for the MCAN controller 1. All the
 * 				Interrupts are hooked to MCAN1INT.
 *              This ISR sends the TX confirmation for the successful tx msg.
 *              This ISR will call the RX processing function for the received
 *              message.
 *              This ISR will call the Error processing function in case of any
 *              error interrupts.
 *
 *  \param[in]  void
 *
 *  \context    ISR
 ******************************************************************************/
/* Design : CAN_DesignId_026 */
FUNC(void, CAN_CODE) Can_0_Int0ISR_Fun(void)
{
    uint32 ctrlId =
            Can_DriverObj.controllerIDMap[CAN_CONTROLLER_INSTANCE_MCAN0];

    Can_mcanProcessISR(&Can_DriverObj.canController[ctrlId],
                       &Can_DriverObj.canMailbox[0],
					   &Can_DriverObj.canTxMessageObj[0],
                       Can_DriverObj.maxMbCnt);
}

/*******************************************************************************
 * Can_0_Int1ISR_Fun
 ******************************************************************************/
/*! \brief      This is the ISR function for the MCAN controller 1 ECC Error.
 *              This ISR will clear and report the ECC Error through call back
 *              function.
 *
 *  \param[in]  void
 *
 *  \context    ISR
 ******************************************************************************/
/*Requirements : SWS_Can_00033*/
/* Design : CAN_DesignId_026 */
FUNC(void, CAN_CODE) Can_0_Int1ISR_Fun(void)
{
#if (STD_ON == CAN_ECC_ENABLE)
    uint32 ctrlId =
            Can_DriverObj.controllerIDMap[CAN_CONTROLLER_INSTANCE_MCAN0];
        Can_mcanProcessECCISR(&Can_DriverObj.canController[ctrlId]);
#endif

}
#endif

#if defined (CAN_CONTROLLER_MCAN1)
/*******************************************************************************
 * Can_1_Int0ISR_Fun
 ******************************************************************************/
/*! \brief      This is the ISR function for the MCAN controller 2. All the
 * Interrupts are hooked to MCAN2INT.
 *              This ISR sends the TX confirmation for the successful tx msg.
 *              This ISR will call the RX processing function for the received
 *              message.
 *              This ISR will call the Error processing function in case of any
 *              error interrupts.
 *
 *  \param[in]  void
 *
 *  \context    ISR
 ******************************************************************************/
/* Design : CAN_DesignId_026 */
FUNC(void, CAN_CODE) Can_1_Int0ISR_Fun(void)
{
    uint32 ctrlId =
            Can_DriverObj.controllerIDMap[CAN_CONTROLLER_INSTANCE_MCAN1];
    Can_mcanProcessISR(&Can_DriverObj.canController[ctrlId],
                           &Can_DriverObj.canMailbox[0],
						   &Can_DriverObj.canTxMessageObj[0],
                           Can_DriverObj.maxMbCnt);
}


/*******************************************************************************
 * Can_1_Int1ISR_Fun
 ******************************************************************************/
/*! \brief      This is the ISR function for the MCAN controller 2 ECC Error.
 *              This ISR will clear and report the ECC Error through call back
 *              function.
 *
 *  \param[in]  void
 *
 *  \context    ISR
 ******************************************************************************/
 /*Requirements : SWS_Can_00033*/
 /*Requirements : SWS_Can_00033*/
/* Design : CAN_DesignId_026 */
FUNC(void, CAN_CODE) Can_1_Int1ISR_Fun(void)
{
#if (STD_ON == CAN_ECC_ENABLE)
    uint32 ctrlId =
            Can_DriverObj.controllerIDMap[CAN_CONTROLLER_INSTANCE_MCAN1];
        Can_mcanProcessECCISR(&Can_DriverObj.canController[ctrlId]);
#endif
}
#endif

#if  defined(CAN_CONTROLLER_MCAN2)
/*******************************************************************************
 * Can_2_Int0ISR_Fun
 ******************************************************************************/
/*! \brief      This is the ISR function for the MCAN controller 3. All the
 * Interrupts are hooked to MCAN3INT.
 *              This ISR sends the TX confirmation for the successful tx msg.
 *              This ISR will call the RX processing function for the received
 *              message.
 *              This ISR will call the Error processing function in case of any
 *              error interrupts.
 *
 *  \param[in]  void
 *
 *  \context    ISR
 ******************************************************************************/
/* Design : CAN_DesignId_026 */
FUNC(void, CAN_CODE) Can_2_Int0ISR_Fun(void)
{
    uint32 ctrlId =
            Can_DriverObj.controllerIDMap[CAN_CONTROLLER_INSTANCE_MCAN2];

    Can_mcanProcessISR(&Can_DriverObj.canController[ctrlId],
                       &Can_DriverObj.canMailbox[0],
					   &Can_DriverObj.canTxMessageObj[0],
                       Can_DriverObj.maxMbCnt);
}

/*******************************************************************************
 * Can_2_Int1ISR_Fun
 ******************************************************************************/
/*! \brief      This is the ISR function for the MCAN controller 3 ECC Error.
 *              This ISR will clear and report the ECC Error through call back
 *              function.
 *
 *  \param[in]  void
 *
 *  \context    ISR
 ******************************************************************************/
/*Requirements : SWS_Can_00033*/
/* Design : CAN_DesignId_026 */
FUNC(void, CAN_CODE) Can_2_Int1ISR_Fun(void)
{
#if (STD_ON == CAN_ECC_ENABLE)
    uint32 ctrlId =
            Can_DriverObj.controllerIDMap[CAN_CONTROLLER_INSTANCE_MCAN2];
        Can_mcanProcessECCISR(&Can_DriverObj.canController[ctrlId]);
#endif

}
#endif

#if  defined(CAN_CONTROLLER_MCAN3)
/*******************************************************************************
 * Can_3_Int0ISR_Fun
 ******************************************************************************/
/*! \brief      This is the ISR function for the MCAN controller 4. All the
 * Interrupts are hooked to MCAN4NT.
 *              This ISR sends the TX confirmation for the successful tx msg.
 *              This ISR will call the RX processing function for the received
 *              message.
 *              This ISR will call the Error processing function in case of any
 *              error interrupts.
 *
 *  \param[in]  void
 *
 *  \context    ISR
 ******************************************************************************/
/* Design : CAN_DesignId_026 */
FUNC(void, CAN_CODE) Can_3_Int0ISR_Fun(void)
{
    uint32 ctrlId =
            Can_DriverObj.controllerIDMap[CAN_CONTROLLER_INSTANCE_MCAN3];

    Can_mcanProcessISR(&Can_DriverObj.canController[ctrlId],
                       &Can_DriverObj.canMailbox[0],
					   &Can_DriverObj.canTxMessageObj[0],
                       Can_DriverObj.maxMbCnt);
}

/*******************************************************************************
 * Can_3_Int1ISR_Fun
 ******************************************************************************/
/*! \brief      This is the ISR function for the MCAN controller 4 ECC Error.
 *              This ISR will clear and report the ECC Error through call back
 *              function.
 *
 *  \param[in]  void
 *
 *  \context    ISR
 ******************************************************************************/
/*Requirements : SWS_Can_00033*/
/* Design : CAN_DesignId_026 */
FUNC(void, CAN_CODE) Can_3_Int1ISR_Fun(void)
{
#if (STD_ON == CAN_ECC_ENABLE)
    uint32 ctrlId =
            Can_DriverObj.controllerIDMap[CAN_CONTROLLER_INSTANCE_MCAN3];
        Can_mcanProcessECCISR(&Can_DriverObj.canController[ctrlId]);
#endif

}
#endif

/* [SWS_Can_91004], [SWS_Can_91005], [SWS_Can_91006], [SWS_Can_91007],
   [SWS_Can_91008] */
/*******************************************************************************
 * Can_GetControllerErrorState
 ******************************************************************************/
 /**
 * \brief This function updates the error state of the controller.
 *
 * \verbatim
 * Service name: Can_GetControllerErrorState
 * Syntax      : Std_ReturnType Can_GetControllerErrorState( uint8 ControllerId,
 *               Can_ErrorStateType* ErrorStatePtr )
 * Service ID[hex]:0x11
 * Sync/Async     :Synchronous
 * Reentrancy     :Non Reentrant
 * Parameters (in):ControllerId
 *                 - ID of the CAN controller whose error state is requested.
 * Parameters (inout):None
 * Parameters (out)  :ErrorStatePtr
 *                    - Pointer where the error state is updated.
 * Return value:Std_ReturnType
 *              - E_OK: Error state request accepted
 *              - E_NOT_OK: Error state request not accepted
 * Description:This function updates the error state of the controller.
 * \endverbatim 
  *****************************************************************************/
FUNC(Std_ReturnType, CAN_CODE) Can_GetControllerErrorState(uint8 ControllerId,
  Can_ErrorStateType* ErrorStatePtr)
{
  Std_ReturnType          retVal = (Std_ReturnType)E_NOT_OK;
  Can_ErrorStateType    errorState;
#if (CAN_DEV_ERROR_DETECT == STD_ON)
  if (Can_DrvState == CAN_UNINIT)
  {
    (void)Det_ReportError((uint16)CAN_MODULE_ID,
      (uint8)CAN_INSTANCE_ID,
      (uint8)CAN_GETCTRERRST_ID,
      (uint8)CAN_E_UNINIT);
  }
  else if (ControllerId >= (Can_DriverObj.canMaxControllerCount))
  {
    (void)Det_ReportError((uint16)CAN_MODULE_ID,
      (uint8)CAN_INSTANCE_ID,
      (uint8)CAN_GETCTRERRST_ID,
      (uint8)CAN_E_PARAM_CONTROLLER);
  }
  else if (ErrorStatePtr == NULL_PTR)
  {
    (void)Det_ReportError((uint16)CAN_MODULE_ID,
      (uint8)CAN_INSTANCE_ID,
      (uint8)CAN_GETCTRERRST_ID,
      (uint8)CAN_E_PARAM_POINTER);
  }
  else
#endif
  {
    retVal = (Std_ReturnType)E_OK;
    errorState = Can_mcanGetProtocolStatus
    (&Can_DriverObj.canController[ControllerId]);
    *ErrorStatePtr = errorState;
  }

  return retVal;
}

/*******************************************************************************
 * Can_GetControllerMode
 ******************************************************************************/
 /**
 * \brief This service reports about the current status of the requested
 *        CAN controller.
 *
 * \verbatim
 * Service name: Can_GetControllerMode
 * Syntax: Std_ReturnType Can_GetControllerMode(uint8 Controller,
 *                     Can_ControllerStateType* ControllerModePtr)
 * Service ID[hex]: 0x12
 * Sync/Async: Synchronous
 * Reentrancy: Non Reentrant
 * Parameters (in): Controller
 *                  - CAN controller for which the status shall be changed
 * Parameters (out): ControllerModePtr
 *                  - Pointer to a memory location, where
 *                    the current mode of the CAN controller will be stored.
 * Parameters (inout):  None
 * Return value: Std_ReturnType
 *               - E_OK: request accepted
 *               - E_NOT_OK: request not accepted.
 * Description:  This service reports about the current status of the requested
 *               CAN controller.
 * \endverbatim 
  *****************************************************************************/

/* [SWS_Can_91014][SWS_Can_91015],[SWS_Can_91016],[SWS_Can_91017],
   [SWS_Can_91018] */

FUNC(Std_ReturnType, CAN_CODE)Can_GetControllerMode(uint8 Controller,
  Can_ControllerStateType* ControllerModePtr)
{
  Std_ReturnType    retVal = (Std_ReturnType)E_NOT_OK;
#if (CAN_DEV_ERROR_DETECT == STD_ON)
  if (Can_DrvState == CAN_UNINIT)
  {
    (void)Det_ReportError((uint16)CAN_MODULE_ID,
      (uint8)CAN_INSTANCE_ID,
      (uint8)CAN_GETCTRMODE_ID,
      (uint8)CAN_E_UNINIT);
  }
  else if (Controller >= (Can_DriverObj.canMaxControllerCount))
  {
    (void)Det_ReportError((uint16)CAN_MODULE_ID,
      (uint8)CAN_INSTANCE_ID,
      (uint8)CAN_GETCTRMODE_ID,
      (uint8)CAN_E_PARAM_CONTROLLER);
  }
  else if (ControllerModePtr == NULL_PTR)
  {
    (void)Det_ReportError((uint16)CAN_MODULE_ID,
      (uint8)CAN_INSTANCE_ID,
      (uint8)CAN_GETCTRMODE_ID,
      (uint8)CAN_E_PARAM_POINTER);
  }
  else
#endif
  {
    retVal = (Std_ReturnType)E_OK;
    *ControllerModePtr = Can_DriverObj.canController[Controller].canState;
  }

  return retVal;
}

#define CAN_STOP_SEC_ISR_CODE
#include "Can_MemMap.h"

/*******************************************************************************
 *  End of File: Can.c
 ******************************************************************************/
