/* ======================================================================
*   Copyright (C) 2022-2023 Texas Instruments Incorporated
*
*   All rights reserved. Property of Texas Instruments Incorporated.
*   Restricted rights to use, duplicate or disclose this code are
*   granted through contract.
*
*   The program may not be used without the written permission
*   of Texas Instruments Incorporated or against the terms and conditions
*   stipulated in the agreement under which this program has been
*   supplied.
* ==================================================================== */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "mcan.h"
#define CAN_START_SEC_CODE
#include "Can_MemMap.h"
#include "hw_types.h"
#define CAN_STOP_SEC_CODE
#include "Can_MemMap.h"
/* Includes for MCAL */
#include "string.h"
#include "Can_Priv.h"
#include "EcuM_Cbk.h"
#include "Can_Cfg.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/**
 * \brief  Macro for standard Message ID filter.
 */
#define MCANSS_STD_ID_FILTER_SIZE_WORDS                          (1U)

/**
 * \brief  Macro for extended Message ID filter.
 */
#define MCANSS_EXT_ID_FILTER_SIZE_WORDS                          (2U)

/**
 * \brief  Macro for Interrupt Line enable mask.
 */
#define MCANSS_INTR_LINE_EN_MASK   ((MCAN_ILE_EINT0_MASK | MCAN_ILE_EINT1_MASK))

/**
 * \brief  Mask and shift for Tx Buffers elements.
 */
#define MCANSS_TX_BUFFER_ELEM_ID_SHIFT                           (0U)
#define MCANSS_TX_BUFFER_ELEM_ID_MASK                            (0x1FFFFFFFU)
#define MCANSS_TX_BUFFER_ELEM_RTR_SHIFT                          (29U)
#define MCANSS_TX_BUFFER_ELEM_RTR_MASK                           (0x20000000U)
#define MCANSS_TX_BUFFER_ELEM_XTD_SHIFT                          (30U)
#define MCANSS_TX_BUFFER_ELEM_XTD_MASK                           (0x40000000U)
#define MCANSS_TX_BUFFER_ELEM_ESI_SHIFT                          (31U)
#define MCANSS_TX_BUFFER_ELEM_ESI_MASK                           (0x80000000U)
#define MCANSS_TX_BUFFER_ELEM_DLC_SHIFT                          (16U)
#define MCANSS_TX_BUFFER_ELEM_DLC_MASK                           (0x000F0000U)
#define MCANSS_TX_BUFFER_ELEM_BRS_SHIFT                          (20U)
#define MCANSS_TX_BUFFER_ELEM_BRS_MASK                           (0x00100000U)
#define MCANSS_TX_BUFFER_ELEM_FDF_SHIFT                          (21U)
#define MCANSS_TX_BUFFER_ELEM_FDF_MASK                           (0x00200000U)
#define MCANSS_TX_BUFFER_ELEM_EFC_SHIFT                          (23U)
#define MCANSS_TX_BUFFER_ELEM_EFC_MASK                           (0x00800000U)
#define MCANSS_TX_BUFFER_ELEM_MM_SHIFT                           (24U)
#define MCANSS_TX_BUFFER_ELEM_MM_MASK                            (0xFF000000U)

/**
 * \brief  Mask and shift for Rx Buffers elements.
 */
#define MCANSS_RX_BUFFER_ELEM_ID_SHIFT                           (0U)
#define MCANSS_RX_BUFFER_ELEM_ID_MASK                            (0x1FFFFFFFU)
#define MCANSS_RX_BUFFER_ELEM_RTR_SHIFT                          (29U)
#define MCANSS_RX_BUFFER_ELEM_RTR_MASK                           (0x20000000U)
#define MCANSS_RX_BUFFER_ELEM_XTD_SHIFT                          (30U)
#define MCANSS_RX_BUFFER_ELEM_XTD_MASK                           (0x40000000U)
#define MCANSS_RX_BUFFER_ELEM_ESI_SHIFT                          (31U)
#define MCANSS_RX_BUFFER_ELEM_ESI_MASK                           (0x80000000U)
#define MCANSS_RX_BUFFER_ELEM_RXTS_SHIFT                         (0U)
#define MCANSS_RX_BUFFER_ELEM_RXTS_MASK                          (0x0000FFFFU)
#define MCANSS_RX_BUFFER_ELEM_DLC_SHIFT                          (16U)
#define MCANSS_RX_BUFFER_ELEM_DLC_MASK                           (0x000F0000U)
#define MCANSS_RX_BUFFER_ELEM_BRS_SHIFT                          (20U)
#define MCANSS_RX_BUFFER_ELEM_BRS_MASK                           (0x00100000U)
#define MCANSS_RX_BUFFER_ELEM_FDF_SHIFT                          (21U)
#define MCANSS_RX_BUFFER_ELEM_FDF_MASK                           (0x00200000U)
#define MCANSS_RX_BUFFER_ELEM_FIDX_SHIFT                         (24U)
#define MCANSS_RX_BUFFER_ELEM_FIDX_MASK                          (0x7F000000U)
#define MCANSS_RX_BUFFER_ELEM_ANMF_SHIFT                         (31U)
#define MCANSS_RX_BUFFER_ELEM_ANMF_MASK                          (0x80000000U)

/**
 * \brief  Mask and shift for Standard Message ID Filter Elements.
 */
#define MCANSS_STD_ID_FILTER_SFID2_SHIFT                         (0U)
#define MCANSS_STD_ID_FILTER_SFID2_MASK                          (0x000003FFU)
#define MCANSS_STD_ID_FILTER_SFID1_SHIFT                         (16U)
#define MCANSS_STD_ID_FILTER_SFID1_MASK                          (0x03FF0000U)
#define MCANSS_STD_ID_FILTER_SFEC_SHIFT                          (27U)
#define MCANSS_STD_ID_FILTER_SFEC_MASK                           (0x38000000U)
#define MCANSS_STD_ID_FILTER_SFT_SHIFT                           (30U)
#define MCANSS_STD_ID_FILTER_SFT_MASK                            (0xC0000000U)

/**
 * \brief  Extended Message ID Filter Element.
 */
#define MCANSS_EXT_ID_FILTER_EFID2_SHIFT                        (0U)
#define MCANSS_EXT_ID_FILTER_EFID2_MASK                         (0x1FFFFFFFU)
#define MCANSS_EXT_ID_FILTER_EFID1_SHIFT                        (0U)
#define MCANSS_EXT_ID_FILTER_EFID1_MASK                         (0x1FFFFFFFU)
#define MCANSS_EXT_ID_FILTER_EFEC_SHIFT                         (29U)
#define MCANSS_EXT_ID_FILTER_EFEC_MASK                          (0xE0000000U)
#define MCANSS_EXT_ID_FILTER_EFT_SHIFT                          (30U)
#define MCANSS_EXT_ID_FILTER_EFT_MASK                           (0xC0000000U)

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

/**
 * \brief   This API will unblock write access to write protected registers.
 *
 * \param   baseAddr        Base Address of the MCAN Registers.
 *
 * \return  None.
 */
static void MCAN_writeProtectedRegAccessUnlock(uint32 baseAddr);

/**
 * \brief   This API will block write access to write protected registers.
 *
 * \param   baseAddr        Base Address of the MCAN Registers.
 *
 * \return  None.
 */
static void MCAN_writeProtectedRegAccessLock(uint32 baseAddr);

/**
 * \brief   This API will load the register from ECC memory bank.
 *
 * \param   baseAddr        Base Address of the MCAN Registers.
 * \param   regOffset       Offset of the register to read.
 *
 * \return  None.
 */
static void MCAN_eccLoadRegister(uint32 baseAddr, uint32 regOffSet);

/**
 * \brief   This API will read the message object from Message RAM.
 *
 * \param   baseAddr        Base Address of the MCAN Registers.
 * \param   elemAddr        Address of the message object.
 * \param   elem            Message Object.
 *                          Refer struct #MCAN_RxBufElement.
 *
 * \return  None.
 */
static void MCAN_readMsg(uint32           baseAddr,
                         uint32           elemAddr,
                         volatile MCAN_RxBufElement *elem);

/**
 * \brief   This API will write the message object to Message RAM.
 *
 * \param   baseAddr        Base Address of the MCAN Registers.
 * \param   elemAddr        Address of the message object.
 * \param   elem            Message Object.
 *                          Refer struct #MCAN_TxBufElement.
 *
 * \return  None.
 */
static void MCAN_writeMsg(uint32                 baseAddr,
                          uint32                 elemAddr,
                          const MCAN_TxBufElement *elem);

/**
 * \brief   This API will return payload depending on 'data_length'  field.
 *
 * \param   data_length             Data Length Code.
 *
 * \return  data size       Size of the payload.
 */
static uint32 MCAN_getDataSize(uint32 data_length);

/**
 * \brief   This API will return message object size.
 *
 * \param   data_length             Data Length Code.
 *
 * \return  message object size
 *                          Size of the message object stored in Message RAM.
 */
static uint32 MCAN_getMsgObjSize(uint32 elemSize);

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

#define CAN_START_SEC_VAR_INIT_32
#include "Can_MemMap.h"

/* payload depending on 'dlc'  field. */
static const uint32 Can_DataSize[16]  = {0,  1,  2,  3,  4,  5,  6, 7, 8,
                                        12, 16, 20, 24, 32, 48, 64};
/* message object stored in Message RAM. */
static const uint32 Can_MsgObjSize[8] = {4, 5, 6, 7, 8, 10, 14, 18};

#define CAN_STOP_SEC_VAR_INIT_32
#include "Can_MemMap.h"
/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */
/* Memory mapping, done via MemMap.h: Code */
#define CAN_START_SEC_CODE
#include "Can_MemMap.h"
void MCAN_reset(uint32 baseAddr)
{

    HW_WR_FIELD32(baseAddr + MCAN_MCANSS_CTRL,
                  MCAN_MCANSS_CTRL_RESET,
                  0x1U);

}

uint32 MCAN_isInReset(uint32 baseAddr)
{
    return((uint32)HW_RD_FIELD32(baseAddr + MCAN_MCANSS_STAT,
                          MCAN_MCANSS_STAT_RESET));
}

uint32 MCAN_isMemInitDone(uint32 baseAddr)
{
  uint32 memInit;
  uint32 state;
  memInit = HW_RD_FIELD32((baseAddr)+MCAN_MCANSS_STAT,
    MCAN_MCANSS_STAT_MEM_INIT_DONE);
  if (1U == memInit)
  {
    state = (uint32)TRUE;
  }
  else
  {
    state = (uint32)FALSE;
  }
  return state;
}

void MCAN_setOpMode(uint32 baseAddr, uint32 mode)
{
    HW_WR_FIELD32(baseAddr + MCAN_CCCR, MCAN_CCCR_INIT, mode);
}

uint32 MCAN_getOpMode(uint32 baseAddr)
{
    return (HW_RD_FIELD32(baseAddr + MCAN_CCCR, MCAN_CCCR_INIT));
}

int32_t MCAN_init(uint32 baseAddr, const MCAN_InitParams *initParams)
{
    int32_t  status;
    uint32 regVal;
    /* Configure MCAN wakeup and clock stop controls */
    regVal = HW_RD_REG32(baseAddr + MCAN_MCANSS_CTRL);
    HW_SET_FIELD32(regVal,
                   MCAN_MCANSS_CTRL_WAKEUPREQEN,
                   initParams->wkupReqEnable);
    HW_SET_FIELD32(regVal,
                   MCAN_MCANSS_CTRL_AUTOWAKEUP,
                   initParams->autoWkupEnable);

    HW_SET_FIELD32(regVal,
                   MCAN_MCANSS_CTRL_EMUEN,
                   initParams->emulationEnable);
    HW_SET_FIELD32(regVal,
                   MCAN_MCANSS_CTRL_EMUFACK,
                   initParams->emulationFAck);
    HW_SET_FIELD32(regVal,
                   MCAN_MCANSS_CTRL_CLKFACK,
                   initParams->clkStopFAck);

    HW_WR_REG32(baseAddr + MCAN_MCANSS_CTRL, regVal);

    MCAN_writeProtectedRegAccessUnlock(baseAddr);

    /* Configure MCAN mode(FD vs Classic CAN operation) and controls */
    regVal = HW_RD_REG32(baseAddr + MCAN_CCCR);
    HW_SET_FIELD32(regVal,
                   MCAN_CCCR_FDOE,
                   initParams->fdMode);
    HW_SET_FIELD32(regVal,
                   MCAN_CCCR_BRSE,
                   initParams->brsEnable);
    HW_SET_FIELD32(regVal,
                   MCAN_CCCR_TXP,
                   initParams->txpEnable);
    HW_SET_FIELD32(regVal,
                   MCAN_CCCR_EFBI,
                   initParams->efbi);
    HW_SET_FIELD32(regVal,
                   MCAN_CCCR_PXHD,
                   initParams->pxhddisable);
    HW_SET_FIELD32(regVal,
                   MCAN_CCCR_DAR,
                   initParams->darEnable);
    HW_WR_REG32(baseAddr + MCAN_CCCR, regVal);

    if (MCAN_RWD_WDC_MAX >= initParams->wdcPreload)
    {
        /* Configure MSG RAM watchdog counter preload value */
        HW_WR_FIELD32(baseAddr + MCAN_RWD,
                      MCAN_RWD_WDC,
                      initParams->wdcPreload);
        status = STW_SOK;
    }
    else
    {
        status = STW_EFAIL;
    }

    MCAN_writeProtectedRegAccessLock(baseAddr);

    return status;
}

int32_t MCAN_config(uint32 baseAddr, const MCAN_ConfigParams *configParams)
{
    int32_t status;

    MCAN_writeProtectedRegAccessUnlock(baseAddr);

    /* Configure MCAN control registers */
    HW_WR_FIELD32(baseAddr + MCAN_CCCR,
                  MCAN_CCCR_MON,
                  configParams->monEnable);
    HW_WR_FIELD32(baseAddr + MCAN_CCCR,
                  MCAN_CCCR_ASM,
                  configParams->asmEnable);
    /* Configure Global Filter */
    HW_WR_FIELD32(baseAddr + MCAN_GFC,
                  MCAN_GFC_RRFE,
                  configParams->filterConfig.rrfe);
    HW_WR_FIELD32(baseAddr + MCAN_GFC,
                  MCAN_GFC_RRFS,
                  configParams->filterConfig.rrfs);
    HW_WR_FIELD32(baseAddr + MCAN_GFC,
                  MCAN_GFC_ANFE,
                  configParams->filterConfig.anfe);
    HW_WR_FIELD32(baseAddr + MCAN_GFC,
                  MCAN_GFC_ANFS,
                  configParams->filterConfig.anfs);

    if ((MCAN_TSCC_TCP_MAX >= configParams->tsPrescalar) &&
        (MCAN_TOCC_TOP_MAX >= configParams->timeoutPreload))
    {
        /* Configure Time-stamp counter */
        HW_WR_FIELD32(baseAddr + MCAN_TSCC,
                      MCAN_TSCC_TSS,
                      configParams->tsSelect);
        HW_WR_FIELD32(baseAddr + MCAN_TSCC,
                      MCAN_TSCC_TCP,
                      (configParams->tsPrescalar - 1U));
        /* Configure Time-out counter */
        HW_WR_FIELD32(baseAddr + MCAN_TOCC,
                      MCAN_TOCC_TOS,
                      configParams->timeoutSelect);
        HW_WR_FIELD32(baseAddr + MCAN_TOCC,
                      MCAN_TOCC_TOP,
                      configParams->timeoutPreload);
        /* Enable Time-out counter */
        HW_WR_FIELD32(baseAddr + MCAN_TOCC,
                      MCAN_TOCC_ETOC,
                      configParams->timeoutCntEnable);
        status = STW_SOK;
    }
    else
    {
        status = STW_EFAIL;
    }

    MCAN_writeProtectedRegAccessLock(baseAddr);

    return status;
}

void MCAN_eccConfig(uint32                    baseAddr,
                    const MCAN_ECCConfigParams *configParams)
{
  uint32 regVal;

    baseAddr = (baseAddr&(~MCAN_RAM_OFFSET))>>4;
    regVal = HW_RD_REG32(baseAddr + MCAN_ECC_AGGR_CONTROL);
    HW_SET_FIELD32(regVal,
                   MCAN_ECC_AGGR_CONTROL_ECC_CHECK,
                   configParams->enableChk);
    HW_SET_FIELD32(regVal,
                   MCAN_ECC_AGGR_CONTROL_ECC_ENABLE,
                   configParams->enable);
    HW_SET_FIELD32(regVal,
                   MCAN_ECC_AGGR_CONTROL_ENABLE_RMW,
                   configParams->enableRdModWr);
    HW_WR_REG32(baseAddr + MCAN_ECC_AGGR_CONTROL, regVal);

}

int32_t MCAN_setBitTime(uint32                    baseAddr,
                        const MCAN_BitTimingParams *configParams)
{
    int32_t status;
	uint32 regVal;

    MCAN_writeProtectedRegAccessUnlock(baseAddr);
	
	regVal = HW_RD_REG32(baseAddr + MCAN_CCCR);
    HW_SET_FIELD32(regVal, MCAN_CCCR_BRSE, configParams->brsEnable);
    HW_WR_REG32(baseAddr + MCAN_CCCR, regVal);
	
    if ((MCAN_NBTP_NSJW_MAX >= configParams->nomSynchJumpWidth) &&
        (MCAN_NBTP_NTSEG2_MAX >= configParams->nomTimeSeg2) &&
        (MCAN_NBTP_NTSEG1_MAX >= configParams->nomTimeSeg1) &&
        (MCAN_NBTP_NBRP_MAX >= configParams->nomRatePrescalar))
    {
        HW_WR_FIELD32(baseAddr + MCAN_NBTP,
                      MCAN_NBTP_NSJW,
                      configParams->nomSynchJumpWidth);
        HW_WR_FIELD32(baseAddr + MCAN_NBTP,
                      MCAN_NBTP_NTSEG2,
                      configParams->nomTimeSeg2);
        HW_WR_FIELD32(baseAddr + MCAN_NBTP,
                      MCAN_NBTP_NTSEG1,
                      configParams->nomTimeSeg1);
        HW_WR_FIELD32(baseAddr + MCAN_NBTP,
                      MCAN_NBTP_NBRP,
                      configParams->nomRatePrescalar);
        status = STW_SOK;
    }
    else
    {
        status = STW_EFAIL;
    }
    if (STW_SOK == status)
    {
        if ((MCAN_DBTP_DSJW_MAX >= configParams->dataSynchJumpWidth) &&
            (MCAN_DBTP_DTSEG2_MAX >= configParams->dataTimeSeg2) &&
            (MCAN_DBTP_DTSEG1_MAX >= configParams->dataTimeSeg1) &&
            (MCAN_DBTP_DBRP_MAX >= configParams->dataRatePrescalar))
        {
            HW_WR_FIELD32(baseAddr + MCAN_DBTP,
                          MCAN_DBTP_DSJW,
                          configParams->dataSynchJumpWidth);
            HW_WR_FIELD32(baseAddr + MCAN_DBTP,
                          MCAN_DBTP_DTSEG2,
                          configParams->dataTimeSeg2);
            HW_WR_FIELD32(baseAddr + MCAN_DBTP,
                          MCAN_DBTP_DTSEG1,
                          configParams->dataTimeSeg1);
            HW_WR_FIELD32(baseAddr + MCAN_DBTP,
                          MCAN_DBTP_DBRP,
                          configParams->dataRatePrescalar);
            status = STW_SOK;
        }
        else
        {
            status = STW_EFAIL;
        }
    }
	if (STW_SOK == status)
    {
		if ((MCAN_TDCR_TDCF_MAX >= configParams->tdcConfig.tdcf) &&
			(MCAN_TDCR_TDCO_MAX >= configParams->tdcConfig.tdco))
		{
			/* Configure Transceiver Delay Compensation */
			HW_WR_FIELD32(baseAddr + MCAN_TDCR,
						  MCAN_TDCR_TDCF,
						  configParams->tdcConfig.tdcf);
			HW_WR_FIELD32(baseAddr + MCAN_TDCR,
						  MCAN_TDCR_TDCO,
						  configParams->tdcConfig.tdco);
			/* Enable/Disable Transceiver Delay Compensation */
			HW_WR_FIELD32(baseAddr + MCAN_DBTP,
						  MCAN_DBTP_TDC,
						  configParams->tdcEnable);
			status = STW_SOK;
		}
        else
        {
            status = STW_EFAIL;
        }
    }

    MCAN_writeProtectedRegAccessLock(baseAddr);
    return status;
}

int32_t MCAN_msgRAMConfig(uint32                       baseAddr,
                          const MCAN_MsgRAMConfigParams *msgRAMConfigParams)
{
    int32_t  status;
    uint32 elemNum = 0U;

    MCAN_writeProtectedRegAccessUnlock(baseAddr);

    /* Configure Message Filters section */
    if (0U != msgRAMConfigParams->lss)
    {
        HW_WR_FIELD32(baseAddr + MCAN_SIDFC,
                      MCAN_SIDFC_FLSSA,
                      (msgRAMConfigParams->flssa >> 2U));
        HW_WR_FIELD32(baseAddr + MCAN_SIDFC,
                      MCAN_SIDFC_LSS,
                      msgRAMConfigParams->lss);
    }
    if (0U != msgRAMConfigParams->lse)
    {
        HW_WR_FIELD32(baseAddr + MCAN_XIDFC,
                      MCAN_XIDFC_FLESA,
                      (msgRAMConfigParams->flesa >> 2U));
        HW_WR_FIELD32(baseAddr + MCAN_XIDFC,
                      MCAN_XIDFC_LSE,
                      msgRAMConfigParams->lse);
    }
    /* Configure Rx FIFO 0 section */
    if (0U != msgRAMConfigParams->rxFIFO0size)
    {
        HW_WR_FIELD32(baseAddr + MCAN_RXF0C,
                      MCAN_RXF0C_F0SA,
                      (msgRAMConfigParams->rxFIFO0startAddr >> 2U));
        HW_WR_FIELD32(baseAddr + MCAN_RXF0C,
                      MCAN_RXF0C_F0S,
                      msgRAMConfigParams->rxFIFO0size);
        HW_WR_FIELD32(baseAddr + MCAN_RXF0C,
                      MCAN_RXF0C_F0WM,
                      msgRAMConfigParams->rxFIFO0waterMark);
        HW_WR_FIELD32(baseAddr + MCAN_RXF0C,
                      MCAN_RXF0C_F0OM,
                      msgRAMConfigParams->rxFIFO0OpMode);
        /* Configure Rx FIFO0 elements size */
        HW_WR_FIELD32(baseAddr + MCAN_RXESC,
                      MCAN_RXESC_F0DS,
                      msgRAMConfigParams->rxFIFO0ElemSize);
    }
    /* Configure Rx FIFO 1 section */
    if (0U != msgRAMConfigParams->rxFIFO1size)
    {
        HW_WR_FIELD32((baseAddr) + MCAN_RXF1C,
                      MCAN_RXF1C_F1SA,
                      (msgRAMConfigParams->rxFIFO1startAddr >> 2U));
        HW_WR_FIELD32((baseAddr) + MCAN_RXF1C,
                      MCAN_RXF1C_F1S,
                      msgRAMConfigParams->rxFIFO1size);
        HW_WR_FIELD32((baseAddr) + MCAN_RXF1C,
                      MCAN_RXF1C_F1WM,
                      msgRAMConfigParams->rxFIFO1waterMark);
        HW_WR_FIELD32((baseAddr) + MCAN_RXF1C,
                      MCAN_RXF1C_F1OM,
                      msgRAMConfigParams->rxFIFO1OpMode);
        /* Configure Rx FIFO1 elements size */
        HW_WR_FIELD32((baseAddr) + MCAN_RXESC,
                      MCAN_RXESC_F1DS,
                      msgRAMConfigParams->rxFIFO1ElemSize);
    }
    /* Configure Rx Buffer Start Address */
    HW_WR_FIELD32(baseAddr + MCAN_RXBC,
                  MCAN_RXBC_RBSA,
                  (msgRAMConfigParams->rxBufStartAddr >> 2U));
    /* Configure Rx Buffer elements size */
    HW_WR_FIELD32(baseAddr + MCAN_RXESC,
                  MCAN_RXESC_RBDS,
                  msgRAMConfigParams->rxBufElemSize);
    /* Configure Tx Event FIFO section */
    if (0U != msgRAMConfigParams->txEventFIFOSize)
    {
        HW_WR_FIELD32((baseAddr) + MCAN_TXEFC,
                      MCAN_TXEFC_EFSA,
                      (msgRAMConfigParams->txEventFIFOStartAddr >> 2U));
        HW_WR_FIELD32((baseAddr) + MCAN_TXEFC,
                      MCAN_TXEFC_EFS,
                      msgRAMConfigParams->txEventFIFOSize);
        HW_WR_FIELD32((baseAddr) + MCAN_TXEFC,
                      MCAN_TXEFC_EFWM,
                      msgRAMConfigParams->txEventFIFOWaterMark);
    }
    /* Configure Tx Buffer and FIFO/Q section */
    elemNum = msgRAMConfigParams->txBufNum + msgRAMConfigParams->txFIFOSize;
    if ((MCAN_TX_MB_MAX_NUM >= elemNum) &&
        ((0U != msgRAMConfigParams->txBufNum) ||
         (0U != msgRAMConfigParams->txFIFOSize)))
    {
        HW_WR_FIELD32(baseAddr + MCAN_TXBC,
                      MCAN_TXBC_TBSA,
                      (msgRAMConfigParams->txStartAddr >> 2U));
        HW_WR_FIELD32(baseAddr + MCAN_TXBC,
                      MCAN_TXBC_NDTB,
                      msgRAMConfigParams->txBufNum);
        HW_WR_FIELD32(baseAddr + MCAN_TXBC,
                      MCAN_TXBC_TFQS,
                      msgRAMConfigParams->txFIFOSize);
        HW_WR_FIELD32(baseAddr + MCAN_TXBC,
                      MCAN_TXBC_TFQM,
                      msgRAMConfigParams->txBufMode);
        /* Configure Tx Buffer/FIFO0/FIFO1 elements size */
        HW_WR_FIELD32(baseAddr + MCAN_TXESC,
                      MCAN_TXESC_TBDS,
                      msgRAMConfigParams->txBufElemSize);
        status = STW_SOK;
    }
    else
    {
        status = STW_EFAIL;
    }

    MCAN_writeProtectedRegAccessLock(baseAddr);

    return status;
}

uint32 MCAN_writeMsgRam(uint32                 baseAddr,
                      uint32                 memType,
                      uint32                 bufNum,
                      const MCAN_TxBufElement *elem)
{
    uint32 startAddr = 0U, elemSize = 0U, elemAddr = 0U;
    uint32 idx       = 0U;

    if (memType == (uint32)MCAN_MEM_TYPE_BUF)
    {
        idx       = bufNum;
    }
    else
    {
        idx       = HW_RD_FIELD32(baseAddr + MCAN_TXFQS, MCAN_TXFQS_TFQPI);
    }
	startAddr = HW_RD_FIELD32(baseAddr + MCAN_TXBC,
							  MCAN_TXBC_TBSA);
	elemSize = HW_RD_FIELD32(baseAddr + MCAN_TXESC,
						 MCAN_TXESC_TBDS);
    startAddr = (uint32) (startAddr << 2U);
    elemSize  = Can_MsgObjSize[elemSize];
    elemSize *= 4U;
    elemAddr  = startAddr + (elemSize * idx);
    /* elemAddr += MCAN_MCAN_MSG_MEM; */
    MCAN_writeMsg(baseAddr, elemAddr, elem);
	return idx;
}

int32_t MCAN_txBufAddReq(uint32 baseAddr, uint32 bufNum)
{
    int32_t  status;
    uint32 regVal;
    MCAN_writeProtectedRegAccessUnlock(baseAddr);
    if (MCAN_TX_MB_MAX_NUM > bufNum)
    {
        regVal = HW_RD_REG32(baseAddr + MCAN_TXBAR);
        regVal |= ((uint32)1U << bufNum);
        HW_WR_REG32(baseAddr + MCAN_TXBAR, regVal);
        status = STW_SOK;
    }
    else
    {
        status = STW_EFAIL;
    }
    MCAN_writeProtectedRegAccessLock(baseAddr);
    return status;
}

void  MCAN_getNewDataStatus(uint32              baseAddr,
                            volatile MCAN_RxNewDataStatus *newDataStatus)
{
    newDataStatus->statusLow  = HW_RD_REG32(baseAddr + MCAN_NDAT1);
    newDataStatus->statusHigh = HW_RD_REG32(baseAddr + MCAN_NDAT2);
}

void  MCAN_clearNewDataStatus(uint32                    baseAddr,
                              const MCAN_RxNewDataStatus *newDataStatus)
{
    HW_WR_REG32(baseAddr + MCAN_NDAT1, newDataStatus->statusLow);
    HW_WR_REG32(baseAddr + MCAN_NDAT2, newDataStatus->statusHigh);
}

void MCAN_readMsgRam(uint32           baseAddr,
                     uint32           memType,
                     uint32           bufNum,
                     uint32           fifoNum,
                     volatile MCAN_RxBufElement *elem)
{
    uint32 startAddr = 0U, elemSize = 0U, elemAddr = 0U;
    uint32 enableMod = 0U, idx = 0U;

    if (MCAN_MEM_TYPE_BUF == memType)
    {
        startAddr = HW_RD_FIELD32(baseAddr+ MCAN_RXBC,
                                  MCAN_RXBC_RBSA);
        elemSize = HW_RD_FIELD32(baseAddr + MCAN_RXESC,
                                 MCAN_RXESC_RBDS);
        idx       = bufNum;
    }
    if (MCAN_MEM_TYPE_FIFO == memType)
    {
        switch (fifoNum)
        {
            case MCAN_RX_FIFO_NUM_0:
                startAddr = HW_RD_FIELD32(baseAddr + MCAN_RXF0C,
                                          MCAN_RXF0C_F0SA);
                elemSize = HW_RD_FIELD32(baseAddr + MCAN_RXESC,
                                         MCAN_RXESC_F0DS);
                idx = HW_RD_FIELD32(baseAddr + MCAN_RXF0S,
                                    MCAN_RXF0S_F0GI);
                break;
            case MCAN_RX_FIFO_NUM_1:
                startAddr = HW_RD_FIELD32(baseAddr + MCAN_RXF1C,
                                          MCAN_RXF1C_F1SA);
                elemSize = HW_RD_FIELD32(baseAddr + MCAN_RXESC,
                                         MCAN_RXESC_F1DS);
                idx = HW_RD_FIELD32(baseAddr + MCAN_RXF1S,
                                    MCAN_RXF1S_F1GI);
                break;
            default:
                /* Invalid option */
                break;
        }
    }
	startAddr = (uint32) (startAddr << 2U);
	elemSize  = Can_MsgObjSize[elemSize];
	elemSize *= 4U;
	elemAddr  = startAddr + (elemSize * idx);
	MCAN_readMsg(baseAddr, elemAddr, elem);
}


void MCAN_addStdMsgIDFilter(uint32                          baseAddr,
                            uint32                          filtNum,
                            const MCAN_StdMsgIDFilterElement *elem)
{
    uint32 startAddr, elemAddr, regVal;

    startAddr = HW_RD_FIELD32(baseAddr + MCAN_SIDFC,
                              MCAN_SIDFC_FLSSA);
    startAddr = (uint32) (startAddr << 2U);
    elemAddr  = startAddr + (filtNum * MCANSS_STD_ID_FILTER_SIZE_WORDS * 4U);
    /* elemAddr += MCAN_MCAN_MSG_MEM; */

    regVal  = 0U;
    regVal |= (uint32) (elem->sfid2 << MCANSS_STD_ID_FILTER_SFID2_SHIFT);
    regVal |= (uint32) (elem->sfid1 << MCANSS_STD_ID_FILTER_SFID1_SHIFT);
    regVal |= (uint32) (elem->sfec << MCANSS_STD_ID_FILTER_SFEC_SHIFT);
    regVal |= (uint32) (elem->sft << MCANSS_STD_ID_FILTER_SFT_SHIFT);
    HW_WR_REG32(baseAddr + elemAddr, regVal);
}

void MCAN_addExtMsgIDFilter(uint32                          baseAddr,
                            uint32                          filtNum,
                            const MCAN_ExtMsgIDFilterElement *elem)
{
    uint32 startAddr, elemAddr, regVal;

    startAddr = HW_RD_FIELD32(baseAddr + MCAN_XIDFC,
                              MCAN_XIDFC_FLESA);
    startAddr = (uint32) (startAddr << 2U);
    elemAddr  = startAddr + (filtNum * MCANSS_EXT_ID_FILTER_SIZE_WORDS * 4U);
    /* elemAddr += MCAN_MCAN_MSG_MEM; */

    regVal  = 0U;
    regVal |= (uint32) (elem->efid1 << MCANSS_EXT_ID_FILTER_EFID1_SHIFT);
    regVal |= (uint32) (elem->efec << MCANSS_EXT_ID_FILTER_EFEC_SHIFT);
    HW_WR_REG32(baseAddr + elemAddr, regVal);

    elemAddr += 4U;
    regVal    = 0U;
    regVal   |= (uint32) (elem->efid2 << MCANSS_EXT_ID_FILTER_EFID2_SHIFT);
    regVal   |= (uint32) (elem->eft << MCANSS_EXT_ID_FILTER_EFT_SHIFT);
    HW_WR_REG32(baseAddr + elemAddr, regVal);
}

void MCAN_lpbkModeEnable(uint32 baseAddr,
                         uint32 lpbkMode,
                         uint32 enable)
{
    MCAN_writeProtectedRegAccessUnlock(baseAddr);

    if ((uint32)TRUE == enable)
    {
        HW_WR_FIELD32(baseAddr + MCAN_CCCR, MCAN_CCCR_TEST, 0x1U);
        HW_WR_FIELD32(baseAddr + MCAN_TEST,
                      MCAN_TEST_LBCK,
                      enable);
        if (lpbkMode == (uint32)MCAN_LPBK_MODE_INTERNAL)
        {
            HW_WR_FIELD32(baseAddr + MCAN_CCCR,
                          MCAN_CCCR_MON,
                          0x1U);
        }
    }
    else
    {
        HW_WR_FIELD32(baseAddr + MCAN_TEST,
                      MCAN_TEST_LBCK,
                      enable);
        HW_WR_FIELD32(baseAddr + MCAN_CCCR, MCAN_CCCR_TEST, 0x0U);
        if (lpbkMode == (uint32)MCAN_LPBK_MODE_INTERNAL)
        {
            HW_WR_FIELD32(baseAddr + MCAN_CCCR,
                          MCAN_CCCR_MON,
                          0x0U);
        }
    }
    MCAN_writeProtectedRegAccessLock(baseAddr);
}

void  MCAN_getProtocolStatus(uint32             baseAddr,
                             volatile MCAN_ProtocolStatus *protStatus)
{
    uint32 regVal;

    regVal = HW_RD_REG32(baseAddr + MCAN_PSR);
    protStatus->lastErrCode   = HW_GET_FIELD(regVal, MCAN_PSR_LEC);
    protStatus->act           = HW_GET_FIELD(regVal, MCAN_PSR_ACT);
    protStatus->errPassive    = HW_GET_FIELD(regVal, MCAN_PSR_EP);
    protStatus->warningStatus = HW_GET_FIELD(regVal, MCAN_PSR_EW);
    protStatus->busOffStatus  = HW_GET_FIELD(regVal, MCAN_PSR_BO);
    protStatus->dlec          = HW_GET_FIELD(regVal, MCAN_PSR_DLEC);
    protStatus->resi          = HW_GET_FIELD(regVal, MCAN_PSR_RESI);
    protStatus->rbrs          = HW_GET_FIELD(regVal, MCAN_PSR_RBRS);
    protStatus->rfdf          = HW_GET_FIELD(regVal, MCAN_PSR_RFDF);
    protStatus->pxe           = HW_GET_FIELD(regVal, MCAN_PSR_PXE);
    protStatus->tdcv          = HW_GET_FIELD(regVal, MCAN_PSR_TDCV);
}

void MCAN_enableIntr(uint32 baseAddr, uint32 intrMask, uint32 enable)
{
    uint32 regVal;

    if ((uint32)TRUE == enable)
    {
        regVal  = HW_RD_REG32(baseAddr + MCAN_IE);
        regVal |= intrMask;
        HW_WR_REG32(baseAddr + MCAN_IE, regVal);
    }
    else
    {
        regVal  = HW_RD_REG32(baseAddr + MCAN_IE);
        regVal &= ~intrMask;
        HW_WR_REG32(baseAddr + MCAN_IE, regVal);
    }
}

void MCAN_selectIntrLine(uint32 baseAddr,
                         uint32 intrMask,
                         uint32 lineNum)
{
    uint32 regVal;

    if ( lineNum == (uint32)MCAN_INTR_LINE_NUM_0)
    {
        regVal  = HW_RD_REG32(baseAddr + MCAN_ILS);
        regVal &= ~intrMask;
        HW_WR_REG32(baseAddr + MCAN_ILS, regVal);
    }
    else
    {
        regVal  = HW_RD_REG32(baseAddr + MCAN_ILS);
        regVal |= intrMask;
        HW_WR_REG32(baseAddr + MCAN_ILS, regVal);
    }
}

void MCAN_enableIntrLine(uint32 baseAddr,
                         uint32 lineNum,
                         uint32 enable)
{
    uint32 regVal, intrLine;

    intrLine = lineNum;
    intrLine &= MCANSS_INTR_LINE_EN_MASK;
    regVal   = HW_RD_REG32(baseAddr + MCAN_ILE);
    regVal  &= ~((uint32) 0x1U << intrLine);
    regVal  |= (uint32) (enable << intrLine);
    HW_WR_REG32(baseAddr + MCAN_ILE, regVal);
}

uint32 MCAN_getIntrStatus(uint32 baseAddr)
{
    return (HW_RD_REG32(baseAddr + MCAN_IR));
}
void MCAN_clearIntrStatus(uint32 baseAddr, uint32 intrMask)
{
    HW_WR_REG32(baseAddr + MCAN_IR, intrMask);
}

void MCAN_getRxFIFOStatus(uint32           baseAddr,
                          volatile MCAN_RxFIFOStatus *fifoStatus)
{
    uint32 regVal;

    switch (fifoStatus->num)
    {
        case MCAN_RX_FIFO_NUM_0:
            regVal = HW_RD_REG32(baseAddr + MCAN_RXF0S);
            fifoStatus->fillLvl  = HW_GET_FIELD(regVal, MCAN_RXF0S_F0FL);
            fifoStatus->getIdx   = HW_GET_FIELD(regVal, MCAN_RXF0S_F0GI);
            fifoStatus->putIdx   = HW_GET_FIELD(regVal, MCAN_RXF0S_F0PI);
            fifoStatus->fifoFull = HW_GET_FIELD(regVal, MCAN_RXF0S_F0F);
            fifoStatus->msgLost  = HW_GET_FIELD(regVal, MCAN_RXF0S_RF0L);
            break;
        case MCAN_RX_FIFO_NUM_1:
            regVal = HW_RD_REG32(baseAddr + MCAN_RXF1S);
            fifoStatus->fillLvl  = HW_GET_FIELD(regVal, MCAN_RXF1S_F1FL);
            fifoStatus->getIdx   = HW_GET_FIELD(regVal, MCAN_RXF1S_F1GI);
            fifoStatus->putIdx   = HW_GET_FIELD(regVal, MCAN_RXF1S_F1PI);
            fifoStatus->fifoFull = HW_GET_FIELD(regVal, MCAN_RXF1S_F1F);
            fifoStatus->msgLost  = HW_GET_FIELD(regVal, MCAN_RXF1S_RF1L);
            break;
        default:
            /* Invalid option */
            break;
    }
}

int32_t MCAN_writeRxFIFOAck(uint32 baseAddr,
                            uint32 fifoNum,
                            uint32 idx)
{
    int32_t  status;
    uint32 size;

    switch (fifoNum)
    {
        case MCAN_RX_FIFO_NUM_0:
            size = HW_RD_FIELD32(baseAddr + MCAN_RXF0C,
                                 MCAN_RXF0C_F0S);
            if (size >= idx)
            {
                HW_WR_FIELD32(baseAddr + MCAN_RXF0A,
                              MCAN_RXF0A_F0AI,
                              idx);
                status = STW_SOK;
            }
            else
            {
                status = STW_EFAIL;
            }
            break;
        case MCAN_RX_FIFO_NUM_1:
            size = HW_RD_FIELD32(baseAddr + MCAN_RXF1C,
                                 MCAN_RXF1C_F1S);
            if (size >= idx)
            {
                HW_WR_FIELD32(baseAddr + MCAN_RXF1A,
                              MCAN_RXF1A_F1AI,
                              idx);
                status = STW_SOK;
            }
            else
            {
                status = STW_EFAIL;
            }
            break;
        default:
            status = STW_EFAIL;
            break;
    }

    return status;
}

uint32 MCAN_getTxBufReqPend(uint32 baseAddr)
{
    return (HW_RD_REG32(baseAddr + MCAN_TXBRP));
}

int32_t MCAN_txBufCancellationReq(
    uint32 baseAddr, uint32 buffNum)
{
    int32_t  status;
    uint32 regVal;

    if (MCAN_TX_MB_MAX_NUM > buffNum)
    {
        regVal  = HW_RD_REG32(baseAddr + MCAN_TXBCR);
        regVal |= ((uint32) 1U << buffNum);
        HW_WR_REG32(baseAddr + MCAN_TXBCR, regVal);
        status = STW_SOK;
    }
    else
    {
        status = STW_EFAIL;
    }
    return status;
}

uint32 MCAN_getTxBufTransmissionStatus(uint32 baseAddr)
{
    return (HW_RD_REG32(baseAddr + MCAN_TXBTO));
}

uint32 MCAN_txBufCancellationStatus(uint32 baseAddr)
{
    return (HW_RD_REG32(baseAddr + MCAN_TXBCF));
}

void MCAN_txBufTransIntrEnable(uint32 baseAddr, uint32 InterruptMask)
{
    uint32 regVal;
	regVal  = HW_RD_REG32(baseAddr + MCAN_TXBTIE);
	regVal |= InterruptMask;
	HW_WR_REG32(baseAddr + MCAN_TXBTIE, regVal);

}

void MCAN_addClockStopRequest(uint32 baseAddr, uint32 enable)
{
    if ((uint32)TRUE == enable)
    {
        HW_WR_FIELD32(baseAddr + MCAN_CCCR, MCAN_CCCR_CSR, 0x1U);
    }
    else
    {
        HW_WR_FIELD32(baseAddr + MCAN_CCCR, MCAN_CCCR_CSR, 0x0U);
    }
}

void MCAN_eccGetErrorStatus(uint32           baseAddr,
                            MCAN_ECCErrStatus *eccErr)
{
    baseAddr = (baseAddr&(~MCAN_RAM_OFFSET))>>4;
    eccErr->secErr = HW_RD_FIELD32(baseAddr + MCAN_ECC_AGGR_SEC_STATUS_REG0,
                                   MCAN_ECC_AGGR_SEC_STATUS_REG0_MSGMEM_PEND);
    eccErr->dedErr = HW_RD_FIELD32(baseAddr + MCAN_ECC_AGGR_DED_STATUS_REG0,
                                   MCAN_ECC_AGGR_DED_STATUS_REG0_MSGMEM_PEND);
}

void MCAN_eccClearErrorStatus(uint32 baseAddr, MCAN_ECCErrType errType)
{
    baseAddr = (baseAddr&(~MCAN_RAM_OFFSET))>>4;
    MCAN_eccLoadRegister(baseAddr, MCAN_ECC_AGGR_ERROR_STATUS1);
    if (errType == MCAN_ECC_ERR_TYPE_SEC)
    {
        HW_WR_FIELD32(baseAddr + MCAN_ECC_AGGR_ERROR_STATUS1,
                      MCAN_ECC_AGGR_ERROR_STATUS1_CLR_ECC_SEC,
                      0x1U);
    }
    else
    {
        HW_WR_FIELD32(baseAddr + MCAN_ECC_AGGR_ERROR_STATUS1,
                      MCAN_ECC_AGGR_ERROR_STATUS1_CLR_ECC_DED,
                      0x1U);
    }
    MCAN_eccLoadRegister(baseAddr, MCAN_ECC_AGGR_ERROR_STATUS1);
}

void MCAN_eccWriteEOI(uint32 baseAddr, MCAN_ECCErrType errType)
{
    baseAddr = (baseAddr&(~MCAN_RAM_OFFSET))>>4;
    if (errType == MCAN_ECC_ERR_TYPE_SEC)
    {
        HW_WR_FIELD32(baseAddr + MCAN_ECC_AGGR_SEC_EOI_REG,
                      MCAN_ECC_AGGR_SEC_EOI_REG_WR,
                      0x1U);
    }
    else
    {
        HW_WR_FIELD32(baseAddr + MCAN_ECC_AGGR_DED_EOI_REG,
                      MCAN_ECC_AGGR_DED_EOI_REG_WR,
                      0x1U);
    }
    HW_WR_FIELD32(baseAddr + MCAN_ECC_EOI,
                  MCAN_ECC_EOI,
                  0x1U);
}

void MCAN_eccEnableIntr(uint32 baseAddr, MCAN_ECCErrType errType, 
                        uint32 enable)
{
    baseAddr = (baseAddr&(~MCAN_RAM_OFFSET))>>4;
    if ((uint32)TRUE == enable)
    {
        if (errType == MCAN_ECC_ERR_TYPE_SEC)
        {
            HW_WR_FIELD32(baseAddr + MCAN_ECC_AGGR_SEC_ENABLE_SET_REG0,
                          MCAN_ECC_AGGR_SEC_ENABLE_SET_REG0_MSGMEM,
                          0x1U);
        }
        else
        {
            HW_WR_FIELD32(baseAddr + MCAN_ECC_AGGR_DED_ENABLE_SET_REG0,
                          MCAN_ECC_AGGR_DED_ENABLE_SET_REG0_MSGMEM,
                          0x1U);
        }
    }
    else
    {
        if (errType == MCAN_ECC_ERR_TYPE_SEC)
        {
            HW_WR_FIELD32(baseAddr + MCAN_ECC_AGGR_SEC_ENABLE_CLR_REG0,
                          MCAN_ECC_AGGR_SEC_ENABLE_CLR_REG0_MSGMEM,
                          0x1U);
        }
        else
        {
            HW_WR_FIELD32(baseAddr + MCAN_ECC_AGGR_DED_ENABLE_CLR_REG0,
                          MCAN_ECC_AGGR_DED_ENABLE_CLR_REG0_MSGMEM,
                          0x1U);
        }
    }
}

/* ========================================================================== */
/*                          Advance Functions                                 */
/* ========================================================================== */

uint32 MCAN_getClkStopAck(uint32 baseAddr)
{
    return (HW_RD_FIELD32(baseAddr + MCAN_CCCR, MCAN_CCCR_CSA));
}




/* ========================================================================== */
/*                          Internal Functions                                */
/* ========================================================================== */
static void MCAN_writeProtectedRegAccessUnlock(uint32 baseAddr)
{
    HW_WR_FIELD32(baseAddr + MCAN_CCCR, MCAN_CCCR_CCE, 0x1U);
}
static void MCAN_writeProtectedRegAccessLock(uint32 baseAddr)
{
    HW_WR_FIELD32(baseAddr + MCAN_CCCR, MCAN_CCCR_CCE, 0x0U);
}

static void MCAN_eccLoadRegister(uint32 baseAddr, uint32 regOffSet)
{
    uint32 regVal = 0U, offset;
    TickType    startCount = 0U, tempCount, elaspsedCount = 0U;
    boolean     timeOut = (boolean)FALSE;

    offset  = regOffSet & 0xFFU;
    regVal |= ((uint32)MCANSS_MSG_RAM_NUM << MCAN_ECC_AGGR_VECTOR_SHIFT);
    regVal |= (offset << MCAN_ECC_AGGR_VECTOR_RD_SVBUS_ADDRESS_SHIFT);
    regVal |= ((uint32)1U << MCAN_ECC_AGGR_VECTOR_RD_SVBUS_SHIFT);
    HW_WR_REG32(baseAddr + MCAN_ECC_AGGR_VECTOR, regVal);

    (void)GetCounterValue(CAN_OS_COUNTER_ID, &startCount);    
    {
        /* Below API can change start time, so use temp variable */
        tempCount = startCount;
        /* Wait for register from register bank to load into front register */
        do
        {
            (void)GetElapsedValue(
                         CAN_OS_COUNTER_ID,
                         &tempCount,
                         &elaspsedCount);
            if (elaspsedCount >= (uint32)CAN_TIMEOUT_DURATION)
            {
                /* timeout */
                timeOut = (boolean)TRUE;
                break;
            }
        } while (MCAN_ECC_AGGR_VECTOR_RD_SVBUS_DONE_MASK !=
           (HW_RD_REG32(baseAddr + MCAN_ECC_AGGR_VECTOR) &
            MCAN_ECC_AGGR_VECTOR_RD_SVBUS_DONE_MASK));
    }
    if (timeOut == (boolean)TRUE)
    {
      (void)Dem_SetEventStatus(
            CAN_E_HARDWARE_ERROR, DEM_EVENT_STATUS_FAILED);
    }
}

static void MCAN_readMsg(uint32           baseAddr,
                         uint32           elemAddr,
                         volatile MCAN_RxBufElement *elem)
{
    uint32 regVal = 0U, loopCnt = 0U, addrOffset;

    addrOffset = elemAddr;
    regVal   = HW_RD_REG32(baseAddr + addrOffset);
    elem->id = (uint32) ((regVal & MCANSS_RX_BUFFER_ELEM_ID_MASK)
                           >> MCANSS_RX_BUFFER_ELEM_ID_SHIFT);
    elem->rtr = (uint32) ((regVal & MCANSS_RX_BUFFER_ELEM_RTR_MASK)
                            >> MCANSS_RX_BUFFER_ELEM_RTR_SHIFT);
    elem->xtd = (uint32) ((regVal & MCANSS_RX_BUFFER_ELEM_XTD_MASK)
                            >> MCANSS_RX_BUFFER_ELEM_XTD_SHIFT);
    elem->esi = (uint32) ((regVal & MCANSS_RX_BUFFER_ELEM_ESI_MASK)
                            >> MCANSS_RX_BUFFER_ELEM_ESI_SHIFT);

    addrOffset  += 4U;
    regVal     = HW_RD_REG32(baseAddr + addrOffset);
    elem->rxts = (uint32) ((regVal & MCANSS_RX_BUFFER_ELEM_RXTS_MASK)
                             >> MCANSS_RX_BUFFER_ELEM_RXTS_SHIFT);
    elem->data_length = (uint32) ((regVal & MCANSS_RX_BUFFER_ELEM_DLC_MASK)
                            >> MCANSS_RX_BUFFER_ELEM_DLC_SHIFT);
    elem->brs = (uint32) ((regVal & MCANSS_RX_BUFFER_ELEM_BRS_MASK)
                            >> MCANSS_RX_BUFFER_ELEM_BRS_SHIFT);
    elem->fdf = (uint32) ((regVal & MCANSS_RX_BUFFER_ELEM_FDF_MASK)
                            >> MCANSS_RX_BUFFER_ELEM_FDF_SHIFT);
    elem->fidx = (uint32) ((regVal & MCANSS_RX_BUFFER_ELEM_FIDX_MASK)
                             >> MCANSS_RX_BUFFER_ELEM_FIDX_SHIFT);
    elem->anmf = (uint32) ((regVal & MCANSS_RX_BUFFER_ELEM_ANMF_MASK)
                             >> MCANSS_RX_BUFFER_ELEM_ANMF_SHIFT);
    addrOffset += 4U;

    loopCnt = Can_DataSize[elem->data_length];
	
#ifndef __aarch64__
    memcpy((void *)elem->data, (void *)(baseAddr + addrOffset), loopCnt);
#else
    memcpy((void *)elem->data, (void *)((uint64_t)(baseAddr + addrOffset)), loopCnt);
#endif
}

static void MCAN_writeMsg(uint32                 baseAddr,
                          uint32                 elemAddr,
                          const MCAN_TxBufElement *elem)
{
    uint32 regVal = 0U, loopCnt = 0U, addrOffset;

    addrOffset = elemAddr;
    regVal  = 0U;
    regVal |= (((uint32) (elem->id << MCANSS_TX_BUFFER_ELEM_ID_SHIFT)) |
               ((uint32) (elem->rtr << MCANSS_TX_BUFFER_ELEM_RTR_SHIFT)) |
               ((uint32) (elem->xtd << MCANSS_TX_BUFFER_ELEM_XTD_SHIFT)) |
               ((uint32) (elem->esi << MCANSS_TX_BUFFER_ELEM_ESI_SHIFT)));
    HW_WR_REG32(baseAddr + addrOffset, regVal);
    addrOffset += 4U;

    regVal  = 0U;
    regVal |= 
           ((uint32) (elem->data_length << MCANSS_TX_BUFFER_ELEM_DLC_SHIFT)) |
           ((uint32) (elem->brs << MCANSS_TX_BUFFER_ELEM_BRS_SHIFT)) |
           ((uint32) (elem->fdf << MCANSS_TX_BUFFER_ELEM_FDF_SHIFT)) |
           ((uint32) (elem->efc << MCANSS_TX_BUFFER_ELEM_EFC_SHIFT)) |
           ((uint32) (elem->mm << MCANSS_TX_BUFFER_ELEM_MM_SHIFT));
	HW_WR_REG32(baseAddr + addrOffset, regVal);
    addrOffset += 4U;

    loopCnt = 0U;
    /* Framing words out of the payload bytes and writing it to message RAM */
    while ((4U <= (MCAN_getDataSize(elem->data_length) - loopCnt)) &&
           (0U != (MCAN_getDataSize(elem->data_length) - loopCnt)))
    {
        regVal  = 0U;
        regVal |= ((uint32)elem->data[loopCnt] |
                   ((uint32)elem->data[(loopCnt + 1U)] << 8U) |
                   ((uint32)elem->data[(loopCnt + 2U)] << 16U) |
                   ((uint32)elem->data[(loopCnt + 3U)] << 24U));
        HW_WR_REG32(baseAddr + addrOffset, regVal);
        addrOffset += 4U;
        loopCnt  += 4U;
    }
    /* Framing a word out of remaining payload bytes and writing it to
     * message RAM */
    if (0U < (MCAN_getDataSize(elem->data_length) - loopCnt))
    {
        regVal  = 0U;
        regVal |= ((uint32)elem->data[loopCnt] |
                   ((uint32)elem->data[(loopCnt + 1U)] << 8U) |
                   ((uint32)elem->data[(loopCnt + 2U)] << 16U) |
                   ((uint32)elem->data[(loopCnt + 3U)] << 24U));
        HW_WR_REG32(baseAddr + addrOffset, regVal);
    }
}

static uint32 MCAN_getDataSize(uint32 data_length)
{
    uint32 dataSize[16] = {0U,  1U,  2U,  3U,  4U,  5U,  6U, 7U, 8U,
                             12U, 16U, 20U, 24U, 32U, 48U, 64U};

    return (dataSize[data_length]);
}

static uint32 MCAN_getMsgObjSize(uint32 elemSize)
{
    uint32 objSize[8] = {4U, 5U, 6U, 7U, 8U, 10U, 14U, 18U};

    return (objSize[elemSize]);
}


int32_t MCAN_setExtIDAndMask(uint32 baseAddr, uint32 idMask)
{
  int32_t status;

  if (MCAN_XIDAM_EIDM_MAX >= idMask)
  {
    MCAN_writeProtectedRegAccessUnlock(baseAddr);

    HW_WR_FIELD32((baseAddr) + MCAN_XIDAM,
      MCAN_XIDAM_EIDM,
      idMask);

    MCAN_writeProtectedRegAccessLock(baseAddr);
    status = STW_SOK;
  }
  else
  {
    status = STW_EFAIL;
  }
  return status;
}


/* Memory mapping, done via MemMap.h: Code */
#define CAN_STOP_SEC_CODE
#include "Can_MemMap.h"
