/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     Dio_Platform.c
 *
 */

#define DIO_PRIV_SOURCE  (1)

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */


#include "Std_Types.h"
#include "Dio.h"
#include "Dio_priv.h"
#include "soc.h"


/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define DIO_NUM_GPIO_REGS         (5U)

/* ========================================================================== */
/*                         Structure Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

#define DIO_START_SEC_CONST_UNSPECIFIED
#include "Dio_MemMap.h"

STATIC CONST(uint32, DIO_CONST) Dio_GPIOPortAddr[DIO_NUM_GPIO_REGS] =
{
    SOC_GPIO_PORTAB_BASE,
    SOC_GPIO_PORTCD_BASE,
    SOC_GPIO_PORTEF_BASE,
    SOC_GPIO_PORTGH_BASE,
    SOC_GPIO_PORTI_BASE
};

#define DIO_STOP_SEC_CONST_UNSPECIFIED
#include "Dio_MemMap.h"

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#define DIO_START_SEC_CODE
#include "Dio_MemMap.h"
uint32 Dio_GetGPIOPortAddr(uint8 regNum)
{
    uint32 retVal;

    retVal = Dio_GPIOPortAddr[regNum];

    return (retVal);
}

/** @fn uint32 Dio_GpioGetLogicLvl(gioPORT_t *port, uint32 bit)
*   @brief Read Bit
*   @param[in] port pointer to GIO port:
*              - gioPORTA: PortA pointer
*              - gioPORTB: PortB pointer
*              - gioPORTC: PortC pointer
*              - gioPORTD: PortD pointer
*   @param[in] bit number 0-7 that specifies the bit to be written to.
*              - 0: LSB
*              - 7: MSB
*
*   Reads a the current value from the specified pin of the given GIO port
*/
/* SourceId :  */
/* DesignId :  */
/* Requirements :  */
uint32 Dio_GpioGetLogicLvl(const gpioPORT_t *port, uint32 bit)
{
    return (port->DIN >> bit) & 1U;
}



/** @fn uint32 Dio_GpioGetDirection(gioPORT_t *port, uint32 bit)
*   @brief Read Bit
*   @param[in] port pointer to GIO port:
*              - gioPORTA: PortA pointer
*              - gioPORTB: PortB pointer
*              - gioPORTC: PortC pointer
*              - gioPORTD: PortD pointer
*   @param[in] bit number 0-7 that specifies the bit to be written to.
*              - 0: LSB
*              - 7: MSB
*
*   Reads a the current value from the specified pin of the given GIO port
*/
/* SourceId :  */
/* DesignId :  */
/* Requirements :  */
uint32 Dio_GpioGetDirection(const gpioPORT_t *port, uint32 bit)
{

    uint32 dataIn = port->DIR;

    return (dataIn >> bit) & 1U;

}
/** @fn uint32 Dio_GpioGetPortDirection(gioPORT_t *port)
*   @brief Read Bit
*   @param[in] port pointer to GIO port:
*              - gioPORTA: PortA pointer
*              - gioPORTB: PortB pointer
*              - gioPORTC: PortC pointer
*              - gioPORTD: PortD pointer
*
*   Reads a the current value from the specified pin of the given GIO port
*/
/* SourceId :  */
/* DesignId :  */
/* Requirements :  */
uint32 Dio_GpioGetPortDirection(const gpioPORT_t *port)
{
    return ( port->DIR);

}

/** @fn uint32 Dio_GpioSetBit(gioPORT_t *port, uint32 bit, uint32 value)
*   @brief Read Bit
*   @param[in] port pointer to GIO port:
*              - gioPORTA: PortA pointer
*              - gioPORTB: PortB pointer
*              - gioPORTC: PortC pointer
*              - gioPORTD: PortD pointer
*   @param[in] bit number 0-7 that specifies the bit to be written to.
*              - 0: LSB
*              - 7: MSB
*
*   writes the current value from the specified pin of the given GIO port
*/
/* SourceId :  */
/* DesignId :  */
/* Requirements :  */
void Dio_GpioSetBit(gpioPORT_t *port, uint32 bit, uint32 value)
{

    if (value != 0U)
    {
        port->DSET = (uint32)1U << bit;
    }
    else
    {
        port->DCLR = (uint32)1U << bit;
    }

}




/** @fn uint32 Dio_GpioPortWrite(gioPORT_t *port, uint32 bit, uint32 value)
*   @brief Read Bit
*   @param[in] port pointer to GIO port:
*              - gioPORTA: PortA pointer
*              - gioPORTB: PortB pointer
*              - gioPORTC: PortC pointer
*              - gioPORTD: PortD pointer
*   @param[in] bit number 0-7 that specifies the bit to be written to.
*              - 0: LSB
*              - 7: MSB
*
*   Reads a the current value from the specified pin of the given GIO port
*/

uint32 Dio_GpioPortWrite(const gpioPORT_t *port, uint32 setBits)
{
    uint32 regWrFailStatus = 0U;
    regWrFailStatus |= regWriteReadback(&port->DOUT, M_THIRTY_ONE, M_ZERO, setBits);
    return (regWrFailStatus);
}

/** @fn uint32 Dio_GpioGetPort(gioPORT_t *port)
*   @brief Read Port Value
*   @param[in] port pointer to GIO port:
*              - gioPORTA: PortA pointer
*              - gioPORTB: PortB pointer
*              - gioPORTC: PortC pointer
*              - gioPORTD: PortD pointer
*
*   Reads a the current value of a given GIO port
*/
/* SourceId :  */
/* DesignId :  */
/* Requirements :  */
uint32 Dio_GpioGetPort(const gpioPORT_t *port)
{
    return (port->DIN);
}

/** @fn void Dio_GpioBitToggle(gioPORT_t *port, uint32 bit)
*   @brief Write Bit
*   @param[in] port pointer to GIO port:
*              - gioPORTA: PortA pointer
*              - gioPORTB: PortB pointer
*              - gioPORTC: PortC pointer
*              - gioPORTD: PortD pointer
*   @param[in] bit number 0-7 that specifies the bit to be written to.
*              - 0: LSB
*              - 7: MSB
*
*   Toggle a value to the specified pin of the given GIO port
*/
/* SourceId :  */
/* DesignId :  */
/* Requirements :  */
void Dio_GpioBitToggle(gpioPORT_t *port, uint32 bit)
{

    uint32 dataIn = port->DIN;

    if ((dataIn & (uint32)((uint32)1U << bit)) != 0U)
    {
        port->DCLR = (uint32)1U << bit;
    }
    else
    {
        port->DSET = (uint32)1U << bit;
    }

}

#define DIO_STOP_SEC_CODE
#include "Dio_MemMap.h"

