/*
 * Copyright (C) 2023 Texas Instruments Incorporated
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 *   Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the
 *   distribution.
 *
 *   Neither the name of Texas Instruments Incorporated nor the names of
 *   its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 *  \defgroup DRV_CDD_EDMA_MODULE APIs for EDMA
 *  \ingroup DRV_MODULE
 *
 *  This module contains APIs to program and use the EDMA.
 *
 *  @{
 */

/**
 *  \file v0/edma.h
 *
 *  \brief This file contains the prototype of EDMA driver APIs
 *
 *  \details  Please find the below detailed description of edma dal.
 *            -# Programming sequence for initializing the edma controller
 *               -# Perform the clock configuration of channel controller and
 *                  transfer controllers.
 *               -# To perform initialization of controller use the
 *                  -# #CDD_EDMA_init with appropriate #CDD_EDMA_InitParams
 *                     which is part of the #CDD_EDMA_Attrs
 *            -# Programming sequence for setting up edma channel.
 *               -# To configure the PaRAM sets use #CDD_EDMA_setPaRAM. For
 *                  custom use
 *               -# To start transfer use #CDD_EDMA_enableTransferRegion.
 *               -# To stop transfer use #CDD_EDMA_disableTransferRegion.
 */

#ifndef CDD_DMA_EDMA_LLD_H_
#define CDD_DMA_EDMA_LLD_H_

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <stdint.h>
#include "Std_Types.h"
#include "Cdd_Dma_edma.h"
#include "sys_common.h"

/* ========================================================================== */
/*                                 Macros                                     */
/* ========================================================================== */
/** @brief Number of DMA Channels */
#define CDD_EDMA_NUM_DMACH                 (64U)
/** @brief Number of QDMA Channels */
#define CDD_EDMA_NUM_QDMACH                (8U)
/** @brief Number of PaRAM Sets available */
#define CDD_EDMA_NUM_PARAMSETS             (256U)
/** @brief Number of Event Queues available */
#define CDD_EDMA_NUM_EVQUE                 (2U)
/** @brief Support for Channel to PaRAM Set mapping */
#define CDD_EDMA_CHMAPEXIST                (1U)
/** @brief Number of EDMA Regions */
#define CDD_EDMA_NUM_REGIONS               (8U)
/** @brief Support for Memory Protection */
#define CDD_EDMA_MEMPROTECT                (1U)

/** \anchor CDD_EDMA_QUEUE_NUM_DEFS
*   \name EDMA Queue Number Configuration.
*  @{
*/
#define CDD_EDMACC_DMAQNUM_CLR(chNum) \
    (~((uint32) 0x7U << (((chNum) % 8U) * 4U)))
/**< DMAQNUM bits Clear */
#define CDD_EDMACC_DMAQNUM_SET(chNum, queNum) \
    (((uint32) 0x7U & (queNum)) << (((chNum) % 8U) * 4U))
/**< DMAQNUM bits Set */
#define CDD_EDMACC_QDMAQNUM_CLR(chNum) \
    (~((uint32) 0x7U << ((chNum) * 4U)))
/**< QDMAQNUM bits Clear */
#define CDD_EDMACC_QDMAQNUM_SET(chNum, queNum) \
    (((uint32) 0x7U & (queNum)) << ((chNum) * 4U))
/**< QDMAQNUM bits Set */
/** @} */

/** \anchor CDD_EDMA_QDMA_MAPPING_DEFS
*   \name EDMA Qdma Channel mapping Configuration.
*  @{
*/
#define CDD_EDMACC_QCHMAP_PAENTRY_CLR ((uint32) (~((uint32)CDD_EDMA_TPCC_QCHMAPN_PAENTRY_MASK)))
/**< QCHMAP Param ID Clear */
#define CDD_EDMACC_QCHMAP_PAENTRY_SET(paRAMId)                               \
    (((CDD_EDMA_TPCC_QCHMAPN_PAENTRY_MASK >> CDD_EDMA_TPCC_QCHMAPN_PAENTRY_SHIFT) \
      & (paRAMId)) << CDD_EDMA_TPCC_QCHMAPN_PAENTRY_SHIFT)                    \
/**< QCHMAP-PaRAMEntry bitfield Set */
#define CDD_EDMACC_QCHMAP_TRWORD_CLR  ((uint32) (~((uint32)CDD_EDMA_TPCC_QCHMAPN_TRWORD_MASK)))
/**< QCHMAP-TrigWord bitfield Clear */
#define CDD_EDMACC_QCHMAP_TRWORD_SET(paRAMId)                                \
    (((CDD_EDMA_TPCC_QCHMAPN_TRWORD_MASK >> CDD_EDMA_TPCC_QCHMAPN_TRWORD_SHIFT) & \
      (paRAMId)) << CDD_EDMA_TPCC_QCHMAPN_TRWORD_SHIFT)
/**< QCHMAP-TrigWord bitfield Set */
/** @} */

/** \anchor CDD_EDMA_TRIGGER_MODE_DEFS
*   \name EDMA Trigger Mode Configuration.
*  @{
*/
#define CDD_EDMA_TRIG_MODE_MANUAL                ((uint32) 0U)
/**< Manually trigger EDMA transfer */
#define CDD_EDMA_TRIG_MODE_QDMA                  ((uint32) 1U)
/**< Manually trigger QDMA transfer */
#define CDD_EDMA_TRIG_MODE_EVENT                 ((uint32) 2U)
/**< Trigger EDMA transfer upon Event */
/** @} */

/** \anchor CDD_EDMA_CHANNEL_TYPE_DEFS
*   \name EDMA channel type definitions.
*  @{
*/
/** \brief Values that can be used for parameter chType in API's
*/
#define CDD_EDMA_CHANNEL_TYPE_DMA                ((uint32) 0U)
/**< Channel Typr DMA */
#define CDD_EDMA_CHANNEL_TYPE_QDMA               ((uint32) 1U)
/**< Channel Typr QDMA */
/** @} */

/** \anchor CDD_EDMA_EVENT_STATUS_DEFS
*   \name EDMA event status definitions.
*  @{
*/
/** \brief Values that can be used to specify different event status
*/
#define CDD_EDMA_XFER_COMPLETE                   ((uint32) 0U)
/**< Transfer Complete */
#define CDD_EDMA_CC_DMA_EVT_MISS                 ((uint32) 1U)
/**< DMA Event Miss */
#define CDD_EDMA_CC_QDMA_EVT_MISS                ((uint32) 2U)
/**< QDMA Event Miss */
/** @} */

/** \anchor CDD_EDMA_TRANSFER_TYPE_DEFS
*   \name EDMA transfer type definitions.
*  @{
*/
/**
 *   \brief Values that can be used to specify different
 *   synchronization events
 */
#define CDD_EDMA_SYNC_A                          ((uint32) 0U)
/**< A Sync Transfer */
#define CDD_EDMA_SYNC_AB                         ((uint32) 1U)
/** AB Sync Transfer */
/** @} */

/** \anchor CDD_EDMA_ADDRESSING_MODE_DEFS
*   \name EDMA addressing modes definitions.
*  @{
*/
/** \brief Values that can be used to specify different
* addressing modes (relevant for SAM and DAM sub-fields in OPT field).
*/
#define CDD_EDMA_ADDRESSING_MODE_LINEAR          ((uint32) 0U)
/**< Incremental addressing (INCR), not FIFO */
#define CDD_EDMA_ADDRESSING_MODE_FIFO_WRAP       ((uint32) 1U)
/**< Constant addressing (CONST) within the FIFO array, wraps around upon
    reaching FIFO width */
/** @} */

/** \anchor CDD_EDMA_FIFO_WIDTH_DEFS
*   \name EDMA FIFO width definitions.
*  @{
*/
/** \brief Values that can be used to specify different FIFO widths (FWID in OPT field).
*/
#define CDD_EDMA_FIFO_WIDTH_8BIT     ((uint32) CDD_EDMA_TPCC_OPT_FWID_FIFOWIDTH8BIT)
/**< 8-bit FIFO width */
#define CDD_EDMA_FIFO_WIDTH_16BIT    ((uint32) CDD_EDMA_TPCC_OPT_FWID_FIFOWIDTH16BIT)
/**< 16-bit FIFO width */
#define CDD_EDMA_FIFO_WIDTH_32BIT    ((uint32) CDD_EDMA_TPCC_OPT_FWID_FIFOWIDTH32BIT)
/**< 32-bit FIFO width */
#define CDD_EDMA_FIFO_WIDTH_64BIT    ((uint32) CDD_EDMA_TPCC_OPT_FWID_FIFOWIDTH64BIT)
/**< 64-bit FIFO width */
#define CDD_EDMA_FIFO_WIDTH_128BIT   ((uint32) CDD_EDMA_TPCC_OPT_FWID_FIFOWIDTH128BIT)
/**< 128-bit FIFO width */
#define CDD_EDMA_FIFO_WIDTH_256BIT   ((uint32) DMA_TPCC_OPT_FWID_FIFOWIDTH256BIT)
/**< 256-bit FIFO width */
/** @} */

/** \anchor CDD_EDMA_CLEAR_CC_ERROR_DEFS
*   \name EDMA Clear Channel controller Error.
*  @{
*/
/** \brief Values that can be used to Clear any Channel controller Errors
*/
#define CDD_EDMACC_CLR_TCCERR         ((uint32) CDD_EDMA_TPCC_CCERRCLR_TCERR_MASK)
/**< Cleat TCC Error */
#define CDD_EDMACC_CLR_QTHRQ0         ((uint32) CDD_EDMA_TPCC_CCERRCLR_QTHRXCD0_MASK)
/**< Cleat Queue threshold 0 Error */
#define CDD_EDMACC_CLR_QTHRQ1         ((uint32) CDD_EDMA_TPCC_CCERRCLR_QTHRXCD1_MASK)
/**< Cleat Queue threshold 1 Error */
/** @} */

/** \anchor CDD_EDMA_PARAM_OPT_FIELD_DEFS
*   \name EDMA Param OPT fields.
*  @{
*/
/** \brief Values that are used to Chain two specified channels */
#define CDD_EDMA_OPT_TCCHEN_MASK      ((uint32) CDD_EDMA_TPCC_OPT_TCCHEN_MASK)
/**< Transfer Complete Chaining enable */
#define CDD_EDMA_OPT_ITCCHEN_MASK     ((uint32) CDD_EDMA_TPCC_OPT_ITCCHEN_MASK)
/**< Intermediate Transfer Complete Chaining enable */
#define CDD_EDMA_OPT_TCINTEN_MASK     ((uint32) CDD_EDMA_TPCC_OPT_TCINTEN_MASK)
/**< Transfer Complete Interrupt enable */
#define CDD_EDMA_OPT_ITCINTEN_MASK    ((uint32) CDD_EDMA_TPCC_OPT_ITCINTEN_MASK)
/**< Intermediate Transfer Complete Interrupt enable */
#define CDD_EDMA_OPT_TCC_MASK         ((uint32) CDD_EDMA_TPCC_OPT_TCC_MASK)
/**< Transfer Complete Code mask */
#define CDD_EDMA_OPT_TCC_SHIFT        ((uint32) CDD_EDMA_TPCC_OPT_TCC_SHIFT)
/**< Transfer Complete Code shift */
#define CDD_EDMA_OPT_SYNCDIM_MASK     ((uint32) CDD_EDMA_TPCC_OPT_SYNCDIM_MASK)
/**< Sync Type shift */
#define CDD_EDMA_OPT_SYNCDIM_SHIFT    ((uint32) CDD_EDMA_TPCC_OPT_SYNCDIM_SHIFT)
/**< Sync Type mask */
#define CDD_EDMA_OPT_STATIC_MASK      ((uint32) CDD_EDMA_TPCC_OPT_STATIC_MASK)
/**< Param Static mask */
#define CDD_EDMA_OPT_STATIC_SHIFT     ((uint32) CDD_EDMA_TPCC_OPT_STATIC_SHIFT)
/**< Param Static mask */
#define CDD_EDMACC_OPT_TCC_CLR        ((uint32) (~CDD_EDMA_TPCC_OPT_TCC_MASK))
/**< OPT-TCC bitfield Clear */
#define CDD_EDMACC_OPT_TCC_SET(tcc)                                      \
    (((CDD_EDMA_TPCC_OPT_TCC_MASK >> CDD_EDMA_TPCC_OPT_TCC_SHIFT) & (tcc)) << \
     CDD_EDMA_TPCC_OPT_TCC_SHIFT)
/**< OPT-TCC bitfield Set */
/** @} */

/** \anchor CDD_EDMA_PARAM_FIELD_DEFS
*   \name EDMA Param fields.
*  @{
*/
/** \brief PaRAMEntry Fields */
#define    CDD_EDMACC_PARAM_ENTRY_OPT            ((uint32) 0x0U)
/**< The OPT field (Offset Address 0x0 Bytes) */
#define    CDD_EDMACC_PARAM_ENTRY_SRC            ((uint32) 0x1U)
/**< The SRC field (Offset Address 0x4 Bytes)*/
#define    CDD_EDMACC_PARAM_ENTRY_ACNT_BCNT      ((uint32) 0x2U)
/**< The (ACNT+BCNT) field (Offset Address 0x8 Bytes)*/
#define    CDD_EDMACC_PARAM_ENTRY_DST            ((uint32) 0x3U)
/**< The DST field (Offset Address 0xC Bytes)*/
#define    CDD_EDMACC_PARAM_ENTRY_SRC_DST_BIDX   ((uint32) 0x4U)
/**< The (SRCBIDX+DSTBIDX) field (Offset Address 0x10 Bytes)*/
#define    CDD_EDMACC_PARAM_ENTRY_LINK_BCNTRLD   ((uint32) 0x5U)
/**< The (LINK+BCNTRLD) field (Offset Address 0x14 Bytes)*/
#define    CDD_EDMACC_PARAM_ENTRY_SRC_DST_CIDX   ((uint32) 0x6U)
/**< The (SRCCIDX+DSTCIDX) field (Offset Address 0x18 Bytes)*/
#define    CDD_EDMACC_PARAM_ENTRY_CCNT           ((uint32) 0x7U)
/**< The (CCNT+RSVD) field (Offset Address 0x1C Bytes)*/
#define    CDD_EDMACC_PARAM_FIELD_OFFSET         ((uint32) 0x4U)
/**< The offset for each PaRAM Entry field */
#define    CDD_EDMACC_PARAM_ENTRY_FIELDS         ((uint32) 0x8U)
/**< Number of PaRAM Entry fields.
 *   OPT, SRC, A_B_CNT, DST, SRC_DST_BIDX, LINK_BCNTRLD, SRC_DST_CIDX
 *   and CCNT
 */
#define CDD_EDMA_NUM_TCC                        ((uint32) CDD_EDMA_NUM_DMACH)
/**< Number of TCC's available */
/** @} */

/** \anchor CDD_EDMA_RESOURCE_TYPE_DEFS
*   \name EDMA resource type definition used for resource allocation and freeing.
*  @{
*/
/** \brief Values that can be used for parameter resType in API's
*/
#define CDD_EDMA_RESOURCE_TYPE_DMA                ((uint32) 0U)
/**< Resource Type DMA */
#define CDD_EDMA_RESOURCE_TYPE_QDMA               ((uint32) 1U)
/**< Resource Type QDMA */
#define CDD_EDMA_RESOURCE_TYPE_TCC                ((uint32) 2U)
/**< Resource Type TCC */
#define CDD_EDMA_RESOURCE_TYPE_PARAM              ((uint32) 3U)
/**< Resource Type PARAM */
#define CDD_EDMA_RESOURCE_ALLOC_ANY               ((uint32) 0xFFFFU)
/**< Allocate any available resource */
/** @} */

#define CDD_EDMA_SET_ALL_BITS                    ((uint32) 0xFFFFFFFFU)
/**< Used to set all Bits in a register */
#define CDD_EDMA_CLR_ALL_BITS                    ((uint32) 0x00000000U)
/**< Used to clear all Bits in a register */
#define CDD_EDMACC_COMPL_HANDLER_RETRY_COUNT     ((uint32) 10U)
/**< Completion interrupt handler retry count */
#define CDD_EDMACC_ERR_HANDLER_RETRY_COUNT       ((uint32) 10U)
/**< Error interrupt handler retry count */

/* ========================================================================== */
/*                         Structures                                         */
/* ========================================================================== */
/**
 * \brief EDMA Parameter RAM Set in User Configurable format
 *        This is a mapping of the EDMA PaRAM set provided to the user
 *        for ease of modification of the individual fields
 */
typedef struct {
    /* \brief OPT field of PaRAM Set */
    uint32 opt;
    /**
     * \brief Starting byte address of Source
     * For FIFO mode, srcAddr must be a 256-bit aligned address.
     */
    uint32 srcAddr;
    /* \brief Number of bytes in each Array (ACNT) */
    uint16 aCnt;
    /* \brief Number of Arrays in each Frame (BCNT) */
    uint16 bCnt;
    /**
     * \brief Starting byte address of destination
     * For FIFO mode, destAddr must be a 256-bit aligned address.
     * i.e. 5 LSBs should be 0.
     */
    uint32 destAddr;
    /* \brief Index between consec. arrays of a Source Frame (SRCBIDX) */
    sint16  srcBIdx;
    /* \brief Index between consec. arrays of a Destination Frame (DSTBIDX) */
    sint16  destBIdx;
    /**
     * \brief Address for linking (AutoReloading of a PaRAM Set)
     * This must point to a valid aligned 32-byte PaRAM set
     * A value of 0xFFFF means no linking
     */
    uint16 linkAddr;
    /**
     * \brief Reload value of the numArrInFrame (BCNT)
     * Relevant only for A-sync transfers
     */
    uint16 bCntReload;
    /* \brief Index between consecutive frames of a Source Block (SRCCIDX) */
    sint16  srcCIdx;
    /* \brief Index between consecutive frames of a Dest Block (DSTCIDX) */
    sint16  destCIdx;
    /* \brief Number of Frames in a block (CCNT) */
    uint16 cCnt;
    /* \brief Reserved bit field (Should not be written with a non zero value) */
    uint16 reserved;

} __attribute__((packed))
CDD_EDMACCEDMACCPaRAMEntry;

/**
 * \brief EDMA resource allocation structure
 *
 */
typedef struct
{
    /**
     * \brief DMA channels allocated. Each channel will be defined with 1 bit.
     *        ownDmaCh[0] used for 0 to 31, ownDmaCh[1] for 32 to 63 etc.
     */
    uint32    dmaCh[CDD_EDMA_NUM_DMACH/32U];
    /* \brief QDMA channels allocated. Each channel will be defined with 1 bit. */
    uint32    qdmaCh;
    /**
     * \brief TCCs allocated. Each tcc will be defined with 1 bit.
     *        ownTcc[0] used for 0 to 31, ownTcc[1] for 32 to 63 etc.
     */
    uint32    tcc[CDD_EDMA_NUM_TCC/32U];
    /**
     * \brief PaRAMs owned. Each PaRAM will be defined with 1 bit.
     *        ownParamSet[0] used for 0 to 31, ownParamSet[1] for 32 to 63 etc.
     */
    uint32    paramSet[CDD_EDMA_NUM_PARAMSETS/32U];
    /**
     * \brief Used for thread protection during resource allocation.
     */
    uint32    allocMutex;
} CDD_EDMA_ResourceObject;

/**
 * \brief EDMA interrupt handle returned from #CDD_EDMA_registerIntr() function.
 */
typedef struct Edma_IntrObject_t   *Cdd_Edma_IntrHandle;

/**
 * \brief EDMA interrupt callback function prototype
 */
typedef void (*Cdd_Edma_EventCallback)(Cdd_Edma_IntrHandle intrHandle,
                                   void *appData);

/**
 * \brief EDMA interrupt configuration object. The object is passed to the
 *        #CDD_EDMA_registerIntr() function. This is held by the driver till the
 *        #CDD_EDMA_unregisterIntr() function is called with this obect. The
 *        application should not modify this object in between.
 *
 */
typedef struct Edma_IntrObject_t
{
    /* \brief TCC number for which the callback to be reistered. */
    uint32                  tccNum;
    /* \brief Application data pointer passed to callback function. */
    void                     *appData;
    /* \brief Callback function pointer. */
    Cdd_Edma_EventCallback        cbFxn;
    /**
     * \brief pointer to next interrupt object. Used internally by driver.
     *        Should not be modified by the application.
     */
    Cdd_Edma_IntrHandle  nextIntr;
    /**
     * \brief pointer to previous interrupt object. Used internally by driver.
     *        Should not be modified by the application.
     */
    Cdd_Edma_IntrHandle  prevIntr;
} Cdd_Edma_IntrObject;

/**
 *  \brief EDMA driver initialization object
 */
typedef struct
{
    uint32    regionId;
    /**< EDMA region to be used */
    uint32    queNum;
    /**< EDMA Event queue to be used for all channels */
    uint32    initParamSet;
    /**< Parameter to reset the PaRAM memory of the owned PaRAMs */
    CDD_EDMA_ResourceObject ownResource;
    /**< Owned resource configuration */
    uint32    reservedDmaCh[CDD_EDMA_NUM_DMACH/32U];
    /**< Dma channels reserved for Event triggered transfers */
    uint32    compIntrNumber;
    /**< Completion interrupt number. */
    uint32    compIntrPriority;
    /**< Completion interrupt priority. */
    uint32    intrAggEnableAddr;
    /**< Interrupt Aggregator enable address */
    uint32    intrAggEnableMask;
    /**< Interrupt Aggregator enable mask */
    uint32    intrAggStatusAddr;
    /**< Interrupt Aggregator enable address */
    uint32    intrAggClearMask;
    /**< Interrupt Aggregator clear mask */
    uint32    intrEnable;
    /**< Enable interrupt mode */
} Cdd_EDMALLD_InitObject, *Cdd_EDMALLD_InitHandle;

/**
 *  \brief EDMA driver object
 */
typedef struct
{
    /*
     * SOC configuration
     */
    uint32                baseAddr;
    /**< Peripheral base address */

    Cdd_EDMALLD_InitHandle         hEdmaInit;
    /**< [IN] Initialization parameters of QSPI instance */

    /*
     * User parameters
     */
    uint32                isOpen;
    /**< Flag to indicate whether the instance is opened already */
    CDD_EDMA_ResourceObject     allocResource;
    /**< Object to hold the allocated resources */
    Cdd_Edma_IntrHandle  firstIntr;
    /**< Used for thread protection during interrupt callback registration. */
    uint32    intRegisterMutex;
} Cdd_EDMALLD_Object, *Cdd_EDMALLD_Handle;

extern Cdd_EDMALLD_Object gEdmaObject;
extern Cdd_EDMALLD_Handle gEdmaHandle; 
/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/**
 *  \brief   Clear a PaRAM Set .
 *
 *  \param   paramEntry               Parameter RAM set to be cleared.
 *
 */
void CDD_EDMA_lld_ccPaRAMEntry_init(CDD_EDMACCEDMACCPaRAMEntry *paramEntry);

/**
 * \brief  Enable channel to Shadow region mapping
 *
 * This API allocates DMA/QDMA channels or TCCs, and the same resources are
 * enabled in the shadow region specific register (DRAE/DRAEH/QRAE).
 * Here only one shadow region is used since, there is only one Master.
 *
 *  \param   baseAddr     Memory address of the EDMA instance used.\n
 *
 *  \param   regionId     Region id to be used.
 *                        Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   chType       (DMA/QDMA) Channel
 *                        For Example: For DMA it is,
 *                        CDD_EDMA_CHANNEL_TYPE_DMA.\n
 *
 *  \param   chNum       Allocated channel number.
 *                       Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *                       Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA \n
 *  chType can have values
 *        CDD_EDMA_CHANNEL_TYPE_DMA\n
 *        CDD_EDMA_CHANNEL_TYPE_QDMA
 *
 */
void CDD_EDMA_lld_enableChInShadowRegRegion(uint32 baseAddr,
                                    uint32 regionId,
                                    uint32 chType,
                                    uint32 chNum);

/**
 * \brief  Disable channel to Shadow region mapping
 *
 * This API deallocates DMA/QDMA channels or TCCs, and the same resources are
 * disabled in the shadow region specific register (DRAE/DRAEH/QRAE).
 * Here only one shadow region is used since, there is only one Master.
 *
 * \param   baseAddr   Memory address of the EDMA instance used.\n
 *
 * \param   regionId     Region id to be used.
*                        Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 * \param   chType      (DMA/QDMA) Channel
 *
 * \param   chNum       Allocated channel number.
 *                      Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *                      Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA \n
 *
 *
 *  chType can have values
 *        CDD_EDMA_CHANNEL_TYPE_DMA\n
 *        CDD_EDMA_CHANNEL_TYPE_QDMA
 *
 */
void CDD_EDMA_lld_disableChInShadowRegRegion(uint32 baseAddr,
                                     uint32 regionId,
                                     uint32 chType,
                                     uint32 chNum);

/**
 *  \brief   This function maps DMA channel to any of the PaRAM sets
 *           in the PaRAM memory map.
 *
 *  \param   baseAddr   Memory address of the EDMA instance used.
 *
 *  \param   channel   The DMA channel number required to be mapped.
 *
 *  \param   paramSet  It specifies the paramSet to which DMA channel
 *                     required to be mapped.
 *                     Valid values are 0 to CDD_EDMA_NUM_PARAMSETS-1
 *
 */
void CDD_EDMA_lld_channelToParamMap(uint32 baseAddr,
                            uint32 channel,
                            uint32 paramSet);

/**
 *  \brief  Map channel to Event Queue
 *
 *  This API maps DMA/QDMA channels to the Event Queue
 *
 *  \param  baseAddr    Memory address of the EDMA instance used.\n
 *
 *  \param  chType     (DMA/QDMA) Channel
 *                     For Example: For QDMA it is
 *                     CDD_EDMA_CHANNEL_TYPE_QDMA.\n
 *
 *  \param   chNum     Allocated channel number.
 *                     Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *                     Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA \n
 *
 *  \param  evtQNum    Event Queue Number to which the channel
 *                     will be mapped (valid only for the
 *                     Master Channel (DMA/QDMA) request).\n
 *
 *  chtype can have values
 *        CDD_EDMA_CHANNEL_TYPE_DMA\n
 *        CDD_EDMA_CHANNEL_TYPE_QDMA
 *
 */
void CDD_EDMA_lld_mapChToEvtQ(uint32 baseAddr,
                      uint32 chType,
                      uint32 chNum,
                      uint32 evtQNum);

/**
 *  \brief  Remove Mapping of channel to Event Queue
 *
 *  This API Unmaps DMA/QDMA channels from the Event Queue allocated
 *
 *  \param  baseAddr    Memory address of the EDMA instance used.\n
 *
 *  \param  chType     (DMA/QDMA) Channel
 *                     For Example: For DMA it is
 *                     CDD_EDMA_CHANNEL_TYPE_DMA.\n
 *
 *  \param   chNum     Allocated channel number.
 *                     Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *                     Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA \n
 *
 *  chtype can have values
 *        CDD_EDMA_CHANNEL_TYPE_DMA\n
 *        CDD_EDMA_CHANNEL_TYPE_QDMA
 *
 */
void CDD_EDMA_lld_unmapChToEvtQ(uint32 baseAddr,
                        uint32 chType,
                        uint32 chNum);

/**
 *  \brief  Enables the user to map a QDMA channel to PaRAM set
 *          This API Needs to be called before programming the paRAM sets for
 *          the QDMA Channels.Application needs to maitain the paRAMId
 *          provided by this API.This paRAMId is used to set paRAM and get
 *          paRAM. Refer corresponding API's for more details.
 *
 *  \param  baseAddr                  Memory address of the EDMA instance used.\n
 *
 *  \param   chNum                    Allocated channel number.
 *                                    Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA \n
 *
 *  \param  paRAMId                  PaRAM Id to which the QDMA channel will be
 *                                   mapped to.
 *                                   mapped to.
 *
 *  Note : The PaRAMId requested must be greater than 32(CDD_EDMA_NUM_DMACH).
 *         and lesser than CDD_EDMA_NUM_DMACH + chNum  Because, the first
 *         32 PaRAM's are directly mapped to first 32 DMA channels and (32 - 38)
 *         for QDMA Channels. (32 - 38) is assigned by driver in this API.
 *
 */
void CDD_EDMA_lld_mapQdmaChToPaRAM(uint32        baseAddr,
                           uint32        chNum,
                           const uint32 *paRAMId);

/**
 *  \brief  Returns the PaRAM associated with the DMA/QDMA channel.
 *
 * \param  baseAddr     Memory address of the EDMA instance used.\n
 *
 * \param   chNum       Allocated channel number.
 *                      Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *                      Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA \n
 *
 * \param  chType       (DMA/QDMA) Channel
 *                       For Example: For QDMA it is, CDD_EDMA_CHANNEL_TYPE_QDMA.\n
 *
 * \param  paramId      Associated paramId is returned in this pointer.
 *
 * \return  TRUE if parameters are valid else return FALSE.
 *
 */
uint32 CDD_EDMA_lld_getMappedPaRAM(uint32 baseAddr,
                             uint32 chNum,
                             uint32 chType,
                             uint32 *paramId);
/**
 * \brief  Assign a Trigger Word to the specified QDMA channel
 *
 * This API sets the Trigger word for the specific QDMA channel in the QCHMAP
 * Register. Default QDMA trigger word is CCNT.
 *
 * \param  baseAddr             Memory address of the EDMA instance used.\n
 *
 * \param  chNum               QDMA Channel which needs to be assigned
 *                             the Trigger Word.
 *                             Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA \n
 *
 * \param  trigWord            The Trigger Word for the QDMA channel.
 *                             Trigger Word is the word in the PaRAM
 *                             Register Set which, when written to by CPU,
 *                             will start the QDMA transfer automatically.
 *
 */
void CDD_EDMA_lld_setQdmaTrigWord(uint32 baseAddr,
                          uint32 chNum,
                          uint32 trigWord);

/**
 *  \brief   Enables the user to Clear any missed event
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   chNum                  Allocated channel number.\n
 *                                  Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 */
void CDD_EDMA_lld_clrMissEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum);

/**
 *  \brief   Enables the user to Clear any QDMA missed event
 *
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   chNum                  Allocated channel number.\n
 *                                  Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA
 *
 */
void CDD_EDMA_lld_qdmaClrMissEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum);

/**
 *  \brief   Enables the user to Clear any Channel controller Errors
 *
 *  \param   baseAddr              Memory address of the EDMA instance used.\n
 *
 *  \param   flags                Masks to be passed.\n
 *
 *  flags can have values:
 *
 *  CDD_EDMACC_CLR_TCCERR            Clears the TCCERR bit in the EDMACC
 *                                ERR Reg\n
 *  CDD_EDMACC_CLR_QTHRQ0            Queue threshold error clear for queue 0.\n
 *  CDD_EDMACC_CLR_QTHRQ1            Queue threshold error clear for queue 1.
 *
 */
void CDD_EDMA_lld_clrCCErr(uint32 baseAddr, uint32 flags);

/**
 *  \brief   Enables the user to Set an event. This API helps user to manually
 *           set events to initiate DMA transfer requests.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.\n
 *
 *  \param   chNum                  Allocated channel number.
 *                                  Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *
 *  Note :   This API is generally used during Manual transfers.\n
 */
void CDD_EDMA_lld_setEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum);

/**
 *  \brief   Enables the user to Clear an event.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   chNum                  Allocated channel number.\n
 *                                  Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *
 *  Note :   This API is generally used during Manual transfers.\n
 */
void CDD_EDMA_lld_clrEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum);

/**
 *  \brief   Enables the user to enable an DMA event.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   chNum                  Allocated channel number.
 *                                  Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *
 *  Note :   Writes of 1 to the bits in EESR sets the corresponding event
 *           bits in EER. This is generally used for Event Based transfers.\n
 */
void CDD_EDMA_lld_enableDmaEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum);

/**
 *  \brief   Enables the user to Disable an DMA event.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   chNum                  Allocated channel number.
 *                                  Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *
 *  Note :   Writes of 1 to the bits in EECR clear the corresponding event bits
 *           in EER; writes of 0 have no effect.. This is generally used for
 *           Event Based transfers.\n
 */
void CDD_EDMA_lld_disableDmaEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum);

/**
 *  \brief   Enables the user to enable an QDMA event.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   chNum                  Allocated channel number.
 *                                  Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA
 *
 *  Note :   Writes of 1 to the bits in QEESR sets the corresponding event
 *            bits in QEER.\n
 */
void CDD_EDMA_lld_enableQdmaEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum);

/**
 *  \brief   Enables the user to disable an QDMA event.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   chNum                  Allocated channel number.
 *                                  Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA
 *
 *  Note :   Writes of 1 to the bits in QEECR clears the corresponding event
 *            bits in QEER.\n
 */
void CDD_EDMA_lld_disableQdmaEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum);

/**
 *  \brief   This function returns interrupts status of those events
 *           which is less than 32.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 * \return   Interrupt status register value
 *
 **/
uint32 CDD_EDMA_lld_getIntrStatusRegion(uint32 baseAddr, uint32 regionId);

/**
 *  \brief   Enables the user to enable the transfer completion interrupt
 *           generation by the EDMACC for all DMA/QDMA channels.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   chNum                  Allocated channel number.
 *                                  Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *
 *  Note :   To set any interrupt bit in IER, a 1 must be written to the
 *           corresponding interrupt bit in the interrupt enable set register.
 */
void CDD_EDMA_lld_enableEvtIntrRegion(uint32 baseAddr, uint32 regionId, uint32 chNum);

/**
 *  \brief   Enables the user to clear CC interrupts
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   chNum                  Allocated channel number.
 *                                  Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *
 *  Note :   Writes of 1 to the bits in IECR clear the corresponding interrupt
 *           bits in the interrupt enable registers (IER); writes of 0 have
 *           no effect.\n
 */
void CDD_EDMA_lld_disableEvtIntrRegion(uint32 baseAddr, uint32 regionId, uint32 chNum);

/**
 *  \brief   Enables the user to Clear an Interrupt.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param   value                  Value to be set to clear the Interrupt
 *                                  Status.
 *
 */
void CDD_EDMA_lld_clrIntrRegion(uint32 baseAddr, uint32 regionId, uint32 value);

/**
 *  \brief   This function returns interrupt enable status of events which
 *           are less than 32.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 * \return   Interrupt status register value
 *
 **/
uint32 CDD_EDMA_lld_getEnabledIntrRegion(uint32 baseAddr, uint32 regionId);

/**
 *  \brief   This function returns interrupt enable status of events which
 *           are more than 32.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId               Region id to be used.
 *                                  Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 * \return   Interrupt status register value
 *
 **/
uint32 CDD_EDMA_lld_getEnabledIntrHighRegion(uint32 baseAddr, uint32 regionId);

/**
 *  \brief   Retrieve existing PaRAM set associated with specified logical
 *           channel (DMA/Link).
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   paRAMId                paRAMset ID whose parameter set is
 *                                  requested.
 *                                  Valid values are 0 to CDD_EDMA_NUM_PARAMSETS-1\n
 *
 *  \param   currPaRAM              User gets the existing PaRAM here.\n
 *
 */
void CDD_EDMA_lld_getPaRAM(uint32           baseAddr,
                   uint32           paRAMId,
                   CDD_EDMACCEDMACCPaRAMEntry *currPaRAM);

/**
 * \brief   Retrieve existing PaRAM set associated with specified logical
 *          channel (QDMA).
 *
 * \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 * \param   paRAMId                paRAMset ID whose parameter set is
 *                                 requested.
 *                                 Valid values are 0 to CDD_EDMA_NUM_PARAMSETS-1\n
 *
 * \param   currPaRAM              User gets the existing PaRAM here.\n
 *
 */
void CDD_EDMA_lld_qdmaGetPaRAM(uint32           baseAddr,
                       uint32           paRAMId,
                       CDD_EDMACCEDMACCPaRAMEntry *currPaRAM);

/**
 * \brief   Copy the user specified PaRAM Set onto the PaRAM Set associated
 *          with the logical channel (DMA/Link).
 *
 * This API takes a PaRAM Set as input and copies it onto the actual PaRAM Set
 * associated with the logical channel. OPT field of the PaRAM Set is written
 * first and the CCNT field is written last.
 *
 *
 * \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 * \param   paRAMId                paRAMset ID whose parameter set has to be
 *                                 updated
 *                                 Valid values are 0 to CDD_EDMA_NUM_PARAMSETS-1
 *
 * \param   newPaRAM               Parameter RAM set to be copied onto existing
 *                                 PaRAM.\n
 *
 */
void CDD_EDMA_lld_setPaRAM(uint32           baseAddr,
                   uint32           paRAMId,
                   const CDD_EDMACCEDMACCPaRAMEntry *newPaRAM);

/**
 * \brief   Copy the user specified PaRAM Set onto the PaRAM Set associated
 *          with the logical channel (QDMA only).
 *
 * This API takes a PaRAM Set as input and copies it onto the actual PaRAM Set
 * associated with the logical channel. OPT field of the PaRAM Set is written
 * first and the CCNT field is written last.
 *
 *
 * \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *
 * \param  paRAMId                 paRaMset ID whose parameter set has to be
 *                                 updated
 *                                 Valid values are 0 to CDD_EDMA_NUM_PARAMSETS-1
 *
 * \param   newPaRAM               Parameter RAM set to be copied onto existing
 *                                 PaRAM.\n
 *
 */
void CDD_EDMA_lld_qdmaSetPaRAM(uint32           baseAddr,
                       uint32           paRAMId,
                       const CDD_EDMACCEDMACCPaRAMEntry *newPaRAM);

/**
 * \brief   Set a particular PaRAM set entry of the specified PaRAM set
 *
 * \param   baseAddr           Memory address of the EDMA instance used.\n
 *
 * \param   paRAMId           PaRAM Id to which the QDMA channel is
 *                            mapped to.
 *
 * \param   paRAMEntry        Specify the PaRAM set entry which needs
 *                            to be set.
 *
 * \param   newPaRAMEntryVal  The new field setting. Make sure this field is
 *                            packed for setting certain fields in paRAM.
 *
 *  CDD_EDMACC_PARAM_ENTRY_OPT
 *  CDD_EDMACC_PARAM_ENTRY_SRC
 *  CDD_EDMACC_PARAM_ENTRY_ACNT_BCNT
 *  CDD_EDMACC_PARAM_ENTRY_DST
 *  CDD_EDMACC_PARAM_ENTRY_SRC_DST_BIDX
 *  CDD_EDMACC_PARAM_ENTRY_LINK_BCNTRLD
 *  CDD_EDMACC_PARAM_ENTRY_SRC_DST_CIDX
 *  CDD_EDMACC_PARAM_ENTRY_CCNT
 *
 * \note    This API should be used while setting the PaRAM set entry
 *          for QDMA channels. If EDMAQdmaSetPaRAMEntry() used,
 *          it will trigger the QDMA channel before complete
 *          PaRAM set entry is written.
 */
void CDD_EDMA_lld_qdmaSetPaRAMEntry(uint32 baseAddr,
                            uint32 paRAMId,
                            uint32 paRAMEntry,
                            uint32 newPaRAMEntryVal);

/**
 * \brief   Get a particular PaRAM entry of the specified PaRAM set
 *
 * \param   baseAddr           Memory address of the EDMA instance used.\n
 *
 * \param   paRAMId           PaRAM Id to which the QDMA channel is
 *                            mapped to.
 *
 * \param   paRAMEntry        Specify the PaRAM set entry which needs
 *                            to be read.
 *
 *  paRAMEntry can have values:
 *
 *  CDD_EDMACC_PARAM_ENTRY_OPT
 *  CDD_EDMACC_PARAM_ENTRY_SRC
 *  CDD_EDMACC_PARAM_ENTRY_ACNT_BCNT
 *  CDD_EDMACC_PARAM_ENTRY_DST
 *  CDD_EDMACC_PARAM_ENTRY_SRC_DST_BIDX
 *  CDD_EDMACC_PARAM_ENTRY_LINK_BCNTRLD
 *  CDD_EDMACC_PARAM_ENTRY_SRC_DST_CIDX
 *  CDD_EDMACC_PARAM_ENTRY_CCNT
 *
 * \return  paRAMEntryVal     The value of the paRAM field pointed by the
 *                            paRAMEntry.
 *
 * \note    This API should be used while reading the PaRAM set entry
 *          for QDMA channels. And the paRAMEntryVal is a packed value for
 *          certain fields of paRAMEntry.The user has to make sure the value
 *          is unpacked appropriately.
 *          For example, the third field is A_B_CNT. Hence he will have to
 *          unpack it to two 16 bit fields to get ACNT and BCNT.
 */
uint32 CDD_EDMA_lld_qdmaGetPaRAMEntry(uint32 baseAddr,
                                uint32 paRAMId,
                                uint32 paRAMEntry);

/**
 * \brief   Set a particular PaRAM set entry of the specified PaRAM set
 *
 * \param   baseAddr           Memory address of the EDMA instance used.\n
 *
 * \param   paRAMId           PaRAM Id to which the DMA channel is
 *                            mapped to.
 *
 * \param   paRAMEntry        Specify the PaRAM set entry which needs
 *                            to be set.
 *
 * \param   newPaRAMEntryVal  The new field setting. Make sure this field is
 *                            packed for setting certain fields in paRAM.
 *
 *  CDD_EDMACC_PARAM_ENTRY_OPT
 *  CDD_EDMACC_PARAM_ENTRY_SRC
 *  CDD_EDMACC_PARAM_ENTRY_ACNT_BCNT
 *  CDD_EDMACC_PARAM_ENTRY_DST
 *  CDD_EDMACC_PARAM_ENTRY_SRC_DST_BIDX
 *  CDD_EDMACC_PARAM_ENTRY_LINK_BCNTRLD
 *  CDD_EDMACC_PARAM_ENTRY_SRC_DST_CIDX
 *  CDD_EDMACC_PARAM_ENTRY_CCNT
 *
 * \note    This API should be used while setting the PaRAM set entry
 *          for DMA channels.
 */
void CDD_EDMA_lld_dmaSetPaRAMEntry(uint32 baseAddr,
                            uint32 paRAMId,
                            uint32 paRAMEntry,
                            uint32 newPaRAMEntryVal);

/**
 * \brief   Get a particular PaRAM entry of the specified PaRAM set
 *
 * \param   baseAddr           Memory address of the EDMA instance used.\n
 *
 * \param   paRAMId           PaRAM Id to which the DMA channel is
 *                            mapped to.
 *
 * \param   paRAMEntry        Specify the PaRAM set entry which needs
 *                            to be read.
 *
 *  paRAMEntry can have values:
 *
 *  CDD_EDMACC_PARAM_ENTRY_OPT
 *  CDD_EDMACC_PARAM_ENTRY_SRC
 *  CDD_EDMACC_PARAM_ENTRY_ACNT_BCNT
 *  CDD_EDMACC_PARAM_ENTRY_DST
 *  CDD_EDMACC_PARAM_ENTRY_SRC_DST_BIDX
 *  CDD_EDMACC_PARAM_ENTRY_LINK_BCNTRLD
 *  CDD_EDMACC_PARAM_ENTRY_SRC_DST_CIDX
 *  CDD_EDMACC_PARAM_ENTRY_CCNT
 *
 * \return  paRAMEntryVal     The value of the paRAM field pointed by the
 *                            paRAMEntry.
 *
 * \note    This API should be used while reading the PaRAM set entry
 *          for DMA channels. And the paRAMEntryVal is a packed value for
 *          certain fields of paRAMEntry.The user has to make sure the value
 *          is unpacked appropriately.
 *          For example, the third field is A_B_CNT. Hence he will have to
 *          unpack it to two 16 bit fields to get ACNT and BCNT.
 */
uint32 CDD_EDMA_lld_dmaGetPaRAMEntry(uint32 baseAddr,
                                uint32 paRAMId,
                                uint32 paRAMEntry);

/**
 *  \brief Request a DMA/QDMA/Link channel.
 *
 *  Each channel (DMA/QDMA/Link) must be requested  before initiating a DMA
 *  transfer on that channel.
 *
 *  This API is used to allocate a logical channel (DMA/QDMA/Link) along with
 *  the associated resources. For DMA and QDMA channels, TCC and PaRAM Set are
 *  also allocated along with the requested channel.
 *
 *  User can request a specific logical channel by passing the channel number
 *  in 'chNum'.
 *
 *  For DMA/QDMA channels, after allocating all the EDMA resources, this API
 *  sets the TCC field of the OPT PaRAM Word with the allocated TCC. It also
 *  sets the event queue for the channel allocated. The event queue needs to
 *  be specified by the user.
 *
 *  For DMA channel, it also sets the DCHMAP register.
 *
 *  For QDMA channel, it sets the QCHMAP register and CCNT as trigger word and
 *  enables the QDMA channel by writing to the QEESR register.
 *
 *  \param  baseAddr                  Memory address of the EDMA instance used.\n
 *
 *  \param   regionId                 Region id to be used.
 *                                    Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param  chType                   (DMA/QDMA) Channel
 *                                    For Example: For DMA it is
 *                                    CDD_EDMA_CHANNEL_TYPE_DMA.\n
 *
 *  \param  chNum                    This is the channel number requested for a
 *                                   particular event.
 *                                   Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *                                   Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA \n
 *
 *  \param  tccNum                   The tcc number on which the
 *                                   completion/error interrupt is generated.
 *                                   Not used if user requested for a Link
 *                                   channel.\n
 *
 *  \param  paramId                  The param number currosponding to the
 *                                   DMA/QDMA channel.\n
 *
 *  \param  evtQNum                  Event Queue Number to which the channel
 *                                   will be mapped (valid only for the
 *                                   Master Channel (DMA/QDMA) request).\n
 *
 *  \return  TRUE if parameters are valid, else FALSE
 */
uint32 CDD_EDMA_lld_configureChannelRegion(uint32 baseAddr,
                                     uint32 regionId,
                                     uint32 chType,
                                     uint32 chNum,
                                     uint32 tccNum,
                                     uint32 paramId,
                                     uint32 evtQNum);

/**
 *  \brief    Free the specified channel (DMA/QDMA/Link) and its associated
 *            resources (PaRAM Set, TCC etc) and removes various mappings.
 *
 *  For Link channels, this API only frees the associated PaRAM Set.
 *
 *  For DMA/QDMA channels, it does the following operations:
 *  1) Disable any ongoing transfer on the channel,\n
 *  2) Remove the channel to Event Queue mapping,\n
 *  3) For DMA channels, clear the DCHMAP register, if available\n
 *  4) For QDMA channels, clear the QCHMAP register,\n
 *  5) Frees the DMA/QDMA channel in the end.\n
 *
 *  \param  baseAddr                  Memory address of the EDMA instance used.\n
 *
 *  \param   regionId                 Region id to be used.
 *                                    Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param  chType                    (DMA/QDMA) Channel
 *                                      For Example: For QDMA it is,
 *                                      CDD_EDMA_CHANNEL_TYPE_QDMA.\n
 *
 *  \param  chNum                    This is the channel number requested for a
 *                                   particular event.
 *                                   Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *                                   Valid values are 0 to CDD_EDMA_NUM_QDMACH-1 for QDMA \n
 *
 *  \param  trigMode                 Mode of triggering start of transfer.\n
 *
 *  \param  tccNum                   The channel number on which the
 *                                   completion/error interrupt is generated.
 *                                   Not used if user requested for a Link
 *                                   channel.\n
 *
 *  \param  evtQNum                  Event Queue Number to which the channel
 *                                   will be unmapped (valid only for the
 *                                   Master Channel (DMA/QDMA) request).\n
 *
 *  trigMode can have values:
 *        CDD_EDMA_TRIG_MODE_MANUAL\n
 *        CDD_EDMA_TRIG_MODE_QDMA\n
 *        CDD_EDMA_TRIG_MODE_EVENT
 *
 *  \return  TRUE if parameters are valid else return FALSE
 */
uint32 CDD_EDMA_lld_freeChannelRegion(uint32 baseAddr,
                                uint32 regionId,
                                uint32 chType,
                                uint32 chNum,
                                uint32 trigMode,
                                uint32 tccNum,
                                uint32 evtQNum);

/**
 *  \brief    Start EDMA transfer on the specified channel.
 *
 *  There are multiple ways to trigger an EDMA transfer. The triggering mode
 *  option allows choosing from the available triggering modes: Event,
 *  Manual or QDMA.
 *
 *  In event triggered, a peripheral or an externally generated event triggers
 *  the transfer. This API clears the Event and Event Miss Register and then
 *  enables the DMA channel by writing to the EESR.
 *
 *  In manual triggered mode, CPU manually triggers a transfer by writing a 1
 *  in the Event Set Register ESR. This API writes to the ESR to start the
 *  transfer.
 *
 *  In QDMA triggered mode, a QDMA transfer is triggered when a CPU (or other
 *  EDMA programmer) writes to the trigger word of the QDMA channel PaRAM set
 *  (auto-triggered) or when the EDMACC performs a link update on a PaRAM set
 *  that has been mapped to a QDMA channel (link triggered). This API enables
 *  the QDMA channel by writing to the QEESR register.
 *
 *  \param  baseAddr         Memory address of the EDMA instance used.\n
 *
 *  \param   regionId       Region id to be used.
*                           Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param  chNum           Channel being used to enable transfer.
 *                          Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *
 *  \param  trigMode        Mode of triggering start of transfer (Manual,
 *                          QDMA or Event).\n
 *
 *  trigMode can have values:
 *        CDD_EDMA_TRIG_MODE_MANUAL\n
 *        CDD_EDMA_TRIG_MODE_QDMA\n
 *        CDD_EDMA_TRIG_MODE_EVENT\n
 *
 *  \return  retVal         TRUE or FALSE depending on the param passed.\n
 *
 */
uint32 CDD_EDMA_lld_enableTransferRegion(uint32 baseAddr,
                                   uint32 regionId,
                                   uint32 chNum,
                                   uint32 trigMode);

/**
 *  \brief   Disable DMA transfer on the specified channel
 *
 *  There are multiple ways by which an EDMA transfer could be triggered.
 *  The triggering mode option allows choosing from the available triggering
 *  modes.
 *
 *  To disable a channel which was previously triggered in manual mode,
 *  this API clears the Secondary Event Register and Event Miss Register,
 *  if set, for the specific DMA channel.
 *
 *  To disable a channel which was previously triggered in QDMA mode, this
 *  API clears the QDMA Event Enable Register, for the specific QDMA channel.
 *
 *  To disable a channel which was previously triggered in event mode, this API
 *  clears the Event Enable Register, Event Register, Secondary Event Register
 *  and Event Miss Register, if set, for the specific DMA channel.
 *
 *
 *  \param  baseAddr         Memory address of the EDMA instance used.\n
 *
 *  \param   regionId       Region id to be used.
 *                          Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param  chNum           Channel being used to enable transfer.
 *                          Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *
 *  \param  trigMode        Mode of triggering start of transfer (Manual,
 *                          QDMA or Event).\n
 *  \return  retVal         TRUE or FALSE depending on the param passed.\n
 *
 */
uint32 CDD_EDMA_lld_disableTransferRegion(uint32 baseAddr,
                                    uint32 regionId,
                                    uint32 chNum,
                                    uint32 trigMode);

/**
 *  \brief  Clears Event Register and Error Register for a specific
 *          DMA channel and brings back EDMA to its initial state.
 *
 *  This API clears the Event register, Event Miss register, Event Enable
 *  register for a specific DMA channel. It also clears the CC Error register.
 *
 *  \param  baseAddr         Memory address of the EDMA instance used.\n
 *
 *  \param   regionId       Region id to be used.
 *                          Valid values are 0 to CDD_EDMA_NUM_REGIONS-1 \n
 *
 *  \param  chNum           This is the channel number requested for a
 *                          particular event.
 *                          Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA\n
 *
 *  \param  evtQNum         Event Queue Number to which the channel
 *                          will be unmapped (valid only for the
 *                          Master Channel (DMA/QDMA) request).\n
 *
 */
void CDD_EDMA_lld_clearErrorBitsRegion(uint32 baseAddr,
                               uint32 regionId,
                               uint32 chNum,
                               uint32 evtQNum);

/**
 *  \brief   This returns EDMA CC error status.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \return  value                  Status of the Interrupt Pending Register
 *
 */
uint32 CDD_EDMA_lld_getCCErrStatus(uint32 baseAddr);

/**
 *  \brief   This returns error interrupt status for those events whose
 *           event number is less than 32.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \return  value                  Status of the Interrupt Pending Register
 *
 */
uint32 CDD_EDMA_lld_getErrIntrStatus(uint32 baseAddr);

/**
 *  \brief   This returns QDMA error interrupt status.
 *
 *  \param   baseAddr            Memory address of the EDMA instance used.\n
 *
 *  \return  value              Status of the QDMA Interrupt Pending Register
 *
 */
uint32 CDD_EDMA_lld_qdmaGetErrIntrStatus(uint32 baseAddr);

/**
 * \brief   This API return the revision Id of the peripheral.
 *
 * \param   baseAddr     Memory address of the EDMA instance used.\n
 *
 * \return  value        Revision ID of the peripheral
 **/
uint32 CDD_EDMA_lld_peripheralIdGet(uint32 baseAddr);

/**
 *  \brief   This function returns interrupt status of those events
 *           which are greater than 32.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId                Region id to be used.\n
 *
 *  \return  value                   Status of the Interrupt Pending High Register
 *
 **/
uint32 CDD_EDMA_lld_intrStatusHighGetRegion(uint32 baseAddr, uint32 regionId);

/**
 *  \brief   This function reads interrupt status.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   regionId                Region id to be used.\n
 *
 *  \param   tccNum                  The channel number on which the
 *                                   completion/error interrupt is generated.
 *
 *  \return  value                   Status of the Interrupt Pending register
 *
 **/
uint32 CDD_EDMA_lld_readIntrStatusRegion(uint32 baseAddr, uint32 regionId, uint32 tccNum);

/**
 *  \brief   This function returns status of those events
 *           which are less than 32.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 * \return   value                   Event pending status register value
 *
 **/
uint32 CDD_EDMA_lld_getEventStatus(uint32 baseAddr);


/**
 *  \brief   This function returns status of those events
 *           which are greater than 32.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \return  value                   Status of the Event Pending High Register
 *
 **/
uint32 CDD_EDMA_lld_getEventStatusHigh(uint32 baseAddr);

/**
 *  \brief   This function reads Event pending status.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \param   chNum                   Channel being used for transfer.
 *                                   Valid values are 0 to CDD_EDMA_NUM_DMACH-1 for DMA
 *
 *  \return  value                   Status of the Event Pending register
 *
 **/
uint32 CDD_EDMA_lld_readEventStatusRegion(uint32 baseAddr, uint32 chNum);

/**
 *  \brief   This returns error interrupt status for those events whose
 *           event number is greater than 32.
 *
 *  \param   baseAddr                Memory address of the EDMA instance used.\n
 *
 *  \return  value                  Status of the Interrupt Pending Register
 *
 */
uint32 CDD_EDMA_lld_errIntrHighStatusGet(uint32 baseAddr);

/**
 *  \brief  Chain the two specified channels
 *
 *  This API is used to chain a DMA channel to a previously allocated DMA/QDMA
 *  channel
 *
 *  Chaining is different from Linking. The EDMA link feature reloads the
 *  current channel parameter set with the linked parameter set. The EDMA
 *  chaining feature does not modify or update any channel parameter set;
 *  it provides a synchronization event (or trigger) to the chained DMA channel,
 *  as soon as the transfer (final or intermediate) completes on the main
 *  DMA/QDMA channel.
 *
 *  \param  baseAddr         Memory address of the EDMA instance used.\n
 *
 *  \param  paRAMId1        PaRAM set ID of physical channel1 to which
 *                          particular paRAM set will be chained
 *                          or
 *                          PaRAM set ID in case another PaRAM set is being
 *                          chained to this PaRAM set
 *
 *  \param  chId2           DMA channel which needs to be chained to
 *                          the first DMA/QDMA channel.
 *
 *  \param   chainOptions   combination of the following masks which control
 *                          individual PaRAM OPT fields related to
 *                          intermediate/final completion chaining and
 *                          intermediate/final completion interrupt :
 *                            - CDD_EDMA_OPT_TCCHEN_MASK
 *                            - CDD_EDMA_OPT_ITCCHEN_MASK
 *                            - CDD_EDMA_OPT_TCINTEN_MASK
 *                            - CDD_EDMA_OPT_ITCINTEN_MASK
 *                          e.g to enable final completion chaining and enable
 *                          interrupt only for intermediate completion,
 *                          set chainOptions as :
 *                          CDD_EDMA_OPT_TCCHEN_MASK | CDD_EDMA_OPT_ITCINTEN_MASK
 *
 */
void CDD_EDMA_lld_chainChannel(uint32 baseAddr,
                       uint32 paRAMId1,
                       uint32 chId2,
                       uint32 chainOptions);

/**
 *  \brief  Link two channels.
 *
 *  This API is used to link two previously allocated logical (DMA/QDMA/Link)
 *  channels.
 *
 *  It sets the Link field of the PaRAM set associated with first
 *  channel (chId1) to point it to the PaRAM set associated with second
 *  channel (chId2).
 *
 *  It also sets the TCC field of PaRAM set of second channel to the
 *  same as that of the first channel.
 *
 *  \param  baseAddr         Memory address of the EDMA instance used.\n
 *
 *  \param  paRAMId1        PaRAM set ID of physical channel1 to which
 *                          particular paRAM set will be linked
 *                          or
 *                          PaRAM set ID in case another PaRAM set is being
 *                          linked to this PaRAM set
 *
 *  \param  paRAMId2        PaRAM set ID which is linked to
 *                          channel with parameter ID paRAMId1
 *
 *                          After the transfer based on the PaRAM set
 *                          of channel1 is over, the PaRAM set paRAMId2 will
 *                          be copied to the PaRAM set of channel1 and
 *                          transfer will resume.
 *                          For DMA channels, another sync event is
 *                          required to initiate the transfer on the
 *                          Link channel.
 *
 */
void CDD_EDMA_lld_linkChannel(uint32 baseAddr, uint32 paRAMId1, uint32 paRAMId2);

/**
 *  \brief  This function initializes the EDMA instance
 * 
 *  \param  hEdma      Handle to the EDMA instance used
 */
void CDD_EDMA_lld_init(Cdd_EDMALLD_Handle hEdma);

/**
 *  \brief  This function Deinitializes the EDMA instance
 *
 *  \param  hEdma      Handle to the EDMA instance used
 */
void CDD_EDMA_lld_deInit(Cdd_EDMALLD_Handle hEdma);

/**
 *  \brief  Function to check if EDMA interrupt is enabled.
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *
 *  \sa     #CDD_EDMA_open()
 */
uint32 CDD_EDMA_lld_isInterruptEnabled(Cdd_EDMALLD_Handle hEdma);

/**
 *  \brief  Function to check if EDMA initialization is done or not.
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *
 *  \sa     #CDD_EDMA_open()
 */
uint32 CDD_EDMA_lld_isInitialized(Cdd_EDMALLD_Handle hEdma);

/**
 *  \brief  Function to register callback function for a TCC
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *  \param  intrObj     Pointer to interrupt parameters. intrObj will be held
 *                      by the driver till #CDD_EDMA_unregisterIntr() is called
 *                      with the same intrObj and application should not modify
 *                      the memory allocated to intrObj.
 *
 *  \return #MCAL_SystemP_SUCCESS if started successfully; else error on failure
 *
 *  \sa     #CDD_EDMA_open()
 */
sint32 CDD_EDMA_lld_registerIntr(Cdd_EDMALLD_Handle hEdma, Cdd_Edma_IntrObject *intrObj);

/**
 *  \brief  Function to unregister callback function for a TCC
 *
 *  \pre    #CDD_EDMA_registerIntr() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *  \param  intrObj     Pointer to interrupt parameters object. This is the
 *                      same object passed to #CDD_EDMA_registerIntr() for interrupt
 *                      registeration.
 *
 *  \return #MCAL_SystemP_SUCCESS if started successfully; else error on failure
 *
 *  \sa     #CDD_EDMA_open()
 *  \sa     #CDD_EDMA_registerIntr()
 */
sint32 CDD_EDMA_lld_unregisterIntr(Cdd_EDMALLD_Handle hEdma, Cdd_Edma_IntrObject *intrObj);

/**
 *  \brief  Function to get the edma base address
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *
 *  \return edma base address if successful;
 *          0 (NULL) on failure
 *
 *  \sa     #CDD_EDMA_open()
 */
uint32 CDD_EDMA_lld_getBaseAddr(Cdd_EDMALLD_Handle hEdma);

/**
 *  \brief  Function to get the edma region
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *
 *  \return region Id corresponding to the edma handle if successful
 *          CDD_EDMA_NUM_REGIONS upon failure which is invalid region number.
 *          valid region numbers are from 0 to (CDD_EDMA_NUM_REGIONS - 1)
 *
 *  \sa     #CDD_EDMA_open()
 */
uint32 CDD_EDMA_lld_getRegionId(Cdd_EDMALLD_Handle hEdma);

/**
 *  \brief  Function to allocate the Dma Channel
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *  \param  dmaCh       Set the value to DMA channel number to be allocated.
 *                      Set to CDD_EDMA_RESOURCE_ALLOC_ANY to allocate any owned
 *                      channel. Allocated channel number is stored in this.
 *
 *  \return #MCAL_SystemP_SUCCESS if started successfully; else error on failure
 *
 *  \sa     #CDD_EDMA_open()
 */
sint32 CDD_EDMA_lld_allocDmaChannel(Cdd_EDMALLD_Handle hEdma, uint32 *dmaCh);

/**
 *  \brief  Function to allocate the Dma Channel
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *  \param  qdmaCh      Set the value to DMA channel number to be allocated.
 *                      Set to CDD_EDMA_RESOURCE_ALLOC_ANY to allocate any owned
 *                      channel. Allocated channel number is stored in this.
 *
 *  \return #MCAL_SystemP_SUCCESS if started successfully; else error on failure
 *
 *  \sa     #CDD_EDMA_open()
 */
sint32 CDD_EDMA_lld_allocQdmaChannel(Cdd_EDMALLD_Handle hEdma, uint32 *qdmaCh);

/**
 *  \brief  Function to allocate the Qdma Channel
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *  \param  tcc         Set the value to QDMA channel number to be allocated.
 *                      Set to CDD_EDMA_RESOURCE_ALLOC_ANY to allocate any owned
 *                      Qdma channel. Allocated channel number is stored in this.
 *
 *  \return #MCAL_SystemP_SUCCESS if started successfully; else error on failure
 *
 *  \sa     #CDD_EDMA_open()
 */
sint32 CDD_EDMA_lld_allocTcc(Cdd_EDMALLD_Handle hEdma, uint32 *tcc);

/**
 *  \brief  Function to allocate the TCC
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *  \param  param       Set the value to TCC to be allocated.
 *                      Set to CDD_EDMA_RESOURCE_ALLOC_ANY to allocate any owned
 *                      TCC. Allocated TCC number is stored in this.
 *
 *  \return #MCAL_SystemP_SUCCESS if started successfully; else error on failure
 *
 *  \sa     #CDD_EDMA_open()
 */
sint32 CDD_EDMA_lld_allocParam(Cdd_EDMALLD_Handle hEdma, uint32 *param);

/**
 *  \brief  Function to free the Dma Channel
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *  \param  dmaCh       dma channel allocated using the call to
 *                      #CDD_EDMA_allocDmaChannel()
 *
 *  \return #MCAL_SystemP_SUCCESS if started successfully; else error on failure
 *
 *  \sa     #CDD_EDMA_open()
 */
sint32 CDD_EDMA_lld_freeDmaChannel(Cdd_EDMALLD_Handle hEdma, uint32 *dmaCh);

/**
 *  \brief  Function to free the Qdma Channel
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *  \param  qdmaCh      qdma channel allocated using the call to
 *                      #CDD_EDMA_allocQdmaChannel()
 *
 *  \return #MCAL_SystemP_SUCCESS if started successfully; else error on failure
 *
 *  \sa     #CDD_EDMA_open()
 */
sint32 CDD_EDMA_lld_freeQdmaChannel(Cdd_EDMALLD_Handle hEdma, uint32 *qdmaCh);

/**
 *  \brief  Function to free the tcc Channel
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *  \param  tcc         tcc channel allocated using the call to
 *                      #CDD_EDMA_allocTcc()
 *
 *  \return #MCAL_SystemP_SUCCESS if started successfully; else error on failure
 *
 *  \sa     #CDD_EDMA_open()
 */
sint32 CDD_EDMA_lld_freeTcc(Cdd_EDMALLD_Handle hEdma, uint32 *tcc);

/**
 *  \brief  Function to free the Param
 *
 *  \pre    #CDD_EDMA_open() has to be called first
 *
 *  \param  hEdma      #Cdd_EDMALLD_Handle returned from #CDD_EDMA_open()
 *  \param  param       param allocated using the call to
 *                      #CDD_EDMA_allocParam()
 *
 *  \return #MCAL_SystemP_SUCCESS if started successfully; else error on failure
 *
 *  \sa     #CDD_EDMA_open()
 */
sint32 CDD_EDMA_lld_freeParam(Cdd_EDMALLD_Handle hEdma, uint32 *param);

/**
 *  \brief  This is the EDMA ISR and can be used as IRQ handler.
 *
 *  \param  args      Argument to the ISR.
 *
 */
void    CDD_EDMA_lld_transferCompletionMasterIsrFxn(void);

#ifdef __cplusplus
}
#endif

#endif  /* #ifndef CDD_DMA_EDMA_V0_H_ */

/** @} */

