/********************************************************************
 * Copyright (C) 2020 Texas Instruments Incorporated.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Name        : soc_xbar.h
*/

#ifndef CDD_EDMA_XBAR_H
#define CDD_EDMA_XBAR_H

#ifdef __cplusplus
extern "C"
{
#endif

#include <stdint.h>
#include "Cdd_edma_trig_xbar.h"
#include "Cdd_edma_dmaxbar.h"
#include "Std_Types.h"
#include "hw_types.h"

/**
 *  \defgroup DRV_SOC_XBAR_MODULE APIs for SOC Xbars
 *  \ingroup DRV_SOC_MODULE
 *
 * For more details and example usage, see \ref DRIVERS_SOC_PAGE
 *
 *  @{
 */

#define CDD_DMA_CSL_CONTROLSS_DMAXBAR_STEP          (CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR1_GSEL - CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_GSEL)

/**
 * \brief Trip & Sync xbar: API to select input source of DMA XBar
 *
 * \param base [in] DMA XBar base address
 * \param out [in] Instance of DMA XBar
 * \param group0_muxctl [in] Mux control to select input from group 0 mux
 * \param group1_muxctl [in] Mux control to select input from group 1 mux
 * \param group2_muxctl [in] Mux control to select input from group 2 mux
 * \param group3_muxctl [in] Mux control to select input from group 3 mux
 * \param group4_muxctl [in] Mux control to select input from group 4 mux
 * \param group5_muxctl [in] Mux control to select input from group 5 mux
 * \param group_select [in] Mux control to select group 0/1/2/3/4/5
 *
 */
static inline void
CDD_DMA_SOC_xbarSelectDMAXBarInputSource(uint32 base, uint8 out, uint8 group_select, uint8 group0_muxctl, uint8 group1_muxctl, uint8 group2_muxctl, uint8 group3_muxctl, uint8 group4_muxctl, uint8 group5_muxctl)
{
    HW_WR_REG32(base + out*CDD_DMA_CSL_CONTROLSS_DMAXBAR_STEP + CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_GSEL , group_select &  CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_GSEL_GSEL_MASK);
    HW_WR_REG32(base + out*CDD_DMA_CSL_CONTROLSS_DMAXBAR_STEP + CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G0   , group0_muxctl & CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G0_SEL_MASK);
    HW_WR_REG32(base + out*CDD_DMA_CSL_CONTROLSS_DMAXBAR_STEP + CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G1   , group1_muxctl & CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G1_SEL_MASK);
    HW_WR_REG32(base + out*CDD_DMA_CSL_CONTROLSS_DMAXBAR_STEP + CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G2   , group2_muxctl & CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G2_SEL_MASK);
    HW_WR_REG32(base + out*CDD_DMA_CSL_CONTROLSS_DMAXBAR_STEP + CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G3   , group3_muxctl & CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G3_SEL_MASK);
    HW_WR_REG32(base + out*CDD_DMA_CSL_CONTROLSS_DMAXBAR_STEP + CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G4   , group4_muxctl & CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G4_SEL_MASK);
    HW_WR_REG32(base + out*CDD_DMA_CSL_CONTROLSS_DMAXBAR_STEP + CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G5   , group5_muxctl & CDD_DMA_CSL_CONTROLSS_DMAXBAR_DMAXBAR0_G5_SEL_MASK);

}

/**
 * \brief SoC level xbars: API to select input source of EDMA Trigger XBar
 *
 * \param base [in] EDMA Trigger XBar base address
 * \param out [in] Instance of EDMA Trigger XBar
 * \param mux_control [in] Mux control to select input source
 *
 */
static inline void
CDD_DMA_SOC_xbarSelectEdmaTrigXbarInputSource(uint32 base, uint8 out, uint8 mux_control)
{
    HW_WR_REG32(base + CDD_DMA_CSL_EDMA_TRIG_XBAR_MUXCNTL(out), (CDD_DMA_CSL_EDMA_TRIG_XBAR_MUXCNTL_INT_ENABLE_MASK) | (mux_control & CDD_DMA_CSL_EDMA_TRIG_XBAR_MUXCNTL_ENABLE_MASK));
}

/** @} */

#ifdef __cplusplus
}
#endif

#endif // SOC_XBAR_AM263X_H_
