/*
 * Copyright (C) 2023 Texas Instruments Incorporated
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 *   Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the
 *   distribution.
 *
 *   Neither the name of Texas Instruments Incorporated nor the names of
 *   its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 *  \file   CDD_Dma.c
 *
 *  \brief  This file contains device abstraction layer APIs for the DMA device.
 *          There are APIs here to enable the DMA instance, set the required
 *          configurations for communication, transmit or receive data.
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "Cdd_Dma_Cfg.h"
#include "hw_types.h"
#include "Cdd_Dma.h"
#include "stddef.h"

typedef struct
{
    uint32 cpuID;
    /**< CPU/core ID within cluster
     *   Refer \ref CSL_ArmR5CPUID
     */
    uint32 grpId;
    /**< Group Id of the cluster
     *   Refer \ref CSL_ArmR5ClusterGroupID
     */
}CDD_DMA_CSL_ArmR5CPUInfo;

typedef struct
{
    uint32 tcmaSize;
    uint32 tcmbSize;

    CDD_DMA_CSL_ArmR5CPUInfo cpuInfo;

}CDD_DMA_SOC_VirtToPhyMap;

#define CDD_DMA_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Cdd_Dma_MemMap.h"
Cdd_EDMALLD_Object gEdmaObject;
Cdd_EDMALLD_Handle gEdmaHandle;

CDD_DMA_SOC_VirtToPhyMap virtToPhymap;
#define CDD_DMA_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Cdd_Dma_MemMap.h"


#define CDD_DMA_START_SEC_VAR_INIT_8
#include "Cdd_Dma_MemMap.h"
uint8 Cdd_Dma_isMapAvailable = 0u;

//
// Global Init Done flag
//
static boolean Cdd_Dma_InitDone = FALSE;

#define CDD_DMA_STOP_SEC_VAR_INIT_8
#include "Cdd_Dma_MemMap.h"

#define CDD_DMA_CSL_MSS(reg, PER_REG_FIELD)                                        \
    (((reg) & CDD_DMA_CSL_##PER_REG_FIELD##_MASK) >> CDD_DMA_CSL_##PER_REG_FIELD##_SHIFT)

#define CDD_DMA_CSL_R5_MPIDR_AFF0_MASK                                   (0x000000FFU)
#define CDD_DMA_CSL_R5_MPIDR_AFF0_SHIFT                                  (0U)
#define CDD_DMA_CSL_R5_MPIDR_AFF1_MASK                                   (0x0000FF00U)
#define CDD_DMA_CSL_R5_MPIDR_AFF1_SHIFT                                  (8U)

#define CDD_DMA_CSL_MSS_CTRL_U_BASE                                      (0x50D00000ul)

#define CDD_DMA_CSL_MSS_CTRL_R5_STATUS_REG_R5_STATUS_REG_LOCK_STEP_MASK                (0x00000100U)
#define CDD_DMA_CSL_MSS_CTRL_R5_STATUS_REG_R5_STATUS_REG_LOCK_STEP_SHIFT               (0x00000008U)
#define CDD_DMA_CSL_MSS_CTRL_R5_STATUS_REG_R5_STATUS_REG_LOCK_STEP_RESETVAL            (0x00000000U)
#define CDD_DMA_CSL_MSS_CTRL_R5_STATUS_REG_R5_STATUS_REG_LOCK_STEP_MAX                 (0x00000001U)
 
#define CDD_DMA_CSL_MSS_TCMA_RAM_BASE               (0x00000040U)
#define CDD_DMA_CSL_MSS_TCMA_RAM_SIZE               (0x00007FC0U)  
#define CDD_DMA_CSL_MSS_TCMB_RAM_BASE               (0x00080000U)
#define CDD_DMA_CSL_MSS_TCMB_RAM_SIZE               (0x00004000U)

#define CDD_DMA_CSL_MSS_TCMA_CR5A_U_BASE            (0xC1000000U)
#define CDD_DMA_CSL_MSS_TCMB_CR5A_U_BASE            (0xC1800000U)
#define CDD_DMA_CSL_MSS_TCMA_CR5B_U_BASE            (0xC3000000U)
#define CDD_DMA_CSL_MSS_TCMB_CR5B_U_BASE            (0xC3800000U)

#define CDD_DMA_CSL_MSS_L2_U_BASE                   (0x10200000U)
#define CDD_DMA_CSL_MSS_L2_RAM_BASE                 CDD_DMA_CSL_MSS_L2_U_BASE
#define CDD_DMA_CSL_MSS_L2_RAM_SIZE                 CDD_DMA_CSL_MSS_L2_U_SIZE

#define CDD_DMA_CSL_MSS_L2_U_SIZE                   (0x000F0000U) 
#define CDD_DMA_CSL_GLOB_MSS_L2_RAM_BASE            (0xC0200000U) 

uint32 CDD_DMA_Mcal_armR5ReadMpidrReg(void);
static void Cdd_Dma_CSL_armR5GetCpuID(CDD_DMA_CSL_ArmR5CPUInfo *cpuInfo);
static uint32 Cdd_Dma_SOC_rcmIsR5FInLockStepMode(uint32 r5fClusterGroupId);
static CDD_DMA_CSL_mss_ctrlRegs* Cdd_Dma_SOC_rcmGetBaseAddressMSSCTRL (void);

#define CDD_DMA_START_SEC_CODE
#include "Cdd_Dma_MemMap.h"

/* Initializes the DMA module. */
void Cdd_Dma_Init(Cdd_Dma_ConfigType *ConfigPtr)
{
    uint32 instCnt;
    Cdd_Dma_ConfigType *CfgPtr = ConfigPtr;
    
    /* Initialize the Pointer. */
    #if (STD_ON == CDD_DMA_PRE_COMPILE_VARIANT)
        if (NULL_PTR == CfgPtr)
        {
            CfgPtr = (Cdd_Dma_ConfigType *)&CDD_DMA_INIT_CONFIG_PC;
        }
    #endif

    if (FALSE == Cdd_Dma_InitDone)
    {
        gEdmaHandle = &gEdmaObject;
        gEdmaHandle->baseAddr = CfgPtr->baseAddr;
        gEdmaHandle->hEdmaInit = &(CfgPtr->edmaConfig);
        Cdd_Dma_TrigXbar();
        Cdd_Dma_Xbar();
        CDD_EDMA_lld_init(gEdmaHandle);
        Cdd_Dma_InitDone = TRUE;
    }
    else
    {
        //Report DET
    }

    return;
}

/* Provides the DMA Initialization status. */
boolean Cdd_Dma_GetInitStatus(void)
{
    return Cdd_Dma_InitDone;
}


/* De-Initializes the DMA module. */
void Cdd_Dma_DeInit(void)
{
    if(TRUE == Cdd_Dma_InitDone)
    {
        Cdd_Dma_InitDone = FALSE;
        CDD_EDMA_lld_deInit(gEdmaHandle);
    }
    else
    {
        //Report DET
    }

    return;
}

// /* Provides the Handle for the instance. */
CddDma_HandleType Cdd_Dma_GetHandle()
{   
    return(gEdmaHandle);
}

uint64 Cdd_Dma_SOC_virtToPhy(void *virtAddr)
{
    uintptr_t   temp = (uintptr_t) virtAddr;
    uint64    phyAddr = (uint64) temp;                  /* Default case */

    /* R5F overrides */
#if (__ARM_ARCH == 7) && (__ARM_ARCH_PROFILE == 'R')
    uint32    retVal;

    if (0u == Cdd_Dma_isMapAvailable)
    {
        /* Get Core ID Info */
        Cdd_Dma_CSL_armR5GetCpuID(&virtToPhymap.cpuInfo);

        retVal = Cdd_Dma_SOC_rcmIsR5FInLockStepMode(virtToPhymap.cpuInfo.grpId);
        /* LockStep Mode TCM Size is 64KB */

        if (virtToPhymap.cpuInfo.grpId == (uint32)0)
        {
            if (retVal == TRUE)
            {
                virtToPhymap.tcmaSize = (CDD_DMA_CSL_MSS_TCMA_RAM_SIZE * 2U);
                virtToPhymap.tcmbSize = (CDD_DMA_CSL_MSS_TCMB_RAM_SIZE * 2U);
            }
            else
            {
                virtToPhymap.tcmaSize = (CDD_DMA_CSL_MSS_TCMA_RAM_SIZE);
                virtToPhymap.tcmbSize = (CDD_DMA_CSL_MSS_TCMB_RAM_SIZE);
            }
        }

        Cdd_Dma_isMapAvailable = 1u;
    }

    if (virtToPhymap.cpuInfo.grpId == (uint32)0) /* R5SS0-0 */
    {
        if(virtToPhymap.cpuInfo.cpuID == 0)
        {
            /* TCMA R5FSS0-0 */
            if((temp >= CDD_DMA_CSL_MSS_TCMA_RAM_BASE) &&
               (temp < (CDD_DMA_CSL_MSS_TCMA_RAM_BASE + virtToPhymap.tcmaSize)))
            {
                phyAddr -= CDD_DMA_CSL_MSS_TCMA_RAM_BASE;
                phyAddr += CDD_DMA_CSL_MSS_TCMA_CR5A_U_BASE;
            }

            /* TCMB R5FSS0-0 */
            else if((temp >= CDD_DMA_CSL_MSS_TCMB_RAM_BASE) &&
               (temp < (CDD_DMA_CSL_MSS_TCMB_RAM_BASE + virtToPhymap.tcmbSize)))
            {
                phyAddr -= CDD_DMA_CSL_MSS_TCMB_RAM_BASE;
                phyAddr += CDD_DMA_CSL_MSS_TCMB_CR5A_U_BASE;
            }
        }
        else if(virtToPhymap.cpuInfo.cpuID == 1)
        {
            /* TCMA R5FSS0-1 */
            if((temp >= CDD_DMA_CSL_MSS_TCMA_RAM_BASE) &&
               (temp < (CDD_DMA_CSL_MSS_TCMA_RAM_BASE + CDD_DMA_CSL_MSS_TCMA_RAM_SIZE)))
            {
                phyAddr -= CDD_DMA_CSL_MSS_TCMA_RAM_BASE;
                phyAddr += CDD_DMA_CSL_MSS_TCMA_CR5B_U_BASE;
            }

            /* TCMB R5FSS0-1 */
            else if((temp >= CDD_DMA_CSL_MSS_TCMB_RAM_BASE) &&
               (temp < (CDD_DMA_CSL_MSS_TCMB_RAM_BASE + CDD_DMA_CSL_MSS_TCMB_RAM_SIZE)))
            {
                phyAddr -= CDD_DMA_CSL_MSS_TCMB_RAM_BASE;
                phyAddr += CDD_DMA_CSL_MSS_TCMB_CR5B_U_BASE;
            }
        }
    }

    /* MSS L2 */
    if((temp >= CDD_DMA_CSL_MSS_L2_RAM_BASE) &&
       (temp < (CDD_DMA_CSL_MSS_L2_RAM_BASE + CDD_DMA_CSL_MSS_L2_RAM_SIZE)))
    {
        phyAddr -= CDD_DMA_CSL_MSS_L2_RAM_BASE;
        phyAddr += CDD_DMA_CSL_GLOB_MSS_L2_RAM_BASE;
    }

#endif

    return (phyAddr);
}

static void Cdd_Dma_CSL_armR5GetCpuID(CDD_DMA_CSL_ArmR5CPUInfo *cpuInfo)
{
    uint32 regVal;

    if (cpuInfo != NULL)
    {
        regVal = CDD_DMA_Mcal_armR5ReadMpidrReg();
        cpuInfo->cpuID = (uint32)((regVal & CDD_DMA_CSL_R5_MPIDR_AFF0_MASK) >>
                                            CDD_DMA_CSL_R5_MPIDR_AFF0_SHIFT);
        cpuInfo->grpId = (uint32)((regVal & CDD_DMA_CSL_R5_MPIDR_AFF1_MASK) >>
                                            CDD_DMA_CSL_R5_MPIDR_AFF1_SHIFT);
    }
}

static uint32 Cdd_Dma_SOC_rcmIsR5FInLockStepMode(uint32 r5fClusterGroupId)
{
    uint32 retVal = FALSE;
   CDD_DMA_CSL_mss_ctrlRegs *mssCtrl = Cdd_Dma_SOC_rcmGetBaseAddressMSSCTRL();

    if (r5fClusterGroupId == 0)
    {
        if (CDD_DMA_CSL_MSS(mssCtrl->R5_STATUS_REG, 
		                                      MSS_CTRL_R5_STATUS_REG_R5_STATUS_REG_LOCK_STEP) == 1U)
        {
            /* Lockstep Mode */
            retVal = TRUE;
        }
        else
        {
            /* Dualcore Mode */
            retVal = FALSE;
        }
    }

    return retVal;
}

static CDD_DMA_CSL_mss_ctrlRegs* Cdd_Dma_SOC_rcmGetBaseAddressMSSCTRL (void)
{
    return (CDD_DMA_CSL_mss_ctrlRegs*) CDD_DMA_CSL_MSS_CTRL_U_BASE;
}

/* =========================Function separator========================= */
/* FUNCTION DEF: void CSL_armR5ReadMpidrReg */
        asm(".global CDD_DMA_Mcal_armR5ReadMpidrReg         ");
        asm(".arm                                     ");
        asm(".align 2                                 ");
asm("CDD_DMA_Mcal_armR5ReadMpidrReg:");
        asm("dmb ");
		asm("mrc     p15, #0, r0, c1, c0, #5 ");
        asm("bx      lr                      ");

#define CDD_DMA_STOP_SEC_CODE
#include "Cdd_Dma_MemMap.h"
