/*
 * Copyright (C) 2021 Texas Instruments Incorporated
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 *   Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the
 *   distribution.
 *
 *   Neither the name of Texas Instruments Incorporated nor the names of
 *   its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 *  \file   edma.c
 *
 *  \brief  This file contains device abstraction layer APIs for the EDMA device.
 *          There are APIs here to enable the EDMA instance, set the required
 *          configurations for communication, transmit or receive data.
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "SchM_Cdd_Dma.h"
#include "Cdd_Dma.h"
#include "hw_types.h"
#include "stddef.h"

/* ========================================================================== */
/*                        Static Function Declaration                         */
/* ========================================================================== */
// static Cdd_EDMALLD_Handle gEdmaHandle;
static sint32 CDD_EDMA_lld_initialize (Cdd_EDMALLD_Handle hEdma);
static sint32 CDD_EDMA_lld_deinitialize (Cdd_EDMALLD_Handle hEdma);
static sint32 CDD_EDMA_lld_allocResource(Cdd_EDMALLD_Handle hEdma, uint32 resType, uint32 *resId);
static sint32 CDD_EDMA_lld_Alloc_resource(Cdd_EDMALLD_Handle hEdma, uint32 *resId, uint32 resType);
static sint32 CDD_EDMA_lld_freeResource(Cdd_EDMALLD_Handle hEdma, uint32 resType, uint32 *resId);
static void lock_mutex(void);
static void unlock_mutex(void);

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */
#define CDD_DMA_START_SEC_CODE
#include "Cdd_Dma_MemMap.h"

void lock_mutex(void)
{
    SchM_Enter_Cdd_Dma_DMA_EXCLUSIVE_AREA_0();
}

void unlock_mutex(void)
{
    SchM_Exit_Cdd_Dma_DMA_EXCLUSIVE_AREA_0();
}

static sint32 CDD_EDMA_lld_initialize (Cdd_EDMALLD_Handle hEdma)
{
    sint32 retVal = MCAL_SystemP_SUCCESS;
    CDD_EDMACCEDMACCPaRAMEntry paramSet;
    uint32 count = 0;
    uint32 i     = 0;
    uint32 qnumValue;
    uint32 regionId;
    uint32 queNum;
    uint32 baseAddr = hEdma->baseAddr;
    Cdd_EDMALLD_InitHandle hEdmaInit = hEdma->hEdmaInit;
    CDD_EDMA_ResourceObject *ownResource = &hEdmaInit->ownResource;
    gEdmaHandle = hEdma;

    if (hEdmaInit == NULL)
    {
        retVal = MCAL_SystemP_FAILURE;
    }
    else
    {
        regionId = hEdmaInit->regionId;
        queNum = hEdmaInit->queNum;

        /* Clear the Event miss Registers                                     */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EMCR, ownResource->dmaCh[0]);
    #if CDD_EDMA_NUM_DMACH > 32
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EMCRH, ownResource->dmaCh[1]);
    #endif
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QEMCR, ownResource->qdmaCh);

        /* Clear CCERR register.                                               */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_CCERRCLR, CDD_EDMA_SET_ALL_BITS);

        /* Disable and clear DMA events for all own dma channels */
        for (i = 0; i < CDD_EDMA_NUM_DMACH; i++)
        {
            if (((1U << (i%32U)) & ownResource->dmaCh[i/32U]) != 0U)
            {
                CDD_EDMA_lld_disableDmaEvtRegion(baseAddr, regionId, i);
                CDD_EDMA_lld_clrEvtRegion(baseAddr, regionId, i);
                CDD_EDMA_lld_clrMissEvtRegion(baseAddr, regionId, i);
            }
        }

        /* Disable and clear channel interrupts for all own dma channels */
        for (i = 0; i < CDD_EDMA_NUM_TCC; i++)
        {
            if (((1U << (i%32U)) & ownResource->tcc[i/32U]) != 0U)
            {
                CDD_EDMA_lld_disableEvtIntrRegion(baseAddr, regionId, i);
                CDD_EDMA_lld_clrIntrRegion(baseAddr, regionId, i);
            }
        }

        /* Disable and clear channel interrupts for all own qdma channels */
        for (i = 0; i < CDD_EDMA_NUM_QDMACH; i++)
        {
            if (((1U << i) & ownResource->qdmaCh) != 0U)
            {
                CDD_EDMA_lld_disableQdmaEvtRegion(baseAddr, regionId, i);
                CDD_EDMA_lld_qdmaClrMissEvtRegion(baseAddr, regionId, i);
            }
        }
        /* FOR TYPE EDMA*/
        /* Enable the own DMA (0 - 64) channels in the DRAE and DRAEH register */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DRAEM(regionId), ownResource->dmaCh[0]);
    #if CDD_EDMA_NUM_DMACH > 32
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DRAEHM(regionId), ownResource->dmaCh[1]);
    #endif
        /* Enable the own TCCs also for the region in DRAE and DRAEH register */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DRAEM(regionId), ownResource->tcc[0]);
    #if CDD_EDMA_NUM_DMACH > 32
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DRAEHM(regionId), ownResource->tcc[1]);
    #endif
        if( CDD_EDMA_CHMAPEXIST != 0U)
        {
            for (i = 0U; i < CDD_EDMA_NUM_DMACH; i++)
            {
                if (((1U << (i%32U)) & ownResource->dmaCh[i/32U]) != 0U)
                {
                    /* All channels are one to one mapped with the params */
                    HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DCHMAPN(i), i << 5);
                }
            }
        }

        /* Initialize the DMA Queue Number Registers                            */
        for (count = 0U; count < CDD_EDMA_NUM_DMACH; count++)
        {
            if (((1U << (count%32U)) & ownResource->dmaCh[count/32U]) != 0U)
            {
                qnumValue  = HW_RD_REG32(baseAddr + (CDD_EDMA_TPCC_DMAQNUMN((count >> 3U))));
                qnumValue &= CDD_EDMACC_DMAQNUM_CLR(count);
                qnumValue |= CDD_EDMACC_DMAQNUM_SET(count, queNum);
                HW_WR_REG32(baseAddr + (CDD_EDMA_TPCC_DMAQNUMN((count >> 3U))), qnumValue);
            }
        }

        /* FOR TYPE QDMA */
        /* Enable the QDMA (0 - 64) channels in the DRAE register                */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QRAEN(regionId), ownResource->qdmaCh);

        /* Initialize the QDMA Queue Number Registers                           */
        for (count = 0U; count < CDD_EDMA_NUM_QDMACH; count++)
        {
            if (((1U << count) & ownResource->qdmaCh) != 0U)
            {
                qnumValue  = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_QDMAQNUM);
                qnumValue &= CDD_EDMACC_QDMAQNUM_CLR(count);
                qnumValue |= CDD_EDMACC_QDMAQNUM_SET(count, queNum);
                HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QDMAQNUM, qnumValue);
            }
        }

        if (hEdmaInit->initParamSet == TRUE)
        {
            CDD_EDMA_lld_ccPaRAMEntry_init(&paramSet);
            /* cleanup Params, note h/w reset state is all 0s, must be done after
            disabling/clearning channel events (in particular QDMA) */
            for (count = 0; count < CDD_EDMA_NUM_PARAMSETS; count++)
            {
                if (((1U << (count%32U)) & ownResource->paramSet[count/32U]) != 0U)
                {
                    CDD_EDMA_lld_setPaRAM(baseAddr, count, &paramSet);
                }
            }
        }
    }
    return retVal;
}

void CDD_EDMA_lld_ccPaRAMEntry_init(CDD_EDMACCEDMACCPaRAMEntry *paramEntry)
{
    if(paramEntry != NULL)
    {
        /* Initialize all PaRAM entries as 0 */
        paramEntry->opt = 0;
        paramEntry->srcAddr = 0;
        paramEntry->aCnt = 0;
        paramEntry->bCnt = 0;
        paramEntry->destAddr = 0;
        paramEntry->srcBIdx = 0;
        paramEntry->destBIdx = 0;
        paramEntry->linkAddr = 0;
        paramEntry->bCntReload = 0;
        paramEntry->srcCIdx = 0;
        paramEntry->destCIdx = 0;
        paramEntry->cCnt = 0;
        paramEntry->reserved = 0;
    }
    else
    {
        //Do Nothing
    }
}

uint32 CDD_EDMA_lld_peripheralIdGet(uint32 baseAddr)
{
    return (HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_PID));
}

void CDD_EDMA_lld_enableChInShadowRegRegion(uint32 baseAddr,
                                    uint32 regionId,
                                    uint32 chType,
                                    uint32 chNum)
{
    uint32 draeValue;
    /* Allocate the DMA/QDMA channel */
    if (CDD_EDMA_CHANNEL_TYPE_DMA == chType)
    {
        /* FOR TYPE EDMA*/
        if (chNum < 32U)
        {
            draeValue = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_DRAEM(regionId));
            /* Enable the DMA channel in the DRAE registers */
            draeValue |= (uint32) 0x01U << chNum;
            HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DRAEM(regionId), draeValue);
        }
        else
        {
            draeValue = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_DRAEHM(regionId));

            /* Enable the DMA channel in the DRAEH registers */
            draeValue |= (uint32) 0x01U << (chNum - 32U);
            HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DRAEHM(regionId), draeValue);
        }
    }
    else if (CDD_EDMA_CHANNEL_TYPE_QDMA == chType)
    {
        /* FOR TYPE QDMA */
        /* Enable the QDMA channel in the DRAE/DRAEH registers */
        draeValue  = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_QRAEN(regionId));
        draeValue |= (uint32) 0x01U << chNum;
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QRAEN(regionId), draeValue);
    }
    else
    {
        /*An error will be generated automatically.*/
    }
}

void CDD_EDMA_lld_disableChInShadowRegRegion(uint32 baseAddr,
                                     uint32 regionId,
                                     uint32 chType,
                                     uint32 chNum)
{
    uint32 draeValue;
    /* Allocate the DMA/QDMA channel */
    if (CDD_EDMA_CHANNEL_TYPE_DMA == chType)
    {
        /* FOR TYPE EDMA*/
        if (chNum < 32U)
        {
            draeValue = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_DRAEM(regionId));
            /* Enable the DMA channel in the DRAE registers */
            draeValue &= ~((uint32) 0x01U << chNum);
            HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DRAEM(regionId), draeValue);
        }
        else
        {
            draeValue = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_DRAEHM(regionId));
            /* Enable the DMA channel in the DRAEH registers */
            draeValue &= ~((uint32) 0x01U << (chNum - 32U));
            HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DRAEHM(regionId), draeValue);
        }
    }
    else if (CDD_EDMA_CHANNEL_TYPE_QDMA == chType)
    {
        /* FOR TYPE QDMA */
        draeValue = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_QRAEN(regionId));
        /* Enable the QDMA channel in the DRAE/DRAEH registers */
        draeValue &= ~((uint32) 0x01U) << chNum;
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QRAEN(regionId), draeValue);
    }
    else
    {
        /*An error will be generated automatically.*/
    }
}

void CDD_EDMA_lld_channelToParamMap(uint32 baseAddr,
                            uint32 channel,
                            uint32 paramSet)

{
    if( CDD_EDMA_CHMAPEXIST != 0U)
    {
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DCHMAPN(channel), paramSet << 5U);
    }
}

void CDD_EDMA_lld_mapChToEvtQ(uint32 baseAddr,
                      uint32 chType,
                      uint32 chNum,
                      uint32 evtQNum)
{
    uint32 qnumValue;
    if (CDD_EDMA_CHANNEL_TYPE_DMA == chType)
    {
        /* Associate DMA Channel to Event Queue                             */
        qnumValue  = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_DMAQNUMN(chNum >> 3U));
        qnumValue &= CDD_EDMACC_DMAQNUM_CLR(chNum);
        qnumValue |= CDD_EDMACC_DMAQNUM_SET(chNum, evtQNum);
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DMAQNUMN(chNum >> 3U), qnumValue);
    }
    else if (CDD_EDMA_CHANNEL_TYPE_QDMA == chType)
    {
        /* Associate QDMA Channel to Event Queue                            */
        qnumValue  = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_QDMAQNUM);
        qnumValue &= CDD_EDMACC_QDMAQNUM_CLR(chNum);
        qnumValue |= CDD_EDMACC_QDMAQNUM_SET(chNum, evtQNum);
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QDMAQNUM, qnumValue);
    }
    else
    {
        /*An error will be generated automatically.*/
    }
}

void CDD_EDMA_lld_unmapChToEvtQ(uint32 baseAddr,
                        uint32 chType,
                        uint32 chNum)
{
    uint32 qnumValue;
    if (CDD_EDMA_CHANNEL_TYPE_DMA == chType)
    {
        /* Unmap DMA Channel to Event Queue                                */
        qnumValue  = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_DMAQNUMN(chNum >> 3U));
        qnumValue &= CDD_EDMACC_DMAQNUM_CLR(chNum);
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DMAQNUMN(chNum >> 3U), qnumValue);
    }
    else if (CDD_EDMA_CHANNEL_TYPE_QDMA == chType)
    {
        /* Unmap QDMA Channel to Event Queue                               */
        qnumValue  = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_QDMAQNUM);
        qnumValue &= CDD_EDMACC_QDMAQNUM_CLR(chNum);
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QDMAQNUM, qnumValue);
    }
    else
    {
        /*An error will be generated automatically.*/
    }
}

void CDD_EDMA_lld_mapQdmaChToPaRAM(uint32        baseAddr,
                           uint32        chNum,
                           const uint32 *paRAMId)
{
    uint32 qchmapValue;
    /* Map Parameter RAM Set Number for specified channelId             */
    qchmapValue  = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_QCHMAPN(chNum));
    qchmapValue &= CDD_EDMACC_QCHMAP_PAENTRY_CLR;
    qchmapValue |= (uint32) CDD_EDMACC_QCHMAP_PAENTRY_SET(*paRAMId);
    HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QCHMAPN(chNum), qchmapValue);
}

uint32 CDD_EDMA_lld_getMappedPaRAM(uint32 baseAddr,
                             uint32 chNum,
                             uint32 chType,
                             uint32 *paramId)
{
    uint32 retVal = FALSE;
    if ((CDD_EDMA_CHANNEL_TYPE_DMA == chType) &&
        (chNum <= CDD_EDMA_NUM_DMACH))
    {

        /* Bug Fix - Changed to == */
        if( CDD_EDMA_CHMAPEXIST == 0U)
        {
            *paramId = chNum;
        }
        else
        {
            *paramId = HW_RD_FIELD32(baseAddr + CDD_EDMA_TPCC_DCHMAPN(chNum),
                                    CDD_EDMA_TPCC_DCHMAPN_PAENTRY);
        }
        retVal = TRUE;
    }
    else if ((CDD_EDMA_CHANNEL_TYPE_QDMA == chType) &&
             (chNum <= CDD_EDMA_NUM_QDMACH))
    {
        *paramId = HW_RD_FIELD32(baseAddr + CDD_EDMA_TPCC_QCHMAPN(chNum),
                                CDD_EDMA_TPCC_QCHMAPN_PAENTRY);
        retVal = TRUE;
    }
    else
    {
        /*An error will be generated automatically.*/
    }
    return retVal;
}

void CDD_EDMA_lld_setQdmaTrigWord(uint32 baseAddr,
                          uint32 chNum,
                          uint32 trigWord)
{
    uint32 qchmapValue;
    qchmapValue = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_QCHMAPN(chNum));
    /* Clear QDMA Trigger word value */
    qchmapValue &= CDD_EDMACC_QCHMAP_TRWORD_CLR;
    /* Set the Trigger Word */
    qchmapValue |= CDD_EDMACC_QCHMAP_TRWORD_SET(trigWord);
    HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QCHMAPN(chNum), qchmapValue);
}

void CDD_EDMA_lld_clrMissEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum)
{
    if (chNum < 32U)
    {
        /*clear SECR to clean any previous NULL request */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_SECR_RN(
                        regionId), (uint32) 0x01U << chNum);

        /*clear EMCR to clean any previous NULL request */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EMCR, (uint32) 0x01U << chNum);
    }
    else
    {
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_SECRH_RN(regionId),
                    (uint32) 0x01U << (chNum - 32U));
        /*clear EMCRH to clean any previous NULL request */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EMCRH, (uint32) 0x01U << (chNum - 32U));
    }
}

void CDD_EDMA_lld_qdmaClrMissEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum)
{
    /*clear SECR to clean any previous NULL request  */
    HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QSECR_RN(
                    regionId), (uint32) 0x01U << chNum);

    /*clear EMCR to clean any previous NULL request  */
    HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QEMCR, (uint32) 0x01U << chNum);
}

void CDD_EDMA_lld_clrCCErr(uint32 baseAddr, uint32 flags)
{
    /* (CCERRCLR) - clear channel controller error register */
    HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_CCERRCLR, flags);
}

void CDD_EDMA_lld_setEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum)
{
    if (chNum < 32U)
    {
        /* (ESR) - set corresponding bit to set a event */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_ESR_RN(
                        regionId), (uint32) 0x01U << chNum);
    }
    else
    {
        /* (ESRH) - set corresponding bit to set a event */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_ESRH_RN(regionId),
                    (uint32) 0x01U << (chNum - 32U));
    }
}

void CDD_EDMA_lld_clrEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum)
{
    if (chNum < 32U)
    {
        /* (ECR) - set corresponding bit to clear a event */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_ECR_RN(
                        regionId), (uint32) 0x01U << chNum);
    }
    else
    {
        /* (ECRH) - set corresponding bit to clear a event */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_ECRH_RN(regionId),
                    (uint32) 0x01U << (chNum - 32U));
    }
}

void CDD_EDMA_lld_enableDmaEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum)
{
    if (chNum < 32U)
    {
        /* (EESR) - set corresponding bit to enable DMA event */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EESR_RN(
                        regionId), (uint32) 0x01U << chNum);
    }
    else
    {
        /* (EESRH) - set corresponding bit to enable DMA event */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EESRH_RN(regionId),
                    (uint32) 0x01U << (chNum - 32U));
    }
}

void CDD_EDMA_lld_disableDmaEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum)
{
    if (chNum < 32U)
    {
        /* (EECR) - set corresponding bit to disable event */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EECR_RN(
                        regionId), (uint32) 0x01U << chNum);
    }
    else
    {
        /* (EECRH) - set corresponding bit to disable event */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EECRH_RN(
                        regionId), (uint32) 0x01U << (chNum - 32U));
    }
}

void CDD_EDMA_lld_enableQdmaEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum)
{
    /* (QEESR) - set corresponding bit to enable QDMA event                 */
    HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QEESR_RN(
                    regionId), (uint32) 0x01U << chNum);
}

void CDD_EDMA_lld_disableQdmaEvtRegion(uint32 baseAddr, uint32 regionId, uint32 chNum)
{
    /* (QEESR) - set corresponding bit to disable QDMA event                 */
    HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QEECR_RN(
                    regionId), (uint32) 0x01U << chNum);
}

uint32 CDD_EDMA_lld_getCCErrStatus(uint32 baseAddr)
{
    uint32 intrStatusVal = 0;
    intrStatusVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_CCERR);

    return intrStatusVal;
}

uint32 CDD_EDMA_lld_getIntrStatusRegion(uint32 baseAddr, uint32 regionId)
{
    uint32 intrStatusVal = 0;

    intrStatusVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_IPR_RN(regionId));

    return intrStatusVal;
}

uint32 CDD_EDMA_lld_intrStatusHighGetRegion(uint32 baseAddr, uint32 regionId)
{
    uint32 intrStatusVal = 0;

    intrStatusVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_IPRH_RN(regionId));

    return intrStatusVal;
}

uint32 CDD_EDMA_lld_readIntrStatusRegion(uint32 baseAddr, uint32 regionId, uint32 tccNum)
{
    uint32 intrStatus = 0;

    if(tccNum < 32U)
    {
        if ((CDD_EDMA_lld_getIntrStatusRegion(baseAddr, regionId) & (0x1U << tccNum)) ==
            (0x1U << tccNum))
        {
            intrStatus = 1;
        }
    }
    else
    {
        if ((CDD_EDMA_lld_intrStatusHighGetRegion(baseAddr, regionId) &
               (0x1U << (tccNum - 32))) ==
               (0x1U << (tccNum - 32)))
        {
            intrStatus = 1;
        }
    }
    return intrStatus;
}

uint32 CDD_EDMA_lld_getEventStatus(uint32 baseAddr)
{
    uint32 intrStatusVal = 0;

    intrStatusVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_ER);

    return intrStatusVal;
}

uint32 CDD_EDMA_lld_getEventStatusHigh(uint32 baseAddr)
{
    uint32 intrStatusVal = 0;

    intrStatusVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_ERH);

    return intrStatusVal;
}

uint32 CDD_EDMA_lld_readEventStatusRegion(uint32 baseAddr, uint32 chNum)
{
    uint32 eventStatus = 0;

    if(chNum < 32U)
    {
        if ((CDD_EDMA_lld_getEventStatus(baseAddr) & (0x1U << chNum)) ==
            (0x1U << chNum))
        {
            eventStatus = 1;
        }
    }
    else
    {
        if ((CDD_EDMA_lld_getEventStatusHigh(baseAddr) & (0x1U << (chNum - 32))) ==
            (0x1U << (chNum - 32)))
        {
            eventStatus = 1;
        }
    }
    return eventStatus;
}

uint32 CDD_EDMA_lld_getErrIntrStatus(uint32 baseAddr)
{
    uint32 intrStatusVal = 0;

    intrStatusVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_EMR);

    return intrStatusVal;
}

uint32 CDD_EDMA_lld_errIntrHighStatusGet(uint32 baseAddr)
{
    uint32 intrStatusVal = 0;

    intrStatusVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_EMRH);

    return intrStatusVal;
}

uint32 CDD_EDMA_lld_qdmaGetErrIntrStatus(uint32 baseAddr)
{
    uint32 intrStatusVal = 0;
    intrStatusVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_QEMR);

    return intrStatusVal;
}

void CDD_EDMA_lld_enableEvtIntrRegion(uint32 baseAddr, uint32 regionId, uint32 chNum)
{
    if (chNum < 32U)
    {
        /*  Interrupt Enable Set Register (IESR) */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_IESR_RN(
                        regionId), (uint32) 0x01U << chNum);
    }
    else
    {
        /*  Interrupt Enable Set Register (IESRH) */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_IESRH_RN(regionId),
                    (uint32) 0x01U << (chNum - 32U));
    }
}

void CDD_EDMA_lld_disableEvtIntrRegion(uint32 baseAddr, uint32 regionId, uint32 chNum)
{
    if (chNum < 32U)
    {
        /* Interrupt Enable Clear Register (IECR) */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_IECR_RN(
                        regionId), (uint32) 0x01U << chNum);
    }
    else
    {
        /* Interrupt Enable Clear Register (IECRH) */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_IECRH_RN(regionId),
                    (uint32) 0x01U << (chNum - 32U));
    }
}

void CDD_EDMA_lld_clrIntrRegion(uint32 baseAddr, uint32 regionId, uint32 value)
{
    if (value < 32U)
    {
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_ICR_RN(
                        regionId), (uint32) 1U << value);
    }
    else
    {
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_ICRH_RN(regionId), (uint32) 1U <<
                    (value - 32U));
    }
}

uint32 CDD_EDMA_lld_getEnabledIntrRegion(uint32 baseAddr, uint32 regionId)
{
    uint32 intrEnableVal = 0;

    intrEnableVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_IER_RN(regionId));

    return intrEnableVal;
}

uint32 CDD_EDMA_lld_getEnabledIntrHighRegion(uint32 baseAddr, uint32 regionId)
{
    uint32 intrEnableVal = 0;

    intrEnableVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_IERH_RN(regionId));

    return intrEnableVal;
}

void CDD_EDMA_lld_getPaRAM(uint32           baseAddr,
                   uint32           paRAMId,
                   CDD_EDMACCEDMACCPaRAMEntry *currPaRAM)
{
    uint32  i = 0;
    uint32 sr;
    uint32 *ds = (uint32 *) currPaRAM;

    sr = baseAddr + CDD_EDMA_TPCC_OPT(paRAMId);

    for (i = 0; i < CDD_EDMACC_PARAM_ENTRY_FIELDS; i++)
    {
        *ds = HW_RD_REG32(sr);
        ds++;
        sr+= (uint32)sizeof(uint32);
    }
}

void CDD_EDMA_lld_qdmaGetPaRAM(uint32           baseAddr,
                       uint32           paRAMId,
                       CDD_EDMACCEDMACCPaRAMEntry *currPaRAM)
{
    uint32  i = 0;
    uint32 *ds     = (uint32 *) currPaRAM;
    uint32  sr     = baseAddr + CDD_EDMA_TPCC_OPT(paRAMId);

    for (i = 0; i < CDD_EDMACC_PARAM_ENTRY_FIELDS; i++)
    {
        *ds = HW_RD_REG32(sr);
        ds++;
        sr+= (uint32)sizeof(uint32);
    }
}

void CDD_EDMA_lld_setPaRAM(uint32           baseAddr,
                   uint32           paRAMId,
                   const CDD_EDMACCEDMACCPaRAMEntry *newPaRAM)
{
    uint32           i  = 0;
    uint32          *sr = (uint32 *) newPaRAM;
    volatile uint32  ds;
    uint32           dsAddr =baseAddr + CDD_EDMA_TPCC_OPT(paRAMId);

    ds = (uint32 ) (dsAddr);

    for (i = 0; i < CDD_EDMACC_PARAM_ENTRY_FIELDS; i++)
    {
        HW_WR_REG32(ds, *sr);
        ds+= (uint32)sizeof(uint32);
        sr++;
    }
}

void CDD_EDMA_lld_qdmaSetPaRAM(uint32           baseAddr,
                       uint32           paRAMId,
                       const CDD_EDMACCEDMACCPaRAMEntry *newPaRAM)
{
    uint32  i  = 0;
    uint32 *sr = (uint32 *) newPaRAM;
    uint32  ds;
    uint32  dsAddr =baseAddr + CDD_EDMA_TPCC_OPT(paRAMId);

    ds = (uint32 ) (dsAddr);

    for (i = 0; i < CDD_EDMACC_PARAM_ENTRY_FIELDS; i++)
    {
        HW_WR_REG32(ds, *sr);
        ds+= (uint32)sizeof(uint32);
        sr++;
    }
}

void CDD_EDMA_lld_qdmaSetPaRAMEntry(uint32 baseAddr,
                            uint32 paRAMId,
                            uint32 paRAMEntry,
                            uint32 newPaRAMEntryVal)
{
    CDD_EDMA_lld_dmaSetPaRAMEntry(baseAddr, paRAMId,
                          paRAMEntry, newPaRAMEntryVal);
}

uint32 CDD_EDMA_lld_qdmaGetPaRAMEntry(uint32 baseAddr,
                                uint32 paRAMId,
                                uint32 paRAMEntry)
{
    return CDD_EDMA_lld_dmaGetPaRAMEntry(baseAddr, paRAMId,
                          paRAMEntry);
}

void CDD_EDMA_lld_dmaSetPaRAMEntry(uint32 baseAddr,
                            uint32 paRAMId,
                            uint32 paRAMEntry,
                            uint32 newPaRAMEntryVal)
{
    if (paRAMEntry <= CDD_EDMACC_PARAM_ENTRY_CCNT)
    {
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_OPT(paRAMId) +
                    (paRAMEntry * 0x04U), newPaRAMEntryVal);
    }
}

uint32 CDD_EDMA_lld_dmaGetPaRAMEntry(uint32 baseAddr,
                                uint32 paRAMId,
                                uint32 paRAMEntry)
{
    uint32 paRAMEntryVal = 0;
    if (paRAMEntry <= CDD_EDMACC_PARAM_ENTRY_CCNT)
    {
        paRAMEntryVal = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_OPT(paRAMId) +
                                    (paRAMEntry * 0x04U));
    }
    return (paRAMEntryVal);
}

uint32 CDD_EDMA_lld_configureChannelRegion(uint32 baseAddr,
                                     uint32 regionId,
                                     uint32 chType,
                                     uint32 chNum,
                                     uint32 tccNum,
                                     uint32 paramId,
                                     uint32 evtQNum)
{
    uint32 optValue;
    uint32 retVal = FALSE;
    if (((CDD_EDMA_CHANNEL_TYPE_DMA == chType) && (chNum < CDD_EDMA_NUM_DMACH)) ||
        ((CDD_EDMA_CHANNEL_TYPE_QDMA == chType) && (chNum < CDD_EDMA_NUM_QDMACH)))
    {
        /* Enable the DMA channel in the enabled in the shadow region
         * specific register
         */
        CDD_EDMA_lld_enableChInShadowRegRegion(baseAddr, regionId, chType, chNum);

        CDD_EDMA_lld_mapChToEvtQ(baseAddr, chType, chNum, evtQNum);

        if (CDD_EDMA_CHANNEL_TYPE_DMA == chType)
        {
            CDD_EDMA_lld_channelToParamMap(baseAddr, chNum, paramId);
        }
        else
        {
            CDD_EDMA_lld_mapQdmaChToPaRAM(baseAddr, chNum, &paramId);
        }

        optValue  = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_OPT(paramId));
        optValue &= CDD_EDMACC_OPT_TCC_CLR;
        optValue |= CDD_EDMACC_OPT_TCC_SET(tccNum);
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_OPT(paramId), optValue);

        retVal = (uint32) TRUE;
    }
    return retVal;
}

uint32 CDD_EDMA_lld_freeChannelRegion(uint32 baseAddr,
                                uint32 regionId,
                                uint32 chType,
                                uint32 chNum,
                                uint32 trigMode,
                                uint32 tccNum,
                                uint32 evtQNum)
{
    uint32 retVal = FALSE;
    if ((chNum < CDD_EDMA_NUM_DMACH) &&
        ((CDD_EDMA_CHANNEL_TYPE_DMA == chType) || (CDD_EDMA_CHANNEL_TYPE_QDMA == chType)))
    {
        CDD_EDMA_lld_disableTransferRegion(baseAddr, regionId, chNum, trigMode);
        /* Disable the DMA channel in the shadow region specific register
         */
        CDD_EDMA_lld_disableChInShadowRegRegion(baseAddr, regionId, chType, chNum);

        CDD_EDMA_lld_unmapChToEvtQ(baseAddr, chType, chNum);

        retVal = (uint32) TRUE;
    }
    return retVal;
}

uint32 CDD_EDMA_lld_enableTransferRegion(uint32 baseAddr,
                                   uint32 regionId,
                                   uint32 chNum,
                                   uint32 trigMode)
{
    uint32 retVal = FALSE;
    switch (trigMode)
    {
        case CDD_EDMA_TRIG_MODE_MANUAL:
            if (chNum < CDD_EDMA_NUM_DMACH)
            {
                CDD_EDMA_lld_setEvtRegion(baseAddr, regionId, chNum);
                retVal = (uint32) TRUE;
            }
            break;

        case CDD_EDMA_TRIG_MODE_QDMA:
            if (chNum < CDD_EDMA_NUM_QDMACH)
            {
                CDD_EDMA_lld_enableQdmaEvtRegion(baseAddr, regionId, chNum);
                retVal = (uint32) TRUE;
            }
            break;

        case CDD_EDMA_TRIG_MODE_EVENT:
            if (chNum < CDD_EDMA_NUM_DMACH)
            {
                /*clear SECR & EMCR to clean any previous NULL request    */
                CDD_EDMA_lld_clrMissEvtRegion(baseAddr, regionId, chNum);

                /* Set EESR to enable event                               */
                CDD_EDMA_lld_enableDmaEvtRegion(baseAddr, regionId, chNum);
                retVal = (uint32) TRUE;
            }
            break;

        default:
            retVal = (uint32) FALSE;
            break;
    }
    return retVal;
}

uint32 CDD_EDMA_lld_disableTransferRegion(uint32 baseAddr,
                                    uint32 regionId,
                                    uint32 chNum,
                                    uint32 trigMode)
{
    uint32 retVal = FALSE;
    switch (trigMode)
    {
        case CDD_EDMA_TRIG_MODE_MANUAL:
            if (chNum < CDD_EDMA_NUM_DMACH)
            {
                CDD_EDMA_lld_clrEvtRegion(baseAddr, regionId, chNum);
                retVal = (uint32) TRUE;
            }
            break;

        case CDD_EDMA_TRIG_MODE_QDMA:
            if (chNum < CDD_EDMA_NUM_QDMACH)
            {
                CDD_EDMA_lld_disableQdmaEvtRegion(baseAddr, regionId, chNum);
                retVal = (uint32) TRUE;
            }
            break;

        case CDD_EDMA_TRIG_MODE_EVENT:
            if (chNum < CDD_EDMA_NUM_DMACH)
            {
                /*clear SECR & EMCR to clean any previous NULL request    */
                CDD_EDMA_lld_clrMissEvtRegion(baseAddr, regionId, chNum);

                /* Set EESR to enable event                               */
                CDD_EDMA_lld_disableDmaEvtRegion(baseAddr, regionId, chNum);
                retVal = (uint32) TRUE;
            }
            break;

        default:
            retVal = (uint32) FALSE;
            break;
    }
    return retVal;
}

void CDD_EDMA_lld_clearErrorBitsRegion(uint32 baseAddr,
                               uint32 regionId,
                               uint32 chNum,
                               uint32 evtQNum)
{
    if (chNum < CDD_EDMA_NUM_DMACH)
    {
        if (chNum < 32U)
        {
            HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EECR_RN(
                            regionId), (uint32) 0x01U << chNum);
            /* Write to EMCR to clear the corresponding EMR bit */
            HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EMCR, (uint32) 0x01U << chNum);
            /* Clears the SER */
            HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_SECR_RN(
                            regionId), (uint32) 0x01U << chNum);
        }
        else
        {
            HW_WR_REG32(baseAddr +
                        CDD_EDMA_TPCC_EECRH_RN(regionId), (uint32) 0x01U <<
                        (chNum - 32U));
            /* Write to EMCR to clear the corresponding EMR bit */
            HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EMCRH, (uint32) 0x01U <<
                        (chNum - 32U));
            /* Clears the SER */
            HW_WR_REG32(baseAddr +
                        CDD_EDMA_TPCC_SECRH_RN(regionId), (uint32) 0x01U <<
                        (chNum - 32U));
        }
    }

    /* Clear the global CC Error Register */
    if (0U == evtQNum)
    {
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_CCERRCLR,
                    CDD_EDMA_TPCC_CCERRCLR_QTHRXCD0_MASK |
                    CDD_EDMA_TPCC_CCERRCLR_TCERR_MASK);
    }
    else if (1U == evtQNum)
    {
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_CCERRCLR,
                    CDD_EDMA_TPCC_CCERRCLR_QTHRXCD1_MASK |
                    CDD_EDMA_TPCC_CCERRCLR_TCERR_MASK);
    }
#if SOC_CDD_EDMA_NUM_EVQUE > 2
    else if (2U == evtQNum)
    {
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_CCERRCLR,
                    CDD_EDMA_TPCC_CCERRCLR_QTHRXCD2_MASK |
                    CDD_EDMA_TPCC_CCERRCLR_TCERR_MASK);
    }
#if SOC_CDD_EDMA_NUM_EVQUE > 3
    else if (3U == evtQNum)
    {
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_CCERRCLR,
                    CDD_EDMA_TPCC_CCERRCLR_QTHRXCD3_MASK |
                    CDD_EDMA_TPCC_CCERRCLR_TCERR_MASK);
    }
#endif
#endif
    else
    {
        /*Error will be generated automatically*/
    }
}

static sint32 CDD_EDMA_lld_deinitialize (Cdd_EDMALLD_Handle hEdma)
{
    sint32 retVal = MCAL_SystemP_SUCCESS;
    uint32 count = 0;
    uint32 qnumValue;
    uint32 regionId;
    uint32 baseAddr = hEdma->baseAddr;
    Cdd_EDMALLD_InitHandle hEdmaInit = hEdma->hEdmaInit;
    CDD_EDMA_ResourceObject *ownResource = &hEdmaInit->ownResource;

    if (hEdmaInit == NULL)
    {
        retVal = MCAL_SystemP_FAILURE;
    }
    else
    {
        regionId = hEdmaInit->regionId;
        /* Disable the DMA (0 - 62) channels in the DRAE register */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DRAEM(
                        regionId), CDD_EDMA_CLR_ALL_BITS);
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DRAEHM(
                        regionId), CDD_EDMA_CLR_ALL_BITS);

        CDD_EDMA_lld_clrCCErr(baseAddr, CDD_EDMACC_CLR_TCCERR);

        /* Clear the Event miss Registers                      */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EMCR, ownResource->dmaCh[0]);
    #if CDD_EDMA_NUM_DMACH > 32
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_EMCRH, ownResource->dmaCh[1]);
    #endif
        /* Clear CCERR register */
        HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_CCERRCLR, ownResource->qdmaCh);

        /* Disable and clear channel interrupts for all dma channels */
        for (count = 0; count < CDD_EDMA_NUM_TCC; count++)
        {
            if (((1U << (count%32U)) & ownResource->tcc[count/32U]) != 0U)
            {
                CDD_EDMA_lld_disableEvtIntrRegion(baseAddr, regionId, count);
                CDD_EDMA_lld_clrIntrRegion(baseAddr, regionId, count);
            }
        }
        /* Disable and clear channel interrupts for all qdma channels */
        for (count = 0; count < CDD_EDMA_NUM_QDMACH; count++)
        {
            if (((1U << count) & ownResource->qdmaCh) != 0U)
            {
                CDD_EDMA_lld_disableQdmaEvtRegion(baseAddr, regionId, count);
                CDD_EDMA_lld_qdmaClrMissEvtRegion(baseAddr, regionId, count);
            }
        }

        /* Deinitialize the Queue Number Registers */
        for (count = 0; count < CDD_EDMA_NUM_DMACH; count++)
        {
            if (((1U << (count%32U)) & ownResource->dmaCh[count/32U]) != 0U)
            {
                qnumValue  = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_DMAQNUMN((count >> 3U)));
                qnumValue &= CDD_EDMACC_DMAQNUM_CLR(count);
                HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_DMAQNUMN((count >> 3U)), qnumValue);
            }
        }

        for (count = 0; count < CDD_EDMA_NUM_QDMACH; count++)
        {
            if (((1U << count) & ownResource->qdmaCh) != 0U)
            {
                qnumValue  = HW_RD_REG32(baseAddr + CDD_EDMA_TPCC_QDMAQNUM);
                qnumValue &= CDD_EDMACC_QDMAQNUM_CLR(count);
                HW_WR_REG32(baseAddr + CDD_EDMA_TPCC_QDMAQNUM, qnumValue);
            }
        }
    }
    return retVal;
}

void CDD_EDMA_lld_chainChannel(uint32 baseAddr,
                       uint32 paRAMId1,
                       uint32 chId2,
                       uint32 chainOptions)
{
    CDD_EDMACCEDMACCPaRAMEntry *currPaRAM     = NULL;
    uint32           currPaRAMAddr = baseAddr + CDD_EDMA_TPCC_OPT(paRAMId1);
    uint32           optVal;
    uintptr_t          optAddr;

    /* Get param set for the channel Id passed*/
    currPaRAM = (CDD_EDMACCEDMACCPaRAMEntry *) (currPaRAMAddr);

    optAddr    = (uintptr_t) &currPaRAM->opt;
    optVal = HW_RD_REG32((uint32) optAddr);
    optVal &= ~(CDD_EDMA_OPT_TCCHEN_MASK | CDD_EDMA_OPT_ITCCHEN_MASK |
               CDD_EDMA_OPT_TCINTEN_MASK | CDD_EDMA_OPT_ITCINTEN_MASK);
    optVal |= chainOptions;
    optVal &= ~CDD_EDMA_TPCC_OPT_TCC_MASK;
    optVal |= (chId2 << CDD_EDMA_TPCC_OPT_TCC_SHIFT) & CDD_EDMA_TPCC_OPT_TCC_MASK;
    HW_WR_REG32((uint32) optAddr, optVal);
}

void CDD_EDMA_lld_linkChannel(uint32 baseAddr,
                      uint32 paRAMId1,
                      uint32 paRAMId2)
{
    CDD_EDMACCEDMACCPaRAMEntry *currPaRAM1;
    CDD_EDMACCEDMACCPaRAMEntry *currPaRAM2;
    uint32           optVal1, optVal2;
    uint32           currPaRAMAddr1 = baseAddr + CDD_EDMA_TPCC_OPT(paRAMId1);
    uint32           currPaRAMAddr2 = baseAddr + CDD_EDMA_TPCC_OPT(paRAMId2);
    uintptr_t          lnkAddr;

    /* Get param set for the paRAMId1 passed*/
    currPaRAM1 = (CDD_EDMACCEDMACCPaRAMEntry *) (currPaRAMAddr1);

    /* Update the Link field with lch2 PaRAM set */
    lnkAddr = (uintptr_t) &currPaRAM1->linkAddr;
    HW_WR_REG16(lnkAddr,
        (uint16) ((baseAddr + CDD_EDMA_TPCC_OPT(paRAMId2)) & (uint16) 0x0FFFF));

    /* Get param set for the paRAMId2 passed*/
    currPaRAM2 = (CDD_EDMACCEDMACCPaRAMEntry *) (currPaRAMAddr2);

    /*Updated TCC value of param2 with that of param1*/
    optVal1 = HW_RD_REG32((uint32) &currPaRAM1->opt);
    optVal2 = HW_RD_REG32((uint32) &currPaRAM2->opt);
    optVal2 &= ~CDD_EDMA_TPCC_OPT_TCC_MASK;
    optVal2 |= optVal1 & CDD_EDMA_TPCC_OPT_TCC_MASK;
    HW_WR_REG32((uint32) &currPaRAM2->opt, optVal2);
}

void CDD_EDMA_lld_init(Cdd_EDMALLD_Handle hEdma)
{
    sint32 status = MCAL_SystemP_SUCCESS;
    Cdd_EDMALLD_InitHandle  hEdmaInit;

    if((hEdma != NULL) && (hEdma->hEdmaInit != NULL))
    {
        if(TRUE == hEdma->isOpen)
        {
            /* Handle is already opened */
            status = MCAL_SystemP_FAILURE;
        }
    }
    else
    {
        status = MCAL_SystemP_FAILURE;
    }

    if(MCAL_SystemP_SUCCESS == status)
    {
        hEdmaInit = hEdma->hEdmaInit;

        /* Init state */
        CDD_EDMA_lld_initialize(hEdma);
         if (hEdmaInit->intrEnable == TRUE)
        {
            /* Register the master ISR. */
            /* Enable the aggregated interrupt. */
            HW_WR_REG32(hEdmaInit->intrAggEnableAddr, hEdmaInit->intrAggEnableMask);
        }
        hEdma->firstIntr = NULL;
        hEdma->isOpen = TRUE;
    }
}

void CDD_EDMA_lld_deInit(Cdd_EDMALLD_Handle hEdma)
{
    if((NULL != hEdma) && (hEdma->isOpen != (uint32)FALSE))
    {
        CDD_EDMA_lld_deinitialize(hEdma);

        hEdma->isOpen = FALSE;
    }
    else
    {
        //Do Nothing
    }
}

uint32 CDD_EDMA_lld_isInterruptEnabled(Cdd_EDMALLD_Handle hEdma)
{
    return (hEdma->hEdmaInit->intrEnable);
}

uint32 CDD_EDMA_lld_isInitialized(Cdd_EDMALLD_Handle hEdma)
{
    return (hEdma->isOpen);
}

static sint32 CDD_EDMA_validateIntrObject(Cdd_Edma_IntrObject *intrObj)
{
    sint32 status = MCAL_SystemP_SUCCESS;

    if (intrObj->cbFxn == NULL)
    {
        /* Callback function cant be NULL for registering interrupt. */
        status = MCAL_SystemP_FAILURE;
    }
    if (status == MCAL_SystemP_SUCCESS)
    {
        /* TODO: validate tccnum */
    }
    return status;
}

sint32 CDD_EDMA_lld_registerIntr(Cdd_EDMALLD_Handle hEdma, Cdd_Edma_IntrObject *intrObj)
{
    sint32 status = MCAL_SystemP_SUCCESS;
    Cdd_EDMALLD_InitHandle     hEdmaInit;

    if ((hEdma == NULL) || (intrObj == NULL))
    {
        status = MCAL_SystemP_FAILURE;
    }
    if (status == MCAL_SystemP_SUCCESS)
    {
        status = CDD_EDMA_validateIntrObject(intrObj);
    }
    if (status == MCAL_SystemP_SUCCESS)
    {
        hEdmaInit = hEdma->hEdmaInit;

        if((hEdma->isOpen != (uint32)FALSE) &&
        (hEdmaInit->intrEnable != FALSE))
        {
            lock_mutex();
            if (hEdma->firstIntr == NULL)
            {
                hEdma->firstIntr = intrObj;
                intrObj->nextIntr = NULL;
                intrObj->prevIntr = NULL;
            }
            else
            {
                /* Insert the intrObj at the end of the list. */
                Cdd_Edma_IntrObject *tempObj;
                tempObj = hEdma->firstIntr;
                while (tempObj->nextIntr != NULL)
                {
                    tempObj = tempObj->nextIntr;
                }
                tempObj->nextIntr = intrObj;
                intrObj->nextIntr = NULL;
                intrObj->prevIntr = tempObj;
            }
            unlock_mutex();

            /* Enable the tcc interrupt bit. */
            CDD_EDMA_lld_enableEvtIntrRegion(hEdma->baseAddr, hEdmaInit->regionId, intrObj->tccNum);
        }
        else
        {
            status = MCAL_SystemP_FAILURE;
        }
    }
    return status;
}

sint32 CDD_EDMA_lld_unregisterIntr(Cdd_EDMALLD_Handle hEdma, Cdd_Edma_IntrObject *intrObj)
{
    sint32 status = MCAL_SystemP_SUCCESS;
    Cdd_EDMALLD_InitHandle     hEdmaInit;

    if ((hEdma == NULL) || (intrObj == NULL))
    {
        status = MCAL_SystemP_FAILURE;
    }
    if (status == MCAL_SystemP_SUCCESS)
    {
        if(hEdma->isOpen != (uint32)FALSE)
        {
            Cdd_Edma_IntrObject *tempObj;
            uint32   elementFound = (uint32)FALSE;
            hEdmaInit = hEdma->hEdmaInit;

            lock_mutex();
            /* Find the intrObj in the list. */
            if (hEdma->firstIntr == intrObj)
            {
                tempObj = hEdma->firstIntr;
                hEdma->firstIntr = tempObj->nextIntr;
                elementFound = TRUE;
            }
            else
            {
                tempObj = hEdma->firstIntr;
                while ((tempObj != NULL) && (tempObj != intrObj))
                {
                    tempObj = tempObj->nextIntr;
                }
                if (tempObj == intrObj)
                {
                    Cdd_Edma_IntrObject *prevObj;
                    /* Element is found. Remove current object from list. */
                    prevObj = tempObj->prevIntr;
                    prevObj->nextIntr = tempObj->nextIntr;
                    elementFound = TRUE;
                }
            }
            unlock_mutex();

            if (elementFound == TRUE)
            {
                intrObj->prevIntr = NULL;
                intrObj->nextIntr = NULL;
                /* Disable the tcc interrupt bit. */
                CDD_EDMA_lld_disableEvtIntrRegion(hEdma->baseAddr, hEdmaInit->regionId, intrObj->tccNum);
            }
            else
            {
                status = MCAL_SystemP_FAILURE;
            }
        }
    }
    return status;
}

uint32 CDD_EDMA_lld_getBaseAddr(Cdd_EDMALLD_Handle hEdma)
{
    uint32            baseAddr = 0;

    if (hEdma != NULL)
    {
        if(hEdma->isOpen != (uint32)FALSE)
        {
             baseAddr = hEdma->baseAddr;
        }
    }
    return baseAddr;
}

uint32 CDD_EDMA_lld_getRegionId(Cdd_EDMALLD_Handle hEdma)
{
    uint32            regionId = CDD_EDMA_NUM_REGIONS;

    if (hEdma != NULL)
    {
        if(hEdma->isOpen != (uint32)FALSE)
        {
            regionId = hEdma->hEdmaInit->regionId;
        }
    }
    return regionId;
}

sint32 CDD_EDMA_lld_allocDmaChannel(Cdd_EDMALLD_Handle hEdma, uint32 *dmaCh)
{
    return (CDD_EDMA_lld_allocResource(hEdma, CDD_EDMA_RESOURCE_TYPE_DMA, dmaCh));
}

sint32 CDD_EDMA_lld_allocQdmaChannel(Cdd_EDMALLD_Handle hEdma, uint32 *qdmaCh)
{
    return (CDD_EDMA_lld_allocResource(hEdma, CDD_EDMA_RESOURCE_TYPE_QDMA, qdmaCh));
}

sint32 CDD_EDMA_lld_allocTcc(Cdd_EDMALLD_Handle hEdma, uint32 *tcc)
{
    return (CDD_EDMA_lld_allocResource(hEdma, CDD_EDMA_RESOURCE_TYPE_TCC, tcc));
}

sint32 CDD_EDMA_lld_allocParam(Cdd_EDMALLD_Handle hEdma, uint32 *param)
{
    return (CDD_EDMA_lld_allocResource(hEdma, CDD_EDMA_RESOURCE_TYPE_PARAM, param));
}

static sint32 CDD_EDMA_lld_allocResource(Cdd_EDMALLD_Handle hEdma, uint32 resType, uint32 *resId)
{
    sint32 status = MCAL_SystemP_SUCCESS;

    if ((hEdma == NULL) || (resId == NULL))
    {
        status = MCAL_SystemP_FAILURE;
    }
    if (status == MCAL_SystemP_SUCCESS)
    {
        if(hEdma->isOpen != (uint32)FALSE)
        {
            switch (resType)
            {
                case CDD_EDMA_RESOURCE_TYPE_DMA:
                    status = CDD_EDMA_lld_Alloc_resource(hEdma, resId, CDD_EDMA_RESOURCE_TYPE_DMA);
                    break;
                case CDD_EDMA_RESOURCE_TYPE_QDMA:
                    status = CDD_EDMA_lld_Alloc_resource(hEdma, resId, CDD_EDMA_RESOURCE_TYPE_QDMA);
                    break;
                case CDD_EDMA_RESOURCE_TYPE_TCC:
                    status = CDD_EDMA_lld_Alloc_resource(hEdma, resId, CDD_EDMA_RESOURCE_TYPE_TCC);
                    break;
                case CDD_EDMA_RESOURCE_TYPE_PARAM:
                    status = CDD_EDMA_lld_Alloc_resource(hEdma, resId, CDD_EDMA_RESOURCE_TYPE_PARAM);
                    break;
                default:
                    status = MCAL_SystemP_FAILURE;
                    break;
            }
        }
    }
    return status;
}

static sint32 CDD_EDMA_lld_Alloc_resource(Cdd_EDMALLD_Handle hEdma, uint32 *resId, uint32 resType)
{
    uint32    i,j;
    sint32     status = MCAL_SystemP_SUCCESS;
    uint32    *allocPtr;
    const uint32 *ownPtr, *reservedPtr;
    uint32    resPtrLen, maxRes;
    Cdd_EDMALLD_InitHandle hEdmaInit = hEdma->hEdmaInit;
    CDD_EDMA_ResourceObject *ownResource = &hEdmaInit->ownResource;

    switch (resType)
    {
        case CDD_EDMA_RESOURCE_TYPE_DMA:
            allocPtr = &hEdma->allocResource.dmaCh[0];
            ownPtr = &ownResource->dmaCh[0];
            reservedPtr = &hEdmaInit->reservedDmaCh[0];
            resPtrLen = CDD_EDMA_NUM_DMACH/32;
            maxRes = CDD_EDMA_NUM_DMACH;
            break;
        case CDD_EDMA_RESOURCE_TYPE_QDMA:
            allocPtr = &hEdma->allocResource.qdmaCh;
            ownPtr = &ownResource->qdmaCh;
            resPtrLen = 1;
            maxRes = CDD_EDMA_NUM_QDMACH;
            break;
        case CDD_EDMA_RESOURCE_TYPE_TCC:
            allocPtr = &hEdma->allocResource.tcc[0];
            ownPtr = &ownResource->tcc[0];
            resPtrLen = CDD_EDMA_NUM_DMACH/32;
            maxRes = CDD_EDMA_NUM_DMACH;
            break;
        case CDD_EDMA_RESOURCE_TYPE_PARAM:
            allocPtr = &hEdma->allocResource.paramSet[0];
            ownPtr = &ownResource->paramSet[0];
            resPtrLen = CDD_EDMA_NUM_PARAMSETS/32;
            maxRes = CDD_EDMA_NUM_PARAMSETS;
            break;
        default:
            status = MCAL_SystemP_FAILURE;
            break;
    }

    if (status == MCAL_SystemP_SUCCESS)
    {
        /* set the status to failure.
           If allocation is successful status will be updated. */
        status = MCAL_SystemP_FAILURE;
        lock_mutex();
        if (*resId == CDD_EDMA_RESOURCE_ALLOC_ANY)
        {
            /* Find available resource. */
            for (i=0; i < resPtrLen; i++)
            {
                for (j=0; j<32; j++)
                {
                    if (resType == CDD_EDMA_RESOURCE_TYPE_DMA)
                    {
                        /* Check if the dma channel is owned and available and not reserved. */
                        if (((ownPtr[i] & (1 << j)) != 0) &&
                            ((reservedPtr[i] & (1 << j)) == 0) &&
                            ((allocPtr[i] & (1 << j)) == 0))
                        {
                            *resId = (i * 32) + j;
                            allocPtr[i] |= 1 << (j%32);
                            status = MCAL_SystemP_SUCCESS;
                            break;
                        }
                    }
                    else
                    {
                        /* Check if the resource is owned and available. */
                        if (((ownPtr[i] & (1 << j)) != 0) &&
                            ((allocPtr[i] & (1 << j)) == 0))
                        {
                            *resId = (i * 32) + j;
                            allocPtr[i] |= 1 << (j%32);
                            status = MCAL_SystemP_SUCCESS;
                            break;
                        }
                    }
                }
                if (*resId != CDD_EDMA_RESOURCE_ALLOC_ANY)
                {
                    break;
                }
            }
        }
        else
        {
            /* Check if the resource is already allocated. */
            if ((*resId < maxRes) &&
                ((ownPtr[*resId/32] & (1 << (*resId%32))) != 0U) &&
                ((allocPtr[*resId/32] & (1 << (*resId%32))) == 0U))
            {
                allocPtr[*resId/32] |= 1 << (*resId%32);
                status = MCAL_SystemP_SUCCESS;
            }
        }
        unlock_mutex();
    }
    return status;
}

sint32 CDD_EDMA_lld_freeDmaChannel(Cdd_EDMALLD_Handle hEdma, uint32 *dmaCh)
{
    return (CDD_EDMA_lld_freeResource(hEdma, CDD_EDMA_RESOURCE_TYPE_DMA, dmaCh));
}

sint32 CDD_EDMA_lld_freeQdmaChannel(Cdd_EDMALLD_Handle hEdma, uint32 *qdmaCh)
{
    return (CDD_EDMA_lld_freeResource(hEdma, CDD_EDMA_RESOURCE_TYPE_QDMA, qdmaCh));
}

sint32 CDD_EDMA_lld_freeTcc(Cdd_EDMALLD_Handle hEdma, uint32 *tcc)
{
    return (CDD_EDMA_lld_freeResource(hEdma, CDD_EDMA_RESOURCE_TYPE_TCC, tcc));
}

sint32 CDD_EDMA_lld_freeParam(Cdd_EDMALLD_Handle hEdma, uint32 *param)
{
    return (CDD_EDMA_lld_freeResource(hEdma, CDD_EDMA_RESOURCE_TYPE_PARAM, param));
}

static sint32 CDD_EDMA_lld_freeResource(Cdd_EDMALLD_Handle hEdma, uint32 resType, uint32 *resId)
{
    sint32 status = MCAL_SystemP_SUCCESS;

    if ((hEdma == NULL) || (resId == NULL))
    {
        status = MCAL_SystemP_FAILURE;
    }
    if ((status == MCAL_SystemP_SUCCESS) &&
        (((resType == CDD_EDMA_RESOURCE_TYPE_DMA  ) && (*resId >= CDD_EDMA_NUM_DMACH    )) ||
        ((resType == CDD_EDMA_RESOURCE_TYPE_QDMA ) && (*resId >= CDD_EDMA_NUM_QDMACH   )) ||
        ((resType == CDD_EDMA_RESOURCE_TYPE_TCC  ) && (*resId >= CDD_EDMA_NUM_DMACH    )) ||
        ((resType == CDD_EDMA_RESOURCE_TYPE_PARAM) && (*resId >= CDD_EDMA_NUM_PARAMSETS))))
    {
        status = MCAL_SystemP_FAILURE;
    }
    if (status == MCAL_SystemP_SUCCESS)
    {
        if(hEdma->isOpen != (uint32)FALSE)
        {
            lock_mutex();
            switch (resType)
            {
                case CDD_EDMA_RESOURCE_TYPE_DMA:
                    hEdma->allocResource.dmaCh[*resId/32] &= ~(1U << *resId%32);
                    break;
                case CDD_EDMA_RESOURCE_TYPE_QDMA:
                    hEdma->allocResource.qdmaCh &= ~(1U << *resId%32);
                    break;
                case CDD_EDMA_RESOURCE_TYPE_TCC:
                    hEdma->allocResource.tcc[*resId/32] &= ~(1U << *resId%32);
                    break;
                case CDD_EDMA_RESOURCE_TYPE_PARAM:
                    hEdma->allocResource.paramSet[*resId/32] &= ~(1U << *resId%32);
                    break;
                default:
                    status = MCAL_SystemP_FAILURE;
                    break;
            }
            unlock_mutex();
        }
    }
    return status;
}

void CDD_EDMA_lld_transferCompletionMasterIsrFxn(void)
{
    Cdd_Edma_IntrObject *intrObj;
    uint32 baseAddr, regionId;
    uint32 intrLow, intrHigh;
    Cdd_EDMALLD_Handle hEdma;
    Cdd_EDMALLD_InitHandle hEdmaInit;

    if(gEdmaHandle != NULL)
    {
        hEdma = (Cdd_EDMALLD_Handle)gEdmaHandle;
        hEdmaInit = hEdma->hEdmaInit;
        baseAddr = hEdma->baseAddr;
        regionId = hEdmaInit->regionId;

        intrLow = CDD_EDMA_lld_getIntrStatusRegion(baseAddr, regionId);
        intrHigh = CDD_EDMA_lld_intrStatusHighGetRegion(baseAddr, regionId);

        intrObj = hEdma->firstIntr;
        while ((intrObj != NULL) && ((intrLow != 0) || (intrHigh != 0)))
        {
            if ((intrObj->tccNum < 32U) && ((intrLow & (1U << intrObj->tccNum)) != 0U))
            {
                CDD_EDMA_lld_clrIntrRegion(baseAddr, regionId, intrObj->tccNum);
                intrLow &= ~(1U << intrObj->tccNum);
                intrObj->cbFxn(intrObj, intrObj->appData);
            }
            
			if ((intrObj->tccNum >= 32U) && ((intrHigh & (1U << (intrObj->tccNum - 32U))) != 0U))
            {
                CDD_EDMA_lld_clrIntrRegion(baseAddr, regionId, intrObj->tccNum);
                intrHigh &= ~(1U << (intrObj->tccNum - 32U));
                intrObj->cbFxn(intrObj, intrObj->appData);
            }
            /* Get next intr Obj. */
            intrObj = intrObj->nextIntr;
        }

        /* Clear the aggregator interrupt */
        HW_WR_REG32(hEdmaInit->intrAggStatusAddr, hEdmaInit->intrAggClearMask);
        /* re evaluate the edma interrupt. */
        HW_WR_FIELD32(baseAddr + CDD_EDMA_TPCC_IEVAL_RN(regionId), CDD_EDMA_TPCC_IEVAL_RN_EVAL, 1);
    }
}

#define CDD_DMA_STOP_SEC_CODE
#include "Cdd_Dma_MemMap.h"