/* ======================================================================
 *   Copyright (c) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file   cpsw.c
 *
 *  \brief  This file contains the device abstraction layer API implementation
 *          corresponding to CPSW subsystem.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "Std_Types.h"
/* There are static inline functions in hw_types.h file. Map them as well */
#define ETH_START_SEC_CODE
/* MISRAC_2012_R.20.1
  * "Reason - This is the format to use for specifying memory sections " */
#include "Eth_MemMap.h"
#include "hw_types.h"
#define ETH_STOP_SEC_CODE
/* MISRAC_2012_R.20.1
  * "Reason - This is the format to use for specifying memory sections " */
#include "Eth_MemMap.h"
#include "hw_cpsw_ale.h"
#include "hw_cpsw_cpdma.h"
#include "hw_cpsw_cpts.h"
#include "hw_cpsw_port.h"
#include "hw_cpsw_ss.h"
#include "hw_cpsw_stats.h"
#include "cpsw_priv.h"
#include "cpsw.h"
#include "cpsw_mdio.h"
#include "Dem.h"
#include "Os.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */
#define ETH_START_SEC_CODE
/* MISRAC_2012_R.20.1
  * "Reason - This is the format to use for specifying memory sections " */
#include "Eth_MemMap.h"
FUNC(void, ETH_CODE) CPSWEnetReset(uint32 baseAddr, uint8 slavePortNum)
{
    TickType   startCount = 0U, tempCount, elaspsedCount = 0U;
    /* Reset the CPDMA */
    HW_WR_FIELD32((baseAddr + CPSW_PN_OFFSET(slavePortNum) + 
		CPSW_ETH_PN_MAC_SOFT_RESET_REG),
        CPSW_ETH_PN_MAC_SOFT_RESET_REG_SOFT_RESET,
        CPSW_ETH_PN_MAC_SOFT_RESET_REG_SOFT_RESET_MASK);

    (void)GetCounterValue(ETH_OS_COUNTER_ID, &startCount);
	while (CPSW_ETH_PN_MAC_SOFT_RESET_REG_SOFT_RESET_MASK ==
		(uint32) HW_RD_FIELD32((baseAddr + CPSW_PN_OFFSET(slavePortNum) + 
			CPSW_ETH_PN_MAC_SOFT_RESET_REG),
			CPSW_ETH_PN_MAC_SOFT_RESET_REG_SOFT_RESET))
	{
		/* Below API can change start time, so use temp variable */
		tempCount = startCount;
		(void)GetElapsedValue(
			ETH_OS_COUNTER_ID,
			&tempCount,
			&elaspsedCount);
		if (elaspsedCount >= ETH_TIMEOUT_DURATION)
		{
			/* timeout */
#if (ETH_E_HARDWARE_ERROR != ETH_DEM_NO_EVENT)
			Dem_SetEventStatus(
				ETH_E_HARDWARE_ERROR, DEM_EVENT_STATUS_FAILED);
#endif
			break;
		}
	}
    
    return;

}

FUNC(void, ETH_CODE) CPSWCpdmaReset(uint32 baseAddr)
{
    TickType   startCount = 0U, tempCount, elaspsedCount = 0U;
    /* Reset the CPDMA */
    HW_WR_FIELD32(
        (baseAddr + CPSW_CPDMA_SOFT_RESET_REG),
        CPSW_CPDMA_SOFT_RESET,
        CPSW_CPDMA_SOFT_RESET_EN_0x1);

    (void)GetCounterValue(ETH_OS_COUNTER_ID, &startCount);
	while (CPSW_CPDMA_SOFT_RESET_EN_0x1 ==
		   (uint32) HW_RD_FIELD32(
			   (baseAddr + CPSW_CPDMA_SOFT_RESET_REG),
			   CPSW_CPDMA_SOFT_RESET))
	{
		/* Below API can change start time, so use temp variable */
		tempCount = startCount;
		(void)GetElapsedValue(
			ETH_OS_COUNTER_ID,
			&tempCount,
			&elaspsedCount);
		if (elaspsedCount >= ETH_TIMEOUT_DURATION)
		{
			/* timeout */
#if (ETH_E_HARDWARE_ERROR != ETH_DEM_NO_EVENT)
			Dem_SetEventStatus(
				ETH_E_HARDWARE_ERROR, DEM_EVENT_STATUS_FAILED);
#endif
			break;
		}
	}
    return;

}

FUNC(void, ETH_CODE) CPSWMdioInit(uint32 baseAddr,
                  uint32 mdioInputFreq,
                  uint32 mdioOutputFreq)
{
    /* Call the MDIOInit API after adding the MDIO submodule offset. */
    MDIOInit(baseAddr, mdioInputFreq, mdioOutputFreq);

    return;
}

void CPSWCpdmaWriteEoiVector(
    uint32 baseAddr, Eth_CpswWrIntrLine intrLine)
{
    if (CPSW_WR_INTR_LINE_RX == intrLine)
    {
        HW_WR_FIELD32(
            baseAddr + CPSW_CPDMA_EOI_VECTOR_REG,
            CPSW_CPDMA_EOI_VECTOR_DMA,
            0x01U);
    }
    else if (CPSW_WR_INTR_LINE_RX_THR == intrLine)
    {
        HW_WR_FIELD32(
            baseAddr + CPSW_CPDMA_EOI_VECTOR_REG,
            CPSW_CPDMA_EOI_VECTOR_DMA,
            0x00U);
    }
    else if (CPSW_WR_INTR_LINE_TX == intrLine)
    {
        HW_WR_FIELD32(
            baseAddr + CPSW_CPDMA_EOI_VECTOR_REG,
            CPSW_CPDMA_EOI_VECTOR_DMA,
            0x02U);
    }
    else /* CPSW_WR_INTR_LINE_MISC == intrLine */
    {
        HW_WR_FIELD32(
            baseAddr + CPSW_CPDMA_EOI_VECTOR_REG,
            CPSW_CPDMA_EOI_VECTOR_DMA,
            0x03U);
    }

    return;
}

void CPSWCpdmaChIntrEnable(uint32 baseAddr,
                           uint32 chNum,
                           Eth_CpswChIntr chIntr)
{
    uint32 regVal = 0U;

    if (CPSW_CH_INTR_RX == chIntr )
    {
        regVal  = HW_RD_REG32(baseAddr + CPSW_CPDMA_TH_INTSTAT_SET_REG);
        regVal |= ((uint32) 1U << chNum);
        HW_WR_REG32(baseAddr + CPSW_CPDMA_TH_INTSTAT_SET_REG, regVal);
    }
    else if (CPSW_CH_INTR_RX_THR == chIntr )
    {
        regVal  = HW_RD_REG32(baseAddr + CPSW_CPDMA_TH_INTSTAT_SET_REG);
        regVal |= ((uint32) 1U << (chNum + (uint32)0x8U));
        HW_WR_REG32(baseAddr + CPSW_CPDMA_TH_INTSTAT_SET_REG, regVal);
    }
    else /* if (CPSW_CH_INTR_TX == chIntr ) */
    {
        regVal  = HW_RD_REG32(baseAddr + CPSW_CPDMA_FH_INTSTAT_MASKED_SET_REG);
        regVal |= ((uint32) 1U << chNum);
        HW_WR_REG32(baseAddr + CPSW_CPDMA_FH_INTSTAT_MASKED_SET_REG, regVal);
    }

    return;
}

 void CPSWMDIOUsrIntrEnable(uint32 baseAddr)
{
     MDIOUsrIntrEnable(baseAddr);
    return;
}

uint32 CPSWSlGetMacCtrl(
     uint32 baseAddr, uint8 slavePortNum)
 {
     return (HW_RD_REG32(baseAddr + CPSW_PN_OFFSET(slavePortNum) + 
												CPSW_ETH_PN_MAC_CONTROL_REG));
 }

 void CPSWSetMacCtrl(
    uint32 baseAddr, uint8 slavePortNum, uint32 regVal)
{
    HW_WR_REG32(baseAddr + CPSW_PN_OFFSET(slavePortNum) + 
										CPSW_ETH_PN_MAC_CONTROL_REG, regVal);
    return;
}

 void CPSWSetRxMaxLen(
     uint32 baseAddr, uint8 slavePortNum, uint32 rxMaxLen)
 {
     if (ETH_CPSW_MAX_PACKET_LEN >= rxMaxLen)
     {
         HW_WR_REG32(baseAddr + CPSW_PN_OFFSET(slavePortNum) + 
										CPSW_ETH_PN_RX_MAXLEN_REG, rxMaxLen);
     }

     return;
 }

 uint32 CPSWCheckHostErr(uint32 baseAddr)
 {
     volatile uint32 retVal = FALSE;
     volatile uint32 regVal = 0U;

     regVal = HW_RD_REG32(baseAddr);
     if (regVal != CPSW_CPSW_ID_VER_REG_DEFAULT)
     {
         retVal = (uint32) TRUE;
     }
     else
     {
         retVal = (uint32) FALSE;
     }
     return retVal;
 }

 void CPSWSetPortSrcAddr(uint32 baseAddr,
                         uint8 slavePortNum,
                         const uint8 macAddr[6])
 {
     uint32 regVal = 0U;

     if( ((uint8)1U == slavePortNum) || ((uint8)2U == slavePortNum))
     {
         regVal = ((uint32) macAddr[0U] |
               (((uint32) macAddr[1U])
                << CPSW_ETH_PN_SA_H_REG_MACSRCADDR_39_32_SHIFT) |
               (((uint32) macAddr[2U])
                << CPSW_ETH_PN_SA_H_REG_MACSRCADDR_31_24_SHIFT) |
               (((uint32) macAddr[3U])
                << CPSW_ETH_PN_SA_H_REG_MACSRCADDR_23_16_SHIFT));
         HW_WR_REG32((baseAddr + CPSW_PN_OFFSET(slavePortNum) + 
												CPSW_ETH_PN_SA_H_REG), regVal);

         regVal = (((uint32) macAddr[4U]) |
                   ((uint32) macAddr[5U]
                    << CPSW_ETH_PN_SA_L_REG_MACSRCADDR_7_0_SHIFT));
         HW_WR_REG32((baseAddr + CPSW_PN_OFFSET(slavePortNum) + 
												CPSW_ETH_PN_SA_L_REG), regVal);
     }
     else
     {

     }
     return;
 }

 void CPSWSetPrescale(uint32 baseAddr, uint32 preScalarVal)
 {
     HW_WR_FIELD32((baseAddr + CPSW_SS_INT_CONTROL_REG),
                   CPSW_SS_INT_CONTROL_PRESCALE, preScalarVal);

     return;
 }

Std_ReturnType CPSWGetPortSrcAddr(
     uint32 baseAddr, uint32 slavePortNum, uint8 macAddr[6])
 {
     Std_ReturnType retVal = (Std_ReturnType) E_NOT_OK;
     uint32         macAddrLo = 0U, macAddrHi = 0U;

     if (((uint32)1U == slavePortNum) || ((uint32)2U == slavePortNum))
     {
         macAddrHi = HW_RD_REG32(baseAddr + CPSW_PN_OFFSET(slavePortNum) + 
														CPSW_ETH_PN_SA_H_REG);

         macAddrLo = HW_RD_REG32(baseAddr + CPSW_PN_OFFSET(slavePortNum) + 
														CPSW_ETH_PN_SA_L_REG);
     }
     else
     {
        /*only two PORT */
     }

     if ((uint32)0U == macAddrHi)
     {
         retVal = (Std_ReturnType) E_NOT_OK;
     }
     else
     {
         macAddr[0U] = (uint8) HW_GET_FIELD(macAddrHi,
                                         CPSW_ETH_PN_SA_H_REG_MACSRCADDR_47_40);
         macAddr[1U] = (uint8) HW_GET_FIELD(macAddrHi,
                                         CPSW_ETH_PN_SA_H_REG_MACSRCADDR_39_32);
         macAddr[2U] = (uint8) HW_GET_FIELD(macAddrHi,
                                         CPSW_ETH_PN_SA_H_REG_MACSRCADDR_31_24);
         macAddr[3U] = (uint8) HW_GET_FIELD(macAddrHi,
                                         CPSW_ETH_PN_SA_H_REG_MACSRCADDR_23_16);
         macAddr[4U] = (uint8) HW_GET_FIELD(macAddrLo,
                                         CPSW_ETH_PN_SA_L_REG_MACSRCADDR_15_8);
         macAddr[5U] = (uint8) HW_GET_FIELD(macAddrLo,
                                          CPSW_ETH_PN_SA_L_REG_MACSRCADDR_7_0);
         retVal = (Std_ReturnType) E_OK;
     }

     return (retVal);
 }

uint32 CPSWChIntrStatus(
    uint32 baseAddr, Eth_CpswChIntr chIntr)
{
    uint32 regVal = 0U;

    if (CPSW_CH_INTR_RX_THR == chIntr )
    {
        regVal = HW_RD_REG32(
            baseAddr + CPSW_SS_TH_THRESH_PULSE_STATUS_REG);
    }
    else if (CPSW_CH_INTR_RX == chIntr )
    {
        regVal = HW_RD_REG32(baseAddr + CPSW_SS_TH_PULSE_STATUS_REG);
    }
    else /* if (CPSW_CH_INTR_TX == chIntr )*/
    {
        regVal = HW_RD_REG32(baseAddr + CPSW_SS_FH_PULSE_STATUS_REG);
    }

    return regVal;
}

void CPSWEnableIntrPacing(
    uint32 baseAddr, Eth_CpswWrInt intrType, uint32 paceVal)
{
    if (CPSW_INTR_TYPE_RX == intrType)
    {
        /* Set  maximum receive Interrupts per Millisecond*/
        HW_WR_FIELD32((baseAddr + CPSW_SS_TH_IMAX_REG),
                      CPSW_SS_TH_IMAX_REG, paceVal);
        /* Enable pacing for Rx interrupts int control */
        HW_WR_FIELD32((baseAddr + CPSW_SS_INT_CONTROL_REG),
                      CPSW_SS_INT_CONTROL_REG_INT_BYPASS_RX, 1U);
    }
    if (CPSW_INTR_TYPE_TX == intrType)
    {
        /* Set  maximum receive Interrupts per Millisecond*/
        HW_WR_FIELD32((baseAddr + CPSW_SS_FH_IMAX_REG),
                      CPSW_SS_FH_IMAX_REG, paceVal);
        /* Enable pacing for Tx interrupts in wrapper int control */
        HW_WR_FIELD32((baseAddr + CPSW_SS_INT_CONTROL_REG),
                      CPSW_SS_INT_CONTROL_REG_INT_BYPASS_TX, 1U);
    }

    return;
}

void CPSWCpdmaDmaIntrEnable(
    uint32 baseAddr, Eth_CpswDMAIntr chIntr)
{
    uint32 regVal = 0U;

    if (CPSW_STAT_INT == chIntr )
    {
        regVal  = HW_RD_REG32(baseAddr + CPSW_CPDMA_INTSTAT_SET_REG);
        regVal |= ((uint32) 1U << (uint32)chIntr);
        HW_WR_REG32(baseAddr + CPSW_CPDMA_INTSTAT_SET_REG, regVal);
    }

    if (CPSW_HOST_ERR_INT == chIntr)
    {
        regVal  = HW_RD_REG32(baseAddr + CPSW_CPDMA_INTSTAT_SET_REG);
        regVal |= ((uint32) 1U << (uint32)chIntr);
        HW_WR_REG32(baseAddr + CPSW_CPDMA_INTSTAT_SET_REG, regVal);
    }

    return;
}

void CPSWCpdmaDmaIntrDisable(
    uint32 baseAddr, Eth_CpswDMAIntr chIntr)
{
    uint32 regVal = 0U;

    if (CPSW_STAT_INT == chIntr)
    {
        regVal = ((uint32) 1U << (uint32)chIntr);
        HW_WR_REG32(baseAddr + CPSW_CPDMA_INTSTAT_CLR_REG, regVal);
    }

    if (CPSW_HOST_ERR_INT == chIntr)
    {
        regVal = ((uint32) 1U << (uint32)chIntr);
        HW_WR_REG32(baseAddr + CPSW_CPDMA_INTSTAT_CLR_REG, regVal);
    }

    return;
}

void CPSWCpdmaXferEnable(uint32 baseAddr, Eth_XferModeType xferMode)
{
    if (ETH_XFER_MODE_RX == xferMode)
    {
        HW_WR_FIELD32(
            baseAddr + CPSW_CPDMA_TH_CONTROL_REG,
            CPSW_CPDMA_TH_CONTROL_EN,
            CPSW_CPDMA_TH_CONTROL_EN_EN_0x1);
    }
    else /*ETH_XFER_MODE_TX == xferMode*/
    {
        HW_WR_FIELD32(
            baseAddr + CPSW_CPDMA_FH_CONTROL_REG,
            CPSW_CPDMA_FH_CONTROL_EN,
            CPSW_CPDMA_FH_CONTROL_EN_EN_0x1);
    }

    return;
}

void CPSWCpdmaXferDisable(uint32 baseAddr, Eth_XferModeType xferMode)
{
    if (ETH_XFER_MODE_RX == xferMode)
    {
        HW_WR_FIELD32(
            baseAddr + CPSW_CPDMA_TH_CONTROL_REG,
            CPSW_CPDMA_TH_CONTROL_EN,
            CPSW_CPDMA_TH_CONTROL_EN_EN_0x0);
    }
    else /*ETH_XFER_MODE_TX == xferMode*/
    {
        HW_WR_FIELD32(
            baseAddr + CPSW_CPDMA_TH_CONTROL_REG,
            CPSW_CPDMA_FH_CONTROL_EN,
            CPSW_CPDMA_FH_CONTROL_EN_EN_0x0);
    }

    return;
}

void CPSWCpdmaXferCpWrite(uint32 baseAddr,
                          Eth_XferModeType xferMode,
                          uint32 chNum,
                          uint32 comPtr)
{
    if (ETH_XFER_MODE_RX == xferMode)
    {
        HW_WR_REG32(
            baseAddr + CPSW_CPDMA_TH_CP_REG(chNum),
            comPtr);
    }
    else /* ETH_XFER_MODE_TX == xferMode)*/
    {
        HW_WR_REG32(
            baseAddr + CPSW_CPDMA_FH_CP_REG(chNum),
            comPtr);
    }

    return;
}

void CPSWSetCpdmaNumFreeBuf(uint32 baseAddr,
                            uint32 chNum,
                            uint32 numFreeBuf)
{
    HW_WR_REG32(
        baseAddr + CPSW_CPDMA_TH_FREEBUFFER_REG(chNum),
        numFreeBuf);

    return;
}

void CPSWSetCpdmaRxThreshold(uint32 baseAddr,
                             uint32 chNum,
                             uint32 rxTreshVal)
{
    HW_WR_REG32(
        baseAddr + CPSW_CPDMA_TH_PENDTHRESH_REG(chNum),
        rxTreshVal);

    return;
}

void CPSWCpdmaChTearDown(
    uint32 baseAddr, Eth_XferModeType xferMode, uint32 chNum)
{
    if (ETH_XFER_MODE_RX == xferMode)
    {
        HW_WR_FIELD32(
            baseAddr + CPSW_CPDMA_TH_TEARDOWN_REG,
            CPSW_CPDMA_TH_TEARDOWN_TDN_CH,
            chNum);
    }
    else /*ETH_XFER_MODE_TX == xferMode)*/
    {
        HW_WR_FIELD32(
            baseAddr + CPSW_CPDMA_FH_TEARDOWN_REG,
            CPSW_CPDMA_FH_TEARDOWN_TDN_CH,
            chNum);
    }
}

void CPSWCpdmaChIntrDisable(uint32 baseAddr,
                            uint32 chNum,
                            Eth_CpswChIntr chIntr)
{
    uint32 regVal = 0U;

    if (CPSW_CH_INTR_RX == chIntr )
    {
        regVal  = HW_RD_REG32(baseAddr + CPSW_CPDMA_TH_INTSTAT_CLR_REG);
        regVal |= ((uint32) 1U << chNum);
        HW_WR_REG32(baseAddr + CPSW_CPDMA_TH_INTSTAT_CLR_REG, regVal);
    }
    else if (CPSW_CH_INTR_RX_THR == chIntr )
    {
        regVal =
            HW_RD_REG32(baseAddr + CPSW_CPDMA_INTSTAT_CLR_REG);
        regVal |= ((uint32) 1U << (chNum + (uint32)0x8U));
        HW_WR_REG32(baseAddr + CPSW_CPDMA_INTSTAT_CLR_REG, regVal);
    }
    else 	/* if (CPSW_CH_INTR_TX == chIntr) */
    {
        regVal  = HW_RD_REG32(baseAddr + CPSW_CPDMA_FH_INTSTAT_MASKED_CLR_REG);
        regVal |= ((uint32) 1U << chNum);
        HW_WR_REG32(baseAddr + CPSW_CPDMA_FH_INTSTAT_MASKED_CLR_REG, regVal);
    }

    return;
}

void CPSWCpdmaReadRxCP(
    uint32  baseAddr, uint32  chNum, uint32 *regValPtr)
{
    *regValPtr = HW_RD_REG32(
        baseAddr + CPSW_CPDMA_TH_CP_REG(chNum));
}

void CPSWCpdmaReadTxCP(
    uint32  baseAddr, uint32  chNum, uint32 *regValPtr)
{
    *regValPtr = HW_RD_REG32(
        baseAddr + CPSW_CPDMA_FH_CP_REG(chNum));
}

uint32 CPSWMiscIntrStatus(uint32 baseAddr)
{
    return (HW_RD_REG32(baseAddr + CPSW_SS_MISC_STATUS_REG));
}

void CPSWMdioPhyRegRead(uint32  baseAddr,
                        uint8   phyAddr,
                        uint8  regNum,
                        uint16 *pData)
{
    /* Call the MDIOPhy Read API after adding the MDIO submodule offset. */
    MDIOPhyRegRead(baseAddr, phyAddr, regNum, pData);

    return;
}

 void CPSWMdioPhyRegWrite(uint32 baseAddr,
                         uint8  phyAddr,
                         uint8 regNum,
                         uint16 wrVal)
{
    /* Call the MDIOPhy Write API. */
    MDIOPhyRegWrite(baseAddr, phyAddr, regNum, wrVal);

    return;
}

uint32 CPSWMdioRegRead(uint32 baseAddr,
                       uint32 regAddr)
{
    /* Call the MDIOPhy Read API */
    return (MDIORegRead(baseAddr, regAddr));
}

void CPSWChIntrEnable(uint32 baseAddr,
                        uint32 chNum,
                        Eth_CpswChIntr chIntr)
{
    uint32 regVal = 0U;

    if (CPSW_CH_INTR_RX_THR == chIntr )
    {
        regVal =
            HW_RD_REG32(baseAddr + CPSW_SS_TH_THRESH_PULSE_EN_REG);
        regVal |= ((uint32) 1U << chNum);
        HW_WR_REG32(
            baseAddr + CPSW_SS_TH_THRESH_PULSE_EN_REG, regVal);
	}		
    else if (CPSW_CH_INTR_RX == chIntr)
    {
        regVal  = HW_RD_REG32(baseAddr + CPSW_SS_TH_PULSE_EN_REG);
        regVal |= ((uint32) 1U << chNum);
        HW_WR_REG32(baseAddr + CPSW_SS_TH_PULSE_EN_REG, regVal);
    }
    else /*if (CPSW_CH_INTR_TX == chIntr)*/
    {
        regVal  = HW_RD_REG32(baseAddr + CPSW_SS_FH_PULSE_EN_REG);
        regVal |= ((uint32) 1U << chNum);
        HW_WR_REG32(baseAddr + CPSW_SS_FH_PULSE_EN_REG, regVal);
    }

    return;
}

 void CPSWMiscIntrEnable(
    uint32 baseAddr, uint32 miscIntrMask)
{
    uint32 regVal = 0U, regFieldVal = 0U;

    regVal = HW_RD_REG32(
        baseAddr + CPSW_SS_MISC_EN_REG);
    regFieldVal =
        HW_GET_FIELD(regVal, CPSW_SS_MISC_EN);
    regFieldVal |= miscIntrMask;
    HW_WR_FIELD32(
        baseAddr + CPSW_SS_MISC_EN_REG,
        CPSW_SS_MISC_EN,
        regFieldVal);

    return;
}

void CPSWMiscIntrDisable(uint32 baseAddr,
                           uint32 miscIntrMask)
{
    uint32 regVal = 0U, regFieldVal = 0U;

    regVal = HW_RD_REG32(
        baseAddr + CPSW_SS_MISC_EN_REG);
    regFieldVal =
        HW_GET_FIELD(regVal, CPSW_SS_MISC_EN);
    regFieldVal &= ~miscIntrMask;

    HW_WR_FIELD32(
        baseAddr + CPSW_SS_MISC_EN_REG,
        CPSW_SS_MISC_EN,
        regFieldVal);

    return;
}

 void CPSWStatsEnable(
    uint32 baseAddr, uint32 portNum, uint32 enableStats)
{
    uint32 regVal = 0U;

    regVal = HW_RD_REG32(baseAddr + CPSW_STAT_PORT_EN_REG);

    if (((uint32) TRUE) == enableStats)
    {
        regVal |= ((uint32) 1U << portNum);
    }
    else
    {
        regVal &= ~((uint32) 1U << portNum);
    }

    HW_WR_REG32(baseAddr + CPSW_STAT_PORT_EN_REG, regVal);

    return;
}

#define ETH_STOP_SEC_CODE
/* MISRAC_2012_R.20.1
  * "Reason - This is the format to use for specifying memory sections " */
#include "Eth_MemMap.h"
