/* ======================================================================
 *   Copyright (c) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     Eth_Priv.h
 *
 *  \brief    This file contains the common declarations and macros used across
 *            all the Ethernet driver files.
 */

#ifndef ETH_PRIV_H_
#define ETH_PRIV_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "Eth_GeneralTypes.h"
#include "cpsw_priv.h"
#include "Os.h"

#ifdef __cplusplus
extern "C" {
#endif
/* ========================================================================== */
/*                                 Macros                                     */
/* ========================================================================== */

/* Default frame type for receive and transmit */
#define ETH_FRAME_DEFAULT_TYPE  (0x0800U)

/* Default TX CPDMA Channel used for reception */
#define ETH_CPDMA_DEFAULT_RX_CHANNEL_NUM    (0U)

/**
  * Default TX CPDMA Channel used for transmission
  * Note: Rate limiting can be only enabled starting with highest priority
  *       channels; in this driver we support only single channel hence we use
  *       channel 7 as default to support rate limiting
  */
#define ETH_CPDMA_DEFAULT_TX_CHANNEL_NUM    (7U)

/* Host port id
 * Note - This is fixed, dont change */
#define ETH_HOST_PORT_ID                (0U)

/* VLAN ID ZERO - no VLAN */
#define ETH_NOVLAN_ID                    (0U)

#define ETH_GMII_SEL_GMII_MODE          (0x0U)
#define ETH_GMII_SEL_RMII_MODE          (0x1U)
#define ETH_GMII_SEL_RGMII_MODE         (0x2U)
/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Variables Declarations                      */
/* ========================================================================== */

extern volatile uint8        Eth_ControllerModeChangeFlag;

extern Eth_DrvObject Eth_DrvObj;

extern volatile VAR(Eth_StateType, ETH_VAR_ZERO_INIT) Eth_DrvStatus;

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/** \brief Initializes the Ethernet Driver
*
* \param[in] CfgPtr
*
* \param[out] None
*/
void Eth_HwInit(const Eth_ConfigType *CfgPtr);

/** \brief Enables / disables the indexed controller
*
* \param[in] CtrlIdx
*			   CtrlMode
*
* \param[out] None
*/
Std_ReturnType Eth_HwSetControllerMode
                (uint8 CtrlIdx, Eth_ModeType CtrlMode);

/** \brief Sets the physical source address used by the indexed controller.
*
* \param[in] PhysAddrPtr
*
* \param[out] None
*/
void Eth_HwSetPhysAddr(P2CONST(uint8, AUTOMATIC, ETH_APPL_DATA)
                                                                  PhysAddrPtr);

/** \brief Update the physical source address to/from the indexed 
*				controller filter. If the Ethernet Controller is not capable to 
*				do the filtering, the software has to do this.
*
* \param[in] 	PhysAddrPtr
*				Action
*
* \param[out] None
*/                                                             
Std_ReturnType Eth_HwUpdatePhysAddrFilter(
                          P2VAR(uint8, AUTOMATIC,
                          ETH_APPL_DATA) PhysAddrPtr, 
						  Eth_FilterActionType Action);
						  
						  
/** \brief Provides access to a transmit buffer
*
* \param[inout]LenBytePtr
*
* \param[out] 	BufIdxPtr
*				BufPtr
*/  						  
BufReq_ReturnType Eth_HwProvideTxBufferIdx(
                        P2VAR(Eth_BufIdxType,AUTOMATIC,ETH_APPL_DATA)BufIdxPtr,
                        P2VAR(uint8,AUTOMATIC,ETH_APPL_DATA)*BufPtr,
                        P2VAR(uint16,AUTOMATIC,ETH_APPL_DATA)LenBytePtr);
						
/** \brief Provides access to a transmit buffer
*
* \param[inout]LenBytePtr
*
* \param[out] 	BufIdxPtr
*				BufPtr
*/  						
BufReq_ReturnType Eth_HwProvideTxBuffer(
                        P2VAR(Eth_BufIdxType,AUTOMATIC,ETH_APPL_DATA)BufIdxPtr,
                        P2VAR(uint8,AUTOMATIC,ETH_APPL_DATA)*BufPtr,
                        P2VAR(uint16,AUTOMATIC,ETH_APPL_DATA)LenBytePtr);

/** \brief Triggers transmission of a previously filled transmit buffer.
*
* \param[in]  BufIdx
*				FrameType
*				TxConfirmation
*				LenByte
*				PhysAddrPtr
*
* \param[out] None
*/
Std_ReturnType Eth_HwTransmit(
                            VAR(Eth_BufIdxType,AUTOMATIC)BufIdx,
                            VAR(Eth_FrameType,AUTOMATIC) FrameType,
                            VAR(boolean,AUTOMATIC) TxConfirmation,
                            VAR(uint16,AUTOMATIC) LenByte,
                            P2CONST(uint8, AUTOMATIC,ETH_APPL_DATA) PhysAddrPtr);


/** \brief Receive a frame from the related fifo.
*
*
* \param[out] RxStatusPtr
*/							
void Eth_HwReceive(
                 P2VAR(Eth_RxStatusType,AUTOMATIC,ETH_APPL_DATA)RxStatusPtr);
				 
/** \brief Reads a list with drop counter values of the corresponding 
*				controller. The meaning of these values is described at 
*				Eth_CounterType.
*
* \param[out] CounterPtr	
*/		 
void Eth_HwGetCounterValues(
                P2VAR(Eth_CounterType,AUTOMATIC,ETH_APPL_DATA)CounterPtr);
				
/** \brief Returns the following list according to IETF RFC2819, where the 
*				maximal possible value shall denote an invalid value.
* \param[out] RxStats
*/				 
void Eth_HwGetRxStats(
                P2VAR(Eth_RxStatsType,AUTOMATIC,ETH_APPL_DATA)RxStats);

/** \brief Returns the list of Transmission Statistics out of IETF RFC1213 
*				defined with Eth_TxStatsType, where the maximal possible value 
*				shall denote an invalid value, e.g. this counter is not 
*				available.
*
* \param[out] TxStats
*/				
void Eth_HwGetTxStats(
                P2VAR(Eth_TxStatsType,AUTOMATIC,ETH_APPL_DATA)TxStats);
				
/** \brief Returns the list of Transmission Error Counters out of IETF 
*				RFC1213 and RFC1643 defined with Eth_TxErrorCounterValuesType, 
*				where the maximal possible value shall denote an invalid value, 
*				e.g. this counter is not available.
*
* \param[out] TxErrorCounterValues	
*/
void Eth_HwGetTxErrorCounterValues(
    P2VAR(Eth_TxErrorCounterValuesType,AUTOMATIC,ETH_APPL_DATA)
	                                      TxErrorCounterValues);
/** \brief Reads back the egress time stamp on a dedicated message object.
*				It must be called within the TxConfirmation() function
*
* \param[in] 	BufIdx
*
* \param[out] 	timeQualPtr
*				timeStampPtr
*/	
void Eth_HwGetEgressTimeStamp(
            VAR(Eth_BufIdxType, AUTOMATIC)BufIdx,
            P2VAR(Eth_TimeStampQualType, AUTOMATIC,ETH_APPL_DATA)timeQualPtr,
            P2VAR(Eth_TimeStampType,AUTOMATIC,ETH_APPL_DATA)timeStampPtr);

/** \brief Reads back the ingress time stamp on a dedicated message object.
*				It must be called within the RxIndication() function
*
* \param[in]   DataPtr
*
* \param[out] 	timeQualPtr
*				timeStampPtr
*/			
void Eth_HwGetIngressTimeStamp(
            P2CONST(Eth_DataType,AUTOMATIC,ETH_APPL_DATA)DataPtr,
            P2VAR(Eth_TimeStampQualType,AUTOMATIC,ETH_APPL_DATA)timeQualPtr,
            P2VAR(Eth_TimeStampType ,AUTOMATIC,ETH_APPL_DATA)timeStampPtr);
/**
 * \brief   calls the the CPDMA Status function.
 */
Std_ReturnType Eth_cpswCheckHostErr(void);

/**
 * \brief   check for controller errors.
 */
void Eth_HwcheckCtrlrErrors(void);

/**
 * \brief Initializes driver object for the CPSW instance
 *
 * \param  instNum   Pointer to  CPSW driver object to be initialised.
 */
void  EthResetDrvObj(Eth_DrvObject *pEthDrvObj);

/**
 * \brief  Function to setup the instance parameters inside the interface
 *
 * \param  instNum  The instance of CPSW.
 */
void EthCpswifInstConfig(uint32 instNum);

/**
 * \brief Initializes the CPSW instance
 *
 * \param  None
 */
void EthCpswInstInit(uint32 portNum);

/**
 * \brief Reset the CPSW instance
 *
 * \param  None.
 */
void EthCpswInstUnInit(void);

/**
 * \brief Initializes the CPSW port
 *
 * \param pPortObj      Pointer to port object.
 *
 *
 * \retval E_OK        Port initialization is successful
 * \retval E_NOT_OK    Failed to initialize port
 */
void Eth_macSetConfig(uint8 portNum, Eth_MacConfigType *pMACConfig);

/**
 * \brief This function calls Cache function pointer.
 *
 *\param  pCPDMABuffDesc - Pointer to CPDMA descriptor.
 */
void EthBuffDescCacheInit(Eth_CpdmaBuffDescType *pCurrBuffDesc);

/**
 * \brief This function initialize the buffer descriptor ring for given mode of
 *        transfer.
 *
 * \param ctrlIdx       CPSW device instance number.
 * \param pRing       The buffer descriptor ring.
 * \param xferType    Transfer mode.
 * \param numBuffDesc Number of buffer descriptors to be initialized.
 * \param startAddr   The start address of Ring.
 */
void EthBuffDescInit(uint8                 ctrlIdx,
                     Eth_CpdmaBuffDescQueue *pRing,
                     enum ethXferMode        xferType,
                     uint32                numBuffDesc,
                     uint32                startAddr);

/**
 * \brief This function allocates the rx buffer descriptors ring. The function
 *        internally calls EthPktBufDescAllocate and allocates the pbufs to
 *        the rx buffer descriptors.
 *
 * \param ctrlIdx       CPSW device instance number.
 * \param pRing    The buffer descriptor ring.
 */
void EthCpswRxBuffDescAllocate(Eth_CpdmaBuffDescQueue *pRing);

/**
 * \brief Process receive buffer descriptor for single packet Rx Status.
 *
 * \param  pCPDMABuffDesc - Pointer to CPDMA descriptor.
 * \param rxStatus - Rx Status
 */
void EthRxBuffDescRxStatus(Eth_CpdmaBuffDescType *pCurrRxBuffDesc, 
												Eth_RxStatusType  *rxStatus);
/**
 * \brief Process receive buffer descriptor for single packet.
 *
 * \param ctrlIdx   The instance of CPSW interface.
 * \param chNum     Channel number.
 */
Eth_RxStatusType EthRxBuffDescProcessSingle(uint8 ctrlIdx, uint32 chNum);

/**
 * \brief Process receive packets buffer descriptors for all received packets.
 *
 * \param ctrlIdx   The instance of CPSW interface.
 * \param chNum     Channel number.
 */
void EthRxBuffDescProcess(uint8 ctrlIdx, uint32 chNum);

void EthTxBuffDescProcessFirst(uint8 ctrlIdx,
									Eth_CpdmaBuffDescType *pCurrTxBuffDesc, 
									Eth_CpdmaBuffDescQueue *pTxDescRing);
									
void EthTxBuffDescProcessSecond(Eth_CpdmaBuffDescType *pLastBuffDesc);

/**
 * \brief Process transmit packet buffer descriptor.
 *
 * \param ctrlIdx   The instance of transmit packet buffer descriptor to process
 * \param chNum     Channel number.
 */
void EthTxBuffDescProcess(uint8 ctrlIdx, uint32 chNum);

/**
 * \brief EthCheckNullMACAddr function.
 *        Check if address passed is NULL MAC address.
 *
 * \param  macAddr[ETH_MAC_ADDR_LEN] - Pointer to physical(MAC) address.
 *
 * \return TRUE if MAC address is NULL
 */
uint32 EthCheckNullMACAddr(const uint8 macAddr[ETH_MAC_ADDR_LEN]);
/*
 * \brief Eth_CPDMABuffDescClearFlags function.
 *        Initializes CPDMA descriptor flags to zero.
 *
 * \param  pCPDMABuffDesc - Pointer to CPDMA descriptor.
 *
 * \return none
 */
void Eth_CPDMABuffDescClearFlags(Eth_CpdmaBuffDescType *pCPDMABuffDesc);
/*
 * \brief Free Ethernet packet buffers.
 *
 * \param  bufObjArray - buffer array object.
 * \param  numBuffers  - no. of buffers in array
 *
 * \return none
 */
void Eth_freeBuffers(Eth_BufObjType *bufObjArray, uint32 numBuffers);


#ifdef __cplusplus
}
#endif

#endif /* #ifndef ETH_PRIV_H_ */
