/* ======================================================================
 *   Copyright (c) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     cpsw.h
 *
 *  \brief    This file contains the function prototypes for the device
 *            abstraction layer for CPSW subsystem. It also contains some
 *            related macro definitions and some files to be included.
 *
 *  \details  Please find the below detailed description of cpsw dal.
 */

#ifndef CPSW_H_
#define CPSW_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "cpsw_priv.h"
#include "hw_cpsw_ss.h"

#ifdef __cplusplus
extern "C" {
#endif
/* ========================================================================== */
/*                                 Macros                                     */
/* ========================================================================== */

/** \brief Maximum length of ethernet packet. */
#define ETH_CPSW_MAX_PACKET_LEN     (2024U)


/**************************************************************************
* Register Macros
**************************************************************************/
#define CPSW_CPSW_ID_VER_REG                                     (0x00020000U)
#define CPSW_CONTROL_REG                                         (0x00020004U)
#define CPSW_EM_CONTROL_REG                                      (0x00020010U)
#define CPSW_STAT_PORT_EN_REG                                    (0x00020014U)
#define CPSW_PTYPE_REG                                           (0x00020018U)
#define CPSW_SOFT_IDLE_REG                                       (0x0002001CU)
#define CPSW_THRU_RATE_REG                                       (0x00020020U)
#define CPSW_GAP_THRESH_REG                                      (0x00020024U)
#define CPSW_TX_START_WDS_REG                                    (0x00020028U)
#define CPSW_EEE_PRESCALE_REG                                    (0x0002002CU)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG                           (0x00020030U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG                           (0x00020034U)
#define CPSW_TX_G_BUF_THRESH_SET_L_REG                           (0x00020038U)
#define CPSW_TX_G_BUF_THRESH_SET_H_REG                           (0x0002003CU)
#define CPSW_TX_G_BUF_THRESH_CLR_L_REG                           (0x00020040U)
#define CPSW_TX_G_BUF_THRESH_CLR_H_REG                           (0x00020044U)
#define CPSW_VLAN_LTYPE_REG                                      (0x00020050U)
#define CPSW_EST_TS_DOMAIN_REG                                   (0x00020054U)
#define CPSW_TX_PRI0_MAXLEN_REG                                  (0x00020100U)
#define CPSW_TX_PRI1_MAXLEN_REG                                  (0x00020104U)
#define CPSW_TX_PRI2_MAXLEN_REG                                  (0x00020108U)
#define CPSW_TX_PRI3_MAXLEN_REG                                  (0x0002010CU)
#define CPSW_TX_PRI4_MAXLEN_REG                                  (0x00020110U)
#define CPSW_TX_PRI5_MAXLEN_REG                                  (0x00020114U)
#define CPSW_TX_PRI6_MAXLEN_REG                                  (0x00020118U)
#define CPSW_TX_PRI7_MAXLEN_REG                                  (0x0002011CU)
#define CPSW_EST_FETCH_LOC(FETCH_LOC)                    (0x00032000U+((FETCH_LOC)*0x4U))


/**************************************************************************
* Field Definition Macros
**************************************************************************/

/* FETCH_LOC */

#define CPSW_EST_FETCH_LOC_LOC_MASK                              (0x003FFFFFU)
#define CPSW_EST_FETCH_LOC_LOC_SHIFT                             (0x00000000U)
#define CPSW_EST_FETCH_LOC_LOC_MAX                               (0x003FFFFFU)

/* CPSW_ID_VER_REG */

#define CPSW_CPSW_ID_VER_REG_MINOR_VER_MASK                      (0x0000003FU)
#define CPSW_CPSW_ID_VER_REG_MINOR_VER_SHIFT                     (0x00000000U)
#define CPSW_CPSW_ID_VER_REG_MINOR_VER_MAX                       (0x0000003FU)

#define CPSW_CPSW_ID_VER_REG_CUSTOM_VER_MASK                     (0x000000C0U)
#define CPSW_CPSW_ID_VER_REG_CUSTOM_VER_SHIFT                    (0x00000006U)
#define CPSW_CPSW_ID_VER_REG_CUSTOM_VER_MAX                      (0x00000003U)

#define CPSW_CPSW_ID_VER_REG_MAJOR_VER_MASK                      (0x00000700U)
#define CPSW_CPSW_ID_VER_REG_MAJOR_VER_SHIFT                     (0x00000008U)
#define CPSW_CPSW_ID_VER_REG_MAJOR_VER_MAX                       (0x00000007U)

#define CPSW_CPSW_ID_VER_REG_RTL_VER_MASK                        (0x0000F800U)
#define CPSW_CPSW_ID_VER_REG_RTL_VER_SHIFT                       (0x0000000BU)
#define CPSW_CPSW_ID_VER_REG_RTL_VER_MAX                         (0x0000001FU)

#define CPSW_CPSW_ID_VER_REG_IDENT_MASK                          (0xFFFF0000U)
#define CPSW_CPSW_ID_VER_REG_IDENT_SHIFT                         (0x00000010U)
#define CPSW_CPSW_ID_VER_REG_IDENT_MAX                           (0x0000FFFFU)

#define CPSW_CPSW_ID_VER_REG_DEFAULT	                         (0x6BA01903U)

/* CONTROL_REG */

#define CPSW_CONTROL_REG_S_CN_SWITCH_MASK                        (0x00000001U)
#define CPSW_CONTROL_REG_S_CN_SWITCH_SHIFT                       (0x00000000U)
#define CPSW_CONTROL_REG_S_CN_SWITCH_MAX                         (0x00000001U)

#define CPSW_CONTROL_REG_VLAN_AWARE_MASK                         (0x00000002U)
#define CPSW_CONTROL_REG_VLAN_AWARE_SHIFT                        (0x00000001U)
#define CPSW_CONTROL_REG_VLAN_AWARE_MAX                          (0x00000001U)

#define CPSW_CONTROL_REG_P0_ENABLE_MASK                          (0x00000004U)
#define CPSW_CONTROL_REG_P0_ENABLE_SHIFT                         (0x00000002U)
#define CPSW_CONTROL_REG_P0_ENABLE_MAX                           (0x00000001U)

#define CPSW_CONTROL_REG_P0_PASS_PRI_TAGGED_MASK                 (0x00000008U)
#define CPSW_CONTROL_REG_P0_PASS_PRI_TAGGED_SHIFT                (0x00000003U)
#define CPSW_CONTROL_REG_P0_PASS_PRI_TAGGED_MAX                  (0x00000001U)

#define CPSW_CONTROL_REG_P1_PASS_PRI_TAGGED_MASK                 (0x00000010U)
#define CPSW_CONTROL_REG_P1_PASS_PRI_TAGGED_SHIFT                (0x00000004U)
#define CPSW_CONTROL_REG_P1_PASS_PRI_TAGGED_MAX                  (0x00000001U)

#define CPSW_CONTROL_REG_P2_PASS_PRI_TAGGED_MASK                 (0x00000020U)
#define CPSW_CONTROL_REG_P2_PASS_PRI_TAGGED_SHIFT                (0x00000005U)
#define CPSW_CONTROL_REG_P2_PASS_PRI_TAGGED_MAX                  (0x00000001U)

#define CPSW_CONTROL_REG_P3_PASS_PRI_TAGGED_MASK                 (0x00000040U)
#define CPSW_CONTROL_REG_P3_PASS_PRI_TAGGED_SHIFT                (0x00000006U)
#define CPSW_CONTROL_REG_P3_PASS_PRI_TAGGED_MAX                  (0x00000001U)

#define CPSW_CONTROL_REG_P4_PASS_PRI_TAGGED_MASK                 (0x00000080U)
#define CPSW_CONTROL_REG_P4_PASS_PRI_TAGGED_SHIFT                (0x00000007U)
#define CPSW_CONTROL_REG_P4_PASS_PRI_TAGGED_MAX                  (0x00000001U)

#define CPSW_CONTROL_REG_P5_PASS_PRI_TAGGED_MASK                 (0x00000100U)
#define CPSW_CONTROL_REG_P5_PASS_PRI_TAGGED_SHIFT                (0x00000008U)
#define CPSW_CONTROL_REG_P5_PASS_PRI_TAGGED_MAX                  (0x00000001U)

#define CPSW_CONTROL_REG_P6_PASS_PRI_TAGGED_MASK                 (0x00000200U)
#define CPSW_CONTROL_REG_P6_PASS_PRI_TAGGED_SHIFT                (0x00000009U)
#define CPSW_CONTROL_REG_P6_PASS_PRI_TAGGED_MAX                  (0x00000001U)

#define CPSW_CONTROL_REG_P7_PASS_PRI_TAGGED_MASK                 (0x00000400U)
#define CPSW_CONTROL_REG_P7_PASS_PRI_TAGGED_SHIFT                (0x0000000AU)
#define CPSW_CONTROL_REG_P7_PASS_PRI_TAGGED_MAX                  (0x00000001U)

#define CPSW_CONTROL_REG_P8_PASS_PRI_TAGGED_MASK                 (0x00000800U)
#define CPSW_CONTROL_REG_P8_PASS_PRI_TAGGED_SHIFT                (0x0000000BU)
#define CPSW_CONTROL_REG_P8_PASS_PRI_TAGGED_MAX                  (0x00000001U)

#define CPSW_CONTROL_REG_P0_TX_CRC_REMOVE_MASK                   (0x00002000U)
#define CPSW_CONTROL_REG_P0_TX_CRC_REMOVE_SHIFT                  (0x0000000DU)
#define CPSW_CONTROL_REG_P0_TX_CRC_REMOVE_MAX                    (0x00000001U)

#define CPSW_CONTROL_REG_P0_RX_PAD_MASK                          (0x00004000U)
#define CPSW_CONTROL_REG_P0_RX_PAD_SHIFT                         (0x0000000EU)
#define CPSW_CONTROL_REG_P0_RX_PAD_MAX                           (0x00000001U)

#define CPSW_CONTROL_REG_P0_RX_PASS_CRC_ERR_MASK                 (0x00008000U)
#define CPSW_CONTROL_REG_P0_RX_PASS_CRC_ERR_SHIFT                (0x0000000FU)
#define CPSW_CONTROL_REG_P0_RX_PASS_CRC_ERR_MAX                  (0x00000001U)

#define CPSW_CONTROL_REG_EEE_ENABLE_MASK                         (0x00010000U)
#define CPSW_CONTROL_REG_EEE_ENABLE_SHIFT                        (0x00000010U)
#define CPSW_CONTROL_REG_EEE_ENABLE_MAX                          (0x00000001U)

#define CPSW_CONTROL_REG_IET_ENABLE_MASK                         (0x00020000U)
#define CPSW_CONTROL_REG_IET_ENABLE_SHIFT                        (0x00000011U)
#define CPSW_CONTROL_REG_IET_ENABLE_MAX                          (0x00000001U)

#define CPSW_CONTROL_REG_EST_ENABLE_MASK                         (0x00040000U)
#define CPSW_CONTROL_REG_EST_ENABLE_SHIFT                        (0x00000012U)
#define CPSW_CONTROL_REG_EST_ENABLE_MAX                          (0x00000001U)

#define CPSW_CONTROL_REG_ECC_CRC_MODE_MASK                       (0x80000000U)
#define CPSW_CONTROL_REG_ECC_CRC_MODE_SHIFT                      (0x0000001FU)
#define CPSW_CONTROL_REG_ECC_CRC_MODE_MAX                        (0x00000001U)

/* EM_CONTROL_REG */

#define CPSW_EM_CONTROL_REG_FREE_MASK                            (0x00000001U)
#define CPSW_EM_CONTROL_REG_FREE_SHIFT                           (0x00000000U)
#define CPSW_EM_CONTROL_REG_FREE_MAX                             (0x00000001U)

#define CPSW_EM_CONTROL_REG_SOFT_MASK                            (0x00000002U)
#define CPSW_EM_CONTROL_REG_SOFT_SHIFT                           (0x00000001U)
#define CPSW_EM_CONTROL_REG_SOFT_MAX                             (0x00000001U)

/* STAT_PORT_EN_REG */

#define CPSW_STAT_PORT_EN_REG_P0_STAT_EN_MASK                    (0x00000001U)
#define CPSW_STAT_PORT_EN_REG_P0_STAT_EN_SHIFT                   (0x00000000U)
#define CPSW_STAT_PORT_EN_REG_P0_STAT_EN_MAX                     (0x00000001U)

#define CPSW_STAT_PORT_EN_REG_P1_STAT_EN_MASK                    (0x00000002U)
#define CPSW_STAT_PORT_EN_REG_P1_STAT_EN_SHIFT                   (0x00000001U)
#define CPSW_STAT_PORT_EN_REG_P1_STAT_EN_MAX                     (0x00000001U)

#define CPSW_STAT_PORT_EN_REG_P2_STAT_EN_MASK                    (0x00000004U)
#define CPSW_STAT_PORT_EN_REG_P2_STAT_EN_SHIFT                   (0x00000002U)
#define CPSW_STAT_PORT_EN_REG_P2_STAT_EN_MAX                     (0x00000001U)

#define CPSW_STAT_PORT_EN_REG_P3_STAT_EN_MASK                    (0x00000008U)
#define CPSW_STAT_PORT_EN_REG_P3_STAT_EN_SHIFT                   (0x00000003U)
#define CPSW_STAT_PORT_EN_REG_P3_STAT_EN_MAX                     (0x00000001U)

#define CPSW_STAT_PORT_EN_REG_P4_STAT_EN_MASK                    (0x00000010U)
#define CPSW_STAT_PORT_EN_REG_P4_STAT_EN_SHIFT                   (0x00000004U)
#define CPSW_STAT_PORT_EN_REG_P4_STAT_EN_MAX                     (0x00000001U)

#define CPSW_STAT_PORT_EN_REG_P5_STAT_EN_MASK                    (0x00000020U)
#define CPSW_STAT_PORT_EN_REG_P5_STAT_EN_SHIFT                   (0x00000005U)
#define CPSW_STAT_PORT_EN_REG_P5_STAT_EN_MAX                     (0x00000001U)

#define CPSW_STAT_PORT_EN_REG_P6_STAT_EN_MASK                    (0x00000040U)
#define CPSW_STAT_PORT_EN_REG_P6_STAT_EN_SHIFT                   (0x00000006U)
#define CPSW_STAT_PORT_EN_REG_P6_STAT_EN_MAX                     (0x00000001U)

#define CPSW_STAT_PORT_EN_REG_P7_STAT_EN_MASK                    (0x00000080U)
#define CPSW_STAT_PORT_EN_REG_P7_STAT_EN_SHIFT                   (0x00000007U)
#define CPSW_STAT_PORT_EN_REG_P7_STAT_EN_MAX                     (0x00000001U)

#define CPSW_STAT_PORT_EN_REG_P8_STAT_EN_MASK                    (0x00000100U)
#define CPSW_STAT_PORT_EN_REG_P8_STAT_EN_SHIFT                   (0x00000008U)
#define CPSW_STAT_PORT_EN_REG_P8_STAT_EN_MAX                     (0x00000001U)

/* PTYPE_REG */

#define CPSW_PTYPE_REG_ESC_PRI_LD_VAL_MASK                       (0x0000001FU)
#define CPSW_PTYPE_REG_ESC_PRI_LD_VAL_SHIFT                      (0x00000000U)
#define CPSW_PTYPE_REG_ESC_PRI_LD_VAL_MAX                        (0x0000001FU)

#define CPSW_PTYPE_REG_P0_PTYPE_ESC_MASK                         (0x00000100U)
#define CPSW_PTYPE_REG_P0_PTYPE_ESC_SHIFT                        (0x00000008U)
#define CPSW_PTYPE_REG_P0_PTYPE_ESC_MAX                          (0x00000001U)

#define CPSW_PTYPE_REG_P1_PTYPE_ESC_MASK                         (0x00000200U)
#define CPSW_PTYPE_REG_P1_PTYPE_ESC_SHIFT                        (0x00000009U)
#define CPSW_PTYPE_REG_P1_PTYPE_ESC_MAX                          (0x00000001U)

#define CPSW_PTYPE_REG_P2_PTYPE_ESC_MASK                         (0x00000400U)
#define CPSW_PTYPE_REG_P2_PTYPE_ESC_SHIFT                        (0x0000000AU)
#define CPSW_PTYPE_REG_P2_PTYPE_ESC_MAX                          (0x00000001U)

#define CPSW_PTYPE_REG_P3_PTYPE_ESC_MASK                         (0x00000800U)
#define CPSW_PTYPE_REG_P3_PTYPE_ESC_SHIFT                        (0x0000000BU)
#define CPSW_PTYPE_REG_P3_PTYPE_ESC_MAX                          (0x00000001U)

#define CPSW_PTYPE_REG_P4_PTYPE_ESC_MASK                         (0x00001000U)
#define CPSW_PTYPE_REG_P4_PTYPE_ESC_SHIFT                        (0x0000000CU)
#define CPSW_PTYPE_REG_P4_PTYPE_ESC_MAX                          (0x00000001U)

#define CPSW_PTYPE_REG_P5_PTYPE_ESC_MASK                         (0x00002000U)
#define CPSW_PTYPE_REG_P5_PTYPE_ESC_SHIFT                        (0x0000000DU)
#define CPSW_PTYPE_REG_P5_PTYPE_ESC_MAX                          (0x00000001U)

#define CPSW_PTYPE_REG_P6_PTYPE_ESC_MASK                         (0x00004000U)
#define CPSW_PTYPE_REG_P6_PTYPE_ESC_SHIFT                        (0x0000000EU)
#define CPSW_PTYPE_REG_P6_PTYPE_ESC_MAX                          (0x00000001U)

#define CPSW_PTYPE_REG_P7_PTYPE_ESC_MASK                         (0x00008000U)
#define CPSW_PTYPE_REG_P7_PTYPE_ESC_SHIFT                        (0x0000000FU)
#define CPSW_PTYPE_REG_P7_PTYPE_ESC_MAX                          (0x00000001U)

#define CPSW_PTYPE_REG_P8_PTYPE_ESC_MASK                         (0x00010000U)
#define CPSW_PTYPE_REG_P8_PTYPE_ESC_SHIFT                        (0x00000010U)
#define CPSW_PTYPE_REG_P8_PTYPE_ESC_MAX                          (0x00000001U)

/* SOFT_IDLE_REG */

#define CPSW_SOFT_IDLE_REG_SOFT_IDLE_MASK                        (0x00000001U)
#define CPSW_SOFT_IDLE_REG_SOFT_IDLE_SHIFT                       (0x00000000U)
#define CPSW_SOFT_IDLE_REG_SOFT_IDLE_MAX                         (0x00000001U)

/* THRU_RATE_REG */

#define CPSW_THRU_RATE_REG_P0_RX_THRU_RATE_MASK                  (0x0000000FU)
#define CPSW_THRU_RATE_REG_P0_RX_THRU_RATE_SHIFT                 (0x00000000U)
#define CPSW_THRU_RATE_REG_P0_RX_THRU_RATE_MAX                   (0x0000000FU)

#define CPSW_THRU_RATE_REG_SL_RX_THRU_RATE_MASK                  (0x0000F000U)
#define CPSW_THRU_RATE_REG_SL_RX_THRU_RATE_SHIFT                 (0x0000000CU)
#define CPSW_THRU_RATE_REG_SL_RX_THRU_RATE_MAX                   (0x0000000FU)

/* GAP_THRESH_REG */

#define CPSW_GAP_THRESH_REG_GAP_THRESH_MASK                      (0x0000001FU)
#define CPSW_GAP_THRESH_REG_GAP_THRESH_SHIFT                     (0x00000000U)
#define CPSW_GAP_THRESH_REG_GAP_THRESH_MAX                       (0x0000001FU)

/* TX_START_WDS_REG */

#define CPSW_TX_START_WDS_REG_TX_START_WDS_MASK                  (0x000007FFU)
#define CPSW_TX_START_WDS_REG_TX_START_WDS_SHIFT                 (0x00000000U)
#define CPSW_TX_START_WDS_REG_TX_START_WDS_MAX                   (0x000007FFU)

/* EEE_PRESCALE_REG */

#define CPSW_EEE_PRESCALE_REG_EEE_PRESCALE_MASK                  (0x00000FFFU)
#define CPSW_EEE_PRESCALE_REG_EEE_PRESCALE_SHIFT                 (0x00000000U)
#define CPSW_EEE_PRESCALE_REG_EEE_PRESCALE_MAX                   (0x00000FFFU)

/* TX_G_OFLOW_THRESH_SET_REG */

#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI0_MASK                 (0x0000000FU)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI0_SHIFT                (0x00000000U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI0_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI1_MASK                 (0x000000F0U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI1_SHIFT                (0x00000004U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI1_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI2_MASK                 (0x00000F00U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI2_SHIFT                (0x00000008U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI2_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI3_MASK                 (0x0000F000U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI3_SHIFT                (0x0000000CU)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI3_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI4_MASK                 (0x000F0000U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI4_SHIFT                (0x00000010U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI4_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI5_MASK                 (0x00F00000U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI5_SHIFT                (0x00000014U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI5_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI6_MASK                 (0x0F000000U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI6_SHIFT                (0x00000018U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI6_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI7_MASK                 (0xF0000000U)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI7_SHIFT                (0x0000001CU)
#define CPSW_TX_G_OFLOW_THRESH_SET_REG_PRI7_MAX                  (0x0000000FU)

/* TX_G_OFLOW_THRESH_CLR_REG */

#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI0_MASK                 (0x0000000FU)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI0_SHIFT                (0x00000000U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI0_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI1_MASK                 (0x000000F0U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI1_SHIFT                (0x00000004U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI1_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI2_MASK                 (0x00000F00U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI2_SHIFT                (0x00000008U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI2_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI3_MASK                 (0x0000F000U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI3_SHIFT                (0x0000000CU)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI3_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI4_MASK                 (0x000F0000U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI4_SHIFT                (0x00000010U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI4_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI5_MASK                 (0x00F00000U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI5_SHIFT                (0x00000014U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI5_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI6_MASK                 (0x0F000000U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI6_SHIFT                (0x00000018U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI6_MAX                  (0x0000000FU)

#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI7_MASK                 (0xF0000000U)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI7_SHIFT                (0x0000001CU)
#define CPSW_TX_G_OFLOW_THRESH_CLR_REG_PRI7_MAX                  (0x0000000FU)

/* TX_G_BUF_THRESH_SET_L_REG */

#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI0_MASK                 (0x000000FFU)
#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI0_SHIFT                (0x00000000U)
#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI0_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI1_MASK                 (0x0000FF00U)
#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI1_SHIFT                (0x00000008U)
#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI1_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI2_MASK                 (0x00FF0000U)
#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI2_SHIFT                (0x00000010U)
#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI2_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI3_MASK                 (0xFF000000U)
#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI3_SHIFT                (0x00000018U)
#define CPSW_TX_G_BUF_THRESH_SET_L_REG_PRI3_MAX                  (0x000000FFU)

/* TX_G_BUF_THRESH_SET_H_REG */

#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI4_MASK                 (0x000000FFU)
#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI4_SHIFT                (0x00000000U)
#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI4_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI5_MASK                 (0x0000FF00U)
#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI5_SHIFT                (0x00000008U)
#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI5_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI6_MASK                 (0x00FF0000U)
#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI6_SHIFT                (0x00000010U)
#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI6_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI7_MASK                 (0xFF000000U)
#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI7_SHIFT                (0x00000018U)
#define CPSW_TX_G_BUF_THRESH_SET_H_REG_PRI7_MAX                  (0x000000FFU)

/* TX_G_BUF_THRESH_CLR_L_REG */

#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI0_MASK                 (0x000000FFU)
#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI0_SHIFT                (0x00000000U)
#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI0_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI1_MASK                 (0x0000FF00U)
#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI1_SHIFT                (0x00000008U)
#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI1_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI2_MASK                 (0x00FF0000U)
#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI2_SHIFT                (0x00000010U)
#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI2_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI3_MASK                 (0xFF000000U)
#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI3_SHIFT                (0x00000018U)
#define CPSW_TX_G_BUF_THRESH_CLR_L_REG_PRI3_MAX                  (0x000000FFU)

/* TX_G_BUF_THRESH_CLR_H_REG */

#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI4_MASK                 (0x000000FFU)
#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI4_SHIFT                (0x00000000U)
#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI4_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI5_MASK                 (0x0000FF00U)
#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI5_SHIFT                (0x00000008U)
#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI5_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI6_MASK                 (0x00FF0000U)
#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI6_SHIFT                (0x00000010U)
#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI6_MAX                  (0x000000FFU)

#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI7_MASK                 (0xFF000000U)
#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI7_SHIFT                (0x00000018U)
#define CPSW_TX_G_BUF_THRESH_CLR_H_REG_PRI7_MAX                  (0x000000FFU)

/* VLAN_LTYPE_REG */

#define CPSW_VLAN_LTYPE_REG_VLAN_LTYPE_INNER_MASK                (0x0000FFFFU)
#define CPSW_VLAN_LTYPE_REG_VLAN_LTYPE_INNER_SHIFT               (0x00000000U)
#define CPSW_VLAN_LTYPE_REG_VLAN_LTYPE_INNER_MAX                 (0x0000FFFFU)

#define CPSW_VLAN_LTYPE_REG_VLAN_LTYPE_OUTER_MASK                (0xFFFF0000U)
#define CPSW_VLAN_LTYPE_REG_VLAN_LTYPE_OUTER_SHIFT               (0x00000010U)
#define CPSW_VLAN_LTYPE_REG_VLAN_LTYPE_OUTER_MAX                 (0x0000FFFFU)

/* EST_TS_DOMAIN_REG */

#define CPSW_EST_TS_DOMAIN_REG_EST_TS_DOMAIN_MASK                (0x000000FFU)
#define CPSW_EST_TS_DOMAIN_REG_EST_TS_DOMAIN_SHIFT               (0x00000000U)
#define CPSW_EST_TS_DOMAIN_REG_EST_TS_DOMAIN_MAX                 (0x000000FFU)

/* TX_PRI0_MAXLEN_REG */

#define CPSW_TX_PRI0_MAXLEN_REG_TX_PRI0_MAXLEN_MASK              (0x00003FFFU)
#define CPSW_TX_PRI0_MAXLEN_REG_TX_PRI0_MAXLEN_SHIFT             (0x00000000U)
#define CPSW_TX_PRI0_MAXLEN_REG_TX_PRI0_MAXLEN_MAX               (0x00003FFFU)

/* TX_PRI1_MAXLEN_REG */

#define CPSW_TX_PRI1_MAXLEN_REG_TX_PRI1_MAXLEN_MASK              (0x00003FFFU)
#define CPSW_TX_PRI1_MAXLEN_REG_TX_PRI1_MAXLEN_SHIFT             (0x00000000U)
#define CPSW_TX_PRI1_MAXLEN_REG_TX_PRI1_MAXLEN_MAX               (0x00003FFFU)

/* TX_PRI2_MAXLEN_REG */

#define CPSW_TX_PRI2_MAXLEN_REG_TX_PRI2_MAXLEN_MASK              (0x00003FFFU)
#define CPSW_TX_PRI2_MAXLEN_REG_TX_PRI2_MAXLEN_SHIFT             (0x00000000U)
#define CPSW_TX_PRI2_MAXLEN_REG_TX_PRI2_MAXLEN_MAX               (0x00003FFFU)

/* TX_PRI3_MAXLEN_REG */

#define CPSW_TX_PRI3_MAXLEN_REG_TX_PRI3_MAXLEN_MASK              (0x00003FFFU)
#define CPSW_TX_PRI3_MAXLEN_REG_TX_PRI3_MAXLEN_SHIFT             (0x00000000U)
#define CPSW_TX_PRI3_MAXLEN_REG_TX_PRI3_MAXLEN_MAX               (0x00003FFFU)

/* TX_PRI4_MAXLEN_REG */

#define CPSW_TX_PRI4_MAXLEN_REG_TX_PRI4_MAXLEN_MASK              (0x00003FFFU)
#define CPSW_TX_PRI4_MAXLEN_REG_TX_PRI4_MAXLEN_SHIFT             (0x00000000U)
#define CPSW_TX_PRI4_MAXLEN_REG_TX_PRI4_MAXLEN_MAX               (0x00003FFFU)

/* TX_PRI5_MAXLEN_REG */

#define CPSW_TX_PRI5_MAXLEN_REG_TX_PRI5_MAXLEN_MASK              (0x00003FFFU)
#define CPSW_TX_PRI5_MAXLEN_REG_TX_PRI5_MAXLEN_SHIFT             (0x00000000U)
#define CPSW_TX_PRI5_MAXLEN_REG_TX_PRI5_MAXLEN_MAX               (0x00003FFFU)

/* TX_PRI6_MAXLEN_REG */

#define CPSW_TX_PRI6_MAXLEN_REG_TX_PRI6_MAXLEN_MASK              (0x00003FFFU)
#define CPSW_TX_PRI6_MAXLEN_REG_TX_PRI6_MAXLEN_SHIFT             (0x00000000U)
#define CPSW_TX_PRI6_MAXLEN_REG_TX_PRI6_MAXLEN_MAX               (0x00003FFFU)

/* TX_PRI7_MAXLEN_REG */

#define CPSW_TX_PRI7_MAXLEN_REG_TX_PRI7_MAXLEN_MASK              (0x00003FFFU)
#define CPSW_TX_PRI7_MAXLEN_REG_TX_PRI7_MAXLEN_SHIFT             (0x00000000U)
#define CPSW_TX_PRI7_MAXLEN_REG_TX_PRI7_MAXLEN_MAX               (0x00003FFFU)

/* CPSW_SS_IDVER_REG */

#define CPSW_SS_IDVER_REG_MINOR_VER_MASK                          (0x000000FFU)
#define CPSW_SS_IDVER_REG_MINOR_VER_SHIFT                         (0x00000000U)
#define CPSW_SS_IDVER_REG_MINOR_VER_MAX                           (0x000000FFU)

#define CPSW_SS_IDVER_REG_MAJOR_VER_MASK                          (0x00000700U)
#define CPSW_SS_IDVER_REG_MAJOR_VER_SHIFT                         (0x00000008U)
#define CPSW_SS_IDVER_REG_MAJOR_VER_MAX                           (0x00000007U)

#define CPSW_SS_IDVER_REG_RTL_VER_MASK                            (0x0000F800U)
#define CPSW_SS_IDVER_REG_RTL_VER_SHIFT                           (0x0000000BU)
#define CPSW_SS_IDVER_REG_RTL_VER_MAX                             (0x0000001FU)

#define CPSW_SS_IDVER_REG_IDENT_MASK                              (0xFFFF0000U)
#define CPSW_SS_IDVER_REG_IDENT_SHIFT                             (0x00000010U)
#define CPSW_SS_IDVER_REG_IDENT_MAX                               (0x0000FFFFU)

/* SYNCE_COUNT_REG */

#define SYNCE_COUNT_REG_SYNCE_CNT_MASK                            (0xFFFFFFFFU)
#define SYNCE_COUNT_REG_SYNCE_CNT_SHIFT                           (0x00000000U)
#define SYNCE_COUNT_REG_SYNCE_CNT_MAX                             (0xFFFFFFFFU)

/* SYNCE_MUX_REG */

#define SYNCE_MUX_REG_SYNCE_SEL_MASK                              (0x0000003FU)
#define SYNCE_MUX_REG_SYNCE_SEL_SHIFT                             (0x00000000U)
#define SYNCE_MUX_REG_SYNCE_SEL_MAX                               (0x0000003FU)

/* CONTROL_REG */

#define CONTROL_REG_EEE_EN_MASK                                   (0x00000001U)
#define CONTROL_REG_EEE_EN_SHIFT                                  (0x00000000U)
#define CONTROL_REG_EEE_EN_MAX                                    (0x00000001U)

#define CONTROL_REG_EEE_PHY_ONLY_MASK                             (0x00000002U)
#define CONTROL_REG_EEE_PHY_ONLY_SHIFT                            (0x00000001U)
#define CONTROL_REG_EEE_PHY_ONLY_MAX                              (0x00000001U)

/* SGMII_MODE_REG */

#define SGMII_MODE_REG_SYNCE_SEL_MASK                             (0x00000001U)
#define SGMII_MODE_REG_SYNCE_SEL_SHIFT                            (0x00000000U)
#define SGMII_MODE_REG_SYNCE_SEL_MAX                              (0x00000001U)

/* RGMII_STATUS_REG */

#define RGMII_STATUS_REG_LINK_MASK                                (0x00000001U)
#define RGMII_STATUS_REG_LINK_SHIFT                               (0x00000000U)
#define RGMII_STATUS_REG_LINK_MAX                                 (0x00000001U)

#define RGMII_STATUS_REG_SPEED_MASK                               (0x00000006U)
#define RGMII_STATUS_REG_SPEED_SHIFT                              (0x00000001U)
#define RGMII_STATUS_REG_SPEED_MAX                                (0x00000003U)

#define RGMII_STATUS_REG_FULLDUPLEX_MASK                          (0x00000008U)
#define RGMII_STATUS_REG_FULLDUPLEX_SHIFT                         (0x00000003U)
#define RGMII_STATUS_REG_FULLDUPLEX_MAX                           (0x00000001U)

/* SUBSSYSTEM_STATUS_REG */

#define SUBSSYSTEM_STATUS_REG_EEE_CLKSTOP_ACK_MASK                (0x00000001U)
#define SUBSSYSTEM_STATUS_REG_EEE_CLKSTOP_ACK_SHIFT               (0x00000000U)
#define SUBSSYSTEM_STATUS_REG_EEE_CLKSTOP_ACK_MAX                 (0x00000001U)


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/** \brief Enumerates interrupt configurations for channel. */
typedef enum 
{
    CPSW_CH_INTR_NONE = 0x00U,
    /**< Receive threshold interrupt. */
    CPSW_CH_INTR_RX_THR = 0x01U,
    /**< Receive threshold interrupt. */
    CPSW_CH_INTR_RX = 0x02U,
    /**< Receive interrupt. */
    CPSW_CH_INTR_TX = 0x03U,
    /**< Transmit interrupt. */
}Eth_CpswChIntr;

/** \brief Enumerates interrupt types for pacing.
 *            RX_PULSE and TX_PULSE interrupts can be paced.
 *         The RX_THRESH_PULSE and MISC_PULSE interrupts are not paced
 */
typedef enum
{
    CPSW_INTR_TYPE_RX,
    /**< Receive interrupt. */
    CPSW_INTR_TYPE_TX
    /**< Transmit interrupt. */
}Eth_CpswWrInt;

/** \brief Enumerates interrupt configurations for DMA. */
typedef enum 
{
    CPSW_STAT_INT = 0x00U,
    /**< Statistics interrupt. */
    CPSW_HOST_ERR_INT = 0x01U, /* (0x1U << 0x0U) */
    /**< Host error interrupt interrupt. */
}Eth_CpswDMAIntr;

/** \brief Enumerates interrupt configurations for channel. */
typedef enum
{
    CPSW_WR_INTR_LINE_NONE = 0x0U,
    /**< None. */
    CPSW_WR_INTR_LINE_RX_THR = 0x1U,
    /**< Receive threshold interrupt. */
    CPSW_WR_INTR_LINE_RX = 0x2U,
    /**< Receive interrupt. */
    CPSW_WR_INTR_LINE_TX = 0x3U,
    /**< Transmit interrupt. */
    CPSW_WR_INTR_LINE_MISC = 0x4U
    /**< Miscellaneous interrupt. */
}Eth_CpswWrIntrLine;

/* ========================================================================== */
/*                         Global Variables Declarations                      */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */
/**
 * \brief   This API sets the MacCtrl registers.
 *
 * \param   baseAddr    Base address of the CPSW.
  * \param   regVal     register value to be written.
 */
void CPSWSetMacCtrl(
   uint32 baseAddr, uint8 slavePortNum, uint32 regVal);
   
/**
 * \brief   This API sets the Rx Maximum length.
 *
 * \param   baseAddr    Base address of the CPSW.
  * \param   rxMaxLen   Rx Maximum length.
 */   
void CPSWSetRxMaxLen(
    uint32 baseAddr, uint8 slavePortNum,  uint32 rxMaxLen);

/**
 * \brief   This API resets the CPSW.
 *
 * \param   baseAddr    Base address of the CPSW.
 */
void CPSWEnetReset(uint32 baseAddr, uint8 slavePortNum);

/**
 * \brief   This API resets the CPDMA module.
 *
 * \param   baseAddr Base address of the CPDMA.
 */
void CPSWCpdmaReset(uint32 baseAddr);

/**
 * \brief   This API returns value of MAC control register of the sliver.
 *
 * \param   baseAddr     Base address of the CPSW.
 *
 * \retval  value of MAC control register.
 */

uint32 CPSWSlGetMacCtrl(uint32 baseAddr, uint8 slavePortNum);

/**
 * \brief   This API enables RX, TX or RX threshold interrupts of a channel. It
 *          configures given interrupt core interfaced to processor.
 *
 * \param   baseAddr     Base address of the CPSW.
 * \param   chNum        Channel number to be enabled.
 * \param   chIntr       Interrupts to be enabled on the given channel of core.
 *                       It takes mask of values #Eth_CpswChIntr.
 */
void CPSWChIntrEnable(uint32 baseAddr,
                        uint32 chNum,
                        Eth_CpswChIntr chIntr);

/**
 * \brief   This API sets the prescale value for WR.
 *
 * \details     WR prescale is used for interrupt pacing.
 *            Prescale value is number of MAIN_CLK R 0x0
 *            periods in 4us.
 *
 * \param   baseAddr     Base Address of the CPSW
 * \param   preScalarVal The prescale value.
 */
void CPSWSetPrescale(uint32 baseAddr,
                       uint32 preScalarVal);

/**
 * \brief   This API enables interrupt pacing for receive/transmit channel.
 *
 * \param   baseAddr     Base address of the CPSW.
 * \param   intrType      Interrupt type.It takes
 *                                   mask of values #Eth_CpswWrInt.
 * \param   paceVal       Maximum number of interrupts per millisecond generated
 *                         on RX_PULSE if pacing is enabled for this interrupt.
 * \Note    Enables interrupt pacing for core 0 only by default.
 */
void CPSWEnableIntrPacing(uint32 baseAddr,
                            Eth_CpswWrInt intrType,
                            uint32 paceVal);

/**
 * \brief   This API enables miscellaneous interrupts. It configures given
 *          interrupt core in wrapper module interfaced to processor.
 *
 * \param   baseAddr     Base address of the CPSW.
 * \param   miscIntrMask Interrupts to be enabled on the given core. It takes
 *                       mask of values #cpswMiscIntr_t.
 */
void CPSWMiscIntrEnable(uint32 baseAddr,
                          uint32 miscIntrMask);

/**
 * \brief   This API gives status given channel interrupt of all channels for
 *          given interrupt core in wrapper module interfaced to processor.
 *
 * \param   baseAddr       Base address of the CPSW.
 * \param   chIntr         Channel interrupt line takes values Eth_CpswChIntr.
 *
 * \retval  status         Given interrupt status of all channels.
 */
uint32 CPSWChIntrStatus(uint32 baseAddr,
                          Eth_CpswChIntr chIntr);

/**
 * \brief   This API returns value of completion pointer for CPDMA channel.
 *
 * \param   baseAddr       Base address of the CPSW.
 * \param   xferMode       Transfer mode takes values #ethernetXferMode_t.
 * \param   chNum          Channel number.
 * \param   regValPtr      Poiter to store value of register.
 *
 * \retval  None.
 */
void CPSWCpdmaReadRxCP(uint32  baseAddr, uint32  chNum, uint32 *regValPtr);

/**
 * \brief   This API returns value of completion pointer for Tx CPDMA channel.
 *
 * \param   baseAddr       Base address of the CPSW.
 * \param   chNum          Channel number.
 * \param   regValPtr      Poiter to store value of register.
 *
 * \retval  None
 */
void CPSWCpdmaReadTxCP(uint32  baseAddr, uint32  chNum, uint32 *regValPtr);

/**
 * \brief   This API gives status for miscellaneous interrupts of given
 *          interrupt core in wrapper module interfaced to processor.
 *
 * \param   baseAddr       Base address of the CPSW.
 *
 * \retval  miscIntrStatus Status of all miscellaneous interrupts.
 *
 * \note    To get status of specific miscellaneous use the values from
 *          #cpswWrMiscIntr_t.
 */
uint32 CPSWMiscIntrStatus(uint32 baseAddr);

/**
 * \brief   This API enables/disables the CPSW statistics for the given port
 *
 * \param   baseAddr    Base Address of the CPSW.
 * \param   portNum     The port number.
 * \param   enableStats Statistics configuration for a port takes
 *                      -# TRUE  - Enable statistics on the given port.
 *                      -# FALSE - Disable statistics on the given port.
 */
void CPSWStatsEnable(uint32 baseAddr,
                       uint32 portNum,
                       uint32 enableStats);

/**
 * \brief   This API configures the MAC address of the slave port.
 *
 * \param   baseAddr     Base address of the CPSW Port Module registers
 * \param   slavePortNum Slave port number.
 * \param   macAddr     The 6 byte MAC address.
 */
void           CPSWSetPortSrcAddr(uint32      baseAddr,
                                  uint8      slavePortNum,
                                  const uint8 macAddr[6]);

/**
 * \brief   This API returns the MAC address of the slave port.
 *
 * \param   baseAddr     Base address of the CPSW Port Module registers
 * \param   slavePortNum Slave port number.
 * \param   macAddr      Pointer to  6 byte MAC address.
 *
 * \return   E_OK: If address is valid
 *               E_NOT_OK: Address not valid
 */
Std_ReturnType CPSWGetPortSrcAddr(uint32 baseAddr,
                                  uint32 slavePortNum,
                                  uint8 macAddr[6]);

/**
 * \brief   This API enables the transfer interrupt for given channel.
 *
 * \param   baseAddr     Base address of the CPSW.
 * \param   chIntr       Interrupt configuration mask of channel takes values
 *                       #Eth_CpswDMAIntr.
 */
void CPSWCpdmaDmaIntrEnable(uint32 baseAddr,
                            Eth_CpswDMAIntr chIntr);

/**
 * \brief   This API enables the transfer interrupt for given channel.
 *
 * \param   baseAddr     Base address of the CPSW.
 * \param   chIntr       Interrupt configuration mask of DMA interrupt takes
 *                       values #Eth_CpswChIntr.
 *
 */
void CPSWCpdmaChIntrEnable(uint32 baseAddr,
                           uint32 chNum,
                           Eth_CpswChIntr chIntr);

/**
 * \brief   This API enables the transfer. Any write to hardware descriptor of
 *          a channel will start transfer.
 *
 * \param   baseAddr Base Address of the CPSW.
 * \param   xferMode Transfer mode takes values #ethernetXferMode_t.
 */
void CPSWCpdmaXferEnable(uint32 baseAddr, Eth_XferModeType xferMode);

/**
 * \brief   This API disables the transfer.
 *
 * \param   baseAddr Base Address of the CPSW.
 * \param   xferMode Transfer mode mask takes values #ethernetXferMode_t.
 */
void CPSWCpdmaXferDisable(uint32 baseAddr, Eth_XferModeType xferMode);

/**
 * \brief   This API writes the transfer HDP register. If transfer is enabled,
 *          write to the transfer HDP will immediately start transfer.
 *          The data will be taken from the buffer pointer of the transfer
 *          buffer descriptor written to the transfer HDP.
 *
 * \param   baseAddr Base Address of the CPSW.
 * \param   xferMode Transfer type.
 * \param   descHdr  Address of the transfer buffer descriptor.
 * \param   chNum    Channel Number.
 */
void CPSWCpdmaXferHdrDescPtrWrite(uint32                 baseAddr,
                                  Eth_XferModeType  xferMode,
                                  const Eth_CpdmaBuffDescType *descHdr,
                                  uint32                 chNum);

/**
 * \brief   This API writes the the transfer Completion Pointer for a specific
 *          channel.
 *
 * \param   baseAddr Base Address of the CPSW.
 * \param   xferMode Transfer type.
 * \param   chNum    Channel Number.
 * \param   comPtr   Completion Pointer Value to be written.
 */
void CPSWCpdmaXferCpWrite(uint32 baseAddr,
                          Eth_XferModeType xferMode,
                          uint32 chNum,
                          uint32 comPtr);

/**
 * \brief   This API writes the interrupt line number to End Of Interrupt
 *          Vector.
 *
 * \param   baseAddr      Base Address of the CPDMA module registers.
 * \param   intrLine      Type of interrupt to acknowledge to the CPDMA from
 *                        interrupt lines #Eth_CpswWrIntrLine.
 */
void CPSWCpdmaWriteEoiVector(uint32 baseAddr, Eth_CpswWrIntrLine intrLine);

/**
 * \brief   This API sets the free buffers for a specific channel.
 *
 * \param   baseAddr   Base Address of the CPSW.
 * \param   chNum      Channel Number.
 * \param   numFreeBuf Number of free buffers.
 */
void CPSWSetCpdmaNumFreeBuf(uint32 baseAddr,
                            uint32 chNum,
                            uint32 numFreeBuf);

/**
 * \brief   This API sets threshold value for issuing receive threshold
 *           pending interrupts.
 *
 * \param   baseAddr   Base Address of the CPSW.
 * \param   chNum      Channel Number.
 * \param   rxTreshVal Rx Flow Threshold values.
 */
void CPSWSetCpdmaRxThreshold(uint32 baseAddr,
                             uint32 chNum,
                             uint32 rxTreshVal);

/**
 * \brief   Returns the CPDMA Status.
 *
 * \param   baseAddr    Base Address of the CPSW.
 *
 * \retval  TRUE= Host error detected.
 *             FALSE = Host error not detected.
 */
uint32 CPSWCheckHostErr(uint32 baseAddr);

/**
 * \brief   This API will teardown CPDMA channel. CPDMA teardown is used for
 *          gracefully shutting down CPDMA channel transfers. When a teardown
 *          command is issued to an enabled receive channel the following will
 *          occur:
 *              1. Any current frame in reception will complete normally.
 *              2. The teardown_complete bit will be set in the next buffer
 *                 descriptor in the chain (if there is one).
 *              3. The channel head descriptor pointer will be cleared to 0.
 *              4. A receive interrupt for the channel will be issued to the
 *                 host.
 *
 * \param   baseAddr      Base Address of the CPSW.
 * \param   xferMode      Type of channel
 *                        Transfer mode takes values #ethernetXferMode_t
 * \param   chNum         Channel number to be torn.
 */
void CPSWCpdmaChTearDown(uint32 baseAddr, Eth_XferModeType xferMode, uint32 chNum);

/**
 * \brief   This API reads a PHY register using MDIO.
 *
 * \param   baseAddr Base Address of the CPSW.
 * \param   phyAddr  PHY Address.
 * \param   regNum   Register Number to be read.
 * \param   pData    Pointer where the read value shall be written.
 *
 * \retval  TRUE     Read is successful.
 * \retval  FALSE    Read is not acknowledged properly.
 */
void CPSWMdioPhyRegRead(uint32  baseAddr,
                        uint8   phyAddr,
                        uint8  regNum,
                        uint16 *pData);

/**
 * \brief   This API writes a PHY register using MDIO.
 *
 * \param   baseAddr      Base Address of the CPSW.
 * \param   phyAddr       PHY Address.
 * \param   regNum        Register Number to be written.
 * \param   wrVal         Value to be written.
 */
void CPSWMdioPhyRegWrite(uint32 baseAddr,
                         uint8  phyAddr,
                         uint8 regNum,
                         uint16 wrVal);

/**
 * \brief   This API reads the link status of all PHY connected to this MDIO.
 *          The bit corresponding to the PHY address will be set if the PHY
 *          link is active.
 *
 * \param   baseAddr Base Address of the CPSW.
 * \param   phyAddr  PHY Address.
 *
 * \retval  TRUE   PHY link is active.
 * \retval  FALSE  PHY link is inactive.
 */
uint32 CPSWMdioPhyLinkStatus(uint32 baseAddr, uint8 phyAddr);

/**
 * \brief   This API initializes the MDIO peripheral. This enables the MDIO
 *          state machine, uses standard pre-amble and set the clock divider
 *          value.
 *
 * \param   baseAddr       Base Address of the CPSW.
 * \param   mdioInputFreq  The clock input to the MDIO module.
 * \param   mdioOutputFreq The clock output required on the MDIO bus.
 */
void CPSWMdioInit(uint32 baseAddr,
                  uint32 mdioInputFreq,
                  uint32 mdioOutputFreq);

/**
 * \brief   This API returns value of MDIO register
 *
 * \param   baseAddr       Base Address of the CPSW.
 */
uint32 CPSWMdioRegRead(uint32 baseAddr,
                       uint32 regAddr);

/**
 * \brief   This API enables MDIO user interrupt
 *
 * \param   baseAddr       Base Address of the CPSW.
 */
void CPSWMDIOUsrIntrEnable(uint32 baseAddr);

/**
 * \brief   This API disables miscellaneous interrupts. It configures given
 *          interrupt core in wrapper module interfaced to processor.
 *
 * \param   baseAddr     Base address of the CPSW.
 * \param   miscIntrMask Interrupts to be disabled on the given core. It takes
 *                       mask of values #cpswWrMiscIntr_t.
 */
void CPSWMiscIntrDisable(uint32 baseAddr,
                           uint32 miscIntrMask);
/**
 * \brief   This API enables the transfer interrupt for given channel.
 *
 * \param   baseAddr     Base address of the CPSW.
 * \param   chIntrMask   Interrupt configuration mask of DMA interrupt takes
 *                       values #cpswDMAIntr.
 */
void CPSWCpdmaDmaIntrDisable(uint32 baseAddr,
                             Eth_CpswDMAIntr chIntrMask);


/**
 * \brief   This API disables the transfer interrupt for given channel.
 *
 * \param   baseAddr     Base address of the CPSW.
 * \param   chNum        Channel number for which interrupt to be disabled
 * \param   chIntrMask   Interrupt configuration mask of channel takes values
 *                       #Eth_CpswChIntr.
 */
void CPSWCpdmaChIntrDisable(uint32 baseAddr,
                            uint32 chNum,
                            Eth_CpswChIntr chIntrMask);



#ifdef __cplusplus
}
#endif

#endif /* #ifndef CPSW_H_ */
