/* ======================================================================
 *   Copyright (c) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file       cpsw_ale.h
 *
 *  \brief      This file contains the function prototypes for the device
 *              contains some abstraction layer for ALE module of CPSW
 *              subsystem.
 */

#ifndef CPSW_ALE_H_
#define CPSW_ALE_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#ifdef __cplusplus
extern "C" {
#endif
/* ========================================================================== */
/*                                 Macros                                     */
/* ========================================================================== */

/** \brief ALE entry length in words. */
#define CPSW_ALE_ENTRY_MAX                                 (512U)

/** \brief ALE entry length in words. */
#define CPSW_ALE_ENTRY_NUM_WORDS                           (3U)

/** \brief Broadcast MAC address byte mask. */
#define CPSW_ALE_BROADCAST_MAC_ADDR_MASK                   (0xFFU)

/** \brief Multicast MAC address upper byte mask. */
#define CPSW_ALE_MULTICAST_MAC_ADDR_MASK                   (0x01U)

/** \brief Enumerates types of Unicast ALE table entry. */
/**< Unicast address that is not ageable. */
#define  CPSW_ALE_TBL_UNICAST_TYPE_NOT_AGEABLE             (0x0U)

/**< Ageable unicast address that has not been touched. */
#define CPSW_ALE_TBL_UNICAST_TYPE_AGEABLE_UNTOUCH          (0x1U)

/**< Non ageable OUI address. */
#define CPSW_ALE_TBL_UNICAST_TYPE_OUI                      (0x2U)

/**< Ageable unicast address that has been touched. */
#define CPSW_ALE_TBL_UNICAST_TYPE_AGEABLE_TOUCH                (0x3U)

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/** \brief Enumerates types of ALE table entry. */
typedef enum
{
    CPSW_ALE_TBL_ENTRY_TYPE_FREE = 0x0U,
    /**< Free entry. */
    CPSW_ALE_TBL_ENTRY_TYPE_ADDR = 0x1U,
    /**< Address type ALE entry. */
    CPSW_ALE_TBL_ENTRY_TYPE_VLAN = 0x2U,
    /**< VLAN type ALE entry. */
    CPSW_ALE_TBL_ENTRY_TYPE_VLAN_ADDR = 0x3U,
    /**< VLAN Address type ALE entry. */
}Eth_CpswAleTblEntryType;

/** \brief Structure holding the Ale table entry context .*/
typedef struct
{
    uint32 word0;
    /**< Pre-scalar register value.*/
    uint32 word1;
    /**< SCL low time register value.*/
    uint32 word2;
    /**< SCL high time register value.*/
}Eth_CpswAleTblEntry;


/** \brief Enumerates port state configuration values in address lookup engine.
**/
typedef enum
{
    CPSW_ALE_PORT_STATE_DISABLED = 0x00U,
    /**< Port is disabled in address lookup engine. */
    CPSW_ALE_PORT_STATE_BLOCKED = 0x01U,
    /**< Port is blocked in address lookup engine. */
    CPSW_ALE_PORT_STATE_LEARN = 0x02U,
    /**< Port is configured for learning in address lookup engine. */
    CPSW_ALE_PORT_STATE_FWD = 0x03U
                              /**< Port is configured to forward in address
                               *lookup engine. */
}Eth_CpswAlePortState;

/** \brief Structure holding the Ale Vlan config context .*/
typedef struct
{
    uint32 enableUnregMcastFlood;
    /**< Mask used for multicast when the multicast address is not found.*/
    uint32 enableRegMcastFlood;
    /**< Mask used for multicast when the multicast address is found.*/
    uint32 enableForceUntaggedEgress;
    /**< This field causes the packet VLAN tag to be removed on egress (except
     *on port 0).*/
}Eth_CpswAleVlanConfigParams;

/** \brief Structure holding the Ale Multicast config context .*/
typedef struct
{
    uint32 fwdState;
    /**<
     *  Indicates the port state(s) required for the received port on a
     *  destination address lookup in order for the multicast packet to be
     *  forwarded to the transmit port(s). A transmit port must be in the
     *  Forwarding state in order to forward the packet. If the transmit
     *  port_mask has multiple set bits then each forward decision is
     *  independent of the other transmit port(s) forward decision.
     */
    uint32 enableSuper;
    /**<
     *  This field indicates that the packet with a matching multicast
     *  destination address is a supervisory packet.
     */
}Eth_CpswAleMcastConfigParams;

/** \brief Structure holding the Ale Unicast config context .*/
typedef struct
{
    uint32 unicastType;
    /**< This field indicates the type of unicast address the table entry
     *contains. */
    uint32 enableSecure;
    /**<
     *  This bit indicates that a packet with a matching source address should
     *  be dropped if the received port number is not equal to the table entry
     *  port_number.
     */
    uint32 enableBlock;
    /**<
     *  The block bit indicates that a packet with a matching source or
     *  destination address should be dropped (block the address).
     */
    uint32 enableDlr;
    /**<
     *  This indicates that the address is a Device Level Ring (DLR) unicast
     *  address. Received packets with a matching destination address will be
     *  flooded to the vlan_member_list (minus the receive port and the host
     *  port).
     */
}Eth_CpswAleUcastConfigParams;

/* ========================================================================== */
/*                         Global Variables Declarations                      */
/* ========================================================================== */

/* None*/

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/**
 * \brief   This API initializes ALE. It clears the ALE table and enables packet
 *          processing by ALE.
 *
 * \param   baseAddr    Base address of the CPSW.
 */
void CPSWAleInit(uint32 baseAddr);

/**
 * \brief   This API checks the MAC address is of the specified type.
 *
 * \param   macAddr   Ethernet MAC address.
 * \param   addrType   Type of address takes values cpswPktType_t.
 *
 * \retval  TRUE       Ethernet MAC address is of the specified type.
 * \retval  FALSE      Ethernet MAC address is not of the specified type.
 */
uint32 CPSWAleValidateMacAddr(const uint8 macAddr[ETH_MAC_ADDR_LEN],
                                     Eth_MACAddrType      addrType);
/**
 * \brief   This API removes any ageable ALE table entries that does not have a
 *          touch bit set.
 *
 * \param   baseAddr    Base address of the CPSW.
 */
void CPSWAleAgeOut(uint32 baseAddr);

/**
 * \brief   This API configures the port state for a given port in the ALE.
 *
 * \param   baseAddr    Base address of the CPSW.
 * \param   portNum     The port number
 * \param   portState   The port state takes values #Eth_CpswAlePortState.
 */
void CPSWAleSetPortState(uint32 baseAddr,
                         uint32 portNum,
                         uint32 portState);

/**
 * \brief   This API configures VLAN Aware in ALE module to determine what
 *          to be done if VLAN is not found.
 *
 * \param   baseAddr        Base Address of the CPSW.
 * \param   enableVlanAware ALE VLAN Aware configuration for packet.
 *                          -# TRUE  - Drop packet if VLAN not found
 *                          -# FALSE - Flood if VLAN not found
 */
void CPSWAleVlanAwareEnable(uint32 baseAddr, uint32 enableVlanAware);

/*!
 * \brief   Sets the Unknown VLAN Registered Multicast Flood Mask
 *
 * \param   baseAddr      Base Address of the ALE module
 * \param   rfmVal        Unknown VLAN Registered Multicast Flood Mask Value
 *
 * \return  None
 *
 **/
void CPSWAleUnknownRegFloodMaskSet(uint32 baseAddr, uint32 rfmVal);


/*!
 * \brief   Sets the Unknown VLAN Member List
 *
 * \param   baseAddr      Base Address of the ALE module
 * \param   mlVal        Unknown VLAN UnRegistered Multicast Flood Mask Value
 *
 * \return  None
 *
 **/
void CPSWAleUnknownMemberListSet(uint32 baseAddr, uint32 mlVal);


/**
 * \brief   This API enables/disables the bypassing of the ALE logic for all
 *          packets received on slave ports to the host port.
 *
 * \param   baseAddr     Base Address of the CPSW.
 * \param   enableBypass Bypass ALE logic configuration takes values
 *                       -# TRUE  - Enable bypass mode
 *                       -# FALSE - Disable bypass mode
 */
void CPSWAleBypassEnable(uint32 baseAddr, uint32 enableBypass);

/**
 * \brief Adds an unicast entry in the ALE table.
 *
 * \param baseAddr      Base address of the CPSW.
 * \param pMacAddr      Ethernet address
 * \param vlanId        VLAN ID
 * \param portNum       The port number
 * \param unicastParams VLAN config parameters #Eth_CpswAleUcastConfigParams
 *
 * \retval index of the ALE entry added
 * \retval CPSW_ALE_ENTRY_MAX if table entry is not free
 */
uint32 CPSWAleUnicastEntryAdd(
    uint32                            baseAddr,
    const uint8                       macAddr[6],
    uint32                            vlanId,
    uint32                            portNum,
    const Eth_CpswAleUcastConfigParams *unicastParams);


/**
 * \brief Deletes an unicast entry in the ALE table.
 *
 * \param baseAddr   Base address of the CPSW.
 * \param pMacAddr   Ethernet address
 * \param vlanId     VLAN ID
 * \param portNum    The port number
 *
 * \retval index of the ALE entry deleted
 * \retval CPSW_ALE_ENTRY_MAX if table entry is not present
 */
uint32 CPSWAleUnicastEntryDel(uint32      baseAddr,
                              const uint8 macAddr[6],
                              uint32      vlanId,
                              uint32      portNum);

/**
 * \brief Adds a multicast entry in the ALE table
 *
 * \param baseAddr        Base address of the CPSW.
 * \param pMacAddr        Ethernet Address
 * \param vlanId          VLAN ID
 * \param portNum         The port number
 * \param multicastParams VLAN config parameters #Eth_CpswAleVlanConfigParams
 *
 * \retval index of the ALE entry added
 * \retval CPSW_ALE_ENTRY_MAX if table entry is not free
 */
uint32 CPSWAleMulticastEntryAdd(
    uint32                            baseAddr,
    const uint8                       macAddr[6],
    uint32                            vlanId,
    uint32                            portMask,
    const Eth_CpswAleMcastConfigParams *multicastParams);

/**
 * \brief Deletes a multicast entry in the ALE table
 *
 * \param baseAddr   Base address of the CPSW.
 * \param pMacAddr   Ethernet Address
 * \param vlanId     VLAN ID
 * \param portNum    The port number
 *
 * \retval index of the ALE entry deleted
 * \retval CPSW_ALE_ENTRY_MAX if table entry is not present
 */
uint32 CPSWAleMulticastEntryPortDel(uint32      baseAddr,
                                    const uint8 macAddr[6],
                                    uint32      vlanId,
                                    uint8      portNum);

/**
 * \brief   This API clears the entries of address lookup table.
 *
 * \param   baseAddr    Base address of the CPSW.
 */
void CPSWAleClearTable(uint32 baseAddr);

/* ========================================================================== */
/*                        Deprecated Function Declarations                    */
/* ========================================================================== */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef CPSW_ALE_H_ */
