/* ======================================================================
 *   Copyright (c) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     cpsw_mdio.h
 *
 *  \brief    This file contains the dal API prototypes and macro definitions
 *            which interact with the MDIO hardware registers. This file
 *            provides APIs to initialise configure and use MDIO.
 */

#ifndef MDIO_H_
#define MDIO_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "hw_cpsw_mdio.h"

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                           Macros                                           */
/* ========================================================================== */

/*None*/

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/*None*/

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/**
 * \brief   This API reads a PHY register using MDIO.
 *
 * \param   baseAddr Base Address of the MDIO module.
 * \param   phyAddr  PHY Address.
 * \param   regNum   Register Number to be read.
 * \param   pData    Pointer where the read value shall be written.
 *
 * \retval  TRUE     Read is successful.
 * \retval  FALSE    Read is not acknowledged properly.
 */
void MDIOPhyRegRead(uint32  baseAddr,
                    uint8   phyAddr,
                    uint8  regNum,
                    uint16 *pData);

/**
 * \brief   This API writes a PHY register using MDIO.
 *
 * \param   baseAddr      Base Address of the MDIO module.
 * \param   phyAddr       PHY Address.
 * \param   regNum        Register Number to be written.
 * \param   wrVal         Value to be written.
 */
void MDIOPhyRegWrite(uint32 baseAddr,
                     uint8  phyAddr,
                     uint8 regNum,
                     uint16 wrVal);


/**
 * \brief   This API initializes the MDIO peripheral. This enables the MDIO
 *          state machine, uses standard pre-amble and set the clock divider
 *          value.
 *
 * \param   baseAddr       Base Address of the MDIO module.
 * \param   mdioInputFreq  The clock input to the MDIO module.
 * \param   mdioOutputFreq The clock output required on the MDIO bus.
 */

void MDIOInit(uint32 baseAddr,
              uint32 mdioInputFreq,
              uint32 mdioOutputFreq);

/**
 * \brief   This API returns value of MDIO register.
 *
 * \param   baseAddr       Base Address of the MDIO module.
 */
uint32 MDIORegRead(uint32 baseAddr,
                   uint32 regAddr);

/**
 * \brief   This API will enable MDIO user command interrupt.
 *
 * \param   baseAddr       Base Address of the MDIO module.
 */
void MDIOUsrIntrEnable(uint32 baseAddr);


/**
 * \brief   This API initializes clears MDIO user command interrupt.
 *
 * \param   baseAddr       Base Address of the MDIO module.
 */
void MDIOUsrIntrClr(uint32 baseAddr);

/* ========================================================================== */
/*                        Deprecated Function Declarations                    */
/* ========================================================================== */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef MDIO_H_ */
