/* ======================================================================
 *   Copyright (c) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     cpsw_priv.h
 *
 *  \brief    This file contains the generic ethernet definitions
 *
 */
#ifndef CPSW_PRIV_H_
#define CPSW_PRIV_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */


#include "Std_Types.h"

#include "Eth_Types.h"
#include "cpsw_cpts.h"


#ifdef __cplusplus
extern "C" {
#endif
/* ========================================================================== */
/*                                 Macros                                     */
/* ========================================================================== */

/*---------------------------------------------------------------------------*\
 |                           Macros/Defines for MAC switch                     |
 \*---------------------------------------------------------------------------*/

/** MAC module return codes */
#define MAC_SUCCESS                 (0x00000000U)
#define MAC_FAILURE                 (0xFFFFFFFFU)
#define MAC_UNKNOWN_IOCTL           (0x00000001U)
#define MAC_MALFORMED_IOCTL         (0x00000002U)

#define MSS_CPSW_CONTROL_REG_P1_MODE_SEL_SHIFT              (0x0U)
#define MSS_CPSW_CONTROL_REG_P1_MODE_SEL_MASK               (0x00000007U)

#define MSS_CPSW_CONTROL_REG_P2_MODE_SEL_SHIFT              (0x10U)
#define MSS_CPSW_CONTROL_REG_P2_MODE_SEL_MASK               (0x00070000U)

#define MAC_CONFIG_MODEFLG_CHPRIORITY           0x00001U
/**< Use Tx channel priority                                                */
#define MAC_CONFIG_MODEFLG_MACLOOPBACK          0x00002U
/**< MAC internal loopback                                                  */
#define MAC_CONFIG_MODEFLG_RXCRC                0x00004U
/**< Include CRC in RX frames                                               */
#define MAC_CONFIG_MODEFLG_TXCRC                0x00008U
/**< Tx frames include CRC                                                  */
#define MAC_CONFIG_MODEFLG_PASSERROR            0x00010U
/**< Pass error frames                                                      */
#define MAC_CONFIG_MODEFLG_PASSCONTROL          0x00020U
/**< Pass control frames                                                    */
#define MAC_CONFIG_MODEFLG_PASSALL              0x00040U
/**< pass all frames                                                        */
#define MAC_CONFIG_MODEFLG_RXQOS                0x00080U
/**< Enable QOS at receive side                                             */
#define MAC_CONFIG_MODEFLG_RXNOCHAIN            0x00100U
/**< Select no buffer chaining                                              */
#define MAC_CONFIG_MODEFLG_RXOFFLENBLOCK        0x00200U
/**< Enable offset/length blocking                                          */
#define MAC_CONFIG_MODEFLG_RXOWNERSHIP          0x00400U
/**< Use ownership bit as 1                                                 */
#define MAC_CONFIG_MODEFLG_RXFIFOFLOWCNTL       0x00800U
/**< Enable rx fifo flow control                                            */
#define MAC_CONFIG_MODEFLG_CMDIDLE              0x01000U
/**< Enable IDLE command                                                    */
#define MAC_CONFIG_MODEFLG_TXSHORTGAPEN         0x02000U
/**< Enable tx short gap                                                    */
#define MAC_CONFIG_MODEFLG_TXPACE               0x04000U
/**< Enable tx pacing                                                       */
#define MAC_CONFIG_MODEFLG_TXFLOWCNTL           0x08000U
/**< Enable tx flow control                                                 */
#define MAC_CONFIG_MODEFLG_RXBUFFERFLOWCNTL     0x10000U
/**< Enable rx buffer flow control                                          */

/*---------------------------------------------------------------------------*\
 |                           Macros/Defines for CPDMA                          |
 \*---------------------------------------------------------------------------*/
#define CPSW_CPDMA_WRD0_NDP_MASK                           (0xFFFFFFFFU)
#define CPSW_CPDMA_WRD0_NDP_SHIFT                          (0U)
#define CPSW_CPDMA_WRD1_BUFF_PTR_MASK                      (0xFFFFFFFFU)
#define CPSW_CPDMA_WRD1_BUFF_PTR_SHIFT                     (0U)
#define CPSW_CPDMA_WRD3_SOP_MASK                           (0x80000000U)
#define CPSW_CPDMA_WRD3_SOP_SHIFT                          (31U)
#define CPSW_CPDMA_WRD3_SOP_ENABLE                         (1U)
#define CPSW_CPDMA_WRD3_SOP_DISABLE                        (0U)
#define CPSW_CPDMA_WRD3_EOP_MASK                           (0x40000000U)
#define CPSW_CPDMA_WRD3_EOP_SHIFT                          (30U)
#define CPSW_CPDMA_WRD3_EOP_ENABLE                         (1U)
#define CPSW_CPDMA_WRD3_EOP_DISABLE                        (0U)
#define CPSW_CPDMA_WRD3_OWN_MASK                           (0x20000000U)
#define CPSW_CPDMA_WRD3_OWN_SHIFT                          (29U)
#define CPSW_CPDMA_WRD3_OWN_ENABLE                         (1U)
#define CPSW_CPDMA_WRD3_OWN_DISABLE                        (0U)
#define CPSW_CPDMA_WRD3_EOQ_MASK                           (0x10000000U)
#define CPSW_CPDMA_WRD3_EOQ_SHIFT                          (28U)
#define CPSW_CPDMA_WRD3_EOQ_ENABLE                         (1U)
#define CPSW_CPDMA_WRD3_EOQ_DISABLE                        (0U)
#define CPSW_CPDMA_WRD3_TEAR_DWN_COMP_MASK                 (0x08000000U)
#define CPSW_CPDMA_WRD3_TEAR_DWN_COMP_SHIFT                (27U)
#define CPSW_CPDMA_WRD3_TEAR_DWN_COMP_COMPLETE       (1U)
#define CPSW_CPDMA_WRD3_PASS_CRC_MASK                      (0x04000000U)
#define CPSW_CPDMA_WRD3_PASS_CRC_SHIFT                     (26U)
#define CPSW_CPDMA_WRD3_TO_PORT_EN_MASK                    (0x00100000U)
#define CPSW_CPDMA_WRD3_TO_PORT_EN_SHIFT                   (20U)
#define CPSW_CPDMA_WRD3_TO_PORT_MASK                       (0x000F0000U)
#define CPSW_CPDMA_WRD3_TO_PORT_SHIFT                      (16U)
#define CPSW_CPDMA_WRD3_PKT_LEN_MASK                       (0x000007FFU)
#define CPSW_CPDMA_WRD3_PKT_LEN_SHIFT                      (0U)

/** \brief Multicast MAC address upper byte mask. */
#define CPSW_CPDMA_RX_WRD2_BUFF_LEN_MASK                   (0x000007FFU)
#define CPSW_CPDMA_RX_WRD2_BUFF_LEN_SHIFT                  (0U)
#define CPSW_CPDMA_RX_WRD2_BUFF_OFF_MASK                   (0x07FF0000U)
#define CPSW_CPDMA_RX_WRD2_BUFF_OFF_SHIFT                  (16U)
#define CPSW_CPDMA_RX_WRD3_FROM_PORT_MASK                  (0x00070000U)
#define CPSW_CPDMA_RX_WRD3_FROM_PORT_SHIFT                 (16U)
#define CPSW_CPDMA_RX_WRD3_VLAN_ENCAP_MASK                 (0x00080000U)
#define CPSW_CPDMA_RX_WRD3_VLAN_ENCAP_SHIFT                (19U)
#define CPSW_CPDMA_RX_WRD3_PKT_ERR_MASK                    (0x00300000U)
#define CPSW_CPDMA_RX_WRD3_PKT_ERR_SHIFT                   (20U)
#define CPSW_CPDMA_RX_WRD3_OVERRUN_MASK                    (0x00400000U)
#define CPSW_CPDMA_RX_WRD3_OVERRUN_SHIFT                   (22U)
#define CPSW_CPDMA_RX_WRD3_MAC_CTRL_MASK                   (0x00800000U)
#define CPSW_CPDMA_RX_WRD3_MAC_CTRL_SHIFT                  (23U)
#define CPSW_CPDMA_RX_WRD3_SHORT_MASK                      (0x01000000U)
#define CPSW_CPDMA_RX_WRD3_SHORT_SHIFT                     (24U)
#define CPSW_CPDMA_RX_WRD3_LONG_MASK                       (0x02000000U)
#define CPSW_CPDMA_RX_WRD3_LONG_SHIFT                      (25U)

/** \brief Multicast MAC address upper byte mask. */
#define CPSW_CPDMA_TX_WRD2_BUFF_LEN_MASK                   (0x0000FFFFU)
#define CPSW_CPDMA_TX_WRD2_BUFF_LEN_SHIFT                  (0U)
#define CPSW_CPDMA_TX_WRD2_BUFF_OFF_MASK                   (0xFFFF0000U)
#define CPSW_CPDMA_TX_WRD2_BUFF_OFF_SHIFT                  (16U)
#define CPSW_CPDMA_TX_WRD3_TO_PORT_MASK                    (0x00030000U)
#define CPSW_CPDMA_TX_WRD3_TO_PORT_SHIFT                   (16U)
#define CPSW_CPDMA_TX_WRD3_TO_PORT_EN_MASK                 (0x00100000U)
#define CPSW_CPDMA_TX_WRD3_TO_PORT_EN_SHIFT                (20U)
#define CPSW_CPDMA_TX_WRD3_TO_PORT_EN_ENABLE               (1U)
#define CPSW_CPDMA_TX_WRD3_TO_PORT_EN_DISABLE              (0U)

#define CPSW_CPPI_RAM_SIZE                                  (0x2000U)
#define CPSW_CPDMA_TEAR_DWN_ACK                             (0xFFFFFFFCU)
#define CPSW_CPDMA_MAX_CHANNELS                             (8U)
#define CPSW_IF_CLK_FREQ                                    (125U)
#define CPSW_CPDMA_BUS_WIDTH                                (32U)

#define CPDMA_DESC_MEM_SIZE ((ETH_NUM_TX_BUFFERS + ETH_NUM_RX_BUFFERS) * 20U)

#define htons(a) ( (((a)>>8)&0xffU) + (((a)<<8)&0xff00U) )
#define htonl(a) ((((a) & 0xff000000U) >> 24) | (((a) & 0x00ff0000U) >> 8) | \
                  (((a) & 0x0000ff00U) << 8)  | (((a) & 0x000000ffU) << 24) )

#define ntohl(a) (htonl(a))
#define ntohs(a) (htons(a))

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/** \brief Enumerates mode of transfer operation. */
typedef enum ethXferMode
{
    ETH_XFER_MODE_TX = 0x00U,
    /**< Transfer mode is transmitter. */
    ETH_XFER_MODE_RX = 0x01U
                       /**< Transfer mode is receiver. */
}Eth_XferModeType;

/** \brief Enumerates type of MAC address. */
typedef enum 
{
    ETH_MACADDR_TYPE_UNICAST = 0x00U,
    /**< Unicast address. */
    ETH_MACADDR_TYPE_MULTICAST = 0x01U,
    /**< Multicast address. */
    ETH_MACADDR_TYPE_BROADCAST = 0x02U
    /**< Broadcast address. */
}Eth_MACAddrType;


/** \brief Structure holding Eth frame header format */
typedef struct
{
    uint8  dstMacAddr[ETH_MAC_ADDR_LEN];
    /**< Destination MAC address */
    uint8  srcMacAddr[ETH_MAC_ADDR_LEN];
    /**< Source MAC address */
    uint16 h_proto;
    /**< The frame type */
}Eth_FrameHeaderType;

/** \brief Structure holding Eth frame format (header and data)*/
typedef struct
{
    /**<Header */
    Eth_FrameHeaderType        header;
    /**< The generic data payload */
    uint8                payload[ETH_DATA_LEN];
} Eth_FrameObjType;

/** \brief Enumeration for buffer states */
typedef enum
{
    ETH_BUF_STATE_FREE = 0U,
    /**< Buffer is free for allocation */
    ETH_BUF_STATE_IN_USE,
    /**< Buffer is in use */
    ETH_BUF_STATE_QUED
    /**< Buffer qued for transmit */
} Eth_Buf_StateType;

/**
 *  \brief Implementation specific structures for port configuration.
 */
/* Requirements : SWS_Eth_00004 */
typedef struct
{
    uint16            bufIdx;
    /**< Buffer index */
    void             *payload;
    /**< pointer to the actual data in the buffer */
    uint16            len;
    /**< length of this buffer */
    Eth_FrameType     type;
    /**< Eth FrameType */
    Eth_Buf_StateType bufState;
    /**< Buffer use status */
    boolean           txConfirmation;
    /**< Callback enable/disable */
#if (ETH_GLOBALTIMESUPPORT_API == STD_ON)
    boolean           enableEgressTimeStamp;
    /**< Flag to enable/disable Timestamping on Tx packet */
    Eth_CptsEvent     bufCptsEventInfo;
    /**< CPTS ETH TX packet event info containing messageType & sequenceId
     *   Used when event lookup is done in subsequent Eth_GetEgressTimeStamp
     *   call after Eth_EnableEgressTimeStamp */
#endif

} Eth_BufObjType;

/** \brief Buffer descriptor data structure. */
typedef struct Eth_CpdmaBuffDesc
{
    struct Eth_CpdmaBuffDesc *pNextBuffDesc;
    /**< Next buffer descriptor. */
    void                *pDataBuffer;
    /**< Associated packet buffer. */
    uint32               bufferOffsetAndLength;
    /**< Buffer length and offset. */
    uint32               flagsAndPacketLength;
    /**< Packet length and configuration flags. */
	Eth_BufObjType      *pBufObj;
    /**< Associated buffer descriptor(mostly used for Transmit logging */
} Eth_CpdmaBuffDescType;

	
/** \brief Hold the data used to operate on the buffer descriptor ring. */
typedef struct
{
    Eth_CpdmaBuffDescType *pFreeHead;
    /**< The head of the bd chain which can be allocated for transmission. */
    Eth_CpdmaBuffDescType *pHead;
    /**< The head of the bd chain which is transmitting data. */
    Eth_CpdmaBuffDescType *pTail;
    /**< The tail of the bd chain which is transmitting data. */
    uint32           freeBuffDesc;
    /**< The number of free bd's, which can be allocated for transmission. */
} Eth_CpdmaBuffDescQueue;

/** \brief Eth port object
 *         This structure will contain information provided by
 *         application for port.
 */
typedef struct
{
    Eth_PortConfigType portCfg;
    /**< Port configuration */
    uint8             portNum;
    /**< CPSW port number in the device*/
    Eth_BufObjType     rxBufObjArray[ETH_NUM_RX_BUFFERS];
    /**< Buffer object for Receive buffer ring */
    Eth_BufObjType     txBufObjArray[ETH_NUM_TX_BUFFERS];
    /**< Buffer object for Transmit buffer ring */
} Eth_PortObject;

/**
 *  \brief  CPSW Statistics Structure
 *
 *  The statistics structure is used to hold, in software, the
 *  hardware statistics reported by the STATS module of the CPSW Eth Port.
 */
typedef struct {
    uint32 RXGOODFRAMES;              /* RxGoodFrames */
    uint32 RXBROADCASTFRAMES;         /* RxBroadcastFrames */
    uint32 RXMULTICASTFRAMES;         /* RxMulticastFrames */
    uint32 RXPAUSEFRAMES;             /* RxPauseFrames */
    uint32 RXCRCERRORS;               /* RxCRCErrors */
    uint32 RXALIGNCODEERRORS;         /* RxAlignCodeErrors */
    uint32 RXOVERSIZEDFRAMES;         /* RxOversizedFrames */
    uint32 RXJABBERFRAMES;            /* RxJabberFrames */
    uint32 RXUNDERSIZEDFRAMES;        /* RxUndersizedFrames */
    uint32 RXFRAGMENTS;               /* RxFragments */
    uint32 ALE_DROP;                  /* ALE_Drop */
    uint32 ALE_OVERRUN_DROP;          /* ALE_Overrun_Drop */
    uint32 RXOCTETS;                  /* RxOctets */
    uint32 TXGOODFRAMES;              /* TxGoodFrames */
    uint32 TXBROADCASTFRAMES;         /* TxBroadcastFrames */
    uint32 TXMULTICASTFRAMES;         /* TxMulticastFrames */
    uint32 TXPAUSEFRAMES;             /* TxPauseFrames */
    uint32 TXDEFERREDFRAMES;          /* TxDeferredFrames */
    uint32 TXCOLLISIONFRAMES;         /* TxCollisionFrames */
    uint32 TXSINGLECOLLFRAMES;        /* TxSingleCollFrames */
    uint32 TXMULTCOLLFRAMES;          /* TxMultCollFrames */
    uint32 TXEXCESSIVECOLLISIONS;     /* TxExcessiveCollisions */
    uint32 TXLATECOLLISIONS;          /* TxLateCollisions */
    uint32 RXIPGERROR;                /* RxIPGError */
    uint32 TXCARRIERSENSEERRORS;      /* TxCarrierSenseErrors */
    uint32 TXOCTETS;                  /* TxOctets */
    uint32 OCTETFRAMES64;             /* OctetFrames64 */
    uint32 OCTETFRAMES65T127;         /* OctetFrames65t127 */
    uint32 OCTETFRAMES128T255;        /* OctetFrames128t255 */
    uint32 OCTETFRAMES256T511;        /* OctetFrames256t511 */
    uint32 OCTETFRAMES512T1023;       /* OctetFrames512t1023 */
    uint32 OCTETFRAMES1024TUP;        /* OctetFrames1024tUP */
    uint32 NETOCTETS;                 /* NetOctets */
    uint32 RX_BOTTOM_OF_FIFO_DROP;    /* Rx_Bottom_of_FIFO_Drop */
    uint32 PORTMASK_DROP;             /* Portmask_Drop */
    uint32 RX_TOP_OF_FIFO_DROP;       /* Rx_Top_of_FIFO_Drop */
    uint32 ALE_RATE_LIMIT_DROP;       /* ALE_Rate_Limit_Drop */
    uint32 ALE_VID_INGRESS_DROP;      /* ALE_VID_Ingress_Drop */
    uint32 ALE_DA_EQ_SA_DROP;         /* ALE_DA_EQ_SA_Drop */
    uint32 ALE_BLOCK_DROP;            /* ALE_Block_Drop */
    uint32 ALE_SECURE_DROP;           /* ALE_Secure_Drop */
    uint32 ALE_AUTH_DROP;             /* ALE_Auth_Drop */
    uint32 ALE_UNKN_UNI;              /* ALE_Unkn_Uni */
    uint32 ALE_UNKN_UNI_BCNT;         /* ALE_Unkn_Uni_Bcnt */
    uint32 ALE_UNKN_MLT;              /* ALE_Unkn_Mlt */
    uint32 ALE_UNKN_MLT_BCNT;         /* ALE_Unkn_Mlt_Bcnt */
    uint32 ALE_UNKN_BRD;              /* ALE_Unkn_Brd */
    uint32 ALE_UNKN_BRD_BCNT;         /* ALE_Unkn_Brd_Bcnt */
    uint32 ALE_POL_MATCH;             /* ALE_Pol_Match */
    uint32 ALE_POL_MATCH_RED;         /* ALE_Pol_Match_Red */
    uint32 ALE_POL_MATCH_YELLOW;      /* ALE_Pol_Match_Yellow */
    uint32 ALE_MULT_SA_DROP;          /* ALE_MULT_SA_DROP */
    uint32 ALE_DUAL_VLAN_DROP;        /* ALE_DUAL_VLAN_DROP */
    uint32 ALE_LEN_ERROR_DROP;        /* ALE_LEN_ERROR_DROP */
    uint32 ALE_IP_NEXT_HDR_DROP;      /* ALE_IP_NEXT_HDR_DROP */
    uint32 ALE_IPV4_FRAG_DROP;        /* ALE_IPV4_FRAG_DROP */
    uint32 Resv_1[24];
    uint32 IET_RX_ASSEMBLY_ERROR_REG;   /* iet_rx_assembly_error */
    uint32 IET_RX_ASSEMBLY_OK_REG;    /* iet_rx_assembly_ok */
    uint32 IET_RX_SMD_ERROR_REG;      /* iet_rx_smd_error */
    uint32 IET_RX_FRAG_REG;           /* iet_rx_frag */
    uint32 IET_TX_HOLD_REG;           /* iet_tx_hold */
    uint32 IET_TX_FRAG_REG;           /* iet_tx_frag */
    uint32 Resv_2[9];
    uint32 TX_MEMORY_PROTECT_ERROR;   /* Tx_Memory_Protect_Error */
	uint32 ENET_PN_TX_PRI_REG[8];     /* enet_pn_tx_pri */
    uint32 ENET_PN_TX_PRI_BCNT_REG[8];   /* enet_pn_tx_pri_bcnt */
    uint32 ENET_PN_TX_PRI_DROP_REG[8];   /* enet_pn_tx_pri_drop */
    uint32 ENET_PN_TX_PRI_DROP_BCNT_REG[8];   /* enet_pn_tx_pri_drop_bcnt */
}Eth_StatsType;

/** \brief Eth stats object
 *         This structure will contain statistics information.
 */
typedef struct
{
    uint32        enableStatistics;
    /**< Flag to indicate if the statistics gathering is enabled or disabled. */
    Eth_StatsType stats;
    /**< Statistics Structure. */
}
Eth_StatsObj;

/** \brief Eth controller driver object
 *         This structure will contain information provided by application
 *         and common information shared by ports */
typedef struct
{
    uint8                  ctrlIdx;
    /**< Controller index */
	uint8                  portIdx;
	/**< Port >*/
    Eth_ConfigType          ethConfig;
    /**< Eth driver Init Configuration */
    uint32                  enableCacheOps;
    /**< Since the Eth driver is updating (writing/reading) the packet header,
     *   if the packets are in cached region, then cache flush should be done
     *   for transmit and cache invalidate should be done for receive.
     *
     *   This flag determines whether the cache operations need to be enabled
     *   or not. If enabled, then the below two callback will be called to
     *   perform the cache operation
     */
    Eth_CacheFlushType      cacheFlushFnPtr;
    /**< Cache flush function pointer */
    Eth_CacheInvalidateType cacheInvalidateFnPtr;
    /**< Cache invalidate function pointer */
    Eth_ModeType            ctrlMode;
    /**< CPSW instance in the device  */
    uint32                  baseAddr;
    /**< Base address */
    uint32                  descMemBaseAddr __attribute__ ((aligned(128)));
    /**< CPPI RAM Base address */
    uint8                   activeMACPortCount;
    /**< Total number active ports */
    Eth_PortObject          portObj;
    /**< Port interface */
    Eth_CpdmaBuffDescQueue    rxDescRing;
    /**< Receive buffer descriptor queue*/
    uint32                  maxRxBuffDesc;
    /**< Maximum Rx buffer descriptors */
    Eth_CpdmaBuffDescQueue    txDescRing;
    /**< Transmit buffer descriptor queue */
    uint32                  maxTxBuffDesc;
    /**< Maximum Tx buffer descriptors */
    Eth_StatsObj            statsObj;
    /**< Statistics object */
    Eth_CptsStateObj        cptsObj; 
    /**< CPTS object */
} Eth_DrvObject;

/* ========================================================================== */
/*                         Global Variables Declarations                      */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                        Deprecated Function Declarations                    */
/* ========================================================================== */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef CPSW_PRIV_H_ */

