 /*   Copyright (C) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     EthTrcv.h
 *
 *  \brief    This file contains interface header for ETHTRCV MCAL driver
 *  @{
 */

/**
 *
 * \defgroup MCAL_ETHTRCV_API Ethernet Transceiver Driver API
 *
 * The EthTrcv Driver MCAL provides low level access to Ethernet transceiver
 * present on AM273 Device platform.<br>
 * The EthTrcv Driver implements a standardized interface specified in the
 * AUTOSAR_SWS_EthernetTransceiverDriver document.<br>
 * The EthTrcv Driver is part of the Communication Stack in the AUTOSAR Basic
 * Software (BSW)<br>
 *  @{
 */

/*
 * Below are the global requirements which are met by this Eth
 * driver which can't be mapped to a particular source ID
 */
/*
 * Requirements : SWS_EthTrcv_00004, SWS_EthTrcv_00007, SWS_EthTrcv_00008,
 *                SWS_EthTrcv_00009, SWS_EthTrcv_00010, SWS_EthTrcv_00085
 */

#ifndef ETHTRCV_H_
#define ETHTRCV_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "ComStack_Types.h"
/* Requirements : SWS_EthTrcv_00095 */
#include "Eth_GeneralTypes.h"
#include "EthTrcv_Cfg.h"
#include "EcuM_Cbk.h"
#include "EthTrcv_Types.h"

#define ETHTRCV_START_SEC_VAR_INIT_UNSPECIFIED
#include "EthTrcv_MemMap.h"

extern VAR(volatile EthTrcv_StateType, ETH_VAR_ZERO_INIT) EthTrcv_DrvStatus ;

#define ETHTRCV_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "EthTrcv_MemMap.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/**
 *  \name EthTrcv Driver Module SW Version Info
 *
 *  Defines for ETHTRCV Driver version used for compatibility checks
 *  @{
 */
/* Requirements : SWS_EthTrcv_00005 */
/** \brief Driver Implementation Major Version. */
#define ETHTRCV_SW_MAJOR_VERSION             (8U)
/** \brief Driver Implementation Minor Version. */
#define ETHTRCV_SW_MINOR_VERSION             (6U)
/** \brief Driver Implementation Patch Version. */
#define ETHTRCV_SW_PATCH_VERSION             (2U)
/* @} */

/**
 *  \name EthTrcv Driver Module AUTOSAR Version Info
 *
 *  Defines for ETHTRCV Driver AUTOSAR version used for compatibility checks
 *  @{
 */
/* Requirements : SWS_EthTrcv_00005 */
/** \brief AUTOSAR Major Version implemented. */
#define ETHTRCV_AR_RELEASE_MAJOR_VERSION     (4U)
/** \brief AUTOSAR Minor Version implemented. */
#define ETHTRCV_AR_RELEASE_MINOR_VERSION     (3U)
/** \brief AUTOSAR patch Version implemented. */
#define ETHTRCV_AR_RELEASE_REVISION_VERSION  (1U)
/* @} */

/**
 *  \name EthTrcv Driver ID Info
 *  @{
 */
/** \brief Texas Instruments Vendor ID */
#define ETHTRCV_VENDOR_ID                    (44U)
/** \brief EthTrcv Driver  ID */
#define ETHTRCV_MODULE_ID                    (73U)
/** \brief EthTrcv Instance ID */
#define ETHTRCV_INSTANCE_ID                  (0U)
/* @} */

/**
 *  \name EthTrcv Driver Service Id
 *
 *  The Service Id is one of the argument to Det_ReportError function and is
 *  used to identify the source of the error
 *  @{
 */
/** \brief EthTrcv_Init() API Service ID */
#define ETHTRCV_INIT_ID                             ((uint8) 0x01U)

/** \brief EthTrcv_SetTransceiverMode() API Service ID */
#define ETHTRCV_SETTRCVMODE_ID                      ((uint8) 0x03U)

/** \brief EthTrcv_GetTransceiverMode() API Service ID */
#define ETHTRCV_GETTRCVMODE_ID                      ((uint8) 0x04U)

/** \brief EthTrcv_StartAutoNegotiation() API Service ID */
#define ETHTRCV_SRTAUTONEGOTIA_ID                   ((uint8) 0x05U)

/** \brief EthTrcv_GetLinkState() API Service ID */
#define ETHTRCV_GETLNKSTATE_ID                      ((uint8) 0x06U)

/** \brief EthTrcv_GetBaudRate() API Service ID */
#define ETHTRCV_GETBAUD_ID                          ((uint8) 0x07U)

/** \brief EthTrcv_GetDuplexMode() API Service ID */
#define ETHTRCV_GETDPLXMODE_ID                      ((uint8) 0x08U)

/** \brief EthTrcv_ReadMiiIndication() API Service ID */
#define ETHTRCV_READMIIIND_ID                       ((uint8) 0x09U)

/** \brief EthTrcv_WriteMiiIndication() API Service ID */
#define ETHTRCV_WRITEMIIIND_ID                      ((uint8) 0x0AU)

/** \brief EthTrcv_GetVersionInfo() API Service ID */
#define ETHTRCV_VERSION_ID                          ((uint8) 0x0BU)

/** \brief EthTrcv_MainFunction() API Service ID */
#define ETHTRCV_MAINFCT_WRITE_ID                    ((uint8) 0x0CU)

/** \brief  EthTrcv_SetTransceiverWakeupMode() API Service ID */
#define ETHTRCV_SETTRCVWKUPMODE_ID                  ((uint8) 0x0DU)

/** \brief EthTrcv_GetTransceiverWakeupMode() API Service ID */
#define ETHTRCV_GETTRCVWKUPMODE_ID                  ((uint8) 0x0EU)

/** \brief EthTrcv_CheckWakeup() API Service ID */
#define ETHTRCV_CHKWKUP_ID                          ((uint8) 0x0FU)

/** \brief EthTrcv_CheckWakeup() API Service ID */
#define ETHTRCV_SETPHYTESTMODE_ID                   ((uint8) 0x11U)

/** \brief EthTrcv_CheckWakeup() API Service ID */
#define ETHTRCV_SETPHYLOOPBCKMODE_ID                ((uint8) 0x12U)

/** \brief EthTrcv_SetPhyTxMode() API Service ID */
#define ETHTRCV_SETPHYTXMODE_ID					 	((uint8) 0x13U)

/** \brief EthTrcv_GetCableDiagnosticsResult() API Service ID */
#define ETHTRCV_GETCBLDIAGNSTICRSLT_ID		     	((uint8) 0x14U)

/** \brief EthTrcv_GetPhyIdentifier() API Service ID */
#define ETHTRCV_GETPHYIDENTIFIER_ID		         	((uint8) 0x15U)

/**
 *  \name EthTrcv Error Codes
 *
 *  Error codes returned by EthTrcv functions
 *  @{
 */
/* SourceId :  */
/* DesignId :  */
/* Requirements : SWS_EthTrcv_00017 */

/** \brief ERROR: Invalid transceiver index */
#define ETHTRCV_E_INV_TRCV_IDX                       ((uint8) 0x01U)

/** \brief ERROR: EthTrcv module was not initialized */
#define ETHTRCV_E_UNINIT      			             ((uint8) 0x02U)

/**< brief NULL_PTR pointer in parameter list */
#define ETHTRCV_E_PARAM_POINTER                      ((uint8) 0x03U)

/** \brief ERROR: EthTrcv mode is not supported */
#define ETHTRCV_E_NOT_SUPPORTED				         ((uint8) 0x04U)


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/**
 *  \addtogroup MCAL_ETHTRCV_CFG EthTrcv Configuration
 *  @{
 */

/* @} */

/* ========================================================================== */
/*                API AND FUNCTION PROTOTYPES                                 */
/* ========================================================================== */

#if (STD_ON == ETHTRCV_VERSION_INFO_API)
/**
 * \brief Function returns the version information of this module.
 *
 * \verbatim
 * Service name      : EthTrcv_GetVersionInfo
 * Syntax            : void EthTrcv_GetVersionInfo(Std_VersionInfoType* 
 *                                                           versioninfo)
 * Service ID[hex]   : 0x07
 * Sync/Async        : Synchronous
 * Reentrancy        : Reentrant
 * Parameters (in)   : None
 * Parameters (inout): None
 * Parameters (out)  : versioninfo
 *                     Pointer to where to store the version information of this
 *                     module.
 * Return value      : None
 * Description       : This function returns the version information of this
 *                     module
 * \endverbatim
 */
FUNC(void, ETHTRCV_CODE)
EthTrcv_GetVersionInfo(P2VAR(Std_VersionInfoType, AUTOMATIC,
                             ETHTRCV_APPL_DATA)
                       VersionInfoPtr);
#endif /* #if (STD_ON == ETHTRCV_VERSION_INFO_API) */

/**
 * \brief This function initializes the ETHTRCV controller
 *        All the required initialisation for the ETHTRCV driver and hardware is
 *        performed in this function.
 *
 * \verbatim
 * Service name      : EthTrcv_Init
 * Syntax            : void EthTrcv_Init( const EthTrcv_ConfigType* ConfigPtr )
 * Service ID[hex]   : 0x01
 * Sync/Async        : Synchronous
 * Reentrancy        : Non Reentrant
 * Parameters (in)   : CfgPtr
 *                     Points to the implementation specific structure
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : None
 * Description       : This function initializes the module.
 * \endverbatim
 */
FUNC(void, ETHTRCV_CODE) EthTrcv_Init(P2CONST(EthTrcv_ConfigType, AUTOMATIC,
                                              ETHTRCV_PBCFG) CfgPtr);

#if (STD_ON == ETHTRCV_SETTRANSCEIVERMODE_API)
/**
 * \brief This function enables / disables the indexed transceiver
 *
 * \verbatim
 * Service name      : EthTrcv_SetTransceiverMode
 * Syntax            : Std_ReturnType EthTrcv_SetTransceiverMode(uint8
 *                     TrcvIdx,EthTrcv_ModeType CtrlMode)
 * Service ID[hex]   : 0x03
 * Sync/Async        : Asynchronous
 * Reentrancy        : Non Reentrant
 * Parameters (in)   : TrcvIdx
 *                     Index of the transceiver within the context of the
 *                     Ethernet Transceiver Driver.
 *                     CtrlMode
 *                      ETHTRCV_MODE_DOWN: disable the transceiver
 *                      ETHTRCV_MODE_ACTIVE: enable the transceiver
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : Std_ReturnType
 *                      E_OK: Service accepted
 *                      E_NOT_OK: Service denied
 * Description       : This function enables / disables the indexed transceiver
 * \endverbatim
 */
FUNC(Std_ReturnType, ETHTRCV_CODE) EthTrcv_SetTransceiverMode(
    uint8 TrcvIdx, EthTrcv_ModeType CtrlMode);
#endif /* #if (STD_ON == ETHTRCV_SETTRANSCEIVERMODE_API) */

#if (STD_ON == ETHTRCV_GETTRANSCEIVERMODE_API)
/**
 * \brief This function obtains the state of the indexed transceiver
 *
 * \verbatim
 * Service name      : EthTrcv_GetTransceiverMode
 * Syntax            : Std_ReturnType EthTrcv_GetTransceiverMode(uint8
 *                      TrcvIdx,EthTrcv_ModeType* TrcvModePtr)
 * Service ID[hex]   : 0x04
 * Sync/Async        : Synchronous
 * Reentrancy        : Non Reentrant
 * Parameters (in)   : TrcvIdx
 *                     Index of the transceiver within the context of the
 *                     Ethernet Transceiver Driver.
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : Std_ReturnType
 *                      E_OK: success
 *                      E_NOT_OK: transceiver could not be initialized
 * Description       : This function obtains the state of the indexed
 *                     transceiver
 * \endverbatim
 */
FUNC(Std_ReturnType, ETHTRCV_CODE) EthTrcv_GetTransceiverMode(
    uint8 TrcvIdx, EthTrcv_ModeType * TrcvModePtr);
#endif /* #if (STD_ON == ETHTRCV_GETTRANSCEIVERMODE_API) */


 #if (STD_ON == ETHTRCV_STARTAUTONEGOTIATION_API)
/**
 * \brief This function restarts the negotiation of the transmission parameters
 *                     used by the indexed.
 *
 * \verbatim
 * Service name      : EthTrcv_StartAutoNegotiation
 * Syntax            : Std_ReturnType EthTrcv_StartAutoNegotiation(uint8
 *                     TrcvIdx)
 * Service ID[hex]   : 0x05
 * Sync/Async        : Synchronous
 * Reentrancy        : Non Reentrant
 * Parameters (in)   : TrcvIdx
 *                     Index of the transceiver within the context of the
 *                     Ethernet Transceiver Driver.
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : Std_ReturnType
 *                      E_OK    : success
 *                      E_NOT_OK: transceiver could not be initialized
 * Description       : Restarts the negotiation of the transmission parameters
 *                     used by the indexed
 * \endverbatim
 */
FUNC(Std_ReturnType, ETHTRCV_CODE) EthTrcv_StartAutoNegotiation(uint8 TrcvIdx);
#endif /* #if (STD_ON == ETHTRCV_STARTAUTONEGOTIATION_API) */

#if (STD_ON == ETHTRCV_GETLINKSTATE_API)
/**
 * \brief This function obtains the link state of the indexed transceiver
 *
 * \verbatim
 * Service name      : EthTrcv_GetLinkState
 * Syntax            : Std_ReturnType EthTrcv_GetLinkState(uint8
 *                     TrcvIdx,EthTrcv_LinkStateType* LinkStatePtr)
 * Service ID[hex]   : 0x06
 * Sync/Async        : Synchronous
 * Reentrancy        : Non Reentrant
 * Parameters (in)   : TrcvIdx
 *                     Index of the transceiver within the context of the
 *                     Ethernet Transceiver Driver.
 * Parameters (inout): None
 * Parameters (out)  : LinkStatePtr
 *                      ETHTRCV_LINK_STATE_DOWN: transceiver is disconnected
 *                      ETHTRCV_LINK_STATE_ACTIVE: transceiver is connected
 * Return value      : Std_ReturnType
 *                      E_OK    : success
 *                      E_NOT_OK: transceiver could not be initialized
 * Description       : Obtains the link state of the indexed transceiver
 * \endverbatim
 */
FUNC(Std_ReturnType, ETHTRCV_CODE) EthTrcv_GetLinkState(
    uint8 TrcvIdx, EthTrcv_LinkStateType * LinkStatePtr);
#endif /* #if (STD_ON == ETHTRCV_GETLINKSTATE_API) */

/*
 * Requirements: SWS_EthTrcv_00089: The function is not required or called by
 *               an upper layer BSW software component. So making internal.
 */
#if (STD_ON == ETHTRCV_GETBAUDRATE_API)
/**
 * \brief This function obtains the baud rate of the indexed transceiver
 *
 * \verbatim
 * Service name      : EthTrcv_GetBaudRate
 * Syntax            : Std_ReturnType EthTrcv_GetBaudRate(uint8
 *                     TrcvIdx,EthTrcv_BaudRateType* BaudRatePtr)
 * Service ID[hex]   : 0x07
 * Sync/Async        : Synchronous
 * Reentrancy        : Non Reentrant
 * Parameters (in)   : TrcvIdx
 *                     Index of the transceiver within the context of the
 *                     Ethernet Transceiver Driver.
 * Parameters (inout): None
 * Parameters (out)  : BaudRatePtr
 *                      ETHTRCV_BAUD_RATE_10MBIT: 10MBit connection
 *                      ETHTRCV_BAUD_RATE_100MBIT: 100MBit connection
 * Return value      : Std_ReturnType
 *                      E_OK    : success
 *                      E_NOT_OK: transceiver could not be initialized
 * Description       : Obtains the baud rate of the indexed transceiver
 * \endverbatim
 */
FUNC(Std_ReturnType, ETHTRCV_CODE) EthTrcv_GetBaudRate(
    uint8 TrcvIdx, EthTrcv_BaudRateType * BaudRatePtr);
#endif /* #if (STD_ON == ETHTRCV_GETBAUDRATE_API) */

#if (STD_ON == ETHTRCV_GETDUPLEXMODE_API)
/**
 * \brief This function obtains the duplex mode of the indexed transceiver
 *
 * \verbatim
 * Service name      : EthTrcv_GetDuplexMode
 * Syntax            : Std_ReturnType EthTrcv_GetDuplexMode(uint8
 *                     TrcvIdx,EthTrcv_DuplexModeType* DuplexModePtr)
 * Service ID[hex]   : 0x08
 * Sync/Async        : Synchronous
 * Reentrancy        : Non Reentrant
 * Parameters (in)   : TrcvIdx
 *                     Index of the transceiver within the context of the
 *                     Ethernet Transceiver Driver.
 * Parameters (inout): None
 * Parameters (out)  : DuplexModePtr
 *                      ETHTRCV_DUPLEX_MODE_HALF: half duplex connections
 *                      ETHTRCV_DUPLEX_MODE_FULL: full duplex connection
 * Return value      : Std_ReturnType
 *                      E_OK    : success
 *                      E_NOT_OK: transceiver could not be initialized
 * Description       : Obtains the duplex mode of the indexed transceiver
 * \endverbatim
 */
FUNC(Std_ReturnType, ETHTRCV_CODE) EthTrcv_GetDuplexMode(
    uint8 TrcvIdx, EthTrcv_DuplexModeType * DuplexModePtr);
#endif /* #if (STD_ON == ETHTRCV_GETDUPLEXMODE_API) */

/**
 * \brief This function is called when information has been read out via MII
 *                     interface.
 *        Triggered by previous Eth_ReadMii call. Can directly be called within
 *                     Eth_ReadMii.
 *
 * \verbatim
 * Service name      : EthTrcv_ReadMiiIndication
 * Syntax            : void EthTrcv_ReadMiiIndication(uint8 CtrlIdx,uint8
 *                     TrcvIdx,uint8 RegIdx,uint8 RegVal)
 * Service ID[hex]   : 0x09
 * Sync/Async        : Synchronous
 * Reentrancy        : Non Reentrant for the same CtrlIdx, reentrant for
 *                     different
 * Parameters (in)   : CtrlIdx
 *                     Index of the controller within the context of the
 *                     Ethernet Driver
 *                     TrcvIdx
 *                     Index of the transceiver on the MII
 *                     RegIdx
 *                     Index of the transceiver register on the MII
 *                     RegVal
 *                     Value contained in the indexed register
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : None
 * Description       : This function is called when information has been read
 *                     out via MII interface.
 *                     Triggered by previous Eth_ReadMii call. Can directly be
 *                     called within Eth_ReadMii.
 * \endverbatim
 */
/** IMP Note : There is discrepancy in the Eth and EthTrcv specs. Data type for
  *            RegVal in EthTrcv_ReadMiiIndication is uint8 as per Eth Trcv spec
  *            where as Eth_ReadMii takes uint16 pointer RegValPtr. To avoid
  *            data loss we change RegVal data type to uint16
  */
FUNC(void, ETHTRCV_CODE) EthTrcv_ReadMiiIndication(uint8 CtrlIdx, uint8 TrcvIdx,
                                                   uint8 RegIdx,
                                                   uint16 RegVal);

/**
 * \brief This function is called when information has been written via MII
 *                     interface.
 *        Triggered by previous Eth_WriteMii call. Can directly be called within
 *                     Eth_WriteMii.
 *
 * \verbatim
 * Service name      : EthTrcv_WriteMiiIndication
 * Syntax            : void EthTrcv_WriteMiiIndication(uint8 CtrlIdx,uint8
 *                     TrcvIdx,uint8 RegId)
 * Service ID[hex]   : 0x0A
 * Sync/Async        : Synchronous
 * Reentrancy        : Non Reentrant for the same CtrlIdx, reentrant for
 *                     different
 * Parameters (in)   : CtrlIdx
 *                     Index of the controller within the context of the
 *                     Ethernet Driver
 *                     TrcvIdx
 *                     Index of the transceiver on the MII
 *                     RegIdx
 *                     Index of the transceiver register on the MII
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : None
 * Description       : This function is called when information has been written
 *                     via MII interface.
 *                     Triggered by previous Eth_WriteMii call. Can directly be
 *                     called within Eth_WriteMii.
 * \endverbatim
 */
FUNC(void, ETHTRCV_CODE) EthTrcv_WriteMiiIndication(uint8 CtrlIdx,
                                                    uint8 TrcvIdx,
                                                    uint8 RegId);

/**
 * \brief This function is used for polling state changes and wakeup reasons.
 *        Calls EthIf_TrcvModeIndication when the transceiver mode changed. 
 *        Stores wakeup events if EthTrcvWakeUpSupport is set to 
 *        ETHTRCV_WAKEUP_BY_POLLING..
 *
 * \verbatim
 * Service name      : EthTrcv_MainFunction
 * Syntax            : void EthTrcv_MainFunction(void)
 * Service ID[hex]   : 0x0c
 * Parameters (in)   : None
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : None
 * Description       : This function is used for polling state changes and
 *                     wakeup reasons. Calls
 *                     EthIf_TrcvModeIndication when the transceiver mode
 *                     changed. Stores wakeup
 *                     events if EthTrcvWakeUpSupport is set to
 *                     ETHTRCV_WAKEUP_BY_POLLING.
 * \endverbatim
 */
FUNC(void, ETHTRCV_CODE) EthTrcv_MainFunction(void);

/**
 * \brief This function is called when information has been written via MII
 *                     interface.
 *        Triggered by previous Eth_WriteMii call. Can directly be called within
 *                     Eth_WriteMii.
 *
 * \verbatim
 * Service name      : EthTrcv_SetPhyLoopbackMode
 * Syntax            : void EthTrcv_SetPhyLoopbackMode(uint8 TrcvIdx, 
 *                            EthTrcv_PhyLoopbackModeType Mode)
 * Service ID[hex]   : 0x0A
 * Sync/Async        : Synchronous
 * Reentrancy        : Non Reentrant for the same CtrlIdx, reentrant for
 *                     different
 * Parameters (in)   : CtrlIdx
 *                     Index of the controller within the context of the
 *                     Ethernet Driver
 *                     TrcvIdx
 *                     Index of the transceiver on the MII
 *                     RegIdx
 *                     Index of the transceiver register on the MII
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : None
 * Description       : This function is called when information has been written
 *                     via MII interface.
 *                     Triggered by previous Eth_WriteMii call. Can directly be
 *                     called within Eth_WriteMii.
 * \endverbatim
 */
 FUNC(Std_ReturnType, ETHTRCV_CODE)
    EthTrcv_SetPhyLoopbackMode(uint8 TrcvIdx, EthTrcv_PhyLoopbackModeType Mode);

#if (STD_ON == ETHTRCV_SETPHYTXMODE_API)		
/**
 * \brief This function is used for Activates a given transmission mode.
 *
 * \verbatim
 * Service name      : EthTrcv_SetPhyTxMode
 * Syntax            : Std_ReturnType EthTrcv_SetPhyTxMode(uint8 TrcvIdx, 
 *					    EthTrcv_PhyTxModeType Mode)
 * Service ID[hex]   : 0x13U
 * Parameters (in)   : TrcvIdx
 * 					   Index of the transceiver within the context of the 
 * 					   Ethernet Transceiver Driver
 * 					   Mode
 * 					   Transmission mode to be activated
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : Std_ReturnType
 *                      E_OK    : success
 *                      E_NOT_OK: transceiver could not be initialized
 * Description       : This function is used for Activates a given transmission 
 *                     mode
 * \endverbatim
 */
 FUNC(Std_ReturnType, ETHTRCV_CODE)
    EthTrcv_SetPhyTxMode(uint8 TrcvIdx, EthTrcv_PhyTxModeType Mode);
#endif /* #if (STD_ON == ETHTRCV_SETPHYTXMODE_API) */

#if (STD_ON == ETHTRCV_GETCABLEDIAGNOSTICSRESULT_API)
/**
 * \brief This function is used for Retrieves the cable diagnostics result of a 
 *		given transceiver
 *
 * \verbatim
 * Service name      : EthTrcv_GetCableDiagnosticsResult
 * Syntax            : Std_ReturnType EthTrcv_GetCableDiagnosticsResult(uint8  
 *					   TrcvIdx, EthTrcv_CableDiagResultType* ResultPtr)
 * Service ID[hex]   : 0x14U
 * Parameters (in)   : TrcvIdx
 * 					   Index of the transceiver within the context of the 
 * 					   Ethernet Transceiver Driver
 * Parameters (inout): None
 * Parameters (out)  : ResultPtr
 *					   Pointer to the location where the cable diagnostics 
 *					   result shall be stored
 * Return value      : Std_ReturnType
 *                      E_OK    : success
 *                      E_NOT_OK: transceiver could not be initialized
 * Description       : This function is used for Retrieves the cable diagnostics 
 *						result of a given transceiver
 * \endverbatim
 */
 FUNC(Std_ReturnType, ETHTRCV_CODE) EthTrcv_GetCableDiagnosticsResult(
							uint8 TrcvIdx,EthTrcv_CableDiagResultType* ResultPtr);
#endif /* #if (STD_ON == ETHTRCV_GETCABLEDIAGNOSTICSRESULT_API) */

#if (STD_ON == ETHTRCV_GETPHYIDENTIFIER_API)
/**
 * \brief This function Obtains the PHY identifier of the Ethernet Transceiver  
 * 			according to IEEE 802.3-2015 chapter 22.2.4.3.1 PHY Identifer
 *
 * \verbatim
 * Service name      : EthTrcv_GetPhyIdentifier
 * Syntax            : Std_ReturnType EthTrcv_GetPhyIdentifier(uint8 TrcvIdx, 
 * 						uint8 TrcvIdx, uint32* OrgUniqueIdPtr,
 * 						uint8* ModelNrPtr, uint8* RevisionNrPtr)
 * Service ID[hex]   : 0x15U
 * Parameters (in)   : TrcvIdx
 * 					   Index of the transceiver within the context of the 
 * 					   Ethernet Transceiver Driver
 * Parameters (inout): None
 * Parameters (out)  : OrgUniqueIdPtr
 *					   Pointer to the memory where the Organizationally Unique
 *					   Identifier shall be stored.
 *					   ModelNrPtr
 * 					   Pointer to the memory where the Manufacturer's Model 
 * 					   Number shall be stored.
 * 					   RevisionNrPtr
 *					   Pointer to the memory where the Revision Number shall be
 *                     stored.
 * Return value      : Std_ReturnType
 *                      E_OK    : success
 *                      E_NOT_OK: transceiver could not be initialized
 * Description       : Obtains the PHY identifier of the Ethernet Transceiver  
 * 					   according to IEEE 802.3-2015 chapter 22.2.4.3.1 PHY 
 *                     Identifer
 * \endverbatim
 */
 FUNC(Std_ReturnType, ETHTRCV_CODE) EthTrcv_GetPhyIdentifier(uint8 TrcvIdx, 
			uint32* OrgUniqueIdPtr, uint8* ModelNrPtr, uint8* RevisionNrPtr);
#endif /* #if (STD_ON == ETHTRCV_GETPHYIDENTIFIER_API) */

#if (STD_ON == ETHTRCV_SETPHYTESTMODE_API)
/**
 * \brief This function activates given test mode  
 *
 * \verbatim
 * Service name      : EthTrcv_SetPhyTestMode
 * Syntax            : Std_ReturnType EthTrcv_SetPhyTestMode(uint8 TrcvIdx, 
 * 					   EthTrcv_PhyTestModeType Mode)
 * Service ID[hex]   : 0x11U
 * Parameters (in)   : TrcvIdx
 * 					   Index of the transceiver within the context of the 
 * 					   Ethernet Transceiver Driver
 *                     Mode
 *                     Test mode to be activated
 * Parameters (inout): None
 * Parameters (out)  : None
 * Return value      : Std_ReturnType
 *                      E_OK    : The request has been accepted
 *                      E_NOT_OK: The request has not been accepted
 * Description       : Activates a given test mode  
 * \endverbatim
 */
 FUNC(Std_ReturnType, ETHTRCV_CODE) EthTrcv_SetPhyTestMode(uint8 TrcvIdx, 
			            EthTrcv_PhyTestModeType Mode);
#endif /* #if (STD_ON == ETHTRCV_SETPHYTESTMODE_API) */

#endif /* ETHTRCV_H_ end file*/
/*********************************************************************************************************************
 *  End of File: EthTrcv.h
 *********************************************************************************************************************/

