/*
*
* Copyright (c) 2022 Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
* license under copyrights and patents it now or hereafter owns or controls to make,
* have made, use, import, offer to sell and sell ("Utilize") this software subject to the
* terms herein.  With respect to the foregoing patent license, such license is granted
* solely to the extent that any such patent is necessary to Utilize the software alone.
* The patent license shall not apply to any combinations which include this software,
* other than combinations with devices manufactured by or for TI ("TI Devices").
* No hardware patent is licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this license
* (including the above copyright notice and the disclaimer and (if applicable) source
* code license limitations below) in the documentation and/or other materials provided
* with the distribution
*
* Redistribution and use in binary form, without modification, are permitted provided
* that the following conditions are met:
*
* No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* any redistribution and use are licensed by TI for use only with TI Devices.
*
* Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution of the
* source code are permitted provided that the following conditions are met:
*
* any redistribution and use of the source code, including any resulting derivative
* works, are licensed by TI for use only with TI Devices.
*
* any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

/**
  *  \file      Fls.c
  *
  *  \brief    This file contains Fls MCAL driver
  *
  */

 /* ========================================================================== */
 /*                             Include Files                                  */
 /* ========================================================================== */
#include "string.h"
#include "Fls.h"
#include "Fls_Brd_Nor.h"
#include "Fls_NOR_s25fl128sa.h"
#include "Det.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* AUTOSAR version information check has to match definition in header file */
#if ((FLS_AR_RELEASE_MAJOR_VERSION != (4U)) || \
    (FLS_AR_RELEASE_MINOR_VERSION != (3U)) ||  \
    (FLS_AR_RELEASE_REVISION_VERSION != (1U)))
    #error "Fls: AUTOSAR Version Numbers of Fls are different!!"
#endif

/* AUTOSAR version information check has to match definition in header file */
#if ((FLS_SW_MAJOR_VERSION != (8U)) || \
    (FLS_SW_MINOR_VERSION != (6U)) ||  \
    (FLS_SW_PATCH_VERSION != (2U)))
    #error "Fls: Software Version Numbers are inconsistent!!"
#endif

#define MAX_SECTOR_ERASETIME_CONV_IN_USEC  1000U
#define MAX_BLOCK_ERASETIME_CONV_IN_USEC   10000U
#define MAX_CHIP_ERASETIME_CONV_IN_USEC    4000000U

#define MAX_SECTOR_READWRITETIME_CONV_IN_USEC  1000U
#define MAX_BLOCK_READWRITETIME_CONV_IN_USEC   10000U
#define MAX_CHIP_READWRITETIME_CONV_IN_USEC    4000000U

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

#if (STD_ON == FLS_DEV_ERROR_DETECT)
/**
 * Check Flash bank ranges for a valid address
 *
 */
static Std_ReturnType checkValidAddress( Fls_AddressType SourceAddress )
{
    Std_ReturnType retVal = E_NOT_OK;
    uint32 startAddr_sectorset1 = 0x0U;
    uint32 EndAddr_sectorset1 = 0x1FFFFU;
    uint32 startAddr_sectorset2 = 0XFE0000U;
    uint32 EndAddr_sectorset2 = 0XFFFFFFU;
    uint32 startAddr_block = 0U;
    uint32 EndAddr_block = 0XFFFFFFU;

    
        if(Fls_DrvObj.typeoferase == FLS_SECTOR_ERASE)
        {
            if( ((SourceAddress >= startAddr_sectorset1) && (SourceAddress < EndAddr_sectorset1)) || ((SourceAddress >= startAddr_sectorset2) && (SourceAddress < EndAddr_sectorset2)) )
            {
                    retVal = E_OK;
            }
        }
        if(Fls_DrvObj.typeoferase == FLS_BLOCK_ERASE)
        {
            if( (SourceAddress >= startAddr_block) && (SourceAddress < EndAddr_block) )
            {
                    retVal = E_OK;
            }
        }
        if(Fls_DrvObj.typeoferase == FLS_CHIP_ERASE)
        {
                 retVal = E_OK;
        }
    
    return retVal;
}

static Std_ReturnType checkValidSectorBlockLength(Fls_LengthType Length )
{
    Std_ReturnType retVal = E_NOT_OK;

        if(Fls_DrvObj.typeoferase == FLS_SECTOR_ERASE)
        {
                if((Length >= NOR_SECTOR_SIZE)&&(Length % NOR_SECTOR_SIZE == 0))
                {
                    retVal = E_OK;
                }
        }
        if(Fls_DrvObj.typeoferase == FLS_BLOCK_ERASE)
        {
                if((Length >= NOR_BLOCK_SIZE)&&(Length % NOR_BLOCK_SIZE == 0))
                {
                    retVal = E_OK;
                }
        }
        if(Fls_DrvObj.typeoferase == FLS_CHIP_ERASE)
        {
                 retVal = E_OK;
        }
    
    return retVal;
}

/**
 * Checks sector alignment for a valid address
 *
 */ 
static Std_ReturnType checkSectorAlignment( Fls_AddressType SourceAddress ) 
{
   Std_ReturnType retVal = E_NOT_OK;
   if( (SourceAddress % NOR_SECTOR_SIZE) == 0 )
   {
       retVal = E_OK;
   }
   return retVal;
}

/**
 * Checks block alignment for a valid address
 *
 */ 
static Std_ReturnType checkBlockAlignment( Fls_AddressType SourceAddress )
{
   Std_ReturnType retVal = E_NOT_OK;
   if( (SourceAddress % NOR_BLOCK_SIZE) == 0 )
   {
       retVal = E_OK;
   }
   return retVal;
}

/**
 * Checks page alignment for a valid address
 *
 */
static Std_ReturnType checkPageAlignment( Fls_AddressType SourceAddress )
{
   Std_ReturnType retVal = E_NOT_OK;
   if( (SourceAddress % NOR_PAGE_SIZE) == 0 )
   {
       retVal = E_OK;
   }
   return retVal;
}
#endif /*#if (STD_ON == FLS_DEV_ERROR_DETECT)*/

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

#define FLS_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Fls_MemMap.h"
/** \brief FLS driver object */
Fls_DriverObjType Fls_DrvObj =
{
    .status = MEMIF_UNINIT
};
#define FLS_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Fls_MemMap.h"

#define FLS_START_SEC_VAR_NO_INIT_32
#include "Fls_MemMap.h"

/*Set Erase Type Parameters */
VAR(uint32, FLS_VAR_NO_INIT) Data_Size_Test;
VAR(uint32, FLS_VAR_NO_INIT) Data_Value;
VAR(uint32, FLS_VAR_NO_INIT) sector_or_blocksize;

#define FLS_STOP_SEC_VAR_NO_INIT_32
#include "Fls_MemMap.h"

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#define FLS_START_SEC_CODE
#include "Fls_MemMap.h"

/**
 *  \Function Name: Fls_Init
 *
 *  The function Fls_Init shall initialize the FLS module and all 
 *	flash memory relevant registers (hardware) with parameters
 *	provided in the given configuration set
 *
 */
FUNC(void, FLS_CODE) Fls_Init(
        P2CONST(Fls_ConfigType, AUTOMATIC, FLS_CONFIG_DATA) ConfigPtr)
{
   const Fls_ConfigType *CfgPtr = NULL_PTR;

#if (STD_ON == FLS_PRE_COMPILE_VARIANT)
    if (NULL_PTR == ConfigPtr)
    {
      CfgPtr = &FLS_INIT_CONFIG_PC;
    }
#endif  /* (STD_ON == FLS_PRE_COMPILE_VARIANT) */

#if (STD_ON == FLS_POST_BUILD_VARIANT)
   if (NULL_PTR != ConfigPtr)
   {
     CfgPtr = ConfigPtr;
   }
#endif  /* (STD_ON == FLS_POST_BUILD_VARIANT) */

#if (STD_ON == FLS_DEV_ERROR_DETECT)
    if (MEMIF_BUSY == Fls_DrvObj.status)
    {
        (void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_INIT, FLS_E_BUSY);
    }
    else if(NULL_PTR == CfgPtr)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_INIT, FLS_E_PARAM_CONFIG);
    }
    else if ((CfgPtr->maxReadNormalMode == 0U) || ((CfgPtr->maxReadNormalMode % NOR_PAGE_SIZE) != 0U))
    {
        (void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_INIT, FLS_E_PARAM_CONFIG);
    }
    else if ((CfgPtr->maxWriteNormalMode == 0U) || ((CfgPtr->maxWriteNormalMode % NOR_PAGE_SIZE) != 0U))
    {
        (void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_INIT, FLS_E_PARAM_CONFIG);
    }
    else
#endif /* #if (STD_ON == FLS_DEV_ERROR_DETECT) */
    {
        /* Instantiate a Driver Obj to be used by module */
        Fls_resetDrvObj(&Fls_DrvObj);
        /*Copy the input config parameters to Driver Object */
        Fls_copyConfig(&Fls_DrvObj, CfgPtr);
        Fls_DrvObj.jobResultType = MEMIF_JOB_PENDING;

        /*Init the HW */
        if(Fls_hwUnitInit() == E_OK)
        {
            Fls_DrvObj.jobResultType = MEMIF_JOB_OK;
            Fls_DrvObj.status = MEMIF_IDLE;
        }
    }
    return;
}

/**
 *  \Function Name: Fls_Erase
 *
 *  The function Fls_Erase shall erase one or more complete flash sectors.
 *
 */
FUNC(Std_ReturnType, FLS_CODE) Fls_Erase(
        Fls_AddressType TargetAddress, Fls_LengthType Length)
{
    Std_ReturnType retVal = (Std_ReturnType) E_OK;
    Fls_AddressType eraseStartAddress = TargetAddress + (FLS_BASE_ADDRESS & FLS_BASE_ADDRESS_REQ);

#if (STD_ON == FLS_DEV_ERROR_DETECT)
    if (MEMIF_UNINIT == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_ERASE, FLS_E_UNINIT);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if(MEMIF_BUSY == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_ERASE, FLS_E_BUSY);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if (((Length >= NOR_SECTOR_SIZE) && (Length < NOR_BLOCK_SIZE) && (E_NOT_OK == checkSectorAlignment(eraseStartAddress))) || 
             ((Length >= NOR_BLOCK_SIZE) && (E_NOT_OK == checkBlockAlignment(eraseStartAddress))) ||
             (E_NOT_OK == checkValidAddress(eraseStartAddress)))                                      
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_ERASE, FLS_E_PARAM_ADDRESS);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if ((Length == 0U) || ((Length >= NOR_SECTOR_SIZE)&&(Length < NOR_BLOCK_SIZE) && (E_NOT_OK == checkSectorAlignment(eraseStartAddress + Length))) ||
             ((Length >= NOR_BLOCK_SIZE) && (E_NOT_OK == checkBlockAlignment(eraseStartAddress + Length))) ||
             (E_NOT_OK == checkValidAddress(eraseStartAddress + Length - 1U)) || 
             (E_NOT_OK == checkValidSectorBlockLength(Length)))
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_ERASE, FLS_E_PARAM_LENGTH);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else
#endif /* #if (STD_ON == FLS_DEV_ERROR_DETECT) */
    {
        Fls_DrvObj.status = MEMIF_BUSY;
        Fls_DrvObj.jobResultType = MEMIF_JOB_PENDING;
        Fls_DrvObj.jobType = FLS_JOB_ERASE;

        /* [ SWS_Fls_00221 ] */
        Fls_DrvObj.flashAddr = eraseStartAddress;
        Fls_DrvObj.ramAddr = NULL;
        Fls_DrvObj.length = Length;
        Fls_DrvObj.transferred = 0;
		
		if(Fls_DrvObj.typeoferase == FLS_SECTOR_ERASE)
			Fls_DrvObj.jobChunkSize = NOR_SECTOR_SIZE;
		else if(Fls_DrvObj.typeoferase == FLS_BLOCK_ERASE)
			Fls_DrvObj.jobChunkSize = NOR_BLOCK_SIZE;
		else
		    Fls_DrvObj.jobChunkSize = NOR_SIZE;

#if (STD_ON == FLS_USE_INTERRUPTS)
        uint32 intrStatus = 0u;
        intrStatus = HW_RD_REG32(FLS_QSPI_CTRL_BASE_ADDR+INTR_STATUS_RAW_SET);
        if((uint32)QSPI_INTR_STATUS_MASK == (uint32)(intrStatus & QSPI_INTR_STATUS_MASK))
        {
            Fls_QspiIntDisable((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                                    QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
                
            Fls_QspiIntClear((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                                    QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
        }
        else
        {

        }
        Fls_QspiIntEnable((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                           QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
#endif
    }

    return retVal;
};

/**
 *  \Function Name: Fls_Read
 *
 *  The function Fls_Read shall read from flash memory.
 *
 */

FUNC(Std_ReturnType, FLS_CODE) Fls_Read(
        Fls_AddressType SourceAddress,
        P2VAR(uint8, AUTOMATIC, FLS_APPL_DATA) TargetAddressPtr,
        Fls_LengthType Length)
{
    Std_ReturnType      retVal = (Std_ReturnType) E_OK;
    Fls_AddressType ReadStartAddress = SourceAddress + (FLS_BASE_ADDRESS & FLS_BASE_ADDRESS_REQ);

#if (STD_ON == FLS_DEV_ERROR_DETECT)
    if (MEMIF_UNINIT == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_READ, FLS_E_UNINIT);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if(MEMIF_BUSY == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_READ, FLS_E_BUSY);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if(NULL_PTR == TargetAddressPtr)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_READ, FLS_E_PARAM_DATA);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if (E_NOT_OK == checkValidAddress(ReadStartAddress))
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_READ, FLS_E_PARAM_ADDRESS);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if ((Length == 0U) || (E_NOT_OK == checkValidAddress(ReadStartAddress + Length - 1U)))
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_READ, FLS_E_PARAM_LENGTH);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else
#endif /* #if (STD_ON == FLS_DEV_ERROR_DETECT) */
    {
        Fls_DrvObj.status = MEMIF_BUSY;
        Fls_DrvObj.jobResultType = MEMIF_JOB_PENDING;
        Fls_DrvObj.jobType = FLS_JOB_READ;

        /* [SWS_Fls_00239] */
        Fls_DrvObj.flashAddr = SourceAddress + (FLS_BASE_ADDRESS & FLS_BASE_ADDRESS_REQ);
        Fls_DrvObj.ramAddr = (uint8*) TargetAddressPtr;
        Fls_DrvObj.length = Length;
        Fls_DrvObj.transferred = 0;

        Fls_DrvObj.jobChunkSize = Fls_DrvObj.maxReadNormalMode;
#if (STD_ON == FLS_USE_INTERRUPTS)
        uint32 intrStatus = 0u;
        intrStatus = HW_RD_REG32(FLS_QSPI_CTRL_BASE_ADDR+INTR_STATUS_RAW_SET);
        if((uint32)QSPI_INTR_STATUS_MASK == (uint32)(intrStatus & QSPI_INTR_STATUS_MASK))
        {
            Fls_QspiIntDisable((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                                    QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
                
            Fls_QspiIntClear((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                                    QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
        }
        else
        {

        }
        Fls_QspiIntEnable((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                           QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
#endif
    }

    return retVal;
}

/**
 *  \Function Name: Fls_Write
 *
 *  The function Fls_Write shall write one or more complete flash pages to the 
 *  flash device. 
 *
 */

FUNC(Std_ReturnType, FLS_CODE) Fls_Write(
        Fls_AddressType TargetAddress,
        P2VAR(const uint8, AUTOMATIC, FLS_APPL_DATA) SourceAddressPtr,
        Fls_LengthType Length)
{
    Std_ReturnType      retVal = (Std_ReturnType) E_OK;
    /* [SWS_Fls_00226] */
    Fls_AddressType writeStartAddress = TargetAddress + (FLS_BASE_ADDRESS & FLS_BASE_ADDRESS_REQ);

#if (STD_ON == FLS_DEV_ERROR_DETECT)
    if (MEMIF_UNINIT == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_WRITE, FLS_E_UNINIT);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if(MEMIF_BUSY == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_WRITE, FLS_E_BUSY);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if(NULL_PTR == SourceAddressPtr)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_WRITE, FLS_E_PARAM_DATA);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if ((E_NOT_OK == checkPageAlignment(writeStartAddress)) 
                                || (E_NOT_OK == checkValidAddress(writeStartAddress)))
	
    {
        /* [SWS_Fls_00026] */
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_WRITE, FLS_E_PARAM_ADDRESS);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if (((Length == 0U) || ((checkPageAlignment(writeStartAddress + Length))
                            || (E_NOT_OK == checkValidAddress(writeStartAddress + Length - 1U)))))
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_WRITE, FLS_E_PARAM_LENGTH);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else
#endif /* #if (STD_ON == FLS_DEV_ERROR_DETECT) */
    {
        Fls_DrvObj.jobResultType = MEMIF_JOB_PENDING;
        Fls_DrvObj.status = MEMIF_BUSY;
        Fls_DrvObj.jobType = FLS_JOB_WRITE;

        Fls_DrvObj.flashAddr = writeStartAddress;
        Fls_DrvObj.ramAddr = (uint8*)SourceAddressPtr;
        Fls_DrvObj.length = Length;
        Fls_DrvObj.transferred = 0;

        Fls_DrvObj.jobChunkSize = Fls_DrvObj.maxWriteNormalMode;
#if (STD_ON == FLS_USE_INTERRUPTS)
        uint32 intrStatus = 0u;
        intrStatus = HW_RD_REG32(FLS_QSPI_CTRL_BASE_ADDR+INTR_STATUS_RAW_SET);
        if((uint32)QSPI_INTR_STATUS_MASK == (uint32)(intrStatus & QSPI_INTR_STATUS_MASK))
        {
            Fls_QspiIntDisable((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                                    QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
                
            Fls_QspiIntClear((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                                    QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
        }
        else
        {

        }
        Fls_QspiIntEnable((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                           QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
#endif
    }
    return retVal;

};

/**
 *  \Function Name: Fls_Compare
 *
 *  The function Fls_Compare shall compare the contents of an area of flash 
 *  memory with that of an application data buffer. 
 *
 */
#if ( FLS_COMPARE_API == STD_ON )
FUNC(Std_ReturnType, FLS_CODE) Fls_Compare(
        Fls_AddressType SourceAddress,
        P2VAR(const uint8, AUTOMATIC, FLS_APPL_DATA) TargetAddressPtr,
        Fls_LengthType Length)
{
    Std_ReturnType      retVal = (Std_ReturnType) E_OK;
    Fls_AddressType compareStartAddress = SourceAddress + (FLS_BASE_ADDRESS & FLS_BASE_ADDRESS_REQ);

#if (STD_ON == FLS_DEV_ERROR_DETECT)
    if (MEMIF_UNINIT == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_COMPARE, FLS_E_UNINIT);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if(MEMIF_BUSY == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_COMPARE, FLS_E_BUSY);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if(NULL_PTR == TargetAddressPtr)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_COMPARE, FLS_E_PARAM_DATA);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if (E_NOT_OK == checkValidAddress(compareStartAddress))
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_COMPARE, FLS_E_PARAM_ADDRESS);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if ((Length == 0U) || (E_NOT_OK == checkValidAddress(compareStartAddress + Length - 1U)))
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_COMPARE, FLS_E_PARAM_LENGTH);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else
#endif /* #if (STD_ON == FLS_DEV_ERROR_DETECT) */
    {
        Fls_DrvObj.status = MEMIF_BUSY;
        Fls_DrvObj.jobResultType = MEMIF_JOB_PENDING;
        Fls_DrvObj.jobType = FLS_JOB_COMPARE;

        /* [SWS_Fls_00244] */
        Fls_DrvObj.flashAddr = SourceAddress + (FLS_BASE_ADDRESS & FLS_BASE_ADDRESS_REQ);
        Fls_DrvObj.ramAddr = (uint8*)TargetAddressPtr;
        Fls_DrvObj.length = Length;
        Fls_DrvObj.transferred = 0;

        Fls_DrvObj.jobChunkSize = Fls_DrvObj.maxReadNormalMode;
    }
    
#if (STD_ON == FLS_USE_INTERRUPTS)
        uint32 intrStatus = 0u;
        intrStatus = HW_RD_REG32(FLS_QSPI_CTRL_BASE_ADDR+INTR_STATUS_RAW_SET);
        if((uint32)QSPI_INTR_STATUS_MASK == (uint32)(intrStatus & QSPI_INTR_STATUS_MASK))
        {
            Fls_QspiIntDisable((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                                    QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
                
            Fls_QspiIntClear((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                                    QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
        }
        else
        {

        }
        Fls_QspiIntEnable((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                           QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
#endif
    return retVal;
}
#endif

/**
 *  \Function Name: Fls_BlankCheck
 *
 *  The function Fls_BlankCheck shall verify, whether a given memory area  
 *  has been erased. 
 *
 */
#if ( FLS_BLANK_CHECK_API == STD_ON )
FUNC(Std_ReturnType, FLS_CODE) Fls_BlankCheck(
        Fls_AddressType TargetAddress, Fls_LengthType Length)
{
    Std_ReturnType      retVal = (Std_ReturnType) E_OK;
    Fls_AddressType BlankCheckStartAddress = TargetAddress + (FLS_BASE_ADDRESS & FLS_BASE_ADDRESS_REQ);

#if (STD_ON == FLS_DEV_ERROR_DETECT)
    if (MEMIF_UNINIT == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_BLANK_CHECK, FLS_E_UNINIT);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if(MEMIF_BUSY == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_BLANK_CHECK, FLS_E_BUSY);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if (E_NOT_OK == checkValidAddress(BlankCheckStartAddress))                                  
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_BLANK_CHECK, FLS_E_PARAM_ADDRESS);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else if ((Length == 0U) || 
             (E_NOT_OK == checkValidAddress(BlankCheckStartAddress + Length - 1U)))
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_BLANK_CHECK, FLS_E_PARAM_LENGTH);
        retVal = (Std_ReturnType) E_NOT_OK;
    }
    else
#endif /* #if (STD_ON == FLS_DEV_ERROR_DETECT) */
    {
        Fls_DrvObj.status = MEMIF_BUSY;
        Fls_DrvObj.jobResultType = MEMIF_JOB_PENDING;
        Fls_DrvObj.jobType = FLS_JOB_BLANKCHECK;

        /* [SWS_Fls_00379] */
        Fls_DrvObj.flashAddr = TargetAddress + (FLS_BASE_ADDRESS & FLS_BASE_ADDRESS_REQ);
        Fls_DrvObj.ramAddr = NULL;
        Fls_DrvObj.length = Length;
        Fls_DrvObj.transferred = 0;

        Fls_DrvObj.jobChunkSize = Fls_DrvObj.maxReadNormalMode;
#if (STD_ON == FLS_USE_INTERRUPTS)
        uint32 intrStatus = 0u;
        intrStatus = HW_RD_REG32(FLS_QSPI_CTRL_BASE_ADDR+INTR_STATUS_RAW_SET);
        if((uint32)QSPI_INTR_STATUS_MASK == (uint32)(intrStatus & QSPI_INTR_STATUS_MASK))
        {
            Fls_QspiIntDisable((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                                    QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
                
            Fls_QspiIntClear((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                                    QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
        }
        else
        {

        }
        Fls_QspiIntEnable((QSPI_INTR_ENABLE_SET_REG_FIRQ_ENA_SET_MASK |
                           QSPI_INTR_ENABLE_SET_REG_WIRQ_ENA_SET_MASK));
#endif
    }
    return retVal;
}
#endif

/**
 *  \Function Name: Fls_GetStatus
 *
 *  The function Fls_GetStatus shall return the FLS module state synchronously
 *
 */
#if ( STD_ON == FLS_GET_STATUS_API)
FUNC(MemIf_StatusType, FLS_CODE)
        Fls_GetStatus( void )
{
    return Fls_DrvObj.status;
}
#endif

/**
 *  \Function Name: Fls_GetJobResult
 *
 *  The function Fls_GetJobResult shall return the result of the
 *	last job synchronously
 *
 */
#if ( FLS_GET_JOB_RESULT_API == STD_ON )
FUNC(MemIf_JobResultType, FLS_CODE)
        Fls_GetJobResult( void )
{
    MemIf_JobResultType jobResult = MEMIF_JOB_FAILED;
#if (STD_ON == FLS_DEV_ERROR_DETECT)
    if (MEMIF_UNINIT == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_GET_JOB_RESULT, FLS_E_UNINIT);
    }
    else
#endif /* #if (STD_ON == FLS_DEV_ERROR_DETECT) */
    {
        jobResult = Fls_DrvObj.jobResultType;
    }
    return jobResult;
}
#endif

/**
 *  \Function Name: Fls_GetVersionInfo
 *
 *  The function Returns the version information of this module.
 *
 */
#if (STD_ON == FLS_VERSION_INFO_API)
FUNC(Std_ReturnType, FLS_CODE) Fls_GetVersionInfo(
        P2VAR(Std_VersionInfoType, AUTOMATIC, FLS_APPL_DATA) versioninfo)
{
    Std_ReturnType      retVal = (Std_ReturnType) E_OK;

#if (STD_ON == FLS_DEV_ERROR_DETECT)
    if (NULL_PTR == versioninfo)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_GET_VERSION_INFO, FLS_E_PARAM_POINTER);
        retVal = E_NOT_OK;
    }
    else
#endif  /* #if (STD_ON == FLS_DEV_ERROR_DETECT) */
    {
        versioninfo->vendorID         = FLS_VENDOR_ID;
        versioninfo->moduleID         = FLS_MODULE_ID;
        versioninfo->sw_major_version = (uint8) FLS_SW_MAJOR_VERSION;
        versioninfo->sw_minor_version = (uint8) FLS_SW_MINOR_VERSION;
        versioninfo->sw_patch_version = (uint8) FLS_SW_PATCH_VERSION;
    }
    return retVal;
}
#endif  /* #if (STD_ON == FLS_VERSION_INFO_API) */

/**
 *  \Function Name: Fls_MainFunction
 *
 *  The function Fls_MainFunction shall perform the processing
 *	of the flash read, write, erase and compare jobs
 *
 */
FUNC(void, FLS_CODE) Fls_MainFunction(void)
{
  #if (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON) 
    uint32     elapsedTime_usec = 0U;
    TickType startCount = 0U,tempCount = 0U, elapsedCount = 0U; 
    StatusType status; 
  #endif 

#if (STD_ON == FLS_DEV_ERROR_DETECT)
    if (MEMIF_UNINIT == Fls_DrvObj.status)
    {
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_UNINIT);
    }
    else
#endif  /* #if (STD_ON == FLS_DEV_ERROR_DETECT) */
    {
        if ( Fls_DrvObj.jobResultType == MEMIF_JOB_PENDING)
        {
            switch (Fls_DrvObj.jobType)
            {
                case FLS_JOB_COMPARE:
                      processJobs(FLS_JOB_COMPARE);
                      break;

                case FLS_JOB_ERASE:
                  #if (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON) 
                    status = GetCounterValue(FLS_OS_COUNTER_ID, &startCount); 
                  #endif 

                    processJobs(FLS_JOB_ERASE); 

                  #if (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON) 
                      if (E_OK == status)
                      {
                        /* Below API can change start time, so use temp variable */
                        tempCount = startCount;
                        status = GetElapsedValue(FLS_OS_COUNTER_ID, &tempCount, &elapsedCount);

                        if(Fls_DrvObj.typeoferase == FLS_SECTOR_ERASE)
                        {
                            if ((E_OK == status) && (elapsedCount > (MAX_SECTOR_ERASETIME_CONV_IN_USEC * (uint32)FLS_MAX_ERASE_TIME)))
                            {
                                Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                                if( Fls_DrvObj.Fls_JobErrorNotification != NULL_PTR )
                                {
                                    Fls_DrvObj.Fls_JobErrorNotification();
                                }
                                else{
                                    /* Do Nothing */
                                }
                            }
                            else
                            {
                              /* Do Nothing */
                            }
                        }
                         else if(Fls_DrvObj.typeoferase == FLS_BLOCK_ERASE)
                         {
                            if ((E_OK == status) && (elapsedCount > (MAX_BLOCK_ERASETIME_CONV_IN_USEC * (uint32)FLS_MAX_ERASE_TIME)))
                            {
                                Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                                if( Fls_DrvObj.Fls_JobErrorNotification != NULL_PTR )
                                {
                                    Fls_DrvObj.Fls_JobErrorNotification();
                                }
                                else{
                                    /* Do Nothing */
                                }
                            }
                            else
                            {
                              /* Do Nothing */
                            }
                         }
                          else 
                         {
                            if ((E_OK == status) && (elapsedCount > (MAX_CHIP_ERASETIME_CONV_IN_USEC * (uint32)FLS_MAX_ERASE_TIME)))
                            {
                                Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                                if( Fls_DrvObj.Fls_JobErrorNotification != NULL_PTR )
                                {
                                    Fls_DrvObj.Fls_JobErrorNotification();
                                }
                                else{
                                    /* Do Nothing */
                                }
                            }
                            else
                            {
                                /* Do Nothing */
                            }
                          }
                    }
                    else
                    {
                      /* Do Nothing */
                    }
                  #endif 
                    break;
                case FLS_JOB_READ:
                 #if (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON) 
                    status = GetCounterValue(FLS_OS_COUNTER_ID, &startCount); 
                  #endif 

                    processJobs(FLS_JOB_READ);

                    #if (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON) 
                    if (E_OK == status)
                    {
                        /* Below API can change start time, so use temp variable */
                        tempCount = startCount;
                        status = GetElapsedValue(FLS_OS_COUNTER_ID, &tempCount, &elapsedCount);
                        if(Fls_DrvObj.typeoferase == FLS_SECTOR_ERASE)
                        {
                            if ((E_OK == status) && (elapsedCount > (MAX_SECTOR_READWRITETIME_CONV_IN_USEC * (uint32)FLS_MAX_WRITE_TIME)))
                            {
                                Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                                if( Fls_DrvObj.Fls_JobErrorNotification != NULL_PTR )
                                {
                                    Fls_DrvObj.Fls_JobErrorNotification();
                                }
                                else{
                                    /* Do Nothing */
                                }
                            }
                            else
                            {
                              /* Do Nothing */
                            }
                        }
                         else if(Fls_DrvObj.typeoferase == FLS_BLOCK_ERASE)
                         {
                            if ((E_OK == status) && (elapsedCount > (MAX_BLOCK_READWRITETIME_CONV_IN_USEC * (uint32)FLS_MAX_WRITE_TIME)))
                            {
                                Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                                if( Fls_DrvObj.Fls_JobErrorNotification != NULL_PTR )
                                {
                                    Fls_DrvObj.Fls_JobErrorNotification();
                                }
                                else{
                                    /* Do Nothing */
                                }
                            }
                            else
                            {
                              /* Do Nothing */
                            }
                         }
                          else 
                         {
                            if ((E_OK == status) && (elapsedCount > (MAX_CHIP_READWRITETIME_CONV_IN_USEC * (uint32)FLS_MAX_WRITE_TIME)))
                            {
                                Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                                if( Fls_DrvObj.Fls_JobErrorNotification != NULL_PTR )
                                {
                                    Fls_DrvObj.Fls_JobErrorNotification();
                                }
                                else{
                                    /* Do Nothing */
                                }
                            }
                         }
                    }
                    else{
                    /* Do Nothing */
                    }
                  #endif 
                    break;
                case FLS_JOB_WRITE:
                  #if (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON) 
                    status = GetCounterValue(FLS_OS_COUNTER_ID, &startCount); 
                  #endif 

                    processJobs(FLS_JOB_WRITE);
                    
                  #if (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON) 
                    if (E_OK == status)
                    {
                        /* Below API can change start time, so use temp variable */
                        tempCount = startCount;
                        status = GetElapsedValue(FLS_OS_COUNTER_ID, &tempCount, &elapsedCount);
                        if(Fls_DrvObj.typeoferase == FLS_SECTOR_ERASE)
                        {
                            if ((E_OK == status) && (elapsedCount > (MAX_SECTOR_READWRITETIME_CONV_IN_USEC * (uint32)FLS_MAX_WRITE_TIME)))
                            {
                                Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                                if( Fls_DrvObj.Fls_JobErrorNotification != NULL_PTR )
                                {
                                    Fls_DrvObj.Fls_JobErrorNotification();
                                }
                                else{
                                    /* Do Nothing */
                                }
                            }
                            else
                            {
                              /* Do Nothing */
                            }
                        }
                         else if(Fls_DrvObj.typeoferase == FLS_BLOCK_ERASE)
                         {
                            if ((E_OK == status) && (elapsedCount > (MAX_BLOCK_READWRITETIME_CONV_IN_USEC * (uint32)FLS_MAX_WRITE_TIME)))
                            {
                                Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                                if( Fls_DrvObj.Fls_JobErrorNotification != NULL_PTR )
                                {
                                    Fls_DrvObj.Fls_JobErrorNotification();
                                }
                                else{
                                    /* Do Nothing */
                                }
                            }
                            else
                            {
                              /* Do Nothing */
                            }
                         }
                          else 
                         {
                            if ((E_OK == status) && (elapsedCount > (MAX_CHIP_READWRITETIME_CONV_IN_USEC * (uint32)FLS_MAX_WRITE_TIME)))
                            {
                                Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                                if( Fls_DrvObj.Fls_JobErrorNotification != NULL_PTR )
                                {
                                    Fls_DrvObj.Fls_JobErrorNotification();
                                }
                                else{
                                    /* Do Nothing */
                                }
                            }
                         }
                    }
                    else{
                      /* Do Nothing */
                    }
                  #endif 
                    break;
                case FLS_JOB_BLANKCHECK:
                    processJobs(FLS_JOB_BLANKCHECK);
                    break;
                case FLS_JOB_NONE:
                    break;
                default:
                    break;
            } /* switch */
        }   /* if */
        else if ( Fls_DrvObj.jobResultType == MEMIF_JOB_CANCELED) 
        { 
          if( Fls_DrvObj.Fls_JobErrorNotification != NULL )
            {
              Fls_DrvObj.Fls_JobErrorNotification();
            }
        } 
        else
        {
          if(Fls_DrvObj.Fls_JobEndNotification != NULL_PTR) 
          { 
           /* If not pending job, then return end notification */
           Fls_DrvObj.Fls_JobEndNotification();
          }
        }
    }
}

/**
 *  \Function Name: Fls_Cancel
 *
 *  The function Fls_Cancel shall cancel an ongoing flash read,
 *  write, erase or compare job synchronously so that directly 
 *  after returning from this function a new job can be started
 *
 *  Design : MCAL-7264,MCAL-7270,MCAL-7274,MCAL-7458,MCAL-7299,
 *           MCAL-7319,MCAL-7389,MCAL-7351,MCAL-7349,MCAL-7412,
 *           MCAL-7260,MCAL-7326,MCAL-7346,MCAL-7296,
 */
#if ( FLS_CANCEL_API == STD_ON )
FUNC(void, FLS_CODE) Fls_Cancel(void)
{
	#if (STD_ON == FLS_DEV_ERROR_DETECT)
	if (MEMIF_UNINIT == Fls_DrvObj.status)
	{
		(void) Det_ReportError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_CANCEL, FLS_E_UNINIT);
	}
	else
	#endif /* #if (STD_ON == FLS_DEV_ERROR_DETECT) */
	{
		/* Reset internal job processing variables (like address, length and data pointer) */
		Fls_DrvObj.flashAddr = 0U;
		Fls_DrvObj.ramAddr = NULL;
		Fls_DrvObj.length = 0U;
		Fls_DrvObj.transferred = 0;


		Fls_DrvObj.status = MEMIF_IDLE;
		Fls_DrvObj.jobType = FLS_JOB_NONE;
        if (Fls_DrvObj.jobResultType == MEMIF_JOB_PENDING)
        {
                Fls_DrvObj.jobResultType = MEMIF_JOB_CANCELED;
        }
		if( Fls_DrvObj.Fls_JobErrorNotification != NULL )
			{
				Fls_DrvObj.Fls_JobErrorNotification();
			}
	}

};
#endif

/**
 *  \Function Name: Fls_SetEraseType
 *
 *  The function Fls_SetEraseType sets the parameters like data size, 
 *  data_value, erase type(sector/block/chip),typeoferase for Fls_DrvObj, 
 *  which are required in application, based on the erase type parameter 
 *  passed to this function while calling from Application.
 *  This function is called from application based on the macros enabled in 
 *  application itself.
 *  Implementation of this function is done to allow user to select one or more 
 *  erase types based on the requirement.
 */
FUNC(uint32, FLS_CODE) Fls_SetEraseType(Fls_EraseType erasetype)
{
   switch(erasetype) {
		case FLS_SECTOR_ERASE:
                Data_Size_Test = NOR_SECTOR_SIZE;
                Data_Value = 0xAA;
                Fls_DrvObj.typeoferase = FLS_SECTOR_ERASE;
                sector_or_blocksize = NOR_SECTOR_SIZE;
            break;
		case FLS_BLOCK_ERASE:
                Data_Size_Test = NOR_BLOCK_SIZE;
                Data_Value = 0xBB;
                Fls_DrvObj.typeoferase = FLS_BLOCK_ERASE;
                sector_or_blocksize = NOR_BLOCK_SIZE;
            break;
        case FLS_CHIP_ERASE:
                Data_Size_Test = NOR_BLOCK_SIZE;
                Data_Value = 0xCC;
                Fls_DrvObj.typeoferase = FLS_CHIP_ERASE;
                sector_or_blocksize = NOR_BLOCK_SIZE;
        break;
        default:
                Data_Size_Test = NOR_SECTOR_SIZE;
                Data_Value = 0xAA;
                Fls_DrvObj.typeoferase = FLS_SECTOR_ERASE;
                sector_or_blocksize = NOR_SECTOR_SIZE;
            break;
    }
   return sector_or_blocksize;
}


#define FLS_STOP_SEC_CODE
#include "Fls_MemMap.h"
