/* ======================================================================
 *   Copyright (C) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     Gpt.h
 *
 *  \brief    This file contains interface header for GPT MCAL driver
 *
 */

/**
 *  \defgroup MCAL_GPT_API GPT Handler and Driver API
 *
 *  The GPT Driver implements a standardized interface specified in the
 *  AUTOSAR_SWS_GPTDriver document.<br>
 *
 *
 *  @sa MCAL_GPT_CFG
 *  @{
 */

/*  -------------------------------------------------------------------------------------------------------------------
*  FILE DESCRIPTION
*  -------------------------------------------------------------------------------------------------------------------
*         File:  Gpt.h
*    Component:  AM263x MCAL Driver
*       Module:  GptDriver
*    Generator:  None
*
*  Description: This component provides services for initialization and control of the microcontroller internal
*                GPT unit (General Purpose Timer).
*********************************************************************************************************************/

/*
 * Below are the global requirements which are met by this GPT
 * driver which can't be mapped to a particular source ID or maps
 * to multiple source.
 */

/* Design : GPT_DesignId_001, GPT_DesignId_002, GPT_DesignId_003,
 *          GPT_DesignId_018, GPT_DesignId_019 */
/*
 * Requirements : SWS_Gpt_00345, SWS_Gpt_00346, SWS_Gpt_00387, SWS_Gpt_00347,
 *                SWS_Gpt_00404, SWS_Gpt_00348, SWS_Gpt_00349, SWS_Gpt_00350,
 *                SWS_Gpt_00351, SWS_Gpt_00388, SWS_Gpt_00332, SWS_Gpt_00278,
 *                SWS_Gpt_00257, SWS_Gpt_00337, SWS_Gpt_00380, AUTORADAR_MCAL-8
 */


/*
 * Below are the GPT module environment requirements which can't be mapped
 * to this driver
 */
/*
 * Requirements : SWS_Gpt_00353, SWS_Gpt_00354, SWS_Gpt_00380
 */

#ifndef GPT_H_
#define GPT_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

/*LDRA_NOANALYSIS*/
#include "Std_Types.h"
/*LDRA_ANALYSIS*/
#include "Gpt_Cfg.h"
#if (GPT_WAKEUP_FUNCTIONALITY_API == STD_ON)
/* include for Wakeup handling */
    #include "EcuM_Cbk.h"
#endif

#ifdef __cplusplus
extern "C"
{
#endif


/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/**
 *  \name GPT Driver Module SW Version Info
 *
 *  Defines for GPT Driver version used for compatibility checks
 *  @{
 */
/** \brief Driver Implementation Major Version */
#define GPT_SW_MAJOR_VERSION            (8U)
/** \brief Driver Implementation Minor Version */
#define GPT_SW_MINOR_VERSION            (6U)
/** \brief Driver Implementation Patch Version */
#define GPT_SW_PATCH_VERSION            (2U)
/* @} */

/**
 *  \name GPT Driver Module AUTOSAR Version Info
 *
 *  Defines for GPT Driver AUTOSAR version used for compatibility checks
 *  @{
 */
/** \brief AUTOSAR Major version specification implemented by GPT Driver */
#define GPT_AR_RELEASE_MAJOR_VERSION    (4U)
/** \brief AUTOSAR Minor  version specification implemented by GPT Driver */
#define GPT_AR_RELEASE_MINOR_VERSION    (3U)
/** \brief AUTOSAR Patch version specification implemented by GPT Driver */
#define GPT_AR_RELEASE_REVISION_VERSION (1U)
/* @} */


/**
 *  \name GPT Driver ID Info
 *  @{
 */
/** \brief Texas Instruments Vendor ID */
#define GPT_VENDOR_ID                   ((uint16) 44U)
/** \brief GPT Driver Module ID */
#define GPT_MODULE_ID                   ((uint16) 100U)
/** \brief GPT Driver Instance ID */
#define GPT_INSTANCE_ID                 ((uint8) 0U)
/* @} */

/*
 * GPT channel notification/wakeup function index.
 * This is used store function index into ISR array.
 */
/** \brief GPT continuous mode index */
#define GPT_CH_ISR_MODE_CONTINUOUS     (0U)
/** \brief GPT oneshot mode index */
#define GPT_CH_ISR_MODE_ONESHOT        (1U)
/** \brief GPT continuous wakeup mode index */
#define GPT_CH_ISR_MODE_CONT_WAKEUP    (2U)
/** \brief GPT oneshot wakeup mode index */
#define GPT_CH_ISR_MODE_ONESHOT_WAKEUP (3U)


/** \brief GP timer channels available for AM263x */
#define GPT_RTI_CH_MIN              (0U)
#define GPT_RTI_CH_MAX              (15U)

#define GPT_RTI_MAX              	(16U)

/**< \brief Driver status UN INITIALIZED */
#define GPT_DRIVER_UNINITIALIZED    ((uint32) 0U)
/**< \brief Driver status UN INITIALIZED */
#define GPT_DRIVER_INITIALIZED      ((uint32) 1U)
/*
 * GPT channels available for user
 */
/** \brief Module 1 */
#define GPT_RTI_CHANNEL1            (0U)
#define GPT_RTI_CHANNEL2            (1U)
#define GPT_RTI_CHANNEL3            (2U)
#define GPT_RTI_CHANNEL4            (3U)
/** \brief Module 2 */
#define GPT_RTI_CHANNEL5            (4U)
#define GPT_RTI_CHANNEL6            (5U)
#define GPT_RTI_CHANNEL7            (6U)
#define GPT_RTI_CHANNEL8            (7U)
/** \brief Module 3 */
#define GPT_RTI_CHANNEL9            (8U)
#define GPT_RTI_CHANNEL10           (9U)
#define GPT_RTI_CHANNEL11           (10U)
#define GPT_RTI_CHANNEL12           (11U)
/** \brief Module 4 */
#define GPT_RTI_CHANNEL13            (12U)
#define GPT_RTI_CHANNEL14           (13U)
#define GPT_RTI_CHANNEL15           (14U)
#define GPT_RTI_CHANNEL16           (15U)

/* Masks the higher order bits to evaluate compare block (lsb0 and lsb1) */
#define CH_COMP_MASK                ((uint32)0x00000003)

/*
 * GPT counters available for user
 */
/** \brief Counter Block 0 */
#define GPT_RTI_COUNTER0            (0U)
/** \brief Counter Block 1 */
#define GPT_RTI_COUNTER1            (1U)
/** \brief Number of Counter Blocks per RTI Module */
#define GPT_RTI_COUNTERS            (2U)

/*
 * GPT RTI modules available for user
 */
/** \brief GP timer A instance */
#define GPT_RTI_MODULE1             (0U)
/** \brief GP timer B instance */
#define GPT_RTI_MODULE2             (1U)
/** \brief GP timer C instance */
#define GPT_RTI_MODULE3             (2U)
/** \brief GP timer D instance */
#define GPT_RTI_MODULE4             (3U)
/** \brief Number of GPT Modules */
#define GPT_RTI_MODULES             (4U)

/**
 *  \name GPT Error Codes
 *
 *  Error codes returned by GPT functions
 *  @{
 */
#ifndef GPT_E_UNINIT
/** \brief API service used without module initialization */
#define GPT_E_UNINIT                (0x0AU)
#endif
#ifndef GPT_E_BUSY
/**
 *  \brief API service called during ongoing process (Timer is already
 *  running)
 */
#define GPT_E_BUSY                  (0x0BU)
#endif
#ifndef GPT_E_MODE
/** \brief API service called when driver is in wrong mode */
#define GPT_E_MODE                  (0x0CU)
#endif
#ifndef GPT_E_ALREADY_INITIALIZED
/** \brief API Gpt_Init is called but the module is already initialized. */
#define GPT_E_ALREADY_INITIALIZED   (0x0DU)
#endif
#ifndef GPT_E_INIT_FAILED
/** \brief API Gpt_Init is called but the module is already initialized. */
#define GPT_E_INIT_FAILED   (0x0EU)
#endif
#ifndef GPT_E_PARAM_CHANNEL
/** \brief API called with invalid channel. */
#define GPT_E_PARAM_CHANNEL         (0x14U)
#endif
#ifndef GPT_E_PARAM_VALUE
/** \brief API servcie called with invalid start timer parameter */
#define GPT_E_PARAM_VALUE           (0x15U)
#endif
#ifndef GPT_E_PARAM_POINTER
/** \brief API servcie called with invalid(Null_Ptr) data buffer pointer. */
#define GPT_E_PARAM_POINTER         (0x16U)
#endif
#ifndef GPT_E_PARAM_PREDEF_TIMER
/** \brief API service called with invalid Predef Timer. */
#define GPT_E_PARAM_PREDEF_TIMER   (0x17U)
#endif
#ifndef GPT_E_PARAM_MODE
/** \brief Gpt_SetMode is called with wrong mode parameter. */
#define GPT_E_PARAM_MODE            (0x1FU)
#endif
#ifndef GPT_E_INVALID_ISR
/** \brief Invalid entry to ISR. */
#define GPT_E_INVALID_ISR            (0x80U)
#endif
/* @} */


/**
 *  \name GPT Service Ids
 *
 *  The Service Id is one of the argument to Det_ReportError function and
 *  is used to identify the source of the error
 *  @{
 */
/** \brief Gpt_GetVersionInfo() API Service ID */
#define GPT_SID_GET_VERSION_INFO    (0x00U)
/** \brief Gpt_Init() API Service ID */
#define GPT_SID_INIT                (0x01U)
/** \brief Gpt_DeInit() API Service ID */
#define GPT_SID_DEINIT              (0x02U)
/** \brief Gpt_GetTimeElapsed() API Service ID */
#define GPT_SID_GET_TIME_ELAPSED    (0x03U)
/** \brief Gpt_GetTimeRemaining() API Service ID */
#define GPT_SID_GET_TIME_REMAINING  (0x04U)
/** \brief Gpt_StartTimer() API Service ID */
#define GPT_SID_START_TIMER         (0x05U)
/** \brief Gpt_StopTimer() API Service ID */
#define GPT_SID_STOP_TIMER          (0x06U)
/** \brief Gpt_EnableNotification() API Service ID */
#define GPT_SID_ENABLE_NOTIFY       (0x07U)
/** \brief Gpt_DisableNotification() API Service ID */
#define GPT_SID_DISABLE_NOTIFY      (0x08U)
/** \brief Gpt_SetMode() API Service ID */
#define GPT_SID_SET_MODE            (0x09U)
/** \brief Gpt_DisableWakeup() API Service ID */
#define GPT_SID_DISABLE_WAKEUP      (0x0AU)
/** \brief Gpt_EnableWakeup() API Service ID */
#define GPT_SID_ENABLE_WAKEUP       (0x0BU)
/** \brief Gpt_CheckWakeup() API Service ID */
#define GPT_SID_CHECK_WAKEUP        (0x0CU)
/** \brief Gpt_GetPredefTimerValue() API Service ID */
#define GPT_SID_GET_PREDEFTIMERVALUE    (0x0DU)
/** \brief GPT Get Hw Unit Obj API Service ID */
#define GPT_SID_GET_GETHWUNITOBJ        (0x0EU)
/* @} */

/** \brief Maximum resolution for the timer */
#define MAX_RESOLUTION    0xFFFFFFFFU 



/* Klocwork Inspected
 * MISRA C 2012 Dir 4.9
   Reason: Macro cannot be avoided here as sending type as an argument
 */

#define SET(type, var, bit)   (var |= (type)(((uint32)1U) << bit))
#define CLEAR(type, var, bit) (var &= (~((type)(((uint32)1U) << bit))))

#if (STD_ON == GPT_REGISTER_READBACK_API)
/**
 *  \brief GPT register readback structure
 * Requirements : AUTORADAR_MCAL-53
 */
/*Type for Register readback of configuration registers*/
typedef struct GptConfigRegTag
{
    uint32 GPT_GCTRL;
    /**<RTI Global Control Register*/
    uint32 GPT_TBCTRL;
    /**<RTI Timebas Control Register*/
    uint32 GPT_COMPCTRL;
    /**<RTI Compare Control Register*/
    uint32 GPT_UDCP0;
    /**<RTI Update Compare 0 Register*/
    uint32 GPT_UDCP1;
    /**<RTI Update Compare 1 Register*/
    uint32 GPT_UDCP2;
    /**<RTI Update Compare 2 Register*/
    uint32 GPT_UDCP3;
    /**<RTI Update Compare 3 Register*/
} GptConfigReg;

#endif  /* #if (STD_ON == GPT_REGISTER_READBACK_API) */

/*********************************************************************************************************************
 * Exported Preprocessor #define Macros
 *********************************************************************************************************************/

/*********************************************************************************************************************
 * Exported Type Declarations
 *********************************************************************************************************************/

/* Requirements: SWS_Gpt_00358, ECUC_Gpt_00308 */
/** \brief Type describing the Gpt channel */
typedef uint32 Gpt_ChannelType;

/** \brief  Type describing the timeout value  */
/* Requirements: SWS_Gpt_00359 */
typedef uint32 Gpt_ValueType;

/** \brief  Type describing the counter block to use for GPT */
typedef uint32 Gpt_CntId;

/** \brief  List of process modes  */
/* Requirements: SWS_Gpt_00360 */
typedef enum
{
    GPT_MODE_UNITIALIZED = 0U,
    /**< GPT initialized */
    GPT_MODE_NORMAL,
    /**< Normal operation mode of the GPT */
    GPT_MODE_SLEEP
    /**< Operation for reduced power operation mode. In sleep mode only wakeup
     *   capable channels are available.*/
} Gpt_ModeType;


/** \brief Type for GPT Predefined(free running) Timers */
/* Requirements: SWS_Gpt_00389, ECUC_Gpt_00334*/
typedef enum
{
    GPT_PREDEF_UNDEFINED

}Gpt_PredefTimerType;

/** \brief  Notification callback function pointer  */
typedef void (*Gpt_NotifyType)(void);

/** \brief Type for specifying source clock selection  */
typedef uint32 Gpt_ClockSourceType;

/** \brief  GPT channel mode macros.  */
/* Requirements: ECUC_Gpt_00309 */
typedef enum
{
    GPT_CH_MODE_CONTINUOUS = 0U,
    /**< GPT initialized */
    GPT_CH_MODE_ONESHOT,
    /**< Normal operation mode of the GPT */
    GPT_CH_MODE_CONT_WAKEUP,
    /**< Normal operation mode of the GPT */
    GPT_CH_MODE_ONESHOT_WAKEUP
    /**< Normal operation mode of the GPT */

} Gpt_ChannelMode;

/**
 *  \brief  List of Prescale values for General purpose timer channels
 *          The timer counter is prescaled with the value 2^(PTV+1).
 *          Example: PTV = 3, counter increases value (if started) after 16
 *          functional clock periods
 *  Note: Not applicable to GPT Predef timers.
 */
typedef enum
{
    GPT_PRESCALE_UNDEFINED
} Gpt_PrescaleValueType;


/**
 * \brief  GPT channel state
 *         GPT will be in one of this state during its lifetime.
 */
typedef enum
{
    GPT_UNINITIALIZED = 0U,
    /**< GPT is uninitialized */
    GPT_INITIALIZED,
    /**< GPT is initialized */
    GPT_RUNNING,
    /**< GPT is running */
    GPT_STOPPED,
    /**< GPT is stopped */
    GPT_EXPIRED
    /**< GPT is expired */
} Gpt_ChannelStateType;

/** \brief  Configuration per channel */
/* Requirements : ECUC_Gpt_00308, ECUC_Gpt_00309, ECUC_Gpt_00331, ECUC_Gpt_00332
 *                ECUC_Gpt_00311, ECUC_Gpt_00312, ECUC_Gpt_00333, ECUC_Gpt_00313
 */
typedef struct
{
    Gpt_ChannelType     ChannelId;
    /**< Channel Id */
    Gpt_ChannelMode     ChannelMode;
    /**< Channel mode */
    /*TI_INSPECTED 217 S : MISRAC_2012_R.4.5
    * "Reason - the variable is used in the GPT context not an issue. " */
    Gpt_CntId            CounterBlk;
    /**< GPT Counter Block */
    uint32              GptChannelTickFrequency;
    /**< Specifies the tick frequency of the timer channel in Hz */
    uint32              tickValueMax;
    /**< Maximum value in ticks, the timer channel is able to count. With the
     *   next tick, the timer rolls over to zero
     *   TODO: ECUC_Gpt_00332 how to implement this with current HW? */
    uint8               GptEnableWakeup;
    /**< Enables wakeup capability of MCU for a channel */
    Gpt_NotifyType      Notification_pt;
    /**< Function pointer to callback function (for non-wakeup notification) */
    Gpt_ClockSourceType GptChannelClksrcRef;
    /**< Reference to the GptClockReferencePoint from which the channel
     *   clock is derived */
    uint8               GptWakeupSourceRef;
    /**< Channel wakeup source, in case the wakeup-capability is true this value
     *   is transmitted to the EcuState Manager */
} Gpt_ChannelConfigType;

/** \brief  global configuration of the driver  */
/* Requirements : SWS_Gpt_00357 */
typedef struct Gpt_ConfigType_s
{
    P2CONST(Gpt_ChannelConfigType, AUTOMATIC, GPT_CONST) ChannelConfig_pt;
    /**<  Pointer to Channel list */
    uint32 ChannelCount;
    /**<  Number of channels configured  */
} Gpt_ConfigType;

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

#if (STD_ON == GPT_VERSION_INFO_API)
/**
 *  \brief This service returns the version information of this module.
 *
 *  \verbatim
 *  Service name        : Gpt_GetVersionInfo
 *  Syntax              : void Gpt_GetVersionInfo(Std_VersionInfoType*
 *                                                          VersionInfoPtr)
 *  Service ID[hex]     : 0x00
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : None
 *  Parameters (inout)  : None
 *  Parameters (out)    : VersionInfoPtr - Pointer to where to store the version
 *                        information of this module
 *  Return value        : None
 *  Description         : Returns the version information of this module
 *  \endverbatim
 */
FUNC(void,
     GPT_CODE) Gpt_GetVersionInfo(P2VAR(Std_VersionInfoType, AUTOMATIC,
                                        GPT_APPL_DATA) VersionInfoPtr);
#endif /*(STD_ON == GPT_VERSION_INFO_API)*/

/**
 *  \brief This service initializes all the configured Gpt channels. This will
 *         set the state of the each channel to"initialized", also set the Gpt
 *         driver state to "Normal".
 *
 *  \verbatim
 *  Service name        : Gpt_Init
 *  Syntax              : void Gpt_Init(const Gpt_ConfigType* ConfigPtr)
 *  Service ID[hex]     : 0x01
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : Gpt_ConfigType* ConfigPtr - configuration structure
 *                        for initializing the module.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Initializes the GPT driver.
 *  \endverbatim
 */
FUNC(void, GPT_CODE) Gpt_Init(P2CONST(Gpt_ConfigType, 
            AUTOMATIC,GPT_CONST) configPtr);

#if (STD_ON == GPT_DEINIT_API)
/**
 *  \brief This service deinitializes the RTI used by Gpt driver to the power on
 *         reset state.
 *         The Gpt driver state is changed to "Uninitialized" state".
 *         All the channel registers are cleared to stop the timer channels.
 *         API will disable all interrupt notifications, wakeup interrupts.
 *
 *  \verbatim
 *  Service name        : Gpt_DeInit
 *  Syntax              : void Gpt_DeInit(void)
 *  Service ID[hex]     : 0x02
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : None
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Deinitializes the GPT driver.
 *  \endverbatim
 */
FUNC(void, GPT_CODE) Gpt_DeInit(void);
#endif /*(STD_ON == GPT_DEINIT_API)*/

#if (STD_ON == GPT_TIME_ELAPSED_API)
/**
 *  \brief Gpt_GetTimeElapsed will return the time elapsed for channel which is
 *         referenced. The user can configure the channel in two modes, One-shot
 *         and Continuous mode. In one shot mode, if the timer is in stopped
 *         state, the function will return time value at the moment of stopping.
 *         If the timer is expired, the function will return the target time
 *         configured for the channel. In Continuous Mode - The elapsed time
 *         value will be the value relative to last occurrence.
 *
 *  \verbatim
 *  Service name        : Gpt_GetTimeElapsed
 *  Syntax              : Gpt_ValueType Gpt_GetTimeElapsed(Gpt_ChannelType
 *                                                                      Channel)
 *  Service ID[hex]     : 0x03
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Channel - ChannelId of the Gpt channel.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : Gpt_ValueType Elapsed timer value (in number of ticks)
 *  Description         : Returns the time already elapsed.
 *  \endverbatim
 */
FUNC(Gpt_ValueType, GPT_CODE) Gpt_GetTimeElapsed(Gpt_ChannelType channel );
#endif /*(STD_ON == GPT_TIME_ELAPSED_API)*/

#if (STD_ON == GPT_TIME_REMAINING_API)
/**
 *  \brief Gpt_GetTimeRemaining will return the timer value remaining until the
 *         target time will be reached next time. The remaining time is target
 *         time minus time already elapsed. In one shot mode, if the timer is in
 *         stopped state, the function will return remaining time value at the
 *         moment of stopping. If the timer is expired, the function will
 *         return 0.
 *
 *  \verbatim
 *  Service name        : Gpt_GetTimeRemaining
 *  Syntax              : Gpt_ValueType Gpt_GetTimeRemaining(Gpt_ChannelType
 *                                                                      Channel)
 *  Service ID[hex]     : 0x04
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Channel - ChannelId of the Gpt channel.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : Gpt_ValueType Remaining timer value (in number o
 *                        ticks)
 *  Description         : Returns the time remaining until the target time is
 *                        reached.
 *  \endverbatim
 */
FUNC(Gpt_ValueType, GPT_CODE) Gpt_GetTimeRemaining( Gpt_ChannelType channel );
#endif /*(STD_ON == GPT_CFG_CFG_TIME_REMAINING_API)*/

/**
 *  \brief Gpt_StartTimer will start the selected timer channel with defined
 *         target time. If the timer channel is enabled for interrupt
 *         notification, then interrupt notification will be triggered after
 *         expiration of the selected timer channel. In one shot mode, if the
 *         timer is expired then the channel will be stopped in interrupt
 *         subroutine. The selected channel will be moved to "Running" state
 *         after calling this function.
 *
 *  \verbatim
 *  Service name        : Gpt_StartTimer
 *  Syntax              : void Gpt_StartTimer(Gpt_ChannelType Channel,
 *                                            Gpt_ValueType Value)
 *  Service ID[hex]     : 0x05
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : Channel - ChannelId of the Gpt channel.
 *                        value - Target time in number of ticks.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Starts a timer channel.
 *  \endverbatim
 */
FUNC(void, GPT_CODE) Gpt_StartTimer(Gpt_ChannelType channel, Gpt_ValueType value);

/**
 *  \brief Gpt_StopTimer will stop the selected timer channel.This will clear
 *         all the registers corresponding to the selected channel. The state of
 *         the timer channel will be changed to "Stopped".If the channel is in
 *         state "Initialized","Expired","Stopped" before calling this function,
 *         the function will be left without any action.
 *
 *  \verbatim
 *  Service name        : Gpt_StopTimer
 *  Syntax              : void Gpt_StopTimer( Gpt_ChannelType Channel )
 *  Service ID[hex]     : 0x06
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant (but not for the same timer channel)
 *  Parameters (in)     : Channel: Numeric identifier of the GPT channel.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Stops a timer channel.
 *  \endverbatim
 */
FUNC(void, GPT_CODE) Gpt_StopTimer(Gpt_ChannelType channel );

#if (STD_ON == GPT_ENABLE_DISABLE_NOTIFICATION_API)
/**
 *  \brief Gpt_EnableNotification will enable the interrupt notification for the
 *         selected channel. The SETINT bit in RTI Set interrupt register will
 *         be Set to enable the Compare interrupt. The interrupt is triggered
 *         when Free running counter value matches with compare register value.
 *
 *  \verbatim
 *  Service name        : Gpt_EnableNotification
 *  Syntax              : void Gpt_EnableNotification( Gpt_ChannelType Channel)
 *  Service ID[hex]     : 0x07
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant (but not for the same timer channel)
 *  Parameters (in)     : Channel: Numeric identifier of the GPT channel.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Enables the interrupt notification for a channel
 *                        relevant in normal mode).
 *  \endverbatim
 */
FUNC(void, GPT_CODE) Gpt_EnableNotification( Gpt_ChannelType channel );
#endif /*(STD_ON == GPT_ENABLE_DISABLE_NOTIFICATION_API)*/

#if (STD_ON == GPT_ENABLE_DISABLE_NOTIFICATION_API)
/**
 *  \brief Gpt_DisableNotification will disable the compare interrupt
 *    notification for the selected channel. The SETINT bit in RTI Set interrupt
 *    register will be cleared to disable the Compare interrupt. The
 *    interrupt is triggered when Free running counter value matches with
 *    compare register value.
 *
 *  \verbatim
 *  Service name        : Gpt_DisableNotification
 *  Syntax              : void Gpt_DisableNotification( Gpt_ChannelType Channel)
 *  Service ID[hex]     : 0x08
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant (but not for the same timer channel)
 *  Parameters (in)     : Channel: Numeric identifier of the GPT channel.
 *  Parameters (inout)  : None
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Disables the interrupt notification for a channel
 *                        relevant in normal mode).
 *  \endverbatim
 */
FUNC(void, GPT_CODE) Gpt_DisableNotification(Gpt_ChannelType channel );
#endif /*(STD_ON == GPT_ENABLE_DISABLE_NOTIFICATION_API)*/

#if (STD_ON == GPT_REGISTER_READBACK_API)
/**
 *  \brief This function reads the important registers of the hardware unit
 *  and returns the value in the structure.
 *
 *  This API should be called after Gpt_Init is called. Otherwise this API will
 *  return E_NOT_OK.
 *
 *  This API could be used to readback the register contents after Gpt_Init
 *  and then the readback value could be compared during GPT execution to
 *  check the correctness of the HW unit. Since this API is used for this
 *  purpose, the register returned are the ones which doesn't change after init
 *  based on conv or channel config.
 *
 *  \verbatim
 *  Service name        : Gpt_ConfigRegReadBack
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : GptChannel - Numeric identifier of the GPT channel.
 *                        If this is invalid, then the API will
 *                        return E_NOT_OK.
 *  Parameters (inout)  : RegRbPtr - Pointer to where to store the readback
 *                        values. If this pointer is NULL_PTR, then the API
 *                        will return E_NOT_OK.
 *  Return value        : Std_ReturnType
 *                        E_OK: Register read back has been done
 *                        E_NOT_OK: Register read back failed
 *  \endverbatim
 */
FUNC(Std_ReturnType, GPT_CODE) Gpt_ConfigRegReadBack(Gpt_ChannelType channel,GptConfigReg* GptConfigReg_pt);
#endif  /* #if (STD_ON == GPT_REGISTER_READBACK_API) */

#ifdef __cplusplus
}
#endif

#endif  /* #ifndef GPT_H_ */

/* @} */

