/*
*
* Copyright (c) 2023 Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
* license under copyrights and patents it now or hereafter owns or controls to make,
* have made, use, import, offer to sell and sell ("Utilize") this software subject to the
* terms herein.  With respect to the foregoing patent license, such license is granted
* solely to the extent that any such patent is necessary to Utilize the software alone.
* The patent license shall not apply to any combinations which include this software,
* other than combinations with devices manufactured by or for TI ("TI Devices").
* No hardware patent is licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this license
* (including the above copyright notice and the disclaimer and (if applicable) source
* code license limitations below) in the documentation and/or other materials provided
* with the distribution
*
* Redistribution and use in binary form, without modification, are permitted provided
* that the following conditions are met:
*
* *       No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* *       any redistribution and use are licensed by TI for use only with TI Devices.
*
* *       Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution of the
* source code are permitted provided that the following conditions are met:
*
* *       any redistribution and use of the source code, including any resulting derivative
* works, are licensed by TI for use only with TI Devices.
*
* *       any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

/**
 *  \file     Cdd_I2c.h
 *
 *  \brief    This file contains interface header for CDD I2C driver
 *
 */

#ifndef CDD_I2C_H
#define CDD_I2C_H

/* ================================================================ */
/*                             Include Files                        */
/* ================================================================ */

#include "Platform_Types.h"
#include "Std_Types.h"
#include "Cdd_I2c_Cfg.h"
#include "SchM_Cdd_I2c.h"

/* ================================================================ */
/*                           Macros                                 */
/* ================================================================ */

/**
 *  \name CDD_I2C Driver Module SW Version Info
 *
 *  Defines for CDD_I2C Driver version used for compatibility checks
 *  @{
 */
/** \brief Driver Implementation Major Version */
#define CDD_I2C_SW_MAJOR_VERSION            (8U)
/** \brief Driver Implementation Minor Version */
#define CDD_I2C_SW_MINOR_VERSION            (6U)
/** \brief Driver Implementation Patch Version */
#define CDD_I2C_SW_PATCH_VERSION            (2U)
/* @} */

/**
 *  \name CDD_I2C Driver Module AUTOSAR Version Info
 *
 *  Defines for CDD_I2C Driver AUTOSAR version used for compatibility 
 *   checks
 *  @{
 */
/** \brief AUTOSAR Major version specification implemented by CDD_I2C Driver */
#define CDD_I2C_AR_RELEASE_MAJOR_VERSION    (4U)
/** \brief AUTOSAR Minor  version specification implemented by CDD_I2C Driver */
#define CDD_I2C_AR_RELEASE_MINOR_VERSION    (3U)
/** \brief AUTOSAR Patch version specification implemented by CDD_I2C Driver */
#define CDD_I2C_AR_RELEASE_REVISION_VERSION (1U)
/* @} */

/**
 *  \name CDD_I2C Driver ID Info
 *  @{
 */
/** \brief Texas Instruments Vendor ID */
#define CDD_I2C_VENDOR_ID                   ((uint16) 44U)
/** \brief CDD_I2C Driver Module ID */
#define CDD_I2C_MODULE_ID                   ((uint16) 255U)
/** \brief CDD_I2C Driver Instance ID */
#define CDD_I2C_INSTANCE_ID                 ((uint8) 0U)              
/* @} */

#define CDD_I2C_ISR_VOID                    (0x00U)
/** \brief Category 1 ISR type */
#define CDD_I2C_ISR_CAT1                    (0x01U)
/** \brief Category 2 ISR type */
#define CDD_I2C_ISR_CAT2                    (0x02U)

/** \brief Maximum possible 10 bit address including read/write bit*/
#define CDD_I2C_10_MAX_BIT_ADDRESS          ((uint16) 2047U)

/**
 *  \name CDD_I2C Service Ids
 *
 *  The Service Id is one of the argument to Det_ReportError function
 and is used to identify the source of the error
 *  @{
 */
/** \brief Service ID Cdd_I2c_Init() */
#define CDD_I2C_SID_INIT                           (0x00u)
/** \brief  Service ID Cdd_I2c_DeInit() */
#define CDD_I2C_SID_DEINIT                         (0x01u) 
/** \brief Service ID Cdd_I2c_GetVersionInfo() */
#define CDD_I2C_SID_GET_VERSION_INFO               (0x02u) 
/** \brief Service ID Cdd_I2c_SetupEB() */
#define CDD_I2C_SID_SETUP_EB                       (0x03u) 
/** \brief Service ID Cdd_I2c_SetupDynamicEB() */
#define CDD_I2C_SID_SETUP_EB_DYNAMIC               (0x04u) 
/** \brief Service ID Cdd_I2c_AsyncTransmit() */
#define CDD_I2C_SID_ASYNC_TRANSMIT                 (0x05u) 
/** \brief Service ID Cdd_I2c_Cancel() */
#define CDD_I2C_SID_CANCEL                         (0x06u) 
/** \brief Service ID Cdd_I2c_GetResult() */
#define CDD_I2C_SID_GET_RESULT                     (0x07u) 
/** \brief Service ID Cdd_I2c_GetSequenceResult() */
#define CDD_I2C_SID_GET_SEQUENCE_RESULT            (0x08u) 
/** \brief Service ID Cdd_I2c_MainFunction() */
#define CDD_I2C_SID_MAIN_FUNCTION                  (0x09u) 
/** \brief Service ID Cdd_I2c_PollingModeProcessing() */
#define CDD_I2C_SID_POLLING_MODE_PROCESSING        (0x0Au)
/** \brief Service ID Cdd_I2c_SetHandling() */
#define CDD_I2C_SID_SET_HANDLING                   (0x0Bu) 
/* @} */

/**
 *  \name CDD_I2C Error Codes
 *
 *  Error codes returned by CDD_I2C functions
 *  @{
 */
/** \brief No errors*/
#define CDD_I2C_E_NO_ERROR                         (0x00u)
/** \brief Error code for not initialized module*/
#define CDD_I2C_E_UNINIT                           (0x01u)
/** \brief  Init service called twice without DeInit*/
#define CDD_I2C_E_ALREADY_INITIALIZED              (0x02u)
/** \brief Driver is busy*/
#define CDD_I2C_E_BUSY                             (0x03u)
/** \brief Channel out of bounds, exceeds the 
 * maximum number of configured channels*/
#define CDD_I2C_E_PARAM_CHANNEL                    (0x04u)
/** \brief Sequence out of bounds, exceeds the 
 * maximum number of configured sequences*/
#define CDD_I2C_E_PARAM_SEQUENCE                   (0x05u)
/** \brief Length out of bounds – a zero length 
 * is not accepted. The total length is 
 * limited to 65535*/
#define CDD_I2C_E_PARAM_LENGTH                     (0x06u)
/** \brief An invalid version info pointer has 
 * been passed (a NULL_PTR)*/
#define CDD_I2C_E_PARAM_VINFO_PTR                  (0x07u)
/** \brief An invalid configuration has been 
 * passed (a non NULL_PTR)*/
#define CDD_I2C_E_PARAM_CONFIG                     (0x08u)
/** \brief Two in-valid transmission buffers have 
 * been passed (no NULL_PTR/ or both NULL_PTR)*/
#define CDD_I2C_E_PARAM_TRANS_BUFFER               (0x09u)
/** \brief Address out of bounds, exceeds the 
 * limit of the configured address range*/
#define CDD_I2C_E_PARAM_ADDRESS                    (0x0Au)
/** \brief Buffer direction conflicts with channel
 * direction set via EB*/
#define CDD_I2C_E_PARAM_DIRECTION                  (0x0Bu)
/** \brief Notify callback functions passed as 
 * NULL_PTR*/
#define CDD_I2C_E_PARAM_NOTIFY_CALLBACK            (0x0Cu)
/** \brief Bus error encountered during transmission
 * or reception*/
#define CDD_I2C_E_PARAM_TRANSMISSION_ERROR         (0x0Du)
/** \brief Queue full*/
#define CDD_I2C_E_PARAM_QUEUE_FULL                 (0x0Eu)
/** \brief Timeout error*/
#define CDD_I2C_E_PARAM_TIMEOUT                    (0x0Fu)
/** \brief Hardware unit busy*/
#define CDD_I2C_E_HW_UNIT_BUSY                     (0x10u)
/** \brief Channel unit busy*/
#define CDD_I2C_E_CHANNEL_BUSY                     (0x11u)
/** \brief ARBITRATION_LOSS*/
#define CDD_I2C_E_ARBITRATION_LOSS                 (0x12u)
/** \brief NACK*/
#define CDD_I2C_E_NACK                             (0x13u)
/** \brief Receive shift register full*/
#define CDD_I2C_E_RECEIVE_SHIFT_REGISTER_FULL      (0x14u)

#define CDD_I2C_E_PARAM_POINTER                    (0x15u)
/* @} */

/** \brief CDD_I2C mask value for write action */
#define CDD_I2C_WRITE                              (0x00u)

/** \brief CDD_I2C mask value for read action */
#define CDD_I2C_READ                               (0x01u)

/**
 * Mask values to clear, enable, disable or set
 * I2c specific interrupts
*/
#define CDD_I2C_INT_ARBITRATION_LOST               (0x01u) 
#define CDD_I2C_INT_NO_ACK                         (0x02u)
#define CDD_I2C_INT_ADRR_READY_ACESS               (0x04u)
#define CDD_I2C_INT_RECV_READY                     (0x08u)
#define CDD_I2C_INT_TRANSMIT_READY                 (0x10u)
#define CDD_I2C_INT_STOP_CONDITION                 (0x20u)
#define CDD_I2C_INT_ADRR_ZERO                      (0x100u)
#define CDD_I2C_INT_ADRR_SLAVE                     (0x200u)
#define CDD_I2C_INT_TRANSMIT_UNDER_FLOW            (0x400u)
#define CDD_I2C_INT_RECV_OVER_RUN                  (0x800u)
#define CDD_I2C_INT_BUS_BUSY                       (0x1000u)
#define CDD_I2C_INT_NO_ACK_SENT                    (0x2000u)
#define CDD_I2C_INT_SLAVE_DIRECTION                (0x4000u)
#define CDD_I2C_ALL_INTERRUPTS                     ((CDD_I2C_INT_ARBITRATION_LOST) | (CDD_I2C_INT_NO_ACK) | \
                                                    (CDD_I2C_INT_ADRR_READY_ACESS) | (CDD_I2C_INT_RECV_READY) | \
                                                    (CDD_I2C_INT_TRANSMIT_READY) | (CDD_I2C_INT_STOP_CONDITION) | \
                                                    (CDD_I2C_INT_ADRR_ZERO) | (CDD_I2C_INT_ADRR_SLAVE) | \
                                                    (CDD_I2C_INT_TRANSMIT_UNDER_FLOW) | (CDD_I2C_INT_RECV_OVER_RUN) | \
                                                    (CDD_I2C_INT_BUS_BUSY) | (CDD_I2C_INT_NO_ACK_SENT) | \
                                                    (CDD_I2C_INT_SLAVE_DIRECTION))

#define CDD_I2C_ALL_INTERRUPTS_CLEAR               ((CDD_I2C_INT_ARBITRATION_LOST) | (CDD_I2C_INT_NO_ACK) | \
                                                    (CDD_I2C_INT_ADRR_READY_ACESS) | (CDD_I2C_INT_RECV_READY) | \
                                                    (CDD_I2C_INT_STOP_CONDITION)   | (CDD_I2C_INT_NO_ACK_SENT) | \
                                                    (CDD_I2C_INT_SLAVE_DIRECTION))

/**
 * Control register mask values
*/
#define CDD_I2C_OUT_OF_RESET                        (0x20u)
#define CDD_I2C_MASTER_MODE                         (0x400u)
#define CDD_I2C_FREE_RUN                            (0x4000u)

/**
 * System clock source frequencies
*/
#define CDD_I2C_MODULE_FUNC_CLK_96MHZ               (96000000U)
#define CDD_I2C_MODULE_FUNC_CLK_200MHZ              (200000000U)

/**
 * I2C hardware unit frequencies
*/
#define CDD_I2C_MODULE_INTERNAL_CLK_8MHZ            (8000000U)
#define CDD_I2C_MODULE_INTERNAL_CLK_12P5MHZ         (12500000U)

/**
 * I2C Status macros
*/
#define CDD_I2C_OK                                  (0x00u)
#define CDD_I2C_NOT_OK                              (0x01u)

/**
 * I2C Bus Status macros
*/
#define CDD_I2C_BUS_FREE                            (0x00)
#define CDD_I2C_BUS_BUSY                            (0x01)

/**
 * I2C Address scheme macros
*/
#define CDD_I2C_7_BIT_ADDRESS                       (0x00)
#define CDD_I2C_10_BIT_ADDRESS                      (0x01)

/**
 * Baud rate macros
*/
#define CDD_I2C_3_4MHZ                              (3400000U)
#define CDD_I2C_1MHZ                                (1000000U)
#define CDD_I2C_100KHZ                              (100000U)
#define CDD_I2C_400KHZ                              (400000U)
#define CDD_I2C_1P0MHZ                              (1000000U)
#define CDD_I2C_3P4MHZ                              (3400000U)

/**
 * Chained channel configurations
*/
#define CHAIN_CHANNEL_ON                        (0x01U)
#define CHAIN_CHANNEL_OFF                       (0x00U)

/* ================================================================ */
/*                Typedefs, Structures and enums                    */
/* ================================================================ */

/*Function pointer for sequence error notify*/
typedef void (*Cdd_I2c_SequenceErrorNotification)(uint8 ErrorCode);

/*Function pointer for sequence complete notify*/
typedef void (*Cdd_I2c_SequenceEndNotification)(void);

/*I2c_SequenceType used to pass Sequence ID*/
typedef uint8 Cdd_I2c_SequenceType;

/*Cdd_I2c_AddressType used to pass addresss*/
typedef uint16 Cdd_I2c_AddressType;

/*I2c_DataLengthType used to pass message size*/
typedef uint16 Cdd_I2c_DataLengthType;

/*I2c_DataPtrType used to store Rx data address*/
typedef uint8* Cdd_I2c_DataPtrType;

/*I2c_DataPtrType used to store Tx data address*/
typedef uint8* Cdd_I2c_DataConstPtrType;

/**
 * This is an enum containing the possible HW unit states states 
 * The default HW unit state is CDD_I2C_HW_UNIT_FREE,
*/
typedef enum
{
    CDD_I2C_HW_UNIT_FREE,
    /*I2C Hardware unit not in use by a sequence*/
    CDD_I2C_HW_UNIT_BUSY
    /*I2C Hardware unit in use by a sequence*/
}Cdd_I2c_HwUnitStatus;

/**
 * This is an enum containing the possible driver states 
 * The default driver state is CDD_I2C_UNINIT,
 * global variable used to access and update driver states is CddI2cDrvState
*/
typedef enum
{
    CDD_I2C_UNINIT,
    /*I2C driver is uninitialized*/
    CDD_I2C_IDLE,
    /*I2C driver is idle*/
    CDD_I2C_BUSY
    /*I2C driver is busy*/
}Cdd_I2c_ComponentStatusType;

/**
 * This is an enum containing the possible variants 
 * of arbitration loss timeout handling
*/
typedef enum
{
    CDD_I2C_BURST_MODE,
    /*In case bus is busy detach from I2C bus*/
    CDD_I2C_RECURRENT_MODE
    /*In case bus is busy try to recover I2C bus*/
}Cdd_I2c_HandlingType;

/**
 * This is an enum containing all the possible channel results
 * The default channel state is CDD_I2C_CH_RESULT_OK
*/
typedef enum
{
    CDD_I2C_CH_RESULT_OK,
    /*I2C channel transmission not in progress*/
    CDD_I2C_CH_RESULT_IN_PROGRESS,
    /*I2C channel transmission is in progress*/
    CDD_I2C_CH_RESULT_BUSFAIL,
    /*I2C channel transmission failed due to bus unavailability*/
    CDD_I2C_CH_RESULT_ARBFAIL,
    /*I2C channel transmission failed due to arbitration loss*/
    CDD_I2C_CH_RESULT_NACKFAIL
    /*I2C channel transmission failed due to bus NACK*/
}Cdd_I2c_ChannelResultType;

/**
 * This is an enum containing all the possible results
 * The default sequence result is CDD_I2C_SEQ_OK
*/
typedef enum
{
    CDD_I2C_SEQ_OK,
    /*I2C sequence transmission not in progress*/
    CDD_I2C_SEQ_PENDING,
    /*I2C sequence transmission in progress*/
    CDD_I2C_SEQ_QUEUED,
    /*I2C sequence transmission queued*/
    CDD_I2C_SEQ_NACK,
    /*I2C sequence transmission failed due to NACK*/
    CDD_I2C_SEQ_FAILED,
    /*I2C sequence transmission failed*/
    CDD_I2C_SEQ_CANCELLED
    /*I2C sequence transmission cancelled from queue*/
}Cdd_I2c_SeqResultType;

/**
 * This is a structure for CDD_I2C which contains the HW unit initialization parameters 
*/
typedef struct
{
    uint8 HwInstance;
    /*This element points to which I2C hardware instance to use for communication with slave, user configurable*/
    uint32 BaudRate;
    /*This element is used to set the baud rate of the desired I2c bus, user configurable*/
    uint32 HwUnitFrequency;
    /*This unit is used set the I2c HW module frequency*/
    uint32 SysClk;
    /*This element contains the system clock frequency being used by the I2c instance*/
}Cdd_I2c_Hw_Cfg;

/**
 * This is a structure for CDD_I2C configuration which contains a pointer
 * to the user configured parameters for a specified I2C HW unit  
*/
typedef struct Cdd_I2c_ConfigType_s
{
    Cdd_I2c_Hw_Cfg CddI2cHwCfgPtr[CDD_I2C_HW_UNITS_USED];
    /*This element points to the I2c HW unit configurations set by the user*/
} Cdd_I2c_ConfigType;

/**
 * This is a structure for CDD_I2C sequence which serves as the primary unit for I2C communication in this driver 
*/
typedef struct __attribute__((packed))
{
    uint8     HwInstance;
    /*This element points to which I2C hardware instance to use for communication with slave, user configurable*/
    uint32    MaxChannels;
    /*This element indicates the maximum number of channels in the given sequence, user configurable*/
    Cdd_I2c_SeqResultType SeqResult;
    /*This element contains the sequence result*/
    Cdd_I2c_SequenceEndNotification SequenceCompleteNotify;
    /*This element is a function pointer which points to the address of user provided API, in case of succesfull sequence completion*/
    Cdd_I2c_SequenceErrorNotification SequenceErrorNotify;
    /*This element is a function pointer which points to the address of user provided API, in case of unsuccesfull sequence completion*/
    Cdd_I2c_SequenceType ChannelList[CDD_I2C_MAXIMUM_CHANNEL_NUMBER];
    /*This element indicates the channel numbers used by a sequence*/
}Cdd_I2c_Sequence_Config;
/**
 * This is a structure for CDD_I2C Channel which is a part of the sequence 
*/
typedef struct __attribute__((packed))
{  
    uint8 Dir;
    /*This element indicates whether we read from or write to the slave, 
     *initially assigned some random value updated during Cdd_I2c_SetupEB function call*/
    Cdd_I2c_AddressType SlaveAddress;
    /*This element contains the address of the slave, user configurable*/
    uint8 SlaveAddressScheme;
    /*If this element is 0, the slave has a 7 bit address and
     *a 10 bit address if this element is 1, user configurable*/
    uint8 ChainedChannel;
    /*If this element is 0 the channel is not chained, else it is chained, user configurable*/
    Cdd_I2c_DataLengthType DataBufferLength;
    /*This element contains the size of the data buffer, initially 0 
     *will be updated after Cdd_I2c_SetupEB function call*/
    Cdd_I2c_DataPtrType DataBuffer;
    /*This element contains a pointer to which initially points to NULL
     * but will be updated after Cdd_I2c_SetupEB function call*/
    Cdd_I2c_ChannelResultType ChannelResult;
    /*This element contains the channel result*/
}Cdd_I2c_Channel_Config;

/**
 * This is a structure for Cdd_I2c queue implementation
*/
struct Cdd_I2c_Queue_obj_ {
    Cdd_I2c_SequenceType sequenceId;
    /*This element contains the sequence ID to be executed*/
    struct Cdd_I2c_Queue_obj_ *pNext;
    /*This element is used to create a linked list implementation of the queue*/
};

typedef struct Cdd_I2c_Queue_obj_ Cdd_I2c_Queue_obj;

/*Pointer type to Cdd_I2c_ConfigType*/
typedef Cdd_I2c_ConfigType* Cdd_I2c_ConfigPtrType;

/*I2c_DataType used to store Tx/Rx data*/
typedef uint8 Cdd_I2c_DataType;

/*I2c_ChannelType used to pass Channel ID*/
typedef uint8 Cdd_I2c_ChannelType; 

/* ================================================================ */
/*                          Global variables                        */
/* ================================================================ */

extern VAR(Cdd_I2c_HwUnitStatus,CDD_I2C_VAR_INIT) CddI2cHwUnitStatus[CDD_I2C_HW_UNITS_MAX];
extern VAR(Cdd_I2c_ComponentStatusType,CDD_I2C_VAR_INIT) CddI2cDrvState;
extern VAR(Cdd_I2c_Sequence_Config, CDD_I2C_VAR_INIT) Cdd_I2cSequenceContainerLocal[CDD_I2C_MAXIMUM_SEQUENCE_NUMBER];
extern VAR(Cdd_I2c_Channel_Config, CDD_I2C_VAR_INIT) Cdd_I2cChannelContainerLocal[CDD_I2C_MAXIMUM_CHANNEL_NUMBER];

/** \brief Cdd I2c global initialization parameter pointer */
extern CONST(Cdd_I2c_ConfigPtrType, CDD_I2C_CONFIG_DATA) pCddI2cInitParams;

/** \brief Cdd I2c global sequence array */
extern CONST(Cdd_I2c_Sequence_Config, CDD_I2C_CONFIG_DATA) Cdd_I2cSequenceContainer[CDD_I2C_MAXIMUM_SEQUENCE_NUMBER];

/** \brief Cdd I2c HW unit base address array */
extern CONST(uint32,CDD_I2C_CONST) CddI2cHwUnitBaseAddr[CDD_I2C_HW_UNITS_MAX];

/** \brief Cdd I2c queue array */
extern VAR(Cdd_I2c_Queue_obj,CDD_I2C_VAR_INIT) CddI2cTransmissionQueue[CDD_I2C_MAX_QUEUE_SIZE];

/** \brief Pointer to Cdd I2c queue array[0] */
extern VAR(Cdd_I2c_Queue_obj *,CDD_I2C_VAR_INIT) pCddI2cTransmissionQueue;

/** \brief Arbitration loss parameter*/
extern CONST(Cdd_I2c_HandlingType, CDD_I2C_CONFIG_DATA) CddI2cArbitrationLossParam;

/** \brief I2c Channel configuration variables*/
extern VAR(Cdd_I2c_Channel_Config, CDD_I2C_CONFIG_DATA) Cdd_I2cChannelContainer[CDD_I2C_MAXIMUM_CHANNEL_NUMBER];

/* ================================================================ */
/*                          Function Declarations                   */
/* ================================================================ */

/** \brief Service for CDD_I2C initialization
 *  \verbatim
 *  Service name        : Cdd_I2c_Init
 *  Syntax              : void Cdd_I2c_Init( Cdd_I2c_ConfigPtrType configPtr)
 *  Service ID[hex]     : 0x00
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : configPtr - Pointer to configuration set
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Service for CDD_I2C initialization
 *  \endverbatim
*/
FUNC(void, CDD_I2C_CODE) Cdd_I2c_Init( Cdd_I2c_ConfigType* configPtr);

/** \brief CDD_I2C De-initialization function
 *  \verbatim
 *  Service name        : Cdd_I2c_DeInit
 *  Syntax              : Std_ReturnType Cdd_I2c_DeInit(void)
 *  Service ID[hex]     : 0x01
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : void
 *  Parameters (out)    : None
 *  Return value        : CDD_I2C status type
 *  Description         : This function de-initializes the module I2c. During de-initialization the module is shut down.
 *                        The module is put into its power-on reset state.
 *  \endverbatim
*/
FUNC(Std_ReturnType, CDD_I2C_CODE) Cdd_I2c_DeInit(void);

/** \brief Returns the version information
 *  \verbatim
 *  Service name        : Cdd_I2c_GetVersionInfo
 *  Syntax              : void Cdd_I2c_GetVersionInfo(Cdd_I2c_VersionInfoPtrType *versionInfoPtr)
 *  Service ID[hex]     : 0x02
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : versionInfoPtr - pointer to version information, this must not be NULL
 *  Parameters (out)    : None
 *  Return value        : None
 *  Description         : Cdd_I2c_GetVersionInfo() returns version information, 
 *                        vendor ID and AUTOSAR module ID of the component.
 *  \endverbatim
*/
#if (STD_ON == CDD_I2C_VERSION_INFO_API)  /*STD_ON == CDD_I2C_VERSION_INFO_API*/
FUNC(void, CDD_I2C_CODE) Cdd_I2c_GetVersionInfo(Std_VersionInfoType* versionInfoPtr);
#endif  /*<STD_ON == CDD_I2C_VERSION_INFO_API>*/

/** \brief Setup external buffer
 *  \verbatim
 *  Service name        : Cdd_I2c_SetupEB
 *  Syntax              : Std_ReturnType Cdd_I2c_SetupEB(I2c_ChannelType channelId,
                                                  Cdd_I2c_DataPtrType txDataBufferPtr,
                                                  Cdd_I2c_DataPtrType rxDataBufferPtr,
                                                  Cdd_I2c_DataLengthType length)
 *  Service ID[hex]     : 0x03
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : channelId 
 *                        txDataBufferPtr             - Pointer to a buffer containing data for transmission
 *                        rxDataBufferPtr             - Pointer to a buffer provided for receiving data 
 *                        length                      - Number of bytes which shall be transmit/received
 *  Parameters (out)    : Receive data buffer pointer - Pointer to a buffer provided for receiving data
 *  Return value        : External buffer setup result 
 *  Description         : The service takes over the given parameters, and prepares a channel for 
 *                        transmission (transmit buffer) and reception (receive buffer). 
 *                        
 *  \endverbatim
*/
FUNC(Std_ReturnType, CDD_I2C_CODE) Cdd_I2c_SetupEB(Cdd_I2c_ChannelType channelId,
                                                  Cdd_I2c_DataConstPtrType txDataBufferPtr,
                                                  Cdd_I2c_DataPtrType rxDataBufferPtr,
                                                  Cdd_I2c_DataLengthType length);

/** \brief Dynamically setup external buffer
 *  \verbatim
 *  Service name        : Cdd_I2c_SetupEBDynamic
 *  Syntax              : Std_ReturnType Cdd_I2c_SetupEBDynamic(I2c_ChannelType channelId,
                                                                Cdd_I2c_AddressType address,
                                                                Cdd_I2c_DataPtrType txDataBufferPtr,
                                                                Cdd_I2c_DataPtrType rxDataBufferPtr,
                                                                Cdd_I2c_DataLengthType length)
 *  Service ID[hex]     : 0x04
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : channelId 
 *                        Address                     - Address given by the service's caller,
 *                                                      used for the transmission/reception
 *                        txDataBufferPtr             - Pointer to a buffer containing data for transmission
 *                        rxDataBufferPtr             - Pointer to a buffer provided for receiving data 
 *                        length                      - Number of bytes which shall be transmit/received
 *  Parameters (out)    : Receive data buffer pointer - Pointer to a buffer provided for receiving data
 *  Return value        : External buffer setup result 
 *  Description         : The service takes over the given parameters, 
 *                        and prepares a channel for transmission (transmit buffer)
 *                        and reception (receive buffer).
 *                        This service does not use the channel's configured address 
 *                        for transmission/reception, but the given address parameter
 *                        
 *  \endverbatim
*/
FUNC(Std_ReturnType, CDD_I2C_CODE) Cdd_I2c_SetupEBDynamic(Cdd_I2c_ChannelType channelId,
                                                          Cdd_I2c_AddressType address,
                                                          Cdd_I2c_DataConstPtrType txDataBufferPtr,
                                                          Cdd_I2c_DataPtrType rxDataBufferPtr,
                                                          Cdd_I2c_DataLengthType length);
                                                          
/** \brief Asynchronous data transmission
 *  \verbatim
 *  Service name        : Cdd_I2c_AsyncTransmit
 *  Syntax              : Std_ReturnType Cdd_I2c_AsyncTransmit(I2c_SequenceType sequenceId)
 *  Service ID[hex]     : 0x05
 *  Sync/Async          : Non Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : sequenceId
 *  Parameters (out)    : None
 *  Return value        : Transmit sequence result 
 *  Description         : This service executes a data exchange with the prepared channels. 
 *                        Only channels of the given sequence are affected of the data transmission
 *                        
 *  \endverbatim
*/
FUNC(Std_ReturnType, CDD_I2C_CODE) Cdd_I2c_AsyncTransmit(Cdd_I2c_SequenceType sequenceId);

/** \brief Cancel a transmission
 *  \verbatim
 *  Service name        : Cdd_I2c_Cancel
 *  Syntax              : Std_ReturnType Cdd_I2c_Cancel(I2c_SequenceType sequenceId)
 *  Service ID[hex]     : 0x06
 *  Sync/Async          : Non Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : sequenceId
 *  Parameters (out)    : None
 *  Return value        : Cancel sequence transmission result 
 *  Description         : A request that is already in transmission progress,
 *                        is cancelled by finishing the current sequence.
 *                        This is done to achieve a stable state with no undefined data.
 *                        A queued transmission request is cancelled at once.
 *                        
 *  \endverbatim
*/
#if (STD_ON == CDD_I2C_CANCEL_API)  /*STD_ON == CDD_I2C_CANCEL_API*/
FUNC(Std_ReturnType, CDD_I2C_CODE) Cdd_I2c_Cancel(Cdd_I2c_SequenceType sequenceId);
#endif  /*STD_ON == CDD_I2C_CANCEL_API*/

/** \brief Returns the requested channel's result
 *  \verbatim
 *  Service name        : Cdd_I2c_GetResult
 *  Syntax              : Cdd_I2c_ChannelResultType Cdd_I2c_GetResult(I2c_ChannelType channelId)
 *  Service ID[hex]     : 0x07
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : channelId
 *  Parameters (out)    : None
 *  Return value        : Channel result 
 *  Description         : Returns the requested channel's result
 *                        
 *  \endverbatim
*/
FUNC(Cdd_I2c_ChannelResultType, CDD_I2C_CODE) Cdd_I2c_GetResult(Cdd_I2c_ChannelType channelId);

/** \brief Returns the requested sequence's result
 *  \verbatim
 *  Service name        : Cdd_I2c_GetResult
 *  Syntax              : Cdd_I2c_SeqResultType Cdd_I2c_GetSequenceResult(I2c_SequenceType sequenceId)
 *  Service ID[hex]     : 0x08
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : sequenceId
 *  Parameters (out)    : None
 *  Return value        : Sequence result 
 *  Description         : Returns the requested sequence's result
 *                        
 *  \endverbatim
*/
FUNC(Cdd_I2c_SeqResultType, CDD_I2C_CODE) Cdd_I2c_GetSequenceResult(Cdd_I2c_SequenceType sequenceId);

/** \brief The components MainFunction is responsible for 
 *         processing the sequence queue and for 
 *         updating the components status
 *  \verbatim
 *  Service name        : Cdd_I2c_MainFunction
 *  Syntax              : void Cdd_I2c_MainFunction(void)
 *  Service ID[hex]     : 0x09
 *  Sync/Async          : Non Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : None
 *  Parameters (out)    : None
 *  Return value        : None 
 *  Description         : The components MainFunction is responsible for 
 *                        processing the sequence queue and for 
 *                        updating the components status
 *                        
 *  \endverbatim
*/
FUNC(void, CDD_I2C_CODE) Cdd_I2c_MainFunction(void);    

/** \brief Ensures proper processing of I2c communication in the case of disabled interrupts
 *  \verbatim
 *  Service name        : Cdd_I2c_PollingModeProcessing
 *  Syntax              : void Cdd_I2c_PollingModeProcessing(void)
 *  Service ID[hex]     : 0x0A
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Non Reentrant
 *  Parameters (in)     : None
 *  Parameters (out)    : None
 *  Return value        : None 
 *  Description         : Ensures proper processing of I2c communication in the case of disabled interrupts
 *                        
 *  \endverbatim
*/
#if (STD_ON == CDD_I2C_POLLING_MODE) /*STD_ON == CDD_I2C_POLLING_MODE*/
FUNC(void, CDD_I2C_CODE) Cdd_I2c_PollingModeProcessing(void);   
#endif  /*STD_ON == CDD_I2C_POLLING_MODE*/

/** \brief Set the timeout handling
 *  \verbatim
 *  Service name        : Cdd_I2c_SetHandling
 *  Syntax              : void Cdd_I2c_SetHandling(Cdd_I2c_HandlingType handling)
 *  Service ID[hex]     : None
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : handling      - BURST_MODE:     In case of an timeout the I2c module does not perform any further actions,
 *                                                        but detaches from the bus.
 *                                      - RECURRENT_MODE: In case of an timeout the I2c module sends stop signals on the bus
 *  Parameters (out)    : None
 *  Return value        : None 
 *  Description         : This service sets the I2c module's timeout handling
 *                        
 *  \endverbatim
*/
FUNC(void, CDD_I2C_CODE) Cdd_I2c_SetHandling(Cdd_I2c_HandlingType handling);    

/** \brief Returns the module's status
 *  \verbatim
 *  Service name        : Cdd_I2c_GetStatus
 *  Syntax              : Cdd_I2c_ComponentStatusType Cdd_I2c_GetStatus(void)
 *  Service ID[hex]     : None
 *  Sync/Async          : Synchronous
 *  Reentrancy          : Reentrant
 *  Parameters (in)     : None
 *  Parameters (out)    : None
 *  Return value        : Module status 
 *  Description         : Returns the module's status
 *                        
 *  \endverbatim
*/
#if (STD_ON == CDD_I2C_GET_STATUS_API)  /*STD_ON == CDD_I2C_GET_STATUS_API*/
FUNC(Cdd_I2c_ComponentStatusType, CDD_I2C_CODE) Cdd_I2c_GetStatus(void);
#endif  /*STD_ON == CDD_I2C_GET_STATUS_API*/

#endif