/*
*
* Copyright (c) 2022 Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, 
* non-exclusive license under copyrights and patents it now or hereafter owns or 
* controls to make, have made, use, import, offer to sell and sell ("Utilize") 
* this software subject to the terms herein.With respect to the foregoing patent 
* license, such license is granted solely to the extent that any such patent is 
* necessary to Utilize the software alone. The patent license shall not apply to 
* any combinations which include this software, other than combinations with 
* devices manufactured by or for TI ("TI Devices"). No hardware patent is 
* licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this 
* license (including the above copyright notice and the disclaimer and 
* (if applicable) source code license limitations below) in the documentation 
* and/or other materials provided with the distribution
*
* Redistribution and use in binary form, without modification, are permitted 
* provided that the following conditions are met:
*
* *  No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* * any redistribution and use are licensed by TI for use only with TI Devices.
*
* * Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution 
* of the source code are permitted provided that the following conditions are 
* met:
*
* * any redistribution and use of the source code, including any resulting 
* derivative works, are licensed by TI for use only with TI Devices.
*
* * any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI 
* Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its 
* suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED  OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

/**
 *  \file     Cdd_I2c.c
 *
 *  \brief    This file contains the function implementations of the CDD I2C driver
 *   
 *
 */

/* ================================================================ */
/*                             Include Files                        */
/* ================================================================ */

#include "Cdd_I2c.h"
#include "Cdd_I2c_priv.h"
#include "Cdd_I2c_hw_reg.h"
#include "Os.h"
#include <string.h>

/* ================================================================ */
/*                          Global variables                        */
/* ================================================================ */

#define CDD_I2C_START_SEC_VAR_INIT_UNSPECIFIED
#include "Cdd_I2c_MemMap.h"
/*Driver status should be uninitialized at the begining*/
VAR(Cdd_I2c_ComponentStatusType,CDD_I2C_VAR_INIT) CddI2cDrvState = CDD_I2C_UNINIT;
VAR(Cdd_I2c_HwUnitStatus,CDD_I2C_VAR_INIT) CddI2cHwUnitStatus[CDD_I2C_HW_UNITS_MAX] = {CDD_I2C_HW_UNIT_FREE};
#define CDD_I2C_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Cdd_I2c_MemMap.h"

#define CDD_I2C_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Cdd_I2c_MemMap.h"
VAR(Cdd_I2c_Sequence_Config, CDD_I2C_VAR_NO_INIT) Cdd_I2cSequenceContainerLocal[CDD_I2C_MAXIMUM_SEQUENCE_NUMBER];
VAR(Cdd_I2c_Channel_Config, CDD_I2C_VAR_NO_INIT) Cdd_I2cChannelContainerLocal[CDD_I2C_MAXIMUM_CHANNEL_NUMBER];
#define CDD_I2C_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Cdd_I2c_MemMap.h"

#define CDD_I2C_START_SEC_CODE
#include "Cdd_I2c_MemMap.h"

/* ================================================================ */
/*                          Function Definitions                    */
/* ================================================================ */

FUNC(void, CDD_I2C_CODE) Cdd_I2c_Init( Cdd_I2c_ConfigType* configPtr){
    uint8 i = 0;

    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
    if(NULL_PTR != configPtr)
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_INIT, CDD_I2C_E_PARAM_CONFIG);
        /*Invalid pointer passed as argument*/
    }

    else if(CDD_I2C_UNINIT != CddI2cDrvState)
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_INIT, CDD_I2C_E_ALREADY_INITIALIZED);
        /*Driver already initialized*/
    }
    #endif

    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
    else
    #endif
    {
        if(CDD_I2C_UNINIT == CddI2cDrvState){
            for(i = 0;  i<CDD_I2C_HW_UNITS_USED;    i++){
                /**
                * Function call initialises I2C hardware unit
                * at frequency specified from GUI 
                * and specified I2C bus frequency
                */
                Cdd_I2c_Setup_HW_Unit_Clk(pCddI2cInitParams->CddI2cHwCfgPtr[i].HwInstance, 
                                        pCddI2cInitParams->CddI2cHwCfgPtr[i].BaudRate,
                                        pCddI2cInitParams->CddI2cHwCfgPtr[i].HwUnitFrequency,
                                        pCddI2cInitParams->CddI2cHwCfgPtr[i].SysClk);

                /*Function call to clear all I2C related interrupts*/
                Cdd_I2c_Clear_interrupts(pCddI2cInitParams->CddI2cHwCfgPtr[i].HwInstance, CDD_I2C_ALL_INTERRUPTS_CLEAR);

                /*Function call to disable all I2C related interrupts*/
                Cdd_I2c_Disable_interrupts(pCddI2cInitParams->CddI2cHwCfgPtr[i].HwInstance, (uint16)~(CDD_I2C_ALL_INTERRUPTS));

                /**
                * Function call to enable Cdd_I2c master, set up I2C in free run 
                * and bring I2c out of reset
                */
                Cdd_I2c_Enable_master(pCddI2cInitParams->CddI2cHwCfgPtr[i].HwInstance);

                /*Function call to set own address as 0x7F by default*/
                Cdd_I2c_Set_Own_Address(pCddI2cInitParams->CddI2cHwCfgPtr[i].HwInstance, 0x7F);
            }
            /*Copy sequence configurations into a local sequence array*/
            memcpy(&Cdd_I2cSequenceContainerLocal, &Cdd_I2cSequenceContainer, sizeof(Cdd_I2cSequenceContainer));
            /*Copy channel configurations into a local channel array*/
            memcpy(&Cdd_I2cChannelContainerLocal, &Cdd_I2cChannelContainer, sizeof(Cdd_I2cChannelContainer));
            /*Function call to set all queue parameters to default value*/
            Cdd_I2c_queue_init(pCddI2cTransmissionQueue);
        }
    }
        CddI2cDrvState = CDD_I2C_IDLE;
}

FUNC(Std_ReturnType, CDD_I2C_CODE) Cdd_I2c_DeInit(void){
    uint8 i = 0;
    uint8 status = 0;

    if(CDD_I2C_UNINIT != CddI2cDrvState){
        for(i = 0;  i<CDD_I2C_HW_UNITS_USED;    i++){
            /*Function call to disable all I2C related interrupts*/
            Cdd_I2c_Disable_interrupts(pCddI2cInitParams->CddI2cHwCfgPtr[i].HwInstance,(uint16)~(CDD_I2C_ALL_INTERRUPTS));
            /*Function call to put I2C peripheral in reset mode*/
            Cdd_I2c_Disable_master(pCddI2cInitParams->CddI2cHwCfgPtr[i].HwInstance);
        }
        /*Update the driver status global variable to UNINIT*/
        CddI2cDrvState = CDD_I2C_UNINIT;
        status = (uint8)CddI2cDrvState;
    }
    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
    else{
        Cdd_I2cReportDetError(CDD_I2C_SID_DEINIT, CDD_I2C_E_UNINIT);
        /*Driver already un-initialized*/
    }
    #endif
    /*return driver state*/
    return status;
}

#if (STD_ON == CDD_I2C_VERSION_INFO_API)  /*STD_ON == CDD_I2C_VERSION_INFO_API*/
FUNC(void, CDD_I2C_CODE) Cdd_I2c_GetVersionInfo(Std_VersionInfoType* versionInfoPtr){
    if(NULL_PTR != versionInfoPtr){
        versionInfoPtr->vendorID = CDD_I2C_VENDOR_ID;
        versionInfoPtr->moduleID = CDD_I2C_MODULE_ID;
        versionInfoPtr->sw_major_version = (uint8)CDD_I2C_SW_MAJOR_VERSION;
        versionInfoPtr->sw_minor_version = (uint8)CDD_I2C_SW_MINOR_VERSION;
        versionInfoPtr->sw_patch_version = (uint8)CDD_I2C_SW_PATCH_VERSION;
    }
    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
    else{
        Cdd_I2cReportDetError(CDD_I2C_SID_GET_VERSION_INFO, CDD_I2C_E_PARAM_VINFO_PTR);
        /*NULL_PTR passed as argument*/
    }
    #endif
}
#endif  /*<STD_ON == CDD_I2C_VERSION_INFO_API>*/

FUNC(Std_ReturnType, CDD_I2C_CODE) Cdd_I2c_SetupEB(Cdd_I2c_ChannelType channelId,
                                                  Cdd_I2c_DataConstPtrType txDataBufferPtr,
                                                  Cdd_I2c_DataPtrType rxDataBufferPtr,
                                                  Cdd_I2c_DataLengthType length)
{
    uint8 status = CDD_I2C_NOT_OK;

    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
    if(CDD_I2C_UNINIT == CddI2cDrvState)
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB, CDD_I2C_E_UNINIT);
        /*Driver un-initialized*/
    }

    else if(CDD_I2C_TX_RX_BUFFER_MAXIMUM_SIZE < length)
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB, CDD_I2C_E_PARAM_LENGTH);
        /*Message size exceeds maximum allowed message length*/
    }

    else if(CDD_I2C_MAXIMUM_CHANNEL_NUMBER <= channelId)
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB, CDD_I2C_E_PARAM_CHANNEL);
        /*Channel ID exceeds maximum number of channels configured*/
    }

    else if((NULL_PTR == txDataBufferPtr)    &&  (NULL_PTR == rxDataBufferPtr))
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB, CDD_I2C_E_PARAM_TRANS_BUFFER);
        /*Both buffers passed are NULL_PTR*/
    }

    else if((NULL_PTR != txDataBufferPtr)   &&  (NULL_PTR != rxDataBufferPtr))
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB, CDD_I2C_E_PARAM_TRANS_BUFFER);
        /*Both buffers passed are non NULL_PTR*/
    }
    #endif

    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
    else if((NULL_PTR != txDataBufferPtr)   &&  (NULL_PTR == rxDataBufferPtr))
    #elif (STD_OFF == CDD_I2C_DEV_ERROR_DETECT)
    if((NULL_PTR != txDataBufferPtr)   &&  (NULL_PTR == rxDataBufferPtr))
    #endif
    {
        if(CDD_I2C_WRITE == Cdd_I2cChannelContainerLocal[channelId].Dir){
            /*Copy user input to conifgurations*/
            Cdd_I2cChannelContainer[channelId].DataBufferLength = length;
            Cdd_I2cChannelContainer[channelId].DataBuffer = txDataBufferPtr;
            /*Copy user input to local driver objects*/
            Cdd_I2cChannelContainerLocal[channelId].DataBufferLength = Cdd_I2cChannelContainer[channelId].DataBufferLength;
            Cdd_I2cChannelContainerLocal[channelId].DataBuffer = Cdd_I2cChannelContainer[channelId].DataBuffer;
            status = CDD_I2C_OK;
        }

        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        else{
            Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB, CDD_I2C_E_PARAM_DIRECTION);
            /*Buffer direction conflicts with channel direction configured in EB*/
        }
        #endif

    }
    else if((NULL_PTR == txDataBufferPtr)   &&  (NULL_PTR != rxDataBufferPtr)){
        if(CDD_I2C_READ == Cdd_I2cChannelContainerLocal[channelId].Dir){
            /*Copy user input to conifgurations*/
            Cdd_I2cChannelContainer[channelId].DataBufferLength = length;
            Cdd_I2cChannelContainer[channelId].DataBuffer = rxDataBufferPtr;
            /*Copy user input to local driver objects*/
            Cdd_I2cChannelContainerLocal[channelId].DataBufferLength = Cdd_I2cChannelContainer[channelId].DataBufferLength;
            Cdd_I2cChannelContainerLocal[channelId].DataBuffer = Cdd_I2cChannelContainer[channelId].DataBuffer;
            status = CDD_I2C_OK;
        }

        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        else{
            Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB, CDD_I2C_E_PARAM_DIRECTION);
            /*Buffer direction conflicts with channel direction configured in EB*/
        }
        #endif

    }
    return status;
}

FUNC(Std_ReturnType, CDD_I2C_CODE) Cdd_I2c_SetupEBDynamic(Cdd_I2c_ChannelType channelId,
                                                          Cdd_I2c_AddressType address,
                                                          Cdd_I2c_DataConstPtrType txDataBufferPtr,
                                                          Cdd_I2c_DataPtrType rxDataBufferPtr,
                                                          Cdd_I2c_DataLengthType length)
{
    uint8 status = CDD_I2C_NOT_OK;

    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
    if(CDD_I2C_UNINIT == CddI2cDrvState)
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB_DYNAMIC, CDD_I2C_E_UNINIT);
        /*Driver un-initialized*/
    }
    
    else if(CDD_I2C_TX_RX_BUFFER_MAXIMUM_SIZE < length)
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB_DYNAMIC, CDD_I2C_E_PARAM_LENGTH);
        /*Message size exceeds maximum allowed message length*/
    }

    else if(CDD_I2C_MAXIMUM_CHANNEL_NUMBER <= channelId)
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB_DYNAMIC, CDD_I2C_E_PARAM_CHANNEL);
        /*Channel ID exceeds maximum number of channels configured*/
    }

    else if((NULL_PTR == txDataBufferPtr)    &&  (NULL_PTR == rxDataBufferPtr))
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB_DYNAMIC, CDD_I2C_E_PARAM_TRANS_BUFFER);
        /*Both buffers passed are NULL_PTR*/
    }

    else if((NULL_PTR != txDataBufferPtr)   &&  (NULL_PTR != rxDataBufferPtr))
    {
        Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB_DYNAMIC, CDD_I2C_E_PARAM_TRANS_BUFFER);
        /*Both buffers passed are non NULL_PTR*/
    }
    #endif

    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
    else if((NULL_PTR != txDataBufferPtr)   &&  (NULL_PTR == rxDataBufferPtr))
    #elif (STD_OFF == CDD_I2C_DEV_ERROR_DETECT)
    if((NULL_PTR != txDataBufferPtr)   &&  (NULL_PTR == rxDataBufferPtr))
    #endif
    {
        if(CDD_I2C_WRITE == Cdd_I2cChannelContainerLocal[channelId].Dir){
            /*Copy user input to conifgurations*/
            Cdd_I2cChannelContainer[channelId].DataBufferLength = length;
            Cdd_I2cChannelContainer[channelId].DataBuffer = txDataBufferPtr;
            Cdd_I2cChannelContainer[channelId].SlaveAddress = address;
            if(Cdd_I2cChannelContainer[channelId].SlaveAddress > 0x7F){
                Cdd_I2cChannelContainer[channelId].SlaveAddressScheme = CDD_I2C_10_BIT_ADDRESS;
            }
            else{
                Cdd_I2cChannelContainer[channelId].SlaveAddressScheme = CDD_I2C_7_BIT_ADDRESS;
            }
            /*Copy user input to local driver objects*/
            Cdd_I2cChannelContainerLocal[channelId].DataBufferLength = Cdd_I2cChannelContainer[channelId].DataBufferLength;
            Cdd_I2cChannelContainerLocal[channelId].DataBuffer = Cdd_I2cChannelContainer[channelId].DataBuffer;
            Cdd_I2cChannelContainerLocal[channelId].SlaveAddress = Cdd_I2cChannelContainer[channelId].SlaveAddress;
            Cdd_I2cChannelContainerLocal[channelId].SlaveAddressScheme = Cdd_I2cChannelContainer[channelId].SlaveAddressScheme;
            status = CDD_I2C_OK;
        }

        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        else{
            Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB_DYNAMIC, CDD_I2C_E_PARAM_DIRECTION);
            /*Buffer direction conflicts with channel direction configured in EB*/
        }
        #endif
    }
    else if((NULL_PTR == txDataBufferPtr)   &&  (NULL_PTR != rxDataBufferPtr)){
        if(CDD_I2C_READ == Cdd_I2cChannelContainerLocal[channelId].Dir){
            /*Copy user input to conifgurations*/
            Cdd_I2cChannelContainer[channelId].DataBufferLength = length;
            Cdd_I2cChannelContainer[channelId].DataBuffer = rxDataBufferPtr;
            Cdd_I2cChannelContainer[channelId].SlaveAddress = address;
            if(Cdd_I2cChannelContainer[channelId].SlaveAddress > 0x7F){
                Cdd_I2cChannelContainer[channelId].SlaveAddressScheme = CDD_I2C_10_BIT_ADDRESS;
            }
            else{
                Cdd_I2cChannelContainer[channelId].SlaveAddressScheme = CDD_I2C_7_BIT_ADDRESS;
            }
            /*Copy user input to local driver objects*/
            Cdd_I2cChannelContainerLocal[channelId].DataBufferLength = Cdd_I2cChannelContainer[channelId].DataBufferLength;
            Cdd_I2cChannelContainerLocal[channelId].DataBuffer = Cdd_I2cChannelContainer[channelId].DataBuffer;
            Cdd_I2cChannelContainerLocal[channelId].SlaveAddress = Cdd_I2cChannelContainer[channelId].SlaveAddress;
            Cdd_I2cChannelContainerLocal[channelId].SlaveAddressScheme = Cdd_I2cChannelContainer[channelId].SlaveAddressScheme;
            status = CDD_I2C_OK;
        }

        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        else{
            Cdd_I2cReportDetError(CDD_I2C_SID_SETUP_EB_DYNAMIC, CDD_I2C_E_PARAM_DIRECTION);
            /*Buffer direction conflicts with channel direction configured in EB*/
        }
        #endif
    }
    return status;
}

FUNC(Std_ReturnType, CDD_I2C_CODE) Cdd_I2c_AsyncTransmit(Cdd_I2c_SequenceType sequenceId)
{
    uint8 status = CDD_I2C_OK;
    uint8 i = 0;
    uint8 ChannelId = 0;
    uint8 HwUnit = 0;
    uint32 Timeout = 0;
    Cdd_I2c_AddressType SlaveAddress = 0;
    
    #if (STD_ON == CDD_I2C_POLLING_MODE)
    uint32 IsrStatus = 0;
    #endif

    if(CDD_I2C_UNINIT == CddI2cDrvState){
        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        Cdd_I2cReportDetError(CDD_I2C_SID_ASYNC_TRANSMIT, CDD_I2C_E_UNINIT);
        /*Driver un-initialized*/
        #endif
        status = CDD_I2C_NOT_OK;
    }

    else if(CDD_I2C_MAXIMUM_SEQUENCE_NUMBER <= sequenceId){
        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        Cdd_I2cReportDetError(CDD_I2C_SID_ASYNC_TRANSMIT, CDD_I2C_E_PARAM_SEQUENCE);
        /*Sequence ID exceeds maximum number of sequences configured*/
        #endif
        status = CDD_I2C_NOT_OK;
    }

    else if((CDD_I2C_HW_UNIT_BUSY == CddI2cHwUnitStatus[Cdd_I2cSequenceContainerLocal[sequenceId].HwInstance]))
    {
        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        Cdd_I2cSequenceContainer[sequenceId].SequenceErrorNotify(CDD_I2C_E_HW_UNIT_BUSY);
        /*Report Hardware unit is busy*/
        #endif
        Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
        /*Update sequence status*/
        status = CDD_I2C_NOT_OK;
        /*Function call to queue sequence*/
        Cdd_I2c_Enqueue(sequenceId);
    }

    else if(CDD_I2C_OK == status)
    {
        /*Update driver status*/
        CddI2cDrvState = CDD_I2C_BUSY; 
        /*Update hardware unit in use status*/
        CddI2cHwUnitStatus[Cdd_I2cSequenceContainerLocal[sequenceId].HwInstance] = CDD_I2C_HW_UNIT_BUSY;  
        /*Update Sequence result*/
        Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_PENDING;
        /*Determine the HW unit used in the sequence*/
        HwUnit = Cdd_I2cSequenceContainerLocal[sequenceId].HwInstance;
        /*Run a loop to cover all channels of a sequence*/
        for(i = 0;  i<Cdd_I2cSequenceContainerLocal[sequenceId].MaxChannels; i++){
            if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult){
                break;
            }
            /*Find the channel ID*/
            ChannelId = Cdd_I2cSequenceContainerLocal[sequenceId].ChannelList[i];
            /*Find the Slave address*/
            SlaveAddress = Cdd_I2cChannelContainerLocal[ChannelId].SlaveAddress;
            if(NULL_PTR == Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer){
                #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                Cdd_I2cReportDetError(CDD_I2C_SID_ASYNC_TRANSMIT, CDD_I2C_E_PARAM_TRANS_BUFFER);
                /*Async transmit called before setupEB*/
                #endif
                Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                break;
            }
            /*Function call to set the addressing mode for the slave device*/
            Cdd_I2c_Set_Slave_Addressing_mode(HwUnit, Cdd_I2cChannelContainerLocal[ChannelId].SlaveAddressScheme);
            /*Function call to set the slave address when device in master mode*/
            Cdd_I2c_Set_Slave_Address(HwUnit, SlaveAddress);
            if(CDD_I2C_READ == Cdd_I2cChannelContainerLocal[ChannelId].Dir){
                #if (STD_OFF == CDD_I2C_POLLING_MODE)
                if(CDD_I2C_CH_RESULT_IN_PROGRESS == Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult){
                    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                    Cdd_I2cSequenceContainer[sequenceId].SequenceErrorNotify(CDD_I2C_E_CHANNEL_BUSY);
                    /*Channel busy*/
                    #endif
                    Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                    break;
                }
                Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_IN_PROGRESS;
                /*Function call to set the master in receive mode*/
                Cdd_I2c_Set_Master_Receive_Mode(HwUnit);
                /*Function call to set number of bytes to read/write*/
                Cdd_I2c_Data_Count(HwUnit, Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength);
                /*Update Channel status*/ 
                Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_IN_PROGRESS;
                /*Function call to clear all I2C related interrupts*/
                /*Get initial value for timeout period*/
                (void) GetCounterValue(CDD_I2C_OS_COUNTER_ID,&Timeout);
                /*Check if bus is busy*/
                while(CDD_I2C_BUS_BUSY == Cdd_I2cBusCheck(HwUnit)){
                    /*Get elapsed time value*/
                    if(1 == (Cdd_I2c_Timeout(Timeout,CDD_I2C_RESET_TIMEOUT))){
                        /*Call Cdd_I2c_SetHandling if bus is busy for more than specified time period*/
                        Cdd_I2c_SetHandling(CddI2cArbitrationLossParam);
                        Timeout = 0;
                        Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
                        Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                        break;
                    }
                }
                Timeout = 0;
                if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult){
                    break;
                }
                /*Disable global interrupts*/
                SchM_Enter_Cdd_I2c_I2C_EXCLUSIVE_AREA_0();
                /*Function call to disable local interrupts*/
                Cdd_I2c_Disable_interrupts(HwUnit, (uint16)~(CDD_I2C_ALL_INTERRUPTS));
                /*Function call to clear all interrupts*/
                Cdd_I2c_Clear_interrupts(HwUnit, CDD_I2C_ALL_INTERRUPTS_CLEAR);
                /*Function call to enable configurator selected interrupts*/
                Cdd_I2c_Enable_interrupts(HwUnit);  
                /*Enable global interrupts*/
                SchM_Exit_Cdd_I2c_I2C_EXCLUSIVE_AREA_0();
                /*Check if channel is chained or not*/
                if(CHAIN_CHANNEL_OFF == Cdd_I2cChannelContainerLocal[ChannelId].ChainedChannel){
                    /*Function call to send a start signal on the bus in interrupt mode*/
                    Cdd_I2c_Start(HwUnit);
                }
                while(CDD_I2C_CH_RESULT_IN_PROGRESS == Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult){
                    /*Wait while channel reception is in progress*/
                }
                #endif
            }
            else if(CDD_I2C_WRITE == Cdd_I2cChannelContainerLocal[ChannelId].Dir){
                if(CDD_I2C_CH_RESULT_IN_PROGRESS == Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult){
                    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                    Cdd_I2cSequenceContainer[sequenceId].SequenceErrorNotify(CDD_I2C_E_CHANNEL_BUSY);
                    /*Channel busy*/
                    #endif
                    Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                    break;
                }
                /*Update channel result*/
                Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_IN_PROGRESS;
                /*Function call to set master in transmit mode*/
                Cdd_I2c_Set_Master_Transmit_Mode(HwUnit);
                /*Function call to set number of bytes to read/write*/
                Cdd_I2c_Data_Count(HwUnit, Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength);
                #if (STD_OFF == CDD_I2C_POLLING_MODE)   /*STD_OFF == CDD_I2C_POLLING_MODE*/
                /*Get initial value for timeout period*/
                (void) GetCounterValue(CDD_I2C_OS_COUNTER_ID,&Timeout);
                /*Check if bus is busy*/
                while(CDD_I2C_BUS_BUSY == Cdd_I2cBusCheck(HwUnit)){
                    /*Get elapsed time value*/
                    if(1 == (Cdd_I2c_Timeout(Timeout,CDD_I2C_RESET_TIMEOUT))){
                        /*Call Cdd_I2c_SetHandling if bus is busy for more than specified time period*/
                        Cdd_I2c_SetHandling(CddI2cArbitrationLossParam);
                        Timeout = 0;
                        Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
                        Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                        break;
                    }
                }
                Timeout = 0;
                if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult){
                    break;
                }
                /*Disable global interrupts*/
                SchM_Enter_Cdd_I2c_I2C_EXCLUSIVE_AREA_0();
                /*Function call to disable local interrupts*/
                Cdd_I2c_Disable_interrupts(HwUnit, (uint16)~(CDD_I2C_ALL_INTERRUPTS));
                /*Function call to clear all I2C related interrupts*/
                Cdd_I2c_Clear_interrupts(HwUnit, CDD_I2C_ALL_INTERRUPTS_CLEAR);
                /*Function call to enable configurator selected interrupts*/
                Cdd_I2c_Enable_interrupts(HwUnit); 
                /*Enable global interrupts*/
                SchM_Exit_Cdd_I2c_I2C_EXCLUSIVE_AREA_0();
                /*Check if channel is chained or not*/
                if(CHAIN_CHANNEL_OFF == Cdd_I2cChannelContainerLocal[ChannelId].ChainedChannel){
                    /*Function call to send a start signal on the bus in interrupt mode*/
                    Cdd_I2c_Start(HwUnit);
                }
                while(CDD_I2C_CH_RESULT_IN_PROGRESS == Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult){
                    /*Wait while channel transmission is in progress*/
                }
                #endif  /*STD_OFF == CDD_I2C_POLLING_MODE*/
                #if (STD_ON == CDD_I2C_POLLING_MODE)    /*STD_ON == CDD_I2C_POLLING_MODE*/
                /*Get initial value for timeout period*/
                (void) GetCounterValue(CDD_I2C_OS_COUNTER_ID,&Timeout);
                /*Check if bus is busy*/
                while(CDD_I2C_BUS_BUSY == Cdd_I2cBusCheck(HwUnit)){
                    /*Get elapsed time value*/
                    if(1 == (Cdd_I2c_Timeout(Timeout,CDD_I2C_RESET_TIMEOUT))){
                        /*Call Cdd_I2c_SetHandling if bus is busy for more than specified time period*/
                        Cdd_I2c_SetHandling(CddI2cArbitrationLossParam);
                        Timeout = 0;
                        Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
                        Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                        break;
                    }
                }
                Timeout = 0; 
                /*Function call to clear all I2C related interrupts*/
                Cdd_I2c_Clear_interrupts(HwUnit, CDD_I2C_ALL_INTERRUPTS_CLEAR);
                if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult){
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                    break;
                }   
                if(CHAIN_CHANNEL_OFF == Cdd_I2cChannelContainerLocal[ChannelId].ChainedChannel){    
                    /*Disable global interrupts*/
                    SchM_Enter_Cdd_I2c_I2C_EXCLUSIVE_AREA_0();
                    /*Function call to send a start signal on the bus in interrupt mode*/
                    Cdd_I2c_Start(HwUnit);
                }
                while(0 != Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength){
                    /*Check for errors*/
                    IsrStatus = Cdd_I2c_Isr_Check(HwUnit);
                    if(IsrStatus & (CDD_I2C_ICSTR_AL_MASK | CDD_I2C_ICSTR_NACK_MASK | 
                       CDD_I2C_ICSTR_AD0_MASK | CDD_I2C_ICSTR_RSFULL_MASK)){
                        if(NULL_PTR != Cdd_I2cSequenceContainer[sequenceId].SequenceErrorNotify){
                            Cdd_I2cSequenceContainer[sequenceId].SequenceErrorNotify(Cdd_I2c_Error_Handler(IsrStatus, ChannelId, HwUnit));
                        }
                        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                        else{
                            Cdd_I2cReportDetError(CDD_I2C_SID_ASYNC_TRANSMIT, CDD_I2C_E_PARAM_NOTIFY_CALLBACK);
                            /*Sequence error notify callback function passed as NULL_PTR*/
                        }
                        #endif
                        Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                        break;
                    }
                    else if(IsrStatus & CDD_I2C_ICSTR_ICXRDY_MASK){
                        while(0 == ((Cdd_I2c_Isr_Check(HwUnit)) & CDD_I2C_ICSTR_ICXRDY_MASK));
                        Cdd_I2c_Write(HwUnit, *(Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer));
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer++;
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength--;
                    }
                }
                /*Enable global interrupts*/
                SchM_Exit_Cdd_I2c_I2C_EXCLUSIVE_AREA_0();
                if(CDD_I2C_SEQ_FAILED != Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult){
                    /*Get initial value for timeout period*/
                    (void) GetCounterValue(CDD_I2C_OS_COUNTER_ID,&Timeout);
                    /*Check if bus is busy*/
                    while(CDD_I2C_BUS_BUSY == Cdd_I2cBusCheck(HwUnit)){
                        Cdd_I2c_Stop(HwUnit);
                        if(CDD_I2C_BUS_BUSY == Cdd_I2cBusCheck(HwUnit)){
                            /*Get elapsed time value*/
                            if(1 == (Cdd_I2c_Timeout(Timeout,CDD_I2C_RESET_TIMEOUT))){
                                /*Call Cdd_I2c_SetHandling if bus is busy for more than specified time period*/
                                Cdd_I2c_SetHandling(CddI2cArbitrationLossParam);
                                Timeout = 0;
                                Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
                                Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                                break;
                            }
                        }
                    }
                    Timeout = 0;
                    /*Function call to disable all interrupts*/
                    Cdd_I2c_Disable_interrupts(HwUnit, (uint16)~(CDD_I2C_ALL_INTERRUPTS));
                    /*Function call to clear all interrupts*/
                    Cdd_I2c_Clear_interrupts(HwUnit, CDD_I2C_ALL_INTERRUPTS_CLEAR);
                    if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult){
                        Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                        break;
                    }
                    else if(0 == Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength){
                        if(NULL_PTR != Cdd_I2cSequenceContainer[sequenceId].SequenceCompleteNotify){
                            if(ChannelId == Cdd_I2cSequenceContainerLocal[sequenceId].ChannelList[Cdd_I2cSequenceContainerLocal[sequenceId].MaxChannels-1]){
                                Cdd_I2cSequenceContainer[sequenceId].SequenceCompleteNotify();
                            }
                        }
                        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                        else{
                            Cdd_I2cReportDetError(CDD_I2C_SID_ASYNC_TRANSMIT, CDD_I2C_E_PARAM_NOTIFY_CALLBACK);
                            /*Sequence complete notify callback function passed as NULL_PTR*/
                        }
                        #endif
                        Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_OK;
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                        if(CDD_I2C_SEQ_PENDING == Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult){
                            Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_OK;
                            status = CDD_I2C_OK;
                        }
                    }
                }
                #endif  /*STD_ON == CDD_I2C_POLLING_MODE*/
            }
            if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult){
                break;
            }
        } 
        if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult){
            status = CDD_I2C_NOT_OK;
            /*Function call to queue sequence*/
            Cdd_I2c_Enqueue(sequenceId);
        }
        /*Update hardware unit in use status*/
        CddI2cHwUnitStatus[Cdd_I2cSequenceContainerLocal[sequenceId].HwInstance] = CDD_I2C_HW_UNIT_FREE;
    }
        CddI2cDrvState = CDD_I2C_IDLE;
        return status;
}

#if (STD_ON == CDD_I2C_CANCEL_API)  /*STD_ON == CDD_I2C_CANCEL_API*/
FUNC(Std_ReturnType, CDD_I2C_CODE) Cdd_I2c_Cancel(Cdd_I2c_SequenceType sequenceId){
    uint8 status = CDD_I2C_OK;
    uint8 i = 0;

    if(CDD_I2C_UNINIT == CddI2cDrvState){
        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        Cdd_I2cReportDetError(CDD_I2C_SID_CANCEL, CDD_I2C_E_UNINIT);
        /*Driver un-initialized*/
        #endif
        status = CDD_I2C_NOT_OK;
    }
    else if(CDD_I2C_MAXIMUM_SEQUENCE_NUMBER <= sequenceId){
        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        Cdd_I2cReportDetError(CDD_I2C_SID_CANCEL, CDD_I2C_E_PARAM_SEQUENCE);
        /*Sequence ID passed exceeds maximum sequences configured*/
        #endif
        status = CDD_I2C_NOT_OK;
    }
    else if(status == CDD_I2C_OK){
        for(i = 0;  i<CDD_I2C_MAX_QUEUE_SIZE;   i++){
            if(sequenceId == ((pCddI2cTransmissionQueue + i)->sequenceId)){
                if(CDD_I2C_SEQ_CANCELLED != Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult){
                    Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_CANCELLED;
                    status = CDD_I2C_OK;
                    break;
                }
            }
        }
    }
    return status;
}
#endif  /*STD_ON == CDD_I2C_CANCEL_API*/

FUNC(Cdd_I2c_ChannelResultType, CDD_I2C_CODE) Cdd_I2c_GetResult(Cdd_I2c_ChannelType channelId){
    Cdd_I2c_ChannelResultType retval = CDD_I2C_NOT_OK;

    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
    if(CDD_I2C_UNINIT == CddI2cDrvState){
        Cdd_I2cReportDetError(CDD_I2C_SID_GET_RESULT, CDD_I2C_E_UNINIT);
        /*Driver un-initialized*/
    }

    else if(CDD_I2C_MAXIMUM_CHANNEL_NUMBER <= channelId){
        Cdd_I2cReportDetError(CDD_I2C_SID_GET_RESULT, CDD_I2C_E_PARAM_CHANNEL);
        /*Channel ID passed exceeds maximum channels configured*/
    }

    else
    #endif
    {
        retval = Cdd_I2cChannelContainerLocal[channelId].ChannelResult;
    }
    return retval;
}

FUNC(Cdd_I2c_SeqResultType, CDD_I2C_CODE) Cdd_I2c_GetSequenceResult(Cdd_I2c_SequenceType sequenceId){
    Cdd_I2c_SeqResultType retval = CDD_I2C_NOT_OK;

    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
    if(CDD_I2C_UNINIT == CddI2cDrvState){
        Cdd_I2cReportDetError(CDD_I2C_SID_GET_SEQUENCE_RESULT, CDD_I2C_E_UNINIT);
        /*Driver un-initialized*/
    }

    else if(CDD_I2C_MAXIMUM_SEQUENCE_NUMBER <= sequenceId){
        Cdd_I2cReportDetError(CDD_I2C_SID_GET_SEQUENCE_RESULT, CDD_I2C_E_PARAM_SEQUENCE);
        /*Sequence ID passed exceeds maximum configured sequences*/
    }

    else
    #endif
    {
        retval = Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult;
    }
    return retval;
}

#if (STD_ON == CDD_I2C_GET_STATUS_API)  /*STD_ON == CDD_I2C_GET_STATUS_API*/
FUNC(Cdd_I2c_ComponentStatusType, CDD_I2C_CODE) Cdd_I2c_GetStatus(void){
    return CddI2cDrvState;
}
#endif  /*STD_ON == CDD_I2C_GET_STATUS_API*/

FUNC(void, CDD_I2C_CODE) Cdd_I2c_MainFunction(void){
    Cdd_I2c_Queue_obj * pTemp = pCddI2cTransmissionQueue;
    uint32 Timeout = 0;

    /*Execute below loop till next element of queue is not pointing to NULL*/
    while(NULL_PTR != pTemp){
        /*Check if sequence is queued or not*/
        if(((pTemp->sequenceId) <= CDD_I2C_MAXIMUM_SEQUENCE_NUMBER) &&  (CDD_I2C_SEQ_QUEUED == Cdd_I2cSequenceContainerLocal[pTemp->sequenceId].SeqResult)){
            Cdd_I2c_AsyncTransmit(pTemp->sequenceId);
            #if (STD_ON == CDD_I2C_POLLING_MODE)
            Cdd_I2c_PollingModeProcessingFromQueue(pTemp->sequenceId);
            #endif
            /*Get initial timeout value*/
            (void) GetCounterValue(CDD_I2C_OS_COUNTER_ID,&Timeout);
            /*Wait till sequence result is OK*/
            while(CDD_I2C_SEQ_OK != Cdd_I2cSequenceContainerLocal[pTemp->sequenceId].SeqResult){
                /*Check if elapsed time is greater than or equal to specified time*/
                if(1 == (Cdd_I2c_Timeout(Timeout,CDD_I2C_QUEUE_TIMEOUT))){
                    break;
                }
            }
            Timeout = 0;
            if(CDD_I2C_SEQ_QUEUED != Cdd_I2cSequenceContainerLocal[pTemp->sequenceId].SeqResult){
                pTemp->sequenceId = 255;
            }
        }
        /*Once sequence is processed update pointer to next sequence entry*/
        pTemp = pTemp->pNext;
    }
}

#if (STD_ON == CDD_I2C_POLLING_MODE) /*STD_ON == CDD_I2C_POLLING_MODE*/
FUNC(void, CDD_I2C_CODE) Cdd_I2c_PollingModeProcessing(void){
    uint8 SequenceId = 0;
    uint8 ChannelId = 0;
    uint8 i = 0;
    uint16 SlaveAddress = 0;
    uint8 HwUnit = 0;

    /*Transmission/Reception for all sequences*/
    for(SequenceId = 0;  SequenceId<CDD_I2C_MAXIMUM_SEQUENCE_NUMBER;  SequenceId++){
        if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult){
            continue;
        }
        else if((CDD_I2C_HW_UNIT_BUSY == CddI2cHwUnitStatus[Cdd_I2cSequenceContainerLocal[SequenceId].HwInstance])){
            #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
            Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify(CDD_I2C_E_HW_UNIT_BUSY);
            /*Bus busy*/
            #endif
            continue;
        }
        /*Update hardware unit in use status*/
        CddI2cHwUnitStatus[Cdd_I2cSequenceContainerLocal[SequenceId].HwInstance] = CDD_I2C_HW_UNIT_BUSY;
        HwUnit = Cdd_I2cSequenceContainerLocal[SequenceId].HwInstance;
        Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_PENDING;
        /*Reception for all channels of a sequence*/
        for(i = 0;  i<Cdd_I2cSequenceContainerLocal[SequenceId].MaxChannels; i++){
            /*Find the channel ID number*/
            ChannelId = Cdd_I2cSequenceContainerLocal[SequenceId].ChannelList[i];
            /*Check if channel direction is read*/
            if(CDD_I2C_READ == Cdd_I2cChannelContainerLocal[ChannelId].Dir){
                if(CDD_I2C_CH_RESULT_IN_PROGRESS == Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult){
                    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                    Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify(CDD_I2C_E_CHANNEL_BUSY);
                    /*Channel busy*/
                    #endif
                    Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                    break;
                }
                /*Update channel result*/
                Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_IN_PROGRESS;
                #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                if(NULL_PTR == Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer){
                    Cdd_I2cReportDetError(CDD_I2C_SID_POLLING_MODE_PROCESSING, CDD_I2C_E_PARAM_TRANS_BUFFER);
                    /*Reception function called before setupEB*/
                }
                #endif
                /*Find the Slave address*/
                SlaveAddress = Cdd_I2cChannelContainerLocal[ChannelId].SlaveAddress;
                /*Function call to set the addressing mode for the slave device*/
                Cdd_I2c_Set_Slave_Addressing_mode(HwUnit, Cdd_I2cChannelContainerLocal[ChannelId].SlaveAddressScheme);
                /*Function call to set the slave address when device in master mode*/
                Cdd_I2c_Set_Slave_Address(HwUnit, SlaveAddress);
                Cdd_I2c_Polling_Mode_Receive(HwUnit, ChannelId, SequenceId);
            }
        }
        /*Update Hardware unit status*/
        CddI2cHwUnitStatus[Cdd_I2cSequenceContainerLocal[SequenceId].HwInstance] = CDD_I2C_HW_UNIT_FREE;
        if((CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult)  &&  (CDD_I2C_READ == Cdd_I2cChannelContainerLocal[ChannelId].Dir)){
            Cdd_I2c_Enqueue(SequenceId);
            break;
        }
    }
}
#endif  /*STD_ON == CDD_I2C_POLLING_MODE*/

FUNC(void, CDD_I2C_CODE) Cdd_I2c_SetHandling(Cdd_I2c_HandlingType handling){
    uint8 i = 0;
    
    if(CDD_I2C_BURST_MODE == handling){
        (void) Cdd_I2c_DeInit();
    }
    else if(CDD_I2C_RECURRENT_MODE == handling){
        for(i = 0;  i<CDD_I2C_HW_UNITS_USED;    i++){
            Cdd_I2c_Stop(pCddI2cInitParams->CddI2cHwCfgPtr[i].HwInstance);
            (void) Cdd_I2c_DeInit();
            Cdd_I2c_Init(NULL_PTR);
        }
    }
}

#define CDD_I2C_STOP_SEC_CODE
#include "Cdd_I2c_MemMap.h"
