/*
*
* Copyright (c) 2022 Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, 
* non-exclusive license under copyrights and patents it now or hereafter owns or 
* controls to make, have made, use, import, offer to sell and sell ("Utilize") 
* this software subject to the terms herein.With respect to the foregoing patent 
* license, such license is granted solely to the extent that any such patent is 
* necessary to Utilize the software alone. The patent license shall not apply to 
* any combinations which include this software, other than combinations with 
* devices manufactured by or for TI ("TI Devices"). No hardware patent is 
* licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this 
* license (including the above copyright notice and the disclaimer and 
* (if applicable) source code license limitations below) in the documentation 
* and/or other materials provided with the distribution
*
* Redistribution and use in binary form, without modification, are permitted 
* provided that the following conditions are met:
*
* *  No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* * any redistribution and use are licensed by TI for use only with TI Devices.
*
* * Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution 
* of the source code are permitted provided that the following conditions are 
* met:
*
* * any redistribution and use of the source code, including any resulting 
* derivative works, are licensed by TI for use only with TI Devices.
*
* * any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI 
* Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its 
* suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED  OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

/**
 *  \file     Cdd_I2c_priv.c
 *
 *  \brief    This file contains the internal function implementations of the CDD I2C driver
 *   
 *
 */

/* ================================================================ */
/*                             Include Files                        */
/* ================================================================ */

#include "Cdd_I2c.h"
#include "Cdd_I2c_Cfg.h"
#include "Cdd_I2c_priv.h"
#include "Cdd_I2c_hw_reg.h"
#include "hw_types.h"
#include "Os.h"

#define CDD_I2C_START_SEC_VAR_INIT_8
#include "Cdd_I2c_MemMap.h"

/* ================================================================ */
/*                          Global variables                        */
/* ================================================================ */

static uint8 QueueCounter = 0;

#define CDD_I2C_STOP_SEC_VAR_INIT_8
#include "Cdd_I2c_MemMap.h"

#define CDD_I2C_START_SEC_CODE
#include "Cdd_I2c_MemMap.h"

/* ================================================================ */
/*                          Function Definitions                    */
/* ================================================================ */

#if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
void Cdd_I2cReportDetError(uint8 apiId, uint8 errorId)
{
    (void) Det_ReportError(CDD_I2C_MODULE_ID, CDD_I2C_INSTANCE_ID, apiId, errorId);
    return;
}
#endif

uint8 Cdd_I2c_Timeout(uint32 TimeoutPrevious, uint32 TimeoutPeriod){
    uint32 TimeoutPresent = 0;
    uint8 status = 0;
    /*Get current time ticks value*/
    (void) GetCounterValue(CDD_I2C_OS_COUNTER_ID, &TimeoutPresent);
    if((TimeoutPresent - TimeoutPrevious) >= TimeoutPeriod){
        status = 1;
    }
    else{
        status = 0;
    }
    return status;
}

void Cdd_I2c_Setup_HW_Unit_Clk(uint8 HwInstance, uint32 BaudRate, uint32 Cdd_I2cHwUnitFrequency, uint32 SysClk){
    uint8 PreScalerI2cModule = 0;
    uint16 VariableDifference = 0;
    uint16 Divisor = 0;

    /**
     * If input clock to I2C peripheral is 96MHZ or 200MHz, the operating frequency of the HW unit is
     * 8MHz, if the input clock is of any other frequency the I2C HW unit will operate at 12.5MHz
    */
    if((CDD_I2C_MODULE_FUNC_CLK_96MHZ == SysClk)   ||  (CDD_I2C_MODULE_FUNC_CLK_200MHZ == SysClk)){
        Cdd_I2cHwUnitFrequency = CDD_I2C_MODULE_INTERNAL_CLK_8MHZ;
    }
    else{
        Cdd_I2cHwUnitFrequency = CDD_I2C_MODULE_INTERNAL_CLK_12P5MHZ;
    }
    /**
     * If the specified BaudRate for communication is 3.4MHz,
     * the I2C HW unit will operate at 12.5 MHz
    */
    if(3400000 == BaudRate){
        Cdd_I2cHwUnitFrequency = CDD_I2C_MODULE_INTERNAL_CLK_12P5MHZ;
    }
    
    PreScalerI2cModule = ((SysClk/Cdd_I2cHwUnitFrequency)-1);
    /**
     * Calculated pre scaler value as per the above formula
     * written to CDD_I2C_ICPSC register
    */
    HW_WR_REG8((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICPSC), PreScalerI2cModule);

    if(0 == PreScalerI2cModule){
        VariableDifference = 7;
    }
    else if(1 == PreScalerI2cModule){
        VariableDifference = 6;
    }
    else{
        VariableDifference = 5;
    }
    Divisor = (Cdd_I2cHwUnitFrequency/BaudRate);
    Divisor = (Divisor/2);
    /**
     * Writes values to clock low as well as clock high register
     * based on pre-scaler valueus.
    */
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICCLKL), (Divisor - VariableDifference));
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICCLKH), (Divisor - VariableDifference));
}

void Cdd_I2c_Set_Own_Address(uint8 HwInstance, uint16 address){
    /**
     * Writes address passed as parameter to the register,
     * Will add check to see if address is 10 bit or 7 bit
     * When driver supports SOC to function as slave 
    */
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICOAR), address);
}

void Cdd_I2c_Clear_interrupts(uint8 HwInstance, uint16 InterruptMask){
    /**
     * Writes mask value to clear any 
     * existing interrupts to the register
    */
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICSTR), InterruptMask);
}

void Cdd_I2c_Disable_interrupts(uint8 HwInstance, uint16 InterruptMask){
    /**
     * Writes mask value to disable all I2C 
     * related interrupts to the register
    */
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), InterruptMask);
}

void Cdd_I2c_Enable_master(uint8 HwInstance){
    /**
     * Writes mask value to set SOC in master mode, free run 
     * and bring I2c out of reset
    */
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR), (CDD_I2C_MASTER_MODE | CDD_I2C_FREE_RUN | CDD_I2C_OUT_OF_RESET));
}

void Cdd_I2c_queue_init(Cdd_I2c_Queue_obj *pQueue){
    /**
     * Function initialises 
     * sequenceId = 255
     * pNext = NULL
     * for all queue locations
    */
    uint8 i = 0;
    
    for(i = 0;  i<CDD_I2C_MAX_QUEUE_SIZE;   i++){
        (pQueue + i)->sequenceId = 255;
        (pQueue + i)->pNext = NULL_PTR;
    }
}

void Cdd_I2c_Disable_master(uint8 HwInstance){
    /**
     * Puts I2c in reset mode
    */
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR), ~(CDD_I2C_OUT_OF_RESET));
}

void Cdd_I2c_Set_Slave_Addressing_mode(uint8 HwInstance, uint8 SlaveAddressScheme){
    uint16 regVal = 0;    
    /**
     * Writes the give SlaveAddressScheme either 0 or 1 to the 
     * MDR registers XA bit
    */
   regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR);
   HW_SET_FIELD16(regVal,CDD_I2C_ICMDR_XA,SlaveAddressScheme);
   HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR), regVal);
}

void Cdd_I2c_Set_Slave_Address(uint8 HwInstance, Cdd_I2c_AddressType SlaveAddress){
    /**
     * Writes the given 16 bit slave device address in the register
    */
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICSAR), SlaveAddress);
}

void Cdd_I2c_Enable_interrupts(uint8 HwInstance){
    uint16 regVal = 0;

    /**
     * All configurator selected interrupts will be enabled here
     * All bits in the IMR register will be manipulated
    */

    #if (STD_ON == CDD_I2C_ARBITRATION_LOSS) 
    /*Enable arbitration loss interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_AL, 1);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);
    #elif (STD_OFF == CDD_I2C_ARBITRATION_LOSS)
    /*Disable arbitration loss interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_AL, 0);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);
    #endif

    #if (STD_ON == CDD_I2C_TREAT_NACK_AS_ERROR)
    /*Enable NACK interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_NACK, 1);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);
    #elif (STD_OFF == CDD_I2C_TREAT_NACK_AS_ERROR)
    /*Disable NACK interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_NACK, 0);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);
    #endif

    /*Disable Register access ready interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_ARDY, 0);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);

    #if (STD_ON == CDD_I2C_POLLING_MODE)
    /*Disable Receive data ready interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_ICRRDY, 0);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);
    #elif (STD_OFF == CDD_I2C_POLLING_MODE)
    /*Enable Receive data ready interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_ICRRDY, 1);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);  
    #endif

    #if (STD_ON == CDD_I2C_POLLING_MODE)
    /*Disable Transmit Data ready interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_ICXRDY, 0);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);
    #elif (STD_OFF == CDD_I2C_POLLING_MODE)
    /*Enable Transmit Data ready interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_ICXRDY, 1);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);  
    #endif

    /*Disable stop condition interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_SCD, 0);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);

    /*Disable Address as Slave interrupt*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICIMR_AAS, 0);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICIMR), regVal);

}

void Cdd_I2c_Set_Master_Receive_Mode(uint8 HwInstance){
    uint16 regVal = 0;

    /**
     * Set Master in receive mode
    */
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICMDR_TRX, 0);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR), regVal);
}

void Cdd_I2c_Set_Master_Transmit_Mode(uint8 HwInstance){
    uint16 regVal = 0;

    /**
     * Set Master in transmit mode
    */
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICMDR_TRX, 1);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR), regVal);
}

void Cdd_I2c_Data_Count(uint8 HwInstance, uint16 DataLength){
    /**
     * Function call writes a 16 bit value to the register
     * Post completion of speified 2 byte data tranmssion/reception
     * stop signal is generated
    */
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICCNT), DataLength);
}

void Cdd_I2c_Start(uint8 HwInstance){
    uint16 regVal = 0;

    /*Set device back in master mode*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICMDR_MST, 1);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR), regVal);
    /*Generate start condition*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICMDR_STT, 1);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR), regVal);
}

uint16 Cdd_I2c_Isr_Check(uint8 HwInstance){
    volatile uint16 regVal = 0;
    /**
     * Read status register
    */
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICSTR);
    return regVal;
}

uint8 Cdd_I2c_Get_Sequence_Id(uint8 HwUnit){
    uint8 i = 0;
    uint8 j = 0;
    uint8 retVal = SEQUENCE_INITIALIZER;

    for(i = 0;  i<CDD_I2C_MAXIMUM_SEQUENCE_NUMBER;  i++){
        if((HwUnit == Cdd_I2cSequenceContainerLocal[i].HwInstance)   &&  (CDD_I2C_SEQ_PENDING == Cdd_I2cSequenceContainerLocal[i].SeqResult)){
            retVal = i;
            break;
        }
    }
    return retVal;
}

uint8 Cdd_I2c_Get_Channel_Id(uint8 HwUnit, uint8 SequenceId){
    uint8 i = 0;
    uint8 ChannelId = 0;
    uint8 retVal = CHANNEL_INITIALIZER;

    for(i = 0; i<Cdd_I2cSequenceContainerLocal[SequenceId].MaxChannels;  i++){
        ChannelId = Cdd_I2cSequenceContainerLocal[SequenceId].ChannelList[i];
        if((CDD_I2C_CH_RESULT_IN_PROGRESS == Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult)  &&  ((HwUnit == Cdd_I2cSequenceContainerLocal[SequenceId].HwInstance))){
            retVal = ChannelId;
            break;
        }
    }
    return retVal;
}

uint8 Cdd_I2c_Arbitration_Loss(uint8 ChannelId){
    Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_ARBFAIL;
    return (CDD_I2C_E_ARBITRATION_LOSS);
}

uint8 Cdd_I2c_No_Ack(uint8 ChannelId){
    Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_NACKFAIL;
    return (CDD_I2C_E_NACK);
}

uint8 Cdd_I2c_Shift_Register_Full(uint8 ChannelId){
    Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
    return (CDD_I2C_E_RECEIVE_SHIFT_REGISTER_FULL);
}

void Cdd_I2c_Stop(uint8 HwInstance){
    uint16 regVal = 0;

    /*Generate stop condition*/
    regVal = HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR);
    HW_SET_FIELD16(regVal, CDD_I2C_ICMDR_STP, 1);
    HW_WR_REG16((CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICMDR), regVal);
}

void Cdd_I2c_Enqueue(uint8 sequenceId){
    if(QueueCounter < CDD_I2C_MAX_QUEUE_SIZE){
        #if(STD_OFF == CDD_I2C_PROCESS_SAME_SEQUENCE_MULTIPLE_TIMES_IN_QUEUE)
        if(CDD_I2C_SEQ_QUEUED != Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult)
        #endif
        {
            if(QueueCounter > 0){
                (pCddI2cTransmissionQueue + QueueCounter)->sequenceId = sequenceId;
                (pCddI2cTransmissionQueue + (QueueCounter - 1))->pNext = (pCddI2cTransmissionQueue + QueueCounter);
                (pCddI2cTransmissionQueue + QueueCounter)->pNext = NULL_PTR; 
            }
            else if(QueueCounter == 0){
                (pCddI2cTransmissionQueue + QueueCounter)->sequenceId = sequenceId;
                (pCddI2cTransmissionQueue + QueueCounter)->pNext = NULL_PTR;
            }
            QueueCounter++;
            Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_QUEUED;
        }
    }
    else{
        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        Cdd_I2cSequenceContainer[sequenceId].SequenceErrorNotify(CDD_I2C_E_PARAM_QUEUE_FULL);
        /*Queue full error*/
        #endif
        #if(STD_OFF == CDD_I2C_PROCESS_SAME_SEQUENCE_MULTIPLE_TIMES_IN_QUEUE)
        if(CDD_I2C_SEQ_QUEUED != Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult)
        #endif
        {
        QueueCounter = 0;
        (pCddI2cTransmissionQueue + QueueCounter)->sequenceId = sequenceId;
        (pCddI2cTransmissionQueue + QueueCounter)->pNext = NULL_PTR;
        QueueCounter++;
        Cdd_I2cSequenceContainerLocal[sequenceId].SeqResult = CDD_I2C_SEQ_QUEUED;
        }
    } 
}

uint8 Cdd_I2c_Read(uint8 HwInstance){
    /*Read data received from the I2C bus*/
    return(HW_RD_REG16(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICDRR));
}

void Cdd_I2c_Write(uint8 HwInstance, uint8 Data){
    /*Write data passed to the register*/
    HW_WR_REG8(CddI2cHwUnitBaseAddr[HwInstance] + CDD_I2C_ICDXR, Data);
}

#if(STD_ON == CDD_I2C_POLLING_MODE)    /*#if(STD_ON == CDD_I2C_POLLING_MODE)*/
void Cdd_I2c_Polling_Mode_Receive(uint8 HwUnit, uint8 ChannelId, uint8 SequenceId){
    uint32 IsrStatus = 0;
    uint32 Timeout = 0;

    /*Update driver status*/
    CddI2cDrvState = CDD_I2C_BUSY;
    /*Function call to disable all interrupts*/
    Cdd_I2c_Disable_interrupts(HwUnit, (uint16)~(CDD_I2C_ALL_INTERRUPTS));
    /*Function call to clear all interrupts*/
    Cdd_I2c_Clear_interrupts(HwUnit, CDD_I2C_ALL_INTERRUPTS_CLEAR);
    /*Function call to set the master in receive mode*/
    Cdd_I2c_Set_Master_Receive_Mode(HwUnit);
    /*Function call to set number of bytes to read/write*/
    Cdd_I2c_Data_Count(HwUnit, Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength);   
    Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_IN_PROGRESS;
    /*Get initial value for timeout period*/
    (void) GetCounterValue(CDD_I2C_OS_COUNTER_ID,&Timeout);
    /*Wait till bus is free*/
    while(CDD_I2C_BUS_BUSY == Cdd_I2cBusCheck(HwUnit)){
        /*Get elapsed time value*/
        if(1 == (Cdd_I2c_Timeout(Timeout,CDD_I2C_RESET_TIMEOUT))){
            /*Call Cdd_I2c_SetHandling if bus is busy for more than specified time period*/
            Cdd_I2c_SetHandling(CddI2cArbitrationLossParam);
            Timeout = 0;
            Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
            Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
            break;
        }
    }
    Timeout = 0;
    if(CDD_I2C_SEQ_FAILED != Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult){
        /*Send start signal on the bus*/
        Cdd_I2c_Start(HwUnit);
        while(0 != Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength){
            /*Read status register*/
            IsrStatus = Cdd_I2c_Isr_Check(HwUnit);
            if(IsrStatus & CDD_I2C_ICSTR_ICRRDY_MASK){
                *(Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer) = Cdd_I2c_Read(HwUnit);
                Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer++;
                Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength--;
            }
            else if(IsrStatus & (CDD_I2C_ICSTR_AL_MASK | CDD_I2C_ICSTR_NACK_MASK | 
                   CDD_I2C_ICSTR_AD0_MASK | CDD_I2C_ICSTR_RSFULL_MASK)){
                    /*Function call to disable all interrupts*/
                    Cdd_I2c_Disable_interrupts(HwUnit, (uint16)~(CDD_I2C_ALL_INTERRUPTS));
                    /*Function call to send a stop signal on the I2C bus*/
                    Cdd_I2c_Stop(HwUnit);
                    if(NULL_PTR != Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify){
                        Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify(Cdd_I2c_Error_Handler(IsrStatus,ChannelId,HwUnit));
                    }
                    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                    else{
                        Cdd_I2cReportDetError(CDD_I2C_SID_POLLING_MODE_PROCESSING, CDD_I2C_E_PARAM_NOTIFY_CALLBACK);
                        /*Sequence error notify callback function passed as NULL_PTR*/
                    }
                    #endif
                    /*Function call to clear all interrupts*/
                    Cdd_I2c_Clear_interrupts(HwUnit, CDD_I2C_ALL_INTERRUPTS_CLEAR);
                    Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                    break;
                }
        }
        /*Function call to disable all interrupts*/
        Cdd_I2c_Disable_interrupts(HwUnit, (uint16)~(CDD_I2C_ALL_INTERRUPTS));
        /*Function call to send a stop signal on the I2C bus*/
        Cdd_I2c_Stop(HwUnit);
        /*Function call to clear all interrupts*/
        Cdd_I2c_Clear_interrupts(HwUnit, CDD_I2C_ALL_INTERRUPTS_CLEAR);
    }
    if(CDD_I2C_SEQ_FAILED != Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult){
        if((ChannelId == Cdd_I2cSequenceContainerLocal[SequenceId].ChannelList[Cdd_I2cSequenceContainerLocal[SequenceId].MaxChannels-1])    &&
                    (Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength == 0)){
                        Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_OK;
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                        if(NULL_PTR != Cdd_I2cSequenceContainer[SequenceId].SequenceCompleteNotify){
                            if(ChannelId == Cdd_I2cSequenceContainerLocal[SequenceId].ChannelList[Cdd_I2cSequenceContainerLocal[SequenceId].MaxChannels-1]){
                                Cdd_I2cSequenceContainer[SequenceId].SequenceCompleteNotify();
                                Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_OK;
                            }
                        }
                        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                        else{
                            Cdd_I2cReportDetError(CDD_I2C_SID_POLLING_MODE_PROCESSING, CDD_I2C_E_PARAM_NOTIFY_CALLBACK);
                            /*Sequence complete notify callback function passed as NULL_PTR*/
                        }
                        #endif
                    }
    }
    CddI2cDrvState = CDD_I2C_IDLE;
}
#endif  /*#if(STD_ON == CDD_I2C_POLLING_MODE)*/

uint8 Cdd_I2cBusCheck(uint8 HwInstance){
    uint16 status = 0;
    uint8 retVal = 0;
    
    status = Cdd_I2c_Isr_Check(HwInstance);
    if(0 != (status   &   CDD_I2C_ICSTR_BB_MASK)){
        retVal = CDD_I2C_BUS_BUSY;
    }
    else if(0 == (status   &   CDD_I2C_ICSTR_BB_MASK)){
        retVal = CDD_I2C_BUS_FREE;
    }
    return retVal;
}

uint8 Cdd_I2c_Error_Specify(uint32 ErrStatus, uint8 ChannelId){

    uint8 ErrorStatus = 0;

    /*Checking for Arbitration loss error*/
    if(ErrStatus & CDD_I2C_ICSTR_AL_MASK){
        ErrorStatus = Cdd_I2c_Arbitration_Loss(ChannelId);
    }
    /*Checking for No acknowledgement error*/
    else if(ErrStatus & CDD_I2C_ICSTR_NACK_MASK){
        ErrorStatus = Cdd_I2c_No_Ack(ChannelId);
    }
    else if(ErrStatus & CDD_I2C_ICSTR_RSFULL_MASK){
        ErrorStatus = Cdd_I2c_Shift_Register_Full(ChannelId);
    }
    return (ErrorStatus);
}

uint8 Cdd_I2c_Error_Handler(uint32 ErrStatus, uint8 ChannelId, uint8 HwUnit){

    uint8 ErrorStatus = 0;

    /*Check for all specific errors and update channel as well as sequence results*/
    ErrorStatus = Cdd_I2c_Error_Specify(ErrStatus, ChannelId);
    /*Function call to disable all interrupts*/
    Cdd_I2c_Disable_interrupts(HwUnit, (uint16)~(CDD_I2C_ALL_INTERRUPTS));
    /*Function call to send a stop signal on the I2C bus*/
    Cdd_I2c_Stop(HwUnit);
    /*Function call to clear all interrupts*/
    Cdd_I2c_Clear_interrupts(HwUnit, CDD_I2C_ALL_INTERRUPTS_CLEAR);

    return (ErrorStatus);
}

#if(STD_ON == CDD_I2C_POLLING_MODE)
void Cdd_I2c_PollingModeProcessingFromQueue(uint8 SequenceId){
    uint8 i = 0;
    uint8 ChannelId = 0;
    uint16 SlaveAddress = 0;
    uint8 HwUnit = 0;

    if((CDD_I2C_SEQ_FAILED != Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult)  &&
        (CDD_I2C_HW_UNIT_BUSY != CddI2cHwUnitStatus[Cdd_I2cSequenceContainerLocal[SequenceId].HwInstance]))
    {
        CddI2cHwUnitStatus[Cdd_I2cSequenceContainerLocal[SequenceId].HwInstance] = CDD_I2C_HW_UNIT_BUSY;
        Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_PENDING;
        for(i = 0;  i<Cdd_I2cSequenceContainerLocal[SequenceId].MaxChannels; i++){
            /*Find the channel ID number*/
            ChannelId = Cdd_I2cSequenceContainerLocal[SequenceId].ChannelList[i];
            if(CDD_I2C_READ == Cdd_I2cChannelContainerLocal[ChannelId].Dir){
                if(CDD_I2C_CH_RESULT_IN_PROGRESS == Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult){
                    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                    Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify(CDD_I2C_E_CHANNEL_BUSY);
                    /*Channel busy*/
                    #endif
                    Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                    break;
                }
                /*Update channel result*/
                Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_IN_PROGRESS;
                #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                if(NULL_PTR == Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer){
                    Cdd_I2cReportDetError(CDD_I2C_SID_POLLING_MODE_PROCESSING, CDD_I2C_E_PARAM_TRANS_BUFFER);
                    /*Reception function called before setupEB*/
                }
                #endif
                /*Find the Slave address*/
                SlaveAddress = Cdd_I2cChannelContainerLocal[ChannelId].SlaveAddress;
                /*Find HwUnit*/
                HwUnit = Cdd_I2cSequenceContainerLocal[SequenceId].HwInstance;
                /*Function call to set the addressing mode for the slave device*/
                Cdd_I2c_Set_Slave_Addressing_mode(HwUnit, Cdd_I2cChannelContainerLocal[ChannelId].SlaveAddressScheme);
                /*Function call to set the slave address when device in master mode*/
                Cdd_I2c_Set_Slave_Address(HwUnit, SlaveAddress);
                Cdd_I2c_Polling_Mode_Receive(HwUnit, ChannelId, SequenceId);
            }
        }
        CddI2cHwUnitStatus[Cdd_I2cSequenceContainerLocal[SequenceId].HwInstance] = CDD_I2C_HW_UNIT_FREE;
        if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult){
            Cdd_I2c_Enqueue(SequenceId);
        }
    }
}
#endif

#if(STD_OFF == CDD_I2C_POLLING_MODE)
void Cdd_I2c_Isr_Handler(uint8 HwUnit){

    uint32 IsrStatus = 0;
    uint8 ChannelId = 0;
    uint8 SequenceId = 0;
    uint32 Timeout = 0;

    /*Function call to check for errors*/
    IsrStatus = Cdd_I2c_Isr_Check(HwUnit);  
    SequenceId = Cdd_I2c_Get_Sequence_Id(HwUnit);
    ChannelId = Cdd_I2c_Get_Channel_Id(HwUnit, SequenceId);

    if((255 == ChannelId)   ||  ((255 == SequenceId))){
        return;
    }

    /*Check if any of the below error exists*/
    if(IsrStatus & (CDD_I2C_ICSTR_AL_MASK | CDD_I2C_ICSTR_NACK_MASK | 
                   CDD_I2C_ICSTR_AD0_MASK | CDD_I2C_ICSTR_AAS_MASK)){
        if(NULL_PTR != Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify){
            Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify(Cdd_I2c_Error_Handler(IsrStatus,ChannelId,HwUnit));
        }
        #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
        else{
            Cdd_I2cReportDetError(CDD_I2C_SID_ASYNC_TRANSMIT, CDD_I2C_E_PARAM_NOTIFY_CALLBACK);
            /*Sequence error notify callback function passed as NULL_PTR*/
        }
        #endif
        Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
        Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
        Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
    }
    else{
        /*Check if message is available for reading*/
        if(IsrStatus & CDD_I2C_ICSTR_ICRRDY_MASK){
            while(0 != Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength){
                /*Read status register*/
                IsrStatus = Cdd_I2c_Isr_Check(HwUnit);
                if(IsrStatus & CDD_I2C_ICSTR_ICRRDY_MASK){
                    *(Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer) = Cdd_I2c_Read(HwUnit);
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer++;
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength--;
                }
                else if(IsrStatus & (CDD_I2C_ICSTR_AL_MASK | CDD_I2C_ICSTR_NACK_MASK | CDD_I2C_ICSTR_AD0_MASK)){
                    /*Specify error cause*/
                    if(NULL_PTR != Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify){
                        Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify(Cdd_I2c_Error_Handler(IsrStatus,ChannelId,HwUnit));
                    }

                    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                    else{
                        Cdd_I2cReportDetError(CDD_I2C_SID_ASYNC_TRANSMIT, CDD_I2C_E_PARAM_NOTIFY_CALLBACK);
                        /*Sequence error notify callback function passed as NULL_PTR*/
                    }
                    #endif
                    Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                    break;
                }
            }
            if(CDD_I2C_SEQ_FAILED != Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult){
                /*Get initial value for timeout period*/
                (void) GetCounterValue(CDD_I2C_OS_COUNTER_ID,&Timeout);
                /*Check if bus is busy*/
                while(CDD_I2C_BUS_BUSY == Cdd_I2cBusCheck(HwUnit)){
                    /*Function call to disable all interrupts*/
                    Cdd_I2c_Disable_interrupts(HwUnit, (uint16)~(CDD_I2C_ALL_INTERRUPTS));
                    /*Function call to send a stop signal on the bus*/
                    Cdd_I2c_Stop(HwUnit);
                    /*Function call to clear all interrupts*/
                    Cdd_I2c_Clear_interrupts(HwUnit, CDD_I2C_ALL_INTERRUPTS_CLEAR);
                    if(CDD_I2C_BUS_BUSY == Cdd_I2cBusCheck(HwUnit)){
                        /*Get elapsed time value*/
                        if(1 == (Cdd_I2c_Timeout(Timeout,CDD_I2C_RESET_TIMEOUT))){
                            /*Call Cdd_I2c_SetHandling if bus is busy for more than specified time period*/
                            Cdd_I2c_SetHandling(CddI2cArbitrationLossParam);
                            Timeout = 0;
                            Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
                            Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                            break;
                        }
                    }
                }
                Timeout = 0;
                if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult){
                    Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                }
                else if(0 == Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength){
                    Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_OK;
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                    if(NULL_PTR != Cdd_I2cSequenceContainer[SequenceId].SequenceCompleteNotify){
                        if(ChannelId == Cdd_I2cSequenceContainerLocal[SequenceId].ChannelList[Cdd_I2cSequenceContainerLocal[SequenceId].MaxChannels-1]){
                            Cdd_I2cSequenceContainer[SequenceId].SequenceCompleteNotify();
                            if(CDD_I2C_SEQ_PENDING == Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult){
                                Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_OK;
                            }
                        }
                    }
                    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                    else{
                        Cdd_I2cReportDetError(CDD_I2C_SID_ASYNC_TRANSMIT, CDD_I2C_E_PARAM_NOTIFY_CALLBACK);
                        /*Sequence complete notify callback function passed as NULL_PTR*/
                    }
                    #endif
                }
            }
        }
        /*Check if message is available for writing*/
        else if(IsrStatus & CDD_I2C_ICSTR_ICXRDY_MASK){
            while(0 != Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength){
                /*Check for errors*/
                IsrStatus = Cdd_I2c_Isr_Check(HwUnit);
                if(IsrStatus & (CDD_I2C_ICSTR_AL_MASK | CDD_I2C_ICSTR_NACK_MASK | 
                CDD_I2C_ICSTR_AD0_MASK | CDD_I2C_ICSTR_RSFULL_MASK)){

                    if(NULL_PTR != Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify){
                        Cdd_I2cSequenceContainer[SequenceId].SequenceErrorNotify(Cdd_I2c_Error_Handler(IsrStatus, ChannelId, HwUnit));
                    }

                    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                    else{
                        Cdd_I2cReportDetError(CDD_I2C_SID_ASYNC_TRANSMIT, CDD_I2C_E_PARAM_NOTIFY_CALLBACK);
                        /*Sequence error notify callback function passed as NULL_PTR*/
                    }
                    #endif
                    Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                    break;
                }
                else if(IsrStatus & CDD_I2C_ICSTR_ICXRDY_MASK){
                    while(0 == ((Cdd_I2c_Isr_Check(HwUnit)) & CDD_I2C_ICSTR_ICXRDY_MASK)){
                        /*Do nothing*/
                    }
                        Cdd_I2c_Write(HwUnit, *(Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer));
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer++;
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength--;
                }
            }
            if(CDD_I2C_SEQ_FAILED != Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult){
                /*Get initial value for timeout period*/
                (void) GetCounterValue(CDD_I2C_OS_COUNTER_ID,&Timeout);
                /*Check if bus is busy*/
                while(CDD_I2C_BUS_BUSY == Cdd_I2cBusCheck(HwUnit)){
                    /*Function call to disable all interrupts*/
                    Cdd_I2c_Disable_interrupts(HwUnit, (uint16)~(CDD_I2C_ALL_INTERRUPTS));
                    /*Function call to send a stop signal on the bus*/
                    Cdd_I2c_Stop(HwUnit);
                    /*Function call to clear all interrupts*/
                    Cdd_I2c_Clear_interrupts(HwUnit, CDD_I2C_ALL_INTERRUPTS_CLEAR);
                    if(CDD_I2C_BUS_BUSY == Cdd_I2cBusCheck(HwUnit)){
                        /*Get elapsed time value*/
                        if(1 == (Cdd_I2c_Timeout(Timeout,CDD_I2C_RESET_TIMEOUT))){
                            /*Call Cdd_I2c_SetHandling if bus is busy for more than specified time period*/
                            Cdd_I2c_SetHandling(CddI2cArbitrationLossParam);
                            Timeout = 0;
                            Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
                            Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_FAILED;
                            break;
                        }
                    }
                }
                Timeout = 0;
                if(CDD_I2C_SEQ_FAILED == Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult){
                        Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_BUSFAIL;
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                        Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                    }
                else if(0 == Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength){
                    Cdd_I2cChannelContainerLocal[ChannelId].ChannelResult = CDD_I2C_CH_RESULT_OK;
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBufferLength = Cdd_I2cChannelContainer[ChannelId].DataBufferLength; 
                    Cdd_I2cChannelContainerLocal[ChannelId].DataBuffer = Cdd_I2cChannelContainer[ChannelId].DataBuffer;
                    if(NULL_PTR != Cdd_I2cSequenceContainer[SequenceId].SequenceCompleteNotify){
                        if(ChannelId == Cdd_I2cSequenceContainerLocal[SequenceId].ChannelList[Cdd_I2cSequenceContainerLocal[SequenceId].MaxChannels-1]){
                            Cdd_I2cSequenceContainer[SequenceId].SequenceCompleteNotify();
                            if(CDD_I2C_SEQ_PENDING == Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult){
                                Cdd_I2cSequenceContainerLocal[SequenceId].SeqResult = CDD_I2C_SEQ_OK;
                            }
                        }
                    }
                    #if (STD_ON == CDD_I2C_DEV_ERROR_DETECT)
                    else{
                        Cdd_I2cReportDetError(CDD_I2C_SID_ASYNC_TRANSMIT, CDD_I2C_E_PARAM_NOTIFY_CALLBACK);
                        /*Sequence complete notify callback function passed as NULL_PTR*/
                    }
                    #endif
                }
            }
        }
    }
}
#endif

#define CDD_I2C_STOP_SEC_CODE
#include "Cdd_I2c_MemMap.h"