/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     Cdd_Ipc.c
 *
 *  \brief    This file contains Complex Device Driver for IPC
 *
 */
/*  -------------------------------------------------------------------------------------------------------------------
*  FILE DESCRIPTION
*  -------------------------------------------------------------------------------------------------------------------
*         File:  Cdd_Ipc.c
*    Component:  AM263x Complex Device Driver
*       Module:  IPC
*    Generator:  None
*
*  Description: This component provides services for initialization and control of the IPC between different core in the this
*                     Multicore SoC.
*********************************************************************************************************************/
/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "Cdd_Ipc.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* AUTOSAR version information check has to match definition in header file */
#if ((CDD_IPC_AR_RELEASE_MAJOR_VERSION!= (4U)) || \
    (CDD_IPC_AR_RELEASE_MINOR_VERSION != (3U)) ||  \
    (CDD_IPC_AR_RELEASE_REVISION_VERSION != (1U)))
    #error "CDD: AUTOSAR Version Numbers of CDD IPC are different!!"
#endif

/* AUTOSAR version information check has to match definition in header file */
#if ((CDD_IPC_SW_MAJOR_VERSION != (8U)) || \
    (CDD_IPC_SW_MINOR_VERSION != (6U)) )
    #error "CDD: Software Version Numbers are inconsistent!!"
#endif

/* Shared memory check to ensure it is within bounds */
#if (CDD_IPC_SHARED_MEMORY_SIZE >16000)
    #error "CDD IPC:Shared memory size exceeded"
#endif


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
/** \brief CDD IPC handles */
#define CDD_IPC_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Cdd_Ipc_MemMap.h"

IpcNotify_Object CddIpc_NotifyObject;
IpcNotify_InitObject CddIpcInit_Config;
IpcNotifyUtils_Object CddIpc_NotifyUtilsObject;
IpcNotifyUtils_Handle CddIpc_NotifyUtilsHandle = &CddIpc_NotifyUtilsObject;
IpcNotifyUtils_InitObject CddIpc_NotifyUtilsInitObject;

#if(CDD_IPC_RPMSG_ENABLE_API == STD_ON)
RPMessageLLD_Object Cdd_RPMessageObj;

RPMessage_EpObject Cdd_AckReplyMsgObject[CDD_IPC_LOCALEP_COUNT];

RPMessage_CreateParams createParams;

#endif

#define CDD_IPC_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Cdd_Ipc_MemMap.h"

#define CDD_IPC_START_SEC_VAR_INIT_UNSPECIFIED
#include "Cdd_Ipc_MemMap.h"

IpcNotify_Handle CddIpc_NotifyHandle = &CddIpc_NotifyObject;

#if(CDD_IPC_RPMSG_ENABLE_API == STD_ON)
RPMessageLLD_Handle CddIpc_RPMsgHandle = &Cdd_RPMessageObj;
#endif

extern RPMessageLLD_InitObject CddIpc_RPMessageInitObj;

/** \brief CDD IPC driver status */
static VAR(Cdd_IpcStatusType, CDD_VAR_ZERO_INIT) CddIpc_DrvStatus = CDD_IPC_UNINIT;

#define CDD_IPC_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Cdd_Ipc_MemMap.h"


/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#define CDD_IPC_START_SEC_CODE
#include "Cdd_Ipc_MemMap.h"

uint32 Clock_getTicks(void)
{
    TickType startCount = 0U;
    (void)GetCounterValue(CDD_IPC_OS_COUNTER_ID, &startCount);
    return ((uint32)startCount);
}

void Clock_uSleep(uint32 usec) __attribute__((optnone))
{
    volatile int i = 0;
    for(int i =0; i<usec*400;i++){}; /* Here 400 is the number of uSeconds per ticks*/
}

uint32 Clock_ticksToUsec(uint32 ticks)
{
    return (uint32)(ticks * 1000);
}
 
#if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
static inline void Cdd_IpcReportDetError(uint8 apiId, uint8 errorId)
{
    (void) Det_ReportError(CDD_IPC_MODULE_ID, CDD_IPC_INSTANCE_ID, apiId, errorId);
    return;
}
#endif  /* #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT) */


#if (STD_ON == CDD_IPC_VERSION_INFO_API)
FUNC(void, CDD_IPC_CODE) Cdd_Ipc_GetVersionInfo(P2VAR(Std_VersionInfoType, AUTOMATIC, CDD_IPC_APPL_DATA) VersionInfoPtr)
{
#if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
    if (NULL_PTR == VersionInfoPtr)
    {
        Cdd_IpcReportDetError(CDD_IPC_GETVERSIONINFO_SERVICE_ID, CDD_IPC_E_PARAM_POINTER);
    }
    else
#endif  /* #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT) */
    {
        VersionInfoPtr->vendorID         = CDD_IPC_VENDOR_ID;
        VersionInfoPtr->moduleID         = CDD_IPC_MODULE_ID;
        VersionInfoPtr->sw_major_version = (uint8) CDD_IPC_SW_MAJOR_VERSION;
        VersionInfoPtr->sw_minor_version = (uint8) CDD_IPC_SW_MINOR_VERSION;
        VersionInfoPtr->sw_patch_version = (uint8) CDD_IPC_SW_PATCH_VERSION;
    }

    return;
}
#endif  /* #if (STD_ON == CDD_IPC_VERSION_INFO_API) */


FUNC(void, CDD_IPC_CODE) Cdd_Ipc_Init(P2CONST(Cdd_IpcConfigType, AUTOMATIC, CDD_IPC_CFG) ConfigPtr)
{

    uint32 exitCondition = FALSE;
#if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
    if (CDD_IPC_UNINIT != CddIpc_DrvStatus)
    {
        Cdd_IpcReportDetError(CDD_IPC_INIT_SERVICE_ID, CDD_IPC_E_ALREADY_INITIALIZED);
    }
    else if (NULL_PTR == ConfigPtr || NULL == ConfigPtr)
    {
        Cdd_IpcReportDetError(CDD_IPC_INIT_SERVICE_ID, CDD_IPC_E_PARAM_POINTER);
    }
    else if (ConfigPtr-> Cdd_Ipc_SelfCoreId > CDD_IPC_CORE_ID_MAX || ConfigPtr->Cdd_Ipc_numCores > CDD_IPC_CORE_ID_MAX)
    {
        Cdd_IpcReportDetError(CDD_IPC_INIT_SERVICE_ID, CDD_IPC_E_PARAM_VALUE);
    }
    else if (ConfigPtr->Cdd_Ipc_numCores < CDD_IPC_CORE_ID_MAX)
    {
        for(int i=0; i<ConfigPtr->Cdd_Ipc_numCores;i++)
        {
            if(ConfigPtr->Cdd_Ipc_coreIdList[i] > CDD_IPC_CORE_ID_MAX )
            {
                Cdd_IpcReportDetError(CDD_IPC_INIT_SERVICE_ID, CDD_IPC_E_PARAM_VALUE); 
            }
            else
            {
                exitCondition = TRUE;
            }
        }
    }
    if(TRUE == exitCondition)
#endif
    {
        sint32 initStatus = CDD_IPC_UNINIT;

        CddIpcInit_Config.selfCoreId = ConfigPtr->Cdd_Ipc_SelfCoreId;
        CddIpcInit_Config.numCores = ConfigPtr->Cdd_Ipc_numCores;
        for(int i=0;i<ConfigPtr->Cdd_Ipc_numCores;i++)
        {
            CddIpcInit_Config.coreIdList[i] = ConfigPtr->Cdd_Ipc_coreIdList[i];
        }
        
        CddIpc_NotifyHandle->hIpcNotifyInit = &CddIpcInit_Config;
        CddIpc_NotifyUtilsInitObject.hIpcNotify = CddIpc_NotifyHandle;
        CddIpc_NotifyUtilsHandle->hIpcNotifyUtilsInit = &CddIpc_NotifyUtilsInitObject;

        initStatus = IpcNotify_lld_init(CddIpc_NotifyHandle);
        initStatus = IpcNotifyUtils_lld_init(CddIpc_NotifyUtilsHandle);


        #if(CDD_IPC_RPMSG_ENABLE_API == STD_ON) /* Initialize the RpMsg API's only if it is enabled */
            CddIpc_RPMsgHandle->hRpMsgInit = &CddIpc_RPMessageInitObj;
            initStatus = RPMessage_lld_init(CddIpc_RPMsgHandle);
        #endif
    #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
        if((E_OK) == initStatus)
        {
            CddIpc_DrvStatus = CDD_IPC_IDLE;
        }
        else
        {
            Cdd_IpcReportDetError(CDD_IPC_INIT_SERVICE_ID, CDD_IPC_E_INIT_FAILED);
        }
    #endif
    }
    return ;
}

FUNC(void, CDD_IPC_CODE) Cdd_Ipc_Notify_Sync_All(void)
{
    IpcNotifyUtils_lld_syncAll(CddIpc_NotifyUtilsHandle, MCAL_SystemP_TIMEOUT);
}

#if (STD_ON == CDD_IPC_DEINIT_API)
FUNC(void, CDD_IPC_CODE) Cdd_Ipc_DeInit(P2CONST(Cdd_IpcConfigType, AUTOMATIC, CDD_IPC_CFG) ConfigPtr)
{

    Std_ReturnType deInitStatus = (Std_ReturnType)E_NOT_OK;
    uint32 exitCondition = FALSE;
#if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
    if ((CDD_IPC_UNINIT == CddIpc_DrvStatus))
    {
        Cdd_IpcReportDetError(CDD_IPC_DEINIT_SERVICE_ID, CDD_IPC_E_UNINIT);
    }
    else if  (NULL_PTR == ConfigPtr)
    {
        Cdd_IpcReportDetError(CDD_IPC_DEINIT_SERVICE_ID, CDD_IPC_E_PARAM_POINTER);
    }
    else if (ConfigPtr-> Cdd_Ipc_SelfCoreId > CDD_IPC_CORE_ID_MAX || ConfigPtr->Cdd_Ipc_numCores > CDD_IPC_CORE_ID_MAX)
    {
        Cdd_IpcReportDetError(CDD_IPC_DEINIT_SERVICE_ID, CDD_IPC_E_PARAM_VALUE);
    }
    else if (ConfigPtr->Cdd_Ipc_numCores < CDD_IPC_CORE_ID_MAX)
    {
        for(int i=0; i<ConfigPtr->Cdd_Ipc_numCores;i++)
        {
            if(ConfigPtr->Cdd_Ipc_coreIdList[i] > CDD_IPC_CORE_ID_MAX)
            {
                Cdd_IpcReportDetError(CDD_IPC_DEINIT_SERVICE_ID, CDD_IPC_E_PARAM_VALUE); 
            }
            else
            {
                exitCondition = TRUE;
            }
        }
    }
    if(TRUE == exitCondition)
    {
#endif

        CddIpcInit_Config.selfCoreId = ConfigPtr->Cdd_Ipc_SelfCoreId;
        CddIpcInit_Config.numCores = ConfigPtr->Cdd_Ipc_numCores;
        for(int i=0;i<ConfigPtr->Cdd_Ipc_numCores;i++)
        {
            CddIpcInit_Config.coreIdList[i] = ConfigPtr->Cdd_Ipc_coreIdList[i];
        }
        
        CddIpc_NotifyHandle->hIpcNotifyInit = &CddIpcInit_Config;

        deInitStatus = IpcNotify_lld_deInit(CddIpc_NotifyHandle);

        #if(CDD_IPC_RPMSG_ENABLE_API == STD_ON)
        CddIpc_RPMsgHandle->hRpMsgInit = &CddIpc_RPMessageInitObj;
        deInitStatus = RPMessage_lld_deInit(CddIpc_RPMsgHandle);
        #endif

        #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
        if((E_OK) == deInitStatus)
        {
            CddIpc_DrvStatus = CDD_IPC_UNINIT;
        }
        else
        {
            Cdd_IpcReportDetError(CDD_IPC_DEINIT_SERVICE_ID, CDD_IPC_E_DEINIT_FAILED);
        }
        
    }
        #endif

    return;
}
#endif  /* #if (STD_ON == CDD_IPC_DEINIT_API) */



FUNC(Std_ReturnType, CDD_IPC_CODE) Cdd_Ipc_Notify_RegisterClient(uint32 Cdd_Ipc_localClientId, IpcNotify_FxnCallback Cdd_Ipc_notify_msgHandler, P2VAR(void,AUTOMATIC,CDD_IPC_APPL_DATA)args)
{
    sint32 status = MCAL_SystemP_FAILURE;
    Std_ReturnType retVal = (Std_ReturnType)E_OK;
#if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
    if (CDD_IPC_UNINIT == CddIpc_DrvStatus)
    {
        retVal = (Std_ReturnType) E_NOT_OK;
        Cdd_IpcReportDetError(CDD_IPC_NOTIFY_REGISTER_CLIENT_SERVICE_ID, CDD_IPC_E_UNINIT);
    }
    else if(Cdd_Ipc_notify_msgHandler == NULL_PTR)
    {
        retVal = (Std_ReturnType) E_NOT_OK;
        Cdd_IpcReportDetError(CDD_IPC_NOTIFY_REGISTER_CLIENT_SERVICE_ID, CDD_IPC_E_PARAM_POINTER);
    }
    else if(Cdd_Ipc_localClientId < 0 || Cdd_Ipc_localClientId > CDD_IPC_MAX_NOTIFY_CLIENT_ID_PER_CORE)
    {
        retVal = (Std_ReturnType) E_NOT_OK;
        Cdd_IpcReportDetError(CDD_IPC_NOTIFY_REGISTER_CLIENT_SERVICE_ID, CDD_IPC_E_PARAM_VALUE);
    }
    else
#endif
    {
        status = IpcNotify_lld_registerClient(CddIpc_NotifyHandle, Cdd_Ipc_localClientId, Cdd_Ipc_notify_msgHandler, 0);
        if(MCAL_SystemP_SUCCESS != status)
        {
            retVal = (Std_ReturnType) E_NOT_OK;
            #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
            Cdd_IpcReportDetError(CDD_IPC_NOTIFY_REGISTER_CLIENT_SERVICE_ID, CDD_IPC_E_REGISTER_CLIENT_FAILED);
            #endif
        }
    }
    return (retVal);
}



FUNC(Std_ReturnType, CDD_IPC_CODE) Cdd_Ipc_Notify_UnregisterClient(uint32 Cdd_Ipc_localClientId)
{
    sint32 status = MCAL_SystemP_FAILURE;
    Std_ReturnType retVal = (Std_ReturnType)E_OK;
#if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
    if (CDD_IPC_UNINIT == CddIpc_DrvStatus)
    {
        retVal = (Std_ReturnType) E_NOT_OK;
        Cdd_IpcReportDetError(CDD_IPC_NOTIFY_UNREGISTER_CLIENT_SERVICE_ID, CDD_IPC_E_UNINIT);
    }
    else if(Cdd_Ipc_localClientId < 0 || Cdd_Ipc_localClientId > CDD_IPC_MAX_NOTIFY_CLIENT_ID_PER_CORE)
    {
        retVal = (Std_ReturnType) E_NOT_OK;
        Cdd_IpcReportDetError(CDD_IPC_NOTIFY_UNREGISTER_CLIENT_SERVICE_ID, CDD_IPC_E_PARAM_VALUE);
    }
    else
#endif
    {
        status = IpcNotify_lld_unregisterClient(CddIpc_NotifyHandle, Cdd_Ipc_localClientId);
        if( MCAL_SystemP_SUCCESS != status)
        {
            retVal = (Std_ReturnType) E_NOT_OK;
            #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
            Cdd_IpcReportDetError(CDD_IPC_NOTIFY_UNREGISTER_CLIENT_SERVICE_ID, CDD_IPC_E_UNREGISTER_CLIENT_FAILED);
            #endif
        }
    }
    return (retVal);
}

#if (STD_ON == CDD_IPC_WRITE_API)
FUNC(Std_ReturnType, CDD_IPC_CODE) Cdd_Ipc_Notify_Write(uint32 Cdd_Ipc_remoteCoreId,uint16 Cdd_Ipc_remoteClientId, uint32 Cdd_Ipc_msgValue, uint32 Cdd_Ipc_waitForFifoNotFull)
{
    sint32 status = MCAL_SystemP_FAILURE;
    Std_ReturnType retVal = (Std_ReturnType)E_OK;
#if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
    if (CDD_IPC_UNINIT == CddIpc_DrvStatus)
    {
        retVal = (Std_ReturnType) E_NOT_OK;
        Cdd_IpcReportDetError(CDD_IPC_WRITEDATA_SERVICE_ID, CDD_IPC_E_UNINIT);
    }
    else if((Cdd_Ipc_remoteCoreId > CDD_IPC_CORE_ID_MAX) || (Cdd_Ipc_remoteClientId > CDD_IPC_MAX_NOTIFY_CLIENT_ID_PER_CORE) || (Cdd_Ipc_remoteClientId < 0) || (Cdd_Ipc_waitForFifoNotFull > 1) || (Cdd_Ipc_waitForFifoNotFull < 0))
    {
        retVal = (Std_ReturnType) E_NOT_OK;
        Cdd_IpcReportDetError(CDD_IPC_WRITEDATA_SERVICE_ID, CDD_IPC_E_PARAM_VALUE);
    }
    else
#endif
    {

        status = IpcNotify_lld_sendMsg(CddIpc_NotifyHandle, Cdd_Ipc_remoteCoreId, Cdd_Ipc_remoteClientId, Cdd_Ipc_msgValue, Cdd_Ipc_waitForFifoNotFull, (CDD_IPC_TIMEOUT) );
       
        if( MCAL_SystemP_SUCCESS != status)
        {
            retVal = (Std_ReturnType)E_NOT_OK;
            #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
            Cdd_IpcReportDetError(CDD_IPC_WRITEDATA_SERVICE_ID, CDD_IPC_E_WRITE_FAILED);
            #endif
        }
        
    }
    
    return (retVal);
}
#endif  /* #if (STD_ON == CDD_IPC_WRITE_API) */


#if(CDD_IPC_RPMSG_ENABLE_API == STD_ON)
    FUNC(void, CDD_IPC_CODE) Cdd_Ipc_Construct(P2CONST(Cdd_IpcConfigType, AUTOMATIC, CDD_IPC_CFG) ConfigPtr)
    {
        Std_ReturnType status = (Std_ReturnType)E_NOT_OK;
        uint32 exitCondition = FALSE;
    #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
        if (CDD_IPC_UNINIT == CddIpc_DrvStatus)
        {
            Cdd_IpcReportDetError(CDD_IPC_CONSTRUCT_SERVICE_ID, CDD_IPC_E_UNINIT);
        }
        else if(ConfigPtr == NULL_PTR)
        {
            Cdd_IpcReportDetError(CDD_IPC_CONSTRUCT_SERVICE_ID, CDD_IPC_E_PARAM_POINTER);
        }
        else if(CDD_IPC_LOCALEP_COUNT > 0)
        {
            for(int i=0;i<CDD_IPC_LOCALEP_COUNT;i++)
            {
                uint32 localendPoint_check = ConfigPtr->Cdd_Ipc_RpMsgParams[i].Cdd_Ipc_localEndPt;
                if(localendPoint_check > 64 || localendPoint_check < 0)
                {
                    Cdd_IpcReportDetError(CDD_IPC_CONSTRUCT_SERVICE_ID, CDD_IPC_E_PARAM_VALUE);
                }
                else
                {
                    exitCondition = TRUE;
                }
            }
        }
        if(TRUE == exitCondition) 
    #endif
        {
            for(uint32 i=0U ;i<CDD_IPC_LOCALEP_COUNT;i++)
            {
                RPMessage_lld_CreateParams_init(&createParams);
                createParams.localEndPt = ConfigPtr->Cdd_Ipc_RpMsgParams[i].Cdd_Ipc_localEndPt;
                status = RPMessage_lld_construct(CddIpc_RPMsgHandle,&Cdd_AckReplyMsgObject[i],&createParams);
                #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
                if(((Std_ReturnType)E_OK) != status)
                {
                    Cdd_IpcReportDetError(CDD_IPC_CONSTRUCT_SERVICE_ID, CDD_IPC_E_CONSTRUCT_FAILED);
                }
                #endif
            }
        }
        
        return;
    }

    FUNC(void, CDD_IPC_CODE) Cdd_Ipc_Destruct(uint16 Cdd_Ipc_localEndPtId)
    {
    #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
        if (CDD_IPC_UNINIT == CddIpc_DrvStatus)
        {
            Cdd_IpcReportDetError(CDD_IPC_DESTRUCT_SERVICE_ID, CDD_IPC_E_UNINIT);
        }
        else if((Cdd_Ipc_localEndPtId < 0) || (Cdd_Ipc_localEndPtId > CDD_IPC_MAX_ENDPOINTS_PER_CORE))
        {
            Cdd_IpcReportDetError(CDD_IPC_DESTRUCT_SERVICE_ID, CDD_IPC_E_PARAM_VALUE);
        }
        else
    #endif
        {
            RPMessage_lld_destruct(CddIpc_RPMsgHandle,&Cdd_AckReplyMsgObject[Cdd_Ipc_localEndPtId]);
        }
        return;
    }


    FUNC(Std_ReturnType, CDD_IPC_CODE) Cdd_Ipc_RpMsg_SendMsg( void* data, uint16 dataLen, uint16 Cdd_Ipc_remoteCoreId, uint16 Cdd_Ipc_remoteEndPt, uint16 Cdd_Ipc_localEndPtId, uint32 timeout)
    {
        sint32 status = MCAL_SystemP_FAILURE;
        Std_ReturnType retVal = (Std_ReturnType)E_OK;
    #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
        if (CDD_IPC_UNINIT == CddIpc_DrvStatus)
        {
            retVal = (Std_ReturnType) E_NOT_OK;
            Cdd_IpcReportDetError(CDD_IPC_RPMSG_WRITE_SERVICE_ID, CDD_IPC_E_UNINIT);
        }
        else if( (CddIpc_RPMsgHandle == 0) || (data == NULL_PTR) || (dataLen <= 0) || (Cdd_Ipc_localEndPtId < 0 ) || (Cdd_Ipc_localEndPtId > CDD_IPC_MAX_ENDPOINTS_PER_CORE) || (Cdd_Ipc_remoteCoreId > CDD_IPC_CORE_ID_MAX) || (Cdd_Ipc_remoteEndPt < 0) || (Cdd_Ipc_remoteEndPt > CDD_IPC_MAX_ENDPOINTS_PER_CORE))
        {
            retVal = (Std_ReturnType) E_NOT_OK;
            Cdd_IpcReportDetError(CDD_IPC_RPMSG_WRITE_SERVICE_ID, CDD_IPC_E_PARAM_VALUE);
        }
        else
    #endif
        {
            status = RPMessage_lld_send(CddIpc_RPMsgHandle, data, dataLen, Cdd_Ipc_remoteCoreId, Cdd_Ipc_remoteEndPt, CfgPtr->Cdd_Ipc_RpMsgParams[Cdd_Ipc_localEndPtId].Cdd_Ipc_localEndPt , timeout);
            
            if(MCAL_SystemP_SUCCESS != status)
            {
                retVal = (Std_ReturnType)E_NOT_OK;
                #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
                Cdd_IpcReportDetError(CDD_IPC_RPMSG_WRITE_SERVICE_ID, CDD_IPC_E_RPMSG_WRITE_FAILED);
                #endif
            }
            
        }

        return (retVal);
    }

    FUNC(Std_ReturnType, CDD_IPC_CODE) Cdd_Ipc_RpMsg_RecvMsg(void* data, uint16* dataLen, uint16 Cdd_Ipc_localEndPtId ,uint16* Cdd_Ipc_remoteCoreId, uint16* Cdd_Ipc_remoteEndPt, uint32 timeout)
    {
        sint32 status = MCAL_SystemP_FAILURE;
        Std_ReturnType retVal = (Std_ReturnType)E_OK;
    #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
        if (CDD_IPC_UNINIT == CddIpc_DrvStatus)
        {
            retVal = (Std_ReturnType) E_NOT_OK;
            Cdd_IpcReportDetError(CDD_IPC_RPMSG_READ_SERVICE_ID, CDD_IPC_E_UNINIT);
        }
        else if((CddIpc_RPMsgHandle == 0) || (data == NULL_PTR) || (dataLen == NULL_PTR) || (Cdd_Ipc_localEndPtId <0) || (Cdd_Ipc_localEndPtId > CDD_IPC_MAX_ENDPOINTS_PER_CORE) || (Cdd_Ipc_remoteCoreId == NULL_PTR) || (Cdd_Ipc_remoteEndPt == NULL_PTR))
        {
            retVal = (Std_ReturnType) E_NOT_OK;
            Cdd_IpcReportDetError(CDD_IPC_RPMSG_READ_SERVICE_ID, CDD_IPC_E_PARAM_VALUE);
        }
        else
    #endif
        {
            status = RPMessage_lld_recv(CddIpc_RPMsgHandle, &Cdd_AckReplyMsgObject[Cdd_Ipc_localEndPtId], data, dataLen, Cdd_Ipc_remoteCoreId, Cdd_Ipc_remoteEndPt, timeout);
            
            if(MCAL_SystemP_SUCCESS != status)
            {
                retVal = (Std_ReturnType)E_NOT_OK;
                #if (STD_ON == CDD_IPC_DEV_ERROR_DETECT)
                Cdd_IpcReportDetError(CDD_IPC_RPMSG_READ_SERVICE_ID, CDD_IPC_E_RPMSG_READ_FAILED);
                #endif
            }
            
        }
        
        return (retVal);
    }

    void Cdd_Ipc_Mutex_resourceLock(uint32* lock)
    {
        while(*lock == LOCK);
    }

    void Cdd_Ipc_Mutex_resourceUnlock(uint32* lock)
    {
        *lock = UNLOCK;
    }

    uint32 Cdd_Ipc_Mutex_resourceTryLock(uint32* lock)
    {
        if(*lock == LOCK)
        {
            return LOCK;
        }
        else
        {
            *lock = LOCK;
            return UNLOCK;
        }
    }

#endif


#define CDD_IPC_STOP_SEC_CODE
#include "Cdd_Ipc_MemMap.h"
