#include <ipc_notify_v1_lld.h>
#include <ipc_notify_v1_cfg_lld.h>
#include <Cdd_Ipc.h> //Included to bring the Clock related functions
#include <string.h>

#define CDD_IPC_START_SEC_CODE
#include "Cdd_Ipc_MemMap.h"

static inline void IpcNotify_getWriteMailbox(uint32 selfCoreId, uint32 remoteCoreId, uint32 *mailboxBaseAddr, uint32 *intrBitPos, IpcNotify_SwQueue **swQ)
{
    IpcNotify_MailboxConfig *pMailboxConfig;

    pMailboxConfig = IpcNotifyCfg_getMailboxConfig(selfCoreId, remoteCoreId);

    *mailboxBaseAddr = pMailboxConfig->writeDoneMailboxBaseAddr;
    *intrBitPos = pMailboxConfig->intrBitPos;
    *swQ = pMailboxConfig->swQ;
}

static inline void IpcNotify_getReadMailbox(uint32 selfCoreId, uint32 *mailboxBaseAddr)
{
    IpcNotify_MailboxConfig *pMailboxConfig;

    pMailboxConfig = IpcNotifyCfg_getMailboxConfig(selfCoreId, selfCoreId);

    *mailboxBaseAddr = pMailboxConfig->readReqMailboxBaseAddr;
}

static inline void IpcNotify_getReadSwQ(uint32 selfCoreId, uint32 remoteCoreId, IpcNotify_SwQueue **swQ)
{
    IpcNotify_MailboxConfig *pMailboxConfig;

    pMailboxConfig = IpcNotifyCfg_getMailboxConfig(remoteCoreId, selfCoreId);

    *swQ = pMailboxConfig->swQ;
}

sint32 IpcNotify_lld_init(IpcNotify_Handle hIpcNotify)
{
    uint32                clientId, core, intConfigNum;
    uint32                mailboxBaseAddr;
    sint32                 status = MCAL_SystemP_SUCCESS;
    uint32                selfCoreId;
    IpcNotify_InitHandle    hIpcNotifyInit;

    if((hIpcNotify != NULL) && (hIpcNotify->hIpcNotifyInit != NULL))
    {
        hIpcNotifyInit = hIpcNotify->hIpcNotifyInit;
        selfCoreId = hIpcNotifyInit->selfCoreId;

        LLD_PARAMS_CHECK(selfCoreId < MCAL_CSL_CORE_ID_MAX);

        hIpcNotify->interruptConfig = IpcNotifyCfg_getInterruptConfig(selfCoreId);
        hIpcNotify->interruptConfigNum = IpcNotifyCfg_getInterruptConfigNum(selfCoreId);

        for(clientId=0; clientId<IPC_NOTIFY_CLIENT_ID_MAX; clientId++)
        {
            IpcNotify_lld_unregisterClient(hIpcNotify, clientId);
        }
        for(core=0; core<MCAL_CSL_CORE_ID_MAX; core++)
        {
            hIpcNotify->isCoreEnabled[core] = 0;
        }
        /* check parameters and config and assert if invalid */
        for(core=0; core<hIpcNotifyInit->numCores; core++)
        {
            /* mark core as enabled for IPC */
            hIpcNotify->isCoreEnabled[hIpcNotifyInit->coreIdList[core]] = 1;
        }
        for(intConfigNum=0; intConfigNum<hIpcNotify->interruptConfigNum; intConfigNum++)
        {
            IpcNotify_InterruptConfig *pInterruptConfig;

            pInterruptConfig = &hIpcNotify->interruptConfig[intConfigNum];

            IpcNotify_getReadMailbox(selfCoreId, &mailboxBaseAddr);

            if(pInterruptConfig->clearIntOnInit)
            {
                IpcNotify_mailboxClearAllInt(mailboxBaseAddr);
            }
        }
    }
    else
    {
        status = MCAL_SystemP_INVALID_PARAM;
    }

    return status;
}

sint32 IpcNotify_lld_deInit(IpcNotify_Handle hIpcNotify)
{
    uint32 clientId;
    uint32 mailboxBaseAddr;
    sint32  status = MCAL_SystemP_SUCCESS;

    if(hIpcNotify != NULL)
    {
        for(clientId=0; clientId<IPC_NOTIFY_CLIENT_ID_MAX; clientId++)
        {
            IpcNotify_lld_unregisterClient(hIpcNotify, clientId);
        }

        IpcNotify_getReadMailbox(hIpcNotify->hIpcNotifyInit->selfCoreId, &mailboxBaseAddr);
        IpcNotify_mailboxClearAllInt(mailboxBaseAddr);
    }
    else
    {
        status = MCAL_SystemP_INVALID_PARAM;
    }

    return status;
}

static inline uint32 IpcNotify_makeMsg(uint16 clientId, uint32 msgValue)
{
    return ((clientId & (IPC_NOTIFY_CLIENT_ID_MAX-1)) << IPC_NOTIFY_CLIENT_ID_SHIFT) |
            (msgValue & (IPC_NOTIFY_MSG_VALUE_MAX-1))
            ;
}

sint32 IpcNotify_lld_sendMsg(IpcNotify_Handle hIpcNotify, uint32 remoteCoreId, uint16 remoteClientId, uint32 msgValue, uint32 waitForFifoNotFull, uint32 timeout)
{
    uint32 mailboxBaseAddr, intrBitPos;
    IpcNotify_SwQueue *swQ;
    sint32 status = MCAL_SystemP_SUCCESS;
    uint32 elapsedTicks, startTicks;

    if(hIpcNotify != NULL)
    {
        if((remoteCoreId < MCAL_CSL_CORE_ID_MAX )&& (remoteClientId < IPC_NOTIFY_CLIENT_ID_MAX))
        {
            if(hIpcNotify->isCoreEnabled[remoteCoreId])
            {
                uint32 value = IpcNotify_makeMsg(remoteClientId, msgValue);

                IpcNotify_getWriteMailbox(hIpcNotify->hIpcNotifyInit->selfCoreId, remoteCoreId, &mailboxBaseAddr, &intrBitPos, &swQ);

                startTicks = Clock_getTicks();
                do
                {
                    status = IpcNotify_mailboxWrite(mailboxBaseAddr, intrBitPos, swQ, value);
                    elapsedTicks = Clock_getTicks() - startTicks;
                } while(status != MCAL_SystemP_SUCCESS  && waitForFifoNotFull && (elapsedTicks < timeout));

                if(status != MCAL_SystemP_SUCCESS)
                {
                    status = MCAL_SystemP_TIMEOUT;
                }
            }
            else
            {
                status = MCAL_SystemP_FAILURE;
            }
        }
        else
        {
            status = MCAL_SystemP_INVALID_PARAM;
        }
    }
    else
    {
        status = MCAL_SystemP_INVALID_PARAM;
    }

    return status;
}

sint32 IpcNotify_lld_registerClient(IpcNotify_Handle hIpcNotify, uint16 localClientId, IpcNotify_FxnCallback msgCallback, void *args)
{
    sint32                 status = MCAL_SystemP_SUCCESS;

    if((hIpcNotify != NULL) && (localClientId < IPC_NOTIFY_CLIENT_ID_MAX))
    {
        if(hIpcNotify->callback[localClientId] == NULL)
        {
            hIpcNotify->callback[localClientId] = msgCallback;
            hIpcNotify->callbackArgs[localClientId] = args;
        }
        else
        {
            status = MCAL_SystemP_INVALID_PARAM;
        }
    }
    else
    {
        status = MCAL_SystemP_INVALID_PARAM;
    }

    return status;
}

sint32 IpcNotify_lld_unregisterClient(IpcNotify_Handle hIpcNotify, uint16 localClientId)
{
    sint32                 status = MCAL_SystemP_SUCCESS;

    if((hIpcNotify != NULL) && (localClientId < IPC_NOTIFY_CLIENT_ID_MAX))
    {
        hIpcNotify->callback[localClientId] = NULL;
        hIpcNotify->callbackArgs[localClientId] = NULL;
    }
    else
    {
        status = MCAL_SystemP_INVALID_PARAM;
    }

    return status;
}

__attribute__((weak)) void IpcNotify_lld_isr(void* args)
{
    IpcNotify_InterruptConfig *pInterruptConfig;
    uint32 mailboxBaseAddr;
    IpcNotify_SwQueue *swQ;
    uint32 core, value;
    uint16 clientId;
    sint32 status;
    uint32 pendingIntr;
    uint32 selfCoreId;
    IpcNotify_Handle        hIpcNotify = (IpcNotify_Handle)args;

    selfCoreId = hIpcNotify->hIpcNotifyInit->selfCoreId;

    pInterruptConfig = hIpcNotify->interruptConfig;

    IpcNotify_getReadMailbox(selfCoreId, &mailboxBaseAddr);

    pendingIntr = IpcNotify_mailboxGetPendingIntr(mailboxBaseAddr);
    do
    {
        /* We clear pending interrupt unconditional here, and read all the SW queues later */
        IpcNotify_mailboxClearPendingIntr(mailboxBaseAddr, pendingIntr);

        for(core=0; core<pInterruptConfig->numCores; core++)
        {
            IpcNotify_getReadSwQ(selfCoreId, pInterruptConfig->coreIdList[core], &swQ);

            do
            {
                status = IpcNotify_mailboxReadSwQ(swQ, &value);
                if(status == MCAL_SystemP_SUCCESS)
                {
                    clientId = (value >> IPC_NOTIFY_CLIENT_ID_SHIFT) & (IPC_NOTIFY_CLIENT_ID_MAX-1);

                    if(hIpcNotify->callback[clientId]!=NULL)
                    {
                        hIpcNotify->callback[clientId](hIpcNotify,
                                pInterruptConfig->coreIdList[core],
                                clientId,
                                (value & (IPC_NOTIFY_MSG_VALUE_MAX-1)),
                                CDD_IPC_TIMEOUT,
                                hIpcNotify->callbackArgs[clientId]
                                );
                    }
                }
            } while(status == MCAL_SystemP_SUCCESS);
        }

        /* we need to keeping doing this until all status bits are 0, else we dont get new interrupt at R5F */
        pendingIntr = IpcNotify_mailboxGetPendingIntr(mailboxBaseAddr);
    } while ( pendingIntr != 0 );

}

uint32 IpcNotify_lld_getSelfCoreId(IpcNotify_Handle hIpcNotify)
{
    return hIpcNotify->hIpcNotifyInit->selfCoreId;
}

uint32 IpcNotify_lld_isCoreEnabled(IpcNotify_Handle hIpcNotify, uint32 coreId)
{
    uint32 isEnabled = 0;

    if(coreId < MCAL_CSL_CORE_ID_MAX)
    {
        isEnabled = hIpcNotify->isCoreEnabled[coreId];
    }

    return isEnabled;
}

#define CDD_IPC_STOP_SEC_CODE
#include "Cdd_Ipc_MemMap.h"