/*
 *  Copyright (C) 2018-2021 Texas Instruments Incorporated
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <ipc_rpmsg_lld_priv.h>

#define CDD_IPC_START_SEC_CODE
#include "Cdd_Ipc_MemMap.h"

RPMessage_LocalMsg *RPMessage_allocEndPtMsg(RPMessage_Core *coreObj, uint32 remoteCoreId)
{
    RPMessage_LocalMsg *pMsg;

    pMsg = (RPMessage_LocalMsg*)RPMessage_queueGet(&coreObj->freeQ);
    if(pMsg == NULL_PTR)
    {
        coreObj->freeQAllocPending = 1;
    }
    else
    {
        coreObj->freeQAllocPending = 0;
    }

    return pMsg;
}

uint32 RPMessage_freeEndPtMsg(RPMessage_Core *coreObj, uint16 remoteCoreId, RPMessage_LocalMsg *pMsg)
{
    uint32 isAllocPending;

    isAllocPending = coreObj->freeQAllocPending;
    RPMessage_queuePut(&coreObj->freeQ, &pMsg->elem);

    return isAllocPending;
}

void RPMessage_putEndPtMsg(RPMessage_Struct *epObj, RPMessage_LocalMsg *pMsg)
{
    RPMessage_queuePut(&epObj->endPtQ, &pMsg->elem);

    Cdd_Ipc_Mutex_resourceUnlock(&epObj->newEndPtMsgSem);
}

sint32 RPMessage_getEndPtMsg(RPMessage_Struct *epObj, RPMessage_LocalMsg **pMsg, uint32 timeout)
{
    uint32 done, tryLoop;
    sint32 status = MCAL_SystemP_TIMEOUT;
    uint32 elapsedTicks, startTicks;

    done = 0, tryLoop = 0;
    do {
        *pMsg = (RPMessage_LocalMsg*)RPMessage_queueGet(&epObj->endPtQ);

        if(*pMsg==NULL)
        {
            startTicks = Clock_getTicks();

            do
            {
                if(Cdd_Ipc_Mutex_resourceTryLock(&epObj->newEndPtMsgSem) == MUTEX_ARM_UNLOCKED)
                {
                    tryLoop = 1;
                    status = MCAL_SystemP_SUCCESS;
                }
                elapsedTicks = Clock_getTicks() - startTicks;
            } while ((elapsedTicks < timeout) && (tryLoop == 0));

            if(status == MCAL_SystemP_TIMEOUT)
            {
                done = 1;
            }
            if(status == MCAL_SystemP_SUCCESS && epObj->doRecvUnblock)
            {
                status = MCAL_SystemP_TIMEOUT;
                done = 1;
            }
        }
        else
        {
            status = MCAL_SystemP_SUCCESS;
            done = 1;
        }
    } while( ! done );

    return status;
}

/* handle one new received message from vring */
void RPMessage_lld_recvHandler(RPMessageLLD_Handle hRpMsg, uint32 remoteCoreId,uint32 timeout)
{
    uint16 vringBufId;
    sint32 status;
    RPMessage_LocalMsg *pMsg;

    /* get a free message pointer to hold vring buf info
     * if no free message pointer then dont remove from vring
     */
    pMsg = RPMessage_allocEndPtMsg(&hRpMsg->coreObj[remoteCoreId],remoteCoreId);
    if(pMsg!=NULL)
    {
        status = RPMessage_vringGetFullRxBuf(hRpMsg, remoteCoreId, &vringBufId);
        if(status == MCAL_SystemP_SUCCESS)
        {
            /* message in vring, extract it and copy info to message pointer and put in end point Q */
            uint8 *vringBufAddr = RPMessage_vringGetRxBufAddr(hRpMsg, remoteCoreId, vringBufId);
            RPMessage_Header *header = (RPMessage_Header *)vringBufAddr;
            uint16 localEndPt = header->dstEndPt;

            status = MCAL_SystemP_FAILURE;
            if(localEndPt < RPMESSAGE_MAX_LOCAL_ENDPT)
            {
                RPMessage_Struct *epObj = hRpMsg->localEndPtObj[localEndPt];
                if(epObj!=NULL)
                {
                    if(epObj->recvCallback != NULL)
                    {
                        /* recv messages handled in callback mode */
                        epObj->recvCallback( hRpMsg, (RPMessage_EpObject*)epObj,
                            epObj->recvCallbackArgs,
                            &vringBufAddr[sizeof(RPMessage_Header)],
                            header->dataLen,
                            remoteCoreId,
                            header->srcEndPt
                            );
                        status = MCAL_SystemP_SUCCESS;

                        /* pMsg is not used, free it */
                        RPMessage_freeEndPtMsg(&hRpMsg->coreObj[remoteCoreId], remoteCoreId, pMsg);
                        /* done using vring buf, free it */
                        RPMessage_vringPutEmptyRxBuf(hRpMsg, remoteCoreId, vringBufId,timeout);
                    }
                    else
                    {
                        /* recv messages handled in non-callback mode */
                        pMsg->remoteCoreId = remoteCoreId;
                        pMsg->vringBufId = vringBufId;
                        RPMessage_putEndPtMsg(epObj, pMsg);
                        status = MCAL_SystemP_SUCCESS;

                        if(epObj->recvNotifyCallback!=NULL)
                        {
                            epObj->recvNotifyCallback(hRpMsg, (RPMessage_EpObject*)epObj, epObj->recvNotifyCallbackArgs);
                        }
                    }
                }
            }
            if(status!=MCAL_SystemP_SUCCESS)
            {
                /* invalid vring message header or invalid endpt
                * or no object registered for local end pt, so no need handle the message pointer,
                * free it
                */
                RPMessage_vringPutEmptyRxBuf(hRpMsg, remoteCoreId, vringBufId,timeout);
            }
        }
        if(status!=MCAL_SystemP_SUCCESS)
        {
            /* no message in vring or invalid vring message header or invalid endpt
            * or no object registered for local end pt, so no need handle the message pointer,
            * free it
            */
            RPMessage_freeEndPtMsg(&hRpMsg->coreObj[remoteCoreId], remoteCoreId, pMsg);
        }
    }
}

void RPMessage_notifyCallback(void* ipcNotifyHandle, uint32 remoteCoreId, uint16 localClientId, uint32 msgValue,uint32 timeout, void *args)
{
    RPMessageLLD_Handle hRpMsg = (RPMessageLLD_Handle )args;

    if(hRpMsg->isCoreEnable[remoteCoreId] && hRpMsg->isCoreInitialized[remoteCoreId])
    {
        uint16 rxMsgValue = RPMESSAGE_MSG_VRING_NEW_FULL;

        if(RPMessage_isLinuxCore(hRpMsg, remoteCoreId))
        {
            rxMsgValue = RPMESSAGE_LINUX_RX_VRING_ID; /* In linux, we get RX VRING ID, which is 1 in linux */
        }
        if(msgValue == rxMsgValue)
        {   /* check full ring */
            while(RPMessage_vringIsFullRxBuf(hRpMsg, remoteCoreId))
            {
                RPMessage_lld_recvHandler(hRpMsg, remoteCoreId,timeout);
            }
        }
        else
        {   /* check empty ring */

            /* check if there is any new empty buf, if yes then post semaphore to wake up any waiting threads */
            RPMessage_vringCheckEmptyTxBuf(hRpMsg, remoteCoreId);
        }
    }
}

sint32 RPMessage_lld_send( RPMessageLLD_Handle hRpMsg, void*    data,
                        uint16 dataLen,
                        uint16 remoteCoreId,
                        uint16 remoteEndPt,
                        uint16 localEndPt,
                        uint32 timeout
                      )
{
    sint32 status = MCAL_SystemP_FAILURE;

    if(hRpMsg != NULL)
    {
        if(remoteCoreId < MCAL_CSL_CORE_ID_MAX && hRpMsg->isCoreEnable[remoteCoreId]
            && data != NULL && dataLen != 0
            )
        {
            uint16 vringBufId;

            status = RPMessage_vringGetEmptyTxBuf(hRpMsg, remoteCoreId, &vringBufId, timeout);
            if(status == MCAL_SystemP_SUCCESS)
            {
                uint8 *vringBufAddr = RPMessage_vringGetTxBufAddr(hRpMsg, remoteCoreId, vringBufId);
                uint16 vringBufLen = RPMessage_vringGetTxBufLen(hRpMsg, remoteCoreId, vringBufId);
                RPMessage_Header *header = (RPMessage_Header *)vringBufAddr;

                if(dataLen > (vringBufLen - sizeof(RPMessage_Header)) )
                {
                    dataLen = vringBufLen - sizeof(RPMessage_Header);

                    // DebugP_logWarn("[IPC RPMSG] Message send to remote core %d @ %d end point truncated due to lack of space in vring buffer !!!\r\n",
                    //     remoteCoreId, remoteEndPt);
                }

                header->srcEndPt = localEndPt;
                header->dstEndPt = remoteEndPt;
                header->srcCoreId = hRpMsg->selfCoreId;
                header->flags = 0;
                header->dataLen = dataLen;

                memcpy( &vringBufAddr[sizeof(RPMessage_Header)], data, dataLen);

                RPMessage_vringPutFullTxBuf(hRpMsg, remoteCoreId, vringBufId, dataLen + sizeof(RPMessage_Header),timeout);
            }
            else
            {
                // DebugP_logWarn("[IPC RPMSG] Message send to remote core %d @ %d end point failed due to lack of space in vring !!!\r\n",
                //     remoteCoreId, remoteEndPt);
            }
        }
        else
        {
            // DebugP_logError("[IPC RPMSG] Message send to remote core %d @ %d end point failed due to invalid parameters !!!\r\n",
            //     remoteCoreId, remoteEndPt
            //     );
        }
    }
    else
    {
        status = MCAL_SystemP_INVALID_PARAM;
    }
    return status;
}

sint32 RPMessage_lld_recv(RPMessageLLD_Handle hRpMsg, RPMessage_EpObject *epObject, void* data, uint16 *dataLen,
                      uint16 *remoteCoreId, uint16 *remoteEndPt, uint32 timeout)
{
    sint32 status = MCAL_SystemP_FAILURE;
    RPMessage_Struct *epObj;

    if((hRpMsg != NULL) && (epObject != NULL))
    {
        epObj = (RPMessage_Struct *)epObject;
        if( data != NULL && dataLen != NULL && remoteCoreId != NULL && remoteEndPt != NULL
            && epObj->recvCallback == NULL /* i.e non-callback mode */
        )
        {
            RPMessage_LocalMsg *pMsg;

            status = RPMessage_getEndPtMsg(epObj, &pMsg, timeout);
            if(status == MCAL_SystemP_SUCCESS && pMsg != NULL)
            {
                uint32 isAllocPending = 0;
                uint16 vringBufId = pMsg->vringBufId;
                uint8 *vringBufAddr = RPMessage_vringGetRxBufAddr(hRpMsg, pMsg->remoteCoreId, vringBufId);
                RPMessage_Header *header = (RPMessage_Header *)vringBufAddr;

                *remoteCoreId = pMsg->remoteCoreId;
                *remoteEndPt =  header->srcEndPt;

                if( *dataLen < header->dataLen )
                {
                    // DebugP_logWarn("[IPC RPMSG] Message recv @ %d local end point truncated due to insufficient user buffer size !!!\r\n",
                    //     epObj->localEndPt
                    //     );
                }
                else
                {
                    *dataLen = header->dataLen;
                }

                memcpy( data, &vringBufAddr[sizeof(RPMessage_Header)], *dataLen);

                RPMessage_vringPutEmptyRxBuf(hRpMsg, *remoteCoreId, vringBufId,timeout);
                isAllocPending = RPMessage_freeEndPtMsg(&hRpMsg->coreObj[*remoteCoreId], *remoteCoreId, pMsg);
                if(isAllocPending)
                {   /* if any messages are pending message pointer due to free Q being empty,
                    * now there will be atleast one element to handle any pending vring requests.
                    * So check vring and handle pending messages if any
                    */
                    RPMessage_notifyCallback(hRpMsg->hRpMsgInit->hIpcNotify, *remoteCoreId,
                        IPC_NOTIFY_CLIENT_ID_RPMSG,
                        RPMESSAGE_MSG_VRING_NEW_FULL,
                        timeout,
                        NULL);
                }
            }
            else
            {
                if(status != MCAL_SystemP_TIMEOUT)
                {
                    // DebugP_logError("[IPC RPMSG] Message recv @ %d local end point failed due to invalid end point Q !!!\r\n",
                    //     epObj->localEndPt
                    //     );
                }
            }
        }
        else
        {
            // DebugP_logError("[IPC RPMSG] Message recv @ %d local end point failed due to invalid parameters !!!\r\n",
            //     epObj->localEndPt
            //     );
        }
    }
    else
    {
        status = MCAL_SystemP_INVALID_PARAM;
    }
    return status;
}

void RPMessage_lld_unblock(RPMessage_EpObject *epObject)
{
    RPMessage_Struct *epObj = (RPMessage_Struct *)epObject;
    epObj->doRecvUnblock = 1;
    Cdd_Ipc_Mutex_resourceUnlock(&epObj->newEndPtMsgSem);
}

uint16 RPMessage_lld_getLocalEndPt(const RPMessage_EpObject *epObject)
{
    RPMessage_Struct *epObj = (RPMessage_Struct *)epObject;
    return epObj->localEndPt;
}

sint32 RPMessage_lld_construct(RPMessageLLD_Handle hRpMsg, RPMessage_EpObject *epObject, const RPMessage_CreateParams *createParams)
{
    sint32 status = MCAL_SystemP_FAILURE;
    RPMessage_Struct *epObj;

    if((hRpMsg != NULL) && (epObject != NULL) && (createParams != NULL))
    {
        epObj = (RPMessage_Struct *)epObject;
        //DebugP_assert(sizeof(RPMessage_EpObject) >= sizeof(RPMessage_Struct));

        if(createParams->localEndPt < RPMESSAGE_MAX_LOCAL_ENDPT
            && hRpMsg->localEndPtObj[createParams->localEndPt] == NULL)
        {
            epObj->localEndPt = createParams->localEndPt;
            epObj->recvCallback = createParams->recvCallback;
            epObj->recvCallbackArgs = createParams->recvCallbackArgs;
            epObj->recvNotifyCallback = createParams->recvNotifyCallback;
            epObj->recvNotifyCallbackArgs = createParams->recvNotifyCallbackArgs;
            epObj->doRecvUnblock = 0;
            RPMessage_queueReset(&epObj->endPtQ);
            epObj->newEndPtMsgSem = MUTEX_ARM_LOCKED;

            hRpMsg->localEndPtObj[createParams->localEndPt] = epObj;

            status = MCAL_SystemP_SUCCESS;
        }
    }
    else
    {
        status = MCAL_SystemP_INVALID_PARAM;
    }

    return status;
}

void RPMessage_lld_destruct(RPMessageLLD_Handle hRpMsg, RPMessage_EpObject *epObject)
{
    RPMessage_Struct *epObj = (RPMessage_Struct *)epObject;

    if(epObj->localEndPt < RPMESSAGE_MAX_LOCAL_ENDPT &&
        hRpMsg->localEndPtObj[epObj->localEndPt] != NULL)
    {
        hRpMsg->localEndPtObj[epObj->localEndPt] = NULL;

        epObj->localEndPt = RPMESSAGE_MAX_LOCAL_ENDPT;
        epObj->recvCallback = NULL;
        epObj->recvCallbackArgs = NULL;
        epObj->doRecvUnblock = 0;
        RPMessage_queueReset(&epObj->endPtQ);
    }
}

void RPMessage_lld_CreateParams_init(RPMessage_CreateParams *params)
{
    params->localEndPt = RPMESSAGE_MAX_LOCAL_ENDPT;
    params->recvCallback = NULL;
    params->recvCallbackArgs = NULL;
    params->recvNotifyCallback = NULL;
    params->recvNotifyCallbackArgs = NULL;
}

sint32  RPMessage_coreInit(RPMessageLLD_Handle hRpMsg, uint16 remoteCoreId)
{
    sint32 status = MCAL_SystemP_SUCCESS;
    RPMessage_Core *coreObj = &hRpMsg->coreObj[remoteCoreId];
    uint16 elemId;

    coreObj->newEmptyVringBufSem = MUTEX_ARM_LOCKED;
    coreObj->freeQAllocPending = 0;
    RPMessage_queueReset(&coreObj->freeQ);
    for(elemId=0; elemId<RPMESSAGE_MAX_LOCAL_MSG_OBJ; elemId++)
    {
        RPMessage_queuePut(&coreObj->freeQ, &coreObj->localMsgObj[elemId].elem);
    }
    /* Linux VRINGs we will init later inside RPMessage_waitForLinuxReady() */
    if(hRpMsg->isCoreEnable[remoteCoreId] && !RPMessage_isLinuxCore(hRpMsg, remoteCoreId))
    {
        /* reset RX ring */
        RPMessage_vringReset(hRpMsg, remoteCoreId, 0);
        /* reset TX ring */
        RPMessage_vringReset(hRpMsg, remoteCoreId, 1);

        /* mark core data structure as initialized, now we can handle interrupts */
        hRpMsg->isCoreInitialized[remoteCoreId] = 1;
    }
    return status;
}

void RPMessage_coreDeInit(RPMessageLLD_Handle hRpMsg, uint16 remoteCoreId)
{
    RPMessage_Core *coreObj = &hRpMsg->coreObj[remoteCoreId];

    coreObj->freeQAllocPending = 0;
    RPMessage_queueReset(&coreObj->freeQ);
}

void RPMessage_controlEndPtHandler(void *rpMsgHandle, RPMessage_EpObject *epObj, void *arg,
        void *data, uint16 dataLen,
        uint16 remoteCoreId, uint16 remoteEndPt)
{
    RPMessageLLD_Handle hRpMsg = (RPMessageLLD_Handle)rpMsgHandle;

    if(hRpMsg->controlEndPtCallback)
    {
        /* check if message is of correct size */
        if(dataLen == sizeof(RPMessage_AnnounceMsg))
        {
            /* invoke user callback */
            RPMessage_AnnounceMsg *pMsg = (RPMessage_AnnounceMsg*)data;

            hRpMsg->controlEndPtCallback(
                hRpMsg->controlEndPtCallbackArgs,
                remoteCoreId,
                pMsg->remoteEndPt,
                pMsg->name
                );
        }
    }
}

sint32 RPMessage_controlEndPtInit(RPMessageLLD_Handle hRpMsg)
{
    RPMessage_CreateParams createPrms;
    sint32 status;

    RPMessage_lld_CreateParams_init(&createPrms);
    createPrms.localEndPt = RPMESSAGE_CTRL_ENDPOINT_ID;
    createPrms.recvCallback = RPMessage_controlEndPtHandler;
    status = RPMessage_lld_construct(hRpMsg, &hRpMsg->controlEndPtObj, &createPrms);

    return status;
}

void RPMessage_controlEndPtDeInit(RPMessageLLD_Handle hRpMsg)
{
    RPMessage_lld_destruct(hRpMsg, &hRpMsg->controlEndPtObj);
}

sint32  RPMessage_lld_announce(RPMessageLLD_Handle hRpMsg, uint16 remoteCoreId, uint16 localEndPt, const char* name)
{
    sint32 status;
    RPMessage_AnnounceMsg msg;

    msg.type = 0;
    msg.remoteEndPt = localEndPt; /* local end point will be remote end point for the other side */
    strncpy(msg.name, name, RPMESSAGE_ANNOUNCE_SERVICENAME_LEN-1);
    msg.name[RPMESSAGE_ANNOUNCE_SERVICENAME_LEN-1] = '\0';

    status = RPMessage_lld_send(hRpMsg, &msg,
                sizeof(RPMessage_AnnounceMsg),
                remoteCoreId,
                RPMESSAGE_CTRL_ENDPOINT_ID, /* control end point on remote side */
                RPMESSAGE_CTRL_ENDPOINT_ID, /* reply or local end point, set also to control end point */
                CDD_IPC_TIMEOUT /* wait until message is put in VRING */
    );
    return status;
}

void RPMessage_lld_controlEndPtCallback(RPMessageLLD_Handle hRpMsg, RPMessage_ControlEndPtCallback controlEndPtCallback,
    void  *controlEndPtCallbackArgs)
{
    hRpMsg->controlEndPtCallback = controlEndPtCallback;
    hRpMsg->controlEndPtCallbackArgs = controlEndPtCallbackArgs;
}

uint32 RPMessage_isLinuxCore(RPMessageLLD_Handle hRpMsg, uint16 coreId)
{
    uint32 isLinuxCore = 0;

    if(coreId == hRpMsg->linuxCoreId && hRpMsg->linuxResourceTable)
    {
        isLinuxCore = 1;
    }
    return isLinuxCore;
}

sint32  RPMessage_lld_init(RPMessageLLD_Handle hRpMsg)
{
    sint32 status = MCAL_SystemP_SUCCESS;
    uint16 coreId, localEndPtId;
    RPMessageLLD_InitHandle hRpMsgInit;

    if((hRpMsg != NULL) && (hRpMsg->hRpMsgInit != NULL))
    {
        hRpMsgInit = hRpMsg->hRpMsgInit;

        LLD_PARAMS_CHECK(hRpMsgInit->hIpcNotify != NULL);
        LLD_PARAMS_CHECK(hRpMsgInit->vringSize != 0);
        LLD_PARAMS_CHECK(hRpMsgInit->vringNumBuf != 0);
        LLD_PARAMS_CHECK(hRpMsgInit->vringMsgSize != 0);
        LLD_PARAMS_CHECK(hRpMsgInit->linuxCoreId < MCAL_CSL_CORE_ID_MAX);

        if(status == MCAL_SystemP_SUCCESS)
        {
            hRpMsg->selfCoreId = IpcNotify_lld_getSelfCoreId(hRpMsgInit->hIpcNotify);
            hRpMsg->controlEndPtCallback = NULL;
            hRpMsg->controlEndPtCallbackArgs = NULL;
            hRpMsg->linuxResourceTable = hRpMsgInit->linuxResourceTable;
            hRpMsg->linuxCoreId = hRpMsgInit->linuxCoreId;
            for(localEndPtId = 0; localEndPtId < RPMESSAGE_MAX_LOCAL_ENDPT; localEndPtId++)
            {
                hRpMsg->localEndPtObj[localEndPtId] = NULL;
            }
            for(coreId=0; coreId<MCAL_CSL_CORE_ID_MAX; coreId++)
            {
                /* enable a core for RPMessage only when below is satisifed
                * - valid vring ID is set
                * - not self core ID
                * - IPC Notify with that core is enabled
                */
                hRpMsg->isCoreEnable[coreId] = 0;
                hRpMsg->isCoreInitialized[coreId] = 0;
                if(hRpMsgInit->vringTxBaseAddr[coreId] != RPMESSAGE_VRING_ADDR_INVALID
                    &&
                    hRpMsgInit->vringRxBaseAddr[coreId] != RPMESSAGE_VRING_ADDR_INVALID
                    &&
                    coreId != hRpMsg->selfCoreId
                    &&
                    IpcNotify_lld_isCoreEnabled(hRpMsgInit->hIpcNotify, coreId)
                )
                {
                    hRpMsg->isCoreEnable[coreId] = 1;
                }
                if(RPMessage_isLinuxCore(hRpMsg, coreId)
                    && IpcNotify_lld_isCoreEnabled(hRpMsgInit->hIpcNotify, coreId)
                    )
                {
                    hRpMsg->isCoreEnable[coreId] = 1;
                }

            }
            for(coreId=0; coreId<MCAL_CSL_CORE_ID_MAX; coreId++)
            {
                status |= RPMessage_coreInit(hRpMsg, coreId);
            }

            /* create control end point */
            status |= RPMessage_controlEndPtInit(hRpMsg);

            IpcNotify_lld_registerClient(hRpMsgInit->hIpcNotify, IPC_NOTIFY_CLIENT_ID_RPMSG,
                RPMessage_notifyCallback, hRpMsg
                );
        }
    }
    else
    {
        status = MCAL_SystemP_INVALID_PARAM;
    }

    return status;
}

sint32  RPMessage_lld_deInit(RPMessageLLD_Handle hRpMsg)
{
    sint32  status = MCAL_SystemP_SUCCESS;
    uint16 coreId;

    if(hRpMsg != NULL)
    {
        IpcNotify_lld_unregisterClient(hRpMsg->hRpMsgInit->hIpcNotify, IPC_NOTIFY_CLIENT_ID_RPMSG);

        RPMessage_controlEndPtDeInit(hRpMsg);

        for(coreId=0; coreId<MCAL_CSL_CORE_ID_MAX; coreId++)
        {
            RPMessage_coreDeInit(hRpMsg, coreId);
        }
    }
    else
    {
        status = MCAL_SystemP_INVALID_PARAM;
    }

    return status;
}

#define CDD_IPC_STOP_SEC_CODE
#include "Cdd_Ipc_MemMap.h"