/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     Mcu.h
 *
 *  \brief    This file contains interface header for Mcu Driver
 *
 */

/**
 *  \defgroup MCAL_MCU_API MCU Driver API
 *
 *  The MCU driver provides services for basic microcontroller initialization,
 *  power down functionality, reset  and microcontroller specific functions
 *  required by other MCAL software modules.
 *
 *  @sa MCAL_MCU_CFG
 *  @{
 */

/*
 * Below are the global requirements which are met by this MCU
 * driver which can't be mapped to a particular source ID
 */
/*
 * Requirements : SWS_Mcu_00017, SWS_Mcu_00226
 *                 AUTORADAR_MCAL-7
 */

/*
 * Below are the MCU module environment requirements which can't be mapped
 * to this driver
 */
/*
 * Requirements : SWS_Mcu_00116, SWS_Mcu_00051,
 *                SWS_Mcu_00244, SWS_Mcu_00246, SWS_Mcu_00247
 */
/* Design : MCU_DesignId_001 */
#ifndef MCU_H_
#define MCU_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "Std_Types.h"
/* Requirements: SWS_Mcu_00211 ,SWS_Mcu_00215 */
/* Design : MCU_DesignId_002,MCU_DesignId_004 */
#include "Mcu_Cfg.h"

#ifdef __cplusplus
extern "C"
{
#endif

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/**
 *  \name MCU Driver Module SW Version Info
 *
 *  Defines for MCU Driver version used for compatibility checks
 *  @{
 */
/** \brief Driver Implementation Major Version */
#define MCU_SW_MAJOR_VERSION            (8U)
/** \brief Driver Implementation Minor Version */
#define MCU_SW_MINOR_VERSION            (6U)
/** \brief Driver Implementation Patch Version */
#define MCU_SW_PATCH_VERSION            (2U)
/* @} */

/**
 *  \name MCU Driver Module AUTOSAR Version Info
 *
 *  Defines for MCU Driver AUTOSAR version used for compatibility checks
 *  @{
 */
/** \brief AUTOSAR Major version specification implemented by MCU Driver */
#define MCU_AR_RELEASE_MAJOR_VERSION    (4U)
/** \brief AUTOSAR Minor  version specification implemented by MCU Driver */
#define MCU_AR_RELEASE_MINOR_VERSION    (3U)
/** \brief AUTOSAR Patch version specification implemented by MCU Driver */
#define MCU_AR_RELEASE_REVISION_VERSION (1U)
/* @} */

/**
 *  \name MCU Driver ID Info
 *  @{
 */
/** \brief Texas Instruments Vendor ID */
#define MCU_VENDOR_ID                   ((uint16) 44U)
/** \brief MCU Driver Module ID */
#define MCU_MODULE_ID                   ((uint16) 101U)
/** \brief MCU Driver Instance ID */
#define MCU_INSTANCE_ID                 ((uint8) 0U)
/* @} */

/**
 *  \name MCU Service Ids
 *
 *  The Service Id is one of the argument to Det_ReportError function and
 *  is used to identify the source of the error
 *  @{
 */
/** \brief Mcu_Init() API Service ID */
#define MCU_SID_INIT                    ((uint8) 0x00U)
/** \brief Mcu_InitRamSection() API Service ID */
#define MCU_SID_INIT_RAMSECTION         ((uint8) 0x01U)
/** \brief Mcu_InitClock() API Service ID */
#define MCU_SID_INIT_CLOCK              ((uint8) 0x02U)
/** \brief Mcu_DistributePllClock() API Service ID */
#define MCU_SID_DISTRIBUTE_PLL_CLOCK    ((uint8) 0x03U)
/** \brief Mcu_GetPllStatus() API Service ID */
#define MCU_SID_GET_PLL_STATUS          ((uint8) 0x04U)
/** \brief Mcu_GetResetReason() API Service ID */
#define MCU_SID_GET_RESET_REASON        ((uint8) 0x05U)
/** \brief Mcu_GetResetRawValue() API Service ID */
#define MCU_SID_GET_RESET_RAW_VALUE     ((uint8) 0x06U)
/** \brief Mcu_PerformReset() API Service ID */
#define MCU_SID_PERFORM_RESET           ((uint8) 0x07U)
/** \brief Mcu_SetMode() API Service ID */
#define MCU_SID_SET_MODE                ((uint8) 0x08U)
/** \brief Mcu_GetVersionInfo() API Service ID */
#define MCU_SID_GET_VERSION_INFO        ((uint8) 0x09U)
/** \brief Mcu_GetRamState() API Service ID */
#define MCU_SID_GET_RAM_STATE           ((uint8) 0x0AU)
/* @} */

/* Requirements: SWS_Mcu_00012 */
/**
 *  \name MCU Error Codes
 *
 *  Error Codes returned by MCU Driver
 *  @{
 */
#ifndef MCU_E_PARAM_CONFIG
/** \brief ERROR:Mcu_init config param invalid */
#define MCU_E_PARAM_CONFIG              ((uint8) 0x0AU)
#endif
#ifndef MCU_E_PARAM_CLOCK
/** \brief ERROR:Mcu_InitClock clockid param invalid */
#define MCU_E_PARAM_CLOCK               ((uint8) 0x0BU)
#endif
#ifndef MCU_E_PARAM_MODE
/** \brief ERROR:Mcu_SetMode modeid param invalid */
#define MCU_E_PARAM_MODE                ((uint8) 0x0CU)
#endif
#ifndef MCU_E_PARAM_RAMSECTION
/** \brief ERROR:Mcu_InitRamSection ramSect param invalid */
#define MCU_E_PARAM_RAMSECTION          ((uint8) 0x0DU)
#endif
#ifndef MCU_E_PLL_NOT_LOCKED
/** \brief ERROR:PLL not locked */
#define MCU_E_PLL_NOT_LOCKED            ((uint8) 0x0EU)
#endif
#ifndef MCU_E_UNINIT
/** \brief ERROR:API invoked without performing Mcu_Init */
#define MCU_E_UNINIT                    ((uint8) 0x0FU)
#endif
#ifndef MCU_E_PARAM_POINTER
/** \brief ERROR:NULL_PTR passed to MCU driver APIs */
#define MCU_E_PARAM_POINTER             ((uint8) 0x10U)
#endif
#ifndef MCU_E_RUNTIME_ASSERT
/** \brief ERROR:Runtime assert failure */
#define MCU_E_RUNTIME_ASSERT            ((uint8) 0x11U)
#endif
/* @} */
/** \brief Specifief the status of un-initialization*/
#define MCU_STATE_UNINIT                ((uint8) 0x00U) 
/** \brief Specifief the status of un-initialization*/
#define MCU_STATE_INIT                  ((uint8) 0x01U)      


/**
 *  \name MCU Reset Modes
 *
 *  Reset mode enum which can be used in Mcu_PerformReset()
 *  @{
 */
/** \brief Initiate SW COLD RESET sequence */
#define MCU_PERFORM_RESET_MODE_COLD     ((uint8) 0U)
/** \brief Initiate SW WARM RESET sequence */
#define MCU_PERFORM_RESET_MODE_WARM     ((uint8) 1U)
/* @} */


/** \brief Reset Reason Error Mask */
#define MCU_ERRORRST_MASK           ((Mcu_RawResetType)0xFFFF0000U)

/** \brief Reset Reason  */
#define RESETREASON_NRESET                  0x0U
/**
 *  \name MCU RCM modules mode ID defines
 *
 *  PRCM modules ids defines are used tp enable specific modules
 *  @{
 */
#define MCU_RCM_MODULE_ID_FIRST        (0U)
#define MCU_RCM_MODULE_ID_DCAN         (0U)  /*!< ModuleId for DCAN */
#define MCU_RCM_MODULE_ID_MIBSPI1       (1U)  /*!< ModuleId for MIBSPI1 */
#define MCU_RCM_MODULE_ID_MIBSPI2       (2U)  /*!< ModuleId for MIBSPI2 */
#define MCU_RCM_MODULE_ID_QSPI         (3U)  /*!< ModuleId for QSPI */
#define MCU_RCM_MODULE_ID_GPIOA        (4U)  /*!< ModuleId for GPIOA */
#define MCU_RCM_MODULE_ID_WDTIMER      (5U)  /*!< ModuleId for Watchdog */
#define MCU_RCM_MODULE_ID_UART         (6U)  /*!< ModuleId for UART */
#define MCU_RCM_MODULE_ID_GPTIMER      (7U)  /*!< ModuleId for GPTIMER */
#define MCU_RCM_MODULE_ID_LAST         (MCU_RCM_MODULE_ID_GPTIMER)
/* @} */

/* Requirements: SWS_Mcu_00251,SWS_Mcu_00232,SWS_Mcu_00233 */
/**
 *  \brief This type specifies the identification (ID) for a ClockType
 *  status used by Mcu_InitClock()
 */
typedef uint8 Mcu_ClockType;

/* Requirements: SWS_Mcu_00253,SWS_Mcu_00235,SWS_Mcu_00236 */
/**
 *  \brief This type specifies the identification (ID) for a RAW MCU reset
 *  status returned by Mcu_GetResetRawValue()
 */
typedef uint32 Mcu_RawResetType;

/* Requirements : SWS_Mcu_00255,SWS_Mcu_00239,SWS_Mcu_00240 */
/**
 *  \brief This type specifies the identification (ID) for a RAM section used
 *  in Mcu_InitRamSection()
 */
typedef uint8 Mcu_RamSectionType;

/* Requirements : SWS_Mcu_00254,SWS_Mcu_00237,SWS_Mcu_00238 */
/**
 *  \brief This type specifies the identification (ID) for a MCU mode used
 *  in Mcu_SetMode()
 */
typedef uint8 Mcu_ModeType;

/* Requirements: SWS_Mcu_00252,SWS_Mcu_00234 */
/**
 *  \brief  This is the type of the reset enumerator containing the subset of
 *  reset types. <br>
 *  It is not required that all reset types are supported by hardware
 *
 */
typedef enum
{
    /* Requirements: SWS_Mcu_00134 */
    MCU_RESET_CLEAR,               /*!< Reset Reason Cleared */
    MCU_POWER_ON_RESET,            /*!< POR Reset */
    MCU_WARM_RESET_WDT0,           /*!< Warm Reset due to MSS_WDT0 */
	MCU_WARM_RESET_WDT1,           /*!< Warm Reset due to MSS_WDT1 */
	MCU_WARM_RESET_WDT2,           /*!< Warm Reset due to MSS_WDT2 */
	MCU_WARM_RESET_WDT3,           /*!< Warm Reset due to MSS_WDT3 */
    MCU_WARM_RESET,                /*!< Warm Reset due to TOPRCM WARM_RESET_CONFIG */
    MCU_EXT_PAD_RESET,             /*!< External Pad Reset */
    MCU_HSM_WDT_RESET,             /*!< Warm Reset due to HSM WDT */
	MCU_DEBUGGER_RESET,            /*!< Warm Reset due to Debugger reset */
	MCU_WARM_RESET_TEMP0,          /*!< Warm Reset due to Temprature sensor 0 */
	MCU_WARM_RESET_TEMP1,          /*!< Warm Reset due to Temprature sensor 1 */
    MCU_RESET_UNDEFINED            /*!< UNDEFINED  reset */
} Mcu_ResetType;

/* Requirements: SWS_Mcu_00250,SWS_Mcu_00231 */
/**
 *  \enum Mcu_PllStatusType
 *  \brief  This is a status value returned by the function Mcu_GetPllStatus()
 *  of the MCU module.
 */
typedef enum
{
    MCU_PLL_LOCKED = 0,
    MCU_PLL_UNLOCKED,
    MCU_PLL_STATUS_UNDEFINED
} Mcu_PllStatusType;

/* Requirements : SWS_Mcu_00256 */
/**
 *  \enum Mcu_RamStateType
 *  \brief  Enumeration of ranstate queried by Mcu_GetRamState()
 */
typedef enum
{
    MCU_RAMSTATE_INVALID,
    MCU_RAMSTATE_VALID
} Mcu_RamStateType;

/**
 *  \enum Mcu_ClkModuleIdType
 *  \brief Clock source config modules id enum
 *
 *  Clock source config ids defines are used to set clock source for specific
 *  modules
 */
typedef enum
{
    MCU_CLKSRC_MODULE_ID_CR5,               /* MCU/Cortex_R5 clock*/
    MCU_CLKSRC_MODULE_ID_SYSCLK,            /* System Clock, clock select is ignored */
    MCU_CLKSRC_MODULE_ID_MCAN0,             /* MSS MCAN0 clock */
    MCU_CLKSRC_MODULE_ID_MCAN1,             /* MSS MCAN1 clock */
	MCU_CLKSRC_MODULE_ID_MCAN2,             /* MSS MCAN2 clock */
	MCU_CLKSRC_MODULE_ID_MCAN3,             /* MSS MCAN3 clock */
    MCU_CLKSRC_MODULE_ID_RTI0,              /* MSS RTIA clock */
    MCU_CLKSRC_MODULE_ID_RTI1,              /* MSS RTIB clock */
    MCU_CLKSRC_MODULE_ID_RTI2,              /* MSS RTIC clock */
	MCU_CLKSRC_MODULE_ID_RTI3,              /* MSS RTIC clock */
    MCU_CLKSRC_MODULE_ID_WDT0,               /* MSS WDT clock */
	MCU_CLKSRC_MODULE_ID_WDT1,               /* MSS WDT clock */
	MCU_CLKSRC_MODULE_ID_WDT2,               /* MSS WDT clock */
	MCU_CLKSRC_MODULE_ID_WDT3,               /* MSS WDT clock */
    MCU_CLKSRC_MODULE_ID_QSPI,              /* MSS QSPI clock */
    MCU_CLKSRC_MODULE_ID_MCSPI0,              /* MSS SPIA clock */
	MCU_CLKSRC_MODULE_ID_MCSPI1,              /* MSS SPIA clock */
	MCU_CLKSRC_MODULE_ID_MCSPI2,              /* MSS SPIA clock */
	MCU_CLKSRC_MODULE_ID_MCSPI3,              /* MSS SPIA clock */
	MCU_CLKSRC_MODULE_ID_MCSPI4,              /* MSS SPIA clock */
    MCU_CLKSRC_MODULE_ID_SPIB,              /* MSS SPIB clock */
    MCU_CLKSRC_MODULE_ID_I2C,               /* MSS I2C clock */
    MCU_CLKSRC_MODULE_ID_SCI0,              /* MSS SCIA clock */
    MCU_CLKSRC_MODULE_ID_SCI1,              /* MSS SCIB clock */
	MCU_CLKSRC_MODULE_ID_SCI2,              /* MSS SCIB clock */
	MCU_CLKSRC_MODULE_ID_SCI3,              /* MSS SCIB clock */
	MCU_CLKSRC_MODULE_ID_SCI4,              /* MSS SCIB clock */
	MCU_CLKSRC_MODULE_ID_SCI5,              /* MSS SCIB clock */
    MCU_CLKSRC_MODULE_ID_CPSW,              /* MSS CPSW clock */
    MCU_CLKSRC_MODULE_ID_CPTS,              /* MSS CPTS clock */
    MCU_CLKSRC_MODULE_ID_MCU_CLKOUT,        /* MSS MCU CLKOUT clock */
    MCU_CLKSRC_MODULE_ID_PMIC_CLKOUT,       /* MSS PMIC CLKOUT clock */
    MCU_CLKSRC_MODULE_ID_MII100_CLK,        /* MSS MII 100 CLK clock */
    MCU_CLKSRC_MODULE_ID_MII10_CLK,         /* MSS MII 10 CLK clock */    
    MCU_CLKSRC_MODULE_ID_RGMI_CLK,          /* MSS RGMI CLK clock */
    MCU_CLKSRC_MODULE_ID_MAX
} Mcu_ClkModuleIdType;

/**
 * \brief  This is the type of the clock source in clock tree that is selectable
 *         for  peripheral. Please see TRM to map clock source to module.
 */
typedef enum
{
    MCU_CLKSRC_0 = 0,
    MCU_CLKSRC_1,
    MCU_CLKSRC_2,
    MCU_CLKSRC_3,
    MCU_CLKSRC_4,
    MCU_CLKSRC_5,
    MCU_CLKSRC_6,
    MCU_CLKSRC_7,
    MCU_CLKSRC_MAX
}Mcu_ClkSourceIdType;

/**
 *  \brief Type for PLL source selection
 */
typedef enum
{
    MCU_CLKSRC_DPLL = 0,
    MCU_CLKSRC_APLL
} Mcu_PllSourceIdType;

/* Requirements: SWS_Mcu_00030 */
/**
 *  \brief Structure for data pre-setting to be initialized
 */
typedef struct
{
    P2VAR(uint8, AUTOMATIC, MCU_RAM_COPY) Mcu_RamSectionBaseAddress;
    /**< RAM section base address, Destination */
    uint8  Mcu_RamDefaultValue;
    /**< RAM Default Value */
    uint32 Mcu_RamSectionBytes;
    /**< Section Size */
} Mcu_RamSectionConfigType;

/* Requirements: SWS_Mcu_00031 */
/**
 *  \brief Structure for module clock setting
 */
typedef struct
{
    Mcu_ClkModuleIdType Mcu_ClockModuleId;
    /**< ModuleId for which clock setting to be applied */
    Mcu_ClkSourceIdType Mcu_ClockSourceId;
    /**< Clock source Id for the module */
    uint32              Mcu_ClockDiv;
    /**< Clock divider value for the module  */
    boolean             Mcu_InitCfg;
    /**< Apply this configuration at Mcu_init time*/
} Mcu_ClockConfigType;

/**
 *  \brief Structure for PRCM configuration
 */
typedef struct
{
    uint8 Mcu_RcmModuleId;
    /**< Moudle Id to enable/disable */
    uint8 Mcu_RcmModuleMode;
    /**< Moudle mode to set @sa MCAL_MCU_API:RCM modules mode defines */
} Mcu_RCMConfigType;


/**
 *  \brief Structure for PWM clock configuration
 */
typedef struct
{
    uint8 Mcu_PwmChannelId;
    /**< Channel id of PWM */
    uint8 Mcu_PwmGroupId;
    /**< Group id of PWM */
} Mcu_PwmConfigType;

/**
 *  \brief Structure for ADC Buffer configuration
 */
typedef struct
{
    uint8 Mcu_AdcHWUniId;
    /**< Channel id of PWM */
} Mcu_AdcConfigType;

/**
 * \brief Structure for enumerating the clock outputs of HSDIVIDER
 */
typedef struct
{
    uint16  N;
    uint16  M2;
    uint16  FracM;
    uint16  sdDiv;
    uint64  MCU_PLL_CLKOUT;
    uint64  MCU_PLL_HSDIV0;
    uint64  MCU_PLL_HSDIV1;
    uint64  MCU_PLL_HSDIV2;
    
} Mcu_PllClkDivType;

/**
 *  \brief MCU PLL CONFIG structure
 */
typedef struct
{
    Mcu_PllClkDivType Mcu_PllClk1;
    Mcu_PllClkDivType Mcu_PllClk2;
   
} Mcu_PllConfigType;

/**
 * @brief HS Divider Clock Out Mux Clock Sources
 */
typedef enum
{
    DPLL_CORE_HSDIV0_CLKOUT2_PreMux = 0x0,
    DPLL_PER_HSDIV0_CLKOUT1_PreMux = 0x0,
    APLL_1p2G_HSDIV0_CLKOUT0 = 0x1,
    APLL_1p2G_HSDIV0_CLKOUT1 = 0x1,
    APLL_1p2G_HSDIV0_CLKOUT2 = 0x1,
    APLL_1p2G_HSDIV0_CLKOUT3 = 0x1,
    APLL_1p8G_HSDIV0_CLKOUT0 = 0x2,
    APLL_1p8G_HSDIV0_CLKOUT1 = 0x2,
    APLL_400MHZ = 0x3
} Mcu_HSDIVClkOutMuxClockSource;

/**
 *  \brief Pointer to PLL Config structure
 */
typedef P2CONST (Mcu_PllConfigType, AUTOMATIC, MCU_PBCFG)
Mcu_PllConfigPtrType;

/**
 *  \brief Pointer to RamConfig structure
 */
typedef P2CONST (Mcu_RamSectionConfigType, AUTOMATIC, MCU_PBCFG)
Mcu_RamConfigPtrType;

/**
   \brief Pointer to PrcmConfig structure
*
*/
typedef P2CONST (Mcu_RCMConfigType, AUTOMATIC, MCU_PBCFG)
Mcu_RCMConfigPtrType;

/**
 *  \brief Pointer to Clock Config structure
 */
typedef P2CONST (Mcu_ClockConfigType, AUTOMATIC, MCU_PBCFG)
Mcu_ClockConfigPtrType;
/**
 *  \brief Pointer to PWM Clock Config structure
 */
typedef P2CONST (Mcu_PwmConfigType, AUTOMATIC, MCU_PBCFG)
Mcu_PwmConfigPtrType;

/**
 *  \brief Pointer to ADC buffer Config structure
 */
typedef P2CONST (Mcu_AdcConfigType, AUTOMATIC, MCU_PBCFG)
Mcu_AdcConfigPtrType;

/* Requirements: SWS_Mcu_00249,SWS_Mcu_00131 */
/**
 *  \brief MCU CONFIG ROOT structure
 */
typedef struct Mcu_ConfigType_s
{
    uint8                    Mcu_ResetMode;
    /**< Reset mode to be used @sa  MCAL_MCU_API:ResetModes */
    /* Requirements: SWS_Mcu_00030 */
    Mcu_RamConfigPtrType     Mcu_ConfigRamSection;
    /**< Configration of Ram Sections */
    uint8                    Mcu_NumberOfRamSectors;
    /**< Number of entries in Mcu_ConfigRamSection */
    /* Requirements: SWS_Mcu_00031, SWS_Mcu_00019 */
    Mcu_ClockConfigPtrType   Mcu_ClockConfig;
    /**< Configuration of Module functional clock source */
    uint8                    Mcu_NumberOfClockConfig;
	#if (STD_OFF == MCU_NO_PLL)
    /**< Number of entries in Mcu_ClockConfig */
    Mcu_PllConfigPtrType     Mcu_PllConfig;
    /**< PLL Configuration */
    Mcu_PllSourceIdType      Mcu_PllSourceId;
    /**< PLL source selection */
	#endif
	#if (STD_ON == MCU_PWM_ENABLE)
	Mcu_PwmConfigPtrType     Mcu_Pwmconfig;
	/**< PWM Configuration */
	#endif
	#if (STD_ON == MCU_ADC_ENABLE)
	Mcu_AdcConfigPtrType     Mcu_AdcConfig;
	/**< ADC Configuration */
	#endif
} Mcu_ConfigType;

typedef struct
{
    uint32 CLKSRCSEL0;
}Mcu_RegisterReadbackType;
/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/**
 *  \brief  This service initializes the MCU driver
 *
 *  \verbatim
 *  Service name      : Mcu_Init
 *  Syntax            : void Mcu_Init( const Mcu_ConfigType* ConfigPtr )
 *  Service ID[hex]   : 0x0
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non Reentrant
 *  Parameters (in)   : ConfigPtr - Pointer to MCU driver configuration set.
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : This service initializes the MCU driver
 *  \endverbatim
 */
FUNC(void, MCU_CODE) Mcu_Init(
    P2CONST(Mcu_ConfigType, AUTOMATIC, MCU_PBCFG) ConfigPtr);

#if (STD_ON == MCU_INIT_RAM_API)
/**
 *  \brief This service initializes the RAM section wise
 *
 *  \verbatim
 *  Service name      : Mcu_InitRamSection
 *  Syntax            : Std_ReturnType Mcu_InitRamSection(
 *                                  Mcu_RamSectionType RamSection )
 *  Service ID[hex]   : 0x01
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non Reentrant
 *  Parameters (in)   : RamSection - Selects RAM memory section provided
 *                      in configuration set
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : Std_ReturnType
 *                    - E_OK: command has been accepted
 *                    - E_NOT_OK: command has not been accepted e.g. due to
 *                      parameter error
 *  Description       : This service initializes the RAM section wise
 *  \endverbatim
 */
FUNC(Std_ReturnType, MCU_CODE) Mcu_InitRamSection(
    Mcu_RamSectionType RamSection);
#endif /* MCU_INIT_RAM_API */

#if (STD_ON == MCU_INIT_CLOCK_API)
/**
 *  \brief This service initializes the PLL and other MCU specific clock options
 *
 *  \verbatim
 *  Service name      : Mcu_InitClock
 *  Syntax            : Std_ReturnType Mcu_InitClock(
 *                           Mcu_ClockType ClockSetting )
 *  Service ID[hex]   : 0x02
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non Reentrant
 *  Parameters (in)   : ClockSetting - Clock setting
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : Std_ReturnType
 *                    - E_OK: Command has been accepted
 *                    - E_NOT_OK: Command has not been accepted
 *  Description       : This service initializes the PLL and other MCU specific
 *                      clock options
 * \endverbatim
 */
FUNC(Std_ReturnType, MCU_CODE) Mcu_InitClock(Mcu_ClockType ClockSetting);
#endif /* MCU_INIT_CLOCK_API */

#if (STD_ON == MCU_PERFORM_RESET_API)
/**
 *  \brief The service performs a microcontroller reset
 *
 *  \verbatim
 *  Service name      : Mcu_PerformReset
 *  Syntax            : void Mcu_PerformReset( void )
 *  Service ID[hex]   : 0x07
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : The service performs a microcontroller reset
 *  \endverbatim
 */
FUNC(void, MCU_CODE) Mcu_PerformReset(void);
#endif  /* MCU_PERFORM_RESET_API */

/* Requirements: SWS_Mcu_00230 */
/**
 *  \brief This service provides the lock status of the PLL.
 *
 *  \verbatim
 *  Service name      : Mcu_DistributePllClock
 *  Syntax            : Std_ReturnType Mcu_DistributePllClock( void )
 *  Service ID[hex]   : 0x03
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : Std_ReturnType - E_OK Success
 *                                       E_NOT_OK Failure
 *  Description       : This function is provided as dummy. All PLL distribution
																	 activities
 *                      are being performed in Mcu_Init() to improve 
																	performance.
 *  \endverbatim
 */
FUNC(Std_ReturnType, MCU_CODE) Mcu_DistributePllClock(void);

/* Requirements: SWS_Mcu_00230 */
/**
 *  \brief This service provides the lock status of the PLL.
 *
 *  \verbatim
 *  Service name      : Mcu_GetPllStatus
 *  Syntax            : Mcu_PllStatusType Mcu_GetPllStatus( void )
 *  Service ID[hex]   : 0x04
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : Mcu_PllStatusType - PLL Status
 *  Description       : This service provides the lock status of the PLL.
 *  \endverbatim
 */
FUNC(Mcu_PllStatusType, MCU_CODE) Mcu_GetPllStatus(void);

/**
 *  \brief The service reads the reset type from the hardware, if supported
 *
 *  \verbatim
 *  Service name      : Mcu_GetResetReason
 *  Syntax            : Mcu_ResetType Mcu_GetResetReason( void )
 *  Service ID[hex]   : 0x05
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : Mcu_ResetType
 *  Description       : The service reads the reset type from the hardware, if
 *                      supported
 *  \endverbatim
 */
FUNC(Mcu_ResetType, MCU_CODE) Mcu_GetResetReason(void);

/**
 *  \brief The service reads the reset type from the hardware register, if
 *  supported
 *
 *  \verbatim
 *  Service name      : Mcu_GetResetRawValue
 *  Syntax            : Mcu_RawResetType Mcu_GetResetRawValue( void )
 *  Service ID[hex]   : 0x06
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : nONE
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : Mcu_RawResetType - Reset raw value
 *  Description       : The service reads the reset type from the hardware
 *                      register, if supported
 *  \endverbatim
 */
FUNC(Mcu_RawResetType, MCU_CODE) Mcu_GetResetRawValue(void);

/**
 * \brief This service activates the MCU power modes.
 *
 * Supported low power modes - None .
 *  \verbatim
 *  Service name      : Mcu_SetMode
 *  Syntax            : void Mcu_SetMode(Mcu_ModeType McuMode)
 *  Service ID[hex]   : 0x08
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : Mcu_ModeType McuMode 
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : void
 *  Description       : This service activates the MCU power modes.
 * 
 *  \endverbatim
 */
 FUNC(void, MCU_CODE) Mcu_SetMode(Mcu_ModeType McuMode);

#if (STD_ON == MCU_GET_RAM_STATE_API)
/**
 *  \brief This service provides the actual status of the microcontroller Ram.
 *  (if supported)
 *
 *  \verbatim
 *  Service name      : Mcu_GetRamState
 *  Syntax            : Mcu_RamStateType Mcu_GetRamState(void)
 *  Service ID[hex]   : 0x0a
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : Mcu_RamStateType - Status of the Ram Content
 *  Description       : This service provides the actual status of the
 *                      microcontroller Ram. (if supported)
 *  \endverbatim
 */
FUNC(Mcu_RamStateType, MCU_CODE) Mcu_GetRamState(void);
#endif  /* MCU_GET_RAM_STATE_API */

#if (STD_ON == MCU_GET_VERSION_INFO_API)
/**
 *  \brief This service returns the version information of this module
 *
 *  \verbatim
 *  Service name      : Mcu_GetVersionInfo
 *  Syntax            : void Mcu_GetVersionInfo(Std_VersionInfoType*
 *                              versioninfo)
 *  Service ID[hex]   : 0x09
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : versioninfo - Pointer to where to store the
 *                      version information of this module.
 *  Return value      : None
 *  Description       : This service returns the version information of this
 *                      module
 *  \endverbatim
 */
FUNC(void, MCU_CODE) Mcu_GetVersionInfo(
    P2VAR(Std_VersionInfoType, AUTOMATIC, MCU_APPL_DATA) versioninfo);
#endif  /* MCU_GET_VERSION_INFO_API */

/**
 *  \brief This service is only used for UT testing and not for app use
 *
 *  \verbatim
 *  Service name      : Mcu_DeInit
 *  Syntax            : void Mcu_DeInit(void)
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non-Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : This service sets the internal MCU initDone flag to
 *                      FALSE so that Mcu_Init can be invoked again.
 *                      This is required for negative UT test cases
 *  \endverbatim
 */
FUNC(void, MCU_CODE) Mcu_DeInit(void);

/**
 *  \brief This service is to read some of the config registers
 *
 *  \verbatim
 *  Service name      : Mcu_RegisterReadback
 *  Syntax            : Std_ReturnType Mcu_RegisterReadback
                        (Mcu_RegisterReadbackType* RegisterReadbackPtr )
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non-Reentrant
 *  Parameters (in)   :  Mcu_RegisterReadbackType *RegisterReadbackPtr 
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : Std_ReturnType : E_OK Success
                                         E_NOT_OK Failure
 *  Description       : This service is to read some of the config registers
 
 *  \endverbatim
 */
#if (STD_ON == MCU_REGISTER_READBACK_API)
FUNC(Std_ReturnType, MCU_CODE) Mcu_RegisterReadback
   (P2VAR(Mcu_RegisterReadbackType, AUTOMATIC, MCU_APPL_DATA) 
														   RegisterReadbackPtr);
														   
#endif
#ifdef __cplusplus
}
#endif

#endif  /* #ifndef MCU_H_ */

/* @} */

