/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */
/**
      *  \file     Mcu.c
      *
      */
 /* ---------------------------------------------------------------------------
 *  FILE DESCRIPTION
 *  ---------------------------------------------------------------------------
 *         File:  Mcu.c
 *      Project:  AM263 MCAL driver
 *       Module:  MCU Driver
 *
 *  Description:  The MCU driver provides services for basic microcontroller
 *                initialization,power down functionality,reset and 
 *                microcontroller specific functions required by other MCAL 
 *				  software modules.
 *                        
 *                      
 ******************************************************************************/

#define MCU_SOURCE
/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "Mcu.h"
#include "SchM_Mcu.h"
#include "string.h"
#if (STD_ON == MCU_DEV_ERROR_DETECT)
#include "Det.h"
#endif
#include "Dem.h"
#include "Os.h"
#include "soc.h"
#define MCU_START_SEC_CODE
#include "Mcu_MemMap.h"
#include "sys_common.h"
#define MCU_STOP_SEC_CODE
#include "Mcu_MemMap.h"
#include "am263/hw_ctrl_core.h"
#include "Mcu_priv.h"
#include "hw_types.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* AUTOSAR version information check has to match definition in header file */
#if ((MCU_AR_RELEASE_MAJOR_VERSION != (0x04U))   \
    || (MCU_AR_RELEASE_MINOR_VERSION != (0x03U)) \
    || (MCU_AR_RELEASE_REVISION_VERSION != (0x01U)))
  #error "MCU: AUTOSAR Version Numbers of MCU are different"
#endif

/* vendor specific version information is BCD coded */
#if ((MCU_SW_MAJOR_VERSION != (8U)) || (MCU_SW_MINOR_VERSION != (6U)))
  #error "Version numbers of Mcu.c and Mcu.h are inconsistent!"
#endif

#if ((MCU_CFG_MAJOR_VERSION != (8U)) || (MCU_CFG_MINOR_VERSION != (6U)))
   #error "Version numbers of Mcu.c and Mcu_Cfg.h are inconsistent!"
#endif

/* Software Reset Mask */
#define MCU_PERFORM_RESET_MASK      ((uint32) 0x00008000U)

/* ========================================================================== */
/*                         Structure Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

STATIC FUNC(void, MCU_CODE) Mcu_SystemInit(void);
#if (STD_ON == MCU_INIT_CLOCK_API)
STATIC FUNC(Std_ReturnType, MCU_CODE) Mcu_ClockSetSource(
    Mcu_ClkModuleIdType moduleId,
    Mcu_ClkSourceIdType clkSrcId,
    uint32 clkDivId);
STATIC FUNC(void, MCU_CODE) Mcu_SetupClock(void);
#endif
#if (STD_ON == MCU_PWM_ENABLE)
static void Mcu_setEpwmTbClk(uint32 epwmInstance, uint32 enable);
static void Mcu_setEpwmGroup(uint32 epwmInstance, uint32 group);
#endif
#if (STD_ON == MCU_ADC_ENABLE)
static inline void Mcu_enableAdcReference(uint32 adcInstance);
#endif
static void Mcu_controlModuleUnlockMMR(uint32 domainId, uint32 partition);
static void Mcu_controlModuleLockMMR(uint32 domainId, uint32 partition);
/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* Memory mapping, done via MemMap.h Variables that are not initialized */
#define MCU_START_SEC_VAR_INIT_UNSPECIFIED
#include "Mcu_MemMap.h"
/** \brief MCU driver object */
P2CONST(Mcu_ConfigType, MCU_DATA, MCU_PBCFG) Mcu_DrvObj = NULL_PTR;
#define MCU_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Mcu_MemMap.h"


#define MCU_START_SEC_VAR_INIT_UNSPECIFIED
/*TI_INSPECTED 338 S: JUSTIFICATION_14 */
#include "Mcu_MemMap.h"

static VAR(volatile uint8, MCU_VAR_ZERO_INIT) Mcu_DrvStatus =
MCU_STATE_UNINIT;
static VAR(volatile uint8, MCU_VAR_ZERO_INIT) Mcu_PllStatus =
MCU_STATE_UNINIT;

#define MCU_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Mcu_MemMap.h"

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#define MCU_START_SEC_CODE
#include "Mcu_MemMap.h"

#if (STD_ON == MCU_GET_VERSION_INFO_API)
FUNC(void, MCU_CODE) Mcu_GetVersionInfo(Std_VersionInfoType * versioninfo)
{
#if (STD_ON == MCU_DEV_ERROR_DETECT)
    if (NULL_PTR == versioninfo)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_GET_VERSION_INFO,
                               MCU_E_PARAM_POINTER);
    }
    else
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
    {
        versioninfo->vendorID         = (MCU_VENDOR_ID);
        versioninfo->moduleID         = (MCU_MODULE_ID);
        versioninfo->sw_major_version = (uint8) (MCU_SW_MAJOR_VERSION);
        versioninfo->sw_minor_version = (uint8) (MCU_SW_MINOR_VERSION);
        versioninfo->sw_patch_version = (uint8) (MCU_SW_PATCH_VERSION);
    }
}   /* Mcu_GetVersionInfo()*/
#endif  /* STD_ON == MCU_GET_VERSION_INFO_API */

FUNC(void, MCU_CODE) Mcu_Init(P2CONST(Mcu_ConfigType, AUTOMATIC, MCU_PBCFG)
															          ConfigPtr)
{
const Mcu_ConfigType *CfgPtr = NULL_PTR;	
#if (STD_ON == MCU_VARIANT_PRE_COMPILE)
    if (NULL_PTR == ConfigPtr)
    {
        CfgPtr = &MCU_INIT_CONFIG_PC;
    }
#endif  /* (STD_ON == MCU_VARIANT_PRE_COMPILE) */
#if (STD_ON == MCU_VARIANT_POST_BUILD)

	if (NULL_PTR != ConfigPtr)

	{

		CfgPtr = ConfigPtr;

	}

#endif /* (STD_ON == MCU_VARIANT_POST_BUILD) */
#if (STD_ON == MCU_DEV_ERROR_DETECT)
    if (NULL_PTR == CfgPtr)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID, MCU_SID_INIT,
                               MCU_E_PARAM_CONFIG);
    }
    else
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
    {
        Mcu_DrvObj = CfgPtr;
        Mcu_SystemInit();
		Mcu_DrvStatus = MCU_STATE_INIT;
    }
    return;
}

STATIC FUNC(void, MCU_CODE) Mcu_SystemInit(void)
{
	Mcu_unlockMMR();

#if (STD_OFF == MCU_NO_PLL)
	if(Mcu_PllStatus != MCU_STATE_INIT)
	{
		Mcu_PLLInitAll(Mcu_DrvObj);
		Mcu_PllStatus = MCU_STATE_INIT;
	}
#endif

#if (STD_ON == MCU_INIT_CLOCK_API)
    Mcu_SetupClock();
#endif

#if (STD_ON == MCU_PWM_ENABLE)
	uint8 pwm_instance;
	for(pwm_instance = 0U; pwm_instance < MCU_PWM_CHANNEL ;pwm_instance++)
	{
	  Mcu_setEpwmTbClk(Mcu_DrvObj->Mcu_Pwmconfig[pwm_instance].Mcu_PwmChannelId, TRUE);
	  Mcu_setEpwmGroup(Mcu_DrvObj->Mcu_Pwmconfig[pwm_instance].Mcu_PwmChannelId, Mcu_DrvObj->Mcu_Pwmconfig[pwm_instance].Mcu_PwmGroupId);
	}
#endif
#if (STD_ON == MCU_ADC_ENABLE)
	uint8 adc_instance;
	for(adc_instance = 0U; adc_instance < MCU_ADC_HWUNIT ;adc_instance++)
	{
	  Mcu_enableAdcReference(Mcu_DrvObj->Mcu_AdcConfig[adc_instance].Mcu_AdcHWUniId);
	}
#endif

#if (STD_ON == MCU_INTRCROSSBAR_ENABLE)    
	Mcu_IntXbar();
#endif

#if (STD_ON == MCU_INPUT_CROSSBAR_ENABLE)    
	Mcu_InputXbar();
#endif
}

#if (STD_ON == MCU_INIT_CLOCK_API)
STATIC FUNC(void, MCU_CODE) Mcu_SetupClock(void)
{
        uint8 idx;
        /* Config Clock */
        for (idx = 0U; idx < Mcu_DrvObj->Mcu_NumberOfClockConfig; idx++)
        {
            if (Mcu_DrvObj->Mcu_ClockConfig[idx].Mcu_InitCfg ==
                (boolean) TRUE)
            {
               if (MCU_CLKSRC_MAX > Mcu_DrvObj->Mcu_ClockConfig[idx].
															  Mcu_ClockSourceId)
               {
                   Std_ReturnType status = E_NOT_OK;
                    status =
                        Mcu_ClockSetSource(Mcu_DrvObj->Mcu_ClockConfig[idx].                          
                            Mcu_ClockModuleId,Mcu_DrvObj->Mcu_ClockConfig[idx].                            
                            Mcu_ClockSourceId,Mcu_DrvObj->Mcu_ClockConfig[idx].
                                                                Mcu_ClockDiv);
					if (status != ((Std_ReturnType) E_OK))
					{
#if (STD_ON == MCU_DEV_ERROR_DETECT)

                        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                                              MCU_SID_INIT, MCU_E_PARAM_CONFIG);
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
					}
               }
               else
               {
#if (STD_ON == MCU_DEV_ERROR_DETECT)
                    (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                                           MCU_SID_INIT, MCU_E_PARAM_CONFIG);
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
               }

            }
        }
    return;
}

FUNC(Std_ReturnType, MCU_CODE) Mcu_InitClock(Mcu_ClockType ClockSetting)
{
     VAR(Std_ReturnType, MCU_VAR) InitClock_Return = E_NOT_OK;
#if (STD_ON == MCU_DEV_ERROR_DETECT)
	
    if (Mcu_DrvStatus == MCU_STATE_UNINIT)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_INIT_CLOCK, MCU_E_UNINIT);                              
    }

    else if (NULL_PTR == Mcu_DrvObj)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_INIT_CLOCK, MCU_E_PARAM_CONFIG);                              
    }
    else if (ClockSetting >= Mcu_DrvObj->Mcu_NumberOfClockConfig)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_INIT_CLOCK, MCU_E_PARAM_CLOCK);                             
     }
    else
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
    {
        InitClock_Return = Mcu_ClockSetSource(
        Mcu_DrvObj->Mcu_ClockConfig[ClockSetting].Mcu_ClockModuleId,      
        Mcu_DrvObj->Mcu_ClockConfig[ClockSetting].Mcu_ClockSourceId,   
        Mcu_DrvObj->Mcu_ClockConfig[ClockSetting].Mcu_ClockDiv);        
    }
    return (InitClock_Return);
}  /* end of the Mcu_InitClock */
#endif /* STD_ON == MCU_INIT_CLOCK_API */


#if (STD_OFF == MCU_NO_PLL)
FUNC(Std_ReturnType, MCU_CODE) Mcu_DistributePllClock(void)
{
    VAR(Std_ReturnType, MCU_VAR) InitClock_Return = E_NOT_OK;

#if (STD_ON == MCU_DEV_ERROR_DETECT)
	if (Mcu_DrvStatus == MCU_STATE_UNINIT)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_DISTRIBUTE_PLL_CLOCK, MCU_E_UNINIT);                              
    }

    else if (NULL_PTR == Mcu_DrvObj)
    {
        InitClock_Return = (Std_ReturnType) E_NOT_OK;
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                             MCU_SID_DISTRIBUTE_PLL_CLOCK, MCU_E_PARAM_CONFIG);                            
    }
    else
   if (MCU_PLL_LOCKED != Mcu_GetPllStatus())
   {
      /* API is being called before PLL is locked */
      (void)Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,  
						MCU_SID_DISTRIBUTE_PLL_CLOCK, MCU_E_PLL_NOT_LOCKED);
   }
   else
   {
      InitClock_Return = Mcu_PLLInitAll(Mcu_DrvObj);
   }
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
    return (InitClock_Return);

} /*end of the Mcu_DistributePllClock()*/
#endif /* STD_OFF == MCU_NO_PLL */

#if (STD_ON == MCU_INIT_RAM_API)
FUNC(Std_ReturnType, MCU_CODE) Mcu_InitRamSection(Mcu_RamSectionType RamSection)
{
    VAR(Std_ReturnType, MCU_VAR) Init_Ram_Section_Return =
													(Std_ReturnType) E_NOT_OK;
    /* Local word counter */
    VAR(uint32, MCU_VAR) Mcu_RamBytes;
#if (STD_ON == MCU_DEV_ERROR_DETECT)
	if (Mcu_DrvStatus == MCU_STATE_UNINIT)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_INIT_RAMSECTION, MCU_E_UNINIT);                              
    }
    else if (NULL_PTR == Mcu_DrvObj)
    {
        /* API is being called before calling Mcu_Init */
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_INIT_RAMSECTION,MCU_E_PARAM_CONFIG);                               
    }
    else if (RamSection >= Mcu_DrvObj->Mcu_NumberOfRamSectors)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_INIT_RAMSECTION, MCU_E_PARAM_RAMSECTION);                             
    }
    else
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
    {
        /* RAM destination pointer */
        P2VAR(uint8, AUTOMATIC, MCU_APPL_DATA) Mcu_RamDestination = NULL_PTR;
        /* Load the RAM Section Base address from the configuration */
        Mcu_RamDestination = Mcu_DrvObj->Mcu_ConfigRamSection[RamSection].           
													Mcu_RamSectionBaseAddress;
        /* Load the RamSection Size */
        Mcu_RamBytes =
            Mcu_DrvObj->Mcu_ConfigRamSection[RamSection].Mcu_RamSectionBytes;
        while (0U < Mcu_RamBytes)
        {
            /* Initialize the RAM with selected default value */
            *Mcu_RamDestination =Mcu_DrvObj->Mcu_ConfigRamSection[RamSection].            
															Mcu_RamDefaultValue;
            /* Point to next address*/
            Mcu_RamDestination++;
            /* Reduce Byte counter */
            Mcu_RamBytes--;
        }
        Init_Ram_Section_Return = (Std_ReturnType) E_OK;
    }

    return (Init_Ram_Section_Return);
}
#endif /* STD_ON == MCU_INIT_RAM_API */

FUNC(Mcu_PllStatusType, MCU_CODE) Mcu_GetPllStatus(void)
{
    VAR(Mcu_PllStatusType, MCU_VAR)  Pll_Status = MCU_PLL_STATUS_UNDEFINED;
#if (STD_ON == MCU_DEV_ERROR_DETECT)
    if (Mcu_DrvStatus == MCU_STATE_UNINIT)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_GET_PLL_STATUS, MCU_E_UNINIT);                              
    }
    else if (NULL_PTR == Mcu_DrvObj)
    {
        Pll_Status = MCU_PLL_STATUS_UNDEFINED;
        /* API is being called before calling Mcu_Init */
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_GET_PLL_STATUS, MCU_E_PARAM_CONFIG);
    }
    else
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
    {
#if (STD_ON == MCU_NO_PLL)
        /* Requirements: SWS_Mcu_00206*/
        Pll_Status = MCU_PLL_STATUS_UNDEFINED;
#else
        Pll_Status = Mcu_GetPllLockStatus();
#endif  /* STD_ON == MCU_NO_PLL */
    }

    /* Requirements: SWS_Mcu_00008 */
    return (Pll_Status);
} /* end of Mcu_GetPllStatus */

FUNC(Mcu_ResetType, MCU_CODE) Mcu_GetResetReason(void)
{
    VAR(Mcu_ResetType, MCU_VAR) Reset_Reason = MCU_RESET_UNDEFINED;
#if (STD_ON == MCU_DEV_ERROR_DETECT)
	if (Mcu_DrvStatus == MCU_STATE_UNINIT)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_GET_RESET_REASON, MCU_E_UNINIT);                              
    }
    else if (NULL_PTR == Mcu_DrvObj)
    {
        Reset_Reason = MCU_RESET_UNDEFINED;
        /* API is being called before calling Mcu_Init */
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_GET_RESET_REASON,
                               MCU_E_PARAM_CONFIG);
    }
    else
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
    {
        /* Requirements: SWS_Mcu_00005 */
        Reset_Reason = Mcu_GetPlatformResetReason();
    }

    return (Reset_Reason);
} /* end of Mcu_GetResetReason() */

FUNC(void, MCU_CODE) Mcu_SetMode(Mcu_ModeType McuMode)
{

#if (STD_ON == MCU_DEV_ERROR_DETECT)
	
	if (Mcu_DrvStatus == MCU_STATE_UNINIT)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_SET_MODE, MCU_E_UNINIT);                              
    }
   else if (NULL_PTR == Mcu_DrvObj)
   {
      /* API is being called before calling Mcu_Init */
      (void)Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID, MCU_SID_SET_MODE,
														MCU_E_PARAM_CONFIG);
   }
   else
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
   {
         /* Do nothing. Invalid power mode */
   }
} /*end of the Mcu_SetMode()*/

FUNC(Mcu_RawResetType, MCU_CODE) Mcu_GetResetRawValue(void)
{
   VAR(Mcu_RawResetType, MCU_VAR)  Reset_Raw_Value = MCU_ERRORRST_MASK;
#if (STD_ON == MCU_DEV_ERROR_DETECT)
	if (Mcu_DrvStatus == MCU_STATE_UNINIT)
    {
		Reset_Raw_Value = MCU_ERRORRST_MASK;
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_GET_RESET_RAW_VALUE, MCU_E_UNINIT);                              
    }
    else if (NULL_PTR == Mcu_DrvObj)
    {
        Reset_Raw_Value = MCU_ERRORRST_MASK;
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_GET_RESET_RAW_VALUE, MCU_E_PARAM_CONFIG);                              
    }
    else
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
    {
        Reset_Raw_Value = Mcu_GetPlatformRawResetReason();
    }
    return (Reset_Raw_Value);
}  /*end of the Mcu_ResetRawValue()*/


/* Requirements: SWS_Mcu_00209 */
#if (STD_ON == MCU_GET_RAM_STATE_API)
FUNC(Mcu_RamStateType, MCU_CODE) Mcu_GetRamState(void)
{
    VAR(Mcu_RamStateType, MCU_VAR) GetRamState_Return = MCU_RAMSTATE_INVALID;
#if (STD_ON == MCU_DEV_ERROR_DETECT)
	if (Mcu_DrvStatus == MCU_STATE_UNINIT)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_GET_RAM_STATE, MCU_E_UNINIT);                              
    }
    else if (NULL_PTR == Mcu_DrvObj)
    {
        GetRamState_Return = MCU_RAMSTATE_INVALID;
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_GET_RAM_STATE, MCU_E_PARAM_CONFIG);                               
    }
    else
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
    {
        GetRamState_Return = MCU_RAMSTATE_VALID;
    }
    return (GetRamState_Return);
}  /*end of the Mcu_GetRamState()*/
#endif /*STD_ON == MCU_GET_RAM_STATE_API */

#if (STD_ON == MCU_PERFORM_RESET_API)
FUNC(void, MCU_CODE) Mcu_PerformReset(void)
{
#if (STD_ON == MCU_DEV_ERROR_DETECT)
	if (Mcu_DrvStatus == MCU_STATE_UNINIT)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_PERFORM_RESET, MCU_E_UNINIT);                              
    }
    else if (NULL_PTR == Mcu_DrvObj)
    {
        (void) Det_ReportError(MCU_MODULE_ID, MCU_INSTANCE_ID,
                               MCU_SID_PERFORM_RESET, MCU_E_UNINIT);                               
    }
    else
#endif /* STD_ON == MCU_DEV_ERROR_DETECT */
    {
        VAR(uint8, MCU_VAR)  Reset_Mode;

        /* Perform a sofware reset */
        SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0();
        Reset_Mode = Mcu_DrvObj->Mcu_ResetMode;
        Mcu_PerformSoftSysReset(Reset_Mode);
        SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0();
    }
} /*end of the Mcu_PerformReset()*/
#endif /* STD_ON == MCU_PERFORM_RESET_API */


#if (STD_ON == MCU_INIT_CLOCK_API)
STATIC FUNC(Std_ReturnType, MCU_CODE) Mcu_ClockSetSource(
    Mcu_ClkModuleIdType moduleId,
    Mcu_ClkSourceIdType clkSrcId,
    uint32 clkDivId)
{
    VAR(Std_ReturnType, MCU_VAR) RetVal;

    switch (moduleId)
    {
        case MCU_CLKSRC_MODULE_ID_MCAN0:
        {
            RetVal = Mcu_ClockSetSourceMCAN0(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_MCAN1:
        {
            RetVal = Mcu_ClockSetSourceMCAN1(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_MCAN2:
        {
            RetVal = Mcu_ClockSetSourceMCAN2(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_MCAN3:
        {
            RetVal = Mcu_ClockSetSourceMCAN3(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_RTI0:
        {
            RetVal = Mcu_ClockSetSourceRTI0(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_RTI1:
        {
            RetVal = Mcu_ClockSetSourceRTI1(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_RTI2:
        {
            RetVal = Mcu_ClockSetSourceRTI2(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_RTI3:
        {
            RetVal = Mcu_ClockSetSourceRTI3(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_WDT0:
        {
            RetVal = Mcu_ClockSetSourceWDT0(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_WDT1:
        {
            RetVal = Mcu_ClockSetSourceWDT1(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_WDT2:
        {
            RetVal = Mcu_ClockSetSourceWDT2(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_WDT3:
        {
            RetVal = Mcu_ClockSetSourceWDT3(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_QSPI:
        {
            RetVal = Mcu_ClockSetSourceQSPI(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_MCSPI0:
        {
            RetVal = Mcu_ClockSetSourceMCSPI0(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_MCSPI1:
        {
            RetVal = Mcu_ClockSetSourceMCSPI1(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_MCSPI2:
        {
            RetVal = Mcu_ClockSetSourceMCSPI2(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_MCSPI3:
        {
            RetVal = Mcu_ClockSetSourceMCSPI3(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_MCSPI4:
        {
            RetVal = Mcu_ClockSetSourceMCSPI4(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_I2C:
        {
            RetVal = Mcu_ClockSetSourceI2C(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_SCI0:
        {
            RetVal = Mcu_ClockSetSourceSCI0(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_SCI1:
        {
            RetVal = Mcu_ClockSetSourceSCI1(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_SCI2:
        {
            RetVal = Mcu_ClockSetSourceSCI2(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_SCI3:
        {
            RetVal = Mcu_ClockSetSourceSCI3(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_SCI4:
        {
            RetVal = Mcu_ClockSetSourceSCI4(clkSrcId, clkDivId);
            break;
        }
		case MCU_CLKSRC_MODULE_ID_SCI5:
        {
            RetVal = Mcu_ClockSetSourceSCI5(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_CPSW:
        {
            RetVal = Mcu_ClockSetSourceCPSW(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_CPTS:
        {
            RetVal = Mcu_ClockSetSourceCPTS(clkSrcId, clkDivId);
            break;
        }
        case MCU_CLKSRC_MODULE_ID_MCU_CLKOUT:
        {
            RetVal = Mcu_ClockSetSourceMcuClkout(clkSrcId, clkDivId);
            break;
        }                     
        case MCU_CLKSRC_MODULE_ID_MII100_CLK:
        {
            /*Clk source is fixed to DPLL_CORE_HSDIV0_CLKOUT1*/
            RetVal = Mcu_ClockSetSourceMII100Clk(clkSrcId, clkDivId);
            break;
        }       
        case MCU_CLKSRC_MODULE_ID_MII10_CLK:
        {
            /*Clk source is fixed to DPLL_CORE_HSDIV0_CLKOUT1*/
            RetVal = Mcu_ClockSetSourceMII10Clk(clkSrcId, clkDivId);
            break;
        }               
        case MCU_CLKSRC_MODULE_ID_RGMI_CLK:
        {
            /*Clk source is fixed to DPLL_CORE_HSDIV0_CLKOUT1*/
            RetVal = Mcu_ClockSetSourceRGMIClk(clkSrcId, clkDivId);
            break;
        }              
        
        default:
            RetVal = (Std_ReturnType) E_NOT_OK;
            break;
    }
    return RetVal;
}
#endif

#if (STD_ON == MCU_PWM_ENABLE)
/**
 * \brief Enable or disable ePWM time base clock from Control MMR
 *
 * \param epwmInstance [in] ePWM instance number [0 - 31]
 * \param enable       [in] TRUE to enable and FALSE to disable
 */
static void Mcu_setEpwmTbClk(uint32 epwmInstance, uint32 enable)
{
    if(epwmInstance < MCU_CSL_ETPWM_PER_CNT)
    {
        /* Time base clock enable register belongs to partition 1 of the CTRL MMR */

        /* Unlock CONTROLSS_CTRL registers */
        Mcu_controlModuleUnlockMMR(0, MCU_CONTROLSS_CTRL_PARTITION0);

        if(TRUE == enable)
        {
            /* Enable Time base clock in CTRL MMR */
            HW_WR_REG32(MCU_CSL_CONTROLSS_CTRL_U_BASE + MCU_CSL_CONTROLSS_CTRL_EPWM_CLKSYNC,
                ((HW_RD_REG32(MCU_CSL_CONTROLSS_CTRL_U_BASE +
                  MCU_CSL_CONTROLSS_CTRL_EPWM_CLKSYNC) & MCU_CSL_CONTROLSS_CTRL_EPWM_CLKSYNC_BIT_MASK) | (1 << epwmInstance)));
        }
        else
        {
            /* Disable Time base clock in CTRL MMR */
            HW_WR_REG32(MCU_CSL_CONTROLSS_CTRL_U_BASE + MCU_CSL_CONTROLSS_CTRL_EPWM_CLKSYNC,
                ((HW_RD_REG32(MCU_CSL_CONTROLSS_CTRL_U_BASE +
                  MCU_CSL_CONTROLSS_CTRL_EPWM_CLKSYNC) & MCU_CSL_CONTROLSS_CTRL_EPWM_CLKSYNC_BIT_MASK) & ~(1 << epwmInstance)));
        }

        /* Lock CONTROLSS_CTRL registers */
        Mcu_controlModuleLockMMR(0, MCU_CONTROLSS_CTRL_PARTITION0);
    }
}
/**
 * \brief Configure the ePWM group
 *
 * \param epwmInstance [in] ePWM instance number [0 - 31]
 * \param group        [in] The group for this ePWM instance [0 - 3]
 */
static void Mcu_setEpwmGroup(uint32 epwmInstance, uint32 group)
{
    uint32 baseAddr = MCU_CSL_CONTROLSS_CTRL_U_BASE + MCU_CSL_CONTROLSS_CTRL_EPWM_STATICXBAR_SEL0;
    uint32 mask, shift;

    /* Unlock CONTROLSS_CTRL registers */
    Mcu_controlModuleUnlockMMR(0, MCU_CONTROLSS_CTRL_PARTITION0);

    /* Choose the correct base address depending on which ePWM instance is selected*/
    if(epwmInstance > 15)
    {
        baseAddr = baseAddr + 0x4;
        epwmInstance = epwmInstance - 16;
    }

    shift = (epwmInstance << 1);
    /* Create the mask to be written to register */
    mask = (0x3 << shift);

    /* Configure the group for the ePWM instance */
    HW_WR_REG32(baseAddr, (( HW_RD_REG32(baseAddr) & ~mask) | (group <<shift)));

    /* Lock CONTROLSS_CTRL registers */
    Mcu_controlModuleLockMMR(0, MCU_CONTROLSS_CTRL_PARTITION0);
}

#endif

static void Mcu_controlModuleUnlockMMR(uint32 domainId, uint32 partition)
{
    uint32            baseAddr;
    volatile uint32  *kickAddr;

    if(partition==MCU_CONTROLSS_CTRL_PARTITION0)
    {
        /*Unlock CONTROLSS_CTRL*/
        baseAddr = (uint32) MCU_CSL_CONTROLSS_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + MCU_CSL_CONTROLSS_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, MCU_TEST_KICK0_UNLOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + MCU_CSL_CONTROLSS_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, MCU_TEST_KICK1_UNLOCK_VAL);      /* KICK 1 */
    }

    return;
}

static void Mcu_controlModuleLockMMR(uint32 domainId, uint32 partition)
{
    uint32            baseAddr;
    volatile uint32  *kickAddr;
    
    if(partition==MCU_CONTROLSS_CTRL_PARTITION0)
    {
        /*Lock CONTROLSS_CTRL*/
        baseAddr = (uint32) MCU_CSL_CONTROLSS_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + MCU_CSL_CONTROLSS_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, MCU_TEST_KICK_LOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + MCU_CSL_CONTROLSS_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, MCU_TEST_KICK_LOCK_VAL);      /* KICK 1 */
    }

    return;
}

#if (STD_ON == MCU_ADC_ENABLE)
/* The ADC Instance is enabled here. (adcInstance can be between 0 to 4) */
static inline void Mcu_enableAdcReference(uint32 adcInstance)
{
    /* Determine the group number of the ADC and the mask to be written to compctl register */
    uint32 groupnum = (adcInstance / 3);
    uint32 compctlmask = 0x7;

    if(adcInstance == 1 || adcInstance == 2)
    {
        compctlmask = (compctlmask << 4);
    }
    else if(adcInstance == 3 || adcInstance == 4)
    {
        compctlmask = (compctlmask << 8);
    }

    /* Unlock Top Control Space */
    Mcu_controlModuleUnlockMMR(0, MCU_TOP_CTRL_PARTITION0);

    /* Enable ADC references by writing to MMR */
    HW_WR_REG16(MCU_CSL_TOP_CTRL_U_BASE + MCU_CSL_TOP_CTRL_ADC_REFBUF0_CTRL + (groupnum * 4U), 0x7);
    HW_WR_REG16(MCU_CSL_TOP_CTRL_U_BASE + MCU_CSL_TOP_CTRL_ADC_REF_COMP_CTRL,
    HW_RD_REG32(MCU_CSL_TOP_CTRL_U_BASE + MCU_CSL_TOP_CTRL_ADC_REF_COMP_CTRL) | compctlmask);

    /* Lock Top Control Space */
    Mcu_controlModuleLockMMR(0, MCU_TOP_CTRL_PARTITION0);
}


#endif 

#if (STD_ON == MCU_REGISTER_READBACK_API)

FUNC(Std_ReturnType, MCU_CODE) Mcu_RegisterReadback(P2VAR
	   (Mcu_RegisterReadbackType, AUTOMATIC, MCU_APPL_DATA) RegisterReadbackPtr)
{
   VAR(Std_ReturnType, MCU_VAR) RetVal;

   if ((NULL_PTR == RegisterReadbackPtr) || (NULL_PTR == Mcu_DrvObj))
   {
      RetVal = (Std_ReturnType) E_NOT_OK;
   }
   else
   {
      RegisterReadbackPtr->CLKSRCSEL0 = 
									(uint32)(mssrcmREG->RCM_RTI0_CLK_SRC_SEL);
      RetVal = (Std_ReturnType)E_OK;
   }

   return RetVal;
}
#endif

FUNC(void, MCU_CODE) Mcu_DeInit(void)
{
    /* Reset Init Done flag */
    Mcu_DrvObj = NULL_PTR;
	Mcu_DrvStatus = MCU_STATE_UNINIT;
}
#define MCU_STOP_SEC_CODE
#include "Mcu_MemMap.h"

