
/* =============================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ===========================================================================*/

/**
 *  \file     Mcu_Platform.c
 *
 */

#define MCU_PLATFORM_SOURCE (1)

/* ===========================================================================*/
/*                             Include Files                                  */
/* ===========================================================================*/
#include "Std_Types.h"
#include "Os.h"
#include "Dem.h"
#include "Mcu.h"
#include "Mcu_priv.h"
#define MCU_START_SEC_CODE
#include "Mcu_MemMap.h"
#define MCU_STOP_SEC_CODE
#include "Mcu_MemMap.h"
#include "soc.h"
#include "am263/hw_ctrl_core.h"
#include "hal_stdtypes.h"
#include "hw_types.h"

/* ========================================================================== */
/*                         Structure Declarations                             */
/* ========================================================================== */

#define MCU_START_SEC_CODE
#include "Mcu_MemMap.h"

#if (STD_OFF == MCU_NO_PLL)
Mcu_PllStatusType Mcu_GetPllLockStatus(void)
{
    VAR(uint32, MCU_VAR) Reg_Val1 = 0U;
    VAR(uint32, MCU_VAR) Reg_Val2 = 0U;
    VAR(Mcu_PllStatusType, MCU_VAR) Pll_Status;

    Reg_Val1 = HW_RD_REG32(&toprcmREG->PLL_CORE_STATUS);
    Reg_Val2 = HW_RD_REG32(&toprcmREG->PLL_PER_STATUS);

    if ((Reg_Val1 & 0x400) && (Reg_Val2 & 0x400))
    {
        Pll_Status = MCU_PLL_LOCKED;
    }
    else
    {
        Pll_Status = MCU_PLL_UNLOCKED;
    }

    return (Pll_Status);
}
#endif

#if (STD_ON == MCU_PERFORM_RESET_API)
void Mcu_PerformSoftSysReset(uint32 resetVal)
{  
    (void)regWriteReadback(&toprcmREG->WARM_RESET_CONFIG, M_TEN, 
															M_EIGHT, resetVal);
    return;
}
#endif /* STD_ON == MCU_PERFORM_RESET_API */

Mcu_ResetType Mcu_GetPlatformResetReason(void)
{
    VAR(uint32, MCU_VAR) Reset_Read = 0U;
    VAR(Mcu_ResetType, MCU_VAR) Reset_Type = MCU_RESET_UNDEFINED;
    Reset_Read = Mcu_GetPlatformRawResetReason();

    switch(Reset_Read)
    {
        case M_ZERO:
        {
            Reset_Type = MCU_RESET_CLEAR;
            break;
        }
        case M_SIXTY_FIVE:
        {
            Reset_Type = MCU_POWER_ON_RESET;
            break;
        }
        case M_SIXTY_SIX:
        {
            Reset_Type = MCU_WARM_RESET_WDT0;
            break;
        }
		case M_SIXTY_EIGHT:
        {
            Reset_Type = MCU_WARM_RESET_WDT1;
            break;
        }
		case M_SEVENTY_TWO:
        {
            Reset_Type = MCU_WARM_RESET_WDT2;
            break;
        }
		case M_EIGHTY:
        {
            Reset_Type = MCU_WARM_RESET_WDT3;
            break;
        }
        case M_NINTY_SIX:
        {
            Reset_Type = MCU_WARM_RESET;
            break;
        }
        case M_SIXTY_FOUR:
        {
            Reset_Type = MCU_EXT_PAD_RESET;
            break;
        }
        case M_ONE_NINTY_TWO:
        {
            Reset_Type = MCU_HSM_WDT_RESET;
            break;
        }
		case M_THREE_TWENTY:
        {
            Reset_Type = MCU_DEBUGGER_RESET;
            break;
        }
		case M_FIVE_SEVENTY_SIX:
        {
            Reset_Type = MCU_WARM_RESET_TEMP0;
            break;
        }
		case M_VALUE_1088:
        {
            Reset_Type = MCU_WARM_RESET_TEMP1;
            break;
        }
        default:
            Reset_Type = MCU_RESET_UNDEFINED;
            break;
    }

    return Reset_Type;

}

Mcu_RawResetType Mcu_GetPlatformRawResetReason(void)
{
    VAR(Mcu_RawResetType, MCU_VAR) rawResetReason = 0U;
    /* Get reset reason, stored in SPARE_RW0 by startup code */
    rawResetReason = (Mcu_RawResetType)toprcmREG->HW_SPARE_RW0;

    return rawResetReason;
}

#if (STD_ON == MCU_INIT_CLOCK_API)
uint32 Mcu_getMultibitValue(uint32 clk_srcId)
{
    VAR(uint32, MCU_VAR) multibitValue = 0;
    multibitValue = (clk_srcId << 8) | (clk_srcId << 4) | (clk_srcId);
    return multibitValue;
}
Std_ReturnType Mcu_ClockSetSourceCR5(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);
   
    (void)regWriteReadback(&toprcmREG->R5SS0_CLK_DIV_SEL, M_ELEVEN,
															M_ZERO, clkDivVal);
    
    (void)regWriteReadback(&toprcmREG->R5SS_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}
Std_ReturnType Mcu_ClockSetSourceSYSCLK(uint32 clk_divId)
{
   /*Clk source is pre-configured*/
    uint32 clkDivVal = 0;

    clkDivVal = Mcu_getMultibitValue(clk_divId);

    /* Klocwork inspected. MISRA C 2012 Rule 11.4 
       Need to read the corresponding register address */    
    (void)regWriteReadback(&toprcmREG->SYS_CLK_DIV_VAL, M_ELEVEN, M_ZERO, clkDivVal);

    return E_OK;
}
Std_ReturnType Mcu_ClockSetSourceMCAN0(Mcu_ClkSourceIdType clk_srcId,
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_MCAN0_CLK_DIV_VAL, 
												M_ELEVEN, M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_MCAN0_CLK_SRC_SEL,
												M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}
Std_ReturnType Mcu_ClockSetSourceMCAN1(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_MCAN1_CLK_DIV_VAL, 
												M_ELEVEN, M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_MCAN1_CLK_SRC_SEL,
												M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceMCAN2(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_MCAN2_CLK_DIV_VAL, 
												M_ELEVEN, M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_MCAN2_CLK_SRC_SEL,
												M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceMCAN3(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_MCAN3_CLK_DIV_VAL, 
												M_ELEVEN, M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_MCAN3_CLK_SRC_SEL,
												M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceRTI0(Mcu_ClkSourceIdType clk_srcId,
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;

    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_RTI0_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);
    (void)regWriteReadback(&mssrcmREG->RCM_RTI0_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceRTI1(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;

    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_RTI1_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);
    (void)regWriteReadback(&mssrcmREG->RCM_RTI1_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceRTI2(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;

    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_RTI2_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);
    (void)regWriteReadback(&mssrcmREG->RCM_RTI2_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceRTI3(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;

    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_RTI3_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);
    (void)regWriteReadback(&mssrcmREG->RCM_RTI3_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceWDT0(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_WDT0_CLK_DIV_VAL, M_ELEVEN, 
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_WDT0_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}
Std_ReturnType Mcu_ClockSetSourceWDT1(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_WDT1_CLK_DIV_VAL, M_ELEVEN, 
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_WDT1_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}
Std_ReturnType Mcu_ClockSetSourceWDT2(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_WDT2_CLK_DIV_VAL, M_ELEVEN, 
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_WDT2_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceWDT3(Mcu_ClkSourceIdType clk_srcId, 
														       uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_WDT3_CLK_DIV_VAL, M_ELEVEN, 
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_WDT3_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceQSPI(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);    
    (void)regWriteReadback(&mssrcmREG->RCM_QSPI0_CLK_DIV_VAL, M_ELEVEN, 
															M_ZERO, clkDivVal);
    (void)regWriteReadback(&mssrcmREG->RCM_QSPI0_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceMCSPI0(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_MCSPI0_CLK_DIV_VAL, M_ELEVEN,
														    M_ZERO, clkDivVal);    
    (void)regWriteReadback(&mssrcmREG->RCM_MCSPI0_CLK_SRC_SEL, M_ELEVEN, M_ZERO, 
																	clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceMCSPI1(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_MCSPI1_CLK_DIV_VAL, M_ELEVEN,
														    M_ZERO, clkDivVal);    
    (void)regWriteReadback(&mssrcmREG->RCM_MCSPI1_CLK_SRC_SEL, M_ELEVEN, M_ZERO, 
																	clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceMCSPI2(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_MCSPI2_CLK_DIV_VAL, M_ELEVEN,
														    M_ZERO, clkDivVal);    
    (void)regWriteReadback(&mssrcmREG->RCM_MCSPI2_CLK_SRC_SEL, M_ELEVEN, M_ZERO, 
																	clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceMCSPI3(Mcu_ClkSourceIdType clk_srcId, 
                                                               uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_MCSPI3_CLK_DIV_VAL, M_ELEVEN,
														    M_ZERO, clkDivVal);    
    (void)regWriteReadback(&mssrcmREG->RCM_MCSPI3_CLK_SRC_SEL, M_ELEVEN, M_ZERO, 
																	clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceMCSPI4(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_MCSPI4_CLK_DIV_VAL, M_ELEVEN,
														    M_ZERO, clkDivVal);    
    (void)regWriteReadback(&mssrcmREG->RCM_MCSPI4_CLK_SRC_SEL, M_ELEVEN, M_ZERO, 
																	clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceI2C(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;

    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);
    (void)regWriteReadback(&mssrcmREG->RCM_I2C_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);
    (void)regWriteReadback(&mssrcmREG->RCM_I2C_CLK_SRC_SEL, M_ELEVEN,
															M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceSCI0(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_LIN0_UART0_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_LIN0_UART0_CLK_SRC_SEL, 
												M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceSCI1(Mcu_ClkSourceIdType clk_srcId, 
															uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_LIN1_UART1_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_LIN1_UART1_CLK_SRC_SEL, 
												M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceSCI2(Mcu_ClkSourceIdType clk_srcId, 
															uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_LIN2_UART2_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_LIN2_UART2_CLK_SRC_SEL, 
												M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceSCI3(Mcu_ClkSourceIdType clk_srcId, 
															uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_LIN3_UART3_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_LIN3_UART3_CLK_SRC_SEL, 
												M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceSCI4(Mcu_ClkSourceIdType clk_srcId, 
															uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_LIN4_UART4_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_LIN4_UART4_CLK_SRC_SEL, 
												M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceSCI5(Mcu_ClkSourceIdType clk_srcId, 
															uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId); 
    (void)regWriteReadback(&mssrcmREG->RCM_LIN5_UART5_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_LIN5_UART5_CLK_SRC_SEL, 
												M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceCPTS(Mcu_ClkSourceIdType clk_srcId,
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;
    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);
    (void)regWriteReadback(&mssrcmREG->RCM_CPTS_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&mssrcmREG->RCM_CPTS_CLK_SRC_SEL, M_ELEVEN, 
															M_ZERO, clkSrcVal);

    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceCPSW(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    uint32 clkSrcVal = 0;
    uint32 clkDivVal = 0;

    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);
   
    (void)regWriteReadback(&mssrcmREG->RCM_CPTS_CLK_DIV_VAL, M_ELEVEN, M_ZERO,
																	clkDivVal);
    
    (void)regWriteReadback(&mssrcmREG->RCM_CPTS_CLK_SRC_SEL, M_ELEVEN, M_ZERO, 
																	clkSrcVal);

    return E_OK;
}
Std_ReturnType Mcu_ClockSetSourceMcuClkout(Mcu_ClkSourceIdType clk_srcId, 
																uint32 clk_divId)
{
    uint32 clkSrcVal = 0;
    uint32 clkDivVal = 0;

    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);
   
    (void)regWriteReadback(&toprcmREG->R5SS0_CLK_DIV_SEL, M_ELEVEN, M_ZERO, 
																	clkDivVal);   
    (void)regWriteReadback(&toprcmREG->R5SS_CLK_SRC_SEL, M_ELEVEN, M_ZERO, 
																	clkSrcVal);

    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourcePmicClkout(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
    VAR(uint32, MCU_VAR) clkSrcVal = 0;
    VAR(uint32, MCU_VAR) clkDivVal = 0;

    clkSrcVal = Mcu_getMultibitValue((uint32)clk_srcId);
    clkDivVal = Mcu_getMultibitValue(clk_divId);  
    (void)regWriteReadback(&toprcmREG->TRCCLKOUT_DIV_VAL, M_ELEVEN, 
															M_ZERO, clkDivVal);   
    (void)regWriteReadback(&toprcmREG->TRCCLKOUT_CLK_SRC_SEL,
												  M_ELEVEN, M_ZERO, clkSrcVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceMII100Clk(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
   /*Clk source is fixed to DPLL_CORE_HSDIV0_CLKOUT1*/
    VAR(uint32, MCU_VAR) clkDivVal = 0;

    clkDivVal = Mcu_getMultibitValue(clk_divId);    
    (void)regWriteReadback(&mssrcmREG->RCM_RGMII_50_CLK_DIV_VAL, M_ELEVEN,
															M_ZERO, clkDivVal);
    return E_OK;
}
Std_ReturnType Mcu_ClockSetSourceMII10Clk(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
   /*Clk source is fixed to DPLL_CORE_HSDIV0_CLKOUT1*/
    VAR(uint32, MCU_VAR) clkDivVal = 0;
  
	clkDivVal = (clk_divId << 16) | (clk_divId << 8) | (clk_divId);	
    (void)regWriteReadback(&mssrcmREG->RCM_RGMII_5_CLK_DIV_VAL, M_TWENTY_THREE,
															M_ZERO, clkDivVal);
    return E_OK;
}

Std_ReturnType Mcu_ClockSetSourceRGMIClk(Mcu_ClkSourceIdType clk_srcId, 
															   uint32 clk_divId)
{
   /*Clk source is fixed to DPLL_CORE_HSDIV0_CLKOUT1*/
    VAR(uint32, MCU_VAR) clkDivVal = 0;

    clkDivVal = Mcu_getMultibitValue(clk_divId);   
    (void)regWriteReadback(&mssrcmREG->RCM_RGMII_250_CLK_DIV_VAL, 
												M_ELEVEN, M_ZERO, clkDivVal);
    return E_OK;
}

#endif

/* MMR Unlock Functions */

void Mcu_MMR_unlock(uint32 *kick0, uint32 *kick1)
{
    *kick0 = KICK0_UNLOCK_VAL_MCU;
    *kick1 = KICK1_UNLOCK_VAL_MCU;
}

/**
 * \brief  Unlocks MMR registers
 *
 * \return  Std_ReturnType
 */
void Mcu_unlockMMR(void)
{
    uint32 *lock0;
    uint32 *lock1;
   
    lock0 = (uint32 *)(&toprcmREG->LOCK0_KICK0);  
    lock1 = (uint32 *)(&toprcmREG->LOCK0_KICK1);

    Mcu_MMR_unlock(lock0, lock1);
    lock0 = (uint32 *)(&mssrcmREG->RCM_LOCK0_KICK0);  
    lock1 = (uint32 *)(&mssrcmREG->RCM_LOCK0_KICK1);

    Mcu_MMR_unlock(lock0, lock1);
    lock0 = (uint32 *)(&ctrlREG->LOCK0_KICK0);   
    lock1 = (uint32 *)(&ctrlREG->LOCK0_KICK1);

    Mcu_MMR_unlock(lock0, lock1);
}

#if (STD_OFF == MCU_NO_PLL)
/* PLL Init Functions */

/**
 * \brief  Function to initialize all the PLL clocks
 */
Std_ReturnType Mcu_PLLInitAll(const Mcu_ConfigType *Mcu_DrvObj)
{	

	Std_ReturnType init_pll_return = E_OK;

	TickType startCount = 0U, tempCount, elapsedCount = 0U;
	uint32  lock_status;
	uint32  M, D;
   
	Mcu_PllClkDivType pllClk1 = 
			Mcu_DrvObj->Mcu_PllConfig[Mcu_DrvObj->Mcu_PllSourceId].Mcu_PllClk1;
	Mcu_PllClkDivType pllClk2 = 
			Mcu_DrvObj->Mcu_PllConfig[Mcu_DrvObj->Mcu_PllSourceId].Mcu_PllClk2;
  

	if(Mcu_DrvObj->Mcu_PllSourceId == MCU_CLKSRC_DPLL)
	{
		/* NON-ZERO => Enabled */
		if(pllClk1.MCU_PLL_CLKOUT) /* DPLL_CORE */
		{
		/* APPLJ-1 Setting
		* CLOCKOUT = M/(N+1) * CLKINP * (1/M2)
		*/
		/* Switch clock source to CPU CLK,(MCU_CLKSRC_0 is XTALCLK) */
		M = (pllClk1.MCU_PLL_CLKOUT * (pllClk1.N + M_ONE) * pllClk1.M2) /
																XTAL_CLKINP;

		Mcu_ClockSetSourceCR5(MCU_CLKSRC_0, M_ZERO);
		Mcu_ClockSetSourceSYSCLK(M_ZERO);

		(void)regWriteReadback(&toprcmREG->PLL_CORE_CLKCTRL, M_ZERO, M_ZERO, M_ZERO); /* Soft reset Pll */

		(void)regWriteReadback(&toprcmREG->PLL_CORE_M2NDIV, M_TWENTY_TWO, M_SIXTEEN, pllClk1.M2);    /* M2 - post divider */
		(void)regWriteReadback(&toprcmREG->PLL_CORE_M2NDIV, M_SEVEN, M_ZERO, pllClk1.N);       /* N - input clock divider */
		(void)regWriteReadback(&toprcmREG->PLL_CORE_MN2DIV, M_ELEVEN, M_ZERO, M);       /* M - multiplier */
		(void)regWriteReadback(&toprcmREG->PLL_CORE_MN2DIV, M_NINETEEN, M_SIXTEEN, M_ZERO);    /* N2 - divider for by-pass */
		(void)regWriteReadback(&toprcmREG->PLL_CORE_FRACDIV, M_SEVENTEEN, M_ZERO, pllClk1.FracM);    /* fractional multiplier */
		(void)regWriteReadback(&toprcmREG->PLL_CORE_FRACDIV, M_THIRTY_ONE, M_TWENTY_FOUR, pllClk1.sdDiv); /* sigma Delta divider */

		// /* program CLKDCOLDOEN[29] = 1, IDLE[23] = 0, CLKDCOLDOPWDNZ[17] = 1, SELFREQDCO[12:10] = 4 */
		(void)regWriteReadback(&toprcmREG->PLL_CORE_CLKCTRL, M_TWENTY_NINE, M_TWENTY_NINE, M_ONE);
		(void)regWriteReadback(&toprcmREG->PLL_CORE_CLKCTRL, M_TWENTY_THREE, M_TWENTY_THREE, M_ZERO);
		(void)regWriteReadback(&toprcmREG->PLL_CORE_CLKCTRL, M_SEVENTEEN, M_SEVENTEEN, M_ONE);
		(void)regWriteReadback(&toprcmREG->PLL_CORE_CLKCTRL, M_TWELVE, M_TEN, M_FOUR);

		(void)regWriteReadback(&toprcmREG->PLL_CORE_TENABLE, M_ZERO, M_ZERO, M_ONE);       /* TENABLE High */
		(void)regWriteReadback(&toprcmREG->PLL_CORE_CLKCTRL, M_ZERO, M_ZERO, M_ONE);   /* out of reset Pll */
		(void)regWriteReadback(&toprcmREG->PLL_CORE_TENABLE, M_ZERO, M_ZERO, M_ZERO);       /* TENABLE Low */
		(void)regWriteReadback(&toprcmREG->PLL_CORE_TENABLEDIV, M_ZERO, M_ZERO, M_ONE);    /* TENABLEDIV High */
		(void)regWriteReadback(&toprcmREG->PLL_CORE_TENABLEDIV, M_ZERO, M_ZERO, M_ZERO);    /* TENABLEDIV Low */

		/* APPLJ-1  :  loop check to PLLLOCK DONE */
		TickType    startCount = 0U, tempCount, elapsedCount = 0U;

        (void)GetCounterValue(MCU_OS_COUNTER_ID, &startCount);
        {
			/* Below API can change start time, so use temp variable */
			tempCount = startCount;

			/* Wait for FSM to to come out of reset */
            do
            {
                lock_status = HW_RD_REG32(&toprcmREG->PLL_CORE_STATUS); /* PHASELOCK[10] */
                (void)GetElapsedValue(
                             MCU_OS_COUNTER_ID,
                             &tempCount,
                             &elapsedCount);
                if (elapsedCount >= (uint32) MCU_PLL_TIMEOUT_DURATION)
				{
               /* timeout */
					(void) Dem_SetEventStatus (MCU_E_CLOCK_FAILURE, DEM_EVENT_STATUS_FAILED);
					init_pll_return = E_NOT_OK;
					break;
				}
			} while(0x400 != (lock_status & 0x400)) ;
        }
		 /* HSDIV-1 Settings
		  * CLKOUT0_DIV[4:0] = D -- CLKOUT/(D+1)
		  */
			if(pllClk1.MCU_PLL_HSDIV0) /* NON-ZERO => Enabled */
			{
				D = (pllClk1.MCU_PLL_CLKOUT / pllClk1.MCU_PLL_HSDIV0) - 1;
				(void)regWriteReadback(&toprcmREG->PLL_CORE_HSDIVIDER_CLKOUT0, M_FOUR, M_ZERO, D);
			}
			if(pllClk1.MCU_PLL_HSDIV1)
			{
				D = (pllClk1.MCU_PLL_CLKOUT / pllClk1.MCU_PLL_HSDIV1) - 1;
				(void)regWriteReadback(&toprcmREG->PLL_CORE_HSDIVIDER_CLKOUT1, M_FOUR, M_ZERO, D);
			}
			if(pllClk1.MCU_PLL_HSDIV2)
			{
				D = (pllClk1.MCU_PLL_CLKOUT / pllClk1.MCU_PLL_HSDIV2) - 1;
				(void)regWriteReadback(&toprcmREG->PLL_CORE_HSDIVIDER_CLKOUT2, M_FOUR, M_ZERO, D);
			}

			if(pllClk1.MCU_PLL_HSDIV0 || pllClk1.MCU_PLL_HSDIV1 || pllClk1.MCU_PLL_HSDIV2)
			{
				(void)regWriteReadback(&toprcmREG->PLL_CORE_HSDIVIDER, M_TWO, M_TWO, M_ONE);     /* HSDIVIDER[2] = 1 */
				(void)regWriteReadback(&toprcmREG->PLL_CORE_HSDIVIDER, M_TWO, M_TWO, M_ZERO); /* HSDIVIDER[2] = 0 */
				if(pllClk1.MCU_PLL_HSDIV0)
				{
				   (void)regWriteReadback(&toprcmREG->PLL_CORE_HSDIVIDER_CLKOUT0, M_EIGHT, M_EIGHT, M_ONE);  /* +CLKOUT0_GATE[8] = 1 */
				}
				if(pllClk1.MCU_PLL_HSDIV1)
				{
				   regWriteReadback(&toprcmREG->PLL_CORE_HSDIVIDER_CLKOUT1, M_EIGHT, M_EIGHT, M_ONE); /* +CLKOUT1_GATE[8] = 1 */
				}
				if(pllClk1.MCU_PLL_HSDIV2)
				{
				   (void)regWriteReadback(&toprcmREG->PLL_CORE_HSDIVIDER_CLKOUT2, M_EIGHT, M_EIGHT, M_ONE); /* +CLKOUT2_GATE[8] = 1 */
				}
			}
			/* Restore clock source to DPLL (MCU_CLKSRC_2 is DPLL_CORE_HSDIV0_CLKOUT0)*/
			Mcu_ClockSetSourceCR5(MCU_CLKSRC_2, M_ZERO);
			Mcu_ClockSetSourceSYSCLK(M_ONE);
		}

		if(pllClk2.MCU_PLL_CLKOUT) /* DPLL_PER */
		{
			/* APPLJ-3 Setting
			* CLOCKOUT = M/(N+1) * CLKINP * (1/M2)
			*/
			M = (pllClk2.MCU_PLL_CLKOUT * (pllClk2.N + 1) * pllClk2.M2) /
																	XTAL_CLKINP;
			(void)regWriteReadback(&toprcmREG->PLL_PER_M2NDIV, M_TWENTY_TWO, M_SIXTEEN, pllClk2.M2);  /* M2 - post divider */
			(void)regWriteReadback(&toprcmREG->PLL_PER_M2NDIV, M_SEVEN, M_ZERO, pllClk2.N);    /* N - input clock divider */
			(void)regWriteReadback(&toprcmREG->PLL_PER_MN2DIV, M_ELEVEN, M_ZERO, M); /* M - multiplier */
			(void)regWriteReadback(&toprcmREG->PLL_PER_MN2DIV, M_NINETEEN, M_SIXTEEN,  M_ZERO);    /* N2 - divider for by-pass */
			(void)regWriteReadback(&toprcmREG->PLL_PER_FRACDIV, M_SEVENTEEN, M_ZERO, pllClk2.FracM);         /* fractional multiplier */
			(void)regWriteReadback(&toprcmREG->PLL_PER_FRACDIV, M_THIRTY_ONE, M_TWENTY_FOUR, pllClk2.sdDiv); /* sigma Delta divider */

			/* program CLKDCOLDOEN[29] = 1, IDLE[23] = 0, CLKDCOLDOPWDNZ[17] = 1, SELFREQDCO[12:10] = 4 */							  
			(void)regWriteReadback(&toprcmREG->PLL_PER_CLKCTRL, M_TWENTY_NINE, M_TWENTY_NINE, M_ONE); 
			(void)regWriteReadback(&toprcmREG->PLL_PER_CLKCTRL, M_TWENTY_THREE, M_TWENTY_THREE, M_ZERO); /*Idle */
			(void)regWriteReadback(&toprcmREG->PLL_PER_CLKCTRL, M_SEVENTEEN, M_SEVENTEEN, M_ONE); 
			(void)regWriteReadback(&toprcmREG->PLL_PER_CLKCTRL, M_TWELVE, M_TEN, M_FOUR);

			(void)regWriteReadback(&toprcmREG->PLL_PER_CLKCTRL, M_ZERO, M_ZERO, M_ZERO); /* Soft reset Pll */
			(void)regWriteReadback(&toprcmREG->PLL_PER_TENABLE, M_ZERO, M_ZERO, M_ONE); /* TENABLE High */
			(void)regWriteReadback(&toprcmREG->PLL_PER_CLKCTRL, M_ZERO, M_ZERO, M_ONE); /* out of reset Pll */
			(void)regWriteReadback(&toprcmREG->PLL_PER_TENABLE, M_ZERO, M_ZERO, M_ZERO); /* TENABLE Low */
			(void)regWriteReadback(&toprcmREG->PLL_PER_TENABLEDIV, M_ZERO, M_ZERO, M_ONE);  /* TENABLEDIV High */
			(void)regWriteReadback(&toprcmREG->PLL_PER_TENABLEDIV, M_ZERO, M_ZERO, M_ZERO); /* TENABLEDIV Low */
											 
			TickType    startCount = 0U, tempCount, elapsedCount = 0U;

			(void) GetCounterValue (MCU_OS_COUNTER_ID, &startCount);
			{
				/* Below API can change start time, so use temp variable */
				tempCount = startCount;

				/* APPLJ-3 : loop check to PLLLOCK DONE */
				do
				{
					lock_status = HW_RD_REG32(&toprcmREG->PLL_PER_STATUS);
			 
					(void)GetElapsedValue (
							 MCU_OS_COUNTER_ID,
							 &tempCount,
							 &elapsedCount);
					if (elapsedCount >= (uint32) MCU_PLL_TIMEOUT_DURATION)
					{
						/* timeout */
						(void) Dem_SetEventStatus (
							MCU_E_CLOCK_FAILURE, DEM_EVENT_STATUS_FAILED);
						init_pll_return = E_NOT_OK;
						break;
					}
				} while(0x400 != (lock_status & 0x400)) ;
				
			} 

			/* HSDIV-2 Settings
			* CLKOUT0_DIV[4:0] = D -- CLKOUT/(D+1)
			*/
			if(pllClk2.MCU_PLL_HSDIV0)
			{
				D = (pllClk2.MCU_PLL_CLKOUT / pllClk2.MCU_PLL_HSDIV0) - 1;
				(void)regWriteReadback(&toprcmREG->PLL_PER_HSDIVIDER_CLKOUT0, M_FOUR, M_ZERO, D);
			}
			if(pllClk2.MCU_PLL_HSDIV1)
			{
				D = (pllClk2.MCU_PLL_CLKOUT / pllClk2.MCU_PLL_HSDIV1) - 1;
				(void)regWriteReadback(&toprcmREG->PLL_PER_HSDIVIDER_CLKOUT1, M_FOUR, M_ZERO, D);
			}
			if(pllClk2.MCU_PLL_HSDIV0 || pllClk2.MCU_PLL_HSDIV1)
			{
				(void)regWriteReadback(&toprcmREG->PLL_PER_HSDIVIDER, M_TWO, M_TWO, M_ONE);    /* HSDIVIDER[2] = 1 */
				(void)regWriteReadback(&toprcmREG->PLL_PER_HSDIVIDER, M_TWO, M_TWO, M_ZERO);    /* HSDIVIDER[2] = 0 */
				if(pllClk2.MCU_PLL_HSDIV0)
				{
					(void)regWriteReadback(&toprcmREG->PLL_PER_HSDIVIDER_CLKOUT0, M_EIGHT,
									   M_EIGHT, M_ONE);   /* +CLKOUT0_GATE[8] = 1 */
				}
				if(pllClk2.MCU_PLL_HSDIV1)
				{
					(void)regWriteReadback(&toprcmREG->PLL_PER_HSDIVIDER_CLKOUT1, M_EIGHT,
									   M_EIGHT, M_ONE);   /* +CLKOUT1_GATE[8] = 1 */
				}

			}
		}						    
    }
	
   return init_pll_return;
   
}
#endif /* STD_OFF == MCU_NO_PLL */

#define MCU_STOP_SEC_CODE
#include "Mcu_MemMap.h"
