/* ======================================================================
 *   Copyright (c) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     Port.h
 *
 *  \brief    This file contains interface header for PORT MCAL driver
 *
 */
/*  ----------------------------------------------------------------------------
*  FILE DESCRIPTION
*  -----------------------------------------------------------------------------
*         File:  Port.h
*      Project:  AM263 MCAL Driver
*       Module:  PortDriver
*
*  Description:  This module implements a driver in ANSI C programming language 
*                to centralize the configuration of ports and pins which are
*                used by more than one driver module.
*                With this driver you can:
*                - Configure all port pins
*                - Initialize the configuration
*                - Refresh the configuration (EMC purposes)
*                - Switch the port pin configuration during runtime
*******************************************************************************/

/**
 *  \defgroup MCAL_PORT_API PORT Driver API
 *
 *  This module shall provide the service for initializing the whole PORT
 *  structure of the microcontroller.<br>
 *  Many ports and port pins can be assigned to various functionalities, e.g.
 *      - DIO
 *      - SPI
 *      - RTI
 *  For this reason, there shall be an overall configuration and initialization
 *  of this port structure.
 *  The configuration and mode of these port pins is microcontroller and ECU
 *  dependent
 *
 *  @sa MCAL_PORT_CFG
 *  @{
 */

/*
 * Below are the global requirements which are met by this PORT
 * driver which can't be mapped to a particular source ID
 */

/* Design : PORT_DesignId_001, PORT_DesignId_002, PORT_DesignId_009 */

/*
 * Requirements : SWS_Port_00133, SWS_Port_00138, SWS_Port_00006,
 *                SWS_Port_00207, SWS_Port_00076, SWS_Port_00208,
 *                SWS_Port_00075, SWS_Port_00129, SWS_Port_00146,
 *                SWS_Port_00147, SWS_Port_00148,
 *                AUTORADAR_MCAL-6
 */

/* Design : PORT_DesignId_011 */
/*
 * Below are the PORT's module environment requirements which can't be mapped
 * to this driver
 */
/*
 * Requirements : SWS_Port_00078, SWS_Port_00071, SWS_Port_00215,
 *                SWS_Port_00217, SWS_Port_00218,
 */
#ifndef PORT_H_
#define PORT_H_

/* ========================================================================== */
/*                          Include Files                                  */
/* ========================================================================== */

/* Requirements : SWS_Port_00080, SWS_Port_00130 */
#include "Std_Types.h"
/* Requirements : SWS_Port_00080, SWS_Port_00130 */
#include "Port_Cfg.h"

#ifdef __cplusplus
extern "C"
{
#endif
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/**
 *  \name PORT Driver Module SW Version Info
 *
 *  Defines for PORT Driver version used for compatibility checks
 *  @{
 */
/** \brief Driver Implementation Major Version */
#define PORT_SW_MAJOR_VERSION           (8U)
/** \brief Driver Implementation Minor Version */
#define PORT_SW_MINOR_VERSION           (6U)
/** \brief Driver Implementation Patch Version */
#define PORT_SW_PATCH_VERSION           (2U)
/** @} */

/**
 *  \name PORT Driver Module AUTOSAR Version Info
 *
 *  Defines for PORT Driver AUTOSAR version used for compatibility checks
 *  @{
 */
/** \brief AUTOSAR Major version specification implemented by PORT Driver */
#define PORT_AR_RELEASE_MAJOR_VERSION       (4U)
/** \brief AUTOSAR Minor  version specification implemented by PORT Driver */
#define PORT_AR_RELEASE_MINOR_VERSION       (3U)
/** \brief AUTOSAR Patch version specification implemented by PORT Driver */
#define PORT_AR_RELEASE_REVISION_VERSION    (1U)
/** @} */

/**
 *  \name PORT Driver ID Info
 *  @{
 */
/** \brief Texas Instruments Vendor ID */
#define PORT_VENDOR_ID                  ((uint16) 44U)
/** \brief PORT Driver Module ID */
#define PORT_MODULE_ID                  ((uint16) 124U)
/** \brief PORT Driver Instance ID */
#define PORT_INSTANCE_ID                (0U)
/** @} */

/* Design : PORT_DesignId_003 */
/* Requirements : SWS_Port_00051 */
/**
 *  \name PORT Error Codes
 *
 *  Error codes returned by Driver functions
 *  @{
 */
#ifndef PORT_E_PARAM_PIN
/** \brief ERROR: Invalid Port Pin ID requested */
#define PORT_E_PARAM_PIN                ((uint8) 0x0AU)
#endif

#ifndef PORT_E_DIRECTION_UNCHANGEABLE
/** \brief ERROR:Port Pin not configured as changeable */
#define PORT_E_DIRECTION_UNCHANGEABLE   ((uint8) 0x0BU)
#endif

#ifndef PORT_E_INIT_FAILED
/** \brief ERROR:Init service called with wrong parameter */
#define PORT_E_INIT_FAILED             ((uint8) 0x0CU)
#endif

#ifndef PORT_E_PARAM_INVALID_MODE
/** \brief ERROR:Invalid mode setting for PORT pin */
#define PORT_E_PARAM_INVALID_MODE       ((uint8) 0x0DU)
#endif

#ifndef PORT_E_MODE_UNCHANGEABLE
/** \brief ERROR:PORT pin disallows mode change */
#define PORT_E_MODE_UNCHANGEABLE        ((uint8) 0x0EU)
#endif

#ifndef PORT_E_UNINIT
/** \brief ERROR:PORT API called without init being invoked */
#define PORT_E_UNINIT                   ((uint8) 0x0FU)
#endif

#ifndef PORT_E_PARAM_POINTER
/** \brief ERROR:API invoked with NULL_PTR */
#define PORT_E_PARAM_POINTER            ((uint8) 0x10U)
#endif
/** @} */

/**
 *  \name PORT Service Ids
 *
 *  The Service Id is one of the argument to Det_ReportError function and is
 *  used to identify the source of the error
 *  @{
 */
/* PORT API service identification */
/** \brief Port_Init() */
#define PORT_SID_INIT                   ((uint8) 0x0U)
/** \brief Port_SetPinDirection() */
#define PORT_SID_SET_PIN_DIR            ((uint8) 0x1U)
/** \brief Port_RefreshPortDirection() */
#define PORT_SID_REFRESH_PORT_DIR       ((uint8) 0x2U)
/** \brief Port_GetVersionInfo() */
#define PORT_SID_GET_VERSION_INFO       ((uint8) 0x3U)
/** \brief Port_SetPinMode() */
#define PORT_SID_SET_PIN_MODE           ((uint8) 0x4U)
/** \brief Port_CetConfigRegValues() */
#define PORT_SID_SET_CONFIGREG              ((uint8)0x5U)
/** @} */


/**
 *  \name PORT DIO related defines
 *  @{
 */
/**
 *  \brief PORT DIO Invalid GPIO register Id
 */
#define PORT_DIO_INVALID_REG_ID          (0xFFU)

/**
 *  \brief PORT DIO Invalid GPIO Channel Id
 */
#define PORT_DIO_INVALID_CH_ID          (0xFFU)
/** @} */

/** \brief Valid Port width for the AWR1XX devices */
#define PORT_GPIO_PORT_WIDTH                 (0x8U)


#if (STD_ON == PORT_SAFETI_API)
typedef struct{
    uint32 configRegs[10];
} Port_ConfigRegsType;
#endif


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* Requirements : SWS_Port_00230, SWS_Port_00046 */
/**
 *  \brief Enum of Port Direction used in Port_SetPinDirection()
 */
typedef enum
{
    /* Requirements : SWS_Port_00220 */
    PORT_PIN_OUT,
    /**< PORT_PIN_OUT */
    /* Requirements : SWS_Port_00220 */
    PORT_PIN_IN,
    /**< PORT_PIN_IN */
    PORT_PIN_DEFAULT
    /**< PORT_PIN_DEFAULT */
} Port_PinDirectionType;

/**
 *  \brief Enum of Port Level
 */
typedef enum
{
	PORT_PIN_LEVEL_LOW,
    PORT_PIN_LEVEL_HIGH,
} Port_PinLevelValueType;

/**
 *  \brief Enum of Port Slew Control
 */
typedef enum
{
    PORT_PIN_SLEWCONTROL_FAST_SLEW,
    PORT_PIN_SLEWCONTROL_SLOW_SLEW,
    PORT_PIN_SLEWCONTROL_DEFAULT,
} Port_PinSlewControlType;

/**
 *  \brief Enum of Port Pin Output override selection
 */
typedef enum
{
    PORT_PIN_DISABLE_OUTPUT_OVERRIDE,
    PORT_PIN_ENABLE_OUTPUT_OVERRIDE,
    PORT_PIN_OUTPUT_RETAIN_HW_CTRL
} Port_PinOutputOverrideCtrl;

/**
 *  \brief Enum of Port Pin Input override selection
 */
typedef enum
{
    PORT_PIN_DISABLE_INPUT_OVERRIDE,
    PORT_PIN_ENABLE_INPUT_OVERRIDE,
    PORT_PIN_INPUT_RETAIN_HW_CTRL
} Port_PinInputOverrideCtrl;
/**
 *  \brief Enum of Port Pin Up/Down Type selection
 */
typedef enum
{
    PORT_PIN_PULLTYPE_PULLDOWN,
    PORT_PIN_PULLTYPE_PULLUP,
    PORT_PIN_PULLTYPE_DEFAULT
} Port_PinPullSelectType;


/**
 *  \brief Enum of Port Pin pull inihibit selection
 */
typedef enum
{
    PORT_PIN_PULL_INHIBIT_ENABLE,
    PORT_PIN_PULL_INHIBIT_DISABLE,
    PORT_PIN_PULL_INHIBIT_DEFAULT

} Port_PinPullInhibitEnableType;


/**
 *  \brief Enum of Dio module data direction control
 */
typedef enum
{
    PORT_DIO_DATA_DIRECTION_INPUT,
    PORT_DIO_DATA_DIRECTION_OUTPUT,
    PORT_DIO_DATA_DIRECTION_DEFAULT

} Port_DioDataDir;


/**
 *  \brief Enum of DIO module Data Input register that reflects the current
 *          state of the pins
 */
typedef enum
{
    PORT_DIO_DATA_INPUT_LOW,
    PORT_DIO_DATA_INPUT_HIGH,
    PORT_DIO_DATA_INPUT_DEFAULT
} Port_DioDataIn;


/**
 *  \brief Enum of DIO module Data Output register that specify the output state of the
 *    pins when configured as output
 */
typedef enum
{
    PORT_DIO_DATA_OUTPUT_LOW,
    PORT_DIO_DATA_OUTPUT_HIGH,
    PORT_DIO_DATA_OUTPUT_DEFAULT
} Port_DioDataOut;

/**
 *  \brief Enum of DIO module Open drain capability
 */
typedef enum
{
    PORT_DIO_PDR_PUSHPULL_MODE,
    PORT_DIO_PDR_OPEN_DRAIN_MODE,
    PORT_DIO_PDR_DEFAULT
} Port_DioOpenDrain_Mode;


/**
 *  \brief Enum of DIO module SINGLE/BOTH Edge trigger interrupt
 */
typedef enum
{
    PORT_DIO_SINGLE_EDGE_INTERRUPT,
    PORT_DIO_BOTH_EDGE_INTERRUPT,
    PORT_DIO_DEFAULT_INTERRUPT
} Port_DioIntrDET;

/**
 *  \brief Enum of DIO module Rising/Falling edge interrupt
 */
typedef enum
{
    PORT_DIO_FALLING_EDGE_INTERRUPT,
    PORT_DIO_RISING_EDGE_INTERRUPT,
    PORT_DIO_DEFAULT_EDGE_INTERRUPT
} Port_DioIntrPol;

/**
 *  \brief Enum of DIO module high/low level  interrupt
 */
typedef enum
{
    PORT_DIO_LOW_LEVEL_INTERRUPT,
    PORT_DIO_HIGH_LEVEL_INTERRUPT,
    PORT_DIO_DEFAULT_LEVEL_INTERRUPT
} Port_DioIntrLvl;


typedef enum
{
    PORT_RISING_EDGE,
	PORT_FALLING_EDGE,
	BOTH_EDGE
} Port_EdgeTrig;

/**
 *  \brief Enum of DIO module idle mode configuration
 */
typedef enum
{
    PORT_DIO_IDLEMODE_DEFAULT,
    /**< Do not configure IDLEMODE. Retains default configuration */
} Port_DioIdleModeType;

/**
 *  \brief Enum of DIO module Clock gating ratio for event detection
 */
typedef enum
{
    PORT_DIO_UNGATE = 0xFFFF,
} Port_DioClockGatingRatioType;


/**
 *  \brief Enum of Port input inversion
 */
typedef enum
{
    PORT_NONINV,
    PORT_INV,
    PORT_INVTYPE_DEFAULT
} Port_InputInversion;


/**
 *  \brief Enum of Port input qualifier type select
 */
typedef enum
{
    PORT_SYNC_QUAL,
    PORT_THREE_SAMPLE_QUAL,
    PORT_SIX_SAMPLE_QUAL,
    PORT_ASYNC_QUAL,
    PORT_QUALTYPE_DEFAULT
} Port_InputQualType;


/** \brief Type for symbolic name of Port pins */
/* Requirements : SWS_Port_00229, SWS_Port_00219 */
typedef uint16 Port_PinType;

/* Requirements : SWS_Port_00231, SWS_Port_00124, SWS_Port_00221 */
/**
 *  \brief Enum of Port Pin Mode selection
 *
 *  Module level pin enable is workaround since MCAL configurator is not
 *  available. Will be removed in the next release
 */


typedef enum
{
    PORT_PIN_MODE_GPIOAB,
    /**< Set Pin for  GPIO Port AB mode */
   
    PORT_PIN_MODE_GPIOCD,
    /**< Set Pin for GPIO Port CD mode */
   
	PORT_PIN_MODE_GPIOEF,
    /**< Set Pin for  GPIO Port EF mode */
    
    PORT_PIN_MODE_GPIOGH,
    /**< Set Pin for GPIO Port GH mode */
  
	PORT_PIN_MODE_GPIOI,
    /**< Set Pin for GPIO mode */

	PORT_PIN_MODE_MCAN0,
    /**< Set Pin for MCAN0 (CAN FD) mode */
    PORT_PIN_MODE_MCAN1,
    /**< Set Pin for MCAN1 (CAN FD) mode */
	PORT_PIN_MODE_MCAN2,   
	/**< Set Pin for MCAN2(CAN FD) mode */
	PORT_PIN_MODE_MCAN3,   
	/**< Set Pin for MCAN3(CAN FD) mode */
	
    PORT_PIN_MODE_SPI0,
    /**< Set Pin for MIBSPI0 mode */
    PORT_PIN_MODE_SPI1,
    /**< Set Pin for MIBSPI1 mode */
	PORT_PIN_MODE_SPI2,
    /**< Set Pin for MIBSPI2 mode */
	PORT_PIN_MODE_SPI3,
    /**< Set Pin for MIBSPI3 mode */
	PORT_PIN_MODE_SPI4,
    /**< Set Pin for MIBSPI4 mode */    
      
    PORT_PIN_MODE_QSPI0,
    /**< Set Pin for QSPI mode */
    PORT_PIN_MODE_JTAG,
    /**< Set Pin for JTAG mode */
    PORT_PIN_MODE_TRACE,
    /**< Set Pin for TRACE mode */    
    
	PORT_PIN_MODE_I2C0,
    /**< Set Pin for I2C0 mode */
    PORT_PIN_MODE_I2C1,
    /**< Set Pin for I2C1 mode */
    PORT_PIN_MODE_I2C2,
    /**< Set Pin for I2C2 mode */
    PORT_PIN_MODE_I2C3,
    /**< Set Pin for I2C3 mode */
   
    PORT_PIN_MODE_MII,
    /**< Set Pin for MII mode */
    PORT_PIN_MODE_RMII1,
    /**< Set Pin for RMII1 mode */
	PORT_PIN_MODE_RMII2,
    /**< Set Pin for RMII2 mode */
	
    PORT_PIN_MODE_RGMII1,
    /**< Set Pin for RGMII1 mode */
    PORT_PIN_MODE_RGMII2,
    /**< Set Pin for RGMII2 mode */
    
	PORT_PIN_MODE_MDIO0,
    /**< Set Pin for MDIO mode */
    PORT_PIN_MODE_CPTS0,
    /**< Set Pin for CPTS0 mode */
    PORT_PIN_MODE_UART0,
    /**< Set Pin for UART0 modes */
    PORT_PIN_MODE_UART1,
    /**< Set Pin for UART1 modes */
    PORT_PIN_MODE_UART2,
    /**< Set Pin for UART2 modes */
    PORT_PIN_MODE_UART3,
    /**< Set Pin for UART3 modes */ 
    PORT_PIN_MODE_UART4,
    /**< Set Pin for UART3 modes */ 
    PORT_PIN_MODE_UART5,
    /**< Set Pin for UART5 modes */ 
    PORT_PIN_MODE_ECAP0,
    /**< Set Pin for ECAP0 mode */
    PORT_PIN_MODE_PR0_UART0,
    /**< Set Pin for UART0 mode */
    PORT_PIN_MODE_LIN0,
    /**< Set Pin for LON0 mode */
	PORT_PIN_MODE_LIN1,
    /**< Set Pin for LIN1 mode */
    PORT_PIN_MODE_LIN2,
    /**< Set Pin for LIN2 mode */
	PORT_PIN_MODE_LIN3,
    /**< Set Pin for LIN3 mode */
	PORT_PIN_MODE_LIN4,
    /**< Set Pin for LIN mode */
    PORT_PIN_MODE_EPWM0,
    /**< Set Pin for EPWM0 mode */
    PORT_PIN_MODE_EPWM1,
    /**< Set Pin for EPWM1 mode */
    PORT_PIN_MODE_EPWM2,
    /**< Set Pin for EPWM2 mode */
	PORT_PIN_MODE_EPWM3,
    /**< Set Pin for EPWM3A mode */
	PORT_PIN_MODE_EPWM4,
	 /**< Set Pin for EPWM1A mode */
	PORT_PIN_MODE_EPWM5,
	 /**< Set Pin for EPWM5A mode */
    PORT_PIN_MODE_EPWM6,
	 /**< Set Pin for EPWM6A mode */
    PORT_PIN_MODE_EPWM7,
	 /**< Set Pin for EPWM7A mode */	 
	PORT_PIN_MODE_EPWM8,
	 /**< Set Pin for EPWM8A mode */
	PORT_PIN_MODE_EPWM9,
	 /**< Set Pin for EPWM9A mode */
	PORT_PIN_MODE_EPWM10,
	 /**< Set Pin for EPWM510. mode */
    PORT_PIN_MODE_EPWM11,
	 /**< Set Pin for EPWM11A mode */
	PORT_PIN_MODE_EPWM12,
	 /**< Set Pin for EPWM12A mode */
	PORT_PIN_MODE_EPWM13,
	 /**< Set Pin for EPWM13A mode */
	PORT_PIN_MODE_EPWM14,
	 /**< Set Pin for EPWM14A mode */
	PORT_PIN_MODE_EPWM15,
	 /**< Set Pin for EPWM15A mode */
	PORT_PIN_MODE_EPWM16,
	 /**< Set Pin for EPWM16A mode */
	PORT_PIN_MODE_EPWM17,
	 /**< Set Pin for EPWM17A mode */
	PORT_PIN_MODE_EPWM18,
	 /**< Set Pin for EPWM18A mode */
	PORT_PIN_MODE_EPWM19,
	 /**< Set Pin for EPWM19A mode */
	PORT_PIN_MODE_EPWM20,
	 /**< Set Pin for EPWM20A mode */
	PORT_PIN_MODE_EPWM21,
	 /**< Set Pin for EPWM21A mode */
	PORT_PIN_MODE_EPWM22,
	 /**< Set Pin for EPW22A mode */
	PORT_PIN_MODE_EPWM23,
	 /**< Set Pin for EPWM23A mode */
    PORT_PIN_MODE_EPWM24,
     /**< Set Pin for EPWM24A mode */
    PORT_PIN_MODE_EPWM25,
     /**< Set Pin for EPWM25A mode */
    PORT_PIN_MODE_EPWM26,
     /**< Set Pin for EPWM26A mode */
    PORT_PIN_MODE_EPWM27,
     /**< Set Pin for EPWM27A mode */
    PORT_PIN_MODE_EPWM28,
     /**< Set Pin for EPWM28A mode */
    PORT_PIN_MODE_EPWM29,
     /**< Set Pin for EPWM29A mode */
    PORT_PIN_MODE_EPWM30,
     /**< Set Pin for EPWM30A mode */
    PORT_PIN_MODE_EPWM31,
     /**< Set Pin for EPWM31A mode */
    PORT_PIN_MODE_XBAROUT,
    /**< Set Pin for XBAR mode */
	PORT_PIN_MODE_PRU0_GIO,
	/**< Set Pin for PRU0 mode */
	PORT_PIN_MODE_PRU1_GIO,
	/**< Set Pin for PRU0 mode */
	PORT_PIN_MODE_TRC,
	/**< Set Pin for TRC */
	PORT_PIN_MODE_SDFM0,
	/**< Set Pin for SDFM0 */
	PORT_PIN_MODE_SDFM1,
	/**< Set Pin for SDFM1 */
	PORT_PIN_MODE_MMC0,
	/**< Set Pin for MMC0 */
	PORT_PIN_MODE_FSIRX0,
	/**< Set Pin for FSIRX0 */
	PORT_PIN_MODE_FSIRX1,
	/**< Set Pin for FSIRX1 */
	PORT_PIN_MODE_FSIRX2,
	/**< Set Pin for FSIRX2 */
	PORT_PIN_MODE_FSIRX3,
	/**< Set Pin for FSIRX3 */
	PORT_PIN_MODE_FSITX0,
	/**< Set Pin for FSITX0 */
	PORT_PIN_MODE_FSITX1,
	/**< Set Pin for FSITX1 */
	PORT_PIN_MODE_FSITX2,
	/**< Set Pin for FSITX2 */
	PORT_PIN_MODE_FSITX3,
	/**< Set Pin for FSITX3 */
	PORT_PIN_MODE_EQEP0,
	/**< Set Pin for EQEP0 */
	PORT_PIN_MODE_EQEP1,
	/**< Set Pin for EQEP1 */
	PORT_PIN_MODE_EQEP2,
	/**< Set Pin for EQEP2 */
	PORT_PIN_MODE_PR0_IEP0,
	/**< Set Pin for PR0_IEP0 */
	PORT_PIN_MODE_GPMC0,
	/**< Set Pin for GPMC0 */
	PORT_PIN_MODE_CLKOUT,
	/**< Set Pin for CLKOUT */
	PORT_PIN_MODE_EXT_REFCLK,
	/**< Set Pin for EXT_REFCLK */
	PORT_PIN_MODE_PR0_MDIO0,
	/**< Set Pin for PR0_MDIO0 */
    PORT_PIN_MODE_INVALID,
    /**< Invalid pin mode. For internal use */
} Port_PinModeType;


typedef struct
{
    Port_PinModeType mode;
    /**< Selected pin mode */
    uint32          muxmode;
    /**< Function Select/Muxmode for selected pin mode */
} Port_PinModeConfigType;

/**
 *  \brief Pin Configuration structure
 */
typedef struct
{
    /* Requirements : SWS_Port_00013 */
    Port_PinType             Port_PinId;
    /**< Port Pin Id */
    boolean                  Port_DirectionChangeable;
    /**< Flag indicating if pin direction is changeable */
    boolean                  Port_PinModeChangeable;
    /**< Flag indicating if Pin mode is changeable */
    Port_PinLevelValueType   Port_PinLevelValue;
    /**< Pin level value */
    Port_PinDirectionType    Port_PinDirection;
    /**< Pin direction  @sa Port_PinDirectionType */
    uint32                   Port_NumPortModes;
    /**< Number of modes supported for this pin */
    Port_PinModeConfigType   Port_PinMode[PORT_MAX_MUXMODE];
    /**< Pin mode @sa Port_PinModeType */
    Port_PinModeType         Port_PinInitialMode;
    /**< Pin initial mode @sa Port_PinModeType */
    Port_PinPullSelectType   Port_PullTypeSelect;
    /**< Type of PULL U/D selection @sa Port_PinPullSelectType */
    Port_PinSlewControlType  Port_SlewControl;
    /**< Slew control configuration @sa Port_PinSlewControlType */
    Port_PinPullInhibitEnableType Port_PullInhibitEnable;
    /**< Pull inhibit control configuration @sa Port_PinPullInhibitEnableType*/
    Port_PinInputOverrideCtrl Port_InputOverride_Ctrl;
    /**< Active Low input override control*/
    Port_PinOutputOverrideCtrl Port_OutputOverride_Ctrl;
    /**< Active Low output override control*/
    uint32                   Port_RegOffsetAddr;
    /**< Register base address for configuring this pin */
    uint32                   Port_PinDioRegId;
    /**< GPIO register id corresponding to this pin */
    uint32                   Port_PinDioChannelId;
    /**< GPIO channel corresponding to this pin in GPIO register */
    const sint8             *Port_PinSignalName;
    /**< Pin signal name.Just for information. Not used internally.Can be NULL
    **/
    const sint8             *Port_PinName;
	
    /**< Pin name.Just for information. Not used internally.Can be NULL */
	
    boolean                 Port_PinSetEdgeTrigger;
	 /**< To enable Pin Edge trigger*/
	Port_EdgeTrig           Port_PinSelectEdgeTrigger;
	/**< To select Pin Rising/Falling Edge trigger*/

	boolean                 Port_PinHSmasterEnable;
	/** HSMASTER bit value **/

	boolean                 Port_PinHSmodeEnable;
	/** HSMODE bit value **/

	Port_InputInversion     Port_InversionSelect;
	/** select value for choosing inverted version of PAD input for chip**/

	Port_InputQualType      Port_QualifierTypeSelect;
	/** select value for choosing input qualifer type for PAD. **/
} Port_PinConfigType;

/**
 *  \brief Port GPIO initialization configuration
 */
typedef struct
{
    uint8                      Port_DioRegId;
    /**< GPIO register number */
    uint8                      Port_BankInterruptEnable;
    /**< GPIO interrupt polarity select configuration  Rising/Falling*/
	
} Port_DioRegConfigType;




/* Requirements : SWS_Port_00228, SWS_Port_00073, SWS_Port_00072 */
/**
 *  \brief PORT Module ROOT configuration
 */
typedef struct Port_ConfigType_s
{
    uint16 NumberOfPortPins;
    /**< Number of pins individually configure. Should not conflict with
     *   pins enabled by selecting module enable */
    P2CONST(Port_PinConfigType, AUTOMATIC, PORT_PBCFG) PinConfig_pt;
    /**< Pointer to pin configuration structure */
    uint32 NumberOfGPIORegs;
    /**< Number of GPIO registers to configure */
    P2CONST(Port_DioRegConfigType, AUTOMATIC, PORT_PBCFG) DioConfig_pt;
    /**< DIO config structure */
	
} Port_ConfigType;


/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/**
 *  \brief Initializes the Port Driver module
 *
 *  \verbatim
 *  Service name      : Port_Init
 *  Syntax            : void Port_SetPinDirection( Port_ConfigType  ConfigPtr)
 *  Service ID[hex]   : 0x00
 *  Sync/Async        : Non Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : ConfigPtr - Pointer to configuration set.
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : Initializes the Port Driver module
 *  \endverbatim
 */
FUNC(void, PORT_CODE) Port_Init(
    P2CONST(Port_ConfigType, AUTOMATIC, PORT_PBCFG) ConfigPtr);


/* Requirements : SWS_Port_00086 */
#if (STD_ON == PORT_SET_PIN_DIRECTION_API)
/**
 *  \brief Sets the port pin direction
 *
 *  \verbatim
 *  Service name      : Port_SetPinDirection
 *  Syntax            : void Port_SetPinDirection( Port_PinType Pin,
 *                                  Port_PinDirectionType Direction )
 *  Service ID[hex]   : 0x01
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : Pin - Port Pin ID number
 *                      Direction - Port_SetPinDirection
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : Sets the port pin direction
 *  \endverbatim
 */
FUNC(void, PORT_CODE) Port_SetPinDirection(Port_PinType Pin,
                                           Port_PinDirectionType Direction);
#endif /*( STD_ON == PORT_SET_PIN_DIRECTION_API )*/

#if (STD_ON == PORT_REFRESH_PORT_DIRECTION_API)
/**
 *  \brief Refreshes port direction
 *
 *  \verbatim
 *  Service name      : Port_RefreshPortDirection
 *  Syntax            : void Port_RefreshPortDirection(void)
 *  Service ID[hex]   : 0x02
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : Refreshes port direction
 *  \endverbatim
 */
FUNC(void, PORT_CODE) Port_RefreshPortDirection(void);
#endif /*( STD_ON == PORT_REFRESH_PORT_DIRECTION_API )*/

/* Requirements : SWS_Port_00212 */
#if (STD_ON == PORT_SET_PIN_MODE_API)
/**
 *  \brief Sets the port pin mode
 *
 *  \verbatim
 *  Service name      : Port_SetPinMode
 *  Syntax            : void Port_SetPinMode( Port_PinType Pin,
 *                                            Port_PinModeType Mode )
 *  Service ID[hex]   : 0x04
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : Pin - Port Pin ID number
 *                      Mode - New Port Pin mode to be set on port pin
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : Sets the port pin mode
 *  \endverbatim
 */
FUNC(void, PORT_CODE) Port_SetPinMode(Port_PinType Pin, Port_PinModeType Mode);
#endif /*( STD_ON == PORT_SET_PIN_MODE_API )*/

#if (STD_ON == PORT_VERSION_INFO_API)
/**
 *  \brief Returns the version information of this module
 *
 *  \verbatim
 *  Service name      : Port_GetVersionInfo
 *  Syntax            : void Port_GetVersionInfo( Std_VersionInfoType*
 *                                                      versioninfo )
 *  Service ID[hex]   : 0x03
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : versioninfo - Pointer to where to store the version
 *                      information of this module
 *  Return value      : None
 *  Description       : Returns the version information of this module
 *  \endverbatim
 */
FUNC(void, PORT_CODE) Port_GetVersionInfo(
    P2VAR(Std_VersionInfoType, AUTOMATIC, PORT_APPL_DATA) versioninfo);
#endif /*(STD_ON == PORT_VERSION_INFO_API)*/

/**
 *  \brief This service is only used for UT testing and not for app use
 *
 *  \verbatim
 *  Service name      : Port_DeInit
 *  Syntax            : void Port_DeInit(void)
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non-Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : This service sets the internal PORT initDone flag to
 *                      FALSE so that Port_Init can be invoked again.
 *                      This is required for negative UT test cases
 *  \endverbatim
 */
FUNC(void, PORT_CODE) Port_DeInit(void);


/**
 *  \brief   This service returns current/initial configuration items as from the GUI.
 *
 *  \verbatim
 *  Service name      : Port_GetConfigRegValues
 *  Syntax            : void Port_GetConfigRegValues(uint8 initial,  Port_ConfigRegsType *ConfigRegPtr)
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non-Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : Function could be called from interrupt level or from task level,
 *           Output parameter must not be NULL_PTR, TRUE/FALSE should be passed
 *          as initial parameter.
 *  \endverbatim
 */
#if (STD_ON == PORT_SAFETI_API)
FUNC(void, PORT_PBCODE) Port_GetConfigRegValues( uint8 initial,
                                 P2VAR(Port_ConfigRegsType, AUTOMATIC, PORT_PBCFG) ConfigRegPtr );
#endif /* (STD_ON == PORT_SAFETI_API) */

#ifdef __cplusplus
}
#endif

#endif  /* #ifndef PORT_H_ */

/** @} */

