/*
*
* Copyright (c) 2023 Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
* license under copyrights and patents it now or hereafter owns or controls to make,
* have made, use, import, offer to sell and sell ("Utilize") this software subject to the
* terms herein.  With respect to the foregoing patent license, such license is granted
* solely to the extent that any such patent is necessary to Utilize the software alone.
* The patent license shall not apply to any combinations which include this software,
* other than combinations with devices manufactured by or for TI ("TI Devices").
* No hardware patent is licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this license
* (including the above copyright notice and the disclaimer and (if applicable) source
* code license limitations below) in the documentation and/or other materials provided
* with the distribution
*
* Redistribution and use in binary form, without modification, are permitted provided
* that the following conditions are met:
*
* *       No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* *       any redistribution and use are licensed by TI for use only with TI Devices.
*
* *       Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution of the
* source code are permitted provided that the following conditions are met:
*
* *       any redistribution and use of the source code, including any resulting derivative
* works, are licensed by TI for use only with TI Devices.
*
* *       any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

/**
 *  \file     Pwm.c
 *
 *  \brief    This file includes services for initialization and control of the
 *            PWM unit (pulse width modulation).
 *            The PWM module generates pulses with variable pulse width.It
 *            allows the selection of the duty cycle and the signal period time.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
/* Design : DES_PWM_006 */
/* Requirements : MCAL-3506*/
#include "Pwm.h"
#include "SchM_Pwm.h"
#if (STD_ON == PWM_DEV_ERROR_DETECT)
#include "Det.h"
#endif

#include "Pwm_Irq.h"
#include "Pwm_Priv.h"
#include "Pwm_Platform.h"

/* There are static inline functions in hw_types.h file. Map them as well */
#define PWM_START_SEC_CODE
#include "Pwm_MemMap.h"
#include "hw_types.h"
#define PWM_STOP_SEC_CODE
#include "Pwm_MemMap.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* AUTOSAR version information check has to match definition in header file */
#if ((PWM_AR_RELEASE_MAJOR_VERSION != (4U)) || \
    (PWM_AR_RELEASE_MINOR_VERSION != (3U)) ||  \
    (PWM_AR_RELEASE_REVISION_VERSION != (1U)))
    #error "Pwm: AUTOSAR Version Numbers of Pwm are different!!"
#endif

/* AUTOSAR SW version information check has to match definition in header file */
#if ((PWM_SW_MAJOR_VERSION != (8U)) || \
    (PWM_SW_MINOR_VERSION != (6U)) ||  \
    (PWM_SW_PATCH_VERSION != (2U)))
    #error "Pwm: Software Version Numbers are inconsistent!!"
#endif

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */
/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

#define PWM_START_SEC_VAR_INIT_8
#define PWM_ISR_MAX   0xFF
#define PWM_DUTYCYCLE_MAX   0x8000U
#include "Pwm_MemMap.h"
/** \brief Pwm driver init status */
 volatile VAR(uint8, PWM_VAR_INIT)
    Pwm_DrvStatus = PWM_STATUS_UNINIT;

#define PWM_STOP_SEC_VAR_INIT_8
#include "Pwm_MemMap.h"

#define PWM_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Pwm_MemMap.h"
/** \brief Pwm driver interrupt status flag*/
 volatile VAR(Pwm_OutputStateType, PWM_VAR_NO_INIT)
    Pwm_Intrflag[PWM_MAX_NUM_CHANNELS];
/** \brief PWM driver object. */
    VAR(Pwm_ChObjType, PWM_VAR_NO_INIT) Pwm_ChObj[PWM_MAX_NUM_CHANNELS];
    VAR(uint16, PWM_VAR_NO_INIT) Pwm_IsrIndex[PWM_HW_MAX_NUM_CHANNELS];
/* Pwm_ChObj is not static storage class, as this requires to be accessed by
    debug interface provided.  */
#define PWM_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Pwm_MemMap.h"

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#define PWM_START_SEC_CODE
#include "Pwm_MemMap.h"

/* Design : DES_PWM_010, DES_PWM_014 */
/* Requirements : MCAL-3554, MCAL-3555, MCAL-3558, MCAL-3545,
                  MCAL-3546, MCAL-3547, MCAL-3548, MCAL-3551,
                  MCAL-3552, MCAL-3549, MCAL-3550
 */

/* Initializes PWM unit and sets default period and duty cycle values from
 * parameter ConfigPtr.
 */
FUNC(void, PWM_CODE) Pwm_Init(
                           P2CONST(Pwm_ConfigType, AUTOMATIC, PWM_PBCFG) CfgPtr)
{
    uint32                 chIdx,chnum;
    const Pwm_ConfigType  *ConfigPtr = CfgPtr;
#if (STD_ON == PWM_PRE_COMPILE_VARIANT)
    /* Design : DES_PWM_014 */
    /* Requirements : MCAL-3556 */
    ConfigPtr = &PWM_INIT_CONFIG_PC;
#endif /* (STD_ON == PWM_PRE_COMPILE_VARIANT) */
    /* Design : DES_PWM_014*/
    /* Requirements : MCAL-3556,MCAL-3557*/
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    /* Design :DES_PWM_010*/
    /* Requirements : MCAL-3555, MCAL-3527*/
    if (NULL == ConfigPtr)
    {
        /*
         * ConfigPtr must not be NULL. In case it is NULL report an error and
         * return immediately.
         */
         (void)Pwm_reportDetError(PWM_SID_INIT, PWM_E_INIT_FAILED);
    }
    else if (PWM_STATUS_INIT == Pwm_DrvStatus)
    {
        /*
         * Pwm module must not be initialized.
         * Otherwise call the Det with PWM_E_ALREADY_INITIALIZED
         */
        (void)Pwm_reportDetError(PWM_SID_INIT, PWM_E_ALREADY_INITIALIZED);
    }
   else
#endif  /*(STD_ON == PWM_DEV_ERROR_DETECT) */
    {
        
        /* Only initialize configured resources */
        for (chnum = 0U; chnum < PWM_HW_MAX_NUM_CHANNELS; chnum++)
        {
            for (chIdx = 0U; chIdx < PWM_MAX_NUM_CHANNELS; chIdx++)
            {
                if(chnum == PwmChannelConfigSet_PC.chCfg[chIdx].channelHwId)
                {
                    Pwm_IsrIndex[chnum] = chIdx;
					break;
                }
				else
				{
					Pwm_IsrIndex[chnum] = PWM_ISR_MAX;
				}
            }
        }

        /* Design :DES_PWM_014*/
        /* Requirements : MCAL-3546, MCAL-3547*/
        /* Only initialize configured resources */
        for (chnum = 0U; chnum < PWM_MAX_NUM_CHANNELS; chnum++)
        {
#if (STD_ON == PWM_DEV_ERROR_DETECT)
             if(((&PwmChannelConfigSet_PC)->chCfg[chnum].channelId) >=
            (uint32)PWM_HW_MAX_NUM_CHANNELS)
            {
                /*Invalid channel */
                (void)Pwm_reportDetError(PWM_SID_INIT, PWM_E_PARAM_CHANNEL);
            }
#endif
            for (chIdx = 0U; chIdx < PWM_MAX_NUM_CHANNELS; chIdx++)
            {

                if(chIdx == ((&PwmChannelConfigSet_PC)->chCfg[chnum].channelId))
                {
                    /* Reset Channel object */
                    Pwm_ResetChObj(&Pwm_ChObj[chIdx]);
                    /* Copy the configuration */
                    Pwm_CopyConfig(&Pwm_ChObj[chIdx],
                    &ConfigPtr->chCfg[chnum],&PwmChannelConfigSet_PC.chCfg[chnum]);
                    /*Set interrupt flag as per polarity : this is toggled in the
                    ISR for supporting notifications*/
                    Pwm_Intrflag[chIdx] = Pwm_ChObj[chIdx].chCfg.polarity;
                    /* Design : DES_PWM_014 */
                    /* Requirements :MCAL-3548, MCAL-3549, MCAL-3550*/
                    /* HW Channel Init */
                    Pwm_HwUnitInit(&Pwm_ChObj[chIdx]);
                    /* Design : DES_PWM_014 */
                    /* Requirements : MCAL-3551*/
                    /* The function Pwm_Init shall disable all notifications */
                    #if (STD_ON == PWM_NOTIFICATION_SUPPORTED)
                    Pwm_ChObj[chIdx].channelNotifyActiveFlag = (uint32) FALSE;
                    #endif
                    Pwm_ChObj[chIdx].channelState = PWM_STATUS_INIT;
                    break;
                }
            }
        }		
#if (STD_ON == PWM_DEV_ERROR_DETECT)
        /* Requirements : MCAL-3551*/
        Pwm_DrvStatus = PWM_STATUS_INIT;
#endif
    }
	
    return;
}

/* Design : DES_PWM_010, DES_PWM_015 */
/*
 * Requirements : MCAL-3560, MCAL-3559, MCAL-3561,MCAL-3562,MCAL-3563,
 *                MCAL-3564,MCAL-3643, MCAL-3640, MCAL-3641, MCAL-3642
 */

#if (STD_ON == PWM_DEINIT_API)
FUNC(void, PWM_CODE) Pwm_DeInit(void)
{
    uint32                chIdx;
    uint32                baseAddr;
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    if (PWM_STATUS_INIT != Pwm_DrvStatus)
    {
        /* Design : DES_PWM_010, DES_PWM_015 */
        /* Requirements : MCAL-3524 */
        /* If the driver is not initialized, the function shall raise the error
         * PWM_E_UNINIT */
        Pwm_reportDetError(PWM_SID_DEINIT, PWM_E_UNINIT);
    }
    else
#endif
    {
        /* Requirements : MCAL-3560 */
        /* Set driver status  to uninitialized */
        Pwm_DrvStatus = PWM_STATUS_UNINIT;
        for (chIdx = 0U; chIdx < PWM_MAX_NUM_CHANNELS; chIdx++)
        {
            if(PWM_STATUS_INIT == (Pwm_ChObj[chIdx].channelState))
            {
                Pwm_IpDeInit_epwm(&Pwm_ChObj[chIdx]);

                /* Design : DES_PWM_015 */
                /* Requirements : MCAL-3561*/
                /*
                * The function Pwm_Deinit shall disable all interrupt
                * notifications controlled by the PWM driver.
                */
#if (STD_ON == PWM_NOTIFICATION_SUPPORTED)
                Pwm_ChObj[chIdx].channelNotifyActiveFlag = (uint32) FALSE;
#endif
                Pwm_ChObj[chIdx].channelState = PWM_STATUS_UNINIT;
            }
        }
        /* Reset driver object */
        Pwm_ResetChObj(&Pwm_ChObj[0U]);
    }
    return;
}
#endif /*(STD_ON == PWM_DEINIT_API)*/

#if (STD_ON == PWM_SET_DUTY_CYCLE_API)
/*  Sets parameter DutyCycle as new duty cycle for channel ChannelNumber.
 *  DutyCycle is a percentage of the period with 0x8000 being equivalent
 *  to 100%.
 */
/*  DesignId : DES_PWM_016 */
/*  Requirements : MCAL-3565,MCAL-3566, MCAL-3567,MCAL-3568,MCAL-3569,
    MCAL-3533,MCAL-3534,MCAL-3570,MCAL-3643, MCAL-3645,
    MCAL-3640, MCAL-3641,MCAL-3642,MCAL-3571,MCAL-3572
 */
FUNC(void, PWM_CODE) Pwm_SetDutyCycle(
    Pwm_ChannelType ChannelNumber, uint16 DutyCycle)
{
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    if (ChannelNumber >= (uint32)PWM_MAX_NUM_CHANNELS)
    {
        /*Invalid channel */
        (void)Pwm_reportDetError(PWM_SID_SET_DUTY_CYCLE, PWM_E_PARAM_CHANNEL);
    }
    /*  DesignId : DES_PWM_010 */
    /*  Requirements : MCAL-3524, MCAL-3525 */
    else
    {
#endif
        Pwm_ChObjType *chObj = &Pwm_ChObj[ChannelNumber];
#if (STD_ON == PWM_DEV_ERROR_DETECT)		
        if ((PWM_STATUS_INIT != Pwm_DrvStatus) &&
        (PWM_STATUS_INIT != chObj->channelState))
        {
            (void)Pwm_reportDetError(PWM_SID_SET_DUTY_CYCLE, PWM_E_UNINIT);
        }
        else if (DutyCycle > PWM_DUTY_100_PERCENT)
        {
            (void)Pwm_reportDetError(PWM_SID_SET_DUTY_CYCLE, PWM_E_PARAM_CHANNEL);
        }
        else
#endif
        {
            SchM_Enter_Pwm_PWM_EXCLUSIVE_AREA_0();
            /* Requirements : MCAL-3566 */
            Pwm_SetDutyCycle_Internal(chObj, DutyCycle);
            SchM_Exit_Pwm_PWM_EXCLUSIVE_AREA_0();
        }
#if (STD_ON == PWM_DEV_ERROR_DETECT)		
    }
#endif /* (STD_ON == PWM_DEV_ERROR_DETECT) */
    return;
}

#endif /*(STD_ON == PWM_SET_DUTY_CYCLE_API)*/

#if (STD_ON == PWM_SET_PERIOD_AND_DUTY_API)
/*  Sets DutyCycle and Period of ChannelNumber. DutyCycle is a percentage of
 *  the period with 0x8000 being equivalent to 100%. */
/* DesignId : DES_PWM_017 */
/* Requirements : MCAL-3573,MCAL-3574,MCAL-3575, MCAL-3576,MCAL-3533
 *                MCAL-3534,MCAL-3643, MCAL-3644, MCAL-3645, MCAL-3640,
 *                MCAL-3641, MCAL-3642,MCAL-3577,MCAL-3578,MCAL-3579,MCAL-3580
 */
FUNC(void, PWM_CODE) Pwm_SetPeriodAndDuty(
    Pwm_ChannelType ChannelNumber, Pwm_PeriodType Period, uint16 DutyCycle)
{
    uint32 baseAddr;
    EPWM_SignalParams signalParams;
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    if (ChannelNumber >= (uint32)PWM_MAX_NUM_CHANNELS)
    {
        /*Invalid channel */
        (void)Pwm_reportDetError(PWM_SID_SET_PERIOD_AND_DUTY, PWM_E_PARAM_CHANNEL);
    }
    /*  DesignId : DES_PWM_010 */
    /*  Requirements : MCAL-3524, MCAL-3525 ,MCAL-3643, MCAL-3644*/
    else
    {
#endif /* (STD_ON == PWM_DEV_ERROR_DETECT) */
        Pwm_ChObjType *chObj = &Pwm_ChObj[ChannelNumber];

#if (STD_ON == PWM_DEV_ERROR_DETECT)
        if ((PWM_STATUS_INIT != Pwm_DrvStatus) &&
        (PWM_STATUS_INIT != chObj->channelState))
        {
            (void)Pwm_reportDetError(PWM_SID_SET_PERIOD_AND_DUTY, PWM_E_UNINIT);
        }
        else if (chObj->chCfg.channelClass != PWM_VARIABLE_PERIOD)
        {
            (void)Pwm_reportDetError(
                        PWM_SID_SET_PERIOD_AND_DUTY, PWM_E_PERIOD_UNCHANGEABLE);
        }
        else if (DutyCycle > PWM_DUTYCYCLE_MAX)
        {
            (void)Pwm_reportDetError(PWM_SID_SET_DUTY_CYCLE, PWM_E_PARAM_CHANNEL);
        }
        else
#endif /* (STD_ON == PWM_DEV_ERROR_DETECT) */
        {
            SchM_Enter_Pwm_PWM_EXCLUSIVE_AREA_0();
            /* Requirements : MCAL-3575, MCAL-3511*/
            /* Period passed here is in unit ticks */
            chObj->chCfg.hwPeriod = Period;


            EPWM_setTimeBasePeriod(chObj->baseAddr, chObj->chCfg.hwPeriod);

            /* Requirements : MCAL-3580 */
            if (Period == 0U)
            {
                Pwm_SetDutyCycle_Internal(chObj, (uint16) 0U);
            }
            else
            {
                Pwm_SetDutyCycle_Internal(chObj, DutyCycle);
            }
			
            SchM_Exit_Pwm_PWM_EXCLUSIVE_AREA_0();
        }
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    }
#endif /* (STD_ON == PWM_DEV_ERROR_DETECT) */
    return;
}

#endif /*(STD_ON == PWM_SET_PERIOD_AND_DUTY_API)*/

#if (STD_ON == PWM_SET_OUTPUT_TO_IDLE_API)
/* Sets output state immediately to idle level. */
/* DesignId : DES_PWM_018 */
/* Requirements : MCAL-3581, MCAL-3582, MCAL-3583, MCAL-3584, MCAL-3585
                  MCAL-3586,MCAL-3587
 */
FUNC(void, PWM_CODE) Pwm_SetOutputToIdle(
    Pwm_ChannelType ChannelNumber)
{
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    if (ChannelNumber >= (uint32)PWM_MAX_NUM_CHANNELS)
    {
        /*Invalid channel */
        (void)Pwm_reportDetError(PWM_SID_SET_OUTPUT_TO_IDLE, PWM_E_PARAM_CHANNEL);
    }
    /*  DesignId : DES_PWM_010 */
    /*  Requirements : MCAL-3524, MCAL-3525 */
    else
    {
#endif /* (STD_ON == PWM_DEV_ERROR_DETECT) */
        Pwm_ChObjType *chObj = &Pwm_ChObj[ChannelNumber];
		
#if (STD_ON == PWM_DEV_ERROR_DETECT)
        if ((PWM_STATUS_INIT != Pwm_DrvStatus) &&
        (PWM_STATUS_INIT != chObj->channelState))
        {
            (void)Pwm_reportDetError(PWM_SID_SET_OUTPUT_TO_IDLE, PWM_E_UNINIT);
        }
        else
#endif /* (STD_ON == PWM_DEV_ERROR_DETECT) */
        {
            SchM_Enter_Pwm_PWM_EXCLUSIVE_AREA_0();
            /* Requirements : MCAL-3582*/
            /* Set output to idle state */
            Pwm_SetIdleState(chObj);
            chObj->channelForcedIdle = (boolean)TRUE;
            SchM_Exit_Pwm_PWM_EXCLUSIVE_AREA_0();
        }
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    }
#endif /* (STD_ON == PWM_DEV_ERROR_DETECT) */
    return;
}
#endif /*(STD_ON == PWM_SET_OUTPUT_TO_IDLE_API)*/

#if (STD_ON == PWM_NOTIFICATION_SUPPORTED)
    /* Disables notification for channel ChannelNumber. */
    /* DesignId : DES_PWM_018 */
    /* Requirements : MCAL-3592,MCAL-3593,MCAL-3594,MCAL-3595, MCAL-3636, MCAL-3637,
                      MCAL-3643, MCAL-3645, MCAL-3640, MCAL-3641, MCAL-3642 */
FUNC(void, PWM_CODE) Pwm_DisableNotification(
    Pwm_ChannelType ChannelNumber)
{
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    if (ChannelNumber >= (uint32)PWM_MAX_NUM_CHANNELS)
    {
        /*Invalid channel */
        (void)Pwm_reportDetError(PWM_SID_DISABLE_NOTIFICATION, PWM_E_PARAM_CHANNEL);
    }
    else
    {
#endif /* (STD_ON == PWM_DEV_ERROR_DETECT) */
        Pwm_ChObjType *chObj = &Pwm_ChObj[ChannelNumber];
#if (STD_ON == PWM_DEV_ERROR_DETECT)
        if ((PWM_STATUS_INIT != Pwm_DrvStatus) &&
        (PWM_STATUS_INIT != chObj->channelState))
        {
            /* Design : DES_PWM_010 */
            /* Requirements : MCAL-3524, MCAL-3525 */
            (void)Pwm_reportDetError(PWM_SID_DISABLE_NOTIFICATION, PWM_E_UNINIT);
        }
        else
#endif /* (STD_ON == PWM_DEV_ERROR_DETECT) */
        {
            SchM_Enter_Pwm_PWM_EXCLUSIVE_AREA_0();

        /* Requirements : MCAL-3593 */
			
			EPWM_clearEventTriggerInterruptFlag(chObj->baseAddr);
            EPWM_disableInterrupt(chObj->baseAddr);

            /* Set the notification active flag to FALSE */
            chObj->channelNotifyActiveFlag = (uint32) FALSE;
            SchM_Exit_Pwm_PWM_EXCLUSIVE_AREA_0();
        }
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    }
#endif /* (STD_ON == PWM_DEV_ERROR_DETECT) */
    return;
}
#endif /* #if (STD_ON == PWM_NOTIFICATION_SUPPORTED)*/

#if (STD_ON == PWM_NOTIFICATION_SUPPORTED)
    /* Enables notification for channel Channel. */
    /* DesignId : DES_PWM_020 */
    /* Requirements : MCAL-3596, MCAL-3597,MCAL-3598, MCAL-3599, MCAL-3600, MCAL-3637,
                      MCAL-3643, MCAL-3645, MCAL-3640, MCAL-3641, MCAL-3642, MCAL-3635
 */
FUNC(void, PWM_CODE) Pwm_EnableNotification(
    Pwm_ChannelType ChannelNumber, Pwm_EdgeNotificationType Notification)
{
	
	 Pwm_OutputStateType polarity;
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    if (ChannelNumber >= (uint32)PWM_MAX_NUM_CHANNELS)
    {
        /*Invalid channel */
        (void)Pwm_reportDetError(PWM_SID_ENABLE_NOTIFICATION, PWM_E_PARAM_CHANNEL);
    }
    /* Design : DES_PWM_010 */
    /* Requirements : MCAL-3524, MCAL-3525 */
    else
    {
#endif /* #if (STD_ON == PWM_DEV_ERROR_DETECT)*/
         Pwm_ChObjType *chObj = &Pwm_ChObj[ChannelNumber];
         const Pwm_ChannelConfigType_PC *channelConfig_PC = &chObj->chCfg_PC;
		 
#if (STD_ON == PWM_DEV_ERROR_DETECT)
         if ((PWM_STATUS_INIT != Pwm_DrvStatus) &&
        (PWM_STATUS_INIT != chObj->channelState))
        {
            (void)Pwm_reportDetError(PWM_SID_ENABLE_NOTIFICATION, PWM_E_UNINIT);
        }
        else if ((Pwm_NotifyFuncType) NULL_PTR == chObj->chCfg.notificationHandler)
        {
            (void)Pwm_reportDetError(
                        PWM_SID_ENABLE_NOTIFICATION, PWM_E_PARAM_CHANNEL);
        }
        else
#endif /* #if (STD_ON == PWM_DEV_ERROR_DETECT)*/
        {

            if (PWM_STATUS_UNINIT != chObj->channelState)
            {
                /* Design : DES_PWM_018 */
                /* Requirements : MCAL-2056 */
                /* Critical section, block the interrupts */
                /* SchM_Enter_Pwm_PWM_EXCLUSIVE_AREA_0(); */
                /* Cancel all pending interrupts */
                /* Requirements : MCAL-3598, MCAL-3634*/
                EPWM_clearEventTriggerInterruptFlag(chObj->baseAddr);
				EPWM_enableInterrupt(chObj->baseAddr);
                /* Enable notification only if the duty cycle is not 0% or 100% */
                if ((chObj->chCfg.dutyCycle == PWM_DUTY_0_PERCENT) ||
                        (chObj->chCfg.dutyCycle == PWM_DUTY_100_PERCENT))
                {
					
					
	                				
                                    /* Generate interrupt when count equal to zero,
                     * but do not enable rising/falling edge notification as it is
                     * meaningless if duty cycle is 0% or 100% */
                    EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_ZERO,
                                            EPWM_INT_TBCTR_ZERO);
                }
                else
                {
                    polarity = chObj->chCfg.polarity;
                    switch (Notification)
                    {
                    case PWM_FALLING_EDGE:
                        if (polarity == PWM_HIGH)
                        {
                            if ((uint32)PWM_OUTPUT_CH_A == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_D_CMPA, EPWM_INT_TBCTR_ZERO);
                            }

                            if ((uint32)PWM_OUTPUT_CH_B == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_D_CMPB, EPWM_INT_TBCTR_ZERO);
                            }

                            if ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_ETINTMIX, EPWM_INT_TBCTR_CAD_CBD);
                            }
                        }
                        else
                        {
                            if((uint32)PWM_OUTPUT_CH_A == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_U_CMPA, EPWM_INT_TBCTR_ZERO);
                            }

                            if ((uint32)PWM_OUTPUT_CH_B == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_U_CMPB, EPWM_INT_TBCTR_ZERO);
                            }

                            if ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_ETINTMIX, EPWM_INT_TBCTR_CAU_CBU);
                            }
                        }
                        break;
                    case PWM_RISING_EDGE:
                        if (polarity == PWM_HIGH)
                        {
                            if ((uint32)PWM_OUTPUT_CH_A == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_U_CMPA, EPWM_INT_TBCTR_ZERO);
                            }

                            if ((uint32)PWM_OUTPUT_CH_B == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_U_CMPB, EPWM_INT_TBCTR_ZERO);
                            }

                            if ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_ETINTMIX, EPWM_INT_TBCTR_CAU_CBU);
                            }
                        }
                        else
                        {
                            if ((uint32)PWM_OUTPUT_CH_A == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_D_CMPA, EPWM_INT_TBCTR_ZERO);
                            }

                            if ((uint32)PWM_OUTPUT_CH_B == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_D_CMPB, EPWM_INT_TBCTR_ZERO);
                            }

                            if ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == channelConfig_PC->outputCh)
                            {
                                EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_ETINTMIX, EPWM_INT_TBCTR_CAD_CBD);
                            }

                        }
                        break;
                    case PWM_BOTH_EDGES:
                        /*LDRA_INSPECTED 95 S : MISRAC_2012_R.11.4 "Reason - Pointer typecast required here */
                        if ((uint32)PWM_OUTPUT_CH_A == channelConfig_PC->outputCh)
                        {
                            EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_ETINTMIX, 0x000C);
                        }

                        if ((uint32)PWM_OUTPUT_CH_B == channelConfig_PC->outputCh)
                        {
                            EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_ETINTMIX, 0x0020);
                        }

                        if ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == channelConfig_PC->outputCh)
                        {
                            EPWM_setInterruptSource(chObj->baseAddr, EPWM_INT_TBCTR_ETINTMIX, 0x003C);
                        }
                        break;
                    default :
                        /*Do nothing*/
                        break;
                    }

                }
               
	            EPWM_setInterruptEventCount(chObj->baseAddr, PWM_EVENT_COUNT);
	            EPWM_disableInterruptEventCountInit(chObj->baseAddr);
	            EPWM_setInterruptEventCountInitValue(chObj->baseAddr, 0);

                /* Requirements : MCAL-3597*/
                chObj->channelNotifyActiveFlag = (uint32) TRUE;
                /* SchM_Exit_Pwm_PWM_EXCLUSIVE_AREA_0(); */
            }
        }
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    }
#endif
    return;
}
#endif

#if (STD_ON == PWM_REGISTER_READBACK_API)
 /* DesignId : DES_PWM_022 */
 /* Requirements : MCAL-3662,MCAL-3656 */
FUNC(Std_ReturnType, PWM_CODE) Pwm_RegisterReadback(
    Pwm_ChannelType PwmChannel,
    P2VAR(Pwm_RegisterReadbackType, AUTOMATIC, PWM_APPL_DATA) RegRbPtr)
{
    Std_ReturnType        retVal     = ((Std_ReturnType) E_NOT_OK);
    uint32 baseAddr;
    /* HW unit ID is the index, so search for matching HW unit */
    Pwm_ChObjType *chObj = &Pwm_ChObj[PwmChannel];
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    if (PwmChannel >= (uint32)PWM_MAX_NUM_CHANNELS)
    {
        /*Invalid channel */
        (void)Pwm_reportDetError(PWM_SID_REGISTER_READBACK, PWM_E_PARAM_CHANNEL);
    }
    else if ((PWM_STATUS_INIT != Pwm_DrvStatus) &&
    (PWM_STATUS_INIT != chObj->channelState))
    {
        (void)Pwm_reportDetError(PWM_SID_REGISTER_READBACK, PWM_E_UNINIT);
    }
    else if (NULL_PTR == RegRbPtr)
    {
        (void)Pwm_reportDetError(PWM_SID_REGISTER_READBACK, PWM_E_PARAM_POINTER);
    }
    else if (NULL == chObj)
    {
        (void)Pwm_reportDetError(PWM_SID_REGISTER_READBACK, PWM_E_PARAM_CHANNEL);
    }
    else
#endif
    {
        (void) memset(RegRbPtr, 0, sizeof (Pwm_RegisterReadbackType));
        baseAddr = chObj->baseAddr;
        Pwm_IpRegisterReadback_epwm(baseAddr,RegRbPtr);
        retVal = (Std_ReturnType) E_OK;
    }
    return (retVal);
}
#endif

#if (STD_ON == PWM_VERSION_INFO_API)
/* Design : DES_PWM_021 */
/* Requirements : MCAL-3624 */
FUNC(void, PWM_CODE) Pwm_GetVersionInfo(
            P2VAR(Std_VersionInfoType, AUTOMATIC, PWM_APPL_DATA) VersionInfoPtr)
{
#if (STD_ON == PWM_DEV_ERROR_DETECT)
    if (NULL_PTR == VersionInfoPtr)
    {
        /* Design : DES_PWM_010 */
        /* Requirements : MCAL-4480 */
        Pwm_reportDetError(PWM_SID_GET_VERSION_INFO, PWM_E_PARAM_POINTER);
    }
    else
#endif
    {
        VersionInfoPtr->vendorID         = PWM_VENDOR_ID;
        VersionInfoPtr->moduleID         = PWM_MODULE_ID;
        VersionInfoPtr->sw_major_version = (uint8) PWM_SW_MAJOR_VERSION;
        VersionInfoPtr->sw_minor_version = (uint8) PWM_SW_MINOR_VERSION;
        VersionInfoPtr->sw_patch_version = (uint8) PWM_SW_PATCH_VERSION;
    }

    return;
}
#endif /* PWM_VERSION_INFO_API*/

#define PWM_STOP_SEC_CODE
#include "Pwm_MemMap.h"
